/**
  *********************************************************************************
  *
  * @file    md_pmu.h
  * @brief   Header file of PMU module driver.
  *
  * @version V1.0
  * @date    16 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          16 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __MD_PMU_H__
#define __MD_PMU_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "md_utils.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_PMU PMU
  * @brief PMU micro driver
  * @{
  */

/** @defgroup MD_PMU_Public_Types PMU Public Types
  * @{
  */

/**
  * @brief Low power mode
  */
typedef enum {
        MD_PMU_LP_STOP1   = 0x0U,        /**< Stop1 */
        MD_PMU_LP_STOP2   = 0x1U,        /**< Stop2 */
        MD_PMU_LP_STANDBY = 0x2U,        /**< Standby */
} md_pmu_lp_mode_t;

/**
  * @brief LDO_18 mode
  */
typedef enum {
	MD_PMU_LDO_18_DRV_HIGH = 0x0U, /**< High drive */
	MD_PMU_LDO_18_DRV_LOW  = 0x1U, /**< Low drive */
	MD_PMU_LDO_18_HOLD     = 0x2U, /**< Hold */
	MD_PMU_LDO_18_OFF      = 0x3U, /**< Turn off */
} md_pmu_ldo_18_mode_t;

/**
  * @brief Standby wakeup port select
  */
typedef enum {
	MD_PMU_STANDBY_PORT_SEL_PA0  = 0x0U,	/**< Wakeup by PA0 */
	MD_PMU_STANDBY_PORT_SEL_PA1  = 0x1U,	/**< Wakeup by PA1 */
	MD_PMU_STANDBY_PORT_SEL_PA2  = 0x2U,	/**< Wakeup by PA2 */
	MD_PMU_STANDBY_PORT_SEL_PA3  = 0x3U,	/**< Wakeup by PA3 */
	MD_PMU_STANDBY_PORT_SEL_PA4  = 0x4U,	/**< Wakeup by PA4 */
	MD_PMU_STANDBY_PORT_SEL_PA5  = 0x5U,	/**< Wakeup by PA5 */
	MD_PMU_STANDBY_PORT_SEL_PA6  = 0x6U,	/**< Wakeup by PA6 */
	MD_PMU_STANDBY_PORT_SEL_PA7  = 0x7U,	/**< Wakeup by PA7 */
	MD_PMU_STANDBY_PORT_SEL_NONE = 0xFU,	/**< Wakeup by other source */
} md_pmu_wakeup_port_t;

/**
  * @brief Standby wakeup level
  */
typedef enum {
	MD_PMU_STANDBY_LEVEL_HIGH = 0x0U,	/**< High level */
	MD_PMU_STANDBY_LEVEL_LOW  = 0x1U,	/**< Low level */
} md_pmu_wakeup_level_t;


/**
  * @brief status types
  */
typedef enum {
	MD_PMU_SR_WUF      = (1U << 0),	/**< WakeUp status */
	MD_PMU_SR_STANDBYF = (1U << 1),	/**< Standby status */
	MD_PMU_SR_USBRDY   = (1U << 2),	/**< USB Readby status */
	MD_PMU_SR_RSTAT    = (1U << 3),	/**< RUN/LPRUAN status */
} md_pmu_status_t;

/**
  * @brief LVD voltage select
  */
typedef enum {
	MD_PMU_LVD_VOL_SEL_2_2 = 0x0U,	/**< 2.2V ~ 2.25V */
	MD_PMU_LVD_VOL_SEL_2_4 = 0x1U,	/**< 2.4V ~ 2.45V */
	MD_PMU_LVD_VOL_SEL_2_6 = 0x2U,	/**< 2.6V ~ 2.65V */
	MD_PMU_LVD_VOL_SEL_2_8 = 0x3U,	/**< 2.8V ~ 2.85V */
	MD_PMU_LVD_VOL_SEL_3_0 = 0x4U,	/**< 3.0V ~ 3.05V */
	MD_PMU_LVD_VOL_SEL_3_2 = 0x5U,	/**< 3.2V ~ 3.25V */
	MD_PMU_LVD_VOL_SEL_3_4 = 0x6U,	/**< 3.4V ~ 3.45V */
	MD_PMU_LVD_VOL_SEL_3_6 = 0x7U,	/**< 3.6V ~ 3.65V */
	MD_PMU_LVD_VOL_SEL_3_8 = 0x8U,	/**< 3.8V ~ 3.85V */
	MD_PMU_LVD_VOL_SEL_4_0 = 0x9U,	/**< 4.0V ~ 4.05V */
	MD_PMU_LVD_VOL_SEL_4_2 = 0xAU,	/**< 4.2V ~ 4.25V */
	MD_PMU_LVD_VOL_SEL_4_4 = 0xBU,	/**< 4.4V ~ 4.45V */
	MD_PMU_LVD_VOL_SEL_4_6 = 0xCU,	/**< 4.6V ~ 4.65V */
	MD_PMU_LVD_VOL_SEL_4_8 = 0xDU,	/**< 4.8V ~ 4.85V */
	MD_PMU_LVD_VOL_SEL_EXT = 0xFU,	/**< Select external input. It must be 1.2V */
} md_pmu_lvd_voltage_sel_t;

/**
  * @brief LVD trigger mode
  */
typedef enum {
	MD_PMU_LVD_TRIGGER_RISING_EDGE    = 0x0U,	/**< Rising edge */
	MD_PMU_LVD_TRIGGER_FALLING_EDGE   = 0x1U,	/**< Falling edge */
	MD_PMU_LVD_TRIGGER_HIGH_LEVEL     = 0x2U,	/**< High level */
	MD_PMU_LVD_TRIGGER_LOW_LEVEL      = 0x3U,	/**< Low level */
	MD_PMU_LVD_TRIGGER_RISING_FALLING = 0x4U,	/**< Rising and falling edge */
} md_pmu_lvd_trigger_mode_t;

typedef enum {
	MD_PMU_POWER_SRAM0 = 0x1U,	/**< SRAM0 */
	MD_PMU_POWER_SRAM1 = 0x2U,	/**< SRAM1 */
	MD_PMU_POWER_SRAM2 = 0x4U,	/**< SRAM2 */
	MD_PMU_POWER_SRAM3 = 0x8U,	/**< SRAM3 */
	MD_PMU_POWER_SRAM4 = 0x10U,	/**< SRAM4 */
	MD_PMU_POWER_SRAM5 = 0x20U,	/**< SRAM5 */
	MD_PMU_POWER_CAN   = 0x100U,	/**< CAN */
	MD_PMU_POWER_QSPI  = 0x400U,	/**< QSPI */
	MD_PMU_POWER_USB   = 0x800U,	/**< USB */
	MD_PMU_POWER_ROM   = 0x1000U,	/**< ROM */
} md_pmu_perh_power_t;

/**
  * @}
  */

/** @defgroup MD_PMU_Public_Functions PMU Public Functions
  * @{
  */
/** @defgroup MD_PMU_Public_Functions_Group2 CR0
  * @{
  */
/**
  * @brief  Set pmu low power mode.
  * @param  mode: low power mode @see md_pmu_lp_mode_t
  * @retval None
  */
__STATIC_INLINE void md_pmu_set_low_power_mode(md_pmu_lp_mode_t mode)
{
	MODIFY_REG(PMU->CR0, PMU_CR0_LPM_MSK, mode << PMU_CR0_LPM_POSS);
}

/**
  * @brief  Get pmu low power mode.
  * @retval low power mode @see md_pmu_lp_mode_t
  */
__STATIC_INLINE uint32_t md_pmu_get_low_power_mode(void)
{
	return READ_BITS(PMU->CR0, PMU_CR0_LPM_MSK, PMU_CR0_LPM_POSS);
}

/**
  * @brief  Clear wuf flag.
  * @retval None
  */
__STATIC_INLINE void md_pmu_clear_flag_cwuf(void)
{
	SET_BIT(PMU->CR0, PMU_CR0_CWUF_MSK);
}

/**
  * @brief  Clear standby flag.
  * @retval None
  */
__STATIC_INLINE void md_pmu_clear_flag_standby(void)
{
	SET_BIT(PMU->CR0, PMU_CR0_CSTANDBYF_MSK);
}

/**
  * @brief  Set flash mode in sleep mode.
  * @brief  mode: 0-idle, 1-standby
  * @retval None
  */
__STATIC_INLINE void md_pmu_set_flash_mode(uint8_t mode)
{
	MODIFY_REG(PMU->CR0, PMU_CR0_SFPD_MSK, mode << PMU_CR0_SFPD_POS);
}

/**
  * @brief  Get flash mode in sleep mode.
  * @retval Mode: 0-idle, 1-standby.
  */
__STATIC_INLINE uint8_t md_pmu_get_flash_mode(void)
{
	return READ_BITS(PMU->CR0, PMU_CR0_SFPD_MSK, PMU_CR0_SFPD_POS);
}

/**
  * @brief  Enable LDO low power mode.
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_ldo_mtstop(void)
{
	SET_BIT(PMU->CR0, PMU_CR0_MTSTOP_MSK);
}

/**
  * @brief  Disable LDO low power mode.
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_ldo_mtstop(void)
{
	CLEAR_BIT(PMU->CR0, PMU_CR0_MTSTOP_MSK);
}

/**
  * @brief  Check if LDO low power is enabled or disabled.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_enable_stop_mtstop(void)
{
	return (READ_BIT(PMU->CR0, PMU_CR0_MTSTOP_MSK) == (PMU_CR0_MTSTOP_MSK));
}
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group3 CR1
  * @{
  */
/**
  * @brief  Get status of 1.8V LDO.
  * @retval State. 1-stable, 0-don't stable
  */
__STATIC_INLINE uint32_t md_pmu_get_1_8_ldo(void)
{
	return (READ_BITS(PMU->CR1, PMU_CR1_LDO18RDY_MSK, PMU_CR1_LDO18RDY_POS));
}

/**
  * @brief  Set the mode of 1.8v LDO.
  * @param  mode:
  *           0x0 - High drive
  *           0x1 - Low drive
  *           0x2 - Maintain
  *           0x3 - Close
  * @retval None
  */
__STATIC_INLINE void md_pmu_set_1_8_ldo_mode(uint32_t mode)
{
	MODIFY_REG(PMU->CR1, PMU_CR1_LDO18MOD_MSK, (mode << PMU_CR1_LDO18MOD_POSS));
}

/**
  * @brief  Get the mode of 1.8v LDO.
  * @retval  mode:
  *           0x0 - High drive
  *           0x1 - Low drive
  *           0x2 - Maintain
  *           0x3 - Close
  */
__STATIC_INLINE uint32_t md_pmu_get_1_8_ldo_mode(void)
{
	return (READ_BITS(PMU->CR1, PMU_CR1_LDO18MOD_MSK, PMU_CR1_LDO18MOD_POSS));
}
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group4 SR
  * @{
  */
/**
  * @brief  Get Standby flag .
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_active_flag_standby(void)
{
	return (READ_BIT(PMU->SR, PMU_SR_STANDBYF_MSK) == PMU_SR_STANDBYF_MSK);
}

/**
  * @brief  Get Wakeup flag .
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_active_flag_wakeup(void)
{
	return (READ_BIT(PMU->SR, PMU_SR_WUF_MSK) == PMU_SR_WUF_MSK);
}
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group5 LVDCR
  * @{
  */
/**
  * @brief  Get LVD Status flag .
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_active_flag_lvdo(void)
{
	return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDO_MSK) == PMU_LVDCR_LVDO_MSK);
}

/**
  * @brief  Enable the LVD filter .
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_lvd_filter(void)
{
	SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDFLT_MSK);
}

/**
  * @brief  Disable the LVD filter .
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_lvd_filter(void)
{
	CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDFLT_MSK);
}

/**
  * @brief  Check if the LVD filter is enable or disable.
  * @retval State for bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_pmu_is_enable_lvd_filter(void)
{
	return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDFLT_MSK) == PMU_LVDCR_LVDFLT_MSK);
}

/**
  * @brief  Set LVD Interrupt mode.
  * @param  mode: lvd interrupt mode @see md_pmu_lvd_trigger_mode_t.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_set_lvd_it_mode(md_pmu_lvd_trigger_mode_t mode)
{
	MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVIFS_MSK, mode << PMU_LVDCR_LVIFS_POSS);
}

/**
  * @brief  Get LVD Interrupt mode.
  * @retval value for lvd it mode.
  */
__STATIC_INLINE uint32_t md_pmu_get_lvd_it_mode(void)
{
	return READ_BITS(PMU->LVDCR, PMU_LVDCR_LVIFS_MSK, PMU_LVDCR_LVIFS_POSS);
}

/**
  * @brief  Set LVD voltage threshold value.
  * @param  value: threshold value @see md_pmu_lvd_voltage_sel_t.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_set_lvd_vol_threshold(md_pmu_lvd_voltage_sel_t value)
{
	MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVDS_MSK, value << PMU_LVDCR_LVDS_POSS);
}

/**
  * @brief  Get LVD voltage threshold value.
  * @retval value for lvd voltage threshold @see pmu_lvd_voltage_sel_t.
  */
__STATIC_INLINE uint32_t md_pmu_get_lvd_vol_threshold(void)
{
	return (READ_BITS(PMU->LVDCR, PMU_LVDCR_LVDS_MSK, PMU_LVDCR_LVDS_POSS));
}

/**
  * @brief  Set LVD interrupt flag clear bit.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_set_lvd_it_flag(void)
{
	SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
}

/**
  * @brief  Get LVD interrupt flag.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_active_flag_lvd_it(void)
{
	return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDIF_MSK) == PMU_LVDCR_LVDIF_MSK);
}

/**
  * @brief  Enable LVD interrupt.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_enable_lvd_it(void)
{
	SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
}

/**
  * @brief  Disable LVD interrupt.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_disable_lvd_it(void)
{
	CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
}

/**
  * @brief  Check if LVD interrupt is enable or disable.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_enable_lvd_it(void)
{
	return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK) == PMU_LVDCR_LVDIE_MSK);
}

/**
  * @brief  Enable LVD.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_enable_lvd(void)
{
	SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
}

/**
  * @brief  Disable LVD interrupt.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_disable_lvd(void)
{
	CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
}

/**
  * @brief  Check if LVD is enable or disable.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_lvd_is_enable(void)
{
	return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK) == PMU_LVDCR_LVDEN_MSK);
}
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group6 PWRCR
  * @{
  */
/**
  * @brief  Enable BXCAN power.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_enable_bxcan(void)
{
	SET_BIT(PMU->PWRCR, PMU_PWRCR_BXCAN_MSK);
}

/**
  * @brief  Disable BXCAN power.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_disable_bxcan(void)
{
	CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_BXCAN_MSK);
}

/**
  * @brief  Check if BXCAN power is enable or disable.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_enable_bxcan(void)
{
	return (READ_BIT(PMU->PWRCR, PMU_PWRCR_BXCAN_MSK) == PMU_PWRCR_BXCAN_MSK);
}

/**
  * @brief  Set SRAM which can be supply power.
  * @param  idx: sram power supply[0-5]
  * @retval None
  */
__STATIC_INLINE void md_pmu_set_sram_power_supply(uint32_t idx)
{
	MODIFY_REG(PMU->PWRCR, PMU_PWRCR_SRAM_MSK, (1 << idx) << PMU_PWRCR_SRAM_POSS);
}

/**
  * @brief  Get status which SRAM is supply power.
  * @retval Status for SRAM power supply
  */
__STATIC_INLINE uint32_t md_pmu_get_sram_power_supply(void)
{
	return READ_BITS(PMU->PWRCR, PMU_PWRCR_SRAM_MSK, PMU_PWRCR_SRAM_POSS);
}
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group7 PWRCR
  * @{
  */
/**
  * @brief  Enable rom power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_rom_power(void)
{
	SET_BIT(PMU->PWRCR, PMU_PWRCR_ROM_MSK);
}

/**
  * @brief  Disable rom power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_rom_power(void)
{
	CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_ROM_MSK);
}

/**
  * @brief  Check if rom power is enable or disable.
  * @retval State for 1 or 0.
  */
__STATIC_INLINE uint8_t md_pmu_is_enable_rom_power(void)
{
	return (READ_BIT(PMU->PWRCR, PMU_PWRCR_ROM_MSK) == PMU_PWRCR_ROM_MSK);
}

/**
  * @brief  Enable usb power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_usb_power(void)
{
	SET_BIT(PMU->PWRCR, PMU_PWRCR_USB_MSK);
}

/**
  * @brief  Disable usb power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_usb_power(void)
{
	CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_USB_MSK);
}

/**
  * @brief  Check if usb power is enable or disable.
  * @retval State for 1 or 0.
  */
__STATIC_INLINE uint8_t md_pmu_is_enable_usb_power(void)
{
	return (READ_BIT(PMU->PWRCR, PMU_PWRCR_USB_MSK) == PMU_PWRCR_USB_MSK);
}

/**
  * @brief  Enable qspi power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_qspi_power(void)
{
	SET_BIT(PMU->PWRCR, PMU_PWRCR_QSPI_MSK);
}

/**
  * @brief  Disable qspi power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_qspi_power(void)
{
	CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_QSPI_MSK);
}

/**
  * @brief  Check if qspi power is enable or disable.
  * @retval State for 1 or 0.
  */
__STATIC_INLINE uint8_t md_pmu_is_enable_qspi_power(void)
{
	return (READ_BIT(PMU->PWRCR, PMU_PWRCR_QSPI_MSK) == PMU_PWRCR_QSPI_MSK);
}

/**
  * @brief  Enable can power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_can_power(void)
{
	SET_BIT(PMU->PWRCR, PMU_PWRCR_BXCAN_MSK);
}

/**
  * @brief  Disable can power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_can_power(void)
{
	CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_BXCAN_MSK);
}

/**
  * @brief  Check if can power is enable or disable.
  * @retval State for 1 or 0.
  */
__STATIC_INLINE uint8_t md_pmu_is_enable_can_power(void)
{
	return (READ_BIT(PMU->PWRCR, PMU_PWRCR_BXCAN_MSK) == PMU_PWRCR_BXCAN_MSK);
}

/**
  * @brief  Enable sram power.
  * @param  idx: Index of sram, [0-5]
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_sram_power(uint8_t idx)
{
	SET_BIT(PMU->PWRCR, (1 << idx));
}

/**
  * @brief  Disable sram power.
  * @param  idx: Index of sram, [0-5]
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_sram_power(uint8_t idx)
{
	CLEAR_BIT(PMU->PWRCR, (1 << idx));
}

/**
  * @brief  Check if sram power is enable or disable.
  * @param  idx: Index of sram, [0-5]
  * @retval State for 1 or 0.
  */
__STATIC_INLINE uint8_t md_pmu_is_enable_sram_power(uint8_t idx)
{
	return (READ_BIT(PMU->PWRCR, (1 << idx)) == (1 << idx));
}
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group8 VREFCR
  * @{
  */
/**
  * @brief  Enable vref.
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_vrefen(void)
{
	SET_BIT(PMU->VREFCR, PMU_VREFCR_VREFEN_MSK);
}

/**
  * @brief  Disable vref.
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_vrefen(void)
{
	CLEAR_BIT(PMU->VREFCR, PMU_VREFCR_VREFEN_MSK);
}

/**
  * @brief  Check if vref is enable or disable.
  * @retval State for 1 or 0.
  */
__STATIC_INLINE uint8_t md_pmu_is_enable_vrefen(void)
{
	return (READ_BIT(PMU->VREFCR, PMU_VREFCR_VREFEN_MSK) == PMU_VREFCR_VREFEN_MSK);
}
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group1 Initialization
  * @{
  */
__STATIC_INLINE__ void md_pmu_sleep()
{
	__WFI();
}

__STATIC_INLINE__ void md_pmu_sleep_deep()
{
	SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
	__WFI();
}

extern void md_pmu_reset(void);
extern void md_pmu_stop1_enter(void);
extern void md_pmu_stop2_enter(void);
extern void md_pmu_standby_enter(md_pmu_wakeup_port_t port, md_pmu_wakeup_level_t level);
extern void md_pmu_perh_power_config(md_pmu_perh_power_t perh, type_func_t state);
extern void md_pmu_lvd_config(md_pmu_lvd_voltage_sel_t sel, md_pmu_lvd_trigger_mode_t mode, type_func_t state);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
