/**
  *********************************************************************************
  *
  * @file    md_dac.h
  * @brief   Header file of DAC module driver.
  *
  * @version V1.0
  * @date    20 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          20 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */


#ifndef __MD_DAC_H__
#define __MD_DAC_H__

#ifdef __cplusplus
 extern "C" {
#endif

#include "md_utils.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_DAC DAC
  * @brief DAC micro driver
  * @{
  */

/** @defgroup DAC_Pubulic_Types DAC Pubulic Types
  * @{
  */
/**
  * @brief DAC channel
  */
typedef enum {
	MD_DAC_CHANNEL_0    = 1U,	/**< DAC channel 0 */
	MD_DAC_CHANNEL_1    = 2U,	/**< DAC channel 1 */
	MD_DAC_CHANNEL_COMB = 3U,	/**< DAC channel 0 and 1 */
} md_dac_channel_t;

/**
  * @brief DAC convert mode
  */
typedef enum {
	MD_DAC_CONV_MODE_CONTINUOUS = 0U,	/**< DAC set in continuous mode */
	MD_DAC_CONV_MODE_SAMPLEHOLD = 1U,	/**< DAC set in sample/hold mode */
	MD_DAC_CONV_MODE_SAMPLEOFF  = 2U,	/**< DAC set in sample/shut off mode */
} md_dac_conv_mode_t;

/**
  * @brief DAC out mode
  */
typedef enum {
	MD_DAC_OUTPUT_DISABLE = 0U,	/**< DAC output disable */
	MD_DAC_OUTPUT_PIN     = 1U,	/**< DAC output to pin enable */
	MD_DAC_OUTPUT_ADC     = 2U,	/**< DAC output to dac and acmp enable */
	MD_DAC_OUTPUT_PIN_ADC = 3U,	/**< DAC output to pin, dac and acmp enable */
} md_dac_out_mode_t;

/**
  * @brief DAC negative reference voltage definition
  */
typedef enum {
	MD_DAC_NEG_REF_VSS   = 0x0U,	/**< DAC negative regerence voltage vss */
	MD_DAC_NEG_REF_VREFN = 0x1U,	/**< DAC negative regerence voltage vrefn */
} md_dac_neg_ref_t;

/**
  * @brief DAC positive reference voltage definition
  */
typedef enum {
	MD_DAC_POS_REF_VDD        = 0x0U,	/**< DAC posotove reference is VDD */
	MD_DAC_POS_REF_2V         = 0x1U,	/**< DAC posotove reference is internal 2V */
	MD_DAC_POS_REF_VREEFP     = 0x2U,	/**< DAC posotove reference is VREEFP */
	MD_DAC_POS_REF_VREEFP_BUF = 0x3U,	/**< DAC posotove reference is VREEFP BUFFER */
} md_dac_pos_ref_t;

/**
  * @brief Refresh interval select
  */
typedef enum {
	MD_DAC_REFRESH_8  = 0U,	/**< Channel refreshed every 8 cycles */
	MD_DAC_REFRESH_16 = 1U,	/**< Channel refreshed every 16 cycles */
	MD_DAC_REFRESH_32 = 2U,	/**< Channel refreshed every 32 cycles */
	MD_DAC_REFRESH_64 = 3U,	/**< Channel refreshed every 64 cycles */
} md_dac_refresh_t;

/**
  * @brief DAC prescale
  */
typedef enum {
	MD_DAC_PRES_DIV_1   = 0U,	/**< No division */
	MD_DAC_PRES_DIV_2   = 1U,	/**< 2 clock division */
	MD_DAC_PRES_DIV_4   = 2U,	/**< 4 clock division */
	MD_DAC_PRES_DIV_8   = 3U,	/**< 8 clock division */
	MD_DAC_PRES_DIV_16  = 4U,	/**< 16 clock division */
	MD_DAC_PRES_DIV_32  = 5U,	/**< 32 clock division */
	MD_DAC_PRES_DIV_64  = 6U,	/**< 64 clock division */
	MD_DAC_PRES_DIV_128 = 7U,	/**< 128 clock division */
} md_dac_prescale_div_t;

/**
  * @brief DAC pis input channel
  */
typedef enum {
	MD_DAC_PIS_CH_0  = 0U,	/**< PIS channel 0 triggers DAC channel conversion */
	MD_DAC_PIS_CH_1  = 1U,	/**< PIS channel 1 triggers DAC channel conversion */
	MD_DAC_PIS_CH_2  = 2U,	/**< PIS channel 2 triggers DAC channel conversion */
	MD_DAC_PIS_CH_3  = 3U,	/**< PIS channel 3 triggers DAC channel conversion */
	MD_DAC_PIS_CH_4  = 4U,	/**< PIS channel 4 triggers DAC channel conversion */
	MD_DAC_PIS_CH_5  = 5U,	/**< PIS channel 5 triggers DAC channel conversion */
	MD_DAC_PIS_CH_6  = 6U,	/**< PIS channel 6 triggers DAC channel conversion */
	MD_DAC_PIS_CH_7  = 7U,	/**< PIS channel 7 triggers DAC channel conversion */
	MD_DAC_PIS_CH_8  = 8U,	/**< PIS channel 8 triggers DAC channel conversion */
	MD_DAC_PIS_CH_9  = 9U,	/**< PIS channel 9 triggers DAC channel conversion */
	MD_DAC_PIS_CH_10 = 10U,	/**< PIS channel 10 triggers DAC channel conversion */
	MD_DAC_PIS_CH_11 = 11U,	/**< PIS channel 11 triggers DAC channel conversion */
} md_dac_pissel_t;

/**
  * @brief DAC init structure definition
  */
typedef struct {
	md_dac_conv_mode_t conv_mode;	/**< Conversion mode */
	md_dac_out_mode_t out_mode;	/**< Select output mode */
	md_dac_refresh_t refresh;	/**< Refresh interval select */
	md_dac_prescale_div_t div;	/**< Prescaler setting */
	type_func_t ch0_reset;		/**< Select if prescaler is reset on channel 0 start */
	type_func_t o_ctrl_pis;		/**< Enable pis control of dac output enable */
	type_func_t sine;		/**< Sine mode enable/disable */
	type_func_t diff;		/**< Differential mode enable/disable */
	md_dac_neg_ref_t n_ref;		/**< The negative reference voltage select */
	md_dac_pos_ref_t p_ref;		/**< The positive reference voltage select */
} md_dac_init_t;
/**
  * @}
  */
/** @defgroup DAC_Public_Macros DAC Public Macros
  * @{
  */
#define MD_DAC_CH0_ENABLE()	(SET_BIT(DAC0->CH0CTRL, DAC_CH0CTRL_EN_MSK))
#define MD_DAC_CH1_ENABLE()	(SET_BIT(DAC0->CH1CTRL, DAC_CH1CTRL_EN_MSK))
#define MD_DAC_CH0_DISABLE()	(CLEAR_BIT(DAC0->CH0CTRL, DAC_CH0CTRL_EN_MSK))
#define MD_DAC_CH1_DISABLE()	(CLEAR_BIT(DAC0->CH1CTRL, DAC_CH1CTRL_EN_MSK))
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions DAC Public Functions
  * @{
  */
/** @defgroup MD_DAC_Public_Functions_Group2 CON
  * @{
  */
/**
  * @brief  Set diff mode
  * @param  DACx DAC Instance
  * @param  sel Value of the following:
  *         - 0x0:  diff mode disable
  *         - 0x1:  diff mode enable
  * @retval None
  */
__STATIC_INLINE void md_dac_set_diff_mode(DAC_TypeDef *DACx, uint32_t sel)
{
	MODIFY_REG(DACx->CON, DAC_CON_DIFEN_MSK, sel << DAC_CON_DIFEN_POS);
}

/**
  * @brief  Get diff mode selection
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0:  diff mode disable
  *         - 0x1:  diff mode enable
  */
__STATIC_INLINE uint32_t md_dac_get_diff_mode(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->CON, DAC_CON_DIFEN_MSK, DAC_CON_DIFEN_POS);
}

/**
  * @brief  Set sine mode
  * @param  DACx DAC Instance
  * @param  sel Value of the following:
  *         - 0x0:  diff mode disable
  *         - 0x1:  diff mode enable
  * @retval None
  */
__STATIC_INLINE void md_dac_set_sine_mode(DAC_TypeDef *DACx, uint32_t sel)
{
	MODIFY_REG(DACx->CON, DAC_CON_SINEMD_MSK, sel << DAC_CON_SINEMD_POS);
}

/**
  * @brief  Get sine mode selection
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0:  sine mode disable
  *         - 0x1:  sine mode enable
  */
__STATIC_INLINE uint32_t md_dac_get_sine_mode(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->CON, DAC_CON_SINEMD_MSK, DAC_CON_SINEMD_POS);
}

/**
  * @brief  Set convert mode
  * @param  DACx DAC Instance
  * @param  sel Value of the following:
  *         - DAC_CONV_MODE_CONTINUOUS:  Dac set in continuous mode
  *         - DAC_CONV_MODE_SAMPLEHOLD:  Dac set in sample/hold mode
  *         - DAC_CONV_MODE_SAMPLEOFF:   Dac set in sample/shut off mode
  * @retval None
  */
__STATIC_INLINE void md_dac_set_conv_mode(DAC_TypeDef *DACx, md_dac_conv_mode_t sel)
{
	MODIFY_REG(DACx->CON, DAC_CON_CONVMD_MSK, sel << DAC_CON_CONVMD_POSS);
}

/**
  * @brief  Set pis control out function
  * @param  DACx DAC Instance
  * @param  sel Value of the following:
  *         - 0:  Dac out enable forever
  *         - 1:  Dac out control by pis
  * @retval None
  */
__STATIC_INLINE void md_dac_set_pis_ctrl(DAC_TypeDef *DACx, uint32_t sel)
{
	MODIFY_REG(DACx->CON, DAC_CON_OUTENPIS_MSK, sel << DAC_CON_OUTENPIS_POS);
}

/**
  * @brief  Set CH0 prescale reset function
  * @param  DACx DAC Instance
  * @param  sel Value of the following:
  *         - 0:  prescale not reset channel
  *         - 1:  prescale reset channel
  * @retval None
  */
__STATIC_INLINE void md_dac_set_ch0prs_reset(DAC_TypeDef *DACx, uint32_t sel)
{
	MODIFY_REG(DACx->CON, DAC_CON_CH0PRESRST_MSK, sel << DAC_CON_CH0PRESRST_POS);
}

/**
  * @brief  Set prescale value
  * @param  DACx DAC Instance
  * @param  sel Value of the following:
  *         - 0:  prescale 1
  *         - 1:  prescale 2
  *          ......
  *         - 7:  prescale  128
  * @retval None
  */
__STATIC_INLINE void md_dac_set_prs(DAC_TypeDef *DACx, uint32_t sel)
{
	MODIFY_REG(DACx->CON, DAC_CON_PRES_MSK, sel << DAC_CON_PRES_POSS);
}

/**
  * @brief  Set refresh times
  * @param  DACx DAC Instance
  * @param  sel Value of the following:
  *         - DAC_REFRESH_8:  refresh period is 8
  *         - DAC_REFRESH_16:  refresh period is 8
  *         - DAC_REFRESH_32:  refresh period is 8
  *         - DAC_REFRESH_64:  refresh period is 8
  * @retval None
  */
__STATIC_INLINE void md_dac_set_ref_time(DAC_TypeDef *DACx, uint32_t sel)
{
	MODIFY_REG(DACx->CON, DAC_CON_RCYCLSEL_MSK, sel << DAC_CON_RCYCLSEL_POSS);
}
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group3 STAT
  * @{
  */
/**
  * @brief  Get Status of channel 0.
  * @param  DACx DAC Instance
  * @retval Status:
  *         - 0: CONVERTING
  *         - 1: COMPLETED
  */
__STATIC_INLINE uint32_t md_dac_get_stat_ch0(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->STAT, DAC_STAT_CH0BSY_MSK, DAC_STAT_CH0BSY_POS);
}

/**
  * @brief  Get Status of hannel 1.
  * @param  DACx DAC Instance
  * @retval Status:
  *         - 0: CONVERTING
  *         - 1: COMPLETED
  */
__STATIC_INLINE uint32_t md_adc_get_stat_ch1(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->STAT, DAC_STAT_CH1BSY_MSK, DAC_STAT_CH1BSY_POS);
}
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group4 CH0CTRL
  * @{
  */
/**
  * @brief  Choose pis trigger dac channel 0
  * @param  DACx DAC Instance
  * @param  sel value of the following:
  *         - DAC_PISSEL_CH_0:  pis channel 0 trigger dac channel 0
  *         - DAC_PISSEL_CH_1:  pis channel 1 trigger dac channel 0
  *         ......
  *         - DAC_PISSEL_CH_11:  pis channel 11 trigger dac channel 0
  * @retval None
  */
__STATIC_INLINE void md_dac_ch0_pis_sel(DAC_TypeDef *DACx, md_dac_pissel_t sel)
{
	MODIFY_REG(DACx->CH0CTRL, DAC_CH0CTRL_PISSEL_MSK, sel << DAC_CH0CTRL_PISSEL_POSS);
}
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group5 CH1CTRL
  * @{
  */
/**
  * @brief  Choose pis trigger dac channel 1
  * @param  DACx DAC Instance
  * @param  sel value of the following:
  *         - DAC_PISSEL_CH_0:  pis channel 0 trigger dac channel 0
  *         - DAC_PISSEL_CH_1:  pis channel 1 trigger dac channel 0
  *         ......
  *         - DAC_PISSEL_CH_11:  pis channel 11 trigger dac channel 0
  * @retval None
  */
__STATIC_INLINE void md_dac_ch1_pis_sel(DAC_TypeDef *DACx, md_dac_pissel_t sel)
{
	MODIFY_REG(DACx->CH1CTRL, DAC_CH1CTRL_PISSEL_MSK, sel << DAC_CH1CTRL_PISSEL_POSS);
}
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group6 IES
  * @{
  */
/**
  * @brief  channel 0 complete interrupt enable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch0_enable_complete_it(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IES, DAC_IES_CH0_MSK);
}

/**
  * @brief  channel 1 complete interrupt enable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch1_enable_complete_it(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IES, DAC_IES_CH1_MSK);
}

/**
  * @brief  channel 0 underflow interrupt enable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch0_enable_uf_it(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IES, DAC_IES_CH0UDF_MSK);
}

/**
  * @brief  channel 1 underflow interrupt enable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch1_enable_uf_it(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IES, DAC_IES_CH1UDF_MSK);
}

/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group7 IEC
  * @{
  */
/**
  * @brief  channel 0 complete interrupt disable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch0_disable_complete_it(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IEC, DAC_IEC_CH0_MSK);
}

/**
  * @brief  channel 1 complete interrupt disable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch1_disable_complete_it(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IEC, DAC_IEC_CH1_MSK);
}

/**
  * @brief  channel 0 underflow interrupt disable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch0_disable_uf_it(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IEC, DAC_IEC_CH0UDF_MSK);
}

/**
  * @brief  channel 1 underflow interrupt disable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch1_disable_uf_it(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IEC, DAC_IEC_CH1UDF_MSK);
}
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group8 IEV
  * @{
  */
/**
  * @brief  query channel 0 complete interrupt state
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0: Disable
  *         - 0x1: Enable
  */
__STATIC_INLINE uint32_t md_dac_ch0_get_complete_it(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->IEV, DAC_IEV_CH0_MSK, DAC_IEV_CH0_POS);
}

/**
  * @brief  query channel 1 complete interrupt state
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0: Disable
  *         - 0x1: Enable
  */
__STATIC_INLINE uint32_t md_dac_ch1_get_complete_it(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->IEV, DAC_IEV_CH1_MSK, DAC_IEV_CH1_POS);
}

/**
  * @brief  query channel 0 underflow interrupt state
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0: Disable
  *         - 0x1: Enable
  */
__STATIC_INLINE uint32_t md_dac_ch0_get_uf_it(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->IEV, DAC_IEV_CH0UDF_MSK, DAC_IEV_CH0UDF_POS);
}

/**
  * @brief  query channel 1 underflow interrupt state
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0: Disable
  *         - 0x1: Enable
  */
__STATIC_INLINE uint32_t md_dac_ch1_get_uf_it(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->IEV, DAC_IEV_CH1UDF_MSK, DAC_IEV_CH1UDF_POS);
}
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group9 RIF
  * @{
  */
/**
  * @brief  get channel 0 complete interrupt flag
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0: Disable
  *         - 0x1: Enable
  */
__STATIC_INLINE uint32_t md_dac_ch0_get_complete_flag(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->RIF, DAC_RIF_CH0_MSK, DAC_RIF_CH0_POS);
}

/**
  * @brief  get channel 1 complete interrupt flag
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0: Disable
  *         - 0x1: Enable
  */
__STATIC_INLINE uint32_t md_dac_ch1_get_complete_flag(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->RIF, DAC_RIF_CH1_MSK, DAC_RIF_CH1_POS);
}

/**
  * @brief  get channel 0 underflow interrupt flag
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0: Disable
  *         - 0x1: Enable
  */
__STATIC_INLINE uint32_t md_dac_ch0_get_uf_flag(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->RIF, DAC_RIF_CH0UDF_MSK, DAC_RIF_CH0UDF_POS);
}

/**
  * @brief  get channel 1 underflow interrupt flag
  * @param  DACx DAC Instance
  * @retval Value of the following:
  *         - 0x0: Disable
  *         - 0x1: Enable
  */
__STATIC_INLINE uint32_t md_dac_ch1_get_uf_flag(DAC_TypeDef *DACx)
{
	return READ_BITS(DACx->RIF, DAC_RIF_CH1UDF_MSK, DAC_RIF_CH1UDF_POS);
}
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group10 IFC
  * @{
  */
/**
  * @brief  clear channel 0 complete interrupt flag
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch0_clear_complete_flag(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IFC, DAC_IFC_CH0_MSK);
}

/**
  * @brief  clear channel 1 complete interrupt flag
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch1_clear_complete_flag(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IFC, DAC_IFC_CH1_MSK);
}

/**
  * @brief  clear channel 0 underflow interrupt flag
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch0_clear_uf_flag(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IFC, DAC_IFC_CH0UDF_MSK);
}

/**
  * @brief  clear channel 1 underflow interrupt flag
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_ch1_clear_uf_flag(DAC_TypeDef *DACx)
{
	SET_BIT(DACx->IFC, DAC_IFC_CH1UDF_MSK);
}
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group11 CH0DATA
  * @{
  */
/**
  * @brief  set channel 0 input data
  * @param  DACx DAC Instance
  * @param  value input data
  * @retval None
  */
__STATIC_INLINE void md_dac_ch0_set_data(DAC_TypeDef *DACx, uint32_t value)
{
	MODIFY_REG(DACx->CH0DATA, DAC_CH0DATA_VAL_MSK, value << DAC_CH0DATA_VAL_POSS);
}
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group12 CH1DATA
  * @{
  */
/**
  * @brief  set channel 1 input data
  * @param  DACx DAC Instance
  * @param  value input data
  * @retval None
  */
__STATIC_INLINE void md_dac_ch1_set_data(DAC_TypeDef *DACx, uint32_t value)
{
	MODIFY_REG(DACx->CH1DATA, DAC_CH1DATA_VAL_MSK, value << DAC_CH1DATA_VAL_POSS);
}

/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group1 Initialization
  * @{
  */
extern md_status_t md_dac_init(DAC_TypeDef *DACx, md_dac_init_t *init);
extern void md_dac_struct_init(md_dac_init_t *init);
/**
  * @}
  */
/**
  * @}
  */

/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif
#endif
