/**********************************************************************************
 *
 * @file    md_rcu.c
 * @brief   md_rcu C file
 *
 * @date    11 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          11 Aug 2022     Ginger         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "system_es32f0943.h"
#include "md_rcu.h"
#include "md_fc.h"
#include "md_tick.h"
/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup RCU RCU
  * @brief RCU micro driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
static uint32_t  TICKms;
static uint32_t  TICK100us;
static uint32_t  TICK10us;
/* Private constants ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
#define TICK_CLOCKSOURCE 16000000

/** @defgroup MD_RCU_Private_Macros RCU Private Macros
  * @{
  */

/**
  * @} MD_RCU_Private_Macros
  */

/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/
/** @addtogroup MD_RCU_Public_Functions RCU Public Functions
  * @{
  */
void  md_tick_init_rcu()
{
    TICKms = TICK_CLOCKSOURCE / 1000;
    TICK100us = TICK_CLOCKSOURCE / 10000;
    TICK10us = TICK_CLOCKSOURCE / 100000;

    md_tick_set_rvr_reload(TICK, ((1 << 24) - 1));            /*  Maximun ReLoad  */
    md_tick_set_cvr_current(TICK, 0);                         /*  Reset CVR  */
    md_tick_set_csr_clksrc(TICK, MD_SYSTICK_CLKSRC_HCLK);     /*  MCU Clock  */
    md_tick_enable_csr_enable(TICK);                          /*  Counter Enable  */
}

void  md_tick_waitms_rcu(uint8_t Unit, uint16_t msCnt)
{
    uint32_t  mstime;

    mstime = (1 << 24) - (Unit * TICKms);

    while (msCnt--)
    {
        md_tick_set_cvr_current(TICK, ((1 << 24) - 1)); /*  Reset CVR  */

        while (md_tick_get_cvr_current(TICK) > mstime);
    }
}

void  md_tick_wait100us_rcu(uint16_t Unit, uint16_t usCnt)
{
    uint32_t  ustime;

    ustime = (1 << 24) - (Unit * TICK100us);

    while (usCnt--)
    {
        md_tick_set_cvr_current(TICK, ((1 << 24) - 1)); /*  Reset CVR  */

        while (md_tick_get_cvr_current(TICK) > ustime);
    }
}

void  md_tick_wait10us_rcu(uint16_t Unit, uint16_t usCnt)
{
    uint32_t  ustime;

    ustime = (1 << 24) - (Unit * TICK10us);

    while (usCnt--)
    {
        md_tick_set_cvr_current(TICK, ((1 << 24) - 1)); /*  Reset CVR  */

        while (md_tick_get_cvr_current(TICK) > ustime);
    }
}

void md_rcu_check_hosc_ready()
{
    md_tick_init_rcu();

    md_tick_waitms_rcu(1, 1);

    while (md_rcu_is_active_flag_hosc_ready(RCU) == MD_RCU_HOSCRDY_NOT_READY)
    {
        md_rcu_disable_hosc(RCU);

        md_tick_waitms_rcu(1, 1);

        md_rcu_enable_hosc(RCU);

        md_tick_waitms_rcu(1, 1);

    }
}

/**
  * @} MD_RCU_Check_Hosc_Ready
  */

/** @addtogroup MD_RCU_Init
  * @{
  */
void md_rcu_sys_init(RCU_TypeDef *rcu, md_rcu_init_typedef *RCU_InitStruct)
{
    uint32_t    PLL_Frequency;

    md_fc_set_read_latency(FC, MD_FC_WAIT_MORE_THAN_72Mhz);

    if (RCU_InitStruct->HS_Clock & RCU_CON_PLLON)
        md_rcu_enable_pll(rcu);
    else
        md_rcu_disable_pll(rcu);

    if (RCU_InitStruct->HS_Clock & RCU_CON_HOSCON)
        md_rcu_enable_hosc(rcu);
    else
        md_rcu_disable_hosc(rcu);

    if (RCU_InitStruct->HS_Clock & RCU_CON_HRCON)
        md_rcu_enable_hrc(rcu);
    else
        md_rcu_disable_hrc(rcu);

    if (RCU_InitStruct->LS_Clock & RCU_LCON_LOSCON)
        md_rcu_enable_losc(rcu);
    else
        md_rcu_disable_losc(rcu);

    if (RCU_InitStruct->LS_Clock & RCU_LCON_LRCON)
        md_rcu_enable_lrc(rcu);
    else
        md_rcu_disable_lrc(rcu);

    //(Need to check)
    //make sure HOSC CLK Ready
    if ((RCU_InitStruct->HS_Clock & RCU_CON_HOSCON))
        md_rcu_check_hosc_ready();

    while (
        ((RCU->CON  & RCU_CON_PLLON) && (!md_rcu_is_active_flag_pll_ready(rcu)))
        || ((RCU->CON  & RCU_CON_HOSCON) && (!md_rcu_is_active_flag_hosc_ready(rcu)))
        || ((RCU->CON  & RCU_CON_HRCON) && (!md_rcu_is_active_flag_hrc_ready(rcu)))
        || ((RCU->LCON & RCU_LCON_LRCON) && (!md_rcu_is_active_flag_lrc_ready(rcu)))
        || ((RCU->LCON & RCU_LCON_LOSCON) && (!md_rcu_is_active_flag_losc_ready(rcu)))
    );

    md_rcu_set_mco_div(rcu, RCU_InitStruct->Mpre);
    md_rcu_set_mco_source(rcu, RCU_InitStruct->Msw);
    md_rcu_set_pclk_div(rcu, RCU_InitStruct->Ppre);
    md_rcu_set_hclk_div(rcu, RCU_InitStruct->Hpre);
    md_rcu_set_system_clock_source(rcu, RCU_InitStruct->Sw);

    if (RCU_InitStruct->Pllclk == MD_RCU_PLLFREQ_32MHz)
    {
        switch (RCU_InitStruct->PllSrc)
        {
            case MD_RCU_PLLSRC_HRC  :
                PLL_Frequency = 32000000UL;
                break;

            case MD_RCU_PLLSRC_HOSC :
                PLL_Frequency = 32000000UL;
                break;

            case MD_RCU_PLLSRC_LRC  :
                PLL_Frequency = (uint32_t)(__LRC) * 1024;
                break;

            case MD_RCU_PLLSRC_LOSC :
                PLL_Frequency = (uint32_t)(__LOSC) * 1024;
                break;
        }
    }
    else
    {
        switch (RCU_InitStruct->PllSrc)
        {
            case MD_RCU_PLLSRC_HRC  :
                PLL_Frequency = 48000000UL;
                break;

            case MD_RCU_PLLSRC_HOSC :
                PLL_Frequency = 48000000UL;
                break;

            case MD_RCU_PLLSRC_LRC  :
                PLL_Frequency = (uint32_t)(__LRC) * 1536;
                break;

            case MD_RCU_PLLSRC_LOSC :
                PLL_Frequency = (uint32_t)(__LOSC) * 1536;
                break;
        }
    }

    /* System Frequency */
    switch (md_rcu_get_current_system_clock(rcu)) /* System clock switch(SYSCLK) */
    {
        case MD_RCU_SWS_SYSCLK_HRC: /*================= HRC selected as system clock*/
            SystemCoreClock = (uint32_t)(__HRC);
            break;

        case MD_RCU_SWS_SYSCLK_HOSC: /*================= HOSC selected as system clock*/
            SystemCoreClock = (uint32_t)(__HOSC);
            break;

        case MD_RCU_SWS_SYSCLK_PLL: /*================= PLL selected as system clock*/
            SystemCoreClock = PLL_Frequency;
            break;

        case MD_RCU_SWS_SYSCLK_LRC: /*================= LRC selected as system clock*/
            SystemCoreClock = (__LRC);
            break;

        case MD_RCU_SWS_SYSCLK_LOSC: /*================= LOSC selected as system clock*/
            SystemCoreClock = (__LOSC);
            break;

        default:
            SystemCoreClock = (uint32_t)(__HRC);
            break;
    }

    /* Core Frequency */
    SystemFrequency_SysClk = SystemCoreClock;

    /* AHB Frequency */
    if ((md_rcu_get_hclk_div(rcu) >= 12))
        SystemFrequency_AHBClk = SystemCoreClock >> ((md_rcu_get_hclk_div(rcu) & 0x07) + 2);
    else if ((md_rcu_get_hclk_div(rcu)) >= 8)
        SystemFrequency_AHBClk = SystemCoreClock >> ((md_rcu_get_hclk_div(rcu) & 0x07) + 1);
    else
        SystemFrequency_AHBClk = SystemCoreClock;

    /* APB Frequency */
    if (md_rcu_get_pclk_div(rcu))
        SystemFrequency_APBClk = SystemFrequency_AHBClk >> ((md_rcu_get_pclk_div(rcu) & 0x03) + 1);
    else
        SystemFrequency_APBClk = SystemFrequency_AHBClk;

    if (RCU_InitStruct->HS_Clock & RCU_CON_HOSCCSSON)
        md_rcu_enable_hosc_css(rcu);
    else
        md_rcu_disable_hosc_css(rcu);

    if (RCU_InitStruct->LS_Clock & RCU_LCON_LOSCCSSON)
        md_rcu_enable_losc_css(rcu);
    else
        md_rcu_disable_losc_css(rcu);

    md_rcu_set_current_system_frequency(RCU, (SystemFrequency_SysClk / 1000000));

    if (SystemFrequency_AHBClk / 1000000 > 72)
        md_fc_set_read_latency(FC, MD_FC_WAIT_MORE_THAN_72Mhz);
    else if (SystemFrequency_AHBClk / 1000000 > 48)
        md_fc_set_read_latency(FC, MD_FC_WAIT_BETWEEN_48MHz_AND_72Mhz);
    else if (SystemFrequency_AHBClk / 1000000 > 24)
        md_fc_set_read_latency(FC, MD_FC_WAIT_BETWEEN_24MHz_AND_48Mhz);
    else
        md_fc_set_read_latency(FC, MD_FC_WAIT_LESS_THAN_24MHz);
}

/**
  * @} MD_RCU_Init
  */

/** @addtogroup MD_RCU_Pll_Init
  * @{
  */
void md_rcu_pll_init(RCU_TypeDef *rcu, md_rcu_init_typedef *RCU_InitStruct)
{
    md_tick_init_rcu();

    if ((RCU_InitStruct->Sw == MD_RCU_SW_SYSCLK_PLL) && (RCU_InitStruct->PllSrc == MD_RCU_PLLSRC_HRC))
        md_rcu_enable_hrc(rcu);
    else
        md_rcu_disable_hrc(rcu);

    if ((RCU_InitStruct->Sw == MD_RCU_SW_SYSCLK_PLL) && (RCU_InitStruct->PllSrc == MD_RCU_PLLSRC_HOSC))
        md_rcu_enable_hosc(rcu);
    else
        md_rcu_disable_hosc(rcu);

    if ((RCU_InitStruct->Sw == MD_RCU_SW_SYSCLK_PLL) && (RCU_InitStruct->PllSrc & MD_RCU_PLLSRC_LRC))
        md_rcu_enable_lrc(rcu);
    else
        md_rcu_disable_lrc(rcu);

    if ((RCU_InitStruct->Sw == MD_RCU_SW_SYSCLK_PLL) && (RCU_InitStruct->PllSrc & MD_RCU_PLLSRC_LOSC))
        md_rcu_enable_losc(rcu);
    else
        md_rcu_disable_losc(rcu);

    md_rcu_disable_pll(rcu);

    while (md_rcu_is_active_flag_pll_ready(rcu) == MD_RCU_PLLRDY_READY);

    md_rcu_set_pll_source(rcu, RCU_InitStruct->PllSrc);

    switch ((RCU_InitStruct->PllSrc))
    {
        case MD_RCU_PLLSRC_HRC :
            md_rcu_set_pll_prediv(RCU, ((__HRC / 4000000) - 1));
            md_rcu_set_pllrdycnt(RCU, 400);     // Hardware wait 100us (Use HRC/HOSC count)

            while (!(md_rcu_is_active_flag_hrc_ready(rcu)));

            break;

        case MD_RCU_PLLSRC_HOSC :
            md_rcu_set_pll_prediv(RCU, ((__HOSC / 4000000) - 1));
            md_rcu_set_pllrdycnt(RCU, 400);     // Hardware wait 100us (Use HRC/HOSC count)

            while (!(md_rcu_is_active_flag_hosc_ready(rcu)));

            break;

        case MD_RCU_PLLSRC_LRC :
            md_rcu_set_pll_prediv(RCU, MD_RCU_PREDIV_PLL_DIV1);
            md_rcu_set_pllrdycnt(RCU, 48);      // Hardware wait 1.5ms (Use LRC/LOSC count)

            while (!(md_rcu_is_active_flag_lrc_ready(rcu)));

            break;

        case MD_RCU_PLLSRC_LOSC :
            md_rcu_set_pll_prediv(RCU, MD_RCU_PREDIV_PLL_DIV1);
            md_rcu_set_pllrdycnt(RCU, 48);      // Hardware wait 1.5ms (Use LRC/LOSC count)

            while (!(md_rcu_is_active_flag_losc_ready(rcu)));

            break;

        default :
            md_rcu_set_pll_prediv(RCU, ((__HRC / 4000000) - 1));
            md_rcu_set_pllrdycnt(RCU, 400);     // Hardware wait 100us (Use HRC/HOSC count)

            while (!(md_rcu_is_active_flag_hrc_ready(rcu)));

            break;
    }

    md_rcu_set_pllfreq(RCU, RCU_InitStruct->Pllclk);

    md_rcu_enable_pll(rcu);

    if (!(md_rcu_get_pll_source(RCU) >> 1 & 0x1))
        md_tick_wait10us_rcu(1, 20);

    while (md_rcu_is_active_flag_pll_ready(rcu) == 0);
}

/**
  * @} MD_RCU_Pll0_Init
  */

/**
  * @} MD_RCU_Public_Functions
  */

/**
  * @} RCU
  */

/**
  * @} Micro_Driver
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
