/**********************************************************************************
 *
 * @file    md_adc.c
 * @brief   md_adc C file
 *
 * @date    11 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          11 Aug 2022     Ginger         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "md_adc.h"
#include "md_anpwr.h"
#include "md_tick.h"
#include "md_syscfg.h"
#include "md_rcu.h"
#include <stdio.h>
#include <math.h>
#include <string.h>

#define ADC_DMA

/** @addtogroup Micro_Driver
  * @{
  */


/** @defgroup ADC ADC
  * @brief ADC micro driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
#define PI acos(-1)
/* Private function prototypes ------------------------------------------------*/



/* Public functions -----------------------------------------------------------*/


void md_adc_power_init(RCU_TypeDef *rcu, ANPWR_TypeDef *anpwr, md_adc_power_init_typedef *power_init)
{
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);

    //enable anpwr
    md_rcu_enable_anpwr(rcu);

    md_anpwr_enable_vbg_bias(anpwr);
    md_tick_wait10us(1, 1);
    md_anpwr_set_vldo_mode(anpwr, power_init->anpwr_vldo_mode);
    md_anpwr_set_vldo_sel(anpwr, power_init->anpwr_vldo_sel);
    md_tick_wait10us(20, 1);

    md_rcu_set_adc_ckdiv(rcu, power_init->adc_clockdiv);
    md_rcu_set_adc_ckcfg(rcu, power_init->adc_clocksource);

    //chopper clock must be 4MHz
    md_rcu_set_chop_ckdiv(rcu, power_init->chop_clockdiv);
    md_rcu_set_chop_ckcfg(rcu, power_init->chop_clocksource);
}

void md_adc_reset(RCU_TypeDef *rcu)
{
    md_rcu_enable_adc_reset(rcu);
    md_rcu_disable_adc_reset(rcu);
    md_rcu_enable_adc(rcu);
}

void md_ia_power_init(ADC_TypeDef *ADCx, md_ia_init_typedef *ia_init)
{
    md_adc_set_ia_gain(ADCx, ia_init->ia_gain);
    md_adc_set_ia_chps(ADCx, ia_init->ia_chps);
    md_adc_set_ia_chpcks(ADCx, ia_init->ia_chpcks);

    if (ia_init->ia_lpfen)
        md_adc_enable_ia_lpfen(ADCx);
    else
        md_adc_disable_ia_lpfen(ADCx);

    md_adc_enable_ia_en(ADCx);
}

void md_adc_basic_init(ADC_TypeDef *ADCx, md_adc_init_typedef *adc_init)
{
    md_adc_set_current(ADCx, adc_init->adc_cursel);
    md_adc_set_vrrs(ADCx, adc_init->adc_vrrs);
    md_adc_set_ingn(ADCx, adc_init->adc_ingn);
    md_adc_set_chps(ADCx, adc_init->adc_chps);
    md_adc_set_clock(ADCx, adc_init->adc_cks);
    md_adc_set_osr(ADCx, adc_init->adc_osr);
    md_adc_set_cic_order(ADCx, adc_init->cic_order);
}

void md_adc_enable(ADC_TypeDef *ADCx)
{
    md_adc_clear_flag_adc_reset(ADCx);
    md_adc_enable_adc(ADCx);
    md_adc_clear_flag_cic_filter_reset(ADCx);
    md_adc_enable_cic_filter(ADCx);
}

void md_adc_basic_config(ADC_TypeDef *ADCx, md_adc_config_typedef *adc_config)
{
    md_adc_set_inps(ADCx, adc_config->adc_inps);
    md_adc_set_inns(ADCx, adc_config->adc_inns);
    md_adc_set_vrps(ADCx, adc_config->adc_vrps);
    md_adc_set_vrns(ADCx, adc_config->adc_vrns);
}

void md_adc_init(ADC_TypeDef *ADCx, md_adc_power_init_typedef *power_init, md_adc_init_typedef *adc_init, md_adc_config_typedef *adc_config)
{
    md_adc_power_init(RCU, ANPWR, power_init);
    md_adc_basic_init(ADCx, adc_init);
    md_adc_enable(ADCx);
    md_adc_basic_config(ADCx, adc_config);
}

void md_biquad_init(ADC_TypeDef *ADCx, md_biquad_init_typedef *biquad_init)
{
    md_adc_set_biquad_section(ADCx, biquad_init->section);
    md_adc_set_coef_sc_1(ADCx, biquad_init->coef_sc_1);
    md_adc_set_coef_b1_1(ADCx, biquad_init->coef_b1_1);
    md_adc_set_coef_b2_1(ADCx, biquad_init->coef_b2_1);
    md_adc_set_coef_b3_1(ADCx, biquad_init->coef_b3_1);
    md_adc_set_coef_a2_1(ADCx, biquad_init->coef_a2_1);
    md_adc_set_coef_a3_1(ADCx, biquad_init->coef_a3_1);
    md_adc_set_coef_sc_2(ADCx, biquad_init->coef_sc_2);
    md_adc_set_coef_b1_2(ADCx, biquad_init->coef_b1_2);
    md_adc_set_coef_b2_2(ADCx, biquad_init->coef_b2_2);
    md_adc_set_coef_b3_2(ADCx, biquad_init->coef_b3_2);
    md_adc_set_coef_a2_2(ADCx, biquad_init->coef_a2_2);
    md_adc_set_coef_a3_2(ADCx, biquad_init->coef_a3_2);
    md_adc_set_biquad_trigmode(ADCx, biquad_init->mode);
    md_adc_clear_flag_biquad_reset(ADCx);
    md_adc_enable_biqaud(ADCx);
}

void md_biquad_coefficient_calculate(uint32_t *coef, double num)
{
    //limit input protect
    if (num >= 16)
        num = 15.9999999f;
    else if (num < -16)
        num = -16.0f;


    num = num * (2 << 22); //num*2^23

    *coef = (int32_t)(num + 0.5); //num rounding than write in coef.

}

double md_biquad_coefficient_reverse_calculate(uint32_t coef)
{
    return ((double)((int)coef) / (double)(2 << 22));
}

void md_biquad_coefficient_determines(ADC_TypeDef *ADCx, uint32_t sampling_frequency, uint32_t cut_off_frequency, md_biquad_order_pass_filter order)
{
    md_biquad_init_typedef biquad_init;

    double c = (double)tan((double)PI * ((double)cut_off_frequency / (double)sampling_frequency));



    memset(&biquad_init, 0, sizeof(md_biquad_init_typedef));

    biquad_init.mode = MD_ADC_BIQUAD_TRIGMODE_SWTRIG;
    biquad_init.section = MD_ADC_BIQUAD_SECTION_1;

    if (order == MD_ADC_BIQUAD_1ST_ORDER_LOW_PASS_FILTER)
    {
        md_biquad_coefficient_calculate(&biquad_init.coef_b1_1, 1);
        md_biquad_coefficient_calculate(&biquad_init.coef_b2_1, 1);
        md_biquad_coefficient_calculate(&biquad_init.coef_b3_1, 0);
        md_biquad_coefficient_calculate(&biquad_init.coef_a2_1, ((c - 1) / (c + 1)));
        md_biquad_coefficient_calculate(&biquad_init.coef_a3_1, 0);
        md_biquad_coefficient_calculate(&biquad_init.coef_sc_1, c / (c + 1));
    }
    else if (order == MD_ADC_BIQUAD_2ST_ORDER_LOW_PASS_FILTER)
    {
        md_biquad_coefficient_calculate(&biquad_init.coef_b1_1, 1);
        md_biquad_coefficient_calculate(&biquad_init.coef_b2_1, 2);
        md_biquad_coefficient_calculate(&biquad_init.coef_b3_1, 1);
        md_biquad_coefficient_calculate(&biquad_init.coef_a2_1, ((2 * (pow(c, 2) - 1)) / (pow(c, 2) + sqrt(2)*c + 1)));
        md_biquad_coefficient_calculate(&biquad_init.coef_a3_1, ((pow(c, 2) - sqrt(2)*c + 1) / (pow(c, 2) + sqrt(2)*c + 1)));
        md_biquad_coefficient_calculate(&biquad_init.coef_sc_1, (pow(c, 2) / (pow(c, 2) + sqrt(2)*c + 1)));
    }
    else if (order == MD_ADC_BIQUAD_1ST_ORDER_HIGH_PASS_FILTER)
    {
        md_biquad_coefficient_calculate(&biquad_init.coef_b1_1, 1);
        md_biquad_coefficient_calculate(&biquad_init.coef_b2_1, -1);
        md_biquad_coefficient_calculate(&biquad_init.coef_b3_1, 0);
        md_biquad_coefficient_calculate(&biquad_init.coef_a2_1, ((c - 1) / (c + 1)));
        md_biquad_coefficient_calculate(&biquad_init.coef_a3_1, 0);
        md_biquad_coefficient_calculate(&biquad_init.coef_sc_1, 1 / (c + 1));
    }
    else if (order == MD_ADC_BIQUAD_2ST_ORDER_HIGH_PASS_FILTER)
    {
        md_biquad_coefficient_calculate(&biquad_init.coef_b1_1, 1);
        md_biquad_coefficient_calculate(&biquad_init.coef_b2_1, -2);
        md_biquad_coefficient_calculate(&biquad_init.coef_b3_1, 1);
        md_biquad_coefficient_calculate(&biquad_init.coef_a2_1, ((2 * (pow(c, 2) - 1)) / (pow(c, 2) + sqrt(2)*c + 1)));
        md_biquad_coefficient_calculate(&biquad_init.coef_a3_1, ((pow(c, 2) - sqrt(2)*c + 1) / (pow(c, 2) + sqrt(2)*c + 1)));
        md_biquad_coefficient_calculate(&biquad_init.coef_sc_1, (1 / (pow(c, 2) + sqrt(2)*c + 1)));
    }

    md_biquad_init(ADCx, &biquad_init);
}

/**
  * @} ADC_PUB_FUNC ADC Public Functions
  */

/**
  * @} ADC
  */

/**
  * @} Micro_Driver
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
