/**********************************************************************************
 *
 * @file    md_spi.h
 * @brief   header file of md_spi.c
 *
 * @date    5 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 Aug 2022      Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_SPI_H__
#define __MD_SPI_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include "es32f0943.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_SPI SPI
  * @brief SPI micro driver
  * @{
  */

/** @defgroup MD_SPI_Pubulic_Types SPI Pubulic Types
  * @{
  */

/**
  * @brief   SPI Init structure.
  */
typedef struct
{
    uint32_t Mode;                    /*!< Specifies the SPI mode (Master/Slave).
                                         This parameter can be a value of @ref MD_SPI_PC_MODE.

                                         This feature can be modified afterwards using unitary function @ref md_spi_enable_master_mode() and @ref md_spi_enable_slave_mode().*/

    uint32_t ClockPhase;              /*!< Specifies the clock active edge for the bit capture.
                                         This parameter can be a value of @ref MD_SPI_PC_PHASE.

                                         This feature can be modified afterwards using unitary function @ref md_spi_enable_sample_first_edge() and @ref md_spi_enable_sample_sec_edge().*/

    uint32_t ClockPolarity;           /*!< Specifies the serial clock steady state.
                                         This parameter can be a value of @ref MD_SPI_PC_POLARITY.

                                         This feature can be modified afterwards using unitary function @ref md_spi_enable_sck_low_idle() and @ref md_spi_enable_sck_high_idle().*/

    uint32_t BaudRate;                /*!< Specifies the BaudRate prescaler value which will be used to configure the transmit and receive SCK clock.
                                         This parameter can be a value of @ref MD_SPI_PC_BAUDRATEPRESCALER.
                                         @note The communication clock is derived from the master clock. The slave clock does not need to be set.

                                         This feature can be modified afterwards using unitary function @ref md_spi_set_commumication_baud().*/

    uint32_t BitOrder;                /*!< Specifies whether data transfers start from MSB or LSB bit.
                                         This parameter can be a value of @ref MD_SPI_PC_BIT_ORDER.

                                         This feature can be modified afterwards using unitary function @ref md_spi_enable_first_transmit_lsb() and @ref md_spi_enable_first_transmit_msb().*/

    uint32_t TransferDirection;       /*!< Specifies the SPI unidirectional or bidirectional data mode.
                                         This parameter can be a value of @ref MD_SPI_PC_TRANSFER_DIRECTION.

                                         This feature can be modified afterwards using unitary function @ref md_spi_set_simplex_line_duplex() and @ref md_spi_set_duplex_line_simplex().*/

    uint32_t DataWidth;               /*!< Specifies the SPI data width.
                                         This parameter can be a value of @ref MD_SPI_PC_FRAME_FORMAT.

                                         This feature can be modified afterwards using unitary function @ref md_spi_set_con1_dlen().*/

    uint32_t NSS;                     /*!< Specifies whether the NSS signal is managed by hardware (NSS pin) or by software using the SSI bit.
                                         This parameter can be a value of @ref MD_SPI_PC_NSS_MODE.

                                         This feature can be modified afterwards using unitary function @ref md_spi_enable_ss_output_high() and @ref md_spi_enable_ss_output_low().*/

    uint32_t CRCCalculation;          /*!< Specifies if the CRC calculation is enabled or not.
                                         This parameter can be a value of @ref MD_SPI_EC_CRC_CALCULATION.

                                         This feature can be modified afterwards using unitary functions @ref md_spi_enable_con1_crcen() and @ref md_spi_disable_con1_crcen().*/

    uint32_t CRCPoly;                 /*!< Specifies the polynomial used for the CRC calculation.
                                         This parameter must be a number between Min_Data = 0x00 and Max_Data = 0xFFFF.

                                         This feature can be modified afterwards using unitary function.*/

} md_spi_inittypedef, md_i2s_inittypedef;

typedef struct md_spi_handle_s
{
    SPI_TypeDef *SPI;
    md_spi_inittypedef init;
    uint8_t tx_size;
    uint8_t rx_size;
    uint8_t tx_count;
    uint8_t rx_count;
    uint8_t *tx_buf;
    uint8_t *rx_buf;
    uint32_t ifm_buf;
} md_spi_handle_t;
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Macros SPI Public Macros
  * @{
  */

/**
  * @brief SPI Operation Mode
  */
#define MD_SPI_MODE_SLAVE                       (0U)        /*!< Slave configuration   */
#define MD_SPI_MODE_MASTER                      (1U)        /*!< Master configuration  */

/**
  * @brief SPI Serial Protocol
  */
#define MD_SPI_PROTOCOL_MOTOROLA                (0U)        /*!< Motorola mode. Used as default value */
#define MD_SPI_PROTOCOL_TI                      (1U)        /*!< TI mode                              */

/**
  * @brief SPI Clock Phase
  */
#define MD_SPI_PHASE_1EDGE                      (0U)        /*!< First clock transition is the first data capture edge  */
#define MD_SPI_PHASE_2EDGE                      (1U)        /*!< Second clock transition is the first data capture edge */

/**
  * @brief SPI Clock Polarity
  */
#define MD_SPI_POLARITY_LOW                     (0U)        /*!< Clock to 0 when idle */
#define MD_SPI_POLARITY_HIGH                    (1U)        /*!< Clock to 1 when idle */

/**
  * @brief SPI Baud Rate Prescaler
  */
#define MD_SPI_BAUDRATEPRESCALER_DIV2           (0U)        /*!< BaudRate control equal to fPCLK/2   */
#define MD_SPI_BAUDRATEPRESCALER_DIV4           (1U)        /*!< BaudRate control equal to fPCLK/4   */
#define MD_SPI_BAUDRATEPRESCALER_DIV8           (2U)        /*!< BaudRate control equal to fPCLK/8   */
#define MD_SPI_BAUDRATEPRESCALER_DIV16          (3U)        /*!< BaudRate control equal to fPCLK/16   */
#define MD_SPI_BAUDRATEPRESCALER_DIV32          (4U)        /*!< BaudRate control equal to fPCLK/32  */
#define MD_SPI_BAUDRATEPRESCALER_DIV64          (5U)        /*!< BaudRate control equal to fPCLK/64  */
#define MD_SPI_BAUDRATEPRESCALER_DIV128         (6U)        /*!< BaudRate control equal to fPCLK/128 */
#define MD_SPI_BAUDRATEPRESCALER_DIV256         (7U)        /*!< BaudRate control equal to fPCLK/256 */

/**
  * @brief SPI Transmission Bit Order
  */
#define MD_SPI_LSB_FIRST                        (1U)        /*!< Data is transmitted/received with the LSB first */
#define MD_SPI_MSB_FIRST                        (0U)        /*!< Data is transmitted/received with the MSB first */

/**
  * @brief SPI Software control chip select output
  */
#define MD_SPI_NSS_0                            (0U)        /*!< NSS pin input is 0 */
#define MD_SPI_NSS_1                            (1U)        /*!< NSS pin input is 1 */

/**
  * @brief SPI Software control chip select output
  */
#define MD_SPI_TX_RX                            (0U)        /*!< Full duplex (transmit and receive) */
#define MD_SPI_RX_ONLY                          (1U)        /*!< Turn off output (receive mode only) */

/**
  * @brief SPI Transfer Mode
  */
#define MD_SPI_FRAME_FORMAT_8BIT                (0U)        /*!< 8 bits data format */
#define MD_SPI_FRAME_FORMAT_16BIT               (1U)        /*!< 16 bits data format    */

/**
  * @brief SPI transmission CRC Mode
  */
#define MD_SPI_CRC_NOT                          (0U)        /*!< Do not transmit CRC at the end of data transmission */
#define MD_SPI_CRC_TRANSMIT                     (1U)        /*!< CRC is transmitted at the end of data transmission    */

/**
  * @brief SPI Transfer Direction
  */
#define MD_SPI_FULL_DUPLEX                      (0U)        /*!< Full-Duplex mode. Rx and Tx transfer on 2 lines */
#define MD_SPI_HALF_DUPLEX                      (1U)        /*!< Half-Duplex Rx mode. Rx transfer on 1 line      */

/**
  * @brief SPI RX FIFO Threshold
  */
#define MD_SPI_RX_FIFO_1_CHARACTER_IN_THE_FIFO  (0U)        /*!< RXTHIE event is generated if FIFO level is greater than or equel to 1 character */
#define MD_SPI_RX_FIFO_QUARTER_FULL             (1U)        /*!< RXTHIE event is generated if FIFO level is greater than or equel to FIFO quarter full */
#define MD_SPI_RX_FIFO_HALF_FULL                (2U)        /*!< RXTHIE event is generated if FIFO level is greater than or equel to FIFO half full  */
#define MD_SPI_RX_FIFO_2_LESS_THAN_FULL         (3U)        /*!< RXTHIE event is generated if FIFO level is greater than or equel to FIFO 2 less than full  */

/**
  * @brief SPI TX FIFO Threshold
  */
#define MD_SPI_TX_FIFO_EMPTY                    (0U)        /*!< TXTHIE event is generated if FIFO level is equel to FIFO empty */
#define MD_SPI_TX_FIFO_2_CHARACTERS_IN_THE_FIFO (1U)        /*!< TXTHIE event is generated if FIFO level is below than or equel to 2 characters in the FIFO */
#define MD_SPI_TX_FIFO_QUARTER_FULL             (2U)        /*!< TXTHIE event is generated if FIFO level is below than or equel to FIFO quarter full */
#define MD_SPI_TX_FIFO_HALF_FULL                (3U)        /*!< TXTHIE event is generated if FIFO level is below than or equel to FIFO half full */

/**
  * @brief SPI Slave Select Pin Mode
  */
#define MD_SPI_NSS_SOFT                         (1U)        /*!< NSS managed internally  */
#define MD_SPI_NSS_HARD                         (0U)        /*!< NSS pin is controled by hardware */

/**
  * @brief SPI CRC Calculation
  */
#define MD_SPI_CRCCALCULATION_DISABLE           (0U)        /*!< CRC calculation disabled */
#define MD_SPI_CRCCALCULATION_ENABLE            (1U)        /*!< CRC calculation enabled  */

/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions SPI Public Functions
  * @{
  */

/** @defgroup MD_SPI_Public_Functions_Group1 CON1
  * @{
  */
/**
  * @brief  Set SPI control register (SPIx_CON1)
  * @param  SPIx SPI Instance
  * @param  value The value write in SPIx_CON1
  * @retval None
  */
__STATIC_INLINE void md_spi_set_con1(SPI_TypeDef *SPIx, uint32_t value)
{
    WRITE_REG(SPIx->CON1, value);
}

/**
  * @brief  Get SPI control register (SPIx_CON1)
  * @param  SPIx SPI Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_spi_get_con1(SPI_TypeDef *SPIx)
{
    return (uint32_t)(READ_REG(SPIx->CON1));
}

/**
  * @brief  Set spi duplex line simplex direction commumication.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_set_duplex_line_simplex(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_BIDEN);
}

/**
  * @brief  Set spi simplex line duplex direction commumication.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_set_simplex_line_duplex(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_BIDEN);
}

/**
  * @brief  Check if spi is Enabled duplex line simplex direction commumication.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_duplex_line_simplex(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_BIDEN) != SPI_CON1_BIDEN);
}

/**
  * @brief  Check if spi is Enabled simplex line duplex direction commumication
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_simplex_line_duplex(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_BIDEN) == SPI_CON1_BIDEN);
}

/**
  * @brief  Enable spi duplex line mode data transmit.
  * @param  SPIx: SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_simplex_line_tx(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_BIDOEN);
}

/**
  * @brief  Enable spi duplex line mode data receive.
  * @param  SPIx: SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_simplex_line_rx(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_BIDOEN);
}

/**
  * @brief  Check if spi is Enabled duplex line mode data transmit
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_simplex_line_tx(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_BIDOEN) == SPI_CON1_BIDOEN);
}

/**
  * @brief  Check if spi is Enabled duplex line mode data receive
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_simplex_line_rx(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_BIDOEN) != SPI_CON1_BIDOEN);
}

/**
  * @brief  Enable spi crc by hardware.
  * @param  SPIx: SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_crc_hardware(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_CRCEN);
}

/**
  * @brief  Disable spi crc by hardware.
  * @param  SPIx: SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_crc_hardware(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_CRCEN);
}

/**
  * @brief  Check if spi is enabled crc by hardware.
  * @param  SPIx: SPI Instance.
  * @retval State for 1 or 0.
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_crc_hardware(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_CRCEN) == SPI_CON1_CRCEN);
}

/**
  * @brief  Enable spi next time transmit is crc.
  * @param  SPIx: SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_next_crc_transmit(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_NXTCRC);
}

/**
  * @brief  Disable spi next time transmit is crc.
  * @param  SPIx: SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_next_crc_transmit(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_NXTCRC);
}

/**
  * @brief  Check if spi is enable next time transmit is crc.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_next_crc_transmit(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_NXTCRC) == SPI_CON1_NXTCRC);
}

/**
  * @brief  Enable spi 16 bytes commumication data .
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_16_byte_commumication(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_FLEN);
}

/**
  * @brief  Enable spi 8 bytes commumication data .
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_8_byte_commumication(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_FLEN);
}

/**
  * @brief  Check if spi is enabled 16 bytes commumication data.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_16_byte_commumication(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_FLEN) == SPI_CON1_FLEN);
}

/**
  * @brief  Check if spi is enabled 8 bytes commumication data.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_8_byte_commumication(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_FLEN) != SPI_CON1_FLEN);
}

/**
  * @brief  Enable spi only receive mode.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_recv_only(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_RXO);
}

/**
  * @brief  Disable spi only receive mode.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_disable_recv_only(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_RXO);
}

/**
  * @brief  Check if spi is enabled only receive mode.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint8_t md_spi_is_enabled_recv_only(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_RXO) == SPI_CON1_RXO);
}

/**
  * @brief  Enable spi control slave device by software.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_control_slave(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_SSEN);
}

/**
  * @brief  Disable spi control slave device by software.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_disable_control_slave(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_SSEN);
}

/**
  * @brief  Check if spi is enabled control slave device by software.
  * @param  SPIx: SPI Instance.
  * @retval State for bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_control_slave(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_SSEN) == SPI_CON1_SSEN);
}

/**
  * @brief  Enable spi chip select output high.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_ss_output_high(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_SSOUT);
}

/**
  * @brief  Enable spi chip select output low.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_ss_output_low(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_SSOUT);
}

/**
  * @brief  Check if spi is enabled chip select output high.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_ss_output_high(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_SSOUT) == SPI_CON1_SSOUT);
}

/**
  * @brief  Check if spi is enabled chip select output low.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_ss_output_low(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_SSOUT) != SPI_CON1_SSOUT);
}

/**
  * @brief  Enable spi data first transmit MSB.
  * @param  SPIx: SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_first_transmit_msb(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_LSBFST);
}

/**
  * @brief  Enable spi data first transmit LSB.
  * @param  SPIx: SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_first_transmit_lsb(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_LSBFST);
}

/**
  * @brief  Check if spi is enabled data first transmit LSB.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_first_transmit_lsb(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_LSBFST) == SPI_CON1_LSBFST);
}

/**
  * @brief  Check if spi is enabled data first transmit MSB.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_first_transmit_msb(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_LSBFST) != SPI_CON1_LSBFST);
}

/**
  * @brief  Enable SPI module.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_SPIEN);
}

/**
  * @brief  Disable SPI module.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_disable(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_SPIEN);
}

/**
  * @brief  Check if is enabled spi.
  * @param  SPIx: SPI Instance.
  * @retval State for bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_spi_is_enabled(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_SPIEN) == SPI_CON1_SPIEN);
}

/**
  * @brief  Set spi commumication baudrate.
  * @param  SPIx: SPI Instance.
  * @param  BaudRate This parameter can be one of the following values:
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV2
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV4
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV8
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV16
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV32
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV64
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV128
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV256
  * @retval None.
  */
__STATIC_INLINE void md_spi_set_commumication_baud(SPI_TypeDef *SPIx, uint8_t BaudRate)
{
    MODIFY_REG(SPIx->CON1, SPI_CON1_BAUD, BaudRate << SPI_CON1_BAUD_POSS);
}

/**
  * @brief  Get spi commumication baudrate.
  * @param  SPIx: SPI Instance.
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV2
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV4
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV8
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV16
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV32
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV64
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV128
  *         @arg @ref MD_SPI_BAUDRATEPRESCALER_DIV256
  */
__STATIC_INLINE uint32_t md_spi_get_commumication_baud(SPI_TypeDef *SPIx)
{
    return (uint32_t)(READ_BIT(SPIx->CON1, SPI_CON1_BAUD) >> SPI_CON1_BAUD_POSS);
}

/**
  * @brief  Enable spi master mode.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_master_mode(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_MSTREN);
}

/**
  * @brief  Enable spi slave mode.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_slave_mode(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_MSTREN);
}
/**
  * @brief  Check if spi is enabled master mode.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_master_mode(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_MSTREN) == SPI_CON1_MSTREN);
}

/**
  * @brief  Enable spi sck bus keep high polarity in idle mode.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_sck_high_idle(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_CPOL);
}

/**
  * @brief  Enable spi sck bus keep low polarity in idle mode.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_sck_low_idle(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_CPOL);
}

/**
  * @brief  Check if spi is enabled sck keep high in idle.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_sck_high_idle(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_CPOL) == SPI_CON1_CPOL);
}

/**
  * @brief  Check if spi is enabled sck keep low in idle.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_sck_low_idle(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_CPOL) != SPI_CON1_CPOL);
}

/**
  * @brief  Enable spi sample data in seconde time edge.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_sample_sec_edge(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON1, SPI_CON1_CPHA);
}

/**
  * @param  SPIx: SPI Instance.
  * @brief  Enable spi sample data in first time edge.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_sample_first_edge(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON1, SPI_CON1_CPHA);
}

/**
  * @brief  Check is spi is enabled sample data in seconde time edge.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (0 or 1).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_sample_sec_edge(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_CPHA) == SPI_CON1_CPHA);
}

/**
  * @brief  Check is spi is enabled sample data in first time edge.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (0 or 1).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_sample_first_edge(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON1, SPI_CON1_CPHA) != SPI_CON1_CPHA);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group2 CON2
  * @{
  */
/**
  * @brief  Set SPI control register 2 (SPIx_CON2)
  * @param  SPIx SPI Instance
  * @param  value The value write in SPIx_CON2
  * @retval None
  */
__STATIC_INLINE void md_spi_set_con2(SPI_TypeDef *SPIx, uint32_t value)
{
    WRITE_REG(SPIx->CON2, value);
}

/**
  * @brief  Get SPI control register 2(SPIx_CON2)
  * @param  SPIx SPI Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_spi_get_con2(SPI_TypeDef *SPIx)
{
    return (uint32_t)(READ_REG(SPIx->CON2));
}

/**
  * @brief  Set spi receive fifo threshold.
  * @param  SPIx: SPI Instance.
  * @param  Threshold This parameter can be one of the following values:
  *         @arg @ref MD_SPI_RX_FIFO_1_CHARACTER_IN_THE_FIFO
  *         @arg @ref MD_SPI_RX_FIFO_QUARTER_FULL
  *         @arg @ref MD_SPI_RX_FIFO_HALF_FULL
  *         @arg @ref MD_SPI_RX_FIFO_2_LESS_THAN_FULL
  * @retval None.
  */
__STATIC_INLINE void md_spi_set_rxfifo_threshold(SPI_TypeDef *SPIx, uint8_t Threshold)
{
    MODIFY_REG(SPIx->CON2, SPI_CON2_RXFTH, Threshold << SPI_CON2_RXFTH_POSS);
}

/**
  * @brief  Get spi receive fifo threshold.
  * @param  SPIx: SPI Instance.
  * @retval Returned value can be one of the following values:
  *         @arg @ref MD_SPI_RX_FIFO_1_CHARACTER_IN_THE_FIFO
  *         @arg @ref MD_SPI_RX_FIFO_QUARTER_FULL
  *         @arg @ref MD_SPI_RX_FIFO_HALF_FULL
  *         @arg @ref MD_SPI_RX_FIFO_2_LESS_THAN_FULL
  */
__STATIC_INLINE uint8_t md_spi_get_rxfifo_threshold(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON2, SPI_CON2_RXFTH) >> SPI_CON2_RXFTH_POSS);
}

/**
  * @brief  Set spi transmit fifo threshold.
  * @param  SPIx: SPI Instance.
  * @param  Threshold This parameter can be one of the following values:
  *         @arg @ref MD_SPI_TX_FIFO_EMPTY
  *         @arg @ref MD_SPI_TX_FIFO_2_CHARACTERS_IN_THE_FIFO
  *         @arg @ref MD_SPI_TX_FIFO_QUARTER_FULL
  *         @arg @ref MD_SPI_TX_FIFO_HALF_FULL
  * @retval None.
  */
__STATIC_INLINE void md_spi_set_txfifo_threshold(SPI_TypeDef *SPIx, uint8_t Threshold)
{
    MODIFY_REG(SPIx->CON2, SPI_CON2_TXFTH, Threshold << SPI_CON2_TXFTH_POSS);
}

/**
  * @brief  Get spi transmit fifo threshold.
  * @param  SPIx: SPI Instance.
  * @retval Returned value can be one of the following values:
  *         @arg @ref MD_SPI_TX_FIFO_EMPTY
  *         @arg @ref MD_SPI_TX_FIFO_2_CHARACTERS_IN_THE_FIFO
  *         @arg @ref MD_SPI_TX_FIFO_QUARTER_FULL
  *         @arg @ref MD_SPI_TX_FIFO_HALF_FULL
  */
__STATIC_INLINE uint8_t md_spi_get_txfifo_threshold(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON2, SPI_CON2_TXFTH) >> SPI_CON2_TXFTH_POSS);
}

/**
  * @brief  Set MOTO SPI mode.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_set_moto_mode(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON2, SPI_CON2_FRF);
}

/**
  * @brief  Set TI SPI mode.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_set_ti_mode(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON2, SPI_CON2_FRF);
}

/**
  * @brief  Get spi mode.
  * @param  SPIx: SPI Instance.
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_SPI_PROTOCOL_MOTOROLA
  *         @arg @ref MD_SPI_PROTOCOL_TI
  */
__STATIC_INLINE uint8_t md_spi_get_mode(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON2, SPI_CON2_FRF) >> SPI_CON2_FRF_POS);
}

/**
  * @brief  Produce spi NSS pulse management(master mode).
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_nss_pulse(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON2, SPI_CON2_NSSP);
}

/**
  * @brief  Stop spi NSS pulse management(master mode).
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_disable_nss_pulse(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON2, SPI_CON2_NSSP);
}

/**
  * @brief  Check if spi is enabled NSS pulse management(master mode).
  * @param  SPIx: SPI Instance.
  * @retval SPI mode(0: CLK Disable 1: CLK Enable).
  */
__STATIC_INLINE uint8_t md_spi_is_enabled_nss_pulse(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON2, SPI_CON2_NSSP) == SPI_CON2_NSSP);
}

/**
  * @brief  Enable nss output function.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_enable_nss_output(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->CON2, SPI_CON2_NSSOE);
}

/**
  * @brief  Disable nss output function.
  * @param  SPIx: SPI Instance.
  * @retval None.
  */
__STATIC_INLINE void md_spi_disable_nss_output(SPI_TypeDef *SPIx)
{
    CLEAR_BIT(SPIx->CON2, SPI_CON2_NSSOE);
}

/**
  * @brief  Check if spi is enabled nss output function.
  * @param  SPIx: SPI Instance.
  * @retval nss mode(0: Output Disable 1: Output Enable).
  */
__STATIC_INLINE uint8_t md_spi_is_enabled_nss_output(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->CON2, SPI_CON2_NSSOE) == SPI_CON2_NSSOE);
}

/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group3 STAT
  * @{
  */
/**
  * @brief  Get SPI status register(SPIx_STAT)
  * @param  SPIx SPI Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_spi_get_stat(SPI_TypeDef *SPIx)
{
    return (uint32_t)(READ_REG(SPIx->STAT));
}

/**
  * @brief  Get spi rxfifo value.
  * @param  SPIx: SPI Instance.
  * @retval rxfifo threshold.
  */
__STATIC_INLINE uint32_t md_spi_get_rxfifo_value(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_RXFLV) >> SPI_STAT_RXFLV_POSS);
}

/**
  * @brief  Get spi txfifo value.
  * @param  SPIx: SPI Instance.
  * @retval txfifo threshold.
  */
__STATIC_INLINE uint32_t md_spi_get_txfifo_value(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_TXFLV) >> SPI_STAT_TXFLV_POSS);
}

/**
  * @brief  Get spi busy flag.
  * @param  SPIx: SPI Instance.
  * @retval State for bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_busy(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_BUSY) == SPI_STAT_BUSY);
}

/**
  * @brief  Get spi rx threshold flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_rxth(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_RXTH) == SPI_STAT_RXTH);
}

/**
  * @brief  Get spi rx under flow flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_rxud(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_RXUD) == SPI_STAT_RXUD);
}

/**
  * @brief  Get spi rx over flow flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_rxov(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_RXOV) == SPI_STAT_RXOV);
}

/**
  * @brief  Get spi rx buffer full flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_rxf(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_RXF) == SPI_STAT_RXF);
}

/**
  * @brief  Get spi rx buffer not empty flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_rxne(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_RXNE) == SPI_STAT_RXNE);
}

/**
  * @brief  Get spi tx threshold flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_txth(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_TXTH) == SPI_STAT_TXTH);
}

/**
  * @brief  Get spi tx under flow flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_txud(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_TXUD) == SPI_STAT_TXUD);
}

/**
  * @brief  Get spi tx over flow flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_txov(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_TXOV) == SPI_STAT_TXOV);
}

/**
  * @brief  Get spi tx buffer full flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_txf(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_TXF) == SPI_STAT_TXF);
}

/**
  * @brief  Get spi tx buffer empty flag.
  * @param  SPIx: SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_flag_txe(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->STAT, SPI_STAT_TXE) == SPI_STAT_TXE);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group4 DATA
  * @{
  */
/**
  * @brief  Set spi data register value.
  * @param  SPIx: SPI Instance.
  * @param  data: tx data.
  * @retval None.
  */
__STATIC_INLINE void md_spi_set_data_reg(SPI_TypeDef *SPIx, uint32_t Data)
{
    WRITE_REG(SPIx->DATA, Data);
}

/**
  * @brief  Get spi data register value.
  * @param  SPIx: SPI Instance.
  * @retval data: value for data register.
  */
__STATIC_INLINE uint32_t md_spi_get_data_reg(SPI_TypeDef *SPIx)
{
    return READ_REG(SPIx->DATA);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group5 CRCPOLY
  * @{
  */
/**
  * @brief  Set spi crc register value.
  * @note   When CRC calculation is enabled, the RxCRC[15:0] bits contain the computed CRC value of the subsequently \n
            received bytes. This register is reset when the CRCEN bit in SPIx_CR1 register is written to 1. The CRC is \n
            calculated serially using the polynomial  programmed in the SPIx_CRCPLOY register. Only the 8 LSB bits are \n
            considered when the CRC frame format is set to be 8-bit length (CRCL bit in the SPIx_CON1 is cleared). CRC \n
            calculation is done based on any CRC8 standard. The entire 16-bits of this register are considered when a \n
            16-bit CRC frame format is selected (CRCL bit in the SPIx_CR1 register is set). CRC calculation is done based \n
            on any CRC16 standard.
            Note: A read to this register when the BSY Flag is set could return an incorrect value. These bits are not \n
            used in I2S mode.
  * @param  SPIx: SPI Instance.
  * @param  data: crc data.
  * @retval None.
  */
__STATIC_INLINE void md_spi_set_crc_polynomial(SPI_TypeDef *SPIx, uint32_t data)
{
    WRITE_REG(SPIx->CRCPOLY, data);
}

/**
  * @brief  Get spi crc register value.
  * @param  SPIx: SPI Instance.
  * @retval data: value for crc register.
  */
__STATIC_INLINE uint32_t md_spi_get_crc_polynomial(SPI_TypeDef *SPIx)
{
    return READ_REG(SPIx->CRCPOLY);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group6 RXCRC
  * @{
  */
/**
  * @brief  Get spi receive data crc value.
  * @note   When CRC calculation is enabled, the RxCRC[7:0] bits contain the computed CRC value of the subsequently \n
            transmitted bytes. This register is reset when the CRCEN bit of SPIx_CON1 is written to 1. The CRC is calculated \n
            serially using the polynomial programmed in the SPIx_CRCPLOY register. Only the 8 LSB bits are considered \n
            when the CRC frame format is set to be 8-bit length (FLEN bit in the SPIx_CON1 is cleared). CRC calculation is \n
            done based on any CRC8 standard. The entire 16-bits of this register are considered when a 16-bit CRC frame \n
            format is selected (CRCL bit in the SPIx_CR1 register is set). CRC calculation is done based on any CRC16 standard. \n
            Note: A read to this register when the BSY flag is set could return an incorrect value. These bits are not used \n
            in I2S mode.
  * @param  SPIx: SPI Instance.
  * @retval data: value for receive crc.
  */
__STATIC_INLINE uint32_t md_spi_get_rx_data_crc(SPI_TypeDef *SPIx)
{
    return READ_REG(SPIx->RXCRC);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group7 TXCRC
  * @{
  */
/**
  * @brief  Get spi transmit data crc value.
  * @note   When CRC calculation is enabled, the TxCRC[7:0] bits contain the computed CRC value of the subsequently \n
            transmitted bytes. This register is reset when the CRCEN bit of SPIx_CON1 is written to 1. The CRC is calculated \n
            serially using the polynomial programmed in the SPIx_CRCPLOY register. Only the 8 LSB bits are considered \n
            when the CRC frame format is set to be 8-bit length (FLEN bit in the SPIx_CR1 is cleared). CRC calculation is \n
            done based on any CRC8 standard. The entire 16-bits of this register are considered when a 16-bit CRC frame \n
            format is selected (CRCL bit in the SPIx_CR1 register is set). CRC calculation is done based on any CRC16 standard. \n
            Note: A read to this register when the BSY flag is set could return an incorrect value. These bits are not used \n
            in I2S mode.
  * @param  SPIx: SPI Instance.
  * @retval data: value for transmit crc.
  */
__STATIC_INLINE uint32_t md_spi_get_tx_data_crc(SPI_TypeDef *SPIx)
{
    return READ_REG(SPIx->TXCRC);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group8 IER
  * @{
  */
/**
  * @brief  Set SPI Interrupt enable Register
  * @param  SPIx SPI Instance
  * @param  ier
  * @retval None
  */
__STATIC_INLINE void md_spi_set_ier(SPI_TypeDef *SPIx, uint32_t ier)
{
    WRITE_REG(SPIx->IER, ier);
}

/**
  * @brief  Enable spi frame error interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_fre(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_FRE);
}

/**
  * @brief  Enable spi mode fail interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_modf(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_MODF);
}

/**
  * @brief  Enable spi crc error interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_crcerr(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_CRCERR);
}

/**
  * @brief  Enable spi rx threshold interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_rxth(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_RXTH);
}

/**
  * @brief  Enable spi rx under flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_rxud(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_RXUD);
}

/**
  * @brief  Enable spi rx over flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_rxov(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_RXOV);
}

/**
  * @brief  Enable spi rx full interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_rxf(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_RXF);
}

/**
  * @brief  Enable spi rx not empty interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_rxne(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_RXNE);
}

/**
  * @brief  Enable spi tx threshold interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_txth(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_TXTH);
}

/**
  * @brief  Enable spi tx under flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_txud(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_TXUD);
}

/**
  * @brief  Enable spi tx over flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_txov(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_TXOV);
}

/**
  * @brief  Enable spi tx empty interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_enable_it_txe(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IER, SPI_IER_TXE);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group9 IDR
  * @{
  */
/**
  * @brief  Set SPI interrupt disable Register
  * @param  SPIx SPI Instance
  * @retval None
  */
__STATIC_INLINE void md_spi_set_idr(SPI_TypeDef *SPIx, uint32_t idr)
{
    WRITE_REG(SPIx->IDR, idr);
}

/**
  * @brief  Disable spi frame error interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_fre(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_FRE);
}

/**
  * @brief  Disable spi mode fail interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_modf(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_MODF);
}

/**
  * @brief  Disable spi crc error interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_crcerr(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_CRCERR);
}

/**
  * @brief  Disable spi rx threshold interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_rxth(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_RXTH);
}

/**
  * @brief  Disable spi rx under flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_rxud(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_RXUD);
}

/**
  * @brief  Disable spi rx over flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_rxov(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_RXOV);
}

/**
  * @brief  Disable spi rx full interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_rxf(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_RXF);
}

/**
  * @brief  Disable spi rx not empty interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_rxne(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_RXNE);
}

/**
  * @brief  Disable spi tx threshold interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_txth(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_TXTH);
}

/**
  * @brief  Disable spi tx under flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_txud(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_TXUD);
}

/**
  * @brief  Disable spi tx over flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_txov(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_TXOV);
}

/**
  * @brief  Disable spi tx empty interrupt.
  * @param  SPIx SPI Instance.
  * @retval None
  */
__STATIC_INLINE void md_spi_disable_it_txe(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->IDR, SPI_IDR_TXE);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group10 IVS
  * @{
  */
/**
  * @brief  Get SPI_IVS Register
  * @param  SPIx SPI Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_spi_get_ivs(SPI_TypeDef *SPIx)
{
    return (uint32_t)(READ_REG(SPIx->IVS));
}

/**
  * @brief  Check if the frame error interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_fre(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_FRE) == (SPI_IVS_FRE));
}

/**
  * @brief  Check if the mode fail interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_modf(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_MODF) == (SPI_IVS_MODF));
}

/**
  * @brief  Check if the crc error interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_crcerr(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_CRCERR) == (SPI_IVS_CRCERR));
}

/**
  * @brief  Check if the rx threshold interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_rxth(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_RXTH) == (SPI_IVS_RXTH));
}

/**
  * @brief  Check if the rx under flow interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_rxud(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_RXUD) == (SPI_IVS_RXUD));
}

/**
  * @brief  Check if the rx over flow interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_rxov(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_RXOV) == (SPI_IVS_RXOV));
}

/**
  * @brief  Check if the rx full interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_rxf(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_RXF) == (SPI_IVS_RXF));
}

/**
  * @brief  Check if the rx not empty interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_rxne(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_RXNE) == (SPI_IVS_RXNE));
}

/**
  * @brief  Check if the tx threshold interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_txth(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_TXTH) == (SPI_IVS_TXTH));
}

/**
  * @brief  Check if the tx under flow interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_txud(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_TXUD) == (SPI_IVS_TXUD));
}

/**
  * @brief  Check if the tx over flow interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_txov(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_TXOV) == (SPI_IVS_TXOV));
}

/**
  * @brief  Check if the tx empty interrupt is enabled or disabled.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_enabled_it_txe(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IVS, SPI_IVS_TXE) == (SPI_IVS_TXE));
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group11 RIF
  * @{
  */
/**
  * @brief  Get SPI_RIF Register
  * @param  SPIx SPI Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_spi_get_rif(SPI_TypeDef *SPIx)
{
    return (uint32_t)(READ_REG(SPIx->RIF));
}

/**
  * @brief  Get frame error interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_fre(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_FRE) == (SPI_RIF_FRE));
}

/**
  * @brief  Get mode fail interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_modf(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_MODF) == (SPI_RIF_MODF));
}

/**
  * @brief  Get crc error interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_crcerr(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_CRCERR) == (SPI_RIF_CRCERR));
}

/**
  * @brief  Get rx threshold interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_rxth(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_RXTH) == (SPI_RIF_RXTH));
}

/**
  * @brief  Get rx under flow interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_rxud(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_RXUD) == (SPI_RIF_RXUD));
}

/**
  * @brief  Get rx over flow interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_rxov(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_RXOV) == (SPI_RIF_RXOV));
}

/**
  * @brief  Get rx full interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_rxf(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_RXF) == (SPI_RIF_RXF));
}

/**
  * @brief  Get rx not empty interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_rxne(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_RXNE) == (SPI_RIF_RXNE));
}

/**
  * @brief  Get tx threshold interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_txth(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_TXTH) == (SPI_RIF_TXTH));
}

/**
  * @brief  Get tx under flow interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_txud(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_TXUD) == (SPI_RIF_TXUD));
}

/**
  * @brief  Get tx over flow interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_txov(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_TXOV) == (SPI_RIF_TXOV));
}

/**
  * @brief  Get tx empty interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_active_it_txe(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->RIF, SPI_RIF_TXE) == (SPI_RIF_TXE));
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group12 IFM
  * @{
  */
/**
  * @brief  Get SPI_IFM Register
  * @param  SPIx SPI Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_spi_get_ifm(SPI_TypeDef *SPIx)
{
    return (uint32_t)(READ_REG(SPIx->IFM));
}

/**
  * @brief  Mask frame error interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_fre(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_FRE) == SPI_IFM_FRE);
}

/**
  * @brief  Mask mode fail interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_modf(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_MODF) == SPI_IFM_MODF);
}

/**
  * @brief  Mask crc error interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_crcerr(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_CRCERR) == SPI_IFM_CRCERR);
}

/**
  * @brief  Mask rx threshold interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_rxth(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_RXTH) == SPI_IFM_RXTH);
}

/**
  * @brief  Mask rx under flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_rxud(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_RXUD) == SPI_IFM_RXUD);
}

/**
  * @brief  Mask rx over flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_rxov(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_RXOV) == SPI_IFM_RXOV);
}

/**
  * @brief  Mask rx full interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_rxf(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_RXF) == SPI_IFM_RXF);
}

/**
  * @brief  Mask rx not empty interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_rxne(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_RXNE) == SPI_IFM_RXNE);
}

/**
  * @brief  Mask tx threshold interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_txth(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_TXTH) == SPI_IFM_TXTH);
}

/**
  * @brief  Mask tx under flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_txud(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_TXUD) == SPI_IFM_TXUD);
}

/**
  * @brief  Mask tx over flow interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_txov(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_TXOV) == SPI_IFM_TXOV);
}

/**
  * @brief  Mask tx empty interrupt.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_spi_is_masked_it_txe(SPI_TypeDef *SPIx)
{
    return (READ_BIT(SPIx->IFM, SPI_IFM_TXE) == SPI_IFM_TXE);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group13 ICR
  * @{
  */
/**
  * @brief  Set SPI_ICR Register
  * @param  SPIx SPI Instance
  * @retval None
  */
__STATIC_INLINE void md_spi_set_icr(SPI_TypeDef *SPIx, uint32_t icr)
{
    WRITE_REG(SPIx->ICR, icr);
}

/**
  * @brief  Clear frame error interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_fre(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_FRE);
}

/**
  * @brief  Clear mode fail interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_modf(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_MODF);
}

/**
  * @brief  Clear crc error interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_crcerr(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_CRCERR);
}

/**
  * @brief  Clear rx threshold interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_rxth(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_RXTH);
}

/**
  * @brief  Clear rx under flow interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_rxud(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_RXUD);
}

/**
  * @brief  Clear rx over flow interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_rxov(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_RXOV);
}

/**
  * @brief  Clear rx full interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_rxf(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_RXF);
}

/**
  * @brief  Clear rx not empty interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_rxne(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_RXNE);
}

/**
  * @brief  Clear tx threshold interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_txth(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_TXTH);
}

/**
  * @brief  Clear tx under flow interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_txud(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_TXUD);
}

/**
  * @brief  Clear tx over flow interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_txov(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_TXOV);
}

/**
  * @brief  Clear tx empty interrupt flag.
  * @param  SPIx SPI Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_spi_clear_it_txe(SPI_TypeDef *SPIx)
{
    SET_BIT(SPIx->ICR, SPI_ICR_TXE);
}
/**
  * @}
  */

/** @defgroup MD_SPI_Public_Functions_Group11 Initialization
  * @{
  */
ErrorStatus md_spi_init(SPI_TypeDef *SPIx, md_spi_inittypedef *SPI_InitStruct);
ErrorStatus md_spi_deinit(SPI_TypeDef *SPIx);
void md_spi_struct_init(md_spi_inittypedef *SPI_InitStruct);
uint8_t md_spi_recv_byte(SPI_TypeDef *SPIx);
void md_spi_send_byte(SPI_TypeDef *SPIx, uint8_t data);
uint16_t md_spi_recv_halfword(SPI_TypeDef *SPIx);
void md_spi_send_halfword(SPI_TypeDef *SPIx, uint16_t data);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}  Micro_Driver
  */

#ifdef __cplusplus
}
#endif

#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
