/**********************************************************************************
 *
 * @file    md_rcu.h
 * @brief   header file of md_rcu.c
 *
 * @date    21 Jul 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          21 Jul 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_RCU_H__
#define __MD_RCU_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include <stdbool.h>
#include "es32f0943.h"

/** @addtogroup Micro_Driver
  * @{
  */


/** @defgroup MD_RCU RCU
  * @brief RCU micro driver
  * @{
  */

/** @defgroup MD_RCU_Pubulic_Types RCU Pubulic Types
  * @{
  */
/**
  * @brief MD_RCU_Public_Types RCU Init Type
  */

typedef struct
{
    uint32_t  Mpre;        /*! MCO clock prescaler  */
    uint32_t  Msw;         /*! MCO clock output  */
    uint32_t  PllSrc;      /*! PLL clock source  */
    uint32_t  Pllclk;      /*! PLL clock   */
    uint32_t  Ppre;        /*! APB clock prescaler  */
    uint32_t  Hpre;        /*! AHB clock prescaler  */
    uint32_t  Sw;          /*! system clock   */
    uint32_t  HS_Clock;    /*! high spped clock enable  */
    uint32_t  LS_Clock;    /*! low spped clock enable  */
} md_rcu_init_typedef;

/**
  * @}
  */
/** @defgroup MD_RCU_Public_Macros RCU Public Macros
  * @{
  */

#ifndef __HOSC
#define __HOSC          (8000000UL)
#endif
#define __HRC           (16000000UL)
#define __LOSC          (   32768UL)
#define __LRC           (   32000UL)

/**
  * @brief MD_RCU_LOSCRDY_FLAG LOSC clock ready flag
  */
#define MD_RCU_LOSCRDY_NOT_READY    (0x00000000UL)  /** @brief LOSC oscillator not ready */
#define MD_RCU_LOSCRDY_READY        (0x00000001UL)  /** @brief LOSC oscillator ready */

/**
  * @brief MD_RCU_LRCRDY_FLAG LRC clock ready flag
  */
#define MD_RCU_LRCRDY_NOT_READY     (0x00000000UL)  /** @brief LRC oscillator not ready */
#define MD_RCU_LRCRDY_READY         (0x00000001UL)  /** @brief LRC oscillator ready */

/**
  * @brief MD_RCU_PLL0_FLAG PLL0 clock ready flag
  */
#define MD_RCU_PLLRDY_NOT_READY    (0x00000000UL)  /** @brief PLL not ready */
#define MD_RCU_PLLRDY_READY        (0x00000001UL)  /** @brief PLL ready */

/**
  * @brief MD_RCU_HOSCRDY_FLAG HOSC clock ready flag
  */
#define MD_RCU_HOSCRDY_NOT_READY    (0x00000000UL)  /** @brief HOSC oscillator not ready */
#define MD_RCU_HOSCRDY_READY        (0x00000001UL)  /** @brief HOSC oscillator ready */

/**
  * @brief MD_RCU_HRCRDY_FLAG HRC clock ready flag
  */
#define MD_RCU_HRCRDY_NOT_READY     (0x00000000UL)  /** @brief HRC oscillator not ready */
#define MD_RCU_HRCRDY_READY         (0x00000001UL)  /** @brief HRC oscillator ready */

/**
  * @brief MD_RCU_MCO_DIV Microcontroller clock output select
  */
#define MD_RCU_MPRE_MCO_DIV1        (0x00000000UL)  /** @brief MCO DIV = 1  */
#define MD_RCU_MPRE_MCO_DIV2        (0x00000001UL)  /** @brief MCO DIV = 2  */
#define MD_RCU_MPRE_MCO_DIV4        (0x00000002UL)  /** @brief MCO DIV = 4  */
#define MD_RCU_MPRE_MCO_DIV8        (0x00000003UL)  /** @brief MCO DIV = 8  */
#define MD_RCU_MPRE_MCO_DIV16       (0x00000004UL)  /** @brief MCO DIV = 16 */
#define MD_RCU_MPRE_MCO_DIV32       (0x00000005UL)  /** @brief MCO DIV = 32 */
#define MD_RCU_MPRE_MCO_DIV64       (0x00000006UL)  /** @brief MCO DIV = 64 */
#define MD_RCU_MPRE_MCO_DIV128      (0x00000007UL)  /** @brief MCO DIV = 128*/

/**
  * @brief MD_RCU_MCO_OUTPUT Microcontroller clock output select
  */
#define MD_RCU_MSW_MCO_DISABLE      (0x00000000UL)  /** @brief MCO output disabled, no clock on MCO */
#define MD_RCU_MSW_MCO_LRC          (0x00000001UL)  /** @brief Internal low speed (LRC) oscillator clock selected */
#define MD_RCU_MSW_MCO_LOSC         (0x00000002UL)  /** @brief External low speed (LOSC) oscillator clock selected */
#define MD_RCU_MSW_MCO_HRC          (0x00000003UL)  /** @brief Internal high speed (HRC) oscillator clock selected */
#define MD_RCU_MSW_MCO_HOSC         (0x00000004UL)  /** @brief External high speed (HOSC) oscillator clock selected */
#define MD_RCU_MSW_MCO_SYSCLK       (0x00000008UL)  /** @brief System clock selected(SYSCLK) */
#define MD_RCU_MSW_MCO_HCLK         (0x00000009UL)  /** @brief AHB clock selected(HCLK) */
#define MD_RCU_MSW_MCO_PCLK         (0x0000000AUL)  /** @brief APB clock selected(PCLK) */
#define MD_RCU_MSW_MCO_ADC          (0x0000000BUL)  /** @brief ADC clock selected(ADCCLK) */
#define MD_RCU_MSW_MCO_CHOPPER      (0x0000000CUL)  /** @brief CHOP clock selected(CHOPCLK) */

/**
  * @brief MD_RCU_PLLSRC PLL reference clock source
  */
#define MD_RCU_PLLSRC_HRC           (0x00000000UL)  /** @brief HRC/PREDIV selected as PLL reference clock */
#define MD_RCU_PLLSRC_HOSC          (0x00000001UL)  /** @brief HOSC/PREDIV selected as PLL reference clock */
#define MD_RCU_PLLSRC_LRC           (0x00000002UL)  /** @brief LRC/PREDIV selected as PLL reference clock */
#define MD_RCU_PLLSRC_LOSC          (0x00000003UL)  /** @brief LOSC/PREDIV selected as PLL reference clock */

/**
  * @brief MD_RCU_PLL_DIV Microcontroller clock output select
  */
#define MD_RCU_PREDIV_PLL_DIV1      (0x00000000UL)  /** @brief PLL DIV = 1 */
#define MD_RCU_PREDIV_PLL_DIV2      (0x00000001UL)  /** @brief PLL DIV = 2 */
#define MD_RCU_PREDIV_PLL_DIV3      (0x00000002UL)  /** @brief PLL DIV = 3 */
#define MD_RCU_PREDIV_PLL_DIV4      (0x00000003UL)  /** @brief PLL DIV = 4 */
#define MD_RCU_PREDIV_PLL_DIV5      (0x00000004UL)  /** @brief PLL DIV = 5 */
#define MD_RCU_PREDIV_PLL_DIV6      (0x00000005UL)  /** @brief PLL DIV = 6 */
#define MD_RCU_PREDIV_PLL_DIV7      (0x00000006UL)  /** @brief PLL DIV = 7 */
#define MD_RCU_PREDIV_PLL_DIV8      (0x00000007UL)  /** @brief PLL DIV = 8 */
#define MD_RCU_PREDIV_PLL_DIV9      (0x00000008UL)  /** @brief PLL DIV = 9 */
#define MD_RCU_PREDIV_PLL_DIV10     (0x00000009UL)  /** @brief PLL DIV = 10*/
#define MD_RCU_PREDIV_PLL_DIV11     (0x0000000AUL)  /** @brief PLL DIV = 11*/
#define MD_RCU_PREDIV_PLL_DIV12     (0x0000000BUL)  /** @brief PLL DIV = 12*/
#define MD_RCU_PREDIV_PLL_DIV13     (0x0000000CUL)  /** @brief PLL DIV = 13*/
#define MD_RCU_PREDIV_PLL_DIV14     (0x0000000DUL)  /** @brief PLL DIV = 14*/
#define MD_RCU_PREDIV_PLL_DIV15     (0x0000000EUL)  /** @brief PLL DIV = 15*/
#define MD_RCU_PREDIV_PLL_DIV16     (0x0000000FUL)  /** @brief PLL DIV = 16*/

/**
  * @brief MD_RCU_APB_CLK_DIV APB clock prescaler(PCLK)
  */
#define MD_RCU_PPRE_HCLK_DIV_1    (0x00000000UL)  /** @brief HCLK not divided */
#define MD_RCU_PPRE_HCLK_DIV_2    (0x00000004UL)  /** @brief HCLK divided by 2 */
#define MD_RCU_PPRE_HCLK_DIV_4    (0x00000005UL)  /** @brief HCLK divided by 4 */
#define MD_RCU_PPRE_HCLK_DIV_8    (0x00000006UL)  /** @brief HCLK divided by 8 */
#define MD_RCU_PPRE_HCLK_DIV_16   (0x00000007UL)  /** @brief HCLK divided by 16 */


/**
  * @brief MD_RCU_AHB_CLK_DIV AHB clock prescaler(HCLK)
  */
#define MD_RCU_HPRE_SYSCLK_DIV_1    (0x00000000UL)  /** @brief SYSCLK not divided    */
#define MD_RCU_HPRE_SYSCLK_DIV_2    (0x00000008UL)  /** @brief SYSCLK divided by 2   */
#define MD_RCU_HPRE_SYSCLK_DIV_4    (0x00000009UL)  /** @brief SYSCLK divided by 4   */
#define MD_RCU_HPRE_SYSCLK_DIV_8    (0x0000000AUL)  /** @brief SYSCLK divided by 8   */
#define MD_RCU_HPRE_SYSCLK_DIV_16   (0x0000000BUL)  /** @brief SYSCLK divided by 16  */
#define MD_RCU_HPRE_SYSCLK_DIV_64   (0x0000000CUL)  /** @brief SYSCLK divided by 64  */
#define MD_RCU_HPRE_SYSCLK_DIV_128  (0x0000000DUL)  /** @brief SYSCLK divided by 128 */
#define MD_RCU_HPRE_SYSCLK_DIV_256  (0x0000000EUL)  /** @brief SYSCLK divided by 256 */
#define MD_RCU_HPRE_SYSCLK_DIV_512  (0x0000000FUL)  /** @brief SYSCLK divided by 512 */

/**
  * @brief MD_RCU_SYS_CLK_SWS System clock switch(SYSCLK)
  */
#define MD_RCU_SWS_SYSCLK_HRC       (0x00000000UL)  /** @brief HRC selected as system clock   */
#define MD_RCU_SWS_SYSCLK_HOSC      (0x00000001UL)  /** @brief HOSC selected as system clock  */
#define MD_RCU_SWS_SYSCLK_PLL       (0x00000002UL)  /** @brief PLL selected as system clock  */
#define MD_RCU_SWS_SYSCLK_LRC       (0x00000003UL)  /** @brief LRC selected as system clock */
#define MD_RCU_SWS_SYSCLK_LOSC      (0x00000004UL)  /** @brief LOSC selected as system clock */

/**
  * @brief MD_RCU_SYS_CLK_SW System clock switch(SYSCLK)
  */
#define MD_RCU_SW_SYSCLK_HRC        (0x00000000UL)  /** @brief HRC selected as system clock   */
#define MD_RCU_SW_SYSCLK_HOSC       (0x00000001UL)  /** @brief HOSC selected as system clock  */
#define MD_RCU_SW_SYSCLK_PLL        (0x00000002UL)  /** @brief PLL selected as system clock  */
#define MD_RCU_SW_SYSCLK_LRC        (0x00000003UL)  /** @brief LRC selected as system clock */
#define MD_RCU_SW_SYSCLK_LOSC       (0x00000004UL)  /** @brief LOSC selected as system clock */

/**
  * @brief MD_RCU_CFG1_PLLFREQ
  */
#define MD_RCU_PLLFREQ_32MHz        (0x00000000UL)  /** @brief PLL clock frequency 32MHz  */
#define MD_RCU_PLLFREQ_48MHz        (0x00000001UL)  /** @brief PLL clock frequency 48MHz  */

/**
  * @brief MD_RCU_CFG2_LCD_CKDIV
  */
#define MD_RCU_CFG2_LCD_DIV2        (0x00000000UL)  /** @brief LCD clock divide by 2  */
#define MD_RCU_CFG2_LCD_DIV4        (0x00000001UL)  /** @brief LCD clock divide by 4  */
#define MD_RCU_CFG2_LCD_DIV6        (0x00000002UL)  /** @brief LCD clock divide by 6  */
#define MD_RCU_CFG2_LCD_DIV8        (0x00000003UL)  /** @brief LCD clock divide by 8  */

/**
  * @brief MD_RCU_CFG2_CHOP_CKCFG
  */
#define MD_RCU_CFG2_CHOPCK_DISABLE  (0x00000000UL)  /** @brief Chopper clock divide by 2  */
#define MD_RCU_CFG2_CHOPCK_HOSC     (0x00000001UL)  /** @brief Chopper clock divide by 4  */
#define MD_RCU_CFG2_CHOPCK_HRC      (0x00000002UL)  /** @brief Chopper clock divide by 6  */
#define MD_RCU_CFG2_CHOPCK_PLL      (0x00000003UL)  /** @brief Chopper clock divide by 8  */

/**
  * @brief MD_RCU_CFG2_ADC_CKCFG
  */
#define MD_RCU_CFG2_ADCCK_DISABLE   (0x00000000UL)  /** @brief ADC clock Disable  */
#define MD_RCU_CFG2_ADCCK_HOSC      (0x00000001UL)  /** @brief ADC clock Select HOSC clock  */
#define MD_RCU_CFG2_ADCCK_HRC       (0x00000002UL)  /** @brief ADC clock Select HRC clock  */
#define MD_RCU_CFG2_ADCCK_PLL       (0x00000003UL)  /** @brief ADC clock Select PLL clock  */

/**
  * @brief MD_RCU_CFG2_LCD_CKCFG
  */
#define MD_RCU_CFG2_LCDCK_DISABLE   (0x00000000UL)  /** @brief LCD clock Disable  */
#define MD_RCU_CFG2_LCDCK_LRC       (0x00000001UL)  /** @brief LCD clock Select LRC clock  */
#define MD_RCU_CFG2_LCDCK_LOSC      (0x00000002UL)  /** @brief LCD clock Select LOSC clock  */

/**
  * @brief MD_RCU_CFG2_VLCD_CKCFG
  */
#define MD_RCU_CFG2_VLCDCK_DISABLE  (0x00000000UL)  /** @brief VLCD clock Disable  */
#define MD_RCU_CFG2_VLCDCK_HOSC     (0x00000001UL)  /** @brief VLCD clock Select HOSC clock  */
#define MD_RCU_CFG2_VLCDCK_HRC      (0x00000002UL)  /** @brief VLCD clock Select HRC clock  */
#define MD_RCU_CFG2_VLCDCK_PLL      (0x00000003UL)  /** @brief VLCD clock Select PLL clock  */

/**
  * @brief MD_RCU_LCON_LOSCITUNE
  */
#define MD_RCU_LCON_LOSCITUNE_180nA     (0x00000000UL)  /** @brief LOSC clock current adjustment 180nA */
#define MD_RCU_LCON_LOSCITUNE_240nA     (0x00000001UL)  /** @brief LOSC clock current adjustment 240nA */
#define MD_RCU_LCON_LOSCITUNE_300nA     (0x00000002UL)  /** @brief LOSC clock current adjustment 300nA */
#define MD_RCU_LCON_LOSCITUNE_360nA     (0x00000003UL)  /** @brief LOSC clock current adjustment 360nA */
#define MD_RCU_LCON_LOSCITUNE_420nA     (0x00000004UL)  /** @brief LOSC clock current adjustment 420nA */
#define MD_RCU_LCON_LOSCITUNE_480nA     (0x00000005UL)  /** @brief LOSC clock current adjustment 480nA */
#define MD_RCU_LCON_LOSCITUNE_540nA     (0x00000006UL)  /** @brief LOSC clock current adjustment 540nA */
#define MD_RCU_LCON_LOSCITUNE_600nA     (0x00000007UL)  /** @brief LOSC clock current adjustment 600nA */

#define MD_RCU_HRCON    RCU_CON_HRCON       /** @brief HRC Enable */
#define MD_RCU_HOSCON   RCU_CON_HOSCON      /** @brief HOSC Enable */
#define MD_RCU_PLLON    RCU_CON_PLLON       /** @brief PLL Enable */

#define MD_RCU_LRCON    RCU_LCON_LRCON      /** @brief LRC Enable */
#define MD_RCU_LOSCON   RCU_LCON_LOSCON     /** @brief LOSC Enable */

/**
  * @brief MD_RCU_SRCSEL Calibration Clock Source Select
  */
#define MD_RCU_SRCSEL_HRC_4MHZ      (0x00000000UL)

/**
  * @brief MD_RCU_HRCSEL HRCSEL calibration value selection
  */
#define MD_RCU_HRCSEL_OPTIONBYTE    (0x00000000UL)
#define MD_RCU_HRCSEL_HRCTRIM       (0x00000001UL)

/**
  * @brief MD_RCU_LRCSEL LRCSEL calibration value selection
  */
#define MD_RCU_LRCSEL_OPTIONBYTE    (0x00000000UL)
#define MD_RCU_LRCSEL_LRCTRIM       (0x00000001UL)

/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions RCU Public Functions
  * @{
  */

/** @defgroup MD_RCU_Public_Functions_Group2 CON
  * @{
  */
/**
  * @brief  Set RCU_CON Register
  * @param  rcu RCU Instance
  * @param  con
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_con(RCU_TypeDef *rcu, uint32_t con)
{
    WRITE_REG(rcu->CON, con);
}

/**
  * @brief  Get RCU_CON Register
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_con(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->CON));
}

/**
  * @brief  Clock security system enable(CSS)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hosc_css(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HOSCCSSON);
}

/**
  * @brief  Clock security system disable(CSS)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hosc_css(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HOSCCSSON);
}

/**
  * @brief  Get Clock security system (CSS)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Hosc css enable
  *         @arg 0x0 : Hosc css disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hosc_css(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCCSSON) == (RCU_CON_HOSCCSSON));
}

/**
  * @brief  Get PLL0 clock ready flag
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_PLL0RDY_NOT_READY
  *         @arg @ref MD_RCU_PLL0RDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_pll_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_PLLRDY) == (RCU_CON_PLLRDY));
}

/**
  * @brief  PLL enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_pll(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_PLLON);
}

/**
  * @brief  PLL disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_pll(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_PLLON);
}

/**
  * @brief  Get PLL bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : PLL0 enable
  *         @arg 0x0 : PLL0 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_pll(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_PLLON) == (RCU_CON_PLLON));
}

/**
  * @brief  HOSC bypass mode enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hosc_bypass(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HOSCBYP);
}

/**
  * @brief  HOSC bypass mode disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hosc_bypass(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HOSCBYP);
}

/**
  * @brief  Get HOSC bypass mode bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Hosc bypass enable
  *         @arg 0x0 : Hosc bypass disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hosc_bypass(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCBYP) == (RCU_CON_HOSCBYP));
}

/**
  * @brief  Get HOSC clock ready flag
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HOSCRDY_NOT_READY
  *         @arg @ref MD_RCU_HOSCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCRDY) == (RCU_CON_HOSCRDY));
}

/**
  * @brief  HOSC enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HOSCON);
}

/**
  * @brief  HOSC disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hosc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HOSCON);
}

/**
  * @brief  Get HOSC bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Hosc enable
  *         @arg 0x0 : Hosc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCON) == (RCU_CON_HOSCON));
}

/**
  * @brief  Get HRCRDY clock ready flag
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HRCRDY_NOT_READY
  *         @arg @ref MD_RCU_HRCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HRCRDY) == (RCU_CON_HRCRDY));
}

/**
  * @brief  HRC enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hrc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HRCON);
}

/**
  * @brief  HRC disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hrc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HRCON);
}

/**
  * @brief  Get HRC bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Hrc enable
  *         @arg 0x0 : Hrc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hrc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HRCON) == (RCU_CON_HRCON));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group3 CFG
  * @{
  */
/**
  * @brief  Set RCU_CFG Register
  * @param  rcu RCU Instance
  * @param  cfg
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_cfg(RCU_TypeDef *rcu, uint32_t cfg)
{
    WRITE_REG(rcu->CFG, cfg);
}

/**
  * @brief  Get RCU_CFG Register
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_cfg(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->CFG));
}

/**
  * @brief  Set MCO clock output prescaler
  * @note   Output Frequency = MCO/(MPRE+1)
  * @param  rcu RCU Instance
  * @param  mpre This parameter can be one of the following values:
  *         @arg @ref MD_RCU_MPRE_MCO_DIV1
  *         @arg @ref MD_RCU_MPRE_MCO_DIV2
  *         @arg @ref MD_RCU_MPRE_MCO_DIV4
  *         @arg @ref MD_RCU_MPRE_MCO_DIV8
  *         @arg @ref MD_RCU_MPRE_MCO_DIV16
  *         @arg @ref MD_RCU_MPRE_MCO_DIV32
  *         @arg @ref MD_RCU_MPRE_MCO_DIV64
  *         @arg @ref MD_RCU_MPRE_MCO_DIV128
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_mco_div(RCU_TypeDef *rcu, uint32_t mpre)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_MPRE, mpre << RCU_CFG_MPRE_POSS);
}

/**
  * @brief  Get MCO clock output prescaler
  * @note   Output Frequency = MCO/(MPRE+1)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_MPRE_MCO_DIV1
  *         @arg @ref MD_RCU_MPRE_MCO_DIV2
  *         @arg @ref MD_RCU_MPRE_MCO_DIV4
  *         @arg @ref MD_RCU_MPRE_MCO_DIV8
  *         @arg @ref MD_RCU_MPRE_MCO_DIV16
  *         @arg @ref MD_RCU_MPRE_MCO_DIV32
  *         @arg @ref MD_RCU_MPRE_MCO_DIV64
  *         @arg @ref MD_RCU_MPRE_MCO_DIV128
  */
__STATIC_INLINE uint32_t md_rcu_get_mco_div(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_MPRE) >> RCU_CFG_MPRE_POSS);
}

/**
  * @brief  Set MCO clock output (MCO)
  * @param  rcu RCU Instance
  * @param  msw This parameter can be one of the following values:
  *         @arg @ref MD_RCU_MSW_MCO_DISABLE
  *         @arg @ref MD_RCU_MSW_MCO_LRC
  *         @arg @ref MD_RCU_MSW_MCO_LOSC
  *         @arg @ref MD_RCU_MSW_MCO_HRC
  *         @arg @ref MD_RCU_MSW_MCO_HOSC
  *         @arg @ref MD_RCU_MSW_MCO_SYSCLK
  *         @arg @ref MD_RCU_MSW_MCO_HCLK
  *         @arg @ref MD_RCU_MSW_MCO_PCLK
  *         @arg @ref MD_RCU_MSW_MCO_ADC
  *         @arg @ref MD_RCU_MSW_MCO_CHOPPER
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_mco_source(RCU_TypeDef *rcu, uint32_t msw)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_MSW, msw << RCU_CFG_MSW_POSS);
}

/**
  * @brief  Get Microcontroller clock output (MCO)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_MSW_MCO_DISABLE
  *         @arg @ref MD_RCU_MSW_MCO_LRC
  *         @arg @ref MD_RCU_MSW_MCO_LOSC
  *         @arg @ref MD_RCU_MSW_MCO_HRC
  *         @arg @ref MD_RCU_MSW_MCO_HOSC
  *         @arg @ref MD_RCU_MSW_MCO_SYSCLK
  *         @arg @ref MD_RCU_MSW_MCO_HCLK
  *         @arg @ref MD_RCU_MSW_MCO_PCLK
  *         @arg @ref MD_RCU_MSW_MCO_ADC
  *         @arg @ref MD_RCU_MSW_MCO_CHOPPER
  */
__STATIC_INLINE uint32_t md_rcu_get_mco_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_MSW) >> RCU_CFG_MSW_POSS);
}

/**
  * @brief  Set PLL0 reference clock source
  * @param  rcu RCU Instance
  * @param  pllsrc This parameter can be one of the following values:
  *         @arg @ref MD_RCU_PLLSRC_HRC
  *         @arg @ref MD_RCU_PLLSRC_HOSC
  *         @arg @ref MD_RCU_PLLSRC_LRC
  *         @arg @ref MD_RCU_PLLSRC_LOSC
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_pll_source(RCU_TypeDef *rcu, uint32_t pllsrc)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_PLLSRC, pllsrc << RCU_CFG_PLLSRC_POSS);
}

/**
  * @brief  Get PLL0/1 reference clock source
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_PLLSRC_HRC
  *         @arg @ref MD_RCU_PLLSRC_HOSC
  *         @arg @ref MD_RCU_PLLSRC_LRC
  *         @arg @ref MD_RCU_PLLSRC_LOSC
  */
__STATIC_INLINE uint32_t md_rcu_get_pll_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_PLLSRC) >> RCU_CFG_PLLSRC_POSS);
}

/**
  * @brief  Set PLL0 reference clock pre-divider
  * @param  rcu RCU Instance
  * @param  prediv This parameter can be one of the following values:
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV1
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV2
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV3
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV4
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV5
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV6
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV7
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV8
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV9
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV10
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV11
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV12
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV13
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV14
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV15
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV16
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_pll_prediv(RCU_TypeDef *rcu, uint32_t prediv)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_PREDIV, prediv << RCU_CFG_PREDIV_POSS);
}
/**
  * @brief  Get PLL0 reference clock pre-divider
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV1
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV2
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV3
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV4
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV5
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV6
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV7
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV8
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV9
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV10
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV11
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV12
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV13
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV14
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV15
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV16
  */
__STATIC_INLINE uint32_t md_rcu_get_pll_prediv(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_PREDIV) >> RCU_CFG_PREDIV_POSS);
}

/**
  * @brief  Set APB clock prescaler(PCLK)
  * @param  rcu RCU Instance
  * @param  ppre This parameter can be one of the following values:
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_1
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_2
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_4
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_8
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_16
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_pclk_div(RCU_TypeDef *rcu, uint32_t ppre)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_PPRE, ppre << RCU_CFG_PPRE_POSS);
}

/**
  * @brief  Get APB clock prescaler(PCLK)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_1
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_2
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_4
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_8
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_16
  */
__STATIC_INLINE uint32_t md_rcu_get_pclk_div(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_PPRE) >> RCU_CFG_PPRE_POSS);
}

/**
  * @brief  Set AHB clock prescaler(HCLK)
  * @param  rcu RCU Instance
  * @param  hpre This parameter can be one of the following values:
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_1
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_2
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_4
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_8
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_16
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_64
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_128
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_256
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_512
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hclk_div(RCU_TypeDef *rcu, uint32_t hpre)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_HPRE, hpre << RCU_CFG_HPRE_POSS);
}

/**
  * @brief  Get AHB clock prescaler(HCLK)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_1
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_2
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_4
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_8
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_16
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_64
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_128
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_256
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_512
  */
__STATIC_INLINE uint32_t md_rcu_get_hclk_div(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_HPRE) >> RCU_CFG_HPRE_POSS);
}

/**
  * @brief  System clock selection status
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_SWS_SYSCLK_HRC
  *         @arg @ref MD_RCU_SWS_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SWS_SYSCLK_PLL
  *         @arg @ref MD_RCU_SWS_SYSCLK_LRC
  *         @arg @ref MD_RCU_SWS_SYSCLK_LOSC
  */
__STATIC_INLINE uint32_t md_rcu_get_current_system_clock(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_SWS) >> RCU_CFG_SWS_POSS);
}

/**
  * @brief  Select system clock (SYSCLK) source
  * @param  rcu RCU Instance
  * @param  sw This parameter can be one of the following values:
  *         @arg @ref MD_RCU_SW_SYSCLK_HRC
  *         @arg @ref MD_RCU_SW_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SW_SYSCLK_PLL
  *         @arg @ref MD_RCU_SW_SYSCLK_LRC
  *         @arg @ref MD_RCU_SW_SYSCLK_LOSC

  * @retval None
  */
__STATIC_INLINE void md_rcu_set_system_clock_source(RCU_TypeDef *rcu, uint32_t sw)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_SW, sw << RCU_CFG_SW_POSS);
}

/**
  * @brief  Get Select system clock (SYSCLK) source
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_SW_SYSCLK_HRC
  *         @arg @ref MD_RCU_SW_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SW_SYSCLK_PLL
  *         @arg @ref MD_RCU_SW_SYSCLK_LRC
  *         @arg @ref MD_RCU_SW_SYSCLK_LOSC
  */
__STATIC_INLINE uint32_t md_rcu_get_system_clock_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_SW) >> RCU_CFG_SW_POSS);
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group4 PLL0
  * @{
  */
/**
  * @brief  Set RCU_CFG1 Register
  * @param  rcu RCU Instance
  * @param  pll0
  */
__STATIC_INLINE void md_rcu_set_cfg1(RCU_TypeDef *rcu, uint32_t cfg1)
{
    WRITE_REG(rcu->CFG1, cfg1);
}

/**
  * @brief  Get RCU_CFG1 Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_cfg1(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->CFG1));
}

/**
  * @brief  Set PLL0 VCO frequency fractional multiplication value
  * @param  rcu RCU Instance
  * @param  sysfreq
  * @retval
  */
__STATIC_INLINE void md_rcu_set_current_system_frequency(RCU_TypeDef *rcu, uint32_t sysfreq)
{
    MODIFY_REG(rcu->CFG1, RCU_CFG1_SYSFREQ, (sysfreq << RCU_CFG1_SYSFREQ_POSS));
}

/**
  * @brief  Get PLL0 VCO frequency fractional multiplication value
  * @param  rcu RCU Instance
  * @retval sysfreq
  */
__STATIC_INLINE uint32_t md_rcu_get_current_system_frequency(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG1, RCU_CFG1_SYSFREQ) >> RCU_CFG1_SYSFREQ_POSS);
}

/**
  * @brief  Set PLL0 Ready count
  * @param  rcu RCU Instance
  * @param  sysfreq
  * @retval
  */
__STATIC_INLINE void md_rcu_set_pllrdycnt(RCU_TypeDef *rcu, uint32_t pllrdycnt)
{
    MODIFY_REG(rcu->CFG1, RCU_CFG1_PLLRDYCNT, (pllrdycnt << RCU_CFG1_PLLRDYCNT_POSS));
}

/**
  * @brief  Get PLL0 Ready count
  * @param  rcu RCU Instance
  * @retval sysfreq
  */
__STATIC_INLINE uint32_t md_rcu_get_pllrdycnt(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG1, RCU_CFG1_PLLRDYCNT) >> RCU_CFG1_PLLRDYCNT_POSS);
}

/**
  * @brief  Set PLL clock frequency selection
  * @param  rcu RCU Instance
  * @param  pllfreq This parameter can be one of the following values:
  *         @arg @ref MD_RCU_PLLFREQ_32MHz
  *         @arg @ref MD_RCU_PLLFREQ_48MHz
  * @retval
  */
__STATIC_INLINE void md_rcu_set_pllfreq(RCU_TypeDef *rcu, uint32_t pllfreq)
{
    MODIFY_REG(rcu->CFG1, RCU_CFG1_PLLFREQ, (pllfreq << RCU_CFG1_PLLFREQ_POS));
}

/**
  * @brief  Get PLL clock frequency selection
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values
  *         @arg @ref MD_RCU_PLLFREQ_32MHz
  *         @arg @ref MD_RCU_PLLFREQ_48MHz
  */
__STATIC_INLINE uint32_t md_rcu_get_pllfreq(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG1, RCU_CFG1_PLLFREQ) >> RCU_CFG1_PLLFREQ_POS);
}

/**
  * @brief  Set HRC frequency adjustment
  * @param  rcu RCU Instance
  * @param  hrctune
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hrctune(RCU_TypeDef *rcu, uint32_t hrctune)
{
    MODIFY_REG(rcu->CFG1, RCU_CFG1_HRCTUNE, hrctune << RCU_CFG1_HRCTUNE_POSS);
}

/**
  * @brief  Get HRC frequency adjustment
  * @param  rcu RCU Instance
  * @retval hrctune
  */
__STATIC_INLINE uint32_t md_rcu_get_hrctune(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG1, RCU_CFG1_HRCTUNE) >> RCU_CFG1_HRCTUNE_POSS);
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group5 CFG2
  * @{
  */
/**
  * @brief  Set RCU_CFG2 Register
  * @param  rcu RCU Instance
  * @param  cfg2
  */
__STATIC_INLINE void md_rcu_set_cfg2(RCU_TypeDef *rcu, uint32_t cfg2)
{
    WRITE_REG(rcu->CFG2, cfg2);
}

/**
  * @brief  Get RCU_CFG2 Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_cfg2(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->CFG2));
}

/**
  * @brief  Set Current system clock frequency
  * @param  rcu RCU Instance
  * @param  chop_ckdiv
  * @retval
  */
__STATIC_INLINE void md_rcu_set_chop_ckdiv(RCU_TypeDef *rcu, uint32_t chop_ckdiv)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_CHOP_CKDIV, (chop_ckdiv << RCU_CFG2_CHOP_CKDIV_POSS));
}

/**
  * @brief  Get Current system clock frequency
  * @param  rcu RCU Instance
  * @retval chop_ckdiv
  */
__STATIC_INLINE uint32_t md_rcu_get_chop_ckdiv(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_CHOP_CKDIV) >> RCU_CFG2_CHOP_CKDIV_POSS);
}

/**
  * @brief  Select ADC Clock Divider
  * @param  rcu RCU Instance
  * @param  adc_ckdiv
  * @retval
  */
__STATIC_INLINE void md_rcu_set_adc_ckdiv(RCU_TypeDef *rcu, uint32_t adc_ckdiv)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_ADC_CKDIV, (adc_ckdiv << RCU_CFG2_ADC_CKDIV_POSS));
}

/**
  * @brief  Get ADC Clock Divider
  * @param  rcu RCU Instance
  * @retval adc_ckdiv
  */
__STATIC_INLINE uint32_t md_rcu_get_adc_ckdiv(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_ADC_CKDIV) >> RCU_CFG2_ADC_CKDIV_POSS);
}

/**
  * @brief  Select LCD clock divider
  * @param  rcu RCU Instance
  * @param  lcd_ckdiv This parameter can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_LCD_DIV2
  *         @arg @ref MD_RCU_CFG2_LCD_DIV4
  *         @arg @ref MD_RCU_CFG2_LCD_DIV6
  *         @arg @ref MD_RCU_CFG2_LCD_DIV8
  * @retval
  */
__STATIC_INLINE void md_rcu_set_lcd_ckdiv(RCU_TypeDef *rcu, uint32_t lcd_ckdiv)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_LCD_CKDIV, (lcd_ckdiv << RCU_CFG2_LCD_CKDIV_POSS));
}

/**
  * @brief  Get LCD clock divider
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_LCD_DIV2
  *         @arg @ref MD_RCU_CFG2_LCD_DIV4
  *         @arg @ref MD_RCU_CFG2_LCD_DIV6
  *         @arg @ref MD_RCU_CFG2_LCD_DIV8
  */
__STATIC_INLINE uint32_t md_rcu_get_lcd_ckdiv(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_LCD_CKDIV) >> RCU_CFG2_LCD_CKDIV_POSS);
}

/**
  * @brief  Select VLCD clock divider
  * @param  rcu RCU Instance
  * @param  vlcd_ckdiv
  * @retval
  */
__STATIC_INLINE void md_rcu_set_vlcd_ckdiv(RCU_TypeDef *rcu, uint32_t vlcd_ckdiv)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_VLCD_CKDIV, (vlcd_ckdiv << RCU_CFG2_VLCD_CKDIV_POSS));
}

/**
  * @brief  Get VLCD clock divider
  * @param  rcu RCU Instance
  * @retval vlcd_ckdiv
  */
__STATIC_INLINE uint32_t md_rcu_get_vlcd_ckdiv(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_VLCD_CKDIV) >> RCU_CFG2_VLCD_CKDIV_POSS);
}

/**
  * @brief  Set Chopper Clock Configuration Scratchpad
  * @param  rcu RCU Instance
  * @param  chop_ckcfg This parameter can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_CHOPCK_DISABLE
  *         @arg @ref MD_RCU_CFG2_CHOPCK_HOSC
  *         @arg @ref MD_RCU_CFG2_CHOPCK_HRC
  *         @arg @ref MD_RCU_CFG2_CHOPCK_PLL
  * @retval
  */
__STATIC_INLINE void md_rcu_set_chop_ckcfg(RCU_TypeDef *rcu, uint32_t chop_ckcfg)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_CHOP_CKCFG, (chop_ckcfg << RCU_CFG2_CHOP_CKCFG_POSS));
}

/**
  * @brief  Get Chopper Clock Configuration Scratchpad
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_CHOPCK_DISABLE
  *         @arg @ref MD_RCU_CFG2_CHOPCK_HOSC
  *         @arg @ref MD_RCU_CFG2_CHOPCK_HRC
  *         @arg @ref MD_RCU_CFG2_CHOPCK_PLL
  */
__STATIC_INLINE uint32_t md_rcu_get_chop_ckcfg(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_CHOP_CKCFG) >> RCU_CFG2_CHOP_CKCFG_POSS);
}

/**
  * @brief  Set ADC Clock Configuration Scratchpad
  * @param  rcu RCU Instance
  * @param  adc_ckcfg This parameter can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_ADCCK_DISABLE
  *         @arg @ref MD_RCU_CFG2_ADCCK_HOSC
  *         @arg @ref MD_RCU_CFG2_ADCCK_HRC
  *         @arg @ref MD_RCU_CFG2_ADCCK_PLL
  * @retval
  */
__STATIC_INLINE void md_rcu_set_adc_ckcfg(RCU_TypeDef *rcu, uint32_t adc_ckcfg)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_ADC_CKCFG, (adc_ckcfg << RCU_CFG2_ADC_CKCFG_POSS));
}

/**
  * @brief  Get ADC Clock Configuration Scratchpad
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_ADCCK_DISABLE
  *         @arg @ref MD_RCU_CFG2_ADCCK_HOSC
  *         @arg @ref MD_RCU_CFG2_ADCCK_HRC
  *         @arg @ref MD_RCU_CFG2_ADCCK_PLL
  */
__STATIC_INLINE uint32_t md_rcu_get_adc_ckcfg(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_ADC_CKCFG) >> RCU_CFG2_ADC_CKCFG_POSS);
}

/**
  * @brief  Set LCD Clock Configuration Scratchpad
  * @param  rcu RCU Instance
  * @param  lcd_ckcfg This parameter can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_LCDCK_DISABLE
  *         @arg @ref MD_RCU_CFG2_LCDCK_LRC
  *         @arg @ref MD_RCU_CFG2_LCDCK_LOSC
  * @retval
  */
__STATIC_INLINE void md_rcu_set_lcd_ckcfg(RCU_TypeDef *rcu, uint32_t lcd_ckcfg)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_LCD_CKCFG, (lcd_ckcfg << RCU_CFG2_LCD_CKCFG_POSS));
}

/**
  * @brief  Get LCD Clock Configuration Scratchpad
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_LCDCK_DISABLE
  *         @arg @ref MD_RCU_CFG2_LCDCK_LRC
  *         @arg @ref MD_RCU_CFG2_LCDCK_LOSC
  */
__STATIC_INLINE uint32_t md_rcu_get_lcd_ckcfg(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_LCD_CKCFG) >> RCU_CFG2_LCD_CKCFG_POSS);
}

/**
  * @brief  Set VLCD Clock Configuration Scratchpad
  * @param  rcu RCU Instance
  * @param  vlcd_ckcfg This parameter can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_VLCDCK_DISABLE
  *         @arg @ref MD_RCU_CFG2_VLCDCK_HOSC
  *         @arg @ref MD_RCU_CFG2_VLCDCK_HRC
  *         @arg @ref MD_RCU_CFG2_VLCDCK_PLL
  * @retval
  */
__STATIC_INLINE void md_rcu_set_vlcd_ckcfg(RCU_TypeDef *rcu, uint32_t vlcd_ckcfg)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_VLCD_CKCFG, (vlcd_ckcfg << RCU_CFG2_VLCD_CKCFG_POSS));
}

/**
  * @brief  Get VLCD Clock Configuration Scratchpad
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_CFG2_VLCDCK_DISABLE
  *         @arg @ref MD_RCU_CFG2_VLCDCK_HOSC
  *         @arg @ref MD_RCU_CFG2_VLCDCK_HRC
  *         @arg @ref MD_RCU_CFG2_VLCDCK_PLL
  */
__STATIC_INLINE uint32_t md_rcu_get_vlcd_ckcfg(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_VLCD_CKCFG) >> RCU_CFG2_VLCD_CKCFG_POSS);
}

/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group6 IER
  * @{
  */
/**
  * @brief  Set RCU_IER Register
  * @param  rcu RCU Instance
  * @param  ier
  */
__STATIC_INLINE void md_rcu_set_ier(RCU_TypeDef *rcu, uint32_t ier)
{
    WRITE_REG(rcu->IER, ier);
}

/**
  * @brief  Clock security system interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_css_losc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_CSSLOSC);
}

/**
  * @brief  Clock security system interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_css_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_CSSHOSC);
}

/**
  * @brief  PLL clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_pll_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_PLLRDY);
}

/**
  * @brief  HOSC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_hosc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_HOSCRDY);
}

/**
  * @brief  HRC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_hrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_HRCRDY);
}

/**
  * @brief  LOSC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_losc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_LOSCRDY);
}

/**
  * @brief  LRC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_lrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_LRCRDY);
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group7 IDR
  * @{
  */
/**
  * @brief  Set RCU_IDR Register
  * @param  rcu RCU Instance
  * @param  idr
  */
__STATIC_INLINE void md_rcu_set_idr(RCU_TypeDef *rcu, uint32_t idr)
{
    WRITE_REG(rcu->IDR, idr);
}

/**
  * @brief  Clock security system interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_css_losc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_CSSLOSC);
}

/**
  * @brief  Clock security system interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_css_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_CSSHOSC);
}

/**
  * @brief  PLL clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_pll_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_PLLRDY);
}

/**
  * @brief  HOSC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_hosc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_HOSCRDY);
}

/**
  * @brief  HRC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_hrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_HRCRDY);
}

/**
  * @brief  LOSC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_losc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_LOSCRDY);
}

/**
  * @brief  LRC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_lrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_LRCRDY);
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group8 IVS
  * @{
  */
/**
  * @brief  Set RCU_IVS Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_ivs(RCU_TypeDef *rcu)
{
    return (uint32_t) READ_REG(rcu->IVS);
}

/**
  * @brief  Clock security system interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_css_losc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_CSSLOSC) == (RCU_IVS_CSSLOSC));
}

/**
  * @brief  Clock security system interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_css_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_CSSHOSC) == (RCU_IVS_CSSHOSC));
}

/**
  * @brief  PLL clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_pll_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_PLLRDY) == (RCU_IVS_PLLRDY));
}

/**
  * @brief  HOSC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_HOSCRDY) == (RCU_IVS_HOSCRDY));
}

/**
  * @brief  HRC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_HRCRDY) == (RCU_IVS_HRCRDY));
}

/**
  * @brief  LOSC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_losc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_LOSCRDY) == (RCU_IVS_LOSCRDY));
}

/**
  * @brief  LRC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_LRCRDY) == (RCU_IVS_LRCRDY));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group9 RIF
  * @{
  */
/**
  * @brief  Set RCU_RIF Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_rif(RCU_TypeDef *rcu)
{
    return (uint32_t)READ_REG(rcu->RIF);
}

/**
  * @brief  Clock security system unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_css_losc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_CSSLOSC) == (RCU_RIF_CSSLOSC));
}

/**
  * @brief  Clock security system unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_css_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_CSSHOSC) == (RCU_RIF_CSSHOSC));
}

/**
  * @brief  PLL clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_pll_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_PLLRDY) == (RCU_RIF_PLLRDY));
}

/**
  * @brief  HOSC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_HOSCRDY) == (RCU_RIF_HOSCRDY));
}

/**
  * @brief  HRC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_HRCRDY) == (RCU_RIF_HRCRDY));
}

/**
  * @brief  LOSC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_losc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_LOSCRDY) == (RCU_RIF_LOSCRDY));
}

/**
  * @brief  LRC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_LRCRDY) == (RCU_RIF_LRCRDY));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group10 IFM
  * @{
  */
/**
  * @brief  Set RCU_IFM Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_ifm(RCU_TypeDef *rcu)
{
    return (uint32_t)READ_REG(rcu->IFM);
}

/**
  * @brief  Clock security system interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_css_losc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_CSSLOSC) == (RCU_IFM_CSSLOSC));
}

/**
  * @brief  Clock security system interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_css_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_CSSHOSC) == (RCU_IFM_CSSHOSC));
}

/**
  * @brief  PLL clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_pll_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_PLLRDY) == (RCU_IFM_PLLRDY));
}

/**
  * @brief  HOSC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_HOSCRDY) == (RCU_IFM_HOSCRDY));
}

/**
  * @brief  HRC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_HRCRDY) == (RCU_IFM_HRCRDY));
}

/**
  * @brief  LOSC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_losc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_LOSCRDY) == (RCU_IFM_LOSCRDY));
}

/**
  * @brief  LRC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_LRCRDY) == (RCU_IFM_LRCRDY));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group11 ICR
  * @{
  */
/**
  * @brief  Set RCU_ICR Register
  * @param  rcu RCU Instance
  * @param  icr
  */
__STATIC_INLINE void md_rcu_set_icr(RCU_TypeDef *rcu, uint32_t icr)
{
    WRITE_REG(rcu->ICR, icr);
}

/**
  * @brief  Clear Clock security system interrupt
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_css_losc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_CSSLOSC);
}

/**
  * @brief  Clear Clock security system interrupt
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_css_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_CSSHOSC);
}

/**
  * @brief  Clear PLL0 clock source stable interrupt flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_pll_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_PLLRDY);
}

/**
  * @brief  Clear HOSC clock source stable interrupt flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_hosc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_HOSCRDY);
}

/**
  * @brief  Clear HRC clock source stable interrupt flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_hrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_HRCRDY);
}

/**
  * @brief  Clear LOSC clock source stable interrupt flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_losc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_LOSCRDY);
}

/**
  * @brief  Clear LRC clock source stable interrupt flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_lrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_LRCRDY);
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group12 AHBRST
  * @{
  */
/**
  * @brief  Set RCU_AHBRST Register
  * @param  rcu RCU Instance
  * @param  ahbrst
  */
__STATIC_INLINE void md_rcu_set_ahbrst(RCU_TypeDef *rcu, uint32_t ahbrst)
{
    WRITE_REG(rcu->AHBRST, ahbrst);
}

/**
  * @brief  Get RCU_AHBRST Register
  * @param  rcu RCU Instance
  * @param  ahbrst
  */
__STATIC_INLINE uint32_t md_rcu_get_ahbrst(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->AHBRST));
}

/**
  * @brief  GPIOD reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiod_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_GPDEN);
}

/**
  * @brief  GPIOD reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiod_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_GPDEN);
}

/**
  * @brief  Get GPIOD reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiod_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_GPDEN) == (RCU_AHBRST_GPDEN));
}

/**
  * @brief  GPIOC reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioc_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_GPCEN);
}

/**
  * @brief  GPIOC reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioc_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_GPCEN);
}

/**
  * @brief  Get GPIOC reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioc_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_GPCEN) == (RCU_AHBRST_GPCEN));
}

/**
  * @brief  GPIOB reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiob_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_GPBEN);
}

/**
  * @brief  GPIOB reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiob_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_GPBEN);
}

/**
  * @brief  Get GPIOB reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiob_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_GPBEN) == (RCU_AHBRST_GPBEN));
}

/**
  * @brief  GPIOA reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioa_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_GPAEN);
}

/**
  * @brief  GPIOA reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioa_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_GPAEN);
}

/**
  * @brief  Get GPIOA reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioa_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_GPAEN) == (RCU_AHBRST_GPAEN));
}

/**
  * @brief  CALC reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_calc_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_CALCEN);
}

/**
  * @brief  CALC reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_calc_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_CALCEN);
}

/**
  * @brief  Get CALC reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_calc_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_CALCEN) == (RCU_AHBRST_CALCEN));
}

/**
  * @brief  RTC reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_rtc_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_RTCEN);
}

/**
  * @brief  RTC reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_rtc_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_RTCEN);
}

/**
  * @brief  Get RTC reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_rtc_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_RTCEN) == (RCU_AHBRST_RTCEN));
}

/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group13 APB1RST
  * @{
  */
/**
  * @brief  Set RCU_APB1RST Register
  * @param  rcu RCU Instance
  * @param  apb1rst
  */
__STATIC_INLINE void md_rcu_set_apb1rst(RCU_TypeDef *rcu, uint32_t apb1rst)
{
    WRITE_REG(rcu->APB1RST, apb1rst);
}

/**
  * @brief  Get RCU_APB1RST Register
  * @param  rcu RCU Instance
  * @param  apb1rst
  */
__STATIC_INLINE uint32_t md_rcu_get_apb1rst(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB1RST));
}

/**
  * @brief  I2C1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_i2c1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_I2C1EN);
}

/**
  * @brief  I2C1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_i2c1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_I2C1EN);
}

/**
  * @brief  Get I2C1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_i2c1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_I2C1EN) == (RCU_APB1RST_I2C1EN));
}

/**
  * @brief  UART4 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart4_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_UART4EN);
}

/**
  * @brief  UART4 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart4_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_UART4EN);
}

/**
  * @brief  Get UART4 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart4_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_UART4EN) == (RCU_APB1RST_UART4EN));
}

/**
  * @brief  UART3 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart3_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_UART3EN);
}

/**
  * @brief  UART3 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart3_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_UART3EN);
}

/**
  * @brief  Get UART3 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart3_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_UART3EN) == (RCU_APB1RST_UART3EN));
}

/**
  * @brief  UART2 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart2_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_UART2EN);
}

/**
  * @brief  UART2 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart2_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_UART2EN);
}

/**
  * @brief  Get UART2 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart2_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_UART2EN) == (RCU_APB1RST_UART2EN));
}

/**
  * @brief  WWDT reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_wwdt_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_WWDTEN);
}

/**
  * @brief  WWDT reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_wwdt_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_WWDTEN);
}

/**
  * @brief  Get WWDT reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_wwdt_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_WWDTEN) == (RCU_APB1RST_WWDTEN));
}

/**
  * @brief  BS16T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_bs16t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_BS16T1EN);
}

/**
  * @brief  BS16T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_bs16t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_BS16T1EN);
}

/**
  * @brief  Get BS16T1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_bs16t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_BS16T1EN) == (RCU_APB1RST_BS16T1EN));
}

/**
  * @brief  GP32C4T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T1EN);
}

/**
  * @brief  GP32C4T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T1EN);
}

/**
  * @brief  Get GP32C4T1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T1EN) == (RCU_APB1RST_GP32C4T1EN));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group14 APB2RST
  * @{
  */
/**
  * @brief  Set RCU_APB2RST Register
  * @param  rcu RCU Instance
  * @param  apb2rst
  */
__STATIC_INLINE void md_rcu_set_apb2rst(RCU_TypeDef *rcu, uint32_t apb2rst)
{
    WRITE_REG(rcu->APB2RST, apb2rst);
}

/**
  * @brief  Get RCU_APB2RST Register
  * @param  rcu RCU Instance
  * @param  apb2rst
  */
__STATIC_INLINE uint32_t md_rcu_get_apb2rst(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB2RST));
}

/**
  * @brief  CMP reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_cmp_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_CMPEN);
}

/**
  * @brief  CMP reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_cmp_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_CMPEN);
}

/**
  * @brief  Get CMP reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_cmp_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_CMPEN) == (RCU_APB2RST_CMPEN));
}

/**
  * @brief  GP16C2T4 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t4_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T4EN);
}

/**
  * @brief  GP16C2T4 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t4_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T4EN);
}

/**
  * @brief  Get GP16C2T4 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t4_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T4EN) == (RCU_APB2RST_GP16C2T4EN));
}

/**
  * @brief  GP16C2T3 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t3_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T3EN);
}

/**
  * @brief  GP16C2T3 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t3_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T3EN);
}

/**
  * @brief  Get GP16C2T3 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t3_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T3EN) == (RCU_APB2RST_GP16C2T3EN));
}

/**
  * @brief  GP16C2T2 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t2_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T2EN);
}

/**
  * @brief  GP16C2T2 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t2_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T2EN);
}

/**
  * @brief  Get GP16C2T2 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t2_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T2EN) == (RCU_APB2RST_GP16C2T2EN));
}

/**
  * @brief  GP16C2T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T1EN);
}

/**
  * @brief  GP16C2T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T1EN);
}

/**
  * @brief  Get GP16C2T1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T1EN) == (RCU_APB2RST_GP16C2T1EN));
}

/**
  * @brief  UART1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_UART1EN);
}

/**
  * @brief  UART1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_UART1EN);
}

/**
  * @brief  Get UART1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_UART1EN) == (RCU_APB2RST_UART1EN));
}

/**
  * @brief  SPI1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_spi1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_SPI1EN);
}

/**
  * @brief  SPI1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_spi1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_SPI1EN);
}

/**
  * @brief  Get SPI1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_spi1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_SPI1EN) == (RCU_APB2RST_SPI1EN));
}

/**
  * @brief  OPAMP reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_opamp_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_OPAMPEN);
}

/**
  * @brief  OPAMP reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_opamp_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_OPAMPEN);
}

/**
  * @brief  Get OPAMP reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_opamp_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_OPAMPEN) == (RCU_APB2RST_OPAMPEN));
}

/**
  * @brief  ANPWR reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_anpwr_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_ANPWREN);
}

/**
  * @brief  ANPWR reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_anpwr_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_ANPWREN);
}

/**
  * @brief  Get ANPWR reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_anpwr_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_ANPWREN) == (RCU_APB2RST_ANPWREN));
}

/**
  * @brief  ADC reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_adc_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_ADCEN);
}

/**
  * @brief  ADC reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_adc_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_ADCEN);
}

/**
  * @brief  Get ADC reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_adc_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_ADCEN) == (RCU_APB2RST_ADCEN));
}

/**
  * @brief  LCD reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_lcd_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_LCDEN);
}

/**
  * @brief  LCD reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_lcd_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_LCDEN);
}

/**
  * @brief  Get LCD reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_lcd_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_LCDEN) == (RCU_APB2RST_LCDEN));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group15 AHBEN
  * @{
  */
/**
  * @brief  Set RCU_AHBEN Register
  * @param  rcu RCU Instance
  * @param  ahben
  */
__STATIC_INLINE void md_rcu_set_ahben(RCU_TypeDef *rcu, uint32_t ahben)
{
    WRITE_REG(rcu->AHBEN, ahben);
}

/**
  * @brief  Get RCU_AHBEN Register
  * @param  rcu RCU Instance
  * @param  ahben
  */
__STATIC_INLINE uint32_t md_rcu_get_ahben(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->AHBEN));
}

/**
  * @brief  GPIOD clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiod(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_GPDEN);
}

/**
  * @brief  GPIOD Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiod(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_GPDEN);
}

/**
  * @brief  Get GPIOD Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpiod enable
  *         @arg 0x0 : gpiod disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiod(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_GPDEN) == (RCU_AHBEN_GPDEN));
}

/**
  * @brief  GPIOC clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_GPCEN);
}

/**
  * @brief  GPIOC Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_GPCEN);
}

/**
  * @brief  Get GPIOC Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpioc enable
  *         @arg 0x0 : gpioc disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_GPCEN) == (RCU_AHBEN_GPCEN));
}

/**
  * @brief  GPIOB clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiob(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_GPBEN);
}

/**
  * @brief  GPIOB Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiob(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_GPBEN);
}

/**
  * @brief  Get GPIOB Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpiob enable
  *         @arg 0x0 : gpiob disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiob(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_GPBEN) == (RCU_AHBEN_GPBEN));
}

/**
  * @brief  GPIOA clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioa(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_GPAEN);
}

/**
  * @brief  GPIOA Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioa(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_GPAEN);
}

/**
  * @brief  Get GPIOA Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpioa enable
  *         @arg 0x0 : gpioa disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioa(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_GPAEN) == (RCU_AHBEN_GPAEN));
}

/**
  * @brief  CALC clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_calc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_CALCEN);
}

/**
  * @brief  CALC Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_calc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_CALCEN);
}

/**
  * @brief  Get CALC Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : calc enable
  *         @arg 0x0 : calc disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_calc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_CALCEN) == (RCU_AHBEN_CALCEN));
}

/**
  * @brief  RTC clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_rtc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_RTCEN);
}

/**
  * @brief  RTC Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_rtc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_RTCEN);
}

/**
  * @brief  Get RTC Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : rtc enable
  *         @arg 0x0 : rtc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_rtc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_RTCEN) == (RCU_AHBEN_RTCEN));
}

/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group16 APB1EN
  * @{
  */
/**
  * @brief  Set RCU_APB1EN Register
  * @param  rcu RCU Instance
  * @param  apb1en
  */
__STATIC_INLINE void md_rcu_set_apb1en(RCU_TypeDef *rcu, uint32_t apb1en)
{
    WRITE_REG(rcu->APB1EN, apb1en);
}

/**
  * @brief  Get RCU_APB1EN Register
  * @param  rcu RCU Instance
  * @param  apb1en
  */
__STATIC_INLINE uint32_t md_rcu_get_apb1en(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB1EN));
}

/**
  * @brief  I2C1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_i2c1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_I2C1EN);
}

/**
  * @brief  I2C1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_i2c1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_I2C1EN);
}

/**
  * @brief  Get I2C1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : i2c1 enable
  *         @arg 0x0 : i2c1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_i2c1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_I2C1EN) == (RCU_APB1EN_I2C1EN));
}

/**
  * @brief  UART4 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart4(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_UART4EN);
}

/**
  * @brief  UART4 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart4(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_UART4EN);
}

/**
  * @brief  Get UART4 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart4 enable
  *         @arg 0x0 : uart4 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart4(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_UART4EN) == (RCU_APB1EN_UART4EN));
}

/**
  * @brief  UART3 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart3(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_UART3EN);
}

/**
  * @brief  UART3 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart3(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_UART3EN);
}

/**
  * @brief  Get UART3 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart3 enable
  *         @arg 0x0 : uart3 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart3(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_UART3EN) == (RCU_APB1EN_UART3EN));
}

/**
  * @brief  UART2 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart2(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_UART2EN);
}

/**
  * @brief  UART2 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart2(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_UART2EN);
}

/**
  * @brief  Get UART2 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart2 enable
  *         @arg 0x0 : uart2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart2(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_UART2EN) == (RCU_APB1EN_UART2EN));
}

/**
  * @brief  WWDT clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_wwdt(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_WWDTEN);
}

/**
  * @brief  WWDT Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_wwdt(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_WWDTEN);
}

/**
  * @brief  Get WWDT Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : wwdt enable
  *         @arg 0x0 : wwdt disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_wwdt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_WWDTEN) == (RCU_APB1EN_WWDTEN));
}

/**
  * @brief  BS16T1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_bs16t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_BS16T1EN);
}

/**
  * @brief  BS16T1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_bs16t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_BS16T1EN);
}

/**
  * @brief  Get BS16T1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : bs16t1 enable
  *         @arg 0x0 : bs16t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_bs16t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_BS16T1EN) == (RCU_APB1EN_BS16T1EN));
}

/**
  * @brief  GP32C4T1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T1EN);
}

/**
  * @brief  GP32C4T1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T1EN);
}

/**
  * @brief  Get GP32C4T1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp32c4t1 enable
  *         @arg 0x0 : gp32c4t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T1EN) == (RCU_APB1EN_GP32C4T1EN));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group17 APB2EN
  * @{
  */
/**
  * @brief  Set RCU_APB2EN Register
  * @param  rcu RCU Instance
  * @param  apb2en
  */
__STATIC_INLINE void md_rcu_set_apb2en(RCU_TypeDef *rcu, uint32_t apb2en)
{
    WRITE_REG(rcu->APB2EN, apb2en);
}

/**
  * @brief  Get RCU_APB2EN Register
  * @param  rcu RCU Instance
  * @param  apb2en
  */
__STATIC_INLINE uint32_t md_rcu_get_apb2en(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB2EN));
}

/**
  * @brief  CMP clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_cmp(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_CMPEN);
}

/**
  * @brief  CMP Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_cmp(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_CMPEN);
}

/**
  * @brief  Get CMP Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : cmp enable
  *         @arg 0x0 : cmp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_cmp(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_CMPEN) == (RCU_APB2EN_CMPEN));
}

/**
  * @brief  GP16C2T4EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t4(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T4EN);
}

/**
  * @brief  GP16C2T4EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t4(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T4EN);
}

/**
  * @brief  Get GP16C2T4EN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t4 enable
  *         @arg 0x0 : gp16c2t4 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t4(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T4EN) == (RCU_APB2EN_GP16C2T4EN));
}

/**
  * @brief  GP16C2T3EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t3(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T3EN);
}

/**
  * @brief  GP16C2T3EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t3(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T3EN);
}

/**
  * @brief  Get GP16C2T3EN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t3 enable
  *         @arg 0x0 : gp16c2t3 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t3(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T3EN) == (RCU_APB2EN_GP16C2T3EN));
}

/**
  * @brief  GP16C2T2EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t2(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T2EN);
}

/**
  * @brief  GP16C2T2EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t2(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T2EN);
}

/**
  * @brief  Get GP16C2T2EN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t2 enable
  *         @arg 0x0 : gp16c2t2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t2(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T2EN) == (RCU_APB2EN_GP16C2T2EN));
}

/**
  * @brief  GP16C2T1EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T1EN);
}

/**
  * @brief  GP16C2T1EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T1EN);
}

/**
  * @brief  Get GP16C2T1EN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t1 enable
  *         @arg 0x0 : gp16c2t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T1EN) == (RCU_APB2EN_GP16C2T1EN));
}

/**
  * @brief  UART1EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_UART1EN);
}

/**
  * @brief  UART1EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_UART1EN);
}

/**
  * @brief  Get UART1EN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart1 enable
  *         @arg 0x0 : uart1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_UART1EN) == (RCU_APB2EN_UART1EN));
}

/**
  * @brief  SPI1EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_spi1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_SPI1EN);
}

/**
  * @brief  SPI1EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_spi1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_SPI1EN);
}

/**
  * @brief  Get SPI1EN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : spi1 enable
  *         @arg 0x0 : spi1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_spi1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_SPI1EN) == (RCU_APB2EN_SPI1EN));
}

/**
  * @brief  OPAMPEN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_opamp(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_OPAMPEN);
}

/**
  * @brief  OPAMPEN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_opamp(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_OPAMPEN);
}

/**
  * @brief  Get OPAMPEN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : opamp enable
  *         @arg 0x0 : opamp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_opamp(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_OPAMPEN) == (RCU_APB2EN_OPAMPEN));
}

/**
  * @brief  ANPWREN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_anpwr(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_ANPWREN);
}

/**
  * @brief  ANPWREN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_anpwr(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_ANPWREN);
}

/**
  * @brief  Get ANPWREN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : opamp enable
  *         @arg 0x0 : opamp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_anpwr(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_ANPWREN) == (RCU_APB2EN_ANPWREN));
}

/**
  * @brief  ADCEN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_adc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_ADCEN);
}

/**
  * @brief  ADCEN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_adc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_ADCEN);
}

/**
  * @brief  Get ADCEN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : adc enable
  *         @arg 0x0 : adc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_adc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_ADCEN) == (RCU_APB2EN_ADCEN));
}

/**
  * @brief  LCDEN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_lcd(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_LCDEN);
}

/**
  * @brief  LCDEN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_lcd(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_LCDEN);
}

/**
  * @brief  Get LCDEN Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : adc enable
  *         @arg 0x0 : adc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_lcd(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_LCDEN) == (RCU_APB2EN_LCDEN));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group18 AHBSL
  * @{
  */
/**
  * @brief  Set RCU_AHBSL Register
  * @param  rcu RCU Instance
  * @param  ahbsl
  */
__STATIC_INLINE void md_rcu_set_ahbsl(RCU_TypeDef *rcu, uint32_t ahbsl)
{
    WRITE_REG(rcu->AHBSL, ahbsl);
}

/**
  * @brief  Get RCU_AHBSL Register
  * @param  rcu RCU Instance
  * @param  ahbsl
  */
__STATIC_INLINE uint32_t md_rcu_get_ahbsl(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->AHBSL));
}

/**
  * @brief  GPDEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiod_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_GPDEN);
}

/**
  * @brief  GPDEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiod_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_GPDEN);
}

/**
  * @brief  Get GPDEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpiod enable
  *         @arg 0x0 : gpiod disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiod_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_GPDEN) == (RCU_AHBSL_GPDEN));
}

/**
  * @brief  GPCEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioc_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_GPCEN);
}

/**
  * @brief  GPCEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioc_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_GPCEN);
}

/**
  * @brief  Get GPCEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpioc enable
  *         @arg 0x0 : gpioc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioc_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_GPCEN) == (RCU_AHBSL_GPCEN));
}

/**
  * @brief  GPBEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiob_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_GPBEN);
}

/**
  * @brief  GPBEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiob_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_GPBEN);
}

/**
  * @brief  Get GPBEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpiob enable
  *         @arg 0x0 : gpiob disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiob_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_GPBEN) == (RCU_AHBSL_GPBEN));
}

/**
  * @brief  GPAEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioa_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_GPAEN);
}

/**
  * @brief  GPAEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioa_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_GPAEN);
}

/**
  * @brief  Get GPAEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpioa enable
  *         @arg 0x0 : gpioa disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioa_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_GPAEN) == (RCU_AHBSL_GPAEN));
}

/**
  * @brief  CALCEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_calc_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_CALCEN);
}

/**
  * @brief  CALCEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_calc_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_CALCEN);
}

/**
  * @brief  Get CALCEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : calc enable
  *         @arg 0x0 : calc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_calc_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_CALCEN) == (RCU_AHBSL_CALCEN));
}

/**
  * @brief  RTCEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_rtc_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_RTCEN);
}

/**
  * @brief  RTCEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_rtc_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_RTCEN);
}

/**
  * @brief  Get RTCEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : rtc enable
  *         @arg 0x0 : rtc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_rtc_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_RTCEN) == (RCU_AHBSL_RTCEN));
}

/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group19 APB1SL
  * @{
  */
/**
  * @brief  Set RCU_APB1SL Register
  * @param  rcu RCU Instance
  * @param  apb1sl
  */
__STATIC_INLINE void md_rcu_set_apb1sl(RCU_TypeDef *rcu, uint32_t apb1sl)
{
    WRITE_REG(rcu->APB1SL, apb1sl);
}

/**
  * @brief  Get RCU_APB1SL Register
  * @param  rcu RCU Instance
  * @param  apb1sl
  */
__STATIC_INLINE uint32_t md_rcu_get_apb1sl(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB1SL));
}

/**
  * @brief  I2C1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_i2c1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_I2C1EN);
}

/**
  * @brief  I2C1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_i2c1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_I2C1EN);
}

/**
  * @brief  Get I2C1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : i2c1 enable
  *         @arg 0x0 : i2c1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_i2c1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_I2C1EN) == (RCU_APB1SL_I2C1EN));
}

/**
  * @brief  UART4EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart4_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_UART4EN);
}

/**
  * @brief  UART4EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart4_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_UART4EN);
}

/**
  * @brief  Get UART4EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart4 enable
  *         @arg 0x0 : uart4 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart4_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_UART4EN) == (RCU_APB1SL_UART4EN));
}

/**
  * @brief  UART3EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart3_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_UART3EN);
}

/**
  * @brief  UART3EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart3_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_UART3EN);
}

/**
  * @brief  Get UART3EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart3 enable
  *         @arg 0x0 : uart3 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart3_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_UART3EN) == (RCU_APB1SL_UART3EN));
}

/**
  * @brief  UART2EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart2_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_UART2EN);
}

/**
  * @brief  UART2EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart2_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_UART2EN);
}

/**
  * @brief  Get UART2EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart2 enable
  *         @arg 0x0 : uart2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart2_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_UART2EN) == (RCU_APB1SL_UART2EN));
}

/**
  * @brief  WWDTEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_wwdt_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_WWDTEN);
}

/**
  * @brief WWDTEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_wwdt_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_WWDTEN);
}

/**
  * @brief  Get WWDTEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : wwdt enable
  *         @arg 0x0 : wwdt disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_wwdt_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_WWDTEN) == (RCU_APB1SL_WWDTEN));
}

/**
  * @brief  BS16T1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_bs16t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_BS16T1EN);
}

/**
  * @brief  BS16T1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_bs16t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_BS16T1EN);
}

/**
  * @brief  Get BS16T1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : bs16t1 enable
  *         @arg 0x0 : bs16t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_bs16t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_BS16T1EN) == (RCU_APB1SL_BS16T1EN));
}

/**
  * @brief  GP32C4T1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T1EN);
}

/**
  * @brief  GP32C4T1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T1EN);
}

/**
  * @brief  Get GP32C4T1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp32c4t1 enable
  *         @arg 0x0 : gp32c4t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T1EN) == (RCU_APB1SL_GP32C4T1EN));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group20 APB2SL
  * @{
  */
/**
  * @brief  Set RCU_APB2SL Register
  * @param  rcu RCU Instance
  * @param  apb2sl
  */
__STATIC_INLINE void md_rcu_set_apb2sl(RCU_TypeDef *rcu, uint32_t apb2sl)
{
    WRITE_REG(rcu->APB2SL, apb2sl);
}

/**
  * @brief  Get RCU_APB2SL Register
  * @param  rcu RCU Instance
  * @param  apb2sl
  */
__STATIC_INLINE uint32_t md_rcu_get_apb2sl(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB2SL));
}

/**
  * @brief  CMP clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_cmp_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_CMPEN);
}

/**
  * @brief  CMP clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_cmp_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_CMPEN);
}

/**
  * @brief  Get CMP clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : cmp enable
  *         @arg 0x0 : cmp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_cmp_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_CMPEN) == (RCU_APB2SL_CMPEN));
}

/**
  * @brief  GP16C2T4EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t4_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T4EN);
}

/**
  * @brief  GP16C2T4EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t4_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T4EN);
}

/**
  * @brief  Get GP16C2T4EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t4 enable
  *         @arg 0x0 : gp16c2t4 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t4_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T4EN) == (RCU_APB2SL_GP16C2T4EN));
}

/**
  * @brief  GP16C2T3EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t3_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T3EN);
}

/**
  * @brief  GP16C2T3EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t3_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T3EN);
}

/**
  * @brief  Get GP16C2T3EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t3 enable
  *         @arg 0x0 : gp16c2t3 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t3_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T3EN) == (RCU_APB2SL_GP16C2T3EN));
}

/**
  * @brief  GP16C2T2EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T2EN);
}

/**
  * @brief  GP16C2T2EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T2EN);
}

/**
  * @brief  Get GP16C2T2EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t2 enable
  *         @arg 0x0 : gp16c2t2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T2EN) == (RCU_APB2SL_GP16C2T2EN));
}

/**
  * @brief  GP16C2T1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T1EN);
}

/**
  * @brief  GP16C2T1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T1EN);
}

/**
  * @brief  Get GP16C2T1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t1 enable
  *         @arg 0x0 : gp16c2t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T1EN) == (RCU_APB2SL_GP16C2T1EN));
}

/**
  * @brief  UART1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_UART1EN);
}

/**
  * @brief  UART1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_UART1EN);
}

/**
  * @brief  Get UART1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart1 enable
  *         @arg 0x0 : uart1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_UART1EN) == (RCU_APB2SL_UART1EN));
}

/**
  * @brief  SPI1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_spi1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_SPI1EN);
}

/**
  * @brief  SPI1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_spi1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_SPI1EN);
}

/**
  * @brief  Get SPI1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : spi1 enable
  *         @arg 0x0 : spi1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_spi1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_SPI1EN) == (RCU_APB2SL_SPI1EN));
}

/**
  * @brief  OPAMPEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_opamp_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_OPAMPEN);
}

/**
  * @brief  OPAMPEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_opamp_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_OPAMPEN);
}

/**
  * @brief  Get OPAMPEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : opamp enable
  *         @arg 0x0 : opamp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_opamp_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_OPAMPEN) == (RCU_APB2SL_OPAMPEN));
}

/**
  * @brief  ANPWREN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_anpwr_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_ANPWREN);
}

/**
  * @brief  ANPWREN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_anpwr_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_ANPWREN);
}

/**
  * @brief  Get ANPWREN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : opamp enable
  *         @arg 0x0 : opamp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_anpwr_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_ANPWREN) == (RCU_APB2SL_ANPWREN));
}

/**
  * @brief  ADCEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_adc_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_ADCEN);
}

/**
  * @brief  ADCEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_adc_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_ADCEN);
}

/**
  * @brief  Get ADCEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : adc enable
  *         @arg 0x0 : adc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_adc_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_ADCEN) == (RCU_APB2SL_ADCEN));
}

/**
  * @brief  LCDEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_lcd_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_LCDEN);
}

/**
  * @brief  LCDEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_lcd_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_LCDEN);
}

/**
  * @brief  Get LCDEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : adc enable
  *         @arg 0x0 : adc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_lcd_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_LCDEN) == (RCU_APB2SL_LCDEN));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group21 LCON
  * @{
  */
/**
  * @brief  Set RCU_LCON Register
  * @param  rcu RCU Instance
  * @param  lcon
  */
__STATIC_INLINE void md_rcu_set_lcon(RCU_TypeDef *rcu, uint32_t lcon)
{
    WRITE_REG(rcu->LCON, lcon);
}

/**
  * @brief  Get RCU_LCON Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_lcon(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->LCON));
}

/**
  * @brief  Set LOSC clock current adjustment
  * @param  rcu RCU Instance
  * @param  loscitune This parameter can be one of the following values:
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_180nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_240nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_300nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_360nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_420nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_480nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_540nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_600nA
  */
__STATIC_INLINE void md_rcu_set_loscitune(RCU_TypeDef *rcu, uint32_t loscitune)
{
    MODIFY_REG(rcu->LCON, RCU_LCON_LOSCITUNE, loscitune << RCU_LCON_LOSCITUNE_POSS);
}

/**
  * @brief  Get LOSC clock current adjustment
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_180nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_240nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_300nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_360nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_420nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_480nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_540nA
  *         @arg @ref MD_RCU_LCON_LOSCITUNE_600nA
  */
__STATIC_INLINE uint32_t md_rcu_get_loscitune(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->LCON, RCU_LCON_LOSCITUNE) >> RCU_LCON_LOSCITUNE_POSS);
}

/**
  * @brief  Set LOSC Clock Security System enabled
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_losc_css(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->LCON, RCU_LCON_LOSCCSSON);
}

/**
  * @brief  Set LOSC Clock Security System disabled
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_losc_css(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->LCON, RCU_LCON_LOSCCSSON);
}

/**
  * @brief  Get LOSC Clock Security System Enable (CSS)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : losc css enable
  *         @arg 0x0 : losc css disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_losc_css(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_LOSCCSSON) == (RCU_LCON_LOSCCSSON));
}

/**
  * @brief  Set LOSC bypass mode enabled
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_losc_bypass(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->LCON, RCU_LCON_LOSCBYP);
}

/**
  * @brief  Set LOSC bypass mode disabled
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_losc_bypass(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->LCON, RCU_LCON_LOSCBYP);
}

/**
  * @brief  Get LOSC bypass mode
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : losc bypass enable
  *         @arg 0x0 : losc bypass disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_losc_bypass(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_LOSCBYP) == (RCU_LCON_LOSCBYP));
}

/**
  * @brief  Get LOSCRDY clock ready flag
  * @note
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_LOSCRDY_NOT_READY
  *         @arg @ref MD_RCU_LOSCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_losc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_LOSCRDY) == (RCU_LCON_LOSCRDY));
}

/**
  * @brief  Set LOSC enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_losc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->LCON, RCU_LCON_LOSCON);
}

/**
  * @brief  Set LOSC disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_losc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->LCON, RCU_LCON_LOSCON);
}

/**
  * @brief  Get LOSC bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : losc enable
  *         @arg 0x0 : losc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_losc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_LOSCON) == (RCU_LCON_LOSCON));
}

/**
  * @brief  Get LRCRDY clock ready flag
  * @note
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_LRCRDY_NOT_READY
  *         @arg @ref MD_RCU_LRCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_LRCRDY) == (RCU_LCON_LRCRDY));
}

/**
  * @brief  Set LRC enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_lrc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->LCON, RCU_LCON_LRCON);
}

/**
  * @brief  Set LRC disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_lrc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->LCON, RCU_LCON_LRCON);
}

/**
  * @brief  Get LRC clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : lrc enable
  *         @arg 0x0 : lrc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_lrc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_LRCON) == (RCU_LCON_LRCON));
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group22 RSTF
  * @{
  */
/**
  * @brief  Set RCU_RSTF Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE void md_rcu_set_rstf(RCU_TypeDef *rcu, uint32_t rstf)
{
    WRITE_REG(rcu->RSTF, rstf);
}

/**
  * @brief  Get RCU_RSTF Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_rstf(RCU_TypeDef *rcu)
{
    return (uint32_t)READ_REG(rcu->RSTF);
}

/**
  * @brief  Get Low power reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_wakeup(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_LPRSTF) == (RCU_RSTF_LPRSTF));
}

/**
  * @brief  Get WWDT reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_wwdt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_WWDTRSTF) == (RCU_RSTF_WWDTRSTF));
}

/**
  * @brief  Get IWDT reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_iwdt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_IWDTRSTF) == (RCU_RSTF_IWDTRSTF));
}

/**
  * @brief  Get SW reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_software(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_SWRSTF) == (RCU_RSTF_SWRSTF));
}

/**
  * @brief  Get Option Byte reload reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_option_byte_reload(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_OBLRSTF) == (RCU_RSTF_OBLRSTF));
}

/**
  * @brief  Get NRST external pin reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_nrst(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_NRSTF) == (RCU_RSTF_NRSTF));
}

/**
  * @brief  Get BOR reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_bor(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_BORRSTF) == (RCU_RSTF_BORRSTF));
}

/**
  * @brief  Get POR/PDR reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_por(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_PORRSTF) == (RCU_RSTF_PORRSTF));
}

/**
  * @brief  Clear reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_flag_reset_flag(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->RSTF, RCU_RSTF_CLRFLG);
}
/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group27 CKTRIM
  * @{
  */
/**
  * @brief  Set RCU_CKTRIM Register
  * @param  rcu RCU Instance
  * @param  cktrim
  */
__STATIC_INLINE void md_rcu_set_cktrim(RCU_TypeDef *rcu, uint32_t cktrim)
{
    WRITE_REG(rcu->CKTRIM, cktrim);
}

/**
  * @brief  Get RCU_CKTRIM Register
  * @param  rcu RCU Instance
  * @param  cktrim
  */
__STATIC_INLINE uint32_t md_rcu_get_cktrim(RCU_TypeDef *rcu)
{
    return (uint32_t)READ_REG(rcu->CKTRIM);
}

/**
  * @brief  Set HRC calibration value selection
  * @param  rcu RCU Instance
  * @param  hrcsel
  *         @arg @ref MD_RCU_HRCSEL_OPTIONBYTE
  *         @arg @ref MD_RCU_HRCSEL_HRCTRIM
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hrctrim_source(RCU_TypeDef *rcu, uint32_t hrcsel)
{
    MODIFY_REG(rcu->CKTRIM, RCU_CKTRIM_HRCSEL, hrcsel << RCU_CKTRIM_HRCSEL_POS);
}

/**
  * @brief  Get HRC calibration value selection
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HRCSEL_OPTIONBYTE
  *         @arg @ref MD_RCU_HRCSEL_HRCTRIM
  */
__STATIC_INLINE uint32_t md_rcu_get_hrctrim_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CKTRIM, RCU_CKTRIM_HRCSEL) >> RCU_CKTRIM_HRCSEL_POS);
}

/**
  * @brief  Set HRC 24/48 calibration value
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hrc24_48trim(RCU_TypeDef *rcu, uint32_t hrc24_48trim)
{
    MODIFY_REG(rcu->CKTRIM, RCU_CKTRIM_HRCTRIMH_MSK, (hrc24_48trim << RCU_CKTRIM_HRCTRIMH_POSS));
}

/**
  * @brief  Get HRC 24/48 calibration value
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_hrc24_48trim(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CKTRIM, RCU_CKTRIM_HRCTRIMH_MSK) >> RCU_CKTRIM_HRCTRIMH_POSS);
}

/**
  * @brief  Set HRC16/32 calibration value
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hrc16_32trim(RCU_TypeDef *rcu, uint32_t hrc16_32trim)
{
    MODIFY_REG(rcu->CKTRIM, RCU_CKTRIM_HRCTRIML, (hrc16_32trim << RCU_CKTRIM_HRCTRIML_POSS));
}

/**
  * @brief  Get HRC16/32 calibration value
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_hrc16_32trim(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CKTRIM, RCU_CKTRIM_HRCTRIML) >> RCU_CKTRIM_HRCTRIML_POSS);
}

/**
  * @}
  */

/** @defgroup MD_RCU_Public_Functions_Group1 Initialization
  * @{
  */
void md_rcu_pll_init(RCU_TypeDef *, md_rcu_init_typedef *);
void md_rcu_sys_init(RCU_TypeDef *, md_rcu_init_typedef *);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @} Micro_Driver
  */


#ifdef __cplusplus
}
#endif
#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
