/**********************************************************************************
 *
 * @file    md_opamp.h
 * @brief   header file of md_opamp.c
 *
 * @date    8 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          8 Aug 2022      Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_OPAMP_H__
#define __MD_OPAMP_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include "es32f0943.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_OPAMP OPAMP
  * @brief OPAMP micro driver
  * @{
  */

/** @defgroup MD_OPAMP_Public_Macros OPAMP Public Macros
  * @{
  */

#define MD_OPAMP_OPA_VAL_NEGATIVE_GREATER_POSITIVE      (0x00000000U)
#define MD_OPAMP_OPA_VAL_POSITIVE_GREATER_NEGATIVE      (0x00000001U)

#define MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH1            (0x00000000U)
#define MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH2            (0x00000001U)
#define MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH3            (0x00000002U)
#define MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH4            (0x00000003U)
#define MD_OPAMP_OPA_PWM_SOURCE_GP16C2T1_CH1            (0x00000004U)
#define MD_OPAMP_OPA_PWM_SOURCE_GP16C2T2_CH1            (0x00000005U)
#define MD_OPAMP_OPA_PWM_SOURCE_GP16C2T3_CH1            (0x00000006U)
#define MD_OPAMP_OPA_PWM_SOURCE_GP16C2T4_CH1            (0x00000007U)

#define MD_OPAMP_OPA_MULTS_PASS                         (0x00000000U)
#define MD_OPAMP_OPA_MULTS_THROUGH                      (0x00000001U)

#define MD_OPAMP_OPA_POLAS_NON_INVERTING                (0x00000000U)
#define MD_OPAMP_OPA_POLAS_INVERTED                     (0x00000001U)

#define MD_OPAMP_OPA_INNS_AIN3                          (1 << 0x00000000U)
#define MD_OPAMP_OPA_INNS_AIN5                          (1 << 0x00000001U)
#define MD_OPAMP_OPA_INNS_VDIV                          (1 << 0x00000002U)
#define MD_OPAMP_OPA_INNS_OPOUT                         (1 << 0x00000003U)
#define MD_OPAMP_OPA_INNS_R2ROPO                        (1 << 0x00000004U)
#define MD_OPAMP_OPA_INNS_OPOC                          (1 << 0x00000005U)
#define MD_OPAMP_OPA_INNS_AIN2                          (1 << 0x00000006U)

#define MD_OPAMP_OPA_INPS_AIN2                          (1 << 0x00000000U)
#define MD_OPAMP_OPA_INPS_AIN4                          (1 << 0x00000001U)
#define MD_OPAMP_OPA_INPS_VDIV                          (1 << 0x00000002U)
#define MD_OPAMP_OPA_INPS_VREF                          (1 << 0x00000003U)
#define MD_OPAMP_OPA_INPS_AIN5                          (1 << 0x00000004U)
#define MD_OPAMP_OPA_INPS_AIN6                          (1 << 0x00000005U)
#define MD_OPAMP_OPA_INPS_AIN7                          (1 << 0x00000006U)

#define MD_OPAMP_OPA_CAPS_SAMPLE                        (0x00000000U)
#define MD_OPAMP_OPA_CAPS_INTEGRATE                     (0x00000001U)

#define MD_OPAMP_OPA_CHPCKS_1MHZ                        (0x00000000U)
#define MD_OPAMP_OPA_CHPCKS_2MHZ                        (0x00000001U)

#define MD_OPAMP_DBCNT_IMMEDIATELY                      (0x00000000U)
#define MD_OPAMP_DBCNT_SAME_2_TIMES                     (0x00000001U)
#define MD_OPAMP_DBCNT_SAME_3_TIMES                     (0x00000002U)
#define MD_OPAMP_DBCNT_SAME_4_TIMES                     (0x00000003U)
#define MD_OPAMP_DBCNT_SAME_5_TIMES                     (0x00000004U)
#define MD_OPAMP_DBCNT_SAME_6_TIMES                     (0x00000005U)
#define MD_OPAMP_DBCNT_SAME_7_TIMES                     (0x00000006U)
#define MD_OPAMP_DBCNT_SAME_8_TIMES                     (0x00000007U)


typedef struct
{
    uint32_t opa_inns;
    uint32_t opa_inps;
    uint32_t opa_out;
    uint32_t opa_pwm_source;
    uint32_t opa_mults;
    uint32_t opa_polas;
    uint32_t opa_caps;
    uint32_t opa_chpcks;
    uint32_t bounce_counter;
    uint32_t bounce_prescaler;
} md_opamp_config_typedef;

typedef struct
{
    uint32_t chop_clocksource;
    uint32_t chop_clockdiv;
    uint32_t vldo_mode;
    uint32_t vldo_sel;
    uint32_t vbg_chpcks;
} md_opamp_power_init_typedef;


/**
  * @}
  */

/** @defgroup MD_OPAMP_Public_Functions OPAMP Public Functions
  * @{
  */

/** @defgroup MD_OPAMP_Public_Functions_Group2 OPA_CTRL
  * @{
  */

/**
  * @brief  Set OPA_CTRL Register
  * @param  opamp OPAMP Instance
  * @param  opa_ctrl
  */
__STATIC_INLINE void md_opamp_set_opa_ctrl(OPAMP_TypeDef *opamp, uint32_t opa_ctrl)
{
    WRITE_REG(opamp->OPA_CTRL, opa_ctrl);
}

/**
  * @brief  Get OPA_CTRL Register
  * @param  opamp OPAMP Instance
  * @retval opa_ctrl Values
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_ctrl(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA_CTRL);
}

/**
  * @brief  Set OPAMP Numeric value
  * @param  opamp OPAMP Instance
  * @param  opa_out This parameter can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_VAL_NEGATIVE_GREATER_POSITIVE
  *         @arg @ref MD_OPAMP_OPA_VAL_POSITIVE_GREATER_NEGATIVE
  */
__STATIC_INLINE void md_opamp_set_opa_out(OPAMP_TypeDef *opamp, uint32_t opa_out)
{
    MODIFY_REG(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_OUT, opa_out << OPAMP_OPA_CTRL_OPA_OUT_POS);
}

/**
  * @brief  Get OPAMP Numeric value
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_VAL_NEGATIVE_GREATER_POSITIVE
  *         @arg @ref MD_OPAMP_OPA_VAL_POSITIVE_GREATER_NEGATIVE
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_out(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_OUT) >> OPAMP_OPA_CTRL_OPA_OUT_POS);
}

/**
  * @brief  Set OPA_PWM source selection
  * @param  opamp OPAMP Instance
  * @param  source This parameter can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH1
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH2
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH3
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH4
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP16C2T1_CH1
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP16C2T2_CH1
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP16C2T3_CH1
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP16C2T4_CH1
  */
__STATIC_INLINE void md_opamp_set_opa_pwm_source(OPAMP_TypeDef *opamp, uint32_t source)
{
    MODIFY_REG(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_MULTSRC, source << OPAMP_OPA_CTRL_OPA_MULTSRC_POSS);
}

/**
  * @brief  Get OPA_PWM source selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH1
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH2
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH3
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP32C4T1_CH4
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP16C2T1_CH1
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP16C2T2_CH1
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP16C2T3_CH1
  *         @arg @ref MD_OPAMP_OPA_PWM_SOURCE_GP16C2T4_CH1
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_pwm_source(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_MULTSRC) >> OPAMP_OPA_CTRL_OPA_MULTSRC_POSS);
}

/**
  * @brief  Set OPAMP output is selected by OPA_PWM multiplexer
  * @param  opamp OPAMP Instance
  * @param  mults This parameter can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_MULTS_PASS
  *         @arg @ref MD_OPAMP_OPA_MULTS_THROUGH
  */
__STATIC_INLINE void md_opamp_set_opa_mults(OPAMP_TypeDef *opamp, uint32_t mults)
{
    MODIFY_REG(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_MULTS, mults << OPAMP_OPA_CTRL_OPA_MULTS_POS);
}

/**
  * @brief  Get OPAMP output is selected by OPA_PWM multiplexer
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_MULTS_PASS
  *         @arg @ref MD_OPAMP_OPA_MULTS_THROUGH
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_mults(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_MULTS) >> OPAMP_OPA_CTRL_OPA_MULTS_POS);
}

/**
  * @brief  Set OPAMP digital output phase selection
  * @param  opamp OPAMP Instance
  * @param  polas This parameter can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_POLAS_NON_INVERTING
  *         @arg @ref MD_OPAMP_OPA_POLAS_INVERTED
  */
__STATIC_INLINE void md_opamp_set_opa_polas(OPAMP_TypeDef *opamp, uint32_t polas)
{
    MODIFY_REG(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_POLAS, polas << OPAMP_OPA_CTRL_OPA_POLAS_POS);
}

/**
  * @brief  Get OPAMP digital output phase selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_POLAS_NON_INVERTING
  *         @arg @ref MD_OPAMP_OPA_POLAS_INVERTED
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_polas(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_POLAS) >> OPAMP_OPA_CTRL_OPA_POLAS_POS);
}

/**
  * @brief  OPAMP Digital Filter Switch Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_enable_opa_filter(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_FILTEN);
}

/**
  * @brief  OPAMP Digital Filter Switch Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_disable_opa_filter(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_FILTEN);
}

/**
  * @brief  Check if OPAMP Digital Filter Switch is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opamp_is_enabled_opa_filter(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_FILTEN) == (OPAMP_OPA_CTRL_OPA_FILTEN));
}

/**
  * @brief  OPAMP digital output switch Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_enable_opa_output(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_DOEN);
}

/**
  * @brief  OPAMP digital output switch Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_disable_opa_output(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_DOEN);
}

/**
  * @brief  Check if OPAMP digital output switch is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opamp_is_enabled_opa_output(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_DOEN) == (OPAMP_OPA_CTRL_OPA_DOEN));
}

/**
  * @brief  Set OPAMP Negative channel input switch selection
  * @param  opamp OPAMP Instance
  * @param  opa_inns This parameter can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_INNS_AIN3
  *         @arg @ref MD_OPAMP_OPA_INNS_AIN5
  *         @arg @ref MD_OPAMP_OPA_INNS_VDIV
  *         @arg @ref MD_OPAMP_OPA_INNS_OPOUT
  *         @arg @ref MD_OPAMP_OPA_INNS_R2ROPO
  *         @arg @ref MD_OPAMP_OPA_INNS_OPOC
  *         @arg @ref MD_OPAMP_OPA_INNS_AIN2
  */
__STATIC_INLINE void md_opamp_set_opa_inns(OPAMP_TypeDef *opamp, uint32_t opa_inns)
{
    MODIFY_REG(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_INNS, opa_inns << OPAMP_OPA_CTRL_OPA_INNS_POSS);
}

/**
  * @brief  Get OPAMP Negative channel input switch selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_INNS_AIN3
  *         @arg @ref MD_OPAMP_OPA_INNS_AIN5
  *         @arg @ref MD_OPAMP_OPA_INNS_VDIV
  *         @arg @ref MD_OPAMP_OPA_INNS_OPOUT
  *         @arg @ref MD_OPAMP_OPA_INNS_R2ROPO
  *         @arg @ref MD_OPAMP_OPA_INNS_OPOC
  *         @arg @ref MD_OPAMP_OPA_INNS_AIN2
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_inns(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_INNS) >> OPAMP_OPA_CTRL_OPA_INNS_POSS);
}

/**
  * @brief  Set OPAMP Positive channel input switch selection
  * @param  opamp OPAMP Instance
  * @param  opa_inns This parameter can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN2
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN4
  *         @arg @ref MD_OPAMP_OPA_INPS_VDIV
  *         @arg @ref MD_OPAMP_OPA_INPS_VREF
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN5
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN6
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN7
  */
__STATIC_INLINE void md_opamp_set_opa_inps(OPAMP_TypeDef *opamp, uint32_t opa_inns)
{
    MODIFY_REG(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_INPS, opa_inns << OPAMP_OPA_CTRL_OPA_INPS_POSS);
}

/**
  * @brief  Get OPAMP Positive channel input switch selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN2
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN4
  *         @arg @ref MD_OPAMP_OPA_INPS_VDIV
  *         @arg @ref MD_OPAMP_OPA_INPS_VREF
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN5
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN6
  *         @arg @ref MD_OPAMP_OPA_INPS_AIN7
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_inps(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_INPS) >> OPAMP_OPA_CTRL_OPA_INPS_POSS);
}

/**
  * @brief  OPAMP analog output switch Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_enable_opa_analog_output(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_AOEN);
}

/**
  * @brief  OPAMP analog output switch Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_disable_opa_analog_output(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_AOEN);
}

/**
  * @brief  Check if OPAMP analog output switch is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opamp_is_enabled_opa_analog_output(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_AOEN) == (OPAMP_OPA_CTRL_OPA_AOEN));
}

/**
  * @brief  Set OPAMP built-in capacitor usage setting
  * @param  opamp OPAMP Instance
  * @param  opa_caps This parameter can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_CAPS_SAMPLE
  *         @arg @ref MD_OPAMP_OPA_CAPS_INTEGRATE
  */
__STATIC_INLINE void md_opamp_set_opa_caps(OPAMP_TypeDef *opamp, uint32_t opa_caps)
{
    MODIFY_REG(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_CAPS, opa_caps << OPAMP_OPA_CTRL_OPA_CAPS_POS);
}

/**
  * @brief  Get OPAMP built-in capacitor usage setting
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_CAPS_SAMPLE
  *         @arg @ref MD_OPAMP_OPA_CAPS_INTEGRATE
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_caps(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_CAPS) >> OPAMP_OPA_CTRL_OPA_CAPS_POS);
}

/**
  * @brief  Set OPAMP Chopper Clock Selection
  * @param  opamp OPAMP Instance
  * @param  opa_chpcks This parameter can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_CHPCKS_1MHZ
  *         @arg @ref MD_OPAMP_OPA_CHPCKS_2MHZ
  */
__STATIC_INLINE void md_opamp_set_opa_chpcks(OPAMP_TypeDef *opamp, uint32_t opa_chpcks)
{
    MODIFY_REG(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_CHPCKS, opa_chpcks << OPAMP_OPA_CTRL_OPA_CHPCKS_POS);
}

/**
  * @brief  Get OPAMP Chopper Clock Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPAMP_OPA_CHPCKS_1MHZ
  *         @arg @ref MD_OPAMP_OPA_CHPCKS_2MHZ
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_chpcks(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_CHPCKS) >> OPAMP_OPA_CTRL_OPA_CHPCKS_POS);
}

/**
  * @brief  OPAMP chopper switch Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_enable_opa_chp(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_CHPEN);
}

/**
  * @brief  OPAMP chopper switch Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_disable_opa_chp(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_CHPEN);
}

/**
  * @brief  Check if OPAMP chopper switch is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opamp_is_enabled_opa_chp(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_CHPEN) == (OPAMP_OPA_CTRL_OPA_CHPEN));
}

/**
  * @brief  OPAMP switch Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_enable_opa(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_EN);
}

/**
  * @brief  OPAMP switch Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opamp_disable_opa(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_EN);
}

/**
  * @brief  Check if OPAMP switch is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opamp_is_enabled_opa(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA_CTRL, OPAMP_OPA_CTRL_OPA_EN) == (OPAMP_OPA_CTRL_OPA_EN));
}
/**
  * @}
  */

/** @defgroup MD_OPAMP_Public_Functions_Group3 OPA_DBC
  * @{
  */
/**
  * @brief  Set OPA_DBC Register
  * @param  opamp OPAMP Instance
  * @param  opa_dbc
  */
__STATIC_INLINE void md_opamp_set_opa_dbc(OPAMP_TypeDef *opamp, uint32_t opa_dbc)
{
    WRITE_REG(opamp->OPA_DBC, opa_dbc);
}

/**
  * @brief  Get OPA_DBC Register
  * @param  opamp OPAMP Instance
  * @retval opa_dbc Values
  */
__STATIC_INLINE uint32_t md_opamp_get_opa_dbc(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA_DBC);
}

/**
  * @brief  Set Bounce Elimination Counter
  * @param  opamp OPAMP Instance
  * @param  dbcnt This parameter can be one of the following values:
  *         @arg @ref MD_OPAMP_DBCNT_IMMEDIATELY
  *         @arg @ref MD_OPAMP_DBCNT_SAME_2_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_3_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_4_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_5_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_6_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_7_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_8_TIMES
  */
__STATIC_INLINE void md_opamp_set_bounce_counter(OPAMP_TypeDef *opamp, uint32_t dbcnt)
{
    MODIFY_REG(opamp->OPA_DBC, OPAMP_OPA_DBC_DBCNT, dbcnt << OPAMP_OPA_DBC_DBCNT_POSS);
}

/**
  * @brief  Get Bounce Elimination Counter
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPAMP_DBCNT_IMMEDIATELY
  *         @arg @ref MD_OPAMP_DBCNT_SAME_2_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_3_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_4_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_5_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_6_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_7_TIMES
  *         @arg @ref MD_OPAMP_DBCNT_SAME_8_TIMES
  */
__STATIC_INLINE uint32_t md_opamp_get_bounce_counter(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_DBC, OPAMP_OPA_DBC_DBCNT) >> OPAMP_OPA_DBC_DBCNT_POSS);
}

/**
  * @brief  Set Bounce Elimination Prescaler
  * @param  opamp OPAMP Instance
  * @param  dbpre Prescaler
  */
__STATIC_INLINE void md_opamp_set_bounce_prescaler(OPAMP_TypeDef *opamp, uint32_t dbpre)
{
    MODIFY_REG(opamp->OPA_DBC, OPAMP_OPA_DBC_DBPRE, dbpre << OPAMP_OPA_DBC_DBPRE_POSS);
}

/**
  * @brief  Get Bounce Elimination Prescaler
  * @param  opamp OPAMP Instance
  * @retval Prescaler
  */
__STATIC_INLINE uint32_t md_opamp_get_bounce_prescaler(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA_DBC, OPAMP_OPA_DBC_DBPRE) >> OPAMP_OPA_DBC_DBPRE_POSS);
}
/**
  * @}
  */

/**
  * @}
  */


/**
  * @} Micro_Driver
  */
void md_opamp_cofig(OPAMP_TypeDef *OPAMPx, md_opamp_config_typedef *config);
void md_opamp_power_init(RCU_TypeDef *rcu, ANPWR_TypeDef *anpwr, md_opamp_power_init_typedef *anpwr_power_init);
void md_opamp_init(OPAMP_TypeDef *OPAMPx, md_opamp_power_init_typedef *anpwr_power_init, md_opamp_config_typedef *config);
#ifdef __cplusplus
}
#endif
#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
