/**********************************************************************************
 *
 * @file    md_lcd.h
 * @brief   header file of md_lcd.c
 *
 * @date    8 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          8 Aug 2022      Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_LCD_H__
#define __MD_LCD_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include "es32f0943.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_LCD LCD
  * @brief LCD micro driver
  * @{
  */

/** @defgroup MD_OPAMP_Public_Macros OPAMP Public Macros
  * @{
  */

#define MD_LED_POLA_SEL_SEG_POSITIVE_COM_POSITIVE     (0x00000000U)
#define MD_LED_POLA_SEL_SEG_POSITIVE_COM_REVERSE      (0x00000001U)
#define MD_LED_POLA_SEL_SEG_REVERSE_COM_POSITIVE      (0x00000002U)
#define MD_LED_POLA_SEL_SEG_REVERSE_COM_REVERSE       (0x00000003U)

#define MD_LED_GPIO_ANALOG      (0x00000000U)
#define MD_LED_GPIO_DIGITAL     (0x00000001U)

#define MD_LED_QTSEL_1     (0x00000000U)
#define MD_LED_QTSEL_2     (0x00000001U)
#define MD_LED_QTSEL_3     (0x00000002U)
#define MD_LED_QTSEL_4     (0x00000003U)
#define MD_LED_QTSEL_5     (0x00000004U)
#define MD_LED_QTSEL_6     (0x00000005U)
#define MD_LED_QTSEL_7     (0x00000006U)
#define MD_LED_QTSEL_8     (0x00000007U)

#define MD_LED_RSEL_225K        (0x00000000U)
#define MD_LED_RSEL_900K        (0x00000001U)
#define MD_LED_RSEL_60K_1       (0x00000002U)
#define MD_LED_RSEL_60K_2       (0x00000003U)
#define MD_LED_RSEL_225K_60K    (0x00000004U)
#define MD_LED_RSEL_900K_60K    (0x00000005U)

#define MD_VLCD_SRC_CHARGE_PUMP     (0x00000000U)
#define MD_VLCD_SRC_EXTERNAL        (0x00000001U)

#define MD_LCD_TYPE_A     (0x00000000U)
#define MD_LCD_TYPE_B     (0x00000001U)

#define MD_LCD_BIAS_STATIC      (0x00000000U)
#define MD_LCD_BIAS_1D2         (0x00000001U)
#define MD_LCD_BIAS_1D3         (0x00000002U)
#define MD_LCD_BIAS_1D4         (0x00000003U)

#define MD_LCD_DUTY_1D4         (0x00000000U)
#define MD_LCD_DUTY_1D6         (0x00000001U)

#define MD_LCD_WSBLINK_CLOSE         (0x00000000U)
#define MD_LCD_WSBLINK_MODE1         (0x00000001U)
#define MD_LCD_WSBLINK_MODE2         (0x00000002U)
#define MD_LCD_WSBLINK_MODE3         (0x00000003U)

#define MD_LCD_UPDCTRL_SWUPDATE_1_NEXT     (0x00000000U)
#define MD_LCD_UPDCTRL_EVERY_FRAME         (0x00000001U)
#define MD_LCD_UPDCTRL_SWUPDATE_0_NEXT     (0x00000002U)


typedef struct
{
    uint32_t lcd_duty;
    uint32_t lcd_bias;
    uint32_t lcd_type;
    uint32_t lcd_frdiv;
    uint32_t lcd_updctrl;
} md_lcd_init_typedef;

typedef struct
{
    uint32_t lcd_clocksource;
    uint32_t lcd_clockdiv;
    uint32_t vlcd_clocksource;
    uint32_t vlcd_clockdiv;
    uint32_t vlcd_voltage;
} md_lcd_power_init_typedef;
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions LCD Public Functions
  * @{
  */

/** @defgroup MD_LCD_Public_Functions_Group2 LCD_MODE
  * @{
  */

/**
  * @brief  Set LCD_MODE Register
  * @param  lcd LCD Instance
  * @param  mode
  */
__STATIC_INLINE void md_lcd_set_lcd_mode(LCD_TypeDef *lcd, uint32_t mode)
{
    WRITE_REG(lcd->MODE, mode);
}

/**
  * @brief  Get LCD_MODE Register
  * @param  lcd LCD Instance
  * @retval mode Values
  */
__STATIC_INLINE uint32_t md_lcd_get_lcd_mode(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->MODE);
}

/**
  * @brief  Set LCD GPIO polarity selection
  * @param  lcd LCD Instance
  * @param  pola_sel This parameter can be one of the following values:
  *         @arg @ref MD_LED_POLA_SEL_SEG_POSITIVE_COM_POSITIVE
  *         @arg @ref MD_LED_POLA_SEL_SEG_POSITIVE_COM_REVERSE
  *         @arg @ref MD_LED_POLA_SEL_SEG_REVERSE_COM_POSITIVE
  *         @arg @ref MD_LED_POLA_SEL_SEG_REVERSE_COM_REVERSE
  */
__STATIC_INLINE void md_lcd_set_gpio_polarity(LCD_TypeDef *lcd, uint32_t pola_sel)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_POLA_SEL, pola_sel << LCD_MODE_POLA_SEL_POSS);
}

/**
  * @brief  Get LCD GPIO polarity selection
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LED_POLA_SEL_SEG_POSITIVE_COM_POSITIVE
  *         @arg @ref MD_LED_POLA_SEL_SEG_POSITIVE_COM_REVERSE
  *         @arg @ref MD_LED_POLA_SEL_SEG_REVERSE_COM_POSITIVE
  *         @arg @ref MD_LED_POLA_SEL_SEG_REVERSE_COM_REVERSE
  */
__STATIC_INLINE uint32_t md_lcd_get_gpio_polarity(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->MODE, LCD_MODE_POLA_SEL) >> LCD_MODE_POLA_SEL_POSS);
}

/**
  * @brief  Set LCD GPIO mode selection
  * @param  lcd LCD Instance
  * @param  lcd_gpio This parameter can be one of the following values:
  *         @arg @ref MD_LED_GPIO_ANALOG
  *         @arg @ref MD_LED_GPIO_DIGITAL
  */
__STATIC_INLINE void md_lcd_set_gpio(LCD_TypeDef *lcd, uint32_t lcd_gpio)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_LCD_GPIO, lcd_gpio << LCD_MODE_LCD_GPIO_POS);
}

/**
  * @brief  Get LCD GPIO mode selection
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LED_GPIO_ANALOG
  *         @arg @ref MD_LED_GPIO_DIGITAL
  */
__STATIC_INLINE uint32_t md_lcd_get_gpio(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->MODE, LCD_MODE_LCD_GPIO) >> LCD_MODE_LCD_GPIO_POS);
}

/**
  * @brief  Set LCD quick charge time selection
  * @param  lcd LCD Instance
  * @param  qtsel This parameter can be one of the following values:
  *         @arg @ref MD_LED_QTSEL_1
  *         @arg @ref MD_LED_QTSEL_2
  *         @arg @ref MD_LED_QTSEL_3
  *         @arg @ref MD_LED_QTSEL_4
  *         @arg @ref MD_LED_QTSEL_5
  *         @arg @ref MD_LED_QTSEL_6
  *         @arg @ref MD_LED_QTSEL_7
  *         @arg @ref MD_LED_QTSEL_8
  */
__STATIC_INLINE void md_lcd_set_charge_time(LCD_TypeDef *lcd, uint32_t qtsel)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_QTSEL, qtsel << LCD_MODE_QTSEL_POSS);
}

/**
  * @brief  Get LCD quick charge time selection
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LED_QTSEL_1
  *         @arg @ref MD_LED_QTSEL_2
  *         @arg @ref MD_LED_QTSEL_3
  *         @arg @ref MD_LED_QTSEL_4
  *         @arg @ref MD_LED_QTSEL_5
  *         @arg @ref MD_LED_QTSEL_6
  *         @arg @ref MD_LED_QTSEL_7
  *         @arg @ref MD_LED_QTSEL_8
  */
__STATIC_INLINE uint32_t md_lcd_get_charge_time(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->MODE, LCD_MODE_QTSEL) >> LCD_MODE_QTSEL_POSS);
}

/**
  * @brief  Set LCD Bias Resistor Selection
  * @param  lcd LCD Instance
  * @param  rsel This parameter can be one of the following values:
  *         @arg @ref MD_LED_RSEL_225K
  *         @arg @ref MD_LED_RSEL_900K
  *         @arg @ref MD_LED_RSEL_60K_1
  *         @arg @ref MD_LED_RSEL_60K_2
  *         @arg @ref MD_LED_RSEL_225K_60K
  *         @arg @ref MD_LED_RSEL_900K_60K
  */
__STATIC_INLINE void md_lcd_set_bias_resistor(LCD_TypeDef *lcd, uint32_t rsel)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_RSEL, rsel << LCD_MODE_RSEL_POSS);
}

/**
  * @brief  Get LCD Bias Resistor Selection
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LED_RSEL_225K
  *         @arg @ref MD_LED_RSEL_900K
  *         @arg @ref MD_LED_RSEL_60K_1
  *         @arg @ref MD_LED_RSEL_60K_2
  *         @arg @ref MD_LED_RSEL_225K_60K
  *         @arg @ref MD_LED_RSEL_900K_60K
  */
__STATIC_INLINE uint32_t md_lcd_get_bias_resistor(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->MODE, LCD_MODE_RSEL) >> LCD_MODE_RSEL_POSS);
}

/**
  * @brief  Set VLCD voltage selection
  * @param  lcd LCD Instance
  * @param  vlcd_sel
  */
__STATIC_INLINE void md_lcd_set_vlcd_voltage(LCD_TypeDef *lcd, uint32_t vlcd_sel)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_VLCD_SEL, vlcd_sel << LCD_MODE_VLCD_SEL_POSS);
}

/**
  * @brief  Get VLCD voltage selection
  * @param  lcd LCD Instance
  * @retval vlcd_sel
  */
__STATIC_INLINE uint32_t md_lcd_get_vlcd_voltage(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->MODE, LCD_MODE_VLCD_SEL) >> LCD_MODE_VLCD_SEL_POSS);
}

/**
  * @brief  Set VLCD voltage source selection
  * @param  lcd LCD Instance
  * @param  vlcd_src This parameter can be one of the following values:
  *         @arg @ref MD_VLCD_SRC_CHARGE_PUMP
  *         @arg @ref MD_VLCD_SRC_EXTERNAL
  */
__STATIC_INLINE void md_lcd_set_vlcd_src(LCD_TypeDef *lcd, uint32_t vlcd_src)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_VLCD_SRC, vlcd_src << LCD_MODE_VLCD_SRC_POS);
}

/**
  * @brief  Get VLCD voltage source selection
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_VLCD_SRC_CHARGE_PUMP
  *         @arg @ref MD_VLCD_SRC_EXTERNAL
  */
__STATIC_INLINE uint32_t md_lcd_get_vlcd_src(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->MODE, LCD_MODE_VLCD_SRC) >> LCD_MODE_VLCD_SRC_POS);
}

/**
  * @brief  output buffer Enable.
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE void md_lcd_enable_buffer(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->MODE, LCD_MODE_BUFEN);
}

/**
  * @brief  output buffer Disable.
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE void md_lcd_disable_buffer(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->MODE, LCD_MODE_BUFEN);
}

/**
  * @brief  Check if output buffer is Enabled.
  * @param  lcd LCD Instances
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_buffer(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->MODE, LCD_MODE_BUFEN) == (LCD_MODE_BUFEN));
}

/**
  * @brief  Set LCD operating frequency (LCD_PHY_CLK) output type selection
  * @param  lcd LCD Instance
  * @param  type This parameter can be one of the following values:
  *         @arg @ref MD_LCD_TYPE_A
  *         @arg @ref MD_LCD_TYPE_B
  */
__STATIC_INLINE void md_lcd_set_type(LCD_TypeDef *lcd, uint32_t type)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_TYPE, type << LCD_MODE_TYPE_POS);
}

/**
  * @brief  Get LCD operating frequency (LCD_PHY_CLK) output type selection
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LCD_TYPE_A
  *         @arg @ref MD_LCD_TYPE_B
  */
__STATIC_INLINE uint32_t md_lcd_get_type(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->MODE, LCD_MODE_TYPE) >> LCD_MODE_TYPE_POS);
}

/**
  * @brief  Set LCD Bias Selection
  * @param  lcd LCD Instance
  * @param  bias This parameter can be one of the following values:
  *         @arg @ref MD_LCD_BIAS_STATIC
  *         @arg @ref MD_LCD_BIAS_1D2
  *         @arg @ref MD_LCD_BIAS_1D3
  *         @arg @ref MD_LCD_BIAS_1D4
  */
__STATIC_INLINE void md_lcd_set_bias(LCD_TypeDef *lcd, uint32_t bias)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_BIAS, bias << LCD_MODE_BIAS_POSS);
}

/**
  * @brief  Get LCD Bias Selection
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LCD_BIAS_STATIC
  *         @arg @ref MD_LCD_BIAS_1D2
  *         @arg @ref MD_LCD_BIAS_1D3
  *         @arg @ref MD_LCD_BIAS_1D4
  */
__STATIC_INLINE uint32_t md_lcd_get_bias(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->MODE, LCD_MODE_BIAS) >> LCD_MODE_BIAS_POSS);
}

/**
  * @brief  Set LCD Bias Selection
  * @param  lcd LCD Instance
  * @param  duty This parameter can be one of the following values:
  *         @arg @ref MD_LCD_DUTY_1D4
  *         @arg @ref MD_LCD_DUTY_1D6
  */
__STATIC_INLINE void md_lcd_set_duty(LCD_TypeDef *lcd, uint32_t duty)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_DUTY, duty << LCD_MODE_DUTY_POS);
}

/**
  * @brief  Get LCD Bias Selection
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LCD_DUTY_1D4
  *         @arg @ref MD_LCD_DUTY_1D6
  */
__STATIC_INLINE uint32_t md_lcd_get_duty(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->MODE, LCD_MODE_DUTY) >> LCD_MODE_DUTY_POS);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group3 LCD_CTRL
  * @{
  */

/**
  * @brief  Set LCD_CTRL Register
  * @param  lcd LCD Instance
  * @param  ctrl
  */
__STATIC_INLINE void md_lcd_set_lcd_ctrl(LCD_TypeDef *lcd, uint32_t ctrl)
{
    WRITE_REG(lcd->CTRL, ctrl);
}

/**
  * @brief  Get LCD_CTRL Register
  * @param  lcd LCD Instance
  * @retval ctrl Values
  */
__STATIC_INLINE uint32_t md_lcd_get_lcd_ctrl(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->CTRL);
}

/**
  * @brief  Get LCD voltage steady state flag
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_lcd_is_active_flag_vlcd_ready(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_VLCD_RDY) == (LCD_CTRL_VLCD_RDY));
}

/**
  * @brief  Set Full frequency flicker option
  * @param  lcd LCD Instance
  * @param  wsblink This parameter can be one of the following values:
  *         @arg @ref MD_LCD_WSBLINK_CLOSE
  *         @arg @ref MD_LCD_WSBLINK_MODE1
  *         @arg @ref MD_LCD_WSBLINK_MODE2
  *         @arg @ref MD_LCD_WSBLINK_MODE3
  */
__STATIC_INLINE void md_lcd_set_wsblink(LCD_TypeDef *lcd, uint32_t wsblink)
{
    MODIFY_REG(lcd->CTRL, LCD_CTRL_WSBLINK, wsblink << LCD_CTRL_WSBLINK_POSS);
}

/**
  * @brief  Get Full frequency flicker option
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LCD_WSBLINK_CLOSE
  *         @arg @ref MD_LCD_WSBLINK_MODE1
  *         @arg @ref MD_LCD_WSBLINK_MODE2
  *         @arg @ref MD_LCD_WSBLINK_MODE3
  */
__STATIC_INLINE uint32_t md_lcd_get_wsblink(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->CTRL, LCD_CTRL_WSBLINK) >> LCD_CTRL_WSBLINK_POSS);
}

/**
  * @brief  Set Frame counter value
  * @param  lcd LCD Instance
  * @param  fcvalue
  */
__STATIC_INLINE void md_lcd_set_frame_counter(LCD_TypeDef *lcd, uint32_t fcvalue)
{
    MODIFY_REG(lcd->CTRL, LCD_CTRL_FCVALUE, fcvalue << LCD_CTRL_FCVALUE_POSS);
}

/**
  * @brief  Get Frame counter value
  * @param  lcd LCD Instance
  * @retval fcvalue
  */
__STATIC_INLINE uint32_t md_lcd_get_frame_counter(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->CTRL, LCD_CTRL_FCVALUE) >> LCD_CTRL_FCVALUE_POSS);
}

/**
  * @brief  Frame counter Enable.
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE void md_lcd_enable_frame_counter(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->CTRL, LCD_CTRL_FCCTRL);
}

/**
  * @brief  Frame counter Disable.
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE void md_lcd_disable_frame_counter(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->CTRL, LCD_CTRL_FCCTRL);
}

/**
  * @brief  Check if Frame counter is Enabled.
  * @param  lcd LCD Instances
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_frame_counter(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_FCCTRL) == (LCD_CTRL_FCCTRL));
}

/**
  * @brief  Software generated update event.
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE void md_lcd_clear_flag_swupdate(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->CTRL, LCD_CTRL_SWUPDATE);
}

/**
  * @brief  Set Update control selection
  * @param  lcd LCD Instance
  * @param  updctrl This parameter can be one of the following values:
  *         @arg @ref MD_LCD_UPDCTRL_SWUPDATE_1_NEXT
  *         @arg @ref MD_LCD_UPDCTRL_EVERY_FRAME
  *         @arg @ref MD_LCD_UPDCTRL_SWUPDATE_0_NEXT
  */
__STATIC_INLINE void md_lcd_set_update_control(LCD_TypeDef *lcd, uint32_t updctrl)
{
    MODIFY_REG(lcd->CTRL, LCD_CTRL_UPDCTRL, updctrl << LCD_CTRL_UPDCTRL_POSS);
}

/**
  * @brief  Get Update control selection
  * @param  lcd LCD Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LCD_UPDCTRL_SWUPDATE_1_NEXT
  *         @arg @ref MD_LCD_UPDCTRL_EVERY_FRAME
  *         @arg @ref MD_LCD_UPDCTRL_SWUPDATE_0_NEXT
  */
__STATIC_INLINE uint32_t md_lcd_get_update_control(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->CTRL, LCD_CTRL_UPDCTRL) >> LCD_CTRL_UPDCTRL_POSS);
}

/**
  * @brief  Set LCD operating frequency (LCD_PHY_CLK) division selection
  * @param  lcd LCD Instance
  * @param  frdiv
  */
__STATIC_INLINE void md_lcd_set_division(LCD_TypeDef *lcd, uint32_t frdiv)
{
    MODIFY_REG(lcd->CTRL, LCD_CTRL_FRDIV, frdiv << LCD_CTRL_FRDIV_POSS);
}

/**
  * @brief  Get LCD operating frequency (LCD_PHY_CLK) division selection
  * @param  lcd LCD Instance
  * @retval frdiv
  */
__STATIC_INLINE uint32_t md_lcd_get_division(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->CTRL, LCD_CTRL_FRDIV) >> LCD_CTRL_FRDIV_POSS);
}

/**
  * @brief  LCD control Enable.
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE void md_lcd_enable_led(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->CTRL, LCD_CTRL_LCDEN);
}

/**
  * @brief  LCD control Disable.
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE void md_lcd_disable_led(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->CTRL, LCD_CTRL_LCDEN);
}

/**
  * @brief  Check if LCD control is Enabled.
  * @param  lcd LCD Instances
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_led(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_LCDEN) == (LCD_CTRL_LCDEN));
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group4 LCD_BLINK1
  * @{
  */

/**
  * @brief  Set LCD_BLINK1 Register
  * @param  lcd LCD Instance
  * @param  blink1
  */
__STATIC_INLINE void md_lcd_set_lcd_blink1(LCD_TypeDef *lcd, uint32_t blink1)
{
    WRITE_REG(lcd->BLINK1, blink1);
}

/**
  * @brief  Get LCD_BLINK1 Register
  * @param  lcd LCD Instance
  * @retval blink1 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_lcd_blink1(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->BLINK1);
}

/**
  * @brief  Set COM3 flash selection
  * @param  lcd LCD Instance
  * @param  com3
  */
__STATIC_INLINE void md_lcd_set_com3(LCD_TypeDef *lcd, uint32_t com3)
{
    MODIFY_REG(lcd->BLINK1, LCD_BLINK1_COM3, com3 << LCD_BLINK1_COM3_POSS);
}

/**
  * @brief  Get COM3 flash selection
  * @param  lcd LCD Instance
  * @retval com3
  */
__STATIC_INLINE uint32_t md_lcd_get_com3(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->BLINK1, LCD_BLINK1_COM3) >> LCD_BLINK1_COM3_POSS);
}

/**
  * @brief  Set COM2 flash selection
  * @param  lcd LCD Instance
  * @param  com2
  */
__STATIC_INLINE void md_lcd_set_com2(LCD_TypeDef *lcd, uint32_t com2)
{
    MODIFY_REG(lcd->BLINK1, LCD_BLINK1_COM2, com2 << LCD_BLINK1_COM2_POSS);
}

/**
  * @brief  Get COM2 flash selection
  * @param  lcd LCD Instance
  * @retval com2
  */
__STATIC_INLINE uint32_t md_lcd_get_com2(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->BLINK1, LCD_BLINK1_COM2) >> LCD_BLINK1_COM2_POSS);
}

/**
  * @brief  Set COM1 flash selection
  * @param  lcd LCD Instance
  * @param  com1
  */
__STATIC_INLINE void md_lcd_set_com1(LCD_TypeDef *lcd, uint32_t com1)
{
    MODIFY_REG(lcd->BLINK1, LCD_BLINK1_COM1, com1 << LCD_BLINK1_COM1_POSS);
}

/**
  * @brief  Get COM1 flash selection
  * @param  lcd LCD Instance
  * @retval com1
  */
__STATIC_INLINE uint32_t md_lcd_get_com1(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->BLINK1, LCD_BLINK1_COM1) >> LCD_BLINK1_COM1_POSS);
}

/**
  * @brief  Set COM0 flash selection
  * @param  lcd LCD Instance
  * @param  com0
  */
__STATIC_INLINE void md_lcd_set_com0(LCD_TypeDef *lcd, uint32_t com0)
{
    MODIFY_REG(lcd->BLINK1, LCD_BLINK1_COM0, com0 << LCD_BLINK1_COM0_POSS);
}

/**
  * @brief  Get COM0 flash selection
  * @param  lcd LCD Instance
  * @retval com0
  */
__STATIC_INLINE uint32_t md_lcd_get_com0(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->BLINK1, LCD_BLINK1_COM0) >> LCD_BLINK1_COM0_POSS);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group5 LCD_BLINK2
  * @{
  */

/**
  * @brief  Set LCD_BLINK2 Register
  * @param  lcd LCD Instance
  * @param  blink2
  */
__STATIC_INLINE void md_lcd_set_lcd_blink2(LCD_TypeDef *lcd, uint32_t blink2)
{
    WRITE_REG(lcd->BLINK2, blink2);
}

/**
  * @brief  Get LCD_BLINK2 Register
  * @param  lcd LCD Instance
  * @retval blink2 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_lcd_blink2(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->BLINK2);
}

/**
  * @brief  Set COM5 flash selection
  * @param  lcd LCD Instance
  * @param  com5
  */
__STATIC_INLINE void md_lcd_set_com5(LCD_TypeDef *lcd, uint32_t com5)
{
    MODIFY_REG(lcd->BLINK2, LCD_BLINK2_COM5, com5 << LCD_BLINK2_COM5_POSS);
}

/**
  * @brief  Get COM5 flash selection
  * @param  lcd LCD Instance
  * @retval com5
  */
__STATIC_INLINE uint32_t md_lcd_get_com5(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->BLINK2, LCD_BLINK2_COM5) >> LCD_BLINK2_COM5_POSS);
}

/**
  * @brief  Set COM4 flash selection
  * @param  lcd LCD Instance
  * @param  com4
  */
__STATIC_INLINE void md_lcd_set_com4(LCD_TypeDef *lcd, uint32_t com4)
{
    MODIFY_REG(lcd->BLINK2, LCD_BLINK2_COM4, com4 << LCD_BLINK2_COM4_POSS);
}

/**
  * @brief  Get COM4 flash selection
  * @param  lcd LCD Instance
  * @retval com4
  */
__STATIC_INLINE uint32_t md_lcd_get_com4(LCD_TypeDef *lcd)
{
    return (uint32_t)(READ_BIT(lcd->BLINK2, LCD_BLINK2_COM4) >> LCD_BLINK2_COM4_POSS);
}

/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group6 LCD_IER
  * @{
  */
/**
  * @brief  Enable update completion interrupt function
  * @param  lcd LCD Instance
  * @retval update.
  */
__STATIC_INLINE void md_lcd_enable_it_update(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->IER, LCD_IER_UPDATE);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group7 LCD_IDR
  * @{
  */
/**
  * @brief  Disable update completion interrupt function
  * @param  lcd LCD Instance
  * @retval update.
  */
__STATIC_INLINE void md_lcd_disable_it_update(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->IDR, LCD_IDR_UPDATE);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group8 LCD_IVS
  * @{
  */
/**
  * @brief  Update complete interrupt function on/off status
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_it_update(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->ICR, LCD_IVS_UPDATE) == (LCD_IVS_UPDATE));
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group9 LCD_RIF
  * @{
  */
/**
  * @brief  Update complete unmasked interrupt flag status
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_lcd_is_active_it_update(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->RIF, LCD_RIF_UPDATE) == (LCD_RIF_UPDATE));
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group10 LCD_IFM
  * @{
  */
/**
  * @brief  Update complete interrupt flag mask status
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_lcd_is_masked_it_update(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->IFM, LCD_IFM_UPDATE) == (LCD_IFM_UPDATE));
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group11 LCD_ICR
  * @{
  */
/**
  * @brief  Disable update completion interrupt function
  * @param  lcd LCD Instance
  * @retval update.
  */
__STATIC_INLINE void md_lcd_clear_it_update(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->ICR, LCD_ICR_UPDATE);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group12 LCD_BUSY
  * @{
  */

/**
  * @brief  update status
  * @param  lcd LCD Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_lcd_is_active_flag_stat_update(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->BUSY, LCD_BUSY_UPDATE) == (LCD_BUSY_UPDATE));
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group13 LCD_RAM0
  * @{
  */

/**
  * @brief  Set LCD_RAM0 Register
  * @param  lcd LCD Instance
  * @param  ram0
  */
__STATIC_INLINE void md_lcd_set_ram0(LCD_TypeDef *lcd, uint32_t ram0)
{
    WRITE_REG(lcd->RAM0, ram0);
}

/**
  * @brief  Get LCD_RAM0 Register
  * @param  lcd LCD Instance
  * @retval ram0 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram0(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM0);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group14 LCD_RAM1
  * @{
  */

/**
  * @brief  Set LCD_RAM1 Register
  * @param  lcd LCD Instance
  * @param  ram1
  */
__STATIC_INLINE void md_lcd_set_ram1(LCD_TypeDef *lcd, uint32_t ram1)
{
    WRITE_REG(lcd->RAM1, ram1);
}

/**
  * @brief  Get LCD_RAM1 Register
  * @param  lcd LCD Instance
  * @retval ram1 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram1(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM1);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group15 LCD_RAM2
  * @{
  */

/**
  * @brief  Set LCD_RAM2 Register
  * @param  lcd LCD Instance
  * @param  ram2
  */
__STATIC_INLINE void md_lcd_set_ram2(LCD_TypeDef *lcd, uint32_t ram2)
{
    WRITE_REG(lcd->RAM2, ram2);
}

/**
  * @brief  Get LCD_RAM2 Register
  * @param  lcd LCD Instance
  * @retval ram2 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram2(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM2);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group16 LCD_RAM3
  * @{
  */

/**
  * @brief  Set LCD_RAM3 Register
  * @param  lcd LCD Instance
  * @param  ram3
  */
__STATIC_INLINE void md_lcd_set_ram3(LCD_TypeDef *lcd, uint32_t ram3)
{
    WRITE_REG(lcd->RAM3, ram3);
}

/**
  * @brief  Get LCD_RAM3 Register
  * @param  lcd LCD Instance
  * @retval ram3 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram3(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM3);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group17 LCD_RAM4
  * @{
  */

/**
  * @brief  Set LCD_RAM4 Register
  * @param  lcd LCD Instance
  * @param  ram4
  */
__STATIC_INLINE void md_lcd_set_ram4(LCD_TypeDef *lcd, uint32_t ram4)
{
    WRITE_REG(lcd->RAM4, ram4);
}

/**
  * @brief  Get LCD_RAM4 Register
  * @param  lcd LCD Instance
  * @retval ram0 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram4(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM4);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group18 LCD_RAM5
  * @{
  */

/**
  * @brief  Set LCD_RAM5 Register
  * @param  lcd LCD Instance
  * @param  ram5
  */
__STATIC_INLINE void md_lcd_set_ram5(LCD_TypeDef *lcd, uint32_t ram5)
{
    WRITE_REG(lcd->RAM5, ram5);
}

/**
  * @brief  Get LCD_RAM5 Register
  * @param  lcd LCD Instance
  * @retval ram0 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram5(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM5);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group19 LCD_RAM6
  * @{
  */

/**
  * @brief  Set LCD_RAM6 Register
  * @param  lcd LCD Instance
  * @param  ram6
  */
__STATIC_INLINE void md_lcd_set_ram6(LCD_TypeDef *lcd, uint32_t ram6)
{
    WRITE_REG(lcd->RAM6, ram6);
}

/**
  * @brief  Get LCD_RAM6 Register
  * @param  lcd LCD Instance
  * @retval ram0 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram6(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM6);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group20 LCD_RAM7
  * @{
  */

/**
  * @brief  Set LCD_RAM7 Register
  * @param  lcd LCD Instance
  * @param  ram7
  */
__STATIC_INLINE void md_lcd_set_ram7(LCD_TypeDef *lcd, uint32_t ram7)
{
    WRITE_REG(lcd->RAM7, ram7);
}

/**
  * @brief  Get LCD_RAM7 Register
  * @param  lcd LCD Instance
  * @retval ram0 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram7(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM7);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group21 LCD_RAM8
  * @{
  */

/**
  * @brief  Set LCD_RAM8 Register
  * @param  lcd LCD Instance
  * @param  ram8
  */
__STATIC_INLINE void md_lcd_set_ram8(LCD_TypeDef *lcd, uint32_t ram8)
{
    WRITE_REG(lcd->RAM8, ram8);
}

/**
  * @brief  Get LCD_RAM8 Register
  * @param  lcd LCD Instance
  * @retval ram8 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram8(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM8);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group22 LCD_RAM9
  * @{
  */

/**
  * @brief  Set LCD_RAM9 Register
  * @param  lcd LCD Instance
  * @param  ram9
  */
__STATIC_INLINE void md_lcd_set_ram9(LCD_TypeDef *lcd, uint32_t ram9)
{
    WRITE_REG(lcd->RAM9, ram9);
}

/**
  * @brief  Get LCD_RAM9 Register
  * @param  lcd LCD Instance
  * @retval ram9 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram9(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM9);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group23 LCD_RAM10
  * @{
  */

/**
  * @brief  Set LCD_RAM10 Register
  * @param  lcd LCD Instance
  * @param  ram10
  */
__STATIC_INLINE void md_lcd_set_ram10(LCD_TypeDef *lcd, uint32_t ram10)
{
    WRITE_REG(lcd->RAM10, ram10);
}

/**
  * @brief  Get LCD_RAM10 Register
  * @param  lcd LCD Instance
  * @retval ram10 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram10(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM10);
}
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions_Group24 LCD_RAM11
  * @{
  */

/**
  * @brief  Set LCD_RAM11 Register
  * @param  lcd LCD Instance
  * @param  ram11
  */
__STATIC_INLINE void md_lcd_set_ram11(LCD_TypeDef *lcd, uint32_t ram11)
{
    WRITE_REG(lcd->RAM11, ram11);
}

/**
  * @brief  Get LCD_RAM11 Register
  * @param  lcd LCD Instance
  * @retval ram11 Values
  */
__STATIC_INLINE uint32_t md_lcd_get_ram11(LCD_TypeDef *lcd)
{
    return (uint32_t)READ_REG(lcd->RAM11);
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @} Micro_Driver
  */
void md_lcd_power_init(RCU_TypeDef *RCUx, LCD_TypeDef *LCDx, uint32_t lcd_ckcfg, uint32_t lcd_ckdiv, uint32_t vlcd_ckcfg, uint32_t vlcd_ckdiv, uint32_t vlcd_voltage);
void md_lcd_basic_init(LCD_TypeDef *LCDx, md_lcd_init_typedef *init);
void md_lcd_init(LCD_TypeDef *LCDx, md_lcd_power_init_typedef *power_init, md_lcd_init_typedef *init);
#ifdef __cplusplus
}
#endif
#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
