/**********************************************************************************
 *
 * @file    md_cmp.h
 * @brief   header file of md_cmp.c
 *
 * @date    28 Jul 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          28 Jul 2022     Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_CMP_H__
#define __MD_CMP_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include "es32f0943.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_CMP CMP
  * @brief CMP micro driver
  * @{
  */


/** @defgroup MD_CMP_Public_Macros CMP Public Macros
  * @{
  */
#define MD_CMP_CFG1_LOCK_READ_WRITE              (0x00000000UL)       /*!< @brief CMP_CFG1[31:0] bits can be read and written normally        */
#define MD_CMP_CFG1_LOCK_ONLY_READ               (0x00000001UL)       /*!< @brief The CMP_CFG1[31:0] bits are for read operations only.        */

#define MD_CMP_CFG1_VALUE_OUTPUT_LOW_LEVEL       (0x00000000UL)       /*!< @brief CMP output is low level.        */
#define MD_CMP_CFG1_VALUE_OUTPUT_HIGH_LEVEL      (0x00000001UL)       /*!< @brief CMP output is high level.        */

#define MD_CMP_CFG1_BLANKSEL_CLOSE               (0x00000000UL)       /*!< @brief 00000:Close        */
#define MD_CMP_CFG1_BLANKSEL_GP32C4T1_CH4        (0x00000001UL)       /*!< @brief xxxx1:GP32C4T1_CH4 */
#define MD_CMP_CFG1_BLANKSEL_GP16C2T1_CH2        (0x00000002UL)       /*!< @brief xxx1x:GP16C2T1_CH2 */
#define MD_CMP_CFG1_BLANKSEL_GP16C2T2_CH2        (0x00000004UL)       /*!< @brief xx1xx:GP16C2T2_CH2 */
#define MD_CMP_CFG1_BLANKSEL_GP16C2T3_CH2        (0x00000008UL)       /*!< @brief x1xxx:GP16C2T3_CH2 */
#define MD_CMP_CFG1_BLANKSEL_GP16C2T4_CH2        (0x00000010UL)       /*!< @brief 1xxxx:GP16C2T4_CH2 */

#define MD_CMP_CFG1_OUTSEL_NO                    (0x00000000UL)       /*!< @brief 000:NO            */
#define MD_CMP_CFG1_OUTSEL_GP16C2T1_BKIN         (0x00000002UL)       /*!< @brief 001:GP16C2T1_BKIN */
#define MD_CMP_CFG1_OUTSEL_GP16C2T2_BKIN         (0x00000003UL)       /*!< @brief 010:GP16C2T2_BKIN */
#define MD_CMP_CFG1_OUTSEL_GP16C2T3_BKIN         (0x00000004UL)       /*!< @brief 011:GP16C2T3_BKIN */
#define MD_CMP_CFG1_OUTSEL_GP16C2T4_BKIN         (0x00000005UL)       /*!< @brief 100:GP16C2T4_BKIN */

#define MD_CMP_CFG1_INPSEL_VBG                   (0x00000000UL)       /*!< @brief 000:VBG */
#define MD_CMP_CFG1_INPSEL_EXTERNAL              (0x00000001UL)       /*!< @brief 001:EXTERNAL */

#define MD_CMP_CFG1_INNSEL_MULTI_NODE            (0x00000000UL)       /*!< @brief 000:MULTI_NODE */
#define MD_CMP_CFG1_INNSEL_EXTERNAL              (0x00000001UL)       /*!< @brief 001:EXTERNAL */

#define MD_CMP_CFG2_FILT_CNT_IMMEDIATELY         (0x00000000UL)       /*!< @brief The output value immediately reflects the input state        */
#define MD_CMP_CFG2_FILT_CNT_TWICE               (0x00000001UL)       /*!< @brief The output value will not be changed until the same input state is obtained 2 times */
#define MD_CMP_CFG2_FILT_CNT_THREE_TIMES         (0x00000002UL)       /*!< @brief The output value will not be changed until the same input state is obtained 3 times */
#define MD_CMP_CFG2_FILT_CNT_FOUR_TIMES          (0x00000003UL)       /*!< @brief The output value will not be changed until the same input state is obtained 4 times */
#define MD_CMP_CFG2_FILT_CNT_FIVE_TIMES          (0x00000004UL)       /*!< @brief The output value will not be changed until the same input state is obtained 5 times */
#define MD_CMP_CFG2_FILT_CNT_SIX_TIMES           (0x00000005UL)       /*!< @brief The output value will not be changed until the same input state is obtained 6 times */
#define MD_CMP_CFG2_FILT_CNT_SEVEN_TIMES         (0x00000006UL)       /*!< @brief The output value will not be changed until the same input state is obtained 7 times */
#define MD_CMP_CFG2_FILT_CNT_EIGHT_TIMES         (0x00000007UL)       /*!< @brief The output value will not be changed until the same input state is obtained 8 times */

#define MD_CMP_CFG2_DM_DIABLE                    (0x00000000UL)       /*!< @brief Turn off masking  */
#define MD_CMP_CFG2_DM_DA0                       (0x00000001UL)       /*!< @brief xxx1: After configuration, CMP_OUT will replace DA[0]*/
#define MD_CMP_CFG2_DM_DA1                       (0x00000002UL)       /*!< @brief xx1x: After configuration, CMP_OUT will replace DA[1]*/
#define MD_CMP_CFG2_DM_DA2                       (0x00000004UL)       /*!< @brief x1xx: After configuration, CMP_OUT will replace DA[2]*/
#define MD_CMP_CFG2_DM_DA3                       (0x00000008UL)       /*!< @brief 1xxx: After configuration, CMP_OUT will replace DA[3]*/

#define MD_CMP_CFG2_DA_0                         (0x00000000UL)       /*!< @brief 0     */
#define MD_CMP_CFG2_DA_1_16                      (0x00000001UL)       /*!< @brief 1/16  */
#define MD_CMP_CFG2_DA_2_16                      (0x00000002UL)       /*!< @brief 2/16  */
#define MD_CMP_CFG2_DA_3_16                      (0x00000003UL)       /*!< @brief 3/16  */
#define MD_CMP_CFG2_DA_4_16                      (0x00000004UL)       /*!< @brief 4/16  */
#define MD_CMP_CFG2_DA_5_16                      (0x00000005UL)       /*!< @brief 5/16  */
#define MD_CMP_CFG2_DA_6_16                      (0x00000006UL)       /*!< @brief 6/16  */
#define MD_CMP_CFG2_DA_7_16                      (0x00000007UL)       /*!< @brief 7/16  */
#define MD_CMP_CFG2_DA_8_16                      (0x00000008UL)       /*!< @brief 8/16  */
#define MD_CMP_CFG2_DA_9_16                      (0x00000009UL)       /*!< @brief 9/16  */
#define MD_CMP_CFG2_DA_10_16                     (0x0000000AUL)       /*!< @brief 10/16 */
#define MD_CMP_CFG2_DA_11_16                     (0x0000000BUL)       /*!< @brief 11/16 */
#define MD_CMP_CFG2_DA_12_16                     (0x0000000CUL)       /*!< @brief 12/16 */
#define MD_CMP_CFG2_DA_13_16                     (0x0000000DUL)       /*!< @brief 13/16 */
#define MD_CMP_CFG2_DA_14_16                     (0x0000000EUL)       /*!< @brief 14/16 */
#define MD_CMP_CFG2_DA_15_16                     (0x0000000FUL)       /*!< @brief 15/16 */

#define MD_CMP_CFG2_PS_EXTERNAL                  (0x00000000UL)       /*!< @brief Select external input voltage */
#define MD_CMP_CFG2_PS_VDDH                      (0x00000001UL)       /*!< @brief Select VDDH voltage */
#define MD_CMP_CFG2_PS_VDDA                      (0x00000002UL)       /*!< @brief Select VDDA voltage */


typedef struct
{
    uint32_t blanking_source;
    uint32_t output_trigger_select;
    uint32_t input_negative_select;
    uint32_t input_postive_select;
    uint32_t filter_counter;
    uint32_t multinode_voltage_input_mask;
    uint32_t multinode_voltage_input;
    uint32_t multinode_voltage;
} md_cmp_config_typedef;
/**
  * @}
  */

/** @defgroup MD_CMP_Public_Functions CMP Public Functions
  * @{
  */

/** @defgroup MD_CMP_Public_Functions_Group2 CFG1
  * @{
  */
/**
  * @brief  Set CMP control register (CMP_CFG1)
  * @param  cmp CMP Instance
  * @param  value The value write in CMP_CFG1
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_cfg1(CMP_TypeDef *cmp, uint32_t value)
{
    WRITE_REG(cmp->CFG1, value);
}

/**
  * @brief  Get CMP control register (CMP_CFG1)
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_cmp_get_cfg1(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_REG(cmp->CFG1));
}

/**
  * @brief  Comparator locked W/R
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_enable_lock(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG1_LOCK);
}

/**
  * @brief  Get Comparator locked
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG1_LOCK_READ_WRITE
            @arg @ref MD_CMP_CFG1_LOCK_ONLY_READ
  */
__STATIC_INLINE uint32_t md_cmp_is_enabled_lock(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG1, CMP_CFG1_LOCK) == (CMP_CFG1_LOCK));
}

/**
  * @brief  Get Comparator output
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG1_VALUE_OUTPUT_LOW_LEVEL
            @arg @ref MD_CMP_CFG1_VALUE_OUTPUT_HIGH_LEVEL
  */
__STATIC_INLINE uint32_t md_cmp_get_output_level(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG1, CMP_CFG1_VALUE) >> CMP_CFG1_VALUE_POS);
}

/**
  * @brief  Set Comparator masking function
  * @param  cmp CMP Instance
  * @param  blanksel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG1_BLANKSEL_CLOSE
  *         @arg MD_CMP_CFG1_BLANKSEL_GP32C4T1_CH4
  *         @arg MD_CMP_CFG1_BLANKSEL_GP16C2T1_CH2
  *         @arg MD_CMP_CFG1_BLANKSEL_GP16C2T2_CH2
  *         @arg MD_CMP_CFG1_BLANKSEL_GP16C2T3_CH2
  *         @arg MD_CMP_CFG1_BLANKSEL_GP16C2T4_CH2
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_blanking_source(CMP_TypeDef *cmp, uint32_t blanksel)
{
    MODIFY_REG(cmp->CFG1, CMP_CFG1_BLANKSEL, blanksel << CMP_CFG1_BLANKSEL_POSS);
}

/**
  * @brief  Return Comparator masking function
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
  *         @arg MD_CMP_CFG1_BLANKSEL_CLOSE
  *         @arg MD_CMP_CFG1_BLANKSEL_GP32C4T1_CH4
  *         @arg MD_CMP_CFG1_BLANKSEL_GP16C2T1_CH2
  *         @arg MD_CMP_CFG1_BLANKSEL_GP16C2T2_CH2
  *         @arg MD_CMP_CFG1_BLANKSEL_GP16C2T3_CH2
  *         @arg MD_CMP_CFG1_BLANKSEL_GP16C2T4_CH2
  */
__STATIC_INLINE uint32_t md_cmp_get_blanking_source(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG1, CMP_CFG1_BLANKSEL) >> CMP_CFG1_BLANKSEL_POSS);
}

/**
  * @brief  Enable Comparator filter function
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_enable_filter(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG1_FILT);
}
/**
  * @brief  Disable Comparator filter function
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_disable_filter(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG1, CMP_CFG1_FILT);
}

/**
  * @brief  Check if Comparator filter function select is enable
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp_is_enabled_filter(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG1, CMP_CFG1_FILT) == (CMP_CFG1_FILT));
}

/**
  * @brief  Enable Comparator hysteresis function
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_enable_hysteresis(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG1_HYST);
}
/**
  * @brief  Disable Comparator hysteresis function
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_disable_hysteresis(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG1, CMP_CFG1_HYST);
}

/**
  * @brief  Check if Comparator hysteresis function
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp_is_enabled_hysteresis(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG1, CMP_CFG1_HYST) == (CMP_CFG1_HYST));
}

/**
  * @brief  Enable Comparator output polarity
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_enable_polarity(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG1_POL);
}

/**
  * @brief  Disable Comparator output polarity
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_disable_polarity(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG1, CMP_CFG1_POL);
}

/**
  * @brief  Check if Comparator output polarity
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp_is_enabled_polarity(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG1, CMP_CFG1_POL) == (CMP_CFG1_POL));
}

/**
  * @brief  Set Comparator output trigger selection
  * @param  cmp CMP Instance
  * @param  outsel
  * @param  outsel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG1_OUTSEL_NO
  *         @arg MD_CMP_CFG1_OUTSEL_GP16C2T1_BKIN
  *         @arg MD_CMP_CFG1_OUTSEL_GP16C2T2_BKIN
  *         @arg MD_CMP_CFG1_OUTSEL_GP16C2T3_BKIN
  *         @arg MD_CMP_CFG1_OUTSEL_GP16C2T4_BKIN
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_output_trigger_select(CMP_TypeDef *cmp, uint32_t outsel)
{
    MODIFY_REG(cmp->CFG1, CMP_CFG1_OUTSEL, outsel << CMP_CFG1_OUTSEL_POSS);
}

/**
  * @brief  Return Comparator output trigger selection
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
  *         @arg MD_CMP_CFG1_OUTSEL_NO
  *         @arg MD_CMP_CFG1_OUTSEL_GP16C2T1_BKIN
  *         @arg MD_CMP_CFG1_OUTSEL_GP16C2T2_BKIN
  *         @arg MD_CMP_CFG1_OUTSEL_GP16C2T3_BKIN
  *         @arg MD_CMP_CFG1_OUTSEL_GP16C2T4_BKIN
  */
__STATIC_INLINE uint32_t md_cmp_get_output_trigger_select(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG1, CMP_CFG1_OUTSEL) >> CMP_CFG1_OUTSEL_POSS);
}

/**
  * @brief  Set Comparator postive input selection
  * @param  cmp CMP Instance
  * @param  innsel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG1_INPSEL_VBG
  *         @arg MD_CMP_CFG1_INPSEL_EXTERNAL
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_input_postive_select(CMP_TypeDef *cmp, uint32_t innsel)
{
    MODIFY_REG(cmp->CFG1, CMP_CFG1_INPSEL, innsel << CMP_CFG1_INPSEL_POS);
}

/**
  * @brief  Return Comparator postive input selection
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
  *         @arg MD_CMP_CFG1_INPSEL_VBG
  *         @arg MD_CMP_CFG1_INPSEL_EXTERNAL
  */
__STATIC_INLINE uint32_t md_cmp_get_input_postive_select(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG1, CMP_CFG1_INPSEL) >> CMP_CFG1_INPSEL_POS);
}

/**
  * @brief  Set Comparator negative input selection
  * @param  cmp CMP Instance
  * @param  innsel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG1_INNSEL_MULTI_NODE
  *         @arg MD_CMP_CFG1_INNSEL_EXTERNAL
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_input_negative_select(CMP_TypeDef *cmp, uint32_t innsel)
{
    MODIFY_REG(cmp->CFG1, CMP_CFG1_INNSEL, innsel << CMP_CFG1_INNSEL_POS);
}

/**
  * @brief  Return Comparator negative input selection
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
  *         @arg MD_CMP_CFG1_INNSEL_MULTI_NODE
  *         @arg MD_CMP_CFG1_INNSEL_EXTERNAL
  */
__STATIC_INLINE uint32_t md_cmp_get_input_negative_select(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG1, CMP_CFG1_INNSEL) >> CMP_CFG1_INNSEL_POS);
}

/**
  * @brief  Enable Comparator switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_enable(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG1_EN);
}
/**
  * @brief  Disable Comparator switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_disable(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG1, CMP_CFG1_EN);
}

/**
  * @brief  Get Comparator switch
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp_is_enabled(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG1, CMP_CFG1_EN) == (CMP_CFG1_EN));
}
/**
  * @}
  */

/** @defgroup MD_CMP_Public_Functions_Group3 CFG2
  * @{
  */
/**
  * @brief  Set CMP control register (CMP_CFG2)
  * @param  cmp CMP Instance
  * @param  value The value write in CMP_CFG2
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_cfg2(CMP_TypeDef *cmp, uint32_t value)
{
    WRITE_REG(cmp->CFG2, value);
}

/**
  * @brief  Get CMP control register (CMP_CFG1)
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_cmp_get_cfg2(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_REG(cmp->CFG2));
}

/**
  * @brief  Set Comparator filter counter
  * @param  cmp CMP Instance
  * @param  cnt
            @arg @ref MD_CMP_CFG2_FILT_CNT_IMMEDIATELY
            @arg @ref MD_CMP_CFG2_FILT_CNT_TWICE
            @arg @ref MD_CMP_CFG2_FILT_CNT_THREE_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_FOUR_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_FIVE_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_SIX_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_SEVEN_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_EIGHT_TIMES
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_filter_counter(CMP_TypeDef *cmp, uint32_t cnt)
{
    MODIFY_REG(cmp->CFG2, CMP_CFG2_FILT_CNT, cnt << CMP_CFG2_FILT_CNT_POSS);
}

/**
  * @brief  Get Comparator filter counter
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG2_FILT_CNT_IMMEDIATELY
            @arg @ref MD_CMP_CFG2_FILT_CNT_TWICE
            @arg @ref MD_CMP_CFG2_FILT_CNT_THREE_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_FOUR_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_FIVE_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_SIX_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_SEVEN_TIMES
            @arg @ref MD_CMP_CFG2_FILT_CNT_EIGHT_TIMES
  */
__STATIC_INLINE uint32_t md_cmp_get_filter_counter(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG2, CMP_CFG2_FILT_CNT) >> CMP_CFG2_FILT_CNT_POSS);
}

/**
  * @brief  Set Comparator filter prescaler
  * @param  cmp CMP Instance
  * @param  pre Configure the sampling time of the filter function
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_filter_prescaler(CMP_TypeDef *cmp, uint32_t pre)
{
    MODIFY_REG(cmp->CFG2, CMP_CFG2_FILT_PRE, pre << CMP_CFG2_FILT_PRE_POSS);
}

/**
  * @brief  Get Comparator filter prescaler
  * @param  cmp CMP Instance
  * @retval Configure the sampling time of the filter function
  */
__STATIC_INLINE uint32_t md_cmp_get_filter_prescaler(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG2, CMP_CFG2_FILT_PRE) >> CMP_CFG2_FILT_PRE_POSS);
}

/**
  * @brief  Comparator Multinode Voltage Input Mask Selection
  * @param  cmp CMP Instance
  * @param  dm
            @arg @ref MD_CMP_CFG2_DM_DIABLE
            @arg @ref MD_CMP_CFG2_DM_DA0
            @arg @ref MD_CMP_CFG2_DM_DA1
            @arg @ref MD_CMP_CFG2_DM_DA2
            @arg @ref MD_CMP_CFG2_DM_DA3
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_multinode_voltage_input_mask(CMP_TypeDef *cmp, uint32_t dm)
{
    MODIFY_REG(cmp->CFG2, CMP_CFG2_DM, dm << CMP_CFG2_DM_POSS);
}

/**
  * @brief  Get Comparator Multinode Voltage Input Mask
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG2_DM_DIABLE
            @arg @ref MD_CMP_CFG2_DM_DA0
            @arg @ref MD_CMP_CFG2_DM_DA1
            @arg @ref MD_CMP_CFG2_DM_DA2
            @arg @ref MD_CMP_CFG2_DM_DA3
  */
__STATIC_INLINE uint32_t md_cmp_get_multinode_voltage_input_mask(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG2, CMP_CFG2_DM) >> CMP_CFG2_DM_POSS);
}

/**
  * @brief  Comparator Multi-node voltage group voltage input selection
  * @param  cmp CMP Instance
  * @param  da
            @arg @ref MD_CMP_CFG2_DA_0
            @arg @ref MD_CMP_CFG2_DA_1_16
            @arg @ref MD_CMP_CFG2_DA_2_16
            @arg @ref MD_CMP_CFG2_DA_3_16
            @arg @ref MD_CMP_CFG2_DA_4_16
            @arg @ref MD_CMP_CFG2_DA_5_16
            @arg @ref MD_CMP_CFG2_DA_6_16
            @arg @ref MD_CMP_CFG2_DA_7_16
            @arg @ref MD_CMP_CFG2_DA_8_16
            @arg @ref MD_CMP_CFG2_DA_9_16
            @arg @ref MD_CMP_CFG2_DA_10_16
            @arg @ref MD_CMP_CFG2_DA_11_16
            @arg @ref MD_CMP_CFG2_DA_12_16
            @arg @ref MD_CMP_CFG2_DA_13_16
            @arg @ref MD_CMP_CFG2_DA_14_16
            @arg @ref MD_CMP_CFG2_DA_15_16
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_multinode_voltage_input(CMP_TypeDef *cmp, uint32_t da)
{
    MODIFY_REG(cmp->CFG2, CMP_CFG2_DA, da << CMP_CFG2_DA_POSS);
}

/**
  * @brief  Get Comparator Multi-node voltage group voltage input
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG2_DA_0
            @arg @ref MD_CMP_CFG2_DA_1_16
            @arg @ref MD_CMP_CFG2_DA_2_16
            @arg @ref MD_CMP_CFG2_DA_3_16
            @arg @ref MD_CMP_CFG2_DA_4_16
            @arg @ref MD_CMP_CFG2_DA_5_16
            @arg @ref MD_CMP_CFG2_DA_6_16
            @arg @ref MD_CMP_CFG2_DA_7_16
            @arg @ref MD_CMP_CFG2_DA_8_16
            @arg @ref MD_CMP_CFG2_DA_9_16
            @arg @ref MD_CMP_CFG2_DA_10_16
            @arg @ref MD_CMP_CFG2_DA_11_16
            @arg @ref MD_CMP_CFG2_DA_12_16
            @arg @ref MD_CMP_CFG2_DA_13_16
            @arg @ref MD_CMP_CFG2_DA_14_16
            @arg @ref MD_CMP_CFG2_DA_15_16
  */
__STATIC_INLINE uint32_t md_cmp_get_multinode_voltage_input(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG2, CMP_CFG2_DA) >> CMP_CFG2_DA_POSS);
}

/**
  * @brief  Enable Voltage divider low section short-circuit switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_enable_low_short_circuit_function(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG2, CMP_CFG2_RLS);
}
/**
  * @brief  Disable Voltage divider low section short-circuit switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_disable_low_short_circuit_function(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG2, CMP_CFG2_RLS);
}

/**
  * @brief  Check if Voltage divider low section short-circuit switch is enable
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp2_is_enabled_low_short_circuit_function(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG2, CMP_CFG2_RLS) == (CMP_CFG2_RLS));
}

/**
  * @brief  Enable Voltage divider high section short circuit switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_enable_high_short_circuit_function(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG2, CMP_CFG2_RHS);
}
/**
  * @brief  Disable Voltage divider high section short circuit switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_disable_high_short_circuit_function(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG2, CMP_CFG2_RHS);
}

/**
  * @brief  Check if Voltage divider high section short circuit switch is enable
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp2_is_enabled_high_short_circuit_function(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG2, CMP_CFG2_RHS) == (CMP_CFG2_RHS));
}

/**
  * @brief  Comparator Multi-node voltage group voltage supply selection
  * @param  cmp CMP Instance
  * @param  ps
            @arg @ref MD_CMP_CFG2_PS_EXTERNAL
            @arg @ref MD_CMP_CFG2_PS_VDDH
            @arg @ref MD_CMP_CFG2_PS_VDDA
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_multinode_voltage(CMP_TypeDef *cmp, uint32_t ps)
{
    MODIFY_REG(cmp->CFG2, CMP_CFG2_PS, ps << CMP_CFG2_PS_POSS);
}

/**
  * @brief  Get Comparator Multi-node voltage group voltage supply selection
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG2_PS_EXTERNAL
            @arg @ref MD_CMP_CFG2_PS_VDDH
            @arg @ref MD_CMP_CFG2_PS_VDDA
  */
__STATIC_INLINE uint32_t md_cmp_get_multinode_voltage(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG2, CMP_CFG2_PS) >> CMP_CFG2_PS_POSS);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}  Micro_Driver
  */

void md_cmp_config(CMP_TypeDef *CMPx, md_cmp_config_typedef *config);

#ifdef __cplusplus
}
#endif
#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
