/**********************************************************************************
 *
 * @file    md_anpwr.h
 * @brief   header file of md_anpwr.c
 *
 * @date    8 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          8 Aug 2022      Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_ANPWR_H__
#define __MD_ANPWR_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include "es32f0943.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_ANPWR ANPWR
  * @brief ANPWR micro driver
  * @{
  */

/** @defgroup MD_ANPWR_Public_Macros ANPWR Public Macros
  * @{
  */
#define MD_ANPWR_VDIV_SEL_DISABLE  (0x00000000U)
#define MD_ANPWR_VDIV_SEL_1D16     (0x00000001U)
#define MD_ANPWR_VDIV_SEL_2D16     (0x00000002U)
#define MD_ANPWR_VDIV_SEL_3D16     (0x00000003U)
#define MD_ANPWR_VDIV_SEL_4D16     (0x00000004U)
#define MD_ANPWR_VDIV_SEL_5D16     (0x00000005U)
#define MD_ANPWR_VDIV_SEL_6D16     (0x00000006U)
#define MD_ANPWR_VDIV_SEL_7D16     (0x00000007U)
#define MD_ANPWR_VDIV_SEL_8D16     (0x00000008U)
#define MD_ANPWR_VDIV_SEL_9D16     (0x00000009U)
#define MD_ANPWR_VDIV_SEL_10D16    (0x0000000AU)
#define MD_ANPWR_VDIV_SEL_11D16    (0x0000000BU)
#define MD_ANPWR_VDIV_SEL_12D16    (0x0000000CU)
#define MD_ANPWR_VDIV_SEL_13D16    (0x0000000DU)
#define MD_ANPWR_VDIV_SEL_14D16    (0x0000000EU)
#define MD_ANPWR_VDIV_SEL_15D16    (0x0000000FU)

#define MD_ANPWR_VREF_SEL_VBG       (0x00000000U)
#define MD_ANPWR_VREF_SEL_VDIV      (0x00000001U)

#define MD_ANPWR_VBG_CHPCKS_1MHZ    (0x00000000U)
#define MD_ANPWR_VBG_CHPCKS_2MHZ    (0x00000001U)

#define MD_ANPWR_VLDO_SEL_2P2V      (0x00000000U)
#define MD_ANPWR_VLDO_SEL_2P4V      (0x00000001U)
#define MD_ANPWR_VLDO_SEL_2P7V      (0x00000002U)
#define MD_ANPWR_VLDO_SEL_3P0V      (0x00000003U)
#define MD_ANPWR_VLDO_SEL_3P3V      (0x00000004U)

#define MD_ANPWR_VLDO_CHPCKS_1MHZ   (0x00000000U)
#define MD_ANPWR_VLDO_CHPCKS_2MHZ   (0x00000001U)

#define MD_ANPWR_VLDO_MODE_HIGH_IMPEDANCE       (0x00000000U)
#define MD_ANPWR_VLDO_MODE_SHORT_VDDH           (0x00000001U)
#define MD_ANPWR_VLDO_MODE_GROUND_DISCHARGE     (0x00000002U)
#define MD_ANPWR_VLDO_MODE_ADJUSTABLE_VOLTAGE   (0x00000003U)

typedef struct
{
    uint32_t vldo_mode;
    uint32_t vldo_sel;
} md_anpwr_init_typedef;

/**
  * @}
  */

/** @defgroup MD_ANPWR_Public_Functions ANPWR Public Functions
  * @{
  */

/** @defgroup MD_ANPWR_Public_Functions_Group2 PWR_CTRL
  * @{
  */

/**
  * @brief  Set ANPWR_PWR_CTRL Register
  * @param  anpwr ANPWR Instance
  * @param  pwr_ctrl
  */
__STATIC_INLINE void md_anpwr_set_pwr_ctrl(ANPWR_TypeDef *anpwr, uint32_t pwr_ctrl)
{
    WRITE_REG(anpwr->PWR_CTRL, pwr_ctrl);
}

/**
  * @brief  Get ANPWR_PWR_CTRL Register
  * @param  anpwr ANPWR Instance
  * @retval pwr_ctrl Values
  */
__STATIC_INLINE uint32_t md_anpwr_get_pwr_ctrl(ANPWR_TypeDef *anpwr)
{
    return (uint32_t)READ_REG(anpwr->PWR_CTRL);
}

/**
  * @brief  Set VREF divider selection
  * @param  anpwr ANPWR Instance
  * @param  vref_divs This parameter can be one of the following values:
  *         @arg @ref MD_ANPWR_VDIV_SEL_DISABLE
  *         @arg @ref MD_ANPWR_VDIV_SEL_1D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_2D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_3D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_4D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_5D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_6D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_7D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_8D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_9D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_10D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_11D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_12D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_13D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_14D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_15D16
  */
__STATIC_INLINE void md_anpwr_set_vref_divider(ANPWR_TypeDef *anpwr, uint32_t vref_divs)
{
    MODIFY_REG(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VDIV_SEL, vref_divs << ANPWR_PWR_CTRL_VDIV_SEL_POSS);
}

/**
  * @brief  Get VREF divider selection
  * @param  anpwr ANPWR Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ANPWR_VDIV_SEL_DISABLE
  *         @arg @ref MD_ANPWR_VDIV_SEL_1D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_2D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_3D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_4D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_5D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_6D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_7D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_8D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_9D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_10D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_11D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_12D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_13D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_14D16
  *         @arg @ref MD_ANPWR_VDIV_SEL_15D16
  */
__STATIC_INLINE uint32_t md_anpwr_get_vref_divider(ANPWR_TypeDef *anpwr)
{
    return (uint32_t)(READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VDIV_SEL) >> ANPWR_PWR_CTRL_VDIV_SEL_POSS);
}

/**
  * @brief  VREF buffer switch Enable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_enable_vref(ANPWR_TypeDef *anpwr)
{
    SET_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VREF_EN);
}

/**
  * @brief  VREF buffer switch Disable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_disable_vref(ANPWR_TypeDef *anpwr)
{
    CLEAR_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VREF_EN);
}

/**
  * @brief  Check if VREF buffer switch is Enabled.
  * @param  anpwr ANPWR Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_anpwr_is_enabled_vref(ANPWR_TypeDef *anpwr)
{
    return (READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VREF_EN) == (ANPWR_PWR_CTRL_VREF_EN));
}

/**
  * @brief  Set VREF selection
  * @param  anpwr ANPWR Instance
  * @param  vref This parameter can be one of the following values:
  *         @arg @ref MD_ANPWR_VREF_SEL_VBG
  *         @arg @ref MD_ANPWR_VREF_SEL_VDIV
  */
__STATIC_INLINE void md_anpwr_set_vref_sel(ANPWR_TypeDef *anpwr, uint32_t vref)
{
    MODIFY_REG(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VREF_SEL, vref << ANPWR_PWR_CTRL_VREF_SEL_POS);
}

/**
  * @brief  Get VREF selection
  * @param  anpwr ANPWR Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ANPWR_VREF_SEL_VBG
  *         @arg @ref MD_ANPWR_VREF_SEL_VDIV
  */
__STATIC_INLINE uint32_t md_anpwr_get_vref_sel(ANPWR_TypeDef *anpwr)
{
    return (uint32_t)(READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VREF_SEL) >> ANPWR_PWR_CTRL_VREF_SEL_POS);
}

/**
  * @brief  VBG temperature sensing pin RC filter switch Enable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_enable_tsflt(ANPWR_TypeDef *anpwr)
{
    SET_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_TSFLTEN);
}

/**
  * @brief  AFE temperature sensing pin RC filter switch Disable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_disable_tsflt(ANPWR_TypeDef *anpwr)
{
    CLEAR_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_TSFLTEN);
}

/**
  * @brief  Check if VBG temperature sensing pin RC filter switch is Enabled.
  * @param  anpwr ANPWR Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_anpwr_is_enabled_tsflt(ANPWR_TypeDef *anpwr)
{
    return (READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_TSFLTEN) == (ANPWR_PWR_CTRL_TSFLTEN));
}


/**
  * @brief  Set VBG Chopper Clock Selection
  * @param  anpwr ANPWR Instance
  * @param  afevbg_chpcks This parameter can be one of the following values:
  *         @arg @ref MD_ANPWR_VBG_CHPCKS_1MHZ
  *         @arg @ref MD_ANPWR_VBG_CHPCKS_2MHZ
  */
__STATIC_INLINE void md_anpwr_set_vbg_chpcks(ANPWR_TypeDef *anpwr, uint32_t afevbg_chpcks)
{
    MODIFY_REG(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VBG_CHPCKS, afevbg_chpcks << ANPWR_PWR_CTRL_VBG_CHPCKS_POS);
}

/**
  * @brief  Get AFE VBG Chopper Clock Selection
  * @param  anpwr ANPWR Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ANPWR_VBG_CHPCKS_1MHZ
  *         @arg @ref MD_ANPWR_VBG_CHPCKS_2MHZ
  */
__STATIC_INLINE uint32_t md_anpwr_get_vbg_chpcks(ANPWR_TypeDef *anpwr)
{
    return (uint32_t)(READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VBG_CHPCKS) >> ANPWR_PWR_CTRL_VBG_CHPCKS_POS);
}

/**
  * @brief  VBG Chopper Switch Enable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_enable_vbg_chp(ANPWR_TypeDef *anpwr)
{
    SET_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VBG_CHPEN);
}

/**
  * @brief  VBG Chopper Switch Disable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_disable_vbg_chp(ANPWR_TypeDef *anpwr)
{
    CLEAR_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VBG_CHPEN);
}

/**
  * @brief  Check if VBG Chopper Switch is Enabled.
  * @param  anpwr ANPWR Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_anpwr_is_enabled_vbg_chp(ANPWR_TypeDef *anpwr)
{
    return (READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VBG_CHPEN) == (ANPWR_PWR_CTRL_VBG_CHPEN));
}

/**
  * @brief  VBG Switch Enable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_enable_vbg(ANPWR_TypeDef *anpwr)
{
    SET_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VBG_EN);
}

/**
  * @brief  VBG Switch Disable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_disable_vbg(ANPWR_TypeDef *anpwr)
{
    CLEAR_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VBG_EN);
}

/**
  * @brief  Check if VBG Switch is Enabled.
  * @param  anpwr ANPWR Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_anpwr_is_enabled_vbg(ANPWR_TypeDef *anpwr)
{
    return (READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VBG_EN) == (ANPWR_PWR_CTRL_VBG_EN));
}

/**
  * @brief  VBG Bias Switch Enable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_enable_vbg_bias(ANPWR_TypeDef *anpwr)
{
    SET_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_BIAS_EN);
}

/**
  * @brief  VBG Bias Switch Disable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_disable_vbg_bias(ANPWR_TypeDef *anpwr)
{
    CLEAR_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_BIAS_EN);
}

/**
  * @brief  Check if VBG Bias Switch is Enabled.
  * @param  anpwr ANPWR Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_anpwr_is_enabled_vbg_bias(ANPWR_TypeDef *anpwr)
{
    return (READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_BIAS_EN) == (ANPWR_PWR_CTRL_BIAS_EN));
}

/**
  * @brief  Set VLDO Adjustable Voltage Selection
  * @param  anpwr ANPWR Instance
  * @param  vldo_sel This parameter can be one of the following values:
  *         @arg @ref MD_ANPWR_VLDO_SEL_2P2V
  *         @arg @ref MD_ANPWR_VLDO_SEL_2P4V
  *         @arg @ref MD_ANPWR_VLDO_SEL_2P7V
  *         @arg @ref MD_ANPWR_VLDO_SEL_3P0V
  *         @arg @ref MD_ANPWR_VLDO_SEL_3P3V
  */
__STATIC_INLINE void md_anpwr_set_vldo_sel(ANPWR_TypeDef *anpwr, uint32_t vldo_sel)
{
    MODIFY_REG(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VLDO_SEL, vldo_sel << ANPWR_PWR_CTRL_VLDO_SEL_POSS);
}

/**
  * @brief  Get VLDO Adjustable Voltage Selection
  * @param  AANPWRx ANPWR Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ANPWR_VLDO_SEL_2P2V
  *         @arg @ref MD_ANPWR_VLDO_SEL_2P4V
  *         @arg @ref MD_ANPWR_VLDO_SEL_2P7V
  *         @arg @ref MD_ANPWR_VLDO_SEL_3P0V
  *         @arg @ref MD_ANPWR_VLDO_SEL_3P3V
  */
__STATIC_INLINE uint32_t md_anpwr_get_vldo_sel(ANPWR_TypeDef *anpwr)
{
    return (uint32_t)(READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VLDO_SEL) >> ANPWR_PWR_CTRL_VLDO_SEL_POSS);
}

/**
  * @brief  Set VLDO chopper clock selection
  * @param  AANPWRx ANPWR Instance
  * @param  vldo_chpcks This parameter can be one of the following values:
  *         @arg @ref MD_ANPWR_VLDO_CHPCKS_1MHZ
  *         @arg @ref MD_ANPWR_VLDO_CHPCKS_2MHZ
  */
__STATIC_INLINE void md_anpwr_set_vlod_chpcks(ANPWR_TypeDef *anpwr, uint32_t vldo_chpcks)
{
    MODIFY_REG(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VLDO_CHPCKS, vldo_chpcks << ANPWR_PWR_CTRL_VLDO_CHPCKS_POS);
}

/**
  * @brief  Get VLDO chopper clock selection
  * @param  anpwr ANPWR Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ANPWR_VLDO_CHPCKS_1MHZ
  *         @arg @ref MD_ANPWR_VLDO_CHPCKS_2MHZ
  */
__STATIC_INLINE uint32_t md_anpwr_get_vlod_chpcks(ANPWR_TypeDef *anpwr)
{
    return (uint32_t)(READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VLDO_CHPCKS) >> ANPWR_PWR_CTRL_VLDO_CHPCKS_POS);
}

/**
  * @brief  VLDO chopper switch Enable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_enable_vldo_chp(ANPWR_TypeDef *anpwr)
{
    SET_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VLDO_CHPEN);
}

/**
  * @brief  VLDO chopper switch Disable.
  * @param  anpwr ANPWR Instance
  * @retval None.
  */
__STATIC_INLINE void md_anpwr_disable_vldo_chp(ANPWR_TypeDef *anpwr)
{
    CLEAR_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VLDO_CHPEN);
}

/**
  * @brief  Check if VLDO chopper switch is Enabled.
  * @param  anpwr ANPWR Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_anpwr_is_enabled_vldo_chp(ANPWR_TypeDef *anpwr)
{
    return (READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VLDO_CHPEN) == (ANPWR_PWR_CTRL_VLDO_CHPEN));
}

/**
  * @brief  Set VLDO mode selection
  * @param  anpwr ANPWR Instance
  * @param  vldo_chpcks This parameter can be one of the following values:
  *         @arg @ref MD_ANPWR_VLDO_MODE_HIGH_IMPEDANCE
  *         @arg @ref MD_ANPWR_VLDO_MODE_SHORT_VDDH
  *         @arg @ref MD_ANPWR_VLDO_MODE_GROUND_DISCHARGE
  *         @arg @ref MD_ANPWR_VLDO_MODE_ADJUSTABLE_VOLTAGE
  */
__STATIC_INLINE void md_anpwr_set_vldo_mode(ANPWR_TypeDef *anpwr, uint32_t vldo_mode)
{
    MODIFY_REG(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VLDO_MODE, vldo_mode << ANPWR_PWR_CTRL_VLDO_MODE_POSS);
}

/**
  * @brief  Get VLDO mode selection
  * @param  anpwr ANPWR Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ANPWR_VLDO_MODE_HIGH_IMPEDANCE
  *         @arg @ref MD_ANPWR_VLDO_MODE_SHORT_VDDH
  *         @arg @ref MD_ANPWR_VLDO_MODE_GROUND_DISCHARGE
  *         @arg @ref MD_ANPWR_VLDO_MODE_ADJUSTABLE_VOLTAGE
  */
__STATIC_INLINE uint32_t md_anpwr_get_vldo_mode(ANPWR_TypeDef *anpwr)
{
    return (uint32_t)(READ_BIT(anpwr->PWR_CTRL, ANPWR_PWR_CTRL_VLDO_MODE) >> ANPWR_PWR_CTRL_VLDO_MODE_POSS);
}

/**
  * @}
  */

/**
  * @}
  */
void md_anpwr_init(ANPWR_TypeDef *anpwr, md_anpwr_init_typedef *anpwr_init);
/**
  * @} Micro_Driver
  */


#ifdef __cplusplus
}
#endif
#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
