/**********************************************************************************
 *
 * @file    md_adc.h
 * @brief   header file of md_adc.c
 *
 * @date    5 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 Aug 2022      Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_ADC_H__
#define __MD_ADC_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include "es32f0943.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_ADC ADC
  * @brief ADC micro driver
  * @{
  */

/** @defgroup MD_ADC_Pubulic_Types ADC Pubulic Types
  * @{
  */


typedef struct
{
    uint8_t adc_cursel;     /*!< ADC current */
    uint8_t adc_vrrs;       /*!< ADC reference voltage range */
    uint8_t adc_ingn;       /*!< ADC input gain selection */
    uint8_t adc_chps;       /*!< ADC Chopper selection */
    uint8_t adc_cks;          /*!< ADC clock selection */
    uint8_t adc_osr;       /*!< ADC oversampling output frequency */
    uint8_t cic_order;     /*!< CIC filter order */
} md_adc_init_typedef;

typedef struct
{
    uint8_t adc_inps;         /*!< ADC positive Input selection */
    uint8_t adc_inns;         /*!< ADC negative Input selection */
    uint8_t adc_vrps;         /*!< ADC positive reference voltage input selection */
    uint8_t adc_vrns;         /*!< ADC negative reference voltage input selection */
} md_adc_config_typedef;

typedef struct
{
    uint8_t ia_gain;
    uint8_t ia_chps;
    uint8_t ia_chpcks;
    uint8_t ia_lpfen;
} md_ia_init_typedef;


typedef struct
{
    uint8_t mode;
    uint8_t section;
    uint32_t coef_sc_1;
    uint32_t coef_b1_1;
    uint32_t coef_b2_1;
    uint32_t coef_b3_1;
    uint32_t coef_a2_1;
    uint32_t coef_a3_1;
    uint32_t coef_sc_2;
    uint32_t coef_b1_2;
    uint32_t coef_b2_2;
    uint32_t coef_b3_2;
    uint32_t coef_a2_2;
    uint32_t coef_a3_2;
} md_biquad_init_typedef;

typedef enum
{
    MD_ADC_BIQUAD_1ST_ORDER_LOW_PASS_FILTER,
    MD_ADC_BIQUAD_2ST_ORDER_LOW_PASS_FILTER,
    MD_ADC_BIQUAD_1ST_ORDER_HIGH_PASS_FILTER,
    MD_ADC_BIQUAD_2ST_ORDER_HIGH_PASS_FILTER
} md_biquad_order_pass_filter;

typedef enum
{
    ADC_CLOCKSOURCE_DISABLE = 0,
    ADC_CLOCKSOURCE_HOSC = 1,
    ADC_CLOCKSOURCE_HRC = 2,
    ADC_CLOCKSOURCE_PLL = 3
} md_adc_clocksource;

typedef enum
{
    CHOP_CLOCKSOURCE_DISABLE = 0,
    CHOP_CLOCKSOURCE_HOSC = 1,
    CHOP_CLOCKSOURCE_HRC = 2,
    CHOP_CLOCKSOURCE_PLL = 3
} md_chop_clocksource;

typedef enum
{
    ANPWR_VLDO_MODE_HIGH_IMPEDANCE     = 0,
    ANPWR_VLDO_MODE_SHORT_VDDH         = 1,
    ANPWR_VLDO_MODE_GROUND_DISCHARGE   = 2,
    ANPWR_VLDO_MODE_ADJUSTABLE_VOLTAGE = 3
} md_anpwr_vldo_mode;

typedef enum
{
    ANPWR_VLDO_SEL_2P2V = 0,
    ANPWR_VLDO_SEL_2P4V = 1,
    ANPWR_VLDO_SEL_2P7V = 2,
    ANPWR_VLDO_SEL_3P0V = 3,
    ANPWR_VLDO_SEL_3P3V = 4
} md_anpwr_vldo_sel;

typedef struct
{
    md_anpwr_vldo_mode anpwr_vldo_mode;
    md_anpwr_vldo_sel anpwr_vldo_sel;
    md_adc_clocksource adc_clocksource;
    uint8_t adc_clockdiv;
    md_chop_clocksource chop_clocksource;
    uint8_t chop_clockdiv;
} md_adc_power_init_typedef;

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Macros ADC Public Macros
  * @{
  */
#define MD_ADC_IA_GAIN_0P5X             (0x00000000U)
#define MD_ADC_IA_GAIN_1X               (0x00000001U)
#define MD_ADC_IA_GAIN_2X               (0x00000002U)
#define MD_ADC_IA_GAIN_4X               (0x00000003U)
#define MD_ADC_IA_GAIN_8X               (0x00000004U)
#define MD_ADC_IA_GAIN_16X              (0x00000005U)
#define MD_ADC_IA_GAIN_32X              (0x00000006U)
#define MD_ADC_IA_GAIN_64X              (0x00000007U)

#define MD_ADC_IA_INNS_AIN0             (0x00000000U)
#define MD_ADC_IA_INNS_AIN1             (0x00000001U)
#define MD_ADC_IA_INNS_AIN2             (0x00000002U)
#define MD_ADC_IA_INNS_AIN3             (0x00000003U)
#define MD_ADC_IA_INNS_VREF             (0x00000004U)
#define MD_ADC_IA_INNS_VDDA             (0x00000005U)
#define MD_ADC_IA_INNS_VSSH             (0x00000006U)
#define MD_ADC_IA_INNS_TSP              (0x00000007U)
#define MD_ADC_IA_INNS_TSN              (0x00000008U)

#define MD_ADC_IA_INPS_AIN0             (0x00000000U)
#define MD_ADC_IA_INPS_AIN1             (0x00000001U)
#define MD_ADC_IA_INPS_AIN2             (0x00000002U)
#define MD_ADC_IA_INPS_AIN3             (0x00000003U)
#define MD_ADC_IA_INPS_VREF             (0x00000004U)
#define MD_ADC_IA_INPS_VDDA             (0x00000005U)
#define MD_ADC_IA_INPS_VSSH             (0x00000006U)
#define MD_ADC_IA_INPS_TSP              (0x00000007U)
#define MD_ADC_IA_INPS_TSN              (0x00000008U)

#define MD_ADC_IA_CHPCKS_125KHZ         (0x00000000U)
#define MD_ADC_IA_CHPCKS_250KHZ         (0x00000001U)
#define MD_ADC_IA_CHPCKS_500KHZ         (0x00000002U)
#define MD_ADC_IA_CHPCKS_NOCLOCK        (0x00000003U)

#define MD_ADC_IA_LPFEN_DISABLE         (0U)
#define MD_ADC_IA_LPFEN_ENABLE          (1U)

#define MD_ADC_IA_CHPS_ANAOLG_OFF_SYSTEM_OFF  (0x00000000U)
#define MD_ADC_IA_CHPS_ANAOLG_ON_SYSTEM_OFF   (0x00000001U)
#define MD_ADC_IA_CHPS_ANAOLG_OFF_SYSTEM_ON   (0x00000002U)
#define MD_ADC_IA_CHPS_ANAOLG_ON_SYSTEM_ON    (0x00000003U)

#define MD_ADC_CURSEL_LOW               (0x00000000U)
#define MD_ADC_CURSEL_HIGH              (0x00000001U)

#define MD_ADC_VRRS_1X                  (0x00000000U)
#define MD_ADC_VRRS_0P5X                (0x00000001U)

#define MD_ADC_INGN_1X                  (0x00000000U)
#define MD_ADC_INGN_2X                  (0x00000001U)
#define MD_ADC_INGN_4X                  (0x00000002U)
#define MD_ADC_INGN_8X                  (0x00000003U)

#define MD_ADC_VRNS_VSSH                (0x00000000U)
#define MD_ADC_VRNS_AIN3                (0x00000001U)
#define MD_ADC_VRNS_AIN5                (0x00000002U)
#define MD_ADC_VRNS_VREF                (0x00000003U)

#define MD_ADC_VRPS_VDDA                (0x00000000U)
#define MD_ADC_VRPS_AIN2                (0x00000001U)
#define MD_ADC_VRPS_AIN4                (0x00000002U)
#define MD_ADC_VRPS_VREF                (0x00000003U)

#define MD_ADC_INNS_IAOUTP              (0x00000000U)
#define MD_ADC_INNS_IAOUTN              (0x00000001U)
#define MD_ADC_INNS_AIN2                (0x00000002U)
#define MD_ADC_INNS_AIN3                (0x00000003U)
#define MD_ADC_INNS_AIN4                (0x00000004U)
#define MD_ADC_INNS_AIN5                (0x00000005U)
#define MD_ADC_INNS_AIN6                (0x00000006U)
#define MD_ADC_INNS_AIN7                (0x00000007U)
#define MD_ADC_INNS_TSP                 (0x00000008U)
#define MD_ADC_INNS_TSN                 (0x00000009U)
#define MD_ADC_INNS_VREF                (0x0000000AU)
#define MD_ADC_INNS_VDDA                (0x0000000BU)
#define MD_ADC_INNS_OPOI                (0x0000000CU)
#define MD_ADC_INNS_VDIV                (0x0000000DU)
#define MD_ADC_INNS_VBG                 (0x0000000EU)
#define MD_ADC_INNS_VSSH                (0x0000000FU)

#define MD_ADC_INPS_IAOUTP              (0x00000000U)
#define MD_ADC_INPS_IAOUTN              (0x00000001U)
#define MD_ADC_INPS_AIN2                (0x00000002U)
#define MD_ADC_INPS_AIN3                (0x00000003U)
#define MD_ADC_INPS_AIN4                (0x00000004U)
#define MD_ADC_INPS_AIN5                (0x00000005U)
#define MD_ADC_INPS_AIN6                (0x00000006U)
#define MD_ADC_INPS_AIN7                (0x00000007U)
#define MD_ADC_INPS_TSP                 (0x00000008U)
#define MD_ADC_INPS_TSN                 (0x00000009U)
#define MD_ADC_INPS_VREF                (0x0000000AU)
#define MD_ADC_INPS_VDDA                (0x0000000BU)
#define MD_ADC_INPS_OPOI                (0x0000000CU)
#define MD_ADC_INPS_VDIV                (0x0000000DU)
#define MD_ADC_INPS_VBG              (0x0000000EU)
#define MD_ADC_INPS_VSSH                (0x0000000FU)

#define MD_ADC_CHPS_CHOP_DISABLE_CDS_DISABLE  (0x00000000U)
#define MD_ADC_CHPS_CDS_ENABLE                (0x00000001U)
#define MD_ADC_CHPS_CHOP_ENABLE               (0x00000002U)
#define MD_ADC_CHPS_CHOP_ENABLE_CDS_ENABLE    (0x00000003U)

#define MD_ADC_CKS_250KHZ               (0x00000000U)
#define MD_ADC_CKS_500KHZ               (0x00000001U)
#define MD_ADC_CKS_1MHZ                 (0x00000002U)

#define MD_ADC_OSR_64                   (0x00000000U)
#define MD_ADC_OSR_128                  (0x00000001U)
#define MD_ADC_OSR_256                  (0x00000002U)
#define MD_ADC_OSR_512                  (0x00000003U)
#define MD_ADC_OSR_1024                 (0x00000004U)
#define MD_ADC_OSR_2048                 (0x00000005U)
#define MD_ADC_OSR_4096                 (0x00000006U)
#define MD_ADC_OSR_8192                 (0x00000007U)
#define MD_ADC_OSR_16384                (0x00000008U)
#define MD_ADC_OSR_32768                (0x00000009U)

#define MD_ADC_CIC_ORDER_2              (0x00000000U)
#define MD_ADC_CIC_ORDER_3              (0x00000001U)
#define MD_ADC_CIC_ORDER_4              (0x00000002U)


#define MD_ADC_BIQUAD_TRIGMODE_SWTRIG   (0x00000000U)
#define MD_ADC_BIQUAD_TRIGMODE_ADC      (0x00000001U)

#define MD_ADC_BIQUAD_SECTION_1         (0x00000000U)
#define MD_ADC_BIQUAD_SECTION_2         (0x00000001U)

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions ADC Public Functions
  * @{
  */

/** @defgroup MD_ADC_Public_Functions_Group2 IA_CTRL
  * @{
  */
/**
  * @brief  Set ADC_IA_CTRL  Register
  * @param  ADCx ADC Instance
  * @param  ia_ctrl
  */
__STATIC_INLINE void md_adc_set_ia_ctrl(ADC_TypeDef *ADCx, uint32_t ia_ctrl)
{
    WRITE_REG(ADCx->IA_CTRL, ia_ctrl);
}

/**
  * @brief  Get ADC_IA_CTRL  Register
  * @param  ADCx ADC Instance
  * @retval IA_CTRL Values
  */
__STATIC_INLINE uint32_t md_adc_get_ia_ctrl(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->IA_CTRL);
}

/**
  * @brief  Get Instrumentation Amplifier Input Polarity Status
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref 0: Positive and negative signals are maintained
  *         @arg @ref 1: Positive and negative signals are swapped
  */
__STATIC_INLINE uint32_t md_adc_get_ia_pola_st(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_POLA_ST) >> ADC_IA_CTRL_IA_POLA_ST_POS);
}

/**
  * @brief  Set Instrumentation Amplifier Gain Selection
  * @param  ADCx ADC Instance
  * @param  gain This parameter can be one of the following values:
  *         @arg @ref MD_ADC_IA_GAIN_0P5X
  *         @arg @ref MD_ADC_IA_GAIN_1X
  *         @arg @ref MD_ADC_IA_GAIN_2X
  *         @arg @ref MD_ADC_IA_GAIN_4X
  *         @arg @ref MD_ADC_IA_GAIN_8X
  *         @arg @ref MD_ADC_IA_GAIN_16X
  *         @arg @ref MD_ADC_IA_GAIN_32X
  *         @arg @ref MD_ADC_IA_GAIN_64X
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ia_gain(ADC_TypeDef *ADCx, uint32_t gain)
{
    MODIFY_REG(ADCx->IA_CTRL, ADC_IA_CTRL_IA_GAIN, gain << ADC_IA_CTRL_IA_GAIN_POSS);
}

/**
  * @brief  Get Instrumentation Amplifier Gain Selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_IA_GAIN_0P5X
  *         @arg @ref MD_ADC_IA_GAIN_1X
  *         @arg @ref MD_ADC_IA_GAIN_2X
  *         @arg @ref MD_ADC_IA_GAIN_4X
  *         @arg @ref MD_ADC_IA_GAIN_8X
  *         @arg @ref MD_ADC_IA_GAIN_16X
  *         @arg @ref MD_ADC_IA_GAIN_32X
  *         @arg @ref MD_ADC_IA_GAIN_64X
  */
__STATIC_INLINE uint32_t md_adc_get_ia_gain(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_GAIN) >> ADC_IA_CTRL_IA_GAIN_POSS);
}

/**
  * @brief  Set Instrumentation Amplifier Negative Input Selection
  * @param  ADCx ADC Instance
  * @param  inns This parameter can be one of the following values:
  *         @arg @ref MD_ADC_IA_INNS_AIN0
  *         @arg @ref MD_ADC_IA_INNS_AIN1
  *         @arg @ref MD_ADC_IA_INNS_AIN2
  *         @arg @ref MD_ADC_IA_INNS_AIN3
  *         @arg @ref MD_ADC_IA_INNS_VREF
  *         @arg @ref MD_ADC_IA_INNS_VDDA
  *         @arg @ref MD_ADC_IA_INNS_VSSH
  *         @arg @ref MD_ADC_IA_INNS_TSP
  *         @arg @ref MD_ADC_IA_INNS_TSN
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ia_inns(ADC_TypeDef *ADCx, uint32_t inns)
{
    MODIFY_REG(ADCx->IA_CTRL, ADC_IA_CTRL_IA_INNS, inns << ADC_IA_CTRL_IA_INNS_POSS);
}

/**
  * @brief  Get Instrumentation Amplifier Negative Input Selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_IA_INNS_AIN0
  *         @arg @ref MD_ADC_IA_INNS_AIN1
  *         @arg @ref MD_ADC_IA_INNS_AIN2
  *         @arg @ref MD_ADC_IA_INNS_AIN3
  *         @arg @ref MD_ADC_IA_INNS_VREF
  *         @arg @ref MD_ADC_IA_INNS_VDDA
  *         @arg @ref MD_ADC_IA_INNS_VSSH
  *         @arg @ref MD_ADC_IA_INNS_TSP
  *         @arg @ref MD_ADC_IA_INNS_TSN
  */
__STATIC_INLINE uint32_t md_adc_get_ia_inns(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_INNS) >> ADC_IA_CTRL_IA_INNS_POSS);
}

/**
  * @brief  Set Instrumentation Amplifier Positive Input Selection
  * @param  ADCx ADC Instance
  * @param  inps This parameter can be one of the following values:
  *         @arg @ref MD_ADC_IA_INPS_AIN0
  *         @arg @ref MD_ADC_IA_INPS_AIN1
  *         @arg @ref MD_ADC_IA_INPS_AIN2
  *         @arg @ref MD_ADC_IA_INPS_AIN3
  *         @arg @ref MD_ADC_IA_INPS_VREF
  *         @arg @ref MD_ADC_IA_INPS_VDDA
  *         @arg @ref MD_ADC_IA_INPS_VSSH
  *         @arg @ref MD_ADC_IA_INPS_TSP
  *         @arg @ref MD_ADC_IA_INPS_TSN
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ia_inps(ADC_TypeDef *ADCx, uint32_t inps)
{
    MODIFY_REG(ADCx->IA_CTRL, ADC_IA_CTRL_IA_INPS, inps << ADC_IA_CTRL_IA_INPS_POSS);
}

/**
  * @brief  Get Instrumentation Amplifier Positive Input Selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_IA_INPS_AIN0
  *         @arg @ref MD_ADC_IA_INPS_AIN1
  *         @arg @ref MD_ADC_IA_INPS_AIN2
  *         @arg @ref MD_ADC_IA_INPS_AIN3
  *         @arg @ref MD_ADC_IA_INPS_VREF
  *         @arg @ref MD_ADC_IA_INPS_VDDA
  *         @arg @ref MD_ADC_IA_INPS_VSSH
  *         @arg @ref MD_ADC_IA_INPS_TSP
  *         @arg @ref MD_ADC_IA_INPS_TSN
  */
__STATIC_INLINE uint32_t md_adc_get_ia_inps(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_INPS) >> ADC_IA_CTRL_IA_INPS_POSS);
}

/**
  * @brief  Set Instrumentation Amplifier Chopper Clock Selection
  * @param  ADCx ADC Instance
  * @param  chpcks This parameter can be one of the following values:
  *         @arg @ref MD_ADC_IA_CHPCKS_125KHZ
  *         @arg @ref MD_ADC_IA_CHPCKS_250KHZ
  *         @arg @ref MD_ADC_IA_CHPCKS_500KHZ
  *         @arg @ref MD_ADC_IA_CHPCKS_NOCLOCK
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ia_chpcks(ADC_TypeDef *ADCx, uint32_t chpcks)
{
    MODIFY_REG(ADCx->IA_CTRL, ADC_IA_CTRL_IA_CHPCKS, chpcks << ADC_IA_CTRL_IA_CHPCKS_POSS);
}

/**
  * @brief  Get Instrumentation Amplifier Chopper Clock Selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_IA_CHPCKS_125KHZ
  *         @arg @ref MD_ADC_IA_CHPCKS_250KHZ
  *         @arg @ref MD_ADC_IA_CHPCKS_500KHZ
  *         @arg @ref MD_ADC_IA_CHPCKS_NOCLOCK
  */
__STATIC_INLINE uint32_t md_adc_get_ia_chpcks(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_CHPCKS) >> ADC_IA_CTRL_IA_CHPCKS_POSS);
}

/**
  * @brief  Set Instrumentation Amplifier Chopper Clock Selection
  * @param  ADCx ADC Instance
  * @param  chps This parameter can be one of the following values:
  *         @arg @ref MD_ADC_IA_CHPS_ANAOLG_OFF_SYSTEM_OFF
  *         @arg @ref MD_ADC_IA_CHPS_ANAOLG_ON_SYSTEM_OFF
  *         @arg @ref MD_ADC_IA_CHPS_ANAOLG_OFF_SYSTEM_ON
  *         @arg @ref MD_ADC_IA_CHPS_ANAOLG_ON_SYSTEM_ON
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ia_chps(ADC_TypeDef *ADCx, uint32_t chps)
{
    MODIFY_REG(ADCx->IA_CTRL, ADC_IA_CTRL_IA_CHPS, chps << ADC_IA_CTRL_IA_CHPS_POSS);
}

/**
  * @brief  Get Instrumentation Amplifier Chopper Clock Selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_IA_CHPS_ANAOLG_OFF_SYSTEM_OFF
  *         @arg @ref MD_ADC_IA_CHPS_ANAOLG_ON_SYSTEM_OFF
  *         @arg @ref MD_ADC_IA_CHPS_ANAOLG_OFF_SYSTEM_ON
  *         @arg @ref MD_ADC_IA_CHPS_ANAOLG_ON_SYSTEM_ON
  */
__STATIC_INLINE uint32_t md_adc_get_ia_chps(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_CHPS) >> ADC_IA_CTRL_IA_CHPS_POSS);
}

/**
  * @brief  Instrumentation Amplifier Low Pass Filter Switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_ia_lpfen(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_LPFEN);
}

/**
  * @brief  Instrumentation Amplifier Low Pass Filter Switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_ia_lpfen(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_LPFEN);
}

/**
  * @brief  Check if Instrumentation Amplifier Low Pass Filter Switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_ia_lpfen(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_LPFEN) == (ADC_IA_CTRL_IA_LPFEN));
}

/**
  * @brief  Instrumentation Amplifier Input Short Control Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_ia_inshr(ADC_TypeDef *ADCx)
{
    MODIFY_REG(ADCx->IA_CTRL, ADC_IA_CTRL_IA_INSHR, 1U << ADC_IA_CTRL_IA_INSHR_POSS);
}

/**
  * @brief  Instrumentation Amplifier Input Short Control Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_ia_inshr(ADC_TypeDef *ADCx)
{
    MODIFY_REG(ADCx->IA_CTRL, ADC_IA_CTRL_IA_INSHR, 0U << ADC_IA_CTRL_IA_INSHR_POSS);    
}

/**
  * @brief  Check if Instrumentation Amplifier Input Short Control is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_ia_inshr(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IA_CTRL, (ADC_IA_CTRL_IA_INSHR & (1U <<1 ))) == ((ADC_IA_CTRL_IA_INSHR & (1U << 1))));
}

/**
  * @brief  Instrumentation Amplifier Switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_ia_en(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_EN);
}

/**
  * @brief  Instrumentation Amplifier Switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_ia_en(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_EN);
}

/**
  * @brief  Check if Instrumentation Amplifier Switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_ia_en(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IA_CTRL, ADC_IA_CTRL_IA_EN) == (ADC_IA_CTRL_IA_EN));
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group3 ADC_CTRL0
  * @{
  */
/**
  * @brief  Set ADC_CTRL0  Register
  * @param  ADCx ADC Instance
  * @param  ctrl0
  */
__STATIC_INLINE void md_adc_set_ctrl0(ADC_TypeDef *ADCx, uint32_t ctrl0)
{
    WRITE_REG(ADCx->CTRL0, ctrl0);
}

/**
  * @brief  Get ADC_CTRL0  Register
  * @param  ADCx ADC Instance
  * @retval ctrl0 Values
  */
__STATIC_INLINE uint32_t md_adc_get_ctrl0(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->CTRL0);
}

/**
  * @brief  Set ADC reset
  * @param  ADCx ADC Instance
  */
__STATIC_INLINE void md_adc_clear_flag_adc_reset(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_RST);
}

/**
  * @brief  Set ADC current selection
  * @param  ADCx ADC Instance
  * @param  current This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CURSEL_LOW_
  *         @arg @ref MD_ADC_CURSEL_HIGH
  * @retval None
  */
__STATIC_INLINE void md_adc_set_current(ADC_TypeDef *ADCx, uint32_t current)
{
    MODIFY_REG(ADCx->CTRL0, ADC_CTRL0_ADC_CURSEL, current << ADC_CTRL0_ADC_CURSEL_POS);
}

/**
  * @brief  Get ADC current selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CURSEL_LOW_
  *         @arg @ref MD_ADC_CURSEL_HIGH
  */
__STATIC_INLINE uint32_t md_adc_get_current(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_CURSEL) >> ADC_CTRL0_ADC_CURSEL_POS);
}

/**
  * @brief  Set ADC reference voltage range selection
  * @param  ADCx ADC Instance
  * @param  vrrs This parameter can be one of the following values:
  *         @arg @ref MD_ADC_VRRS_1X
  *         @arg @ref MD_ADC_VRRS_0P5X
  * @retval None
  */
__STATIC_INLINE void md_adc_set_vrrs(ADC_TypeDef *ADCx, uint32_t vrrs)
{
    MODIFY_REG(ADCx->CTRL0, ADC_CTRL0_ADC_VRRS, (vrrs & 1U) << ADC_CTRL0_ADC_VRRS_POSS);
}

/**
  * @brief  Get ADC reference voltage range selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_VRRS_1X
  *         @arg @ref MD_ADC_VRRS_0P5X
  */
__STATIC_INLINE uint32_t md_adc_get_vrrs(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL0, (ADC_CTRL0_ADC_VRRS & (1U<<22))) >> ADC_CTRL0_ADC_VRRS_POSS);
}

/**
  * @brief  Set ADC Input Gain Selection
  * @param  ADCx ADC Instance
  * @param  ingn This parameter can be one of the following values:
  *         @arg @ref MD_ADC_INGN_1x
  *         @arg @ref MD_ADC_INGN_2x
  *         @arg @ref MD_ADC_INGN_3x
  *         @arg @ref MD_ADC_INGN_4x
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ingn(ADC_TypeDef *ADCx, uint32_t ingn)
{
    MODIFY_REG(ADCx->CTRL0, ADC_CTRL0_ADC_INGN, ingn << ADC_CTRL0_ADC_INGN_POSS);
}

/**
  * @brief  Get ADC Input Gain Selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_INGN_1x
  *         @arg @ref MD_ADC_INGN_2x
  *         @arg @ref MD_ADC_INGN_3x
  *         @arg @ref MD_ADC_INGN_4x
  */
__STATIC_INLINE uint32_t md_adc_get_ingn(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_INGN) >> ADC_CTRL0_ADC_INGN_POSS);
}

/**
  * @brief  Set ADC reference voltage negative input selection
  * @param  ADCx ADC Instance
  * @param  vrns This parameter can be one of the following values:
  *         @arg @ref MD_ADC_VRNS_VSSH
  *         @arg @ref MD_ADC_VRNS_AIN3
  *         @arg @ref MD_ADC_VRNS_AIN5
  *         @arg @ref MD_ADC_VRNS_VREF
  * @retval None
  */
__STATIC_INLINE void md_adc_set_vrns(ADC_TypeDef *ADCx, uint32_t vrns)
{
    MODIFY_REG(ADCx->CTRL0, ADC_CTRL0_ADC_VRNS, vrns << ADC_CTRL0_ADC_VRNS_POSS);
}

/**
  * @brief  Get ADC reference voltage negative input selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_VRNS_VSSH
  *         @arg @ref MD_ADC_VRNS_AIN3
  *         @arg @ref MD_ADC_VRNS_AIN5
  *         @arg @ref MD_ADC_VRNS_VREF
  */
__STATIC_INLINE uint32_t md_adc_get_vrns(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_VRNS) >> ADC_CTRL0_ADC_VRNS_POSS);
}

/**
  * @brief  Set ADC reference voltage positive terminal input selection
  * @param  ADCx ADC Instance
  * @param  vrps This parameter can be one of the following values:
  *         @arg @ref MD_ADC_VRPS_VDDA
  *         @arg @ref MD_ADC_VRPS_AIN2
  *         @arg @ref MD_ADC_VRPS_AIN4
  *         @arg @ref MD_ADC_VRPS_VREF
  * @retval None
  */
__STATIC_INLINE void md_adc_set_vrps(ADC_TypeDef *ADCx, uint32_t vrps)
{
    MODIFY_REG(ADCx->CTRL0, ADC_CTRL0_ADC_VRPS, vrps << ADC_CTRL0_ADC_VRPS_POSS);
}

/**
  * @brief  Get ADC reference voltage positive terminal input selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_VRPS_VDDA
  *         @arg @ref MD_ADC_VRPS_AIN2
  *         @arg @ref MD_ADC_VRPS_AIN4
  *         @arg @ref MD_ADC_VRPS_VREF
  */
__STATIC_INLINE uint32_t md_adc_get_vrps(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_VRPS) >> ADC_CTRL0_ADC_VRPS_POSS);
}

/**
  * @brief  Set ADC negative input selection
  * @param  ADCx ADC Instance
  * @param  inns This parameter can be one of the following values:
  *         @arg @ref MD_ADC_INNS_IAOUTP
  *         @arg @ref MD_ADC_INNS_IAOUTN
  *         @arg @ref MD_ADC_INNS_AIN2
  *         @arg @ref MD_ADC_INNS_AIN3
  *         @arg @ref MD_ADC_INNS_AIN4
  *         @arg @ref MD_ADC_INNS_AIN5
  *         @arg @ref MD_ADC_INNS_AIN6
  *         @arg @ref MD_ADC_INNS_AIN7
  *         @arg @ref MD_ADC_INNS_TSP
  *         @arg @ref MD_ADC_INNS_TSN
  *         @arg @ref MD_ADC_INNS_VREF
  *         @arg @ref MD_ADC_INNS_VDDA
  *         @arg @ref MD_ADC_INNS_OPOI
  *         @arg @ref MD_ADC_INNS_VDIV
  *         @arg @ref MD_ADC_INNS_VBG
  *         @arg @ref MD_ADC_INNS_VSSH
  * @retval None
  */
__STATIC_INLINE void md_adc_set_inns(ADC_TypeDef *ADCx, uint32_t inns)
{
    MODIFY_REG(ADCx->CTRL0, ADC_CTRL0_ADC_INNS, inns << ADC_CTRL0_ADC_INNS_POSS);
}

/**
  * @brief  Get ADC negative input selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_INNS_IAOUTP
  *         @arg @ref MD_ADC_INNS_IAOUTN
  *         @arg @ref MD_ADC_INNS_AIN2
  *         @arg @ref MD_ADC_INNS_AIN3
  *         @arg @ref MD_ADC_INNS_AIN4
  *         @arg @ref MD_ADC_INNS_AIN5
  *         @arg @ref MD_ADC_INNS_AIN6
  *         @arg @ref MD_ADC_INNS_AIN7
  *         @arg @ref MD_ADC_INNS_TSP
  *         @arg @ref MD_ADC_INNS_TSN
  *         @arg @ref MD_ADC_INNS_VREF
  *         @arg @ref MD_ADC_INNS_VDDA
  *         @arg @ref MD_ADC_INNS_OPOI
  *         @arg @ref MD_ADC_INNS_VDIV
  *         @arg @ref MD_ADC_INNS_VBG
  *         @arg @ref MD_ADC_INNS_VSSH
  */
__STATIC_INLINE uint32_t md_adc_get_inns(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_INNS) >> ADC_CTRL0_ADC_INNS_POSS);
}

/**
  * @brief  Set ADC positive input selection
  * @param  ADCx ADC Instance
  * @param  inps This parameter can be one of the following values:
  *         @arg @ref MD_ADC_INPS_IAOUTP
  *         @arg @ref MD_ADC_INPS_IAOUTN
  *         @arg @ref MD_ADC_INPS_AIN2
  *         @arg @ref MD_ADC_INPS_AIN3
  *         @arg @ref MD_ADC_INPS_AIN4
  *         @arg @ref MD_ADC_INPS_AIN5
  *         @arg @ref MD_ADC_INPS_AIN6
  *         @arg @ref MD_ADC_INPS_AIN7
  *         @arg @ref MD_ADC_INPS_TSP
  *         @arg @ref MD_ADC_INPS_TSN
  *         @arg @ref MD_ADC_INPS_VREF
  *         @arg @ref MD_ADC_INPS_VDDA
  *         @arg @ref MD_ADC_INPS_OPOI
  *         @arg @ref MD_ADC_INPS_VDIV
  *         @arg @ref MD_ADC_INPS_VBG
  *         @arg @ref MD_ADC_INPS_VSSH
  * @retval None
  */
__STATIC_INLINE void md_adc_set_inps(ADC_TypeDef *ADCx, uint32_t inps)
{
    MODIFY_REG(ADCx->CTRL0, ADC_CTRL0_ADC_INPS, inps << ADC_CTRL0_ADC_INPS_POSS);
}

/**
  * @brief  Get ADC positive input selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_INPS_IAOUTP
  *         @arg @ref MD_ADC_INPS_IAOUTN
  *         @arg @ref MD_ADC_INPS_AIN2
  *         @arg @ref MD_ADC_INPS_AIN3
  *         @arg @ref MD_ADC_INPS_AIN4
  *         @arg @ref MD_ADC_INPS_AIN5
  *         @arg @ref MD_ADC_INPS_AIN6
  *         @arg @ref MD_ADC_INPS_AIN7
  *         @arg @ref MD_ADC_INPS_TSP
  *         @arg @ref MD_ADC_INPS_TSN
  *         @arg @ref MD_ADC_INPS_VREF
  *         @arg @ref MD_ADC_INPS_VDDA
  *         @arg @ref MD_ADC_INPS_OPOI
  *         @arg @ref MD_ADC_INPS_VDIV
  *         @arg @ref MD_ADC_INPS_VBG
  *         @arg @ref MD_ADC_INPS_VSSH
  */
__STATIC_INLINE uint32_t md_adc_get_inps(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_INPS) >> ADC_CTRL0_ADC_INPS_POSS);
}

/**
  * @brief  Set ADC chopper selection
  * @param  ADCx ADC Instance
  * @param  chps This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CHPS_CHOP_DISABLE_CDS_DISABLE
  *         @arg @ref MD_ADC_CHPS_CDS_ENABLE
  *         @arg @ref MD_ADC_CHPS_CHOP_ENABLE
  *         @arg @ref MD_ADC_CHPS_CHOP_ENABLE_CDS_ENABLE
  * @retval None
  */
__STATIC_INLINE void md_adc_set_chps(ADC_TypeDef *ADCx, uint32_t chps)
{
    MODIFY_REG(ADCx->CTRL0, ADC_CTRL0_ADC_CHPS, chps << ADC_CTRL0_ADC_CHPS_POSS);
}

/**
  * @brief  Get ADC chopper selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CHPS_CHOP_DISABLE_CDS_DISABLE
  *         @arg @ref MD_ADC_CHPS_CDS_ENABLE
  *         @arg @ref MD_ADC_CHPS_CHOP_ENABLE
  *         @arg @ref MD_ADC_CHPS_CHOP_ENABLE_CDS_ENABLE
  */
__STATIC_INLINE uint32_t md_adc_get_chps(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_CHPS) >> ADC_CTRL0_ADC_CHPS_POSS);
}

/**
  * @brief  Set ADC clock selection
  * @param  ADCx ADC Instance
  * @param  clock This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CKS_250KHZ
  *         @arg @ref MD_ADC_CKS_500KHZ
  *         @arg @ref MD_ADC_CKS_1MHZ
  * @retval None
  */
__STATIC_INLINE void md_adc_set_clock(ADC_TypeDef *ADCx, uint32_t clock)
{
    MODIFY_REG(ADCx->CTRL0, ADC_CTRL0_ADC_CKS, clock << ADC_CTRL0_ADC_CKS_POSS);
}

/**
  * @brief  Get ADC clock selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CKS_250KHZ
  *         @arg @ref MD_ADC_CKS_500KHZ
  *         @arg @ref MD_ADC_CKS_1MHZ
  */
__STATIC_INLINE uint32_t md_adc_get_clock(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_CKS) >> ADC_CTRL0_ADC_CKS_POSS);
}

/**
  * @brief  ADC reference voltage input buffer switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_vr_buffer(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_VRBUF);
}

/**
  * @brief  ADC reference voltage input buffer switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_vr_buffer(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_VRBUF);
}

/**
  * @brief  Check if ADC reference voltage input buffer switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_vr_buffer(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_VRBUF) == (ADC_CTRL0_ADC_VRBUF));
}

/**
  * @brief  ADC reference voltage short-circuit control Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_vr_short(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_VRSHR);
}

/**
  * @brief  ADC reference voltage short-circuit control Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_vr_short(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_VRSHR);
}

/**
  * @brief  Check if ADC reference voltage short-circuit control is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_vr_short(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_VRSHR) == (ADC_CTRL0_ADC_VRSHR));
}

/**
  * @brief  ADC input short-circuit control Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_input_short(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_INSHR);
}

/**
  * @brief  ADC input short-circuit control Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_input_short(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_INSHR);
}

/**
  * @brief  Check if ADC input short-circuit control is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_input_short(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_INSHR) == (ADC_CTRL0_ADC_INSHR));
}

/**
  * @brief  ADC switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_adc(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_EN);
}

/**
  * @brief  ADC switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_adc(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_EN);
}

/**
  * @brief  Check if ADC switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_adc(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CTRL0, ADC_CTRL0_ADC_EN) == (ADC_CTRL0_ADC_EN));
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group4 ADC_CTRL1
  * @{
  */
/**
  * @brief  Set ADC_CTRL1  Register
  * @param  ADCx ADC Instance
  * @param  ctrl1
  */
__STATIC_INLINE void md_adc_set_ctrl1(ADC_TypeDef *ADCx, uint32_t ctrl1)
{
    WRITE_REG(ADCx->CTRL1, ctrl1);
}

/**
  * @brief  Get ADC_CTRL1  Register
  * @param  ADCx ADC Instance
  * @retval ctrl1 Values
  */
__STATIC_INLINE uint32_t md_adc_get_ctrl1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->CTRL1);
}

/**
  * @brief  Get output data stable flag
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_ready(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL1, ADC_CTRL1_READY_FLAG) == (ADC_CTRL1_READY_FLAG));
}

/**
  * @brief  Set ADC oversampling output frequency setting
  * @param  ADCx ADC Instance
  * @param  osr This parameter can be one of the following values:
  *         @arg @ref MD_ADC_OSR_64
  *         @arg @ref MD_ADC_OSR_128
  *         @arg @ref MD_ADC_OSR_256
  *         @arg @ref MD_ADC_OSR_512
  *         @arg @ref MD_ADC_OSR_1024
  *         @arg @ref MD_ADC_OSR_2048
  *         @arg @ref MD_ADC_OSR_4096
  *         @arg @ref MD_ADC_OSR_8192
  *         @arg @ref MD_ADC_OSR_16384
  *         @arg @ref MD_ADC_OSR_32768
  * @retval None
  */
__STATIC_INLINE void md_adc_set_osr(ADC_TypeDef *ADCx, uint32_t osr)
{
    MODIFY_REG(ADCx->CTRL1, ADC_CTRL1_ADC_OSR, osr << ADC_CTRL1_ADC_OSR_POSS);
}

/**
  * @brief  Get ADC oversampling output frequency setting
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_OSR_64
  *         @arg @ref MD_ADC_OSR_128
  *         @arg @ref MD_ADC_OSR_256
  *         @arg @ref MD_ADC_OSR_512
  *         @arg @ref MD_ADC_OSR_1024
  *         @arg @ref MD_ADC_OSR_2048
  *         @arg @ref MD_ADC_OSR_4096
  *         @arg @ref MD_ADC_OSR_8192
  *         @arg @ref MD_ADC_OSR_16384
  *         @arg @ref MD_ADC_OSR_32768
  */
__STATIC_INLINE uint32_t md_adc_get_osr(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL1, ADC_CTRL1_ADC_OSR) >> ADC_CTRL1_ADC_OSR_POSS);
}

/**
  * @brief  Set CIC filter order selection
  * @param  ADCx ADC Instance
  * @param  cic_order This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CIC_ORDER_2
  *         @arg @ref MD_ADC_CIC_ORDER_3
  *         @arg @ref MD_ADC_CIC_ORDER_4
  * @retval None
  */
__STATIC_INLINE void md_adc_set_cic_order(ADC_TypeDef *ADCx, uint32_t cic_order)
{
    MODIFY_REG(ADCx->CTRL1, ADC_CTRL1_CIC_MODE, cic_order << ADC_CTRL1_CIC_MODE_POSS);
}

/**
  * @brief  Get CIC filter order selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CIC_ORDER_2
  *         @arg @ref MD_ADC_CIC_ORDER_3
  *         @arg @ref MD_ADC_CIC_ORDER_4
  *         @arg @ref MD_ADC_CIC_ORDER_RESERVED
  */
__STATIC_INLINE uint32_t md_adc_get_cic_order(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->CTRL1, ADC_CTRL1_CIC_MODE) >> ADC_CTRL1_CIC_MODE_POSS);
}

/**
  * @brief  Set CIC Filter reset
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_clear_flag_cic_filter_reset(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CTRL1, ADC_CTRL1_CIC_RST);
}

/**
  * @brief  ADC CIC Filter switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_cic_filter(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CTRL1, ADC_CTRL1_CIC_EN);
}

/**
  * @brief  ADC CIC Filter switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_cic_filter(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CTRL1, ADC_CTRL1_CIC_EN);
}

/**
  * @brief  Check if CIC Filter switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_cic_filter(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CTRL1, ADC_CTRL1_CIC_EN) == (ADC_CTRL1_CIC_EN));
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group5 ADC_VAL
  * @{
  */
/**
  * @brief  Get ADC_VAL Register
  * @param  ADCx ADC Instance
  * @retval adc value
  */
__STATIC_INLINE uint32_t md_adc_get_value(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->VAL);
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group6 BIQUAD_CTRL
  * @{
  */
/**
  * @brief  Set BIQUAD_CTRL Register
  * @param  ADCx ADC Instance
  * @param  biquad_ctrl
  */
__STATIC_INLINE void md_adc_set_biquad_ctrl(ADC_TypeDef *ADCx, uint32_t biquad_ctrl)
{
    WRITE_REG(ADCx->BIQUAD_CTRL, biquad_ctrl);
}

/**
  * @brief  Get BIQUAD_CTRL Register
  * @param  ADCx ADC Instance
  * @retval biquad_ctrl Values
  */
__STATIC_INLINE uint32_t md_adc_get_biquad_ctrl(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->BIQUAD_CTRL);
}

/**
  * @brief  Set BIQAUD filter software trigger switch
  * @param  ADCx ADC Instance
  */
__STATIC_INLINE void md_adc_clear_flag_biquad_swtrig(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->BIQUAD_CTRL, ADC_BIQUAD_CTRL_BIQUAD_SWTRIG);
}

/**
  * @brief  Set BIQAUD filter trigger mode selection
  * @param  ADCx ADC Instance
  * @param  trigmode This parameter can be one of the following values:
  *         @arg @ref MD_ADC_BIQUAD_TRIGMODE_SWTRIG
  *         @arg @ref MD_ADC_BIQUAD_TRIGMODE_ADC
  * @retval None
  */
__STATIC_INLINE void md_adc_set_biquad_trigmode(ADC_TypeDef *ADCx, uint32_t trigmode)
{
    MODIFY_REG(ADCx->BIQUAD_CTRL, ADC_BIQUAD_CTRL_BIQUAD_TRIGMODE, trigmode << ADC_BIQUAD_CTRL_BIQUAD_TRIGMODE_POS);
}

/**
  * @brief  Get BIQAUD filter trigger mode selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_BIQUAD_TRIGMODE_SWTRIG
  *         @arg @ref MD_ADC_BIQUAD_TRIGMODE_ADC
  */
__STATIC_INLINE uint32_t md_adc_get_biquad_trigmode(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->BIQUAD_CTRL, ADC_BIQUAD_CTRL_BIQUAD_TRIGMODE) >> ADC_BIQUAD_CTRL_BIQUAD_TRIGMODE_POS);
}

/**
  * @brief  Set BIQAUD filter order selection
  * @param  ADCx ADC Instance
  * @param  section This parameter can be one of the following values:
  *         @arg @ref MD_ADC_BIQUAD_SECTION_1
  *         @arg @ref MD_ADC_BIQUAD_SECTION_2
  * @retval None
  */
__STATIC_INLINE void md_adc_set_biquad_section(ADC_TypeDef *ADCx, uint32_t section)
{
    MODIFY_REG(ADCx->BIQUAD_CTRL, ADC_BIQUAD_CTRL_BIQUAD_SECTION, section << ADC_BIQUAD_CTRL_BIQUAD_SECTION_POS);
}

/**
  * @brief  Get BIQAUD filter order selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_BIQUAD_SECTION_1
  *         @arg @ref MD_ADC_BIQUAD_SECTION_2
  */
__STATIC_INLINE uint32_t md_adc_get_biquad_section(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->BIQUAD_CTRL, ADC_BIQUAD_CTRL_BIQUAD_SECTION) >> ADC_BIQUAD_CTRL_BIQUAD_SECTION_POS);
}

/**
  * @brief  Set BIQUAD filter reset
  * @param  ADCx ADC Instance
  */
__STATIC_INLINE void md_adc_clear_flag_biquad_reset(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->BIQUAD_CTRL, ADC_BIQUAD_CTRL_BIQUAD_RST);
}

/**
  * @brief  ADC BIQUAD filter switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_biqaud(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->BIQUAD_CTRL, ADC_BIQUAD_CTRL_BIQAUD_EN);
}

/**
  * @brief  ADC BIQUAD filter switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_biqaud(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->BIQUAD_CTRL, ADC_BIQUAD_CTRL_BIQAUD_EN);
}

/**
  * @brief  Check if BIQUAD filter switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_biqaud(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->BIQUAD_CTRL, ADC_BIQUAD_CTRL_BIQAUD_EN) == (ADC_BIQUAD_CTRL_BIQAUD_EN));
}

/**
  * @}
  */
/** @defgroup MD_ADC_Public_Functions_Group7 BIQUAD_VAL
  * @{
  */

/** @defgroup MD_ADC_Public_Functions_Group7 BIQUAD_VAL
  * @{
  */
/**
  * @brief  Get BIQUAD_VAL Register
  * @param  ADCx ADC Instance
  * @retval biquad value
  */
__STATIC_INLINE uint32_t md_adc_get_biquad_value(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->BIQUAD_VAL);
}
/**
  * @}
  */
/**
  * @brief  Get BIQUAD_VAL Register
  * @param  ADCx ADC Instance
  * @retval biquad value
  */
__STATIC_INLINE void md_adc_set_biquad_in_value(ADC_TypeDef *ADCx, uint32_t biquad_in)
{
    WRITE_REG(ADCx->BIQUAD_IN, biquad_in);
}
/** @defgroup MD_ADC_Public_Functions_Group8 BIQUAD_IN
  * @{
  */
/**
  * @brief  Get BIQUAD_IN Register
  * @param  ADCx ADC Instance
  * @retval biquad in value
  */
__STATIC_INLINE uint32_t md_adc_get_biquad_in_value(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->BIQUAD_IN);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group9 IER
  * @{
  */

/**
  * @brief  Set ADC_IER Register
  * @param  ADCx ADC Instance
  * @param  ier
  */
__STATIC_INLINE void md_adc_set_ier(ADC_TypeDef *ADCx, uint32_t ier)
{
    WRITE_REG(ADCx->IER, ier);
}

/**
  * @brief  BIQUAD complete interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_biquad_done(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_BIQUAD_DONE);
}

/**
  * @brief  ADC complete interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_adc_done(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_ADC_DONE);
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group10 IDR
  * @{
  */
/**
  * @brief  Set ADC_IDR Register
  * @param  ADCx ADC Instance
  * @param  ier
  */
__STATIC_INLINE void md_adc_set_idr(ADC_TypeDef *ADCx, uint32_t idr)
{
    WRITE_REG(ADCx->IDR, idr);
}

/**
  * @brief  BIQUAD complete interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_biquad_done(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_BIQUAD_DONE);
}

/**
  * @brief  ADC complete interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_adc_done(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_ADC_DONE);
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group11 IVS
  * @{
  */
/**
  * @brief  Get ADC_IVS Register
  * @param  ADCx ADC Instance
  * @param  ier
  */
__STATIC_INLINE uint32_t md_adc_get_ivs(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->IVS);
}

/**
  * @brief  Check if BIQUAD complete interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_biquad_done(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_BIQUAD_DONE) == (ADC_IVS_BIQUAD_DONE));
}

/**
  * @brief  Check if ADC complete interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_adc_done(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_ADC_DONE) == (ADC_IVS_ADC_DONE));
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group12 RIF
  * @{
  */

/**
  * @brief  Get ADC_RIF Register
  * @param  ADCx ADC Instance
  * @param  ier
  */
__STATIC_INLINE uint32_t md_adc_get_rif(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->RIF);
}

/**
  * @brief  Check BIQUAD complete interrupt Flag Status.
  * @param  ADCx ADC Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_biquad_done(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_BIQUAD_DONE) == (ADC_RIF_BIQUAD_DONE));
}

/**
  * @brief  Check ADC complete interrupt Flag Status.
  * @param  ADCx ADC Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_adc_done(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_ADC_DONE) == (ADC_RIF_ADC_DONE));
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group13 IFM
  * @{
  */
/**
  * @brief  Get ADC_IFM Register
  * @param  ADCx ADC Instance
  * @param  ier
  */
__STATIC_INLINE uint32_t md_adc_get_ifm(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->IFM);
}

/**
  * @brief  Check BIQUAD complete interrupt Flag Masked Status.
  * @param  ADCx ADC Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_biquad_done(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_BIQUAD_DONE) == (ADC_IFM_BIQUAD_DONE));
}

/**
  * @brief  Check ADC complete interrupt Flag Masked Status.
  * @param  ADCx ADC Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_adc_done(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_ADC_DONE) == (ADC_IFM_ADC_DONE));
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group15 ICR
  * @{
  */
/**
  * @brief  Set ADC_ICR Register
  * @param  ADCx ADC Instance
  * @param  ier
  */
__STATIC_INLINE void md_adc_set_icr(ADC_TypeDef *ADCx, uint32_t icr)
{
    WRITE_REG(ADCx->ICR, icr);
}

/**
  * @brief  ADC BIQUAD complete interrupt Clear.
  * @note   User can set this bit to clear timeout interrupt status.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_biquad_done(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_BIQUAD_DONE);
}

/**
  * @brief  ADC ADC complete interrupt Clear.
  * @note   User can set this bit to clear Analog Watchdog interrupt status.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_adc_done(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_ADC_DONE);
}

/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group16 COEF_SC_1
  * @{
  */

/**
  * @brief  set COEF_SC_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_sc_1
  */
__STATIC_INLINE void md_adc_set_coef_sc_1(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_SC_1, value);
}

/**
  * @brief  Get COEF_SC_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_sc_1
  */
__STATIC_INLINE uint32_t md_adc_get_coef_sc_1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_SC_1);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group17 COEF_B1_1
  * @{
  */

/**
  * @brief  set COEF_B1_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_b1_1
  */
__STATIC_INLINE void md_adc_set_coef_b1_1(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_B1_1, value);
}

/**
  * @brief  Get COEF_B1_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_b1_1
  */
__STATIC_INLINE uint32_t md_adc_get_coef_b1_1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_B1_1);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group18 COEF_B2_1
  * @{
  */

/**
  * @brief  set COEF_B2_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_b2_1
  */
__STATIC_INLINE void md_adc_set_coef_b2_1(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_B2_1, value);
}

/**
  * @brief  Get COEF_B2_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_b2_1
  */
__STATIC_INLINE uint32_t md_adc_get_coef_b2_1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_B2_1);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group19 COEF_B3_1
  * @{
  */

/**
  * @brief  set COEF_B3_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_b3_1
  */
__STATIC_INLINE void md_adc_set_coef_b3_1(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_B3_1, value);
}

/**
  * @brief  Get COEF_B3_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_b3_1
  */
__STATIC_INLINE uint32_t md_adc_get_coef_b3_1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_B3_1);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group20 COEF_A2_1
  * @{
  */

/**
  * @brief  set COEF_A2_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_a2_1
  */
__STATIC_INLINE void md_adc_set_coef_a2_1(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_A2_1, value);
}

/**
  * @brief  Get COEF_A2_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_a2_1
  */
__STATIC_INLINE uint32_t md_adc_get_coef_a2_1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_A2_1);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group21 COEF_A3_1
  * @{
  */

/**
  * @brief  set COEF_A3_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_a3_1
  */
__STATIC_INLINE void md_adc_set_coef_a3_1(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_A3_1, value);
}

/**
  * @brief  Get COEF_A3_1 Register
  * @param  ADCx ADC Instance
  * @retval coef_a3_1
  */
__STATIC_INLINE uint32_t md_adc_get_coef_a3_1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_A3_1);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group22 COEF_SC_2
  * @{
  */

/**
  * @brief  set COEF_SC_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_sc_2
  */
__STATIC_INLINE void md_adc_set_coef_sc_2(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_SC_2, value);
}

/**
  * @brief  Get COEF_SC_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_sc_2
  */
__STATIC_INLINE uint32_t md_adc_get_coef_sc_2(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_SC_2);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group23 COEF_B1_2
  * @{
  */

/**
  * @brief  set COEF_B1_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_b1_2
  */
__STATIC_INLINE void md_adc_set_coef_b1_2(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_B1_2, value);
}

/**
  * @brief  Get COEF_B1_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_b1_2
  */
__STATIC_INLINE uint32_t md_adc_get_coef_b1_2(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_B1_2);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group24 COEF_B2_2
  * @{
  */

/**
  * @brief  set COEF_B2_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_b2_2
  */
__STATIC_INLINE void md_adc_set_coef_b2_2(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_B2_2, value);
}

/**
  * @brief  Get COEF_B2_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_b2_2
  */
__STATIC_INLINE uint32_t md_adc_get_coef_b2_2(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_B2_2);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group25 COEF_B3_2
  * @{
  */

/**
  * @brief  set COEF_B3_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_b3_2
  */
__STATIC_INLINE void md_adc_set_coef_b3_2(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_B3_2, value);
}

/**
  * @brief  Get COEF_B3_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_b3_2
  */
__STATIC_INLINE uint32_t md_adc_get_coef_b3_2(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_B3_2);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group26 COEF_A2_2
  * @{
  */

/**
  * @brief  set COEF_A2_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_a2_2
  */
__STATIC_INLINE void md_adc_set_coef_a2_2(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_A2_2, value);
}

/**
  * @brief  Get COEF_A2_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_a2_2
  */
__STATIC_INLINE uint32_t md_adc_get_coef_a2_2(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_A2_2);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group27 COEF_A3_2
  * @{
  */

/**
  * @brief  set COEF_A3_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_a3_2
  */
__STATIC_INLINE void md_adc_set_coef_a3_2(ADC_TypeDef *ADCx, uint32_t value)
{
    WRITE_REG(ADCx->COEF_A3_2, value);
}

/**
  * @brief  Get COEF_A3_2 Register
  * @param  ADCx ADC Instance
  * @retval coef_a3_2
  */
__STATIC_INLINE uint32_t md_adc_get_coef_a3_2(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->COEF_A3_2);
}
/**
  * @}
  */

/** @defgroup MD_ADC_Public_Functions_Group1 Initialization
  * @{
  */

void md_adc_power_init(RCU_TypeDef *rcu, ANPWR_TypeDef *anpwr, md_adc_power_init_typedef *power_init);
void md_adc_reset(RCU_TypeDef *rcu);
void md_ia_power_init(ADC_TypeDef *ADCx, md_ia_init_typedef *ia_init);
void md_adc_basic_init(ADC_TypeDef *ADCx, md_adc_init_typedef *adc_init);
void md_adc_enable(ADC_TypeDef *ADCx);
void md_adc_basic_config(ADC_TypeDef *ADCx, md_adc_config_typedef *adc_config);
void md_adc_init(ADC_TypeDef *ADCx, md_adc_power_init_typedef *power_init, md_adc_init_typedef *adc_init, md_adc_config_typedef *adc_config);
void md_biquad_init(ADC_TypeDef *ADCx, md_biquad_init_typedef *biquad_init);
void md_biquad_coefficient_calculate(uint32_t *coef, double num);
void md_biquad_coefficient_determines(ADC_TypeDef *ADCx, uint32_t sampling_frequency, uint32_t cut_off_frequency, md_biquad_order_pass_filter order);
double md_biquad_coefficient_reverse_calculate(uint32_t coef);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @} Micro_Driver
  */


#ifdef __cplusplus
}
#endif
#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
