/**********************************************************************************
 *
 * @file    md_uart.c
 * @brief   md_uart C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "system_es32f0930.h"
#include "md_uart.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup UART UART
  * @brief UART micro driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/

/** @defgroup UART_PUB_FUNC UART Public Functions
  * @brief UART Public Functions
  * @{
  */
/**
  * @brief  Set UARTx baudrate.
  * @param  UARTx UART Instance.
  * @param  Baudrate
  * @retval None.
  */
void md_uart_set_baudrate_calculate(UART_TypeDef *UARTx, uint32_t baudrate)
{
    uint32_t brr;
    brr = ((SystemFrequency_APBClk + (baudrate >> 1)) / baudrate);
    md_uart_set_baudrate(UARTx, brr);
}

/**
  * @brief  UART Initialization
  * @param  UART Init Structure
  * @retval None
  */
void md_uart_init(UART_TypeDef *UARTx, md_uart_init_typedef *UART_InitStruct)
{
    /* Check the UART_InitStruct */
    assert_param((UART_InitStruct != NULL));

    md_uart_disable_tx(UARTx);
    md_uart_disable_rx(UARTx);
    md_uart_set_baudrate_calculate(UARTx, UART_InitStruct->BaudRate);
    md_uart_set_datawidth(UARTx, UART_InitStruct->DataWidth);
    md_uart_set_stop(UARTx, UART_InitStruct->StopBits);

    if (UART_InitStruct->Parity)
    {
        if (UART_InitStruct->Parity >> 1)
            md_uart_set_parity(UARTx, MD_UART_LCON_PS_EVEN);
        else
            md_uart_set_parity(UARTx, MD_UART_LCON_PS_ODD);
    }
    else
        md_uart_set_parity(UARTx, MD_UART_LCON_PS_NONE);

    md_uart_enable_tfrst(UARTx);
    md_uart_enable_rfrst(UARTx);
    md_uart_enable_tx(UARTx);
    md_uart_enable_rx(UARTx);
}

/**
  * @brief  UARTx send character
  * @param  Character
  * @retval None
  */
void md_uart_send(UART_TypeDef *UARTx, uint8_t ch)
{
    while (md_uar_is_active_flag_tffull(UARTx)); // Tx FIFO full

    md_uart_set_send_data8(UARTx, ch);                    // Sent byte
}

/**
  * @brief  UARTx receive character.
  * @param  UART_TypeDef *UARTx.
  * @retval unsigned char.
  */
uint8_t md_uart_recv(UART_TypeDef *UARTx)
{
    while (md_uart_is_active_flag_rfempty(UARTx));           // Rx FIFO empty

    return (md_uart_get_recv_data8(UARTx));
}

/**
  * @brief  Get UARTx baudrate.
  * @param  UART_TypeDef *UARTx.
  * @retval Baudrate.
  */
uint32_t md_uart_get_baudrate_calculate(UART_TypeDef *UARTx)
{
    uint32_t brr, baudrate;

    brr = md_uart_get_baudrate(UARTx);
    baudrate = SystemFrequency_APBClk / brr;
    return baudrate;
}


/**
  * @} UART_PUB_FUNC UART Public Functions
  */

/**
  * @} UART
  */

/**
  * @} Micro_Driver
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

