/**********************************************************************************
 *
 * @file    md_spi.c
 * @brief   md_spi C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "md_spi.h"
#include "md_rcu.h"
#include <stdio.h>
#include "stdint.h"
/** @addtogroup Micro_Driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/

/**
  * @brief  Initialize the SPI registers according to the specified parameters in SPI_initStruct.
  * @note   The parameters in md_spi_init should be expected values. Otherwise, ERROR result will be returned.
  * @param  SPIx SPI Instance
  * @param  SPI_initStruct pointer to a @ref SPI_initStruct structure
  * @retval An ErrorStatus enumeration value. (Return always SUCCESS)
  */
void md_spi_struct_init(md_spi_inittypedef *SPI_initStruct)
{
    SPI_initStruct->Mode               = MD_SPI_MODE_MASTER;
    SPI_initStruct->ClockPhase         = MD_SPI_PHASE_1EDGE;
    SPI_initStruct->ClockPolarity      = MD_SPI_POLARITY_LOW;
    SPI_initStruct->BaudRate           = MD_SPI_BAUDRATEPRESCALER_DIV256;
    SPI_initStruct->BitOrder           = MD_SPI_MSB_FIRST;
    SPI_initStruct->TransferDirection  = MD_SPI_FULL_DUPLEX;
    SPI_initStruct->DataWidth          = MD_SPI_FRAME_FORMAT_8BIT;
    SPI_initStruct->CRCCalculation     = MD_SPI_CRCCALCULATION_DISABLE;
    SPI_initStruct->CRCPoly            = 0x7;
    SPI_initStruct->NSS                = MD_SPI_NSS_HARD;
}

/**
  * @brief  Initialize the SPI registers according to the specified parameters in SPI_initStruct.
  * @note   The parameters in md_i2c_init should be expected values. Otherwise, ERROR result will be returned.
  * @param  SPIx SPI Instance
  * @param  SPI_initStruct pointer to a @ref SPI_initStruct structure
  * @retval An ErrorStatus enumeration value. (Return always SUCCESS)
  */
ErrorStatus md_spi_init(SPI_TypeDef *SPIx, md_spi_inittypedef *SPI_InitStruct)
{
    ErrorStatus status = ERROR;

    md_spi_disable(SPIx);

    if (SPI_InitStruct->Mode)
        md_spi_enable_master_mode(SPIx);
    else
        md_spi_enable_slave_mode(SPIx);

    if (SPI_InitStruct->ClockPhase)
        md_spi_enable_sample_sec_edge(SPIx);
    else
        md_spi_enable_sample_first_edge(SPIx);

    if (SPI_InitStruct->ClockPolarity)
        md_spi_enable_sck_high_idle(SPIx);
    else
        md_spi_enable_sck_low_idle(SPIx);

    md_spi_set_commumication_baud(SPIx, SPI_InitStruct->BaudRate);

    if (SPI_InitStruct->BitOrder)
        md_spi_enable_first_transmit_lsb(SPIx);
    else
        md_spi_enable_first_transmit_msb(SPIx);

    if (SPI_InitStruct->TransferDirection)
        md_spi_set_simplex_line_duplex(SPIx);
    else
        md_spi_set_duplex_line_simplex(SPIx);

    if (SPI_InitStruct->NSS)
        md_spi_enable_ss_output_high(SPIx);
    else
        md_spi_enable_ss_output_low(SPIx);

    md_spi_enable_nss_output(SPIx);

    md_spi_enable(SPIx);

    status = SUCCESS;

    return status;
}

/**
  * @brief  use to receive single one byte data.
  * @note   SPI Slave receive data
  * @param  SPIx SPI Instance
  */
uint8_t SPISSingleRd(SPI_TypeDef *SPIx)
{
    while (md_spi_is_active_flag_rxe(SPIx));

    return (md_spi_get_data_reg(SPIx));
}


/**
  * @brief  use to send single one byte data.
  * @note   SPI receive data
  * @param  SPIx SPI Instance
  * @param  data SPI send one byte data.
  */
void SPISingleWr(SPI_TypeDef *SPIx, uint8_t data)
{
    while (md_spi_is_active_flag_txf(SPIx));

    md_spi_set_data_reg(SPIx, data);
}

/**
  * @} Micro_Driver
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
