/**********************************************************************************
 *
 * @file    md_rcu.c
 * @brief   md_rcu C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "md_rcu.h"
#include "md_fc.h"
#include "system_es32f0930.h"
#include <stdio.h>
/** @addtogroup Micro_Driver
  * @{
  */


/** @defgroup RCU RCU
  * @brief RCU micro driver
  * @{
  */


/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/** @defgroup MD_RCU_Private_Macros RCU Private Macros
  * @{
  */


/**
  * @} MD_RCU_Private_Macros
  */

/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/
/** @addtogroup MD_RCU_Public_Functions RCU Public Functions
  * @{
  */

/** @addtogroup MD_RCU_PF_Init RCU Public Init Functions
  * @{
  */

void md_rcu_init_set(RCU_TypeDef *rcu, md_rcu_init_typedef *RCU_InitStruct)
{
    uint32_t HRC_Divider_Frequency;

    md_fc_set_read_latency(3);

    if (RCU_InitStruct->SysClock & RCU_CON_LRCON_MSK)
        md_rcu_enable_lrc(rcu);
    else
        md_rcu_disable_lrc(rcu);

    if (RCU_InitStruct->SysClock & RCU_CON_HOSCON_MSK)
        md_rcu_enable_hosc(rcu);
    else
        md_rcu_disable_hosc(rcu);

    if (RCU_InitStruct->SysClock & RCU_CON_HRCON_MSK)
        md_rcu_enable_hrc(rcu);
    else
        md_rcu_disable_hrc(rcu);


    if (RCU->CON & RCU_CON_HRCON_MSK)                 /* if HRC enbale */
        while (md_rcu_is_active_flag_hrc_ready(rcu) == 0);  /* Wait for HRCRDY = 1 (HRC is ready)*/


    if (RCU->CON & RCU_CON_HOSCON_MSK)                  /* if HOSC enbale */
        while (md_rcu_is_active_flag_hosc_ready(rcu) == 0);  /* Wait for HOSCRDY = 1 (HOSC is ready)*/


    if (RCU->CON & RCU_CON_LRCON_MSK)                  /* if LRC enbale */
        while (md_rcu_is_active_flag_lrc_ready(rcu) == 0);  /* Wait for LRCRDY = 1 (LRC is ready)*/

    md_rcu_set_mco_div(rcu, RCU_InitStruct->Mpre);
    md_rcu_set_mco_source(rcu, RCU_InitStruct->Msw);
    md_rcu_set_cfg_hrcdiv(rcu, RCU_InitStruct->HrcDiv);
    md_rcu_set_pclk_div(rcu, RCU_InitStruct->Ppre);
    md_rcu_set_hclk_div(rcu, RCU_InitStruct->Hpre);
    md_rcu_set_system_clock_source(rcu, RCU_InitStruct->Sw);

    // HRC Divider Frequency
    if (RCU->CFG & RCU_CFG_HRCDIV_MSK)
    {
        HRC_Divider_Frequency = (__HRC) >> (((RCU_InitStruct->HrcDiv) & 0x03) + 1);
    }
    else
    {
        HRC_Divider_Frequency = (uint32_t)(__HRC);
    }

    /* System Frequency */
    switch (md_rcu_get_system_clock_source(RCU)) /* System clock switch(SYSCLK) */
    {
        case 0: /*================= HRC selected as system clock*/
            SystemCoreClock = HRC_Divider_Frequency;
            break;

        case 1: /*================= HOSC selected as system clock*/
            SystemCoreClock = (uint32_t)(__HOSC);
            break;

        case 2: /*================= LRC selected as system clock*/
            SystemCoreClock = (uint32_t)(__LRC);
            break;

        default:
            SystemCoreClock = (uint32_t)(__HRC);
            break;
    }

    /* Core Frequency */
    SystemFrequency_SysClk = SystemCoreClock;

    /* AHB Frequency */
    if (md_rcu_get_hclk_div(RCU))
        SystemFrequency_AHBClk = SystemFrequency_SysClk >> ((md_rcu_get_hclk_div(RCU) & 0x07) + 1);
    else
        SystemFrequency_AHBClk = SystemFrequency_SysClk;

    /* APB Frequency */
    if (md_rcu_get_pclk_div(RCU))
        SystemFrequency_APBClk = SystemFrequency_AHBClk >> ((md_rcu_get_pclk_div(RCU) & 0x03) + 1);
    else
        SystemFrequency_APBClk = SystemFrequency_AHBClk;

    md_fc_set_read_latency(0);
}


/**
  * @} MD_RCU_PF_Init
  */

/**
  * @} MD_RCU_Public_Functions
  */

/**
  * @} RCU
  */

/**
  * @} Micro_Driver
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

