/**********************************************************************************
 *
 * @file    md_lcd.c
 * @brief   md_lcd C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "md_lcd.h"
#include "md_rcu.h"
#include "system_es32f0930.h"
#include <stdio.h>
#include <string.h>
/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup LCD LCD
  * @brief LCD micro driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
static uint16_t u16DispCom0, u16DispCom1, u16DispCom2, u16DispCom3;
static uint16_t u16PointCom0 = 0, u16PointCom1 = 0, u16PointCom2 = 0, u16PointCom3 = 0;
//         3 3 2 2  1 1 0 0
const uint16_t Seven_Segment_TABLE[16] =        //n d  B A G F  C E D p
{
    0xDE, //0 0  1 1 0 1  1 1 1 0
    0x88, //1 1  1 0 0 0  1 0 0 0
    0xE6, //2 2  1 1 1 0  0 1 1 0
    0xEA, //3 3  1 1 1 0  1 0 1 0
    0xB8, //4 4  1 0 1 1  1 0 0 0
    0x7A, //5 5  0 1 1 1  1 0 1 0
    0x7E, //6 6  0 1 1 1  1 1 1 0
    0xC8, //7 7  1 1 0 0  1 0 0 0
    0xFE, //8 8  1 1 1 1  1 1 1 0
    0xFA, //9 9  1 1 1 1  1 0 1 0
    0x00, //A    0 0 0 0  0 0 0 0
    0x01, //B .  0 0 0 0  0 0 0 1
};
/* Private constants ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/** @defgroup MD_LCD_Private_Macros LCD Private Macros
  * @{
  */

/**
  * @} MD_LCD_Private_Macros
  */

/* Private function prototypes ------------------------------------------------*/
void SetTempLcdData(uint16_t *u16pData)
{
    uint16_t u16DispData;
    uint16_t u16D0, u16D1, u16D2, u16D3;

    u16D0 = u16pData[3];
    u16D0 = Seven_Segment_TABLE[u16D0];
    u16D1 = u16pData[2];
    u16D1 = Seven_Segment_TABLE[u16D1];
    u16D2 = u16pData[1];
    u16D2 = Seven_Segment_TABLE[u16D2];
    u16D3 = u16pData[0];
    u16D3 = Seven_Segment_TABLE[u16D3];

    u16DispData   = u16DispCom0 & 0xFF00;
    u16DispData  |= ((u16D0 & 0x00C0) | ((u16D1 & 0x00C0) >> 2) | ((u16D2 & 0x00C0) >> 4) | ((u16D3 & 0x00C0) >> 6));
    u16DispCom0   = u16DispData;

    u16DispData   = u16DispCom1 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x0030) << 2) | (u16D1 & 0x0030) | ((u16D2 & 0x0030) >> 2) | ((u16D3 & 0x0030) >> 4));
    u16DispCom1   = u16DispData;

    u16DispData   = u16DispCom2 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x000C) << 4) | ((u16D1 & 0x000C) << 2) | (u16D2 & 0x000C) | ((u16D3 & 0x000C) >> 2));
    u16DispCom2   = u16DispData;

    u16DispData   = u16DispCom3 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x0003) << 6) | ((u16D1 & 0x0003) << 4) | ((u16D2 & 0x0003) << 2) | (u16D3 & 0x0003));
    u16DispCom3   = u16DispData;
}

void SetPointLcdData(uint16_t *u16pData)
{
    uint16_t u16DispData;
    uint16_t u16D0 = 0, u16D1 = 0, u16D2 = 0, u16D3 = 0;

    if (u16pData[3] == 0xb)
    {
        u16D0 = u16pData[3];
        u16D0 = Seven_Segment_TABLE[u16D0];
    }

    if (u16pData[2] == 0xb)
    {
        u16D1 = u16pData[2];
        u16D1 = Seven_Segment_TABLE[u16D1];
    }

    if (u16pData[1] == 0xb)
    {
        u16D2 = u16pData[1];
        u16D2 = Seven_Segment_TABLE[u16D2];
    }

    if (u16pData[0] == 0xb)
    {
        u16D3 = u16pData[0];
        u16D3 = Seven_Segment_TABLE[u16D3];
    }

    u16DispData   = u16PointCom0 & 0xFF00;
    u16DispData  |= ((u16D0 & 0x00C0) | ((u16D1 & 0x00C0) >> 2) | ((u16D2 & 0x00C0) >> 4) | ((u16D3 & 0x00C0) >> 6));
    u16PointCom0   = u16DispData;



    u16DispData   = u16PointCom1 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x0030) << 2) | (u16D1 & 0x0030) | ((u16D2 & 0x0030) >> 2) | ((u16D3 & 0x0030) >> 4));
    u16PointCom1   = u16DispData;


    u16DispData   = u16PointCom2 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x000C) << 4) | ((u16D1 & 0x000C) << 2) | (u16D2 & 0x000C) | ((u16D3 & 0x000C) >> 2));
    u16PointCom2   = u16DispData;



    u16DispData   = u16PointCom3 & 0xFF00;
    u16DispData  |= (((u16D0 & 0x0003) << 6) | ((u16D1 & 0x0003) << 4) | ((u16D2 & 0x0003) << 2) | (u16D3 & 0x0003));
    u16PointCom3   = u16DispData;


}
void DispLcdData(void)
{
    md_lcd_set_com0(LCD, (u16DispCom0) | u16PointCom0);
    md_lcd_set_com1(LCD, (u16DispCom1) | u16PointCom1);
    md_lcd_set_com2(LCD, (u16DispCom2) | u16PointCom2);
    md_lcd_set_com3(LCD, (u16DispCom3) | u16PointCom3);
}
/* Public functions -----------------------------------------------------------*/
/** @addtogroup MD_LCD_Public_Functions LCD Public Functions
  * @{
  */

void md_rcu_lcd_init(RCU_TypeDef *rcu, md_rcu_lcd_init_typedef *LCD_InitStruct)
{
    if (LCD_InitStruct -> VlcdCpSw == MD_RCU_IPCFG_VLCDCPSW_HOSC)
        md_rcu_set_ipcfg_vlcdsw_hosc(RCU);
    else
        md_rcu_set_ipcfg_vlcdsw_hrc(RCU);


    md_rcu_set_ipcfg_vlcddiv(RCU, LCD_InitStruct -> VlcdCpDiv);

    if (LCD_InitStruct -> VlcdCpEn & RCU_IPCFG_VLCDEN_MSK)
        md_rcu_enable_ipcfg_vlcden(RCU);
    else
        md_rcu_disable_ipcfg_vlcden(RCU);

    if (LCD_InitStruct -> LcdSw == MD_RCU_IPCFG_LCDSW_DSIABLE)
        md_rcu_set_ipcfg_lcdsw(RCU, MD_RCU_IPCFG_LCDSW_DSIABLE);
    else if (LCD_InitStruct -> LcdSw ==  MD_RCU_IPCFG_LCDSW_PCLK)
        md_rcu_set_ipcfg_lcdsw(RCU, MD_RCU_IPCFG_LCDSW_PCLK);
    else
        md_rcu_set_ipcfg_lcdsw(RCU, MD_RCU_IPCFG_LCDSW_LRC);

    md_rcu_set_ipcfg_lcddiv(RCU, LCD_InitStruct->LcdDiv);
}

void md_lcd_mode(LCD_TypeDef *lcd, md_lcd_mode_typedef *LCD_Mode_Struct)
{
    if (LCD_Mode_Struct -> CpEn & LCD_MODE_CPEN_MSK)
        md_lcd_enable_cpen(LCD);
    else
        md_lcd_disable_cpen(LCD);

    if (LCD_Mode_Struct -> BufEn & LCD_MODE_BUFEN_MSK)
        md_lcd_enable_bufen(LCD);
    else
        md_lcd_disable_bufen(LCD);

    md_lcd_set_vsl(LCD, LCD_Mode_Struct->Vsl);


    if (LCD_Mode_Struct -> Rsel == MD_LCD_MODE_RSEL_900K)
        md_lcd_set_rsel_900k(LCD);
    else
        md_lcd_set_rsel_225k(LCD);

    if (LCD_Mode_Struct -> Type == MD_LCD_MODE_TYPE_B)
        md_lcd_set_typeB(LCD);
    else
        md_lcd_set_typeA(LCD);

    md_lcd_set_bias(LCD, LCD_Mode_Struct->Bias);
    md_lcd_set_duty(LCD, LCD_Mode_Struct->Duty);

}

void md_lcd_comx_segx_enable(LCD_TypeDef *lcd, md_lcd_comx_segx_typedef *LCD_Comx_Segx_Enable_Struct)
{
    md_lcd_set_com_en(LCD, LCD_Comx_Segx_Enable_Struct->ComEn);
    md_lcd_set_seg_en(LCD, LCD_Comx_Segx_Enable_Struct->SegEn);
}

void md_lcd_ctrl(LCD_TypeDef *lcd, md_lcd_ctrl_typedef *LCD_ctrl_InitStruct)
{
    md_lcd_set_frdiv(LCD, LCD_ctrl_InitStruct->FrDiv);
    md_lcd_set_updctrl(LCD, LCD_ctrl_InitStruct->UpdCtrl);
    md_lcd_setl_wsblink(LCD, LCD_ctrl_InitStruct->WsBlink);
    md_lcd_set_fcvalue(LCD, LCD_ctrl_InitStruct->FcValue);

    if (LCD_ctrl_InitStruct->FcCtrl & LCD_CTRL_FCCTRL_MSK)
        md_lcd_enable_fcctrl(LCD);
    else
        md_lcd_disable_fcctrl(LCD);

    if (LCD_ctrl_InitStruct->SwUpdate & LCD_CTRL_SWUPDATE_MSK)
        md_lcd_enable_swupdate(LCD);
    else
        md_lcd_disable_swupdate(LCD);


    if (LCD_ctrl_InitStruct->LcdEn & LCD_CTRL_LCDEN_MSK)
        md_lcd_enable_lcden(LCD);
    else
        md_lcd_disable_lcden(LCD);
}

void md_lcd_display_all()
{
    md_lcd_set_com0(LCD, 0xffff);
    md_lcd_set_com1(LCD, 0xffff);
    md_lcd_set_com2(LCD, 0xffff);
    md_lcd_set_com3(LCD, 0xffff);
}

void md_lcd_clear_all()
{
    md_lcd_set_com0(LCD, 0);
    md_lcd_set_com1(LCD, 0);
    md_lcd_set_com2(LCD, 0);
    md_lcd_set_com3(LCD, 0);
}

void md_lcd_display_float_number(float u32DispVal)
{
    uint32_t u32Data;
    uint16_t au16DispNum[4];
    uint16_t au16PointNum[4] = {0};
    int i = 0;
    uint32_t decimal_places = 0;
    int temp_DispVal = (int)u32DispVal;

    for (i = 0; i < 3; i++)
    {
        temp_DispVal /= 10;

        if (temp_DispVal == 0)
        {
            decimal_places = 3 - i;
            break;
        }

    }

    if (decimal_places > 0)
    {
        au16PointNum[4 - decimal_places] = 0xB;

        for (i = 0; i < decimal_places; i++)
        {
            u32DispVal *= 10;
        }
    }



    u32Data  = (uint32_t)u32DispVal;

    while (u32Data >= 10000)
    {
        u32Data /= 10;
    }

//  HDIV->DIVDR = u32Data;
//  HDIV->DIVSR = 10;
    au16DispNum[3]  = u32Data % 10;

    u32Data /= 10;

//  HDIV->DIVDR = HDIV->DIVQR;
//  HDIV->DIVSR = 10;
    au16DispNum[2]  = u32Data % 10;

//  HDIV->DIVDR = HDIV->DIVQR;
//  HDIV->DIVSR = 10;
    u32Data /= 10;

    au16DispNum[1]  = u32Data % 10;

    u32Data /= 10;

    au16DispNum[0]  = u32Data % 10;




    SetTempLcdData(au16DispNum);
    SetPointLcdData(au16PointNum);
    DispLcdData();
}

void md_lcd_write(LCD_TypeDef *hlcd, uint8_t ComIndex, uint32_t RegisterMask, uint32_t Data)
{
    if (ComIndex == 0)
    {
        MODIFY_REG(hlcd->COM0, ~(RegisterMask), Data);
    }
    else if (ComIndex == 1)
    {
        MODIFY_REG(hlcd->COM1, ~(RegisterMask), Data);
    }
    else if (ComIndex == 2)
    {
        MODIFY_REG(hlcd->COM2, ~(RegisterMask), Data);
    }
    else if (ComIndex == 3)
    {
        MODIFY_REG(hlcd->COM3, ~(RegisterMask), Data);
    }
    else
    {
        printf("Error Index\r\n");
    }

}

void md_lcd_swupdate()
{
    md_lcd_is_enabled_swupdate(LCD);
}


/**
  * @} MD_LCD_Public_Functions
  */

/**
  * @} LCD
  */

/**
  * @} Micro_Driver
  */
/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

