/**********************************************************************************
 *
 * @file    md_i2c.c
 * @brief   md_i2c C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "md_i2c.h"
#include "md_rcu.h"
#include "md_tick.h"
#include <stdio.h>
#include "stdint.h"
/** @addtogroup Micro_Driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/

/**
  * @brief  Initialize the I2C registers according to the specified parameters in I2C_InitStruct.
  * @note   The parameters in md_i2c_init should be expected values. Otherwise, ERROR result will be returned.
  * @param  I2Cx I2C Instance
  * @param  I2C_InitStruct pointer to a @ref md_i2c_inittypedef structure
  * @retval An ErrorStatus enumeration value. (Return always SUCCESS)
  */
ErrorStatus md_i2c_init(I2C_TypeDef *I2Cx, md_i2c_inittypedef *I2C_InitStruct)
{
    ErrorStatus status = ERROR;

    /* Check the I2C Instance I2Cx */
    assert_param(IS_MD_I2C_ALL_INSTANCE(I2Cx));

    /* Check the I2C parameters from I2C_InitStruct */
    assert_param(IS_MD_I2C_TIMING(I2C_InitStruct->Timing));
    assert_param(IS_MD_I2C_ADDRSIZE(I2C_InitStruct->AddrSize));
    assert_param(IS_MD_I2C_ADDRESS1(I2C_InitStruct->Address1));
    assert_param(IS_MD_I2C_DUALADDRESSMODE(I2C_InitStruct->DualAddressMode));
    assert_param(IS_MD_I2C_ADDRESS2(I2C_InitStruct->Address2));
    assert_param(IS_MD_I2C_ADDRESS2MASKS(I2C_InitStruct->Address2Masks));
    /* Check the PLL clock, if not corrt, modify it */

    md_i2c_disable_pe(I2Cx);

    md_i2c_set_clock_prescaler(I2Cx, ((I2C_InitStruct->Timing) >> 28) & 0xF);
    md_i2c_set_data_setup_time(I2Cx, ((I2C_InitStruct->Timing) >> 20) & 0xF);
    md_i2c_set_data_hold_time(I2Cx, ((I2C_InitStruct->Timing) >> 16) & 0xF);
    md_i2c_set_clock_high_period(I2Cx, ((I2C_InitStruct->Timing) >> 8) & 0xFF);
    md_i2c_set_clock_low_period(I2Cx, ((I2C_InitStruct->Timing)) & 0xFF);    
    
    md_i2c_enable_pe(I2Cx);

    md_i2c_disable_own1_addr(I2Cx);
    md_i2c_set_own1_addr(I2Cx, I2C_InitStruct->Address1);

    if (I2C_InitStruct->AddrSize == MD_I2C_OA1_10BIT)
        md_i2c_enable_own1_10_bit_addr(I2Cx);
    else
        md_i2c_enable_own1_7_bit_addr(I2Cx);

    md_i2c_enable_own1_addr(I2Cx);

    if (I2C_InitStruct->DualAddressMode == MD_I2C_DUALADDRESS_ENABLE)
    {
        md_i2c_disable_own2_addr(I2Cx);
        md_i2c_set_own2_mask_addr(I2Cx, I2C_InitStruct->Address2Masks);
        md_i2c_set_own2_addr(I2Cx, I2C_InitStruct->Address2);
        md_i2c_enable_own2_addr(I2Cx);
    }

    status = SUCCESS;

    return status;
}

/**
  * @brief  Set each @ref md_i2c_inittypedef field to default value.
  * @param  UART_InitStruct pointer to a @ref md_i2c_inittypedef structure
  *         whose fields will be set to default values.
  * @retval None
  */
void md_i2c_struct_init(md_i2c_inittypedef *I2C_InitStruct)
{
    /* Set UART_InitStruct fields to default values */
    I2C_InitStruct->Timing          = CLK100kHz16M;
    I2C_InitStruct->AddrSize        = MD_I2C_ADDRESSINGMODE_7BIT;
    I2C_InitStruct->Address1        = 0x50 << 1;
    I2C_InitStruct->DualAddressMode = MD_I2C_DUALADDRESS_DISABLE;
    I2C_InitStruct->Address2        = 0x70 << 1;
    I2C_InitStruct->Address2Masks   = MD_I2C_ADDR2_NOMASK;
}

/**
  * @brief  Transmits in master mode an amount of data in blocking mode.
  * @param  I2Cx I2C Instance
  * @param  The number of bytes to be transmitted
  * @param  Enable/Disable 10-bit addressing mode
  * @param  Device(slave) address
  * @param  The pointer to a data buffer
  * @retval None
  */
void md_i2c_master_send(I2C_TypeDef *I2Cx, uint8_t Nbyte, uint32_t addr10, uint16_t DevAddr, uint8_t *txbuf)
{
    uint8_t index;
    int Timeout = 0;

    /* Config Device(slave) address */
    if (addr10 == MD_I2C_ADDRESSINGMODE_10BIT)
        md_i2c_enable_addr_10_bit(I2Cx);
    else
        md_i2c_enable_addr_7_bit(I2Cx);

    md_i2c_set_slave_addr(I2Cx, DevAddr);
    md_i2c_set_transmit_length(I2Cx, Nbyte);
    md_i2c_is_clear_flag_txfrst(I2Cx, MD_I2C_TXFIFO_RESET);
    md_i2c_enable_master_write(I2Cx);
    md_i2c_disable_reload(I2Cx);
    /* When NBYTES is matched, the communication will be automatically stop */
    md_i2c_enable_auto_end(I2Cx);

    if (Nbyte <= 8)
    {
        for (index = 0; index < Nbyte; index++)
            md_i2c_set_tx_reg_data(I2Cx, *txbuf++);

        Nbyte = 0;
    }
    else
    {
        for (index = 0; index < 8; index++)
            md_i2c_set_tx_reg_data(I2Cx, *txbuf++);

        Nbyte -= 8;
    }

    /* Start the I2C communication */
    md_i2c_set_start(I2Cx, MD_I2C_START_GENERATION);


    while (!md_i2c_is_active_flag_busy(I2Cx))
    {
        Timeout++;

        if (Timeout > 100)
        {
            printf("error\n\r");
            return;
        }
    }

    while (Nbyte > 0)
    {
        Timeout = 0;

        while (md_i2c_is_active_flag_txf(I2Cx))
        {
            Timeout++;

            if (Timeout > 100)
            {
                printf("error\n\r");
                return;
            }
        }

        md_i2c_set_tx_reg_data(I2Cx, *txbuf++);
        Nbyte--;
    }
}

/**
  * @brief  Receives in master mode an amount of data in blocking mode.
  * @param  I2Cx I2C Instance
  * @param  The number of bytes to be received
  * @param  Enable/Disable 10-bit addressing mode
  * @param  Device(slave) address
  * @param  The pointer to a data buffer
  * @retval None
  */
void md_i2c_master_rece(I2C_TypeDef *I2Cx, uint8_t Nbyte, uint32_t addr10, uint16_t DevAddr, uint8_t *rxbuf)
{
    int Timeout = 0;
    int Timeout1 = 0;

    /* Config Device(slave) address */
    if (addr10 == MD_I2C_ADDRESSINGMODE_10BIT)
        md_i2c_enable_addr_10_bit(I2Cx);
    else
        md_i2c_enable_addr_7_bit(I2Cx);

    md_i2c_set_slave_addr(I2Cx, DevAddr);
    md_i2c_set_transmit_length(I2Cx, Nbyte);
    md_i2c_is_clear_flag_rxfrst(I2Cx, MD_I2C_RXFIFO_RESET);
    md_i2c_enable_master_read(I2Cx);
    md_i2c_disable_reload(I2Cx);
    /* When NBYTES is matched, the communication will be automatically stop */
    md_i2c_enable_auto_end(I2Cx);
    /* Start the I2C communication */


    md_i2c_set_start(I2Cx, MD_I2C_START_GENERATION);

    Timeout = 0;

    while (!md_i2c_is_active_flag_busy(I2Cx))
    {
        Timeout++;

        if (Timeout > 100)
        {
            printf("error\n\r");
            return;
        }
    }

    Timeout = 0;

    while (md_i2c_is_active_it_nack(I2C1))
    {
        md_i2c_clear_it_nack(I2C1);
        md_tick_waitms(1, 1);

        if (md_i2c_is_active_it_txe(I2C1))
        {
            if (addr10 == MD_I2C_ADDRESSINGMODE_10BIT)
                md_i2c_enable_addr_10_bit(I2Cx);
            else
                md_i2c_enable_addr_7_bit(I2Cx);

            md_i2c_set_slave_addr(I2Cx, DevAddr);
            md_i2c_set_transmit_length(I2Cx, Nbyte);
            md_i2c_is_clear_flag_rxfrst(I2Cx, MD_I2C_RXFIFO_RESET);
            md_i2c_enable_master_read(I2Cx);
            md_i2c_disable_reload(I2Cx);
            /* When NBYTES is matched, the communication will be automatically stop */
            md_i2c_enable_auto_end(I2Cx);

        }

        Timeout++;

        if (Timeout > 100)
        {
            printf("error\n\r");
            return;
        }


        md_i2c_set_start(I2Cx, MD_I2C_START_GENERATION);
        md_tick_wait100us(1, 1);



        while (!md_i2c_is_active_flag_busy(I2Cx))
        {
            Timeout1++;

            if (Timeout1 > 100)
            {
                printf("error\n\r");
                return;
            }
        }

    }


    Timeout = 0;

    while (Nbyte > 0)
    {
        /* Wait Rx FIFO non-empty */
        Timeout = 0;

        if (md_i2c_is_active_it_nack(I2C1))
        {
            md_i2c_clear_it_nack(I2C1);
            md_tick_waitms(1, 1);
            md_i2c_set_start(I2Cx, MD_I2C_START_GENERATION);
            md_tick_wait100us(1, 1);

            Timeout++;

            if (Timeout > 100)
            {
                printf("error\r\n");
                return;
            }
        }
        else if (!md_i2c_is_active_flag_rxe(I2Cx))
        {
            *rxbuf++ = md_i2c_get_rx_reg_data(I2Cx);
            Nbyte--;
        }

    }
}

/**
  * @brief  Transmits in slave mode an amount of data in blocking mode.
  * @param  I2Cx I2C Instance
  * @param  The number of bytes to be transmitted, not for NBYTES
  * @param  The pointer to a data buffer
  * @retval None
  */
void md_i2c_slave_send(I2C_TypeDef *I2Cx, uint8_t Num, uint8_t *txbuf)
{
    int Timeout = 0;

    md_i2c_is_clear_flag_txfrst(I2Cx, MD_I2C_TXFIFO_RESET);


    while (!(md_i2c_is_active_flag_busy(I2Cx)))
    {
        Timeout++;

        if (Timeout > 1000000)
        {
            printf("error\r\n");
            return;
        }
    }

    while (Num > 0)
    {
        Timeout = 0;

        while (md_i2c_is_active_flag_txf(I2Cx))
        {
            Timeout++;

            if (Timeout > 1000000)
            {
                printf("error\r\n");
                return;
            }
        }

        md_i2c_set_tx_reg_data(I2Cx, *txbuf++);
        Num--;
    }
}

/**
  * @brief  Receives in slave mode an amount of data in blocking mode.
  * @param  I2Cx I2C Instance
  * @param  The number of bytes to be transmitted, not for NBYTES
  * @param  The pointer to a data buffer
  * @retval None
  */
void md_i2c_slave_rece(I2C_TypeDef *I2Cx, uint8_t Num, uint8_t *rxbuf)
{
    int Timeout = 0;
    md_i2c_is_clear_flag_rxfrst(I2Cx, MD_I2C_RXFIFO_RESET);



    while (!(md_i2c_is_active_flag_busy(I2Cx)))
    {
        Timeout++;

        if (Timeout > 1000000)
        {
            printf("error\r\n");
            return;
        }
    }

    while (Num > 0)
    {
        Timeout = 0;

        while (md_i2c_is_active_flag_rxe(I2Cx))
        {
            Timeout++;

            if (Timeout > 1000000)
            {
                printf("error\r\n");
                return;
            }
        }

        *rxbuf++ = md_i2c_get_rx_reg_data(I2Cx);
        Num--;
    }
}


/**
  * @brief  i2c timeout plugin
  * @param  I2C status related functions
  * @param  I2Cx I2C Instance
  * @param  The number of timeout
  * @param  The reverse I2C status function returns the value
  * @retval 0:is no timeout 1:us timeout
  */

bool md_i2c_timeout(uint32_t (*func)(I2C_TypeDef *), I2C_TypeDef *I2Cx, uint32_t timeout, bool reverse)
{
    int ii;

    for (ii = 0; (reverse ? !func(I2Cx) : func(I2Cx)) && ii <= timeout ; ii++)
    {
        if (ii == timeout)
        {
            printf("timeout error\r\n");
            return 1;
        }
    }

    return 0;
}

/**
  * @} Micro_Driver
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
