/**********************************************************************************
 *
 * @file    md_fc.c
 * @brief   md_fc C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "md_syscfg.h"
#include "md_fc.h"

/** @addtogroup Micro_Driver
  * @{
  */

#if defined (FC)

/** @defgroup FC FC
  * @brief FC micro driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/

/* Private constants ----------------------------------------------------------*/
/** @defgroup MD_FC_Private_Constants FC Private Constants
  * @{
  */

#define  REMAPDATA_BASE    ((     uint32_t)0x00000000) /*  FLASH Main  (32K Bytes)  */
#define  REMAPINFO_BASE    ((     uint32_t)0x00000000) /*  FLASH Info  ( 6K Bytes) - Boot Loader Memory  */

#define  EFOPT_PAGE3_OFFSET     (MD_FC_PC_EF_IPAGESZ*3)
#define  EFOPT_REMAP_OFFSET     (EFOPT_PAGE3_OFFSET+0x0)

/**
  * @} MD_FC_Private_Constants
  */

/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/
/** @defgroup MD_FC_Public_Functions FC Public Functions
  * @{
  */

/**
  * @brief   FC Read Adc Temperature voltage,unit is V.
  * @param   None
  * @retval  adc temperature refrence voltage,unit is V.
  */
double  md_fc_read_adc_temperature_voltage()
{
    uint32_t   *EFAddr;

    EFAddr = (uint32_t *)(EBDINFO_BASE + 0x0858);


    return (((double) * EFAddr) / (double)(1U << 23)) * 3.0;
}


/**
  * @brief   FC Read Adc Temperature.
  * @param   None
  * @retval  Always return SUCCESS
  *          - Data in specified page address read to data buffer
  */
ErrorStatus  md_fc_read_adc_temperature(uint32_t *ADCTEMP)
{
    ErrorStatus status = SUCCESS;
    uint32_t   *EFAddr;

    EFAddr = (uint32_t *)(EBDINFO_BASE + 0x0858);
    *ADCTEMP       = *EFAddr;

    return status;
}

/**
  * @brief   FC Read Info. page
  * @param   None
  * @retval  Always return SUCCESS
  *          - Data in specified page address read to data buffer
  */
ErrorStatus  md_fc_read_info(uint32_t info_addr, uint32_t *info)
{
    ErrorStatus status = SUCCESS;
    uint32_t   *EFAddr;

    EFAddr = (uint32_t *)(EBDINFO_BASE + info_addr);
    *info       = *EFAddr;

    return status;
}

/**
  * @brief  FC unlock.
  * @param  None
  * @retval  An ErrorStatus enumeration value:
  *          - SUCCESS: FC unlock done
  *          - ERROR: FC unlock fail
  */
ErrorStatus md_fc_unlock(void)
{
    ErrorStatus status = ERROR;

    /* Unlock EF*/
    if (!md_fc_is_active_flag_flash_unlock())
    {
//      md_fc_set_ul(MD_FC_PC_EF_UL_LOCK);
        md_fc_set_flash_unlock(MD_FC_PC_EF_UL_KEY1);
        md_fc_set_flash_unlock(MD_FC_PC_EF_UL_KEY2);

        while (!md_fc_is_active_flag_flash_unlock());
    }

    status = SUCCESS;
    return status;
}

/**
  * @brief  FC lock.
  * @param  None
  * @retval  An ErrorStatus enumeration value:
  *          - SUCCESS: FC lock done
  *          - ERROR: FC lock fail
  */
ErrorStatus md_fc_lock(void)
{
    ErrorStatus status = ERROR;

    /* Unlock EF*/
    if (md_fc_is_active_flag_flash_unlock())
    {
        md_fc_set_flash_unlock(MD_FC_PC_EF_UL_LOCK);

        while (md_fc_is_active_flag_flash_unlock());
    }

    status = SUCCESS;
    return status;
}

/**
  * @brief  FC main erase.
  * @note  To prevent unexpected code branch, the main erase function must specify a dedicated main erase ID
  * @param  MEraseID Key for main erase, must be 0xA5A5AA55
  * @retval  An ErrorStatus enumeration value:
  *          - SUCCESS: Main flash erased
  *          - ERROR: Main erase ID mismatch
  */
ErrorStatus md_fc_merase(uint32_t MEraseID)
{
    ErrorStatus status = ERROR;

    /* Check the ID */
    if (MEraseID != MD_FC_PC_EF_MERASEID)
        return status;

    /* Unlock EF*/
    md_fc_unlock();
    /* Write MERASE to CMD*/
    md_fc_set_flash_command(MD_FC_PC_CMD_MERASE);
    /* Lock EF*/
    md_fc_lock();

    status = SUCCESS;
    return status;
}

/**
  * @brief  FC Page Erase
  * @note  To prevent unexpected code branch, the page erase function must specify the complement start address
  * @param  pErasePara Pointer to structure md_fc_ControlTypeDef
  * @retval  An ErrorStatus enumeration value.
  *          - SUCCESS: Specified page address erased
  *          - ERROR: Complement of start address mismatch
  */
ErrorStatus md_fc_page_erase(md_fc_ControlTypeDef *pErasePara)
{
    ErrorStatus status = ERROR;

    /* Check the start address and complement of start address*/
    if (pErasePara->u32SAddr & pErasePara->u32SAddrC)
        return status;

    /* Unlock EF*/
    md_fc_unlock();
    /* Write page address to PA*/
    md_fc_disable_info_region();    
    md_fc_set_program_address(pErasePara->u32SAddr);
    /* Write ERASE to CMD*/
    md_fc_set_flash_command(MD_FC_PC_CMD_ERASE);
    /* Lock EF*/
    md_fc_lock();

    status = SUCCESS;
    return status;
}

/**
  * @brief  FC Page Program
  * @note  To prevent unexpected code branch, the page program function must specify the complement start address
  * @param  pProgramPara Pointer to structure md_fc_ControlTypeDef
  * @retval An ErrorStatus enumeration value.
  *          - SUCCESS: Data in data buffer programed to specified page address
  *          - ERROR: Complement of start address mismatch
  */
ErrorStatus md_fc_program(md_fc_ControlTypeDef *pProgramPara)
{
    ErrorStatus status = ERROR;
    uint16_t  u16BCnt;

    md_fc_set_program_counter(pProgramPara->u16BCnt);
    md_fc_set_program_address(pProgramPara->u32SAddr);
    u16BCnt = pProgramPara->u16BCnt;
    u16BCnt = (u16BCnt + 3) >> 2;

    /* Unlock EF*/
    md_fc_unlock();

    while (u16BCnt--)
    {
        /* Check the start address and complement of start address*/
        if (pProgramPara->u32SAddr & pProgramPara->u32SAddrC)
        {
            /* Lock EF*/
            md_fc_lock();
            return status;
        }

        /* Write lower 32bit program data to PLD*/
        md_fc_set_program_data(*pProgramPara->pu32Buf++);

        /* Write PROGRAM to CMD*/
        md_fc_set_flash_command(MD_FC_PC_CMD_PROGRAM);
    }

    /* Lock EF*/
    md_fc_lock();

    status = SUCCESS;
    return status;
}

/**
  * @brief  FC Page Read
  * @param  pReadPara Pointer to structure md_fc_ControlTypeDef
  * @retval  Always return SUCCESS
  *         - Data in specified page address read to data buffer
  */
ErrorStatus  md_fc_read(md_fc_ControlTypeDef *pReadPara)
{
    ErrorStatus status = SUCCESS;
    uint32_t   *pu32EFAddr;
    uint16_t  u16BCnt;

    if (md_syscfg_get_memory_mapping(SYSCFG) == MD_SYSCFG_MEMMOD_MAIN)  /*  Reamp from Main  */
        pu32EFAddr = (uint32_t *)(REMAPDATA_BASE + pReadPara->u32SAddr);
    else
        pu32EFAddr = (uint32_t *)(EBDDATA_BASE + pReadPara->u32SAddr);

    u16BCnt = pReadPara->u16BCnt;
    u16BCnt = (u16BCnt + 3) >> 2;

    while (u16BCnt--)
        *pReadPara->pu32Buf++ = *pu32EFAddr++;

    return status;
}

/**
  * @brief  FC Update UCR Protect
  * @param  pUpdPL1Para Pointer to structure md_fc_UpdProtTypeDdef
  * @retval  Always return SUCCESS
  */
ErrorStatus  md_fc_update_ucrp(md_fc_UpdProtTypeDef *pUpdPL1Para)
{
    ErrorStatus status = SUCCESS;

    /* Write hogher 32bit data to UPL*/
    md_fc_set_protect0(pUpdPL1Para->UpdateL);
    /* Write higher 32bit data to UPH*/
    md_fc_set_protect1(pUpdPL1Para->UpdateH);

    /* Write UPDPL1 to CMD*/
    md_fc_set_flash_command(MD_FC_PC_CMD_SETUCRP);

    return status;
}

/**
  * @brief  FC Update Read Protect
  * @param  pUpdPL1Para Pointer to structure md_fc_UpdProtTypeDdef
  * @retval  Always return SUCCESS
  */
ErrorStatus  md_fc_update_rp(md_fc_UpdProtTypeDef *pUpdPL1Para)
{
    ErrorStatus status = SUCCESS;

    /* Write hogher 32bit data to UPL*/
    md_fc_set_protect0(pUpdPL1Para->UpdateL);

    /* Write UPDPL1 to CMD*/
    md_fc_set_flash_command(MD_FC_PC_CMD_SETRP);

    return status;
}

/**
  * @brief  FC Update Write Protect
  * @param  pUpdPL1Para Pointer to structure md_fc_UpdProtTypeDdef
  * @retval  Always return SUCCESS
  */
ErrorStatus  md_fc_update_wp(md_fc_UpdProtTypeDef *pUpdPL1Para)
{
    ErrorStatus status = SUCCESS;

    /* Write hogher 32bit data to UPL*/
    md_fc_set_protect0(pUpdPL1Para->UpdateL);
    /* Write higher 32bit data to UPH*/
    md_fc_set_protect1(pUpdPL1Para->UpdateH);

    /* Write UPDPL1 to CMD*/
    md_fc_set_flash_command(MD_FC_PC_CMD_SETWP);

    return status;
}

/**
  * @brief  FC Update Remap Option
  * @param  UpdRemap New remap setting value (remap address=valus(0x0~0xf)*0x1000, 0x10=BootLoader)
  * @retval  Always return SUCCESS
  */
ErrorStatus  md_fc_update_remap(uint32_t UpdRemap)
{
    ErrorStatus status = SUCCESS;

    /* Write sector address to PA*/
    md_fc_enable_info_region();
    md_fc_set_program_address(EFOPT_PAGE3_OFFSET);
    /* Unlock EF*/
    md_fc_set_flash_unlock(MD_FC_PC_EF_UL_LOCK);
    md_fc_set_flash_unlock(MD_FC_PC_EF_UL_KEY1);
    md_fc_set_flash_unlock(MD_FC_PC_EF_UL_KEY2);
    /* Write SERASE to CMD*/
    md_fc_set_flash_command(MD_FC_PC_CMD_ERASE);
    /* Write remap offset address to PA*/
    md_fc_enable_info_region();
    md_fc_set_program_address(EFOPT_REMAP_OFFSET);
    /* Write lower 32bit remap data to PLD*/
    md_fc_set_program_data(UpdRemap);
    /* Write PROGRAM to CMD*/
    md_fc_set_flash_command(MD_FC_PC_CMD_PROGRAM);
    /* Lock EF*/
    md_fc_set_flash_unlock(MD_FC_PC_EF_UL_LOCK);
    md_fc_disable_info_region();

    return status;
}

/**
  * @} MD_FC_Public_Functions
  */

/**
  * @} FC
  */
#endif

/**
  * @} Micro_Driver
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

