/**********************************************************************************
 *
 * @file    md_timer.h
 * @brief   header file of md_timer.c
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_TIMER_H__
#define __MD_TIMER_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------*/
#include "es32f0930.h"

/** @addtogroup Micro_Driver
  * @{
  */
#if (defined(GP32C4T1)) || (defined(GP16C2T1)) || (defined(GP16C2T2)) || (defined(BS16T1))

/** @defgroup TIMER TIMER
  * @brief TIMER module driver
  * @{
  */


/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private Macros ----------------------------------------------------------*/



/* Public Constants -----------------------------------------------------------*/

/** @defgroup MD_TIMER_Public_Constants Public Constants
  * @brief TIMER module driver
  * @{
  */
/* TIMER_CON1 macros define*/
#define MD_TIMER_DEBUGSELECT_INPUT        (0 << TIMER_CON1_DBGSEL_POS)      /*!<  Channel input*/
#define MD_TIMER_DEBUGSELECT_OUTPUT       (1 << TIMER_CON1_DBGSEL_POS)      /*!<  Channel output*/

#define MD_TIMER_CLOCKDIVSION_DIV1        (0 << TIMER_CON1_DFCKSEL_POSS)    /*!<  tDTS=tCK_INT*/
#define MD_TIMER_CLOCKDIVSION_DIV2        (1 << TIMER_CON1_DFCKSEL_POSS)    /*!<  tDTS=2*tCK_INT*/
#define MD_TIMER_CLOCKDIVSION_DIV4        (2 << TIMER_CON1_DFCKSEL_POSS)    /*!<  tDTS=4*tCK_INT*/

#define MD_TIMER_CENTERALIGNED_DISABLE    (0 << TIMER_CON1_CMSEL_POSS)      /*!<  Edge-aligned mode*/
#define MD_TIMER_CENTERALIGNED_DOWN       (1 << TIMER_CON1_CMSEL_POSS)      /*!<  Center-aligned mode 1*/
#define MD_TIMER_CENTERALIGNED_UP         (2 << TIMER_CON1_CMSEL_POSS)      /*!<  Center-aligned mode 2*/
#define MD_TIMER_CENTERALIGNED_BOTH       (3 << TIMER_CON1_CMSEL_POSS)      /*!<  Center-aligned mode 3*/

#define MD_TIMER_UPCOUNTER                (0 << TIMER_CON1_DIRSEL_POS)      /*!<  Counter used as upcounter*/
#define MD_TIMER_DOWNCOUNTER              (1 << TIMER_CON1_DIRSEL_POS)      /*!<  Counter used as downcounter*/

#define MD_TIMER_ONEPULSEMODE_DISABLE     (0 << TIMER_CON1_SPMEN_POS)       /*!<  Counter is not stopped at update event*/
#define MD_TIMER_ONEPULSEMODE_ENABLE      (1 << TIMER_CON1_SPMEN_POS)       /*!<  Counter stops counting at the next update event*/

#define MD_TIMER_UPDATESOURCE_NORMAL      (0 << TIMER_CON1_UERSEL_POS)      /*!<  Counter overflow/underflow, Setting the UG bit, Update generation through the slave mode controller generate an update interrupt*/
#define MD_TIMER_UPDATESOURCE_COUNTER     (1 << TIMER_CON1_UERSEL_POS)      /*!<  Only counter overflow/underflow generates an update interrupt or DMA request if Enabled*/

/* TIMER_CON2 macros define*/
#define MD_TIMER_IDLEOUTPUT_LOW           0                                   /*!<  OC1=0 (after a dead-time if OC1N is implemented) when MOE=0*/
#define MD_TIMER_IDLEOUTPUT_HIGH          1                                   /*!<  OC1=1 (after a dead-time if OC1N is implemented) when MOE=0*/

#define MD_TIMER_TI1INPUT_CH1             (0 << TIMER_CON2_I1SEL_POS)       /*!<  The TIMx_CH1 pin is connected to TI1 input*/
#define MD_TIMER_TI1INPUT_XOR             (1 << TIMER_CON2_I1SEL_POS)       /*!<  The TIMx_CH1, CH2 and CH3 pins are connected to the TI1 input (XOR combination)*/

#define MD_TIMER_MASTERMODE_RESET         (0 << TIMER_CON2_MMSEL_POSS)      /*!<  Reset mode*/
#define MD_TIMER_MASTERMODE_ENABLE        (1 << TIMER_CON2_MMSEL_POSS)      /*!<  Enable mode*/
#define MD_TIMER_MASTERMODE_UPDATE        (2 << TIMER_CON2_MMSEL_POSS)      /*!<  Update mode*/
#define MD_TIMER_MASTERMODE_COMPAREPULSE  (3 << TIMER_CON2_MMSEL_POSS)      /*!<  Compare Pulse mode*/
#define MD_TIMER_MASTERMODE_COMPARE1      (4 << TIMER_CON2_MMSEL_POSS)      /*!<  Compare 1 mode*/
#define MD_TIMER_MASTERMODE_COMPARE2      (5 << TIMER_CON2_MMSEL_POSS)      /*!<  Compare 2 mode*/
#define MD_TIMER_MASTERMODE_COMPARE3      (6 << TIMER_CON2_MMSEL_POSS)      /*!<  Compare 3 mode*/
#define MD_TIMER_MASTERMODE_COMPARE4      (7 << TIMER_CON2_MMSEL_POSS)      /*!<  Compare 4 mode*/
#define MD_TIMER_DMASELECTION_COMPARE     (0 << TIMER_CON2_CCDMASEL_POS)    /*!<  CCx DMA request sent when CCx event occurs*/
#define MD_TIMER_DMASELECTION_UPDATE      (1 << TIMER_CON2_CCDMASEL_POS)    /*!<  CCx DMA requests sent when update event occurs*/

/* TIMER_SMCON macros define*/
#define MD_TIMER_ETRPOLARITY_NONINVERTED  (0 << TIMER_SMCON_ETPOL_POS)      /*!<  ETR is non-inverted*/
#define MD_TIMER_ETRPOLARITY_INVERTED     (1 << TIMER_SMCON_ETPOL_POS)      /*!<  ETR is inverted*/

#define MD_TIMER_UPDATESELECTION_COMG     (0 << TIMER_CON2_CCUSEL_POS)      /*!<  When capture/compare control bits are preloaded (CCPC=1), they are updated by setting the COMG bit only*/
#define MD_TIMER_UPDATESELECTION_BOTH     (1 << TIMER_CON2_CCUSEL_POS)      /*!<  When capture/compare control bits are preloaded (CCPC=1), they are updated by setting the COMG bit or when an rising edge occurs on TRGI*/


#define MD_TIMER_CCPRELOAD_DISABLE        (0 << TIMER_CON2_CCPCEN_POS)      /*!<  CCxE, CCxNE and OCxM bits are not preloaded*/
#define MD_TIMER_CCPRELOAD_ENABLE         (1 << TIMER_CON2_CCPCEN_POS)      /*!<  CCxE, CCxNE and OCxM bits are preloaded*/

#define MD_TIMER_ETRFILTER_FDIV1          (0 << TIMER_SMCON_ETFLT_POSS)     /*!<  No filter*/
#define MD_TIMER_ETRFILTER_FDIV1N2        (1 << TIMER_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fCK_INT, N = 2*/
#define MD_TIMER_ETRFILTER_FDIV1N4        (2 << TIMER_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fCK_INT, N = 4*/
#define MD_TIMER_ETRFILTER_FDIV1N8        (3 << TIMER_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fCK_INT, N = 8*/
#define MD_TIMER_ETRFILTER_FDIV2N6        (4 << TIMER_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 2, N = 6*/
#define MD_TIMER_ETRFILTER_FDIV2N8        (5 << TIMER_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 2, N = 8*/
#define MD_TIMER_ETRFILTER_FDIV4N6        (6 << TIMER_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 4, N = 6*/
#define MD_TIMER_ETRFILTER_FDIV4N8        (7 << TIMER_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 4, N = 8*/
#define MD_TIMER_ETRFILTER_FDIV8N6        (8 << TIMER_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 8, N = 6*/
#define MD_TIMER_ETRFILTER_FDIV8N8        (9 << TIMER_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 8, N = 8*/
#define MD_TIMER_ETRFILTER_FDIV16N5       (10 << TIMER_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 16, N = 5*/
#define MD_TIMER_ETRFILTER_FDIV16N6       (11 << TIMER_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 16, N = 6*/
#define MD_TIMER_ETRFILTER_FDIV16N8       (12 << TIMER_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 16, N = 8*/
#define MD_TIMER_ETRFILTER_FDIV32N5       (13 << TIMER_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 32, N = 5*/
#define MD_TIMER_ETRFILTER_FDIV32N6       (14 << TIMER_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 32, N = 6*/
#define MD_TIMER_ETRFILTER_FDIV32N8       (15 << TIMER_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 32, N = 8*/

#define MD_TIMER_MASTERSLAVE_NODELAY      (0 << TIMER_SMCON_MSCFG_POS)      /*!<  No action*/
#define MD_TIMER_MASTERSLAVE_DELAY        (1 << TIMER_SMCON_MSCFG_POS)      /*!<  The effect of an event on the trigger input (TRGI) is delayed to allow a perfect*/

#define MD_TIMER_TRIGGERSELECT_ITR0       (0 << TIMER_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 0*/
#define MD_TIMER_TRIGGERSELECT_ITR1       (1 << TIMER_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 1*/
#define MD_TIMER_TRIGGERSELECT_ITR2       (2 << TIMER_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 2*/
#define MD_TIMER_TRIGGERSELECT_ITR3       (3 << TIMER_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 3*/
#define MD_TIMER_TRIGGERSELECT_TI1FED     (4 << TIMER_SMCON_TSSEL_POSS)     /*!<  TI1 Edge Detector*/
#define MD_TIMER_TRIGGERSELECT_TI1FP1     (5 << TIMER_SMCON_TSSEL_POSS)     /*!<  Filtered Timer Input 1*/
#define MD_TIMER_TRIGGERSELECT_TI2FP2     (6 << TIMER_SMCON_TSSEL_POSS)     /*!<  Filtered Timer Input 2*/
#define MD_TIMER_TRIGGERSELECT_ETRF       (7 << TIMER_SMCON_TSSEL_POSS)     /*!<  External Trigger input*/

#define MD_TIMER_OCREFCLEAR_CMP           (0 << TIMER_SMCON_CHCSEL_POS)     /*!<  OCREF_CLR is connected to the CMP_IN input*/
#define MD_TIMER_OCREFCLEAR_ETRF          (1 << TIMER_SMCON_CHCSEL_POS)     /*!<  OCREF_CLR is connected to ETRF*/

#define MD_TIMER_SLAVEMODE_DISABLE        (0 << TIMER_SMCON_SMODS_POSS)     /*!<  Slave mode disabled*/
#define MD_TIMER_SLAVEMODE_ENCODER1       (1 << TIMER_SMCON_SMODS_POSS)     /*!<  Encoder mode 1*/
#define MD_TIMER_SLAVEMODE_ENCODER2       (2 << TIMER_SMCON_SMODS_POSS)     /*!<  Encoder mode 2*/
#define MD_TIMER_SLAVEMODE_ENCODER3       (3 << TIMER_SMCON_SMODS_POSS)     /*!<  Encoder mode 3*/
#define MD_TIMER_SLAVEMODE_RESET          (4 << TIMER_SMCON_SMODS_POSS)     /*!<  Reset Mode*/
#define MD_TIMER_SLAVEMODE_GATED          (5 << TIMER_SMCON_SMODS_POSS)     /*!<  Gated Mode*/
#define MD_TIMER_SLAVEMODE_TRIGGER        (6 << TIMER_SMCON_SMODS_POSS)     /*!<  Trigger Mode*/
#define MD_TIMER_SLAVEMODE_EXTERNALCLOCK  (7 << TIMER_SMCON_SMODS_POSS)     /*!<  External Clock Mode 1*/

/* TIMER_IER IDR IVS RIF IFM ICR macros define*/
#define MD_TIMER_INTERRUPTFALG_CC4OI      (1 << TIMER_IER_CH4OVI_POS)       /*!<  Capture/Compare 4 overcapture interrupt*/
#define MD_TIMER_INTERRUPTFALG_CC3OI      (1 << TIMER_IER_CH3OVI_POS)       /*!<  Capture/Compare 3 overcapture interrupt*/
#define MD_TIMER_INTERRUPTFALG_CC2OI      (1 << TIMER_IER_CH2OVI_POS)       /*!<  Capture/Compare 2 overcapture interrupt*/
#define MD_TIMER_INTERRUPTFALG_CC1OI      (1 << TIMER_IER_CH1OVI_POS)       /*!<  Capture/Compare 1 overcapture interrupt*/
#define MD_TIMER_INTERRUPTFALG_BI         (1 << TIMER_IER_BRKI_POS)         /*!<  Break interrupt*/
#define MD_TIMER_INTERRUPTFALG_TI         (1 << TIMER_IER_TRGI_POS)         /*!<  Trigger interrupt*/
#define MD_TIMER_INTERRUPTFALG_COMI       (1 << TIMER_IER_COMI_POS)         /*!<  COM interrupt*/
#define MD_TIMER_INTERRUPTFALG_CC4I       (1 << TIMER_IER_CH4I_POS)         /*!<  Capture/Compare 4 interrupt*/
#define MD_TIMER_INTERRUPTFALG_CC3I       (1 << TIMER_IER_CH3I_POS)         /*!<  Capture/Compare 3 interrupt*/
#define MD_TIMER_INTERRUPTFALG_CC2I       (1 << TIMER_IER_CH2I_POS)         /*!<  Capture/Compare 2 interrupt*/
#define MD_TIMER_INTERRUPTFALG_CC1I       (1 << TIMER_IER_CH1I_POS)         /*!<  Capture/Compare 1 interrupt*/
#define MD_TIMER_INTERRUPTFALG_UI         (1 << TIMER_IER_UI_POS)           /*!<  Update interrupt*/

/* TIMER_SGE macros define*/
#define MD_TIMER_EVENTGENERATION_BG       (1 << TIMER_SGE_SGBRK_POS)        /*!<  Break generation*/
#define MD_TIMER_EVENTGENERATION_TG       (1 << TIMER_SGE_SGTRG_POS)        /*!<  Trigger generation*/
#define MD_TIMER_EVENTGENERATION_COMG     (1 << TIMER_SGE_SGCOM_POS)        /*!<  Capture/Compare control update generation*/
#define MD_TIMER_EVENTGENERATION_CC4G     (1 << TIMER_SGE_SGCH4_POS)        /*!<  Capture/Compare 4 generation*/
#define MD_TIMER_EVENTGENERATION_CC3G     (1 << TIMER_SGE_SGCH3_POS)        /*!<  Capture/Compare 3 generation*/
#define MD_TIMER_EVENTGENERATION_CC2G     (1 << TIMER_SGE_SGCH2_POS)        /*!<  Capture/Compare 2 generation*/
#define MD_TIMER_EVENTGENERATION_CC1G     (1 << TIMER_SGE_SGCH1_POS)        /*!<  Capture/Compare 1 generation*/
#define MD_TIMER_EVENTGENERATION_UG       (1 << TIMER_SGE_SGU_POS)          /*!<  Update generation*/

/* TIMER_CHMR1 CHMR2 output macros define*/
#define MD_TIMER_OUTPUTMODE_DISABLE       0                                   /*!<  Frozen*/
#define MD_TIMER_OUTPUTMODE_HIGHONMSTCH   1                                   /*!<  Set channel 1 to active level on match*/
#define MD_TIMER_OUTPUTMODE_LOWONMSTCH    2                                   /*!<  Set channel 1 to inactive level on match*/
#define MD_TIMER_OUTPUTMODE_TOGGLE        3                                   /*!<  Toggle*/
#define MD_TIMER_OUTPUTMODE_FORCELOW      4                                   /*!<  Force inactive level*/
#define MD_TIMER_OUTPUTMODE_FORCEHIGH     5                                   /*!<  Force active level*/
#define MD_TIMER_OUTPUTMODE_PWMMODE1      6                                   /*!<  PWM mode 1*/
#define MD_TIMER_OUTPUTMODE_PWMMODE2      7                                   /*!<  PWM mode 2*/

#define MD_TIMER_CHMODE_OUTPUT            0                                   /*!<  CCx channel is configured as output*/
#define MD_TIMER_CHMODE_INPUT_DIRECT      1                                   /*!<  CCx channel is configured as input, ICx is mapped direct*/
#define MD_TIMER_CHMODE_INPUT_INDIRECT    2                                   /*!<  CCx channel is configured as input, ICx is mapped indirect*/
#define MD_TIMER_CHMODE_INPUT_TRC         3                                   /*!<  CCx channel is configured as input, ICx is mapped TRC*/

/* TIMER_CHMR1 CHMR2 input macros define*/
#define MD_TIMER_INPUTFILTER_FDIV1        0                                   /*!<  No filter*/
#define MD_TIMER_INPUTFILTER_FDIV1N2      1                                   /*!<  fSAMPLING = fCK_INT, N = 2*/
#define MD_TIMER_INPUTFILTER_FDIV1N4      2                                   /*!<  fSAMPLING = fCK_INT, N = 4*/
#define MD_TIMER_INPUTFILTER_FDIV1N8      3                                   /*!<  fSAMPLING = fCK_INT, N = 8*/
#define MD_TIMER_INPUTFILTER_FDIV2N6      4                                   /*!<  fSAMPLING = fDTS / 2, N = 6*/
#define MD_TIMER_INPUTFILTER_FDIV2N8      5                                   /*!<  fSAMPLING = fDTS / 2, N = 8*/
#define MD_TIMER_INPUTFILTER_FDIV4N6      6                                   /*!<  fSAMPLING = fDTS / 4, N = 6*/
#define MD_TIMER_INPUTFILTER_FDIV4N8      7                                   /*!<  fSAMPLING = fDTS / 4, N = 8*/
#define MD_TIMER_INPUTFILTER_FDIV8N6      8                                   /*!<  fSAMPLING = fDTS / 8, N = 6*/
#define MD_TIMER_INPUTFILTER_FDIV8N8      9                                   /*!<  fSAMPLING = fDTS / 8, N = 8*/
#define MD_TIMER_INPUTFILTER_FDIV16N5     10                                  /*!<  fSAMPLING = fDTS / 16, N = 5*/
#define MD_TIMER_INPUTFILTER_FDIV16N6     11                                  /*!<  fSAMPLING = fDTS / 16, N = 6*/
#define MD_TIMER_INPUTFILTER_FDIV16N8     12                                  /*!<  fSAMPLING = fDTS / 16, N = 8*/
#define MD_TIMER_INPUTFILTER_FDIV32N5     13                                  /*!<  fSAMPLING = fDTS / 32, N = 5*/
#define MD_TIMER_INPUTFILTER_FDIV32N6     14                                  /*!<  fSAMPLING = fDTS / 32, N = 6*/
#define MD_TIMER_INPUTFILTER_FDIV32N8     15                                  /*!<  fSAMPLING = fDTS / 32, N = 8*/

#define MD_TIMER_INPUTPRESCALE_DIV1       0                                   /*!<  no prescaler*/
#define MD_TIMER_INPUTPRESCALE_DIV2       1                                   /*!<  capture is done once every 2 events*/
#define MD_TIMER_INPUTPRESCALE_DIV4       2                                   /*!<  capture is done once every 4 events*/
#define MD_TIMER_INPUTPRESCALE_DIV8       3                                   /*!<  capture is done once every 8 events*/

/* TIMER_CCEP input macros define*/
#define MD_TIMER_OUTPUTPOLARITY_HIGH      0                                   /*!<  active high*/
#define MD_TIMER_OUTPUTPOLARITY_LOW       1                                   /*!<  active low*/

/* TIMER_BDCFG input macros define*/
#define MD_TIMER_BREAKPOLARITY_LOW        (0 << TIMER_BDCFG_BRKP_POS)       /*!<  Break input BRK is active low*/
#define MD_TIMER_BREAKPOLARITY_HIGH       (1 << TIMER_BDCFG_BRKP_POS)       /*!<  Break input BRK is active high */

#define MD_TIMER_OFFSTATERUN_DISABLE      (0 << TIMER_BDCFG_OFFSSR_POS)     /*!<  This bit is used when MOE=1, when inactive, OC/OCN outputs are disabled*/
#define MD_TIMER_OFFSTATERUN_ENABLE       (1 << TIMER_BDCFG_OFFSSR_POS)     /*!<  This bit is used when MOE=1, when inactive, OC/OCN outputs are enabled*/

#define MD_TIMER_OFFSTATEIDLE_DISABLE     (0 << TIMER_BDCFG_OFFSSI_POS)     /*!<  This bit is used when MOE=0, when inactive, OC/OCN outputs are disabled*/
#define MD_TIMER_OFFSTATEIDLE_ENABLE      (1 << TIMER_BDCFG_OFFSSI_POS)     /*!<  This bit is used when MOE=0, when inactive, OC/OCN outputs are forced*/

#define MD_TIMER_LOCKLEVEL_0              (0 << TIMER_BDCFG_LOCKLVL_POSS)   /*!<  LOCK OFF*/
#define MD_TIMER_LOCKLEVEL_1              (1 << TIMER_BDCFG_LOCKLVL_POSS)   /*!<  LOCK Level 1*/
#define MD_TIMER_LOCKLEVEL_2              (2 << TIMER_BDCFG_LOCKLVL_POSS)   /*!<  LOCK Level 2*/
#define MD_TIMER_LOCKLEVEL_3              (3 << TIMER_BDCFG_LOCKLVL_POSS)   /*!<  LOCK Level 3*/

/**
  * @} MD_TIMER_Public_Constants
  */

/* Public Macro ------------------------------------------------------------*/
/** @defgroup MD_TIMER_Public_Macro Public Macro
  * @brief TIMER module driver
  * @{
  */

/**
  * @brief  Timer CON1 setup.
  * @param  timx TIMER instance
  * @param  value (DBGSEL | CMPSEL | CMPSELP | DFCKSEL | ARPEN | CMSEL | DIRSEL | SPMEN | UERSEL | DISUE | CNTEN)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_con1(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->CON1, value);
}

/**
  * @brief  GET Timer CON1 register value.
  * @param  timx TIMER instance
  * @retval Timer CON1 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_con1(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->CON1));
}

/**
  * @brief  Timer debug mode keep output enable.
  * @param  timx TIMER instance
  * @param  DebugSelect Debug mode output select
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_debug_keep_output(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CON1, TIMER_CON1_DBGSEL_MSK);
}

/**
  * @brief  Timer debug mode keep output disable.
  * @param  timx TIMER instance
  * @param  DebugSelect Debug mode output select
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_debug_keep_output(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CON1, TIMER_CON1_DBGSEL_MSK);
}

/**
  * @brief  Get timer debug mode select.
  * @param  timx TIMER instance
  * @retval Timer debug mode select.
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_debug_keep_output(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON1, TIMER_CON1_DBGSEL_MSK) >> TIMER_CON1_DBGSEL_POS);
}

/**
  * @brief  Timer clock division setup.
  * @param  timx TIMER instance
  * @param  ClockDivision Clock division
            @arg @ref MD_TIMER_CLOCKDIVSION_DIV1
            @arg @ref MD_TIMER_CLOCKDIVSION_DIV2
            @arg @ref MD_TIMER_CLOCKDIVSION_DIV4
  * @retval None
  */

__STATIC_INLINE void md_timer_set_df_clock_dfcksel(TIMER_TypeDef *timx, uint32_t ClockDivision)
{
    MODIFY_REG(timx->CON1, TIMER_CON1_DFCKSEL_MSK, ClockDivision);
}

/**
  * @brief  Get timer Clock division.
  * @param  timx TIMER instance
  * @retval Timer Clock division.
  */

__STATIC_INLINE uint32_t md_timer_get_df_clock_dfcksel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON1, TIMER_CON1_DFCKSEL_MSK) >> TIMER_CON1_DFCKSEL_POSS);
}

/**
  * @brief  Timer auto-reload preload enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_autoreload_preload_arpen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CON1, TIMER_CON1_ARPEN_MSK);
}

/**
  * @brief  Timer auto-reload preload disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_autoreload_preload_arpen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CON1, TIMER_CON1_ARPEN_MSK);
}

/**
  * @brief  Indicates whether the timer auto-reload preload is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_autoreload_preload_arpen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON1, TIMER_CON1_ARPEN_MSK) == (TIMER_CON1_ARPEN_MSK));
}

/**
  * @brief  Timer center-aligned mode selection setup.
  * @param  timx TIMER instance
  * @param  CenterAlignedMode center-aligned mode selection
            @arg @ref MD_TIMER_CENTERALIGNED_DISABLE
            @arg @ref MD_TIMER_CENTERALIGNED_DOWN
            @arg @ref MD_TIMER_CENTERALIGNED_UP
            @arg @ref MD_TIMER_CENTERALIGNED_BOTH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_counter_aligned_mode_cmsel(TIMER_TypeDef *timx, uint32_t CenterAlignedMode)
{
    MODIFY_REG(timx->CON1, TIMER_CON1_CMSEL_MSK, CenterAlignedMode);
}

/**
  * @brief  Get timer center-aligned mode selection.
  * @param  timx TIMER instance
  * @retval Timer center-aligned mode selection.
  */

__STATIC_INLINE uint32_t md_timer_get_counter_aligned_mode_cmsel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON1, TIMER_CON1_CMSEL_MSK) >> TIMER_CON1_CMSEL_POSS);
}

/**
  * @brief  Timer counting direction setup.
  * @param  timx TIMER instance
  * @param  direction Counting direction
            @arg @ref MD_TIMER_UPCOUNTER
            @arg @ref MD_TIMER_DOWNCOUNTER
  * @retval None
  */

__STATIC_INLINE void md_timer_set_counter_direction_dirsel(TIMER_TypeDef *timx, uint32_t direction)
{
    MODIFY_REG(timx->CON1, TIMER_CON1_DIRSEL_MSK, direction);
}

/**
  * @brief  Get timer counting direction.
  * @param  timx TIMER instance
  * @retval Timer counting direction.
  */

__STATIC_INLINE uint32_t md_timer_get_counter_direction_dirsel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON1, TIMER_CON1_DIRSEL_MSK) >> TIMER_CON1_DIRSEL_POS);
}

/**
  * @brief  Timer one pulse mode enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_onepulse_spmen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CON1, TIMER_CON1_SPMEN_MSK);
}

/**
  * @brief  Timer one pulse mode disable.
  * @param  timx TIMER instance
  * @retval None.
  */

__STATIC_INLINE void md_timer_disable_onepulse_spmen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CON1, TIMER_CON1_SPMEN_MSK);
}

/**
  * @brief  Indicates whether the timer one pulse mode is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_onepulse_spmen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON1, TIMER_CON1_SPMEN_MSK) == (TIMER_CON1_SPMEN_MSK));
}

/**
  * @brief  Timer update request source setup.
  * @param  timx TIMER instance
  * @param  UpdateSource Update request source select
            @arg @ref MD_TIMER_UPDATESOURCE_NORMAL
            @arg @ref MD_TIMER_UPDATESOURCE_COUNTER
  * @retval None
  */

__STATIC_INLINE void md_timer_set_update_source_uersel(TIMER_TypeDef *timx, uint32_t UpdateSource)
{
    MODIFY_REG(timx->CON1, TIMER_CON1_UERSEL_MSK, UpdateSource);
}

/**
  * @brief  Get timer update request source.
  * @param  timx TIMER instance
  * @retval Timer update request source.
  */

__STATIC_INLINE uint32_t md_timer_get_update_source_uersel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON1, TIMER_CON1_UERSEL_MSK) >> TIMER_CON1_UERSEL_POS);
}

/**
  * @brief  Timer update event enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_update_disue(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CON1, TIMER_CON1_DISUE_MSK);
}

/**
  * @brief  Timer update event disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_update_disue(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CON1, TIMER_CON1_DISUE_MSK);
}

/**
  * @brief  Indicates whether the timer update event is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_update_disue(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON1, TIMER_CON1_DISUE_MSK) == (TIMER_CON1_DISUE_MSK));
}

/**
  * @brief  Timer counter enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_counter_cnten(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CON1, TIMER_CON1_CNTEN_MSK);
}


/**
  * @brief  Timer counter disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_counter_cnten(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CON1, TIMER_CON1_CNTEN_MSK);
}

/**
  * @brief  Indicates whether the timer counter is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_counter_cnten(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON1, TIMER_CON1_CNTEN_MSK) == (TIMER_CON1_CNTEN_MSK));
}

/**
  * @brief  Timer CON2 setup.
  * @param  timx TIMER instance
  * @param  value (I1SEL | MMSEL | CCDMASEL)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_con2(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->CON2, value);
}

/**
  * @brief  GET Timer CON2 register value.
  * @param  timx TIMER instance
  * @retval Timer CON2 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_con2(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->CON2));
}

/**
  * @brief  Timer output idle state 2 setup.
  * @param  timx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch2o_idle_select_oiss2(TIMER_TypeDef *timx, uint32_t IdleOutput)
{
    MODIFY_REG(timx->CON2, TIMER_CON2_OISS2_MSK, (IdleOutput << TIMER_CON2_OISS2_POS));
}

/**
  * @brief  Get timer output idle state 2.
  * @param  timx TIMER instance
  * @retval Timer output idle state 2.
  */

__STATIC_INLINE uint32_t md_timer_get_ch2o_idle_select_oiss2(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON2, TIMER_CON2_OISS2_MSK) >> TIMER_CON2_OISS2_POS);
}

/**
  * @brief  Timer output idle state 1N setup.
  * @param  timx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch1on_idle_select_oiss1n(TIMER_TypeDef *timx, uint32_t IdleOutput)
{
    MODIFY_REG(timx->CON2, TIMER_CON2_OISS1N_MSK, (IdleOutput << TIMER_CON2_OISS1N_POS));
}

/**
  * @brief  Get timer output idle state 1N.
  * @param  timx TIMER instance
  * @retval Timer output idle state 1N.
  */

__STATIC_INLINE uint32_t md_timer_get_ch1on_idle_select_oiss1n(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON2, TIMER_CON2_OISS1N_MSK) >> TIMER_CON2_OISS1N_POS);
}

/**
  * @brief  Timer output idle state 1 setup.
  * @param  timx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch1o_idle_select_oiss1(TIMER_TypeDef *timx, uint32_t IdleOutput)
{
    MODIFY_REG(timx->CON2, TIMER_CON2_OISS1_MSK, (IdleOutput << TIMER_CON2_OISS1_POS));
}

/**
  * @brief  Get timer output idle state 1.
  * @param  timx TIMER instance
  * @retval Timer output idle state 1.
  */

__STATIC_INLINE uint32_t md_timer_get_ch1o_idle_select_oiss1(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON2, TIMER_CON2_OISS1_MSK) >> TIMER_CON2_OISS1_POS);
}

/**
  * @brief  Timer TI1 selection setup.
  * @param  timx TIMER instance
  * @param  TI1Input TI1 input select
            @arg @ref MD_TIMER_TI1INPUT_CH1
            @arg @ref MD_TIMER_TI1INPUT_XOR
  * @retval None
  */

__STATIC_INLINE void md_timer_set_i1_func_select_i1sel(TIMER_TypeDef *timx, uint32_t TI1Input)
{
    MODIFY_REG(timx->CON2, TIMER_CON2_I1SEL_MSK, TI1Input);
}

/**
  * @brief  Get timer TI1 selection.
  * @param  timx TIMER instance
  * @retval Timer TI1 selection.
  */

__STATIC_INLINE uint32_t md_timer_get_i1_func_select_i1sel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON2, TIMER_CON2_I1SEL_MSK) >> TIMER_CON2_I1SEL_POS);
}

/**
  * @brief  Timer master mode selection setup.
  * @param  timx TIMER instance
  * @param  MasterMode Master mode selection
            @arg @ref MD_TIMER_MASTERMODE_RESET
            @arg @ref MD_TIMER_MASTERMODE_ENABLE
            @arg @ref MD_TIMER_MASTERMODE_UPDATE
            @arg @ref MD_TIMER_MASTERMODE_COMPAREPULSE
            @arg @ref MD_TIMER_MASTERMODE_COMPARE1
            @arg @ref MD_TIMER_MASTERMODE_COMPARE2
            @arg @ref MD_TIMER_MASTERMODE_COMPARE3
            @arg @ref MD_TIMER_MASTERMODE_COMPARE4
  * @retval None
  */

__STATIC_INLINE void md_timer_set_master_trgout_select_mmsel(TIMER_TypeDef *timx, uint32_t MasterMode)
{
    MODIFY_REG(timx->CON2, TIMER_CON2_MMSEL_MSK, MasterMode);
}

/**
  * @brief  Get timer master mode selection.
  * @param  timx TIMER instance
  * @retval Timer master mode selection.
  */

__STATIC_INLINE uint32_t md_timer_get_master_trgout_select_mmsel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON2, TIMER_CON2_MMSEL_MSK) >> TIMER_CON2_MMSEL_POSS);
}
/**
  * @brief  Timer capture/compare control update selection setup.
  * @param  timx TIMER instance
  * @param  UpdateSelection Capture/compare control update selection
            @arg @ref MD_TIMER_UPDATESELECTION_COMG
            @arg @ref MD_TIMER_UPDATESELECTION_BOTH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc_update_select_ccusel(TIMER_TypeDef *timx, uint32_t UpdateSelection)
{
    MODIFY_REG(timx->CON2, TIMER_CON2_CCUSEL_MSK, UpdateSelection);
}

/**
  * @brief  Get timer capture/compare control update selection.
  * @param  timx TIMER instance
  * @retval Timer capture/compare control update selection.
  */

__STATIC_INLINE uint32_t md_timer_get_cc_update_select_ccusel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON2, TIMER_CON2_CCUSEL_MSK) >> TIMER_CON2_CCUSEL_POS);
}

/**
  * @brief  Timer capture/compare preloaded control enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc_preload_ccpcen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CON2, TIMER_CON2_CCPCEN_MSK);
}

/**
  * @brief  Timer capture/compare preloaded control disable.
  * @param  timx TIMER instance
  * @retval None.
  */

__STATIC_INLINE void md_timer_disable_cc_preload_ccpcen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CON2, TIMER_CON2_CCPCEN_MSK);
}

/**
  * @brief  Indicates whether the timer capture/compare preloaded control is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc_preload_ccpcen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CON2, TIMER_CON2_CCPCEN_MSK) == (TIMER_CON2_CCPCEN_MSK));
}
/**
  * @brief  Timer SMCON setup.
  * @param  timx TIMER instance
  * @param  value (ETPOL | ECM2EN | ETFLT | MSCFG | TSSEL | CHCSEL | SMODS)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_smcon(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->SMCON, value);
}

/**
  * @brief  GET Timer SMCON register value.
  * @param  timx TIMER instance
  * @retval Timer SMCON register value.
  */

__STATIC_INLINE uint32_t md_timer_get_smcon(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->SMCON));
}

/**
  * @brief  Timer external trigger polarity setup.
  * @param  timx TIMER instance
  * @param  ETRPolarity External trigger polarity
            @arg @ref MD_TIMER_ETRPOLARITY_NONINVERTED
            @arg @ref MD_TIMER_ETRPOLARITY_INVERTED
  * @retval None
  */

__STATIC_INLINE void md_timer_set_external_trigger_polarity_etpol(TIMER_TypeDef *timx, uint32_t ETRPolarity)
{
    MODIFY_REG(timx->SMCON, TIMER_SMCON_ETPOL_MSK, ETRPolarity);
}

/**
  * @brief  Get timer external trigger polarity.
  * @param  timx TIMER instance
  * @retval Timer external trigger polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_external_trigger_polarity_etpol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->SMCON, TIMER_SMCON_ETPOL_MSK) >> TIMER_SMCON_ETPOL_POS);
}

/**
  * @brief  Timer external clock enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_external_clk2mode_ecm2en(TIMER_TypeDef *timx)
{
    SET_BIT(timx->SMCON, TIMER_SMCON_ECM2EN_MSK);
}


/**
  * @brief  Timer external clock disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_external_clk2mode_ecm2en(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->SMCON, TIMER_SMCON_ECM2EN_MSK);
}

/**
  * @brief  Indicates whether the timer external clock is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_external_clk2mode_ecm2en(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->SMCON, TIMER_SMCON_ECM2EN_MSK) == (TIMER_SMCON_ECM2EN_MSK));
}

/**
  * @brief  Timer external trigger filter setup.
  * @param  timx TIMER instance
  * @param  ETRFilter External trigger filter
            @arg @ref MD_TIMER_ETRFILTER_FDIV1
            @arg @ref MD_TIMER_ETRFILTER_FDIV1N2
            @arg @ref MD_TIMER_ETRFILTER_FDIV1N4
            @arg @ref MD_TIMER_ETRFILTER_FDIV1N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV2N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV2N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV4N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV4N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV8N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV8N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV16N5
            @arg @ref MD_TIMER_ETRFILTER_FDIV16N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV16N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV32N5
            @arg @ref MD_TIMER_ETRFILTER_FDIV32N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_external_trigger_filter_etflt(TIMER_TypeDef *timx, uint32_t ETRFilter)
{
    MODIFY_REG(timx->SMCON, TIMER_SMCON_ETFLT_MSK, ETRFilter);
}

/**
  * @brief  Get timer external trigger filter.
  * @param  timx TIMER instance
  * @retval Timer external trigger filter.
  */

__STATIC_INLINE uint32_t md_timer_get_external_trigger_filter_etflt(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->SMCON, TIMER_SMCON_ETFLT_MSK) >> TIMER_SMCON_ETFLT_POSS);
}

/**
  * @brief  Timer master/slave mode setup.
  * @param  timx TIMER instance
  * @param  MSMode master/slave mode
            @arg @ref MD_TIMER_MASTERSLAVE_NODELAY
            @arg @ref MD_TIMER_MASTERSLAVE_DELAY
  * @retval None
  */

__STATIC_INLINE void md_timer_set_smcon_mscfg(TIMER_TypeDef *timx, uint32_t MSMode)
{
    MODIFY_REG(timx->SMCON, TIMER_SMCON_MSCFG_MSK, MSMode);
}

/**
  * @brief  Get timer master/slave mode.
  * @param  timx TIMER instance
  * @retval Timer master/slave mode.
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_master_mode_mscfg(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->SMCON, TIMER_SMCON_MSCFG_MSK) >> TIMER_SMCON_MSCFG_POS);
}

/**
  * @brief  Timer trigger selection setup.
  * @param  timx GP16C2T instance
  * @param  TriggerSelect Trigger selection
            @arg @ref MD_TIMER_TRIGGERSELECT_ITR0
            @arg @ref MD_TIMER_TRIGGERSELECT_ITR1
            @arg @ref MD_TIMER_TRIGGERSELECT_ITR2
            @arg @ref MD_TIMER_TRIGGERSELECT_ITR3
            @arg @ref MD_TIMER_TRIGGERSELECT_TI1FED
            @arg @ref MD_TIMER_TRIGGERSELECT_TI1FP1
            @arg @ref MD_TIMER_TRIGGERSELECT_TI2FP2
            @arg @ref MD_TIMER_TRIGGERSELECT_ETRF
  * @retval None
  */

__STATIC_INLINE void md_timer_set_slave_trigger_tssel(TIMER_TypeDef *timx, uint32_t TriggerSelect)
{
    MODIFY_REG(timx->SMCON, TIMER_SMCON_TSSEL_MSK, TriggerSelect);
}

/**
  * @brief  Get timer trigger selection.
  * @param  timx TIMER instance
  * @retval Timer trigger selection.
  */

__STATIC_INLINE uint32_t md_timer_get_slave_trigger_tssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->SMCON, TIMER_SMCON_TSSEL_MSK) >> TIMER_SMCON_TSSEL_POSS);
}

/**
  * @brief  Timer trigger selection setup.
  * @param  timx TIMER instance
  * @param  TriggerSelect Trigger selection
            @arg @ref MD_TIMER_TRIGGERSELECT_ITR0
            @arg @ref MD_TIMER_TRIGGERSELECT_ITR1
            @arg @ref MD_TIMER_TRIGGERSELECT_ITR2
            @arg @ref MD_TIMER_TRIGGERSELECT_ITR3
            @arg @ref MD_TIMER_TRIGGERSELECT_TI1FED
            @arg @ref MD_TIMER_TRIGGERSELECT_TI1FP1
            @arg @ref MD_TIMER_TRIGGERSELECT_TI2FP2
            @arg @ref MD_TIMER_TRIGGERSELECT_ETRF
  * @retval None
  */

__STATIC_INLINE void md_timer_set_smcon_tssel(TIMER_TypeDef *timx, uint32_t TriggerSelect)
{
    MODIFY_REG(timx->SMCON, TIMER_SMCON_TSSEL_MSK, TriggerSelect);
}

/**
  * @brief  Get timer trigger selection.
  * @param  timx TIMER instance
  * @retval Timer trigger selection.
  */

__STATIC_INLINE uint32_t md_timer_get_smcon_tssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->SMCON, TIMER_SMCON_TSSEL_MSK) >> TIMER_SMCON_TSSEL_POSS);
}

/**
  * @brief  Timer OCREF clear selection setup.
  * @param  timx TIMER instance
  * @param  OCREFSelect OCREF clear selection
            @arg @ref MD_TIMER_OCREFCLEAR_CMP
            @arg @ref MD_TIMER_OCREFCLEAR_ETRF
  * @retval None
  */

__STATIC_INLINE void md_timer_set_smcon_chcsel(TIMER_TypeDef *timx, uint32_t OCREFSelect)
{
    MODIFY_REG(timx->SMCON, TIMER_SMCON_CHCSEL_MSK, OCREFSelect);
}

/**
  * @brief  Get timer OCREF clear selection.
  * @param  timx TIMER instance
  * @retval Timer OCREF clear selection.
  */

__STATIC_INLINE uint32_t md_timer_get_smcon_chcsel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->SMCON, TIMER_SMCON_CHCSEL_MSK) >> TIMER_SMCON_CHCSEL_POS);
}

/**
  * @brief  Timer slave mode selection setup.
  * @param  timx TIMER instance
  * @param  SlaveMode Slave mode selection
            @arg @ref MD_TIMER_SLAVEMODE_DISABLE
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER1
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER2
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER3
            @arg @ref MD_TIMER_SLAVEMODE_RESET
            @arg @ref MD_TIMER_SLAVEMODE_GATED
            @arg @ref MD_TIMER_SLAVEMODE_TRIGGER
            @arg @ref MD_TIMER_SLAVEMODE_EXTERNALCLOCK
  * @retval None
  */

__STATIC_INLINE void md_timer_set_slave_mode_smods(TIMER_TypeDef *timx, uint32_t SlaveMode)
{
    MODIFY_REG(timx->SMCON, TIMER_SMCON_SMODS_MSK, SlaveMode);
}

/**
  * @brief  Get timer slave mode selection.
  * @param  timx TIMER instance
  * @retval Timer slave mode selection.
  */

__STATIC_INLINE uint32_t md_timer_get_slave_mode_smods(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->SMCON, TIMER_SMCON_SMODS_MSK) >> TIMER_SMCON_SMODS_POSS);
}

/**
  * @brief  Timer IER setup.
  * @param  timx TIMER instance
  * @param  value (CH4OVI | CH3OVI | CH2OVI | CH1OVI | TRGI | CH4I | CH3I | CH2I | CH1I | UI)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ier(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->IER, value);
}

/**
  * @brief  Timer cpture/compare 4 overcapture interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch4ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_CH4OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 3 overcapture interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch3ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_CH3OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 2 overcapture interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch2ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_CH2OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 1 overcapture interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch1ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_CH1OVI_MSK);
}

/**
  * @brief  Timer break interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_brk(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_BRKI_MSK);
}

/**
  * @brief  Timer trigger interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_trgi(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_TRGI_MSK);
}

/**
  * @brief  Timer COM interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_com(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_COMI_MSK);
}

/**
  * @brief  Timer capture/compare 4 interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch4(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_CH4I_MSK);
}

/**
  * @brief  Timer capture/compare 3 interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch3(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_CH3I_MSK);
}

/**
  * @brief  Timer capture/compare 2 interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch2(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_CH2I_MSK);
}

/**
  * @brief  Timer capture/compare 1 interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch1(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_CH1I_MSK);
}

/**
  * @brief  Timer update interrupt enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_upd(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IER, TIMER_IER_UI_MSK);
}

/**
  * @brief  Timer IDR setup.
  * @param  timx TIMER instance
  * @param  value (CH4OVI | CH3OVI | CH2OVI | CH1OVI | TRGI | CH4I | CH3I | CH2I | CH1I | UI)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_idr(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->IDR, value);
}

/**
  * @brief  Timer cpture/compare 4 overcapture interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch4ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_CH4OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 3 overcapture interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch3ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_CH3OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 2 overcapture interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch2ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_CH2OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 1 overcapture interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch1ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_CH1OVI_MSK);
}

/**
  * @brief  Timer break interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_brk(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_BRKI_MSK);
}

/**
  * @brief  Timer trigger interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_trgi(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_TRGI_MSK);
}

/**
  * @brief  Timer COM interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_com(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_COMI_MSK);
}

/**
  * @brief  Timer capture/compare 4 interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch4(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_CH4I_MSK);
}

/**
  * @brief  Timer capture/compare 3 interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch3(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_CH3I_MSK);
}

/**
  * @brief  Timer capture/compare 2 interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch2(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_CH2I_MSK);
}

/**
  * @brief  Timer capture/compare 1 interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch1(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_CH1I_MSK);
}

/**
  * @brief  Timer update interrupt disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_upd(TIMER_TypeDef *timx)
{
    SET_BIT(timx->IDR, TIMER_IDR_UI_MSK);
}

/**
  * @brief  Get timer IVS setup.
  * @param  timx TIMER instance
  * @retval Timer IVS setup.
  */

__STATIC_INLINE uint32_t md_timer_get_ivs(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->IVS));
}

/**
  * @brief  Indicates whether the timer capture/compare 4 overcapture interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch4ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_CH4OVI_MSK) == (TIMER_IVS_CH4OVI_MSK));
}

/**
  * @brief  Indicates whether the timer capture/compare 3 overcapture interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch3ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_CH3OVI_MSK) == (TIMER_IVS_CH3OVI_MSK));
}

/**
  * @brief  Indicates whether the timer cpture/compare 2 overcapture interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch2ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_CH2OVI_MSK) == (TIMER_IVS_CH2OVI_MSK));
}

/**
  * @brief  Indicates whether the timer capture/compare 1 overcapture interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch1ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_CH1OVI_MSK) == (TIMER_IVS_CH1OVI_MSK));
}

/**
  * @brief  Indicates whether the timer break interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_brk(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_BRKI_MSK) == (TIMER_IVS_BRKI_MSK));
}

/**
  * @brief  Indicates whether the timer trigger interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_trgi(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_TRGI_MSK) == (TIMER_IVS_TRGI_MSK));
}

/**
  * @brief  Indicates whether the timer COM interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_com(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_COMI_MSK) == (TIMER_IVS_COMI_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 4 interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch4(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_CH4I_MSK) == (TIMER_IVS_CH4I_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 3 interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch3(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_CH3I_MSK) == (TIMER_IVS_CH3I_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch2(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_CH2I_MSK) == (TIMER_IVS_CH2I_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch1(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_CH1I_MSK) == (TIMER_IVS_CH1I_MSK));
}

/**
  * @brief  Indicates whether the timer update interrupt is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_upd(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IVS, TIMER_IVS_UI_MSK) == (TIMER_IVS_UI_MSK));
}

/**
  * @brief  Get timer RIF flag.
  * @param  timx TIMER instance
  * @retval Timer RIF flag.
  */

__STATIC_INLINE uint32_t md_timer_get_rif(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->RIF));
}

/**
  * @brief  Get timer capture/compare 4 overcapture interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch4ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_CH4OVI_MSK) == (TIMER_RIF_CH4OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 3 overcapture interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch3ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_CH3OVI_MSK) == (TIMER_RIF_CH3OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 2 overcapture interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch2ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_CH2OVI_MSK) == (TIMER_RIF_CH2OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 1 overcapture interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch1ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_CH1OVI_MSK) == (TIMER_RIF_CH1OVI_MSK));
}

/**
  * @brief  Get timer break interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_brk(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_BRKI_MSK) == (TIMER_RIF_BRKI_MSK));
}

/**
  * @brief  Get timer trigger interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_trgi(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_TRGI_MSK) == (TIMER_RIF_TRGI_MSK));
}

/**
  * @brief  Get timer COM interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_com(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_COMI_MSK) == (TIMER_RIF_COMI_MSK));
}

/**
  * @brief  Get timer capture/compare 4 interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch4(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_CH4I_MSK) == (TIMER_RIF_CH4I_MSK));
}

/**
  * @brief  Get timer capture/compare 3 interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch3(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_CH3I_MSK) == (TIMER_RIF_CH3I_MSK));
}

/**
  * @brief  Get timer capture/compare 2 interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch2(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_CH2I_MSK) == (TIMER_RIF_CH2I_MSK));
}

/**
  * @brief  Get timer capture/compare 1 interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch1(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_CH1I_MSK) == (TIMER_RIF_CH1I_MSK));
}

/**
  * @brief  Get timer update interrupt flag.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_upd(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->RIF, TIMER_RIF_UI_MSK) == (TIMER_RIF_UI_MSK));
}

/**
  * @brief  Get timer IFM flag.
  * @param  timx TIMER instance
  * @retval Timer IFM flag.
  */

__STATIC_INLINE uint32_t md_timer_get_ifm(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->IFM));
}

/**
  * @brief  Get timer capture/compare 4 overcapture interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch4ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_CH4OVI_MSK) == (TIMER_IFM_CH4OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 3 overcapture interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch3ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_CH3OVI_MSK) == (TIMER_IFM_CH3OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 2 overcapture interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch2ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_CH2OVI_MSK) == (TIMER_IFM_CH2OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 1 overcapture interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch1ov(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_CH1OVI_MSK) == (TIMER_IFM_CH1OVI_MSK));
}

/**
  * @brief  Get timer break interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_brk(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_BRKI_MSK) == (TIMER_IFM_BRKI_MSK));
}

/**
  * @brief  Get timer trigger interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_trgi(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_TRGI_MSK) == (TIMER_IFM_TRGI_MSK));
}

/**
  * @brief  Get timer COM interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_com(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_COMI_MSK) == (TIMER_IFM_COMI_MSK));
}

/**
  * @brief  Get timer capture/compare 4 interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch4(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_CH4I_MSK) == (TIMER_IFM_CH4I_MSK));
}

/**
  * @brief  Get timer capture/compare 3 interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch3(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_CH3I_MSK) == (TIMER_IFM_CH3I_MSK));
}

/**
  * @brief  Get timer capture/compare 2 interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch2(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_CH2I_MSK) == (TIMER_IFM_CH2I_MSK));
}

/**
  * @brief  Get timer capture/compare 1 interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch1(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_CH1I_MSK) == (TIMER_IFM_CH1I_MSK));
}

/**
  * @brief  Get timer update interrupt flag masked.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_upd(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->IFM, TIMER_IFM_UI_MSK) == (TIMER_IFM_UI_MSK));
}

/**
  * @brief  Timer ICR setup.
  * @param  timx TIMER instance
  * @param  value (CC4OI | CC3OI | CC2OI | CC1OI | TI | CC4I | CC3I | CC2I | CC1I | UI)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_icr(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->ICR, value);
}

/**
  * @brief  Clear timer cpture/compare 4 overcapture interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch4ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_CH4OVI_MSK);
}

/**
  * @brief  Clear timer cpture/compare 3 overcapture interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch3ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_CH3OVI_MSK);
}

/**
  * @brief  Clear timer cpture/compare 2 overcapture interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch2ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_CH2OVI_MSK);
}

/**
  * @brief  Clear timer cpture/compare 1 overcapture interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch1ov(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_CH1OVI_MSK);
}

/**
  * @brief  Clear timer trigger interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_brk(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_TRGI_MSK);
}

/**
  * @brief  Clear timer trigger interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_trgi(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_TRGI_MSK);
}

/**
  * @brief  Clear timer COM interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_com(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_COMI_MSK);
}

/**
  * @brief  Clear timer capture/compare 4 interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch4(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_CH4I_MSK);
}

/**
  * @brief  Clear timer capture/compare 3 interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch3(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_CH3I_MSK);
}

/**
  * @brief  Clear timer capture/compare 2 interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch2(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_CH2I_MSK);
}

/**
  * @brief  Clear timer capture/compare 1 interrupt flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch1(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_CH1I_MSK);
}

/**
  * @brief  Clear timer upadte flag.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_upd(TIMER_TypeDef *timx)
{
    SET_BIT(timx->ICR, TIMER_ICR_UI_MSK);
}

/**
  * @brief  Timer SGE setup.
  * @param  timx TIMER instance
  * @param  value (SGTRG | SGCH4 | SGCH3 | SGCH2 | SGCH1 | SGU)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_sge(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->SGE, value);
}

/**
  * @brief  Timer break generation.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgbrk(TIMER_TypeDef *timx)
{
    SET_BIT(timx->SGE, TIMER_SGE_SGBRK_MSK);
}

/**
  * @brief  Timer trigger generation.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgtrgi(TIMER_TypeDef *timx)
{
    SET_BIT(timx->SGE, TIMER_SGE_SGTRG_MSK);
}

/**
  * @brief  Timer COM generation.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgcom(TIMER_TypeDef *timx)
{
    SET_BIT(timx->SGE, TIMER_SGE_SGCOM_MSK);
}

/**
  * @brief  Timer Capture/Compare 4 generation.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgch4(TIMER_TypeDef *timx)
{
    SET_BIT(timx->SGE, TIMER_SGE_SGCH4_MSK);
}

/**
  * @brief  Timer Capture/Compare 3 generation.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgch3(TIMER_TypeDef *timx)
{
    SET_BIT(timx->SGE, TIMER_SGE_SGCH3_MSK);
}

/**
  * @brief  Timer Capture/Compare 2 generation.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgch2(TIMER_TypeDef *timx)
{
    SET_BIT(timx->SGE, TIMER_SGE_SGCH2_MSK);
}

/**
  * @brief  Timer Capture/Compare 1 generation.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgch1(TIMER_TypeDef *timx)
{
    SET_BIT(timx->SGE, TIMER_SGE_SGCH1_MSK);
}

/**
  * @brief  Timer update generation.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgupd(TIMER_TypeDef *timx)
{
    SET_BIT(timx->SGE, TIMER_SGE_SGU_MSK);
}

/**
  * @brief  Timer CHMR1 setup.
  * @param  timx TIMER instance
  * @param  value output (CH2OCLREN | CH2MOD | CH2PEN | CH2FEN | CC2SSEL | CH1OCLREN | CH1MOD | CH1PEN | CH1FEN | CC1SSEL)
  *               input  (I2FLT | I2PRES | CC2SSEL | I1FLT | I1PRES | CC1SSEL)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_chmr1(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->CHMR1, value);
}

/**
  * @brief  GET Timer CHMR1 register value.
  * @param  timx TIMER instance
  * @retval Timer CHMR1 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_chmr1(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->CHMR1));
}

/**
  * @brief  Timer output compare 2 clear enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare2_clear_ch2oclren(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2OCLREN_MSK);
}

/**
  * @brief  Timer output compare 2 clear disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare2_clear_ch2oclren(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2OCLREN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 2 clear is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare2_clear_ch2oclren(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2OCLREN_MSK) == (TIMER_CHMR1_OUTPUT_CH2OCLREN_MSK));
}

/**
  * @brief  Timer output compare 2 mode setup.
  * @param  timx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare2_mode_ch2mod(TIMER_TypeDef *timx, uint32_t OutputMode)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2MOD_MSK, (OutputMode << TIMER_CHMR1_OUTPUT_CH2MOD_POSS));
}

/**
  * @brief  Get timer output compare 2 mode.
  * @param  timx TIMER instance
  * @retval Timer output compare 2 mode.
  */

__STATIC_INLINE uint32_t md_timer_get_output_compare2_mode_ch2mod(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2MOD_MSK) >> TIMER_CHMR1_OUTPUT_CH2MOD_POSS);
}

/**
  * @brief  Timer output compare 2 preload enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare2_preload_ch2pen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2PEN_MSK);
}

/**
  * @brief  Timer output compare 2 preload disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare2_preload_ch2pen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 2 preload is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare2_preload_ch2pen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2PEN_MSK) == (TIMER_CHMR1_OUTPUT_CH2PEN_MSK));
}

/**
  * @brief  Timer output compare 2 fast enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare2_fast_ch2fen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2FEN_MSK);
}

/**
  * @brief  Timer output compare 2 fast disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare2_fast_ch2fen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 2 fast is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare2_fast_ch2fen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH2FEN_MSK) == (TIMER_CHMR1_OUTPUT_CH2FEN_MSK));
}

/**
  * @brief  Timer cpture/compare 2 selection setup.
  * @param  timx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc2_func_cc2ssel(TIMER_TypeDef *timx, uint32_t ChannelMode)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_OUTPUT_CC2SSEL_MSK, (ChannelMode << TIMER_CHMR1_OUTPUT_CC2SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 2 selection.
  * @param  timx TIMER instance
  * @retval Timer cpture/compare 2 selection.
  */

__STATIC_INLINE uint32_t md_timer_get_cc2_func_cc2ssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CC2SSEL_MSK) >> TIMER_CHMR1_OUTPUT_CC2SSEL_POSS);
}

/**
  * @brief  Timer output compare 1 clear enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare1_clear_ch1oclren(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1OCLREN_MSK);
}

/**
  * @brief  Timer output compare 1 clear disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare1_clear_ch1oclren(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1OCLREN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 1 clear is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare1_clear_ch1oclren(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1OCLREN_MSK) == (TIMER_CHMR1_OUTPUT_CH1OCLREN_MSK));
}

/**
  * @brief  Timer output compare 1 mode setup.
  * @param  timx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare1_mode_ch1mod(TIMER_TypeDef *timx, uint32_t OutputMode)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1MOD_MSK, (OutputMode << TIMER_CHMR1_OUTPUT_CH1MOD_POSS));
}

/**
  * @brief  Get timer output compare 1 mode.
  * @param  timx TIMER instance
  * @retval Timer output compare 1 mode.
  */

__STATIC_INLINE uint32_t md_timer_get_output_compare1_mode_ch1mod(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1MOD_MSK) >> TIMER_CHMR1_OUTPUT_CH1MOD_POSS);
}

/**
  * @brief  Timer output compare 1 preload enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare1_preload_ch1pen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1PEN_MSK);
}

/**
  * @brief  Timer output compare 1 preload disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare1_preload_ch1pen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 1 preload is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare1_preload_ch1pen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1PEN_MSK) == (TIMER_CHMR1_OUTPUT_CH1PEN_MSK));
}

/**
  * @brief  Timer output compare 1 fast enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare1_fast_ch1fen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1FEN_MSK);
}

/**
  * @brief  Timer output compare 1 fast disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare1_fast_ch1fen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 1 fast is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare1_fast_ch1fen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CH1FEN_MSK) == (TIMER_CHMR1_OUTPUT_CH1FEN_MSK));
}

/**
  * @brief  Timer cpture/compare 1 selection setup.
  * @param  timx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc1_func_cc1ssel(TIMER_TypeDef *timx, uint32_t ChannelMode)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_OUTPUT_CC1SSEL_MSK, (ChannelMode << TIMER_CHMR1_OUTPUT_CC1SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 1 selection.
  * @param  timx TIMER instance
  * @retval Timer cpture/compare 1 selection.
  */

__STATIC_INLINE uint32_t md_timer_get_cc1_func_cc1ssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_OUTPUT_CC1SSEL_MSK) >> TIMER_CHMR1_OUTPUT_CC1SSEL_POSS);
}

/**
  * @brief  Timer input capture 2 filter setup.
  * @param  timx TIMER instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture2_filter_i2flt(TIMER_TypeDef *timx, uint32_t InputFliter)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_INPUT_I2FLT_MSK, (InputFliter << TIMER_CHMR1_INPUT_I2FLT_POSS));
}

/**
  * @brief  Get timer input capture 2 filter.
  * @param  timx TIMER instance
  * @retval Timer input capture 2 filter.
  */

__STATIC_INLINE uint32_t md_timer_get_input_capture2_filter_i2flt(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_INPUT_I2FLT_MSK) >> TIMER_CHMR1_INPUT_I2FLT_POSS);
}

/**
  * @brief  Timer input capture 2 prescaler setup.
  * @param  timx TIMER instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture2_prescaler_ic2pres(TIMER_TypeDef *timx, uint32_t InputPrescale)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_INPUT_I2PRES_MSK, (InputPrescale << TIMER_CHMR1_INPUT_I2PRES_POSS));
}

/**
  * @brief  Get timer input capture 2 prescaler.
  * @param  timx TIMER instance
  * @retval Timer input capture 2 prescaler.
  */

__STATIC_INLINE uint32_t md_timer_get_input_capture2_prescaler_ic2pres(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_INPUT_I2PRES_MSK) >> TIMER_CHMR1_INPUT_I2PRES_POSS);
}

/**
  * @brief  Timer cpture/compare 2 selection setup.
  * @param  timx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_chmr1_input_cc2ssel(TIMER_TypeDef *timx, uint32_t ChannelMode)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_INPUT_CC2SSEL_MSK, (ChannelMode << TIMER_CHMR1_INPUT_CC2SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 2 selection.
  * @param  timx TIMER instance
  * @retval Timer cpture/compare 2 selection.
  */

__STATIC_INLINE uint32_t md_timer_get_chmr1_input_cc2ssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_INPUT_CC2SSEL_MSK) >> TIMER_CHMR1_INPUT_CC2SSEL_POSS);
}

/**
  * @brief  Timer input capture 1 filter setup.
  * @param  timx TIMER instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture1_filter_i1flt(TIMER_TypeDef *timx, uint32_t InputFliter)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_INPUT_I1FLT_MSK, (InputFliter << TIMER_CHMR1_INPUT_I1FLT_POSS));
}

/**
  * @brief  Get timer input capture 1 filter.
  * @param  timx TIMER instance
  * @retval Timer input capture 1 filter.
  */

__STATIC_INLINE uint32_t md_timer_get_input_capture1_filter_i1flt(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_INPUT_I1FLT_MSK) >> TIMER_CHMR1_INPUT_I1FLT_POSS);
}

/**
  * @brief  Timer input capture 1 prescaler setup.
  * @param  timx TIMER instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture1_prescaler_ic1pres(TIMER_TypeDef *timx, uint32_t InputPrescale)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_INPUT_I1PRES_MSK, (InputPrescale << TIMER_CHMR1_INPUT_I1PRES_POSS));
}

/**
  * @brief  Get timer input capture 1 prescaler.
  * @param  timx TIMER instance
  * @retval Timer input capture 1 prescaler.
  */

__STATIC_INLINE uint32_t md_timer_get_input_capture1_prescaler_ic1pres(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_INPUT_I1PRES_MSK) >> TIMER_CHMR1_INPUT_I1PRES_POSS);
}

/**
  * @brief  Timer cpture/compare 1 selection setup.
  * @param  timx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_chmr1_input_cc1ssel(TIMER_TypeDef *timx, uint32_t ChannelMode)
{
    MODIFY_REG(timx->CHMR1, TIMER_CHMR1_INPUT_CC1SSEL_MSK, (ChannelMode << TIMER_CHMR1_INPUT_CC1SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 1 selection.
  * @param  timx TIMER instance
  * @retval Timer cpture/compare 1 selection.
  */

__STATIC_INLINE uint32_t md_timer_get_chmr1_input_cc1ssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR1, TIMER_CHMR1_INPUT_CC1SSEL_MSK) >> TIMER_CHMR1_INPUT_CC1SSEL_POSS);
}

/**
  * @brief  Timer CHMR2 setup.

  * @param  timx TIMER instance
  * @param  value output (CH4OCLREN | CH4MOD | CH4PEN | CH4FEN | CC4SSEL | CH3OCLREN | CH3MOD | CH3PEN | CH3FEN | CC3SSEL)
  *               input  (I4FLT | I4PRES | CC4SSEL | I3FLT | I3PRES | CC3SSEL)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_chmr2(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->CHMR2, value);
}

/**
  * @brief  GET Timer CHMR2 register value.
  * @param  timx TIMER instance
  * @retval Timer CHMR2 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_chmr2(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->CHMR2));
}

/**
  * @brief  Timer output compare 4 clear enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare4_clear_ch4oclren(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4OCLREN_MSK);
}

/**
  * @brief  Timer output compare 4 clear disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare4_clear_ch4oclren(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4OCLREN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 4 clear is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare4_clear_ch4oclren(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4OCLREN_MSK) == (TIMER_CHMR2_OUTPUT_CH4OCLREN_MSK));
}

/**
  * @brief  Timer output compare 4 mode setup.
  * @param  timx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare4_mode_ch4mod(TIMER_TypeDef *timx, uint32_t OutputMode)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4MOD_MSK, (OutputMode << TIMER_CHMR2_OUTPUT_CH4MOD_POSS));
}

/**
  * @brief  Get timer output compare 4 mode.
  * @param  timx TIMER instance
  * @retval Timer output compare 4 mode.
  */

__STATIC_INLINE uint32_t md_timer_get_output_compare4_mode_ch4mod(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4MOD_MSK) >> TIMER_CHMR2_OUTPUT_CH4MOD_POSS);
}

/**
  * @brief  Timer output compare 4 preload enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare4_preload_ch4pen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4PEN_MSK);
}

/**
  * @brief  Timer output compare 4 preload disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare4_preload_ch4pen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 4 preload is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare4_preload_ch4pen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4PEN_MSK) == (TIMER_CHMR2_OUTPUT_CH4PEN_MSK));
}

/**
  * @brief  Timer output compare 4 fast enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare4_fast_ch4fen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4FEN_MSK);
}

/**
  * @brief  Timer output compare 4 fast disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare4_fast_ch4fen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 4 fast is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare4_fast_ch4fen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH4FEN_MSK) == (TIMER_CHMR2_OUTPUT_CH4FEN_MSK));
}

/**
  * @brief  Timer cpture/compare 4 selection setup.
  * @param  timx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc4_func_cc4ssel(TIMER_TypeDef *timx, uint32_t ChannelMode)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_OUTPUT_CC4SSEL_MSK, (ChannelMode << TIMER_CHMR2_OUTPUT_CC4SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 4 selection.
  * @param  timx TIMER instance
  * @retval Timer cpture/compare 4 selection.
  */

__STATIC_INLINE uint32_t md_timer_get_cc4_func_cc4ssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CC4SSEL_MSK) >> TIMER_CHMR2_OUTPUT_CC4SSEL_POSS);
}

/**
  * @brief  Timer output compare 3 clear enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare3_clear_ch3oclren(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3OCLREN_MSK);
}

/**
  * @brief  Timer output compare 3 clear disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare3_clear_ch3oclren(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3OCLREN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 3 clear is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare3_clear_ch3oclren(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3OCLREN_MSK) == (TIMER_CHMR2_OUTPUT_CH3OCLREN_MSK));
}

/**
  * @brief  Timer output compare 3 mode setup.
  * @param  timx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare3_mode_ch3mod(TIMER_TypeDef *timx, uint32_t OutputMode)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3MOD_MSK, (OutputMode << TIMER_CHMR2_OUTPUT_CH3MOD_POSS));
}

/**
  * @brief  Get timer output compare 3 mode.
  * @param  timx TIMER instance
  * @retval Timer output compare 3 mode.
  */

__STATIC_INLINE uint32_t md_timer_get_output_compare3_mode_ch3mod(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3MOD_MSK) >> TIMER_CHMR2_OUTPUT_CH3MOD_POSS);
}

/**
  * @brief  Timer output compare 3 preload enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare3_preload_ch3pen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3PEN_MSK);
}

/**
  * @brief  Timer output compare 3 preload disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare3_preload_ch3pen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 3 preload is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare3_preload_ch3pen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3PEN_MSK) == (TIMER_CHMR2_OUTPUT_CH3PEN_MSK));
}

/**
  * @brief  Timer output compare 3 fast enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare3_fast_ch3fen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3FEN_MSK);
}

/**
  * @brief  Timer output compare 3 fast disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare3_fast_ch3fen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 3 fast is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare3_fast_ch3fen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CH3FEN_MSK) == (TIMER_CHMR2_OUTPUT_CH3FEN_MSK));
}

/**
  * @brief  Timer cpture/compare 3 selection setup.
  * @param  timx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc3_func_cc3ssel(TIMER_TypeDef *timx, uint32_t ChannelMode)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_OUTPUT_CC3SSEL_MSK, (ChannelMode << TIMER_CHMR2_OUTPUT_CC3SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 3 selection.
  * @param  timx TIMER instance
  * @retval Timer cpture/compare 3 selection.
  */

__STATIC_INLINE uint32_t md_timer_get_cc3_func_cc3ssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_OUTPUT_CC3SSEL_MSK) >> TIMER_CHMR2_OUTPUT_CC3SSEL_POSS);
}

/**
  * @brief  Timer input capture 4 filter setup.
  * @param  timx TIMER instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture4_filter_i4flt(TIMER_TypeDef *timx, uint32_t InputFliter)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_INPUT_I4FLT_MSK, (InputFliter << TIMER_CHMR2_INPUT_I4FLT_POSS));
}

/**
  * @brief  Get timer input capture 4 filter.
  * @param  timx TIMER instance
  * @retval Timer input capture 4 filter.
  */

__STATIC_INLINE uint32_t md_timer_get_input_capture4_filter_i4flt(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_INPUT_I4FLT_MSK) >> TIMER_CHMR2_INPUT_I4FLT_POSS);
}

/**
  * @brief  Timer input capture 4 prescaler setup.
  * @param  timx TIMER instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture4_prescaler_ic4pres(TIMER_TypeDef *timx, uint32_t InputPrescale)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_INPUT_I4PRES_MSK, (InputPrescale << TIMER_CHMR2_INPUT_I4PRES_POSS));
}

/**
  * @brief  Get timer input capture 4 prescaler.
  * @param  timx TIMER instance
  * @retval Timer input capture 4 prescaler.
  */

__STATIC_INLINE uint32_t md_timer_get_input_capture4_prescaler_ic4pres(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_INPUT_I4PRES_MSK) >> TIMER_CHMR2_INPUT_I4PRES_POSS);
}

/**
  * @brief  Timer cpture/compare 4 selection setup.
  * @param  timx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_chmr2_input_cc4ssel(TIMER_TypeDef *timx, uint32_t ChannelMode)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_INPUT_CC4SSEL_MSK, (ChannelMode << TIMER_CHMR2_INPUT_CC4SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 4 selection.
  * @param  timx TIMER instance
  * @retval Timer cpture/compare 4 selection.
  */

__STATIC_INLINE uint32_t md_timer_get_chmr2_input_cc4ssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_INPUT_CC4SSEL_MSK) >> TIMER_CHMR2_INPUT_CC4SSEL_POSS);
}

/**
  * @brief  Timer input capture 3 filter setup.
  * @param  timx TIMER instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture3_filter_i3flt(TIMER_TypeDef *timx, uint32_t InputFliter)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_INPUT_I3FLT_MSK, (InputFliter << TIMER_CHMR2_INPUT_I3FLT_POSS));
}

/**
  * @brief  Get timer input capture 3 filter.
  * @param  timx TIMER instance
  * @retval Timer input capture 3 filter.
  */

__STATIC_INLINE uint32_t md_timer_get_input_capture3_filter_i3flt(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_INPUT_I3FLT_MSK) >> TIMER_CHMR2_INPUT_I3FLT_POSS);
}

/**
  * @brief  Timer input capture 3 prescaler setup.
  * @param  timx TIMER instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture3_prescaler_ic3pres(TIMER_TypeDef *timx, uint32_t InputPrescale)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_INPUT_I3PRES_MSK, (InputPrescale << TIMER_CHMR2_INPUT_I3PRES_POSS));
}

/**
  * @brief  Get timer input capture 3 prescaler.
  * @param  timx TIMER instance
  * @retval Timer input capture 3 prescaler.
  */

__STATIC_INLINE uint32_t md_timer_get_input_capture3_prescaler_ic3pres(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_INPUT_I3PRES_MSK) >> TIMER_CHMR2_INPUT_I3PRES_POSS);
}

/**
  * @brief  Timer cpture/compare 3 selection setup.
  * @param  timx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_chmr2_input_cc3ssel(TIMER_TypeDef *timx, uint32_t ChannelMode)
{
    MODIFY_REG(timx->CHMR2, TIMER_CHMR2_INPUT_CC3SSEL_MSK, (ChannelMode << TIMER_CHMR2_INPUT_CC3SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 3 selection.
  * @param  timx TIMER instance
  * @retval Timer cpture/compare 3 selection.
  */

__STATIC_INLINE uint32_t md_timer_get_chmr2_input_cc3ssel(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CHMR2, TIMER_CHMR2_INPUT_CC3SSEL_MSK) >> TIMER_CHMR2_INPUT_CC3SSEL_POSS);
}

/**
  * @brief  Timer CCEP setup.
  * @param  timx TIMER instance
  * @param  value (CC4NPOL | CC4POL | CC4EN | CC3NPOL | CC3POL | CC3EN | CC2NPOL |
  *                CC2POL | CC2EN | CC1NPOL | CC1POL | CC1EN )
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ccep(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->CCEP, value);
}

/**
  * @brief  GET Timer CCEP register value.
  * @param  timx TIMER instance
  * @retval Timer CCEP register value.
  */

__STATIC_INLINE uint32_t md_timer_get_ccep(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->CCEP));
}

/**
  * @brief  Timer capture/compare 4 complementary output polarity setup.
  * @param  timx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc4_complementary_polarity_cc4npol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC4NPOL_MSK, (OutputPolarity << TIMER_CCEP_CC4NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 4 complementary output polarity.
  * @param  timx TIMER instance
  * @retval Timer capture/compare 4 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_cc4_complementary_polarity_cc4npol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC4NPOL_MSK) >> TIMER_CCEP_CC4NPOL_POS);
}

/**
  * @brief  Timer capture/compare 4 output polarity setup.
  * @param  timx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc4_output_polarity_cc4pol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC4POL_MSK, (OutputPolarity << TIMER_CCEP_CC4POL_POS));
}

__STATIC_INLINE void md_timer_set_cc4_input_edge_cc4pol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC4POL_MSK, (OutputPolarity << TIMER_CCEP_CC4POL_POS));
}

/**
  * @brief  Get timer capture/compare 4 output polarity.
  * @param  timx TIMER instance
  * @retval Timer capture/compare 4 output polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_cc4_output_polarity_cc4pol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC4POL_MSK) >> TIMER_CCEP_CC4POL_POS);
}

__STATIC_INLINE uint32_t md_timer_get_cc4_input_edge_cc4pol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC4POL_MSK) >> TIMER_CCEP_CC4POL_POS);
}

/**
  * @brief  Timer Capture/Compare 4 output enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc4_output_cc4en(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CCEP, TIMER_CCEP_CC4EN_MSK);
}

__STATIC_INLINE void md_timer_enable_cc4_input_cc4en(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CCEP, TIMER_CCEP_CC4EN_MSK);
}

/**
  * @brief  Timer Capture/Compare 4 output disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc4_output_cc4en(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CCEP, TIMER_CCEP_CC4EN_MSK);
}

__STATIC_INLINE void md_timer_disable_cc4_input_cc4en(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CCEP, TIMER_CCEP_CC4EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 4 output is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc4_output_cc4en(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC4EN_MSK) == (TIMER_CCEP_CC4EN_MSK));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc4_input_cc4en(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC4EN_MSK) == (TIMER_CCEP_CC4EN_MSK));
}

/**
  * @brief  Timer capture/compare 3 complementary output polarity setup.
  * @param  timx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc3_complementary_polarity_cc3npol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC3NPOL_MSK, (OutputPolarity << TIMER_CCEP_CC3NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 3 complementary output polarity.
  * @param  timx TIMER instance
  * @retval Timer capture/compare 3 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_cc3_complementary_polarity_cc3npol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC3NPOL_MSK) >> TIMER_CCEP_CC3NPOL_POS);
}

/**
  * @brief  Timer capture/compare 3 output polarity setup.
  * @param  timx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc3_output_polarity_cc3pol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC3POL_MSK, (OutputPolarity << TIMER_CCEP_CC3POL_POS));
}

__STATIC_INLINE void md_timer_set_cc3_input_edge_cc3pol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC3POL_MSK, (OutputPolarity << TIMER_CCEP_CC3POL_POS));
}

/**
  * @brief  Get timer capture/compare 3 output polarity.
  * @param  timx TIMER instance
  * @retval Timer capture/compare 3 output polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_cc3_output_polarity_cc3pol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC3POL_MSK) >> TIMER_CCEP_CC3POL_POS);
}

__STATIC_INLINE uint32_t md_timer_get_cc3_input_edge_cc3pol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC3POL_MSK) >> TIMER_CCEP_CC3POL_POS);
}

/**
  * @brief  Timer Capture/Compare 3 output enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc3_output_cc3en(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CCEP, TIMER_CCEP_CC3EN_MSK);
}

__STATIC_INLINE void md_timer_enable_cc3_input_cc3en(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CCEP, TIMER_CCEP_CC3EN_MSK);
}


/**
  * @brief  Timer Capture/Compare 3 output disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc3_output_cc3en(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CCEP, TIMER_CCEP_CC3EN_MSK);
}

__STATIC_INLINE void md_timer_disable_cc3_input_cc3en(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CCEP, TIMER_CCEP_CC3EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 3 output is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc3_output_cc3en(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC3EN_MSK) == (TIMER_CCEP_CC3EN_MSK));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc3_input_cc3en(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC3EN_MSK) == (TIMER_CCEP_CC3EN_MSK));
}
/**
  * @brief  Timer capture/compare 2 complementary output polarity setup.
  * @param  timx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc2_complementary_polarity_cc2npol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC2NPOL_MSK, (OutputPolarity << TIMER_CCEP_CC2NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 2 complementary output polarity.
  * @param  timx TIMER instance
  * @retval Timer capture/compare 2 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_cc2_complementary_polarity_cc2npol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC2NPOL_MSK) >> TIMER_CCEP_CC2NPOL_POS);
}

/**
  * @brief  Timer capture/compare 2 output polarity setup.
  * @param  timx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc2_output_polarity_cc2pol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC2POL_MSK, (OutputPolarity << TIMER_CCEP_CC2POL_POS));
}

__STATIC_INLINE void md_timer_set_cc2_input_edge_cc2pol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC2POL_MSK, (OutputPolarity << TIMER_CCEP_CC2POL_POS));
}

/**
  * @brief  Get timer capture/compare 2 output polarity.
  * @param  timx TIMER instance
  * @retval Timer capture/compare 2 output polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_cc2_output_polarity_cc2pol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC2POL_MSK) >> TIMER_CCEP_CC2POL_POS);
}

__STATIC_INLINE uint32_t md_timer_get_cc2_input_edge_cc2pol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC2POL_MSK) >> TIMER_CCEP_CC2POL_POS);
}

/**
  * @brief  Timer Capture/Compare 2 output enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc2_output_cc2en(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CCEP, TIMER_CCEP_CC2EN_MSK);
}

__STATIC_INLINE void md_timer_enable_cc2_input_cc2en(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CCEP, TIMER_CCEP_CC2EN_MSK);
}

/**
  * @brief  Timer Capture/Compare 2 output disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc2_output_cc2en(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CCEP, TIMER_CCEP_CC2EN_MSK);
}

__STATIC_INLINE void md_timer_disable_cc2_input_cc2en(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CCEP, TIMER_CCEP_CC2EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 output is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc2_output_cc2en(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC2EN_MSK) == (TIMER_CCEP_CC2EN_MSK));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc2_input_cc2en(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC2EN_MSK) == (TIMER_CCEP_CC2EN_MSK));
}

/**
  * @brief  Timer capture/compare 1 complementary output polarity setup.
  * @param  timx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc1_complementary_polarity_cc1npol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC1NPOL_MSK, (OutputPolarity << TIMER_CCEP_CC1NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 1 complementary output polarity.
  * @param  timx TIMER instance
  * @retval Timer capture/compare 1 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_cc1_complementary_polarity_cc1npol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC1NPOL_MSK) >> TIMER_CCEP_CC1NPOL_POS);
}

/**
  * @brief  Timer Capture/Compare 1 complementary output enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc1_complementary_cc1nen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CCEP, TIMER_CCEP_CC1NEN_MSK);
}

/**
  * @brief  Timer Capture/Compare 1 complementary output disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc1_complementary_cc1nen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CCEP, TIMER_CCEP_CC1NEN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 complementary output is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc1_complementary_cc1nen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC1NEN_MSK) == (TIMER_CCEP_CC1NEN_MSK));
}

/**
  * @brief  Timer capture/compare 1 output polarity setup.
  * @param  timx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc1_output_polarity_cc1pol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC1POL_MSK, (OutputPolarity << TIMER_CCEP_CC1POL_POS));
}

__STATIC_INLINE void md_timer_set_cc1_input_edge_cc1pol(TIMER_TypeDef *timx, uint32_t OutputPolarity)
{
    MODIFY_REG(timx->CCEP, TIMER_CCEP_CC1POL_MSK, (OutputPolarity << TIMER_CCEP_CC1POL_POS));
}

/**
  * @brief  Get timer capture/compare 1 output polarity.
  * @param  timx TIMER instance
  * @retval Timer capture/compare 1 output polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_cc1_output_polarity_cc1pol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC1POL_MSK) >> TIMER_CCEP_CC1POL_POS);
}

__STATIC_INLINE uint32_t md_timer_get_cc1_input_edge_cc1pol(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC1POL_MSK) >> TIMER_CCEP_CC1POL_POS);
}

/**
  * @brief  Timer Capture/Compare 1 output enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc1_output_cc1en(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CCEP, TIMER_CCEP_CC1EN_MSK);
}

__STATIC_INLINE void md_timer_enable_cc1_input_cc1en(TIMER_TypeDef *timx)
{
    SET_BIT(timx->CCEP, TIMER_CCEP_CC1EN_MSK);
}

/**
  * @brief  Timer Capture/Compare 1 output disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc1_output_cc1en(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CCEP, TIMER_CCEP_CC1EN_MSK);
}

__STATIC_INLINE void md_timer_disable_cc1_input_cc1en(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->CCEP, TIMER_CCEP_CC1EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 output is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc1_output_cc1en(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC1EN_MSK) == (TIMER_CCEP_CC1EN_MSK));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc1_input_cc1en(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCEP, TIMER_CCEP_CC1EN_MSK) == (TIMER_CCEP_CC1EN_MSK));
}

/**
  * @brief  Timer COUNT setup.
  * @param  timx TIMER instance
  * @param  value COUNT
  * @retval None
  */

__STATIC_INLINE void md_timer_set_counter_value_cntv(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->COUNT, value);
}

/**
  * @brief  GET Timer COUNT register value.
  * @param  timx TIMER instance
  * @retval Timer COUNT register value.
  */

__STATIC_INLINE uint32_t md_timer_get_counter_value_cntv(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->COUNT));
}

/**
  * @brief  Timer counter value setup.
  * @param  timx TIMER instance
  * @param  counter Counter value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cntv(TIMER_TypeDef *timx, uint32_t counter)
{
    MODIFY_REG(timx->COUNT, TIMER_COUNT_CNTV_MSK, counter);
}

/**
  * @brief  Get timer counter value.
  * @param  timx TIMER instance
  * @retval Timer counter value.
  */

__STATIC_INLINE uint32_t md_timer_get_cntv(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->COUNT, TIMER_COUNT_CNTV_MSK) >> TIMER_COUNT_CNTV_POSS);
}

/**
  * @brief  Timer PRES setup.
  * @param  timx TIMER instance
  * @param  value PRES
  * @retval None
  */

__STATIC_INLINE void md_timer_set_prescaler_value_pscv(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->PRES, value);
}

/**
  * @brief  GET Timer PRES register value.
  * @param  timx TIMER instance
  * @retval Timer PRES register value.
  */

__STATIC_INLINE uint32_t md_timer_get_prescaler_value_pscv(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->PRES));
}

/**
  * @brief  Timer prescaler value setup.
  * @param  timx TIMER instance
  * @param  prescaler Prescaler value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_pscv(TIMER_TypeDef *timx, uint32_t prescaler)
{
    MODIFY_REG(timx->PRES, TIMER_PRES_PSCV_MSK, prescaler);
}

/**
  * @brief  Get timer prescaler value.
  * @param  timx TIMER instance
  * @retval Timer prescaler value.
  */

__STATIC_INLINE uint32_t md_timer_get_pscv(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->PRES, TIMER_PRES_PSCV_MSK) >> TIMER_PRES_PSCV_POSS);
}

/**
  * @brief  Timer AR setup.
  * @param  timx TIMER instance
  * @param  value AR
  * @retval None
  */

__STATIC_INLINE void md_timer_set_auto_reload_value_arrv(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->AR, value);
}

/**
  * @brief  GET Timer AR register value.
  * @param  timx TIMER instance
  * @retval Timer AR register value.
  */

__STATIC_INLINE uint32_t md_timer_get_auto_reload_value_arrv(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->AR));
}

/**
  * @brief  Timer auto-reload value setup.
  * @param  timx TIMER instance
  * @param  reload Auto-reload value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_arv(TIMER_TypeDef *timx, uint32_t reload)
{
    MODIFY_REG(timx->AR, TIMER_AR_ARV_MSK, reload);
}

/**
  * @brief  Get timer auto-reload value.
  * @param  timx TIMER instance
  * @retval Timer auto-reload value.
  */

__STATIC_INLINE uint32_t md_timer_get_arv(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->AR, TIMER_AR_ARV_MSK) >> TIMER_AR_ARV_POSS);
}

/**
  * @brief  Timer REPAR setup.
  * @param  timx TIMER instance
  * @param  value REPAR
  * @retval None
  */

__STATIC_INLINE void md_timer_set_repetition_counter_value_repv(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->REPAR, value);
}

/**
  * @brief  GET Timer REPAR register value.
  * @param  timx TIMER instance
  * @retval Timer REPAR register value.
  */

__STATIC_INLINE uint32_t md_timer_get_repetition_counter_value_repv(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->REPAR));
}

/**
  * @brief  Timer repetition counter value setup.
  * @param  timx TIMER instance
  * @param  repetition Repetition counter value (between Min_Data=0 and Max_Data=0xFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_repv(TIMER_TypeDef *timx, uint32_t repetition)
{
    MODIFY_REG(timx->REPAR, TIMER_REPAR_REPV_MSK, repetition);
}

/**
  * @brief  Get timer repetition counter value.
  * @param  timx TIMER instance
  * @retval Timer repetition counter value.
  */

__STATIC_INLINE uint32_t md_timer_get_repv(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->REPAR, TIMER_REPAR_REPV_MSK) >> TIMER_REPAR_REPV_POSS);
}

/**
  * @brief  Timer CCVAL1 setup.
  * @param  timx TIMER instance
  * @param  value CCVAL1
  * @retval None
  */

__STATIC_INLINE void md_timer_set_capture_compare1_value_ccrv1(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->CCVAL1, value);
}

/**
  * @brief  GET Timer CCVAL1 register value.
  * @param  timx TIMER instance
  * @retval Timer CCVAL1 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_capture_compare1_value_ccrv1(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->CCVAL1));
}

/**
  * @brief  Timer capture/compare value 1 setup.
  * @param  timx TIMER instance
  * @param  CapCompValue Capture/Compare value 1 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ccrv1(TIMER_TypeDef *timx, uint32_t CapCompValue)
{
    MODIFY_REG(timx->CCVAL1, TIMER_CCVAL1_CCRV1_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 1.
  * @param  timx TIMER instance
  * @retval Timer capture/compare value 1.
  */

__STATIC_INLINE uint32_t md_timer_get_ccrv1(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCVAL1, TIMER_CCVAL1_CCRV1_MSK) >> TIMER_CCVAL1_CCRV1_POSS);
}

/**
  * @brief  Timer CCVAL2 setup.
  * @param  timx TIMER instance
  * @param  value CCVAL2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_capture_compare2_value_ccrv2(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->CCVAL2, value);
}

/**
  * @brief  GET Timer CCVAL2 register value.
  * @param  timx TIMER instance
  * @retval Timer CCVAL2 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_capture_compare2_value_ccrv2(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->CCVAL2));
}

/**
  * @brief  Timer capture/compare value 2 setup.
  * @param  timx TIMER instance
  * @param  CapCompValue Capture/Compare value 2 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ccrv2(TIMER_TypeDef *timx, uint32_t CapCompValue)
{
    MODIFY_REG(timx->CCVAL2, TIMER_CCVAL2_CCRV2_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 2.
  * @param  timx TIMER instance
  * @retval Timer capture/compare value 2.
  */

__STATIC_INLINE uint32_t md_timer_get_ccrv2(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCVAL2, TIMER_CCVAL2_CCRV2_MSK) >> TIMER_CCVAL2_CCRV2_POSS);
}

/**
  * @brief  Timer CCVAL3 setup.
  * @param  timx TIMER instance
  * @param  value CCVAL3
  * @retval None
  */

__STATIC_INLINE void md_timer_set_capture_compare3_value_ccrv3(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->CCVAL3, value);
}

/**
  * @brief  GET Timer CCVAL3 register value.
  * @param  timx TIMER instance
  * @retval Timer CCVAL3 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_capture_compare3_value_ccrv3(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->CCVAL3));
}

/**
  * @brief  Timer capture/compare value 3 setup.
  * @param  timx TIMER instance
  * @param  CapCompValue Capture/Compare value 3 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ccrv3(TIMER_TypeDef *timx, uint32_t CapCompValue)
{
    MODIFY_REG(timx->CCVAL3, TIMER_CCVAL3_CCRV3_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 3.
  * @param  timx TIMER instance
  * @retval Timer capture/compare value 3.
  */

__STATIC_INLINE uint32_t md_timer_get_ccrv3(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCVAL3, TIMER_CCVAL3_CCRV3_MSK) >> TIMER_CCVAL3_CCRV3_POSS);
}

/**
  * @brief  Timer CCVAL4 setup.
  * @param  timx TIMER instance
  * @param  value CCVAL4
  * @retval None
  */

__STATIC_INLINE void md_timer_set_capture_compare4_value_ccrv4(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->CCVAL4, value);
}

/**
  * @brief  GET Timer CCVAL4 register value.
  * @param  timx TIMER instance
  * @retval Timer CCVAL4 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_capture_compare4_value_ccrv4(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->CCVAL4));
}

/**
  * @brief  Timer capture/compare value 4 setup.
  * @param  timx TIMER instance
  * @param  CapCompValue Capture/Compare value 4 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ccrv4(TIMER_TypeDef *timx, uint32_t CapCompValue)
{
    MODIFY_REG(timx->CCVAL4, TIMER_CCVAL4_CCRV4_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 4.
  * @param  timx TIMER instance
  * @retval Timer capture/compare value 4.
  */

__STATIC_INLINE uint32_t md_timer_get_ccrv4(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->CCVAL4, TIMER_CCVAL4_CCRV4_MSK) >> TIMER_CCVAL4_CCRV4_POSS);
}

/**
  * @brief  Timer BDCFG setup.
  * @param  timx TIMER instance
  * @param  value (GOEN | AOEN | BRKP | BRKEN | OFFSSR | OFFSSI | LOCKLVL | DT)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_bdcfg(TIMER_TypeDef *timx, uint32_t value)
{
    WRITE_REG(timx->BDCFG, value);
}

/**
  * @brief  GET Timer BDCFG register value.
  * @param  timx TIMER instance
  * @retval Timer BDCFG register value.
  */

__STATIC_INLINE uint32_t md_timer_get_bdcfg(TIMER_TypeDef *timx)
{
    return (READ_REG(timx->BDCFG));
}

/**
  * @brief  Timer main output enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_main_output_goen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->BDCFG, TIMER_BDCFG_GOEN_MSK);
}

/**
  * @brief  Timer main output disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_main_output_goen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->BDCFG, TIMER_BDCFG_GOEN_MSK);
}

/**
  * @brief  Indicates whether the timer main output is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_main_output_goen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->BDCFG, TIMER_BDCFG_GOEN_MSK) == (TIMER_BDCFG_GOEN_MSK));
}

/**
  * @brief  Timer automatic output enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_auto_output_aoen(TIMER_TypeDef *timx)
{
    SET_BIT(timx->BDCFG, TIMER_BDCFG_AOEN_MSK);
}

/**
  * @brief  Timer automatic output disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_auto_output_aoen(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->BDCFG, TIMER_BDCFG_AOEN_MSK);
}

/**
  * @brief  Indicates whether the timer automatic output is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_auto_output_aoen(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->BDCFG, TIMER_BDCFG_AOEN_MSK) == (TIMER_BDCFG_AOEN_MSK));
}

/**
  * @brief  Timer break polarity setup.
  * @param  timx TIMER instance
  * @param  BreakPolarity Break polarity
  *         @arg @ref MD_TIMER_BREAKPOLARITY_LOW
  *         @arg @ref MD_TIMER_BREAKPOLARITY_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_break_polarity_brkp(TIMER_TypeDef *timx, uint32_t BreakPolarity)
{
    MODIFY_REG(timx->BDCFG, TIMER_BDCFG_BRKP_MSK, BreakPolarity);
}

/**
  * @brief  Get timer break polarity.
  * @param  timx TIMER instance
  * @retval Timer break polarity.
  */

__STATIC_INLINE uint32_t md_timer_get_break_polarity_brkp(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->BDCFG, TIMER_BDCFG_BRKP_MSK) >> TIMER_BDCFG_BRKP_POS);
}

/**
  * @brief  Timer break enable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_break_brken(TIMER_TypeDef *timx)
{
    SET_BIT(timx->BDCFG, TIMER_BDCFG_BRKEN_MSK);
}

/**
  * @brief  Timer break disable.
  * @param  timx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_break_brken(TIMER_TypeDef *timx)
{
    CLEAR_BIT(timx->BDCFG, TIMER_BDCFG_BRKEN_MSK);
}

/**
  * @brief  Indicates whether the timer break is enabled.
  * @param  timx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_break_brken(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->BDCFG, TIMER_BDCFG_BRKEN_MSK) == (TIMER_BDCFG_BRKEN_MSK));
}

/**
  * @brief  Timer off-state selection for run mode setup.
  * @param  timx TIMER instance
  * @param  OffStateRun Off-state selection for run mode
  *         @arg @ref MD_TIMER_OFFSTATERUN_DISABLE
  *         @arg @ref MD_TIMER_OFFSTATERUN_ENABLE
  * @retval None
  */

__STATIC_INLINE void md_timer_set_offstate_run_selection_offssr(TIMER_TypeDef *timx, uint32_t OffStateRun)
{
    MODIFY_REG(timx->BDCFG, TIMER_BDCFG_OFFSSR_MSK, OffStateRun);
}

/**
  * @brief  Get timer off-state selection for run mode.
  * @param  timx TIMER instance
  * @retval Timer off-state selection for run mode.
  */

__STATIC_INLINE uint32_t md_timer_get_offstate_run_selection_offssr(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->BDCFG, TIMER_BDCFG_OFFSSR_MSK) >> TIMER_BDCFG_OFFSSR_POS);
}

/**
  * @brief  Timer off-state selection for idle mode setup.
  * @param  timx TIMER instance
  * @param  OffStateIdle Off-state selection for idle mode
  *         @arg @ref MD_TIMER_OFFSTATEIDLE_DISABLE
  *         @arg @ref MD_TIMER_OFFSTATEIDLE_ENABLE
  * @retval None
  */

__STATIC_INLINE void md_timer_set_offstate_idle_selection_offssi(TIMER_TypeDef *timx, uint32_t OffStateIdle)
{
    MODIFY_REG(timx->BDCFG, TIMER_BDCFG_OFFSSI_MSK, OffStateIdle);
}

/**
  * @brief  Get timer off-state selection for idle mode.
  * @param  timx TIMER instance
  * @retval Timer off-state selection for idle mode.
  */

__STATIC_INLINE uint32_t md_timer_get_offstate_idle_selection_offssi(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->BDCFG, TIMER_BDCFG_OFFSSI_MSK) >> TIMER_BDCFG_OFFSSI_POS);
}

/**
  * @brief  Timer lock configuration setup.
  * @param  timx TIMER instance
  * @param  LockLevel Lock configuration
  *         @arg @ref MD_TIMER_LOCKLEVEL_0
  *         @arg @ref MD_TIMER_LOCKLEVEL_1
  *         @arg @ref MD_TIMER_LOCKLEVEL_2
  *         @arg @ref MD_TIMER_LOCKLEVEL_3
  * @retval None
  */

__STATIC_INLINE void md_timer_set_lock_config_locklvl(TIMER_TypeDef *timx, uint32_t LockLevel)
{
    MODIFY_REG(timx->BDCFG, TIMER_BDCFG_LOCKLVL_MSK, LockLevel);
}

/**
  * @brief  Get timer lock configuration.
  * @param  timx TIMER instance
  * @retval Timer lock configuration.
  */

__STATIC_INLINE uint32_t md_timer_get_lock_config_locklvl(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->BDCFG, TIMER_BDCFG_LOCKLVL_MSK) >> TIMER_BDCFG_LOCKLVL_POSS);
}

/**
  * @brief  Timer dead-time generator setup.
  * @param  timx TIMER instance
  * @param  DeadTime Dead-time generator (between Min_Data=0 and Max_Data=0xFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_dead_time_duration_dt(TIMER_TypeDef *timx, uint32_t DeadTime)
{
    MODIFY_REG(timx->BDCFG, TIMER_BDCFG_DT_MSK, DeadTime);
}

/**
  * @brief  Get timer dead-time generator.
  * @param  timx TIMER instance
  * @retval Timer dead-time generator.
  */

__STATIC_INLINE uint32_t md_timer_get_dead_time_duration_dt(TIMER_TypeDef *timx)
{
    return (READ_BIT(timx->BDCFG, TIMER_BDCFG_DT_MSK) >> TIMER_BDCFG_DT_POSS);
}

/**
  * @} MD_TIMER_Public_Macro
  */

/* Public functions -----------------------------------------------------------*/



/**
  * @} TIMER
  */


#endif

/**
  * @} Micro_Driver
  */


#ifdef __cplusplus
}
#endif

#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

