/**********************************************************************************
 *
 * @file    md_rcu.h
 * @brief   header file of md_rcu.c
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_RCU_H__
#define __MD_RCU_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include <stdbool.h>
#include "es32f0930.h"

/** @addtogroup Micro_Driver
  * @{
  */

#if defined (RCU)

/** @defgroup RCU RCU
  * @brief RCU micro driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/** @defgroup MD_RCU_Private_Constants RCU Private Constants
  * @{
  */

#define __HOSC      (8000000UL)
#define __HRC       (16000000UL)
#define __LRC       (  32000UL)

#define LRC         (1<<8)
#define HOSC        (1<<4)
#define HRC         (1<<0)
/**
  * @} MD_RCU_Private_Constants
  */
/* Private macros -------------------------------------------------------------*/
/* Public types ---------------------------------------------------------------*/
/* Public types ---------------------------------------------------------------*/
/** @defgroup MD_RCU_Public_Types RCU Public Init Type
  * @{
  */

typedef struct
{
    uint32_t  Mpre;    /*! MCO clock prescaler  */
    uint32_t  Msw;     /*! MCO clock output  */
    uint32_t  HrcDiv;  /*! HRC clock prescaler  */
    uint32_t  Ppre;    /*! APB clock prescaler  */
    uint32_t  Hpre;    /*! AHB clock prescaler  */
    uint32_t  Sw;      /*! system clock   */
    uint16_t  SysClock;/*! system clock enable  */
} md_rcu_init_typedef;

/**
  * @} MD_RCU_Public_Types
  */
/* Public constants -----------------------------------------------------------*/
/** @defgroup MD_RCU_Public_Constants RCU Public Constants
  * @{
  */


/** @defgroup MD_RCU_LRCRDY_FLAG LRC clock ready flag
  * @{
  */
#define MD_RCU_LRCRDY_NOT_READY   (0x00000000UL)  /** @brief LRC oscillator not ready */
#define MD_RCU_LRCRDY_READY       (0x00000001UL)  /** @brief LRC oscillator ready */
/**
  * @} MD_RCU_LRCRDY_FLAG
  */

/** @defgroup MD_RCU_HOSCRDY_FLAG HOSC clock ready flag
  * @{
  */
#define MD_RCU_HOSCRDY_NOT_READY   (0x00000000UL)  /** @brief HOSC oscillator not ready */
#define MD_RCU_HOSCRDY_READY       (0x00000001UL)  /** @brief HOSC oscillator ready */
/**
  * @} MD_RCU_HOSCRDY_FLAG
  */

/** @defgroup MD_RCU_HRCRDY_FLAG HRC clock ready flag
  * @{
  */
#define MD_RCU_HRCRDY_NOT_READY   (0x00000000UL)  /** @brief HRC oscillator not ready */
#define MD_RCU_HRCRDY_READY       (0x00000001UL)  /** @brief HRC oscillator ready */
/**
  * @} MD_RCU_HRCRDY_FLAG
  */

/** @defgroup MD_RCU_MCO_PRESCALE Microcontroller clock prescale
  * @{
  */
#define MD_RCU_MPRE_MCO_DIV_1      (0x00000000UL)  /** @brief MCO not divided */
#define MD_RCU_MPRE_MCO_DIV_2      (0x00000004UL)  /** @brief MCO divided by 2 */
#define MD_RCU_MPRE_MCO_DIV_4      (0x00000005UL)  /** @brief MCO divided by 4 */
#define MD_RCU_MPRE_MCO_DIV_8      (0x00000006UL)  /** @brief MCO divided by 8 */
#define MD_RCU_MPRE_MCO_DIV_16     (0x00000007UL)  /** @brief MCO divided by 16 */
/**
  * @} MD_RCU_MCO_PRESCALE
  */

/** @defgroup MD_RCU_MCO_OUTPUT Microcontroller clock output select
  * @{
  */
#define MD_RCU_MCO_DISABLE      (0x00000000UL)  /** @brief MCO output disabled, no clock on MCO */
#define MD_RCU_MCO_LRC          (0x00000001UL)  /** @brief Internal low speed (LRC) oscillator clock selected */
#define MD_RCU_MCO_HRC          (0x00000003UL)  /** @brief Internal high speed (HRC) oscillator clock selected */
#define MD_RCU_MCO_HOSC         (0x00000004UL)  /** @brief External high speed (HOSC) oscillator clock selected */
#define MD_RCU_MCO_SYSCLK       (0x00000007UL)  /** @brief System clock selected(SYSCLK) */
#define MD_RCU_MCO_HCLK         (0x00000008UL)  /** @brief AHB clock selected(HCLK) */
#define MD_RCU_MCO_PCLK         (0x00000009UL)  /** @brief APB clock selected(PCLK) */
#define MD_RCU_MCO_MMC          (0x0000000AUL)  /** @brief MMC clock selected(MMC) */
#define MD_RCU_MCO_LCD          (0x0000000BUL)  /** @brief LCD clock selected(LCD) */
#define MD_RCU_MCO_VPUMP        (0x0000000CUL)  /** @brief VPUMP clock selected(VPUMP) */
#define MD_RCU_MCO_VLCD         (0x0000000DUL)  /** @brief VLCD clock selected(VLCD) */
#define MD_RCU_MCO_EXCLK        (0x0000000EUL)  /** @brief EXCLK clock selected(EXCLK) */
/**
  * @} MD_RCU_MCO_OUTPUT
  */

/** @defgroup MD_RCU_HRC_PRESCALE HRC clock prescale
  * @{
  */
#define MD_RCU_HRCDIV_DIV_1      (0x00000000UL)  /** @brief HRC not divided */
#define MD_RCU_HRCDIV_DIV_2      (0x00000004UL)  /** @brief HRC divided by 2 */
#define MD_RCU_HRCDIV_DIV_4      (0x00000005UL)  /** @brief HRC divided by 4 */
#define MD_RCU_HRCDIV_DIV_8      (0x00000006UL)  /** @brief HRC divided by 8 */
#define MD_RCU_HRCDIV_DIV_16     (0x00000007UL)  /** @brief HRC divided by 16 */
/**
  * @} MD_RCU_HRC_PRESCALE
  */

/** @defgroup MD_RCU_APB_CLK_DIV APB clock prescaler(PCLK)
  * @{
  */
#define MD_RCU_PPRE_HCLK_DIV_1   (0x00000000UL)  /** @brief HCLK not divided */
#define MD_RCU_PPRE_HCLK_DIV_2   (0x00000004UL)  /** @brief HCLK divided by 2 */
#define MD_RCU_PPRE_HCLK_DIV_4   (0x00000005UL)  /** @brief HCLK divided by 4 */
#define MD_RCU_PPRE_HCLK_DIV_8   (0x00000006UL)  /** @brief HCLK divided by 8 */
#define MD_RCU_PPRE_HCLK_DIV_16  (0x00000007UL)  /** @brief HCLK divided by 16 */
/**
  * @} MD_RCU_APB_CLK_DIV
  */

/** @defgroup MD_RCU_AHB_CLK_DIV AHB clock prescaler(HCLK)
  * @{
  */
#define MD_RCU_HPRE_SYSCLK_DIV_1   (0x00000000UL)  /** @brief SYSCLK not divided */
#define MD_RCU_HPRE_SYSCLK_DIV_2   (0x00000008UL)  /** @brief SYSCLK divided by 2 */
#define MD_RCU_HPRE_SYSCLK_DIV_4   (0x00000009UL)  /** @brief SYSCLK divided by 4 */
#define MD_RCU_HPRE_SYSCLK_DIV_8   (0x0000000AUL)  /** @brief SYSCLK divided by 8 */
#define MD_RCU_HPRE_SYSCLK_DIV_16  (0x0000000BUL)  /** @brief SYSCLK divided by 16 */
/**
  * @} MD_RCU_AHB_CLK_DIV
  */

/** @defgroup MD_RCU_SYS_CLK_SW System clock switch(SYSCLK)
  * @{
  */
#define MD_RCU_SYSCLK_HRC   (0x00000000UL)  /** @brief HRC selected as system clock */
#define MD_RCU_SYSCLK_HOSC  (0x00000001UL)  /** @brief HOSC selected as system clock */
#define MD_RCU_SYSCLK_LRC   (0x00000002UL)  /** @brief LRC selected as system clock */
/**
  * @} MD_RCU_SYS_CLK_SW
  */

/** @defgroup MD_RCU_IPCFG_LCDSW System clock switch(LCD)
  * @{
  */
#define MD_RCU_IPCFG_LCDSW_DSIABLE   (0x00000000UL)  /** @brief DISABLE LCD clock */
#define MD_RCU_IPCFG_LCDSW_PCLK      (0x00000001UL)  /** @brief APB selected as LCD clock */
#define MD_RCU_IPCFG_LCDSW_LRC       (0x00000002UL)  /** @brief LRC selected as LCD clock */
/**
  * @} MD_RCU_IPCFG_LCDSW
  */

/** @defgroup MD_RCU_IPCFG_VLCDCPSW System clock switch(LCD)
  * @{
  */
#define MD_RCU_IPCFG_VLCDCPSW_HRC   (0x00000000UL)  /** @brief VLCDCP = HRC */
#define MD_RCU_IPCFG_VLCDCPSW_HOSC  (0x00000001UL)  /** @brief VLCDCP = HOSC */
/**
  * @} MD_RCU_IPCFG_VLCDCPSW
  */


/**
  * @} MD_RCU_Public_Constants
  */

/* Public macro ---------------------------------------------------------------*/
/** @defgroup MD_RCU_Public_Macros RCU Public Macros
  * @{
  */

/**
  * @brief  Clock security system enable(CSS)
  * @note   Set and cleared by software to enable the clock security system.
  *         When CSSON is set, the clock detector is enabled by hardware when the HOSC oscillator is ready,
  *         and disabled by hardware if a HOSC clock failure is detected.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hosc_css(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_CSSHOSCON_MSK);
}
/**
  * @brief  Clock security system disable(CSS)
  * @note   Set and cleared by software to enable the clock security system.
  *         When CSSON is set, the clock detector is enabled by hardware when the HOSC oscillator is ready,
  *         and disabled by hardware if a HOSC clock failure is detected.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hosc_css(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_CSSHOSCON_MSK);
}
/**
  * @brief  Get Clock security system (CSS)
  * @note   Set and cleared by software to enable the clock security system.
  *         When CSSON is set, the clock detector is enabled by hardware when the HOSC oscillator is ready,
  *         and disabled by hardware if a HOSC clock failure is detected.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hosc_css(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_CSSHOSCON_MSK) >> RCU_CON_CSSHOSCON_POS);
}

/**
  * @brief  Get LRCRDY clock ready flag
  * @note   This bit is set by hardware to indicate that the LRC oscillator is stable.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_LRCRDY_NOT_READY
  *         @arg @ref MD_RCU_LRCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_LRCRDY_MSK) >> RCU_CON_LRCRDY_POS);
}

/**
  * @brief  Internal low speed oscillator clock enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_lrc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_LRCON_MSK);
}
/**
  * @brief  Internal low speed oscillator clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_lrc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_LRCON_MSK);
}
/**
  * @brief  Get Internal low speed oscillator clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_lrc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_LRCON_MSK) >> RCU_CON_LRCON_POS);
}

/**
  * @brief  External high speed oscillator clock Filter bypass enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hoscfiltbpy(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HOSCFILTBPY_MSK);
}
/**
  * @brief  External high speed oscillator clock Filter bypass disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hoscfiltbpy(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HOSCFILTBPY_MSK);
}
/**
  * @brief  Get External high speed oscillator clock Filter bypass bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hoscfiltbpy(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCFILTBPY_MSK) >> RCU_CON_HOSCFILTBPY_POS);
}

/**
  * @brief  External high speed oscillator clock bypass enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hosc_bypass(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HOSCBPY_MSK);
}
/**
  * @brief  External high speed oscillator clock bypass disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hosc_bypass(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HOSCBPY_MSK);
}
/**
  * @brief  Get External high speed oscillator clock bypass bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hosc_bypass(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCBPY_MSK) >> RCU_CON_HOSCBPY_POS);
}

/**
  * @brief  Get HOSCRDY clock ready flag
  * @note   This bit is set by hardware to indicate that the HOSC oscillator is stable.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HOSCRDY_NOT_READY
  *         @arg @ref MD_RCU_HOSCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCRDY_MSK) >> RCU_CON_HOSCRDY_POS);
}

/**
  * @brief  External high speed oscillator clock enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HOSCON_MSK);
}
/**
  * @brief  External high speed oscillator clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hosc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HOSCON_MSK);
}
/**
  * @brief  Get External high speed oscillator clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCON_MSK) >> RCU_CON_HOSCON_POS);
}

/**
  * @brief  Get HRCRDY clock ready flag
  * @note   This bit is set by hardware to indicate that the HRC oscillator is stable.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HRCRDY_NOT_READY
  *         @arg @ref MD_RCU_HRCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HRCRDY_MSK) >> RCU_CON_HRCRDY_POS);
}

/**
  * @brief  Internal high speed oscillator clock enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hrc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HRCON_MSK);
}
/**
  * @brief  Internal high speed oscillator clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hrc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HRCON_MSK);
}
/**
  * @brief  Get Internal high speed oscillator clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hrc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HRCON_MSK) >> RCU_CON_HRCON_POS);
}

/** @defgroup MD_RCU_CON_CSSHOSCON CSSHOSCON
  * @brief  Set Clock security system bit for md_rcu_set_con() function used
  * @param  csshoscon This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_con_csshoscon_fun(csshoscon)  (csshoscon<<RCU_CON_CSSON_POS)
/**
  * @} MD_RCU_CON_CSSHOSCON
  */

/** @defgroup MD_RCU_CON_LRCON LRCON
  * @brief  Set Internal low speed oscillator clock bit for md_rcu_set_con() function used
  * @param  lrcon This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_con_lrcon_fun(lrcon)  (lrcon<<RCU_CON_LRCON_POS)
/**
  * @} MD_RCU_CON_LRCON
  */
/** @defgroup MD_RCU_CON_HOSCON HOSCON
  * @brief  Set External high speed oscillator clock bit for md_rcu_set_con() function used
  * @param  hoscon This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_con_hoscon_fun(hoscon)  (hoscon<<RCU_CON_HOSCON_POS)
/**
  * @} MD_RCU_CON_HOSCON
  */
/** @defgroup MD_RCU_CON_HRCON HRCON
  * @brief  Set Internal high speed oscillator clock bit for md_rcu_set_con() function used
  * @param  hrcon This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_con_hrcon_fun(hrcon)  (hrcon<<RCU_CON_HRCON_POS)
/**
  * @} MD_RCU_CON_HRCON
  */
/**
  * @brief  Set Clock Control Register
  * @param  rcu RCU Instance
  * @param  csson This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_con_csson_fun(csson)
  * @param  pll1on This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_con_pll1on_fun(pll1on)
  * @param  pll0on This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_con_pll0on_fun(pll0on)
  * @param  loscon This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_con_loscon_fun(loscon)
  * @param  lrcon This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_con_lrcon_fun(lrcon)
  * @param  hoscon This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_con_hoscon_fun(hoscon)
  * @param  hrcon This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_con_hrcon_fun(hrcon)
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_con(RCU_TypeDef *rcu, uint32_t csson, uint32_t lrcon, uint32_t hoscon, uint32_t hrcon)
{
    WRITE_REG(rcu->CON, (csson | lrcon | hoscon | hrcon));
}

/**
  * @brief  Set Microcontroller clock output prescaler
  * @note   Output Frequency = MCO/(MPRE+1)
  * @param  rcu RCU Instance
  * @param  mpre This parameter can be one of the following values:
  *         @arg Max Value 7
  *         @arg Min Value 0
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_mco_div(RCU_TypeDef *rcu, uint32_t mpre)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_MPRE_MSK, (mpre << RCU_CFG_MPRE_POSS));
}
/**
  * @brief  Get Microcontroller clock output prescaler
  * @note   Output Frequency = MCO/(MPRE+1)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 7
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_rcu_get_mco_div(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CFG, RCU_CFG_MPRE_MSK) >> RCU_CFG_MPRE_POSS);
}
/**
  * @brief  Set Microcontroller clock output (MCO)
  * @param  rcu RCU Instance
  * @param  msw This parameter can be one of the following values:
  *         @arg @ref MD_RCU_MCO_DISABLE
  *         @arg @ref MD_RCU_MCO_LRC
  *         @arg @ref MD_RCU_MCO_HRC
  *         @arg @ref MD_RCU_MCO_HOSC
  *         @arg @ref MD_RCU_MCO_SYSCLK
  *         @arg @ref MD_RCU_MCO_HCLK
  *         @arg @ref MD_RCU_MCO_PCLK
  *         @arg @ref MD_RCU_MCO_MMC
  *         @arg @ref MD_RCU_MCO_LCD
  *         @arg @ref MD_RCU_MCO_VPUMP
  *         @arg @ref MD_RCU_MCO_VLCD
  *         @arg @ref MD_RCU_MCO_EXCLK
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_mco_source(RCU_TypeDef *rcu, uint32_t msw)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_MSW_MSK, (msw << RCU_CFG_MSW_POSS));
}
/**
  * @brief  Get Microcontroller clock output (MCO)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_MCO_DISABLE
  *         @arg @ref MD_RCU_MCO_LRC
  *         @arg @ref MD_RCU_MCO_HRC
  *         @arg @ref MD_RCU_MCO_HOSC
  *         @arg @ref MD_RCU_MCO_SYSCLK
  *         @arg @ref MD_RCU_MCO_HCLK
  *         @arg @ref MD_RCU_MCO_PCLK
  *         @arg @ref MD_RCU_MCO_MMC
  *         @arg @ref MD_RCU_MCO_LCD
  *         @arg @ref MD_RCU_MCO_VPUMP
  *         @arg @ref MD_RCU_MCO_VLCD
  *         @arg @ref MD_RCU_MCO_EXCLK
  */
__STATIC_INLINE uint32_t md_rcu_get_mco_source(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CFG, RCU_CFG_MSW_MSK) >> RCU_CFG_MSW_POSS);
}
/**
  * @brief  Set HRC clock division factor for HRC reference clock
  * @note   These bits are set and cleared by software to select HRCDIV division factor.
  * @note
  * @param  rcu RCU Instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_cfg_hrcdiv(RCU_TypeDef *rcu, uint32_t hrcdiv)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_HRCDIV_MSK, (hrcdiv << RCU_CFG_HRCDIV_POSS));
}
/**
  * @brief  Get HRC clock division factor for HRC reference clock
  * @note   These bits are set and cleared by software to select HRCDIV division factor.
  * @note
  * @param  rcu RCU Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_rcu_get_cfg_hrcdiv(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CFG, RCU_CFG_HRCDIV_MSK) >> RCU_CFG_HRCDIV_POSS);
}
/**
  * @brief  Set APB clock prescaler(PCLK)
  * @param  rcu RCU Instance
  * @param  ppre The retval can be one of the following values:
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_1
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_2
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_4
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_8
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_16
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_pclk_div(RCU_TypeDef *rcu, uint32_t ppre)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_PPRE_MSK, (ppre << RCU_CFG_PPRE_POSS));
}
/**
  * @brief  Get APB clock prescaler(PCLK)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_1
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_2
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_4
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_8
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_16
  */
__STATIC_INLINE uint32_t md_rcu_get_pclk_div(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CFG, RCU_CFG_PPRE_MSK) >> RCU_CFG_PPRE_POSS);
}
/**
  * @brief  Set AHB clock prescaler(HCLK)
  * @param  rcu RCU Instance
  * @param  hpre can be one of the following values:
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_1
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_2
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_4
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_8
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_16
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hclk_div(RCU_TypeDef *rcu, uint32_t hpre)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_HPRE_MSK, (hpre << RCU_CFG_HPRE_POSS));
}
/**
  * @brief  Get AHB clock prescaler(HCLK)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_1
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_2
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_4
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_8
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_16
  */
__STATIC_INLINE uint32_t md_rcu_get_hclk_div(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CFG, RCU_CFG_HPRE_MSK) >> RCU_CFG_HPRE_POSS);
}

/**
  * @brief  Get System clock switch(SYSCLK) status
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_SYSCLK_HRC
  *         @arg @ref MD_RCU_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SYSCLK_LRC
  */
__STATIC_INLINE uint32_t md_rcu_get_current_system_clock(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CFG, RCU_CFG_SWS_MSK) >> RCU_CFG_SWS_POSS);
}

/**
  * @brief  Set System clock switch(SYSCLK)
  * @param  rcu RCU Instance
  * @param  sw can be one of the following values:
  *         @arg @ref MD_RCU_SYSCLK_HRC
  *         @arg @ref MD_RCU_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SYSCLK_LRC
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_system_clock_source(RCU_TypeDef *rcu, uint32_t sw)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_SW_MSK, (sw << RCU_CFG_SW_POSS));
}
/**
  * @brief  Get System clock switch(SYSCLK)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_SYSCLK_HRC
  *         @arg @ref MD_RCU_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SYSCLK_LRC
  */
__STATIC_INLINE uint32_t md_rcu_get_system_clock_source(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CFG, RCU_CFG_SW_MSK) >> RCU_CFG_SW_POSS);
}

/** @defgroup MD_RCU_CFG_MPRE MPRE
  * @brief  Set Microcontroller clock output prescaler bit for md_rcu_set_cfg() function used
  * @param  mpre This parameter can be one of the following values:
  *         @arg Max Value 7
  *         @arg Min Value 0
  * @{
  */
#define md_rcu_set_mco_div_fun(mpre)  (mpre<<RCU_CFG_MPRE_POSS)
/**
  * @} MD_RCU_CFG_MPRE
  */
/** @defgroup MD_RCU_CFG_MSW MSW
  * @brief  Set Microcontroller clock output (MCO) bit for md_rcu_set_cfg() function used
  * @param  msw This parameter can be one of the following values:
  *         @arg @ref MD_RCU_MCO_DISABLE
  *         @arg @ref MD_RCU_MCO_LRC
  *         @arg @ref MD_RCU_MCO_HRC
  *         @arg @ref MD_RCU_MCO_HOSC
  *         @arg @ref MD_RCU_MCO_SYSCLK
  *         @arg @ref MD_RCU_MCO_HCLK
  *         @arg @ref MD_RCU_MCO_PCLK
  *         @arg @ref MD_RCU_MCO_MMC
  *         @arg @ref MD_RCU_MCO_LCD
  *         @arg @ref MD_RCU_MCO_VPUMP
  *         @arg @ref MD_RCU_MCO_VLCD
  *         @arg @ref MD_RCU_MCO_EXCLK
  * @{
  */
#define md_rcu_set_mco_source_fun(msw)  (msw<<RCU_CFG_MSW_POSS)
/**
  * @} MD_RCU_CFG_MSW
  */

/** @defgroup MD_RCU_CFG_PPRE PPRE
  * @brief  Set APB clock prescaler(PCLK) bit for md_rcu_set_cfg() function used
  * @param  ppre This parameter can be one of the following values:
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_1
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_2
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_4
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_8
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_16
  * @{
  */
#define md_rcu_set_pclk_div_fun(ppre)  (ppre<<RCU_CFG_PPRE_POSS)
/**
  * @} MD_RCU_CFG_PPRE
  */
/** @defgroup MD_RCU_CFG_HPRE HPRE
  * @brief  Set AHB clock prescaler(HCLK) bit for md_rcu_set_cfg() function used
  * @param  hpre This parameter can be one of the following values:
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_1
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_2
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_4
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_8
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_16
  * @{
  */
#define md_rcu_set_hclk_div_fun(hpre)  (hpre<<RCU_CFG_HPRE_POSS)
/**
  * @} MD_RCU_CFG_HPRE
  */
/** @defgroup MD_RCU_CFG_SW SW
  * @brief  Set System clock switch(SYSCLK) bit for md_rcu_set_cfg() function used
  * @param  sw This parameter can be one of the following values:
  *         @arg @ref MD_RCU_SYSCLK_HRC
  *         @arg @ref MD_RCU_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SYSCLK_LRC
  * @{
  */
#define md_rcu_set_system_clock_source_fun(sw)  (sw<<RCU_CFG_SW_POSS)
/**
  * @} MD_RCU_CFG_SW
  */
/**
  * @brief  Set Clock Configuration Register
  * @param  rcu RCU Instance
  * @param  mpre This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_mco_div_fun(mpre)
  * @param  msw This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_mco_source_fun(msw)
  * @param  ppre This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_pclk_div_fun(ppre)
  * @param  hpre This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_hclk_div_fun(hpre)
  * @param  sw This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_system_clock_source_fun(sw)
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_cfg(RCU_TypeDef *rcu, uint32_t mpre,    uint32_t msw, uint32_t ppre, uint32_t hpre,   uint32_t sw)
{
    WRITE_REG(rcu->CFG, (mpre | msw | ppre | hpre | sw));
}

/**
  * @brief  Set LCD pre clock config
  * @param  rcu RCU Instance
  * @param  lcddiv can be one of the following values:
  *         @arg @ref 0
  *         @arg @ref 16(n+1)
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_lcddiv(RCU_TypeDef *rcu, uint32_t lcddiv)
{
    MODIFY_REG(rcu->IPCFG, RCU_IPCFG_LCDDIV_MSK, (lcddiv << RCU_IPCFG_LCDDIV_POSS));
}
/**
  * @brief  Get LCD pre clock config
  * @param  rcu RCU Instance
  * @param  lcddiv can be one of the following values:
  *         @arg @ref 0
  *         @arg @ref 16(n+1)
  */
__STATIC_INLINE uint32_t md_rcu_get_ipcfg_lcddiv(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_LCDDIV_MSK) >> RCU_IPCFG_LCDDIV_POSS);
}
/**
  * @brief  Set LCD clock switch
  * @param  rcu RCU Instance
  * @param  lcdsw can be one of the following values:
  *         @arg @ref 0: Disable LCD clock
  *         @arg @ref 1: PCLK
  *         @arg @ref 2: LCR
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_lcdsw(RCU_TypeDef *rcu, uint32_t lcdsw)
{
    MODIFY_REG(rcu->IPCFG, RCU_IPCFG_LCDSW_MSK, (lcdsw << RCU_IPCFG_LCDSW_POSS));
}
/**
  * @brief  Get LCD clock switch
  * @param  rcu RCU Instance
  * @param  lcdsw can be one of the following values:
  *         @arg @ref 0: Disable LCD clock
  *         @arg @ref 1: PCLK
  *         @arg @ref 2: LCR
  */
__STATIC_INLINE uint32_t md_rcu_get_ipcfg_lcdsw(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_LCDSW_MSK) >> RCU_IPCFG_LCDSW_POSS);
}
/**
  * @brief  Set HOSC for MMC clock
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_mmcsw_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IPCFG, RCU_IPCFG_MMCSW_MSK);
}
/**
  * @brief  Set HRC for MMC clock
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_mmcsw_hrc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->IPCFG, RCU_IPCFG_MMCSW_MSK);
}

/**
  * @brief  Get MMC clock
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_ipcfg_mmcsw(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_MMCSW_MSK) >> RCU_IPCFG_MMCSW_POS);
}
/**
  * @brief  Set MMC clock enable
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ipcfg_mmcen(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IPCFG, RCU_IPCFG_MMCEN_MSK);
}
/**
  * @brief  Set MMC clock disable
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ipcfg_mmcen(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->IPCFG, RCU_IPCFG_MMCEN_MSK);
}

/**
  * @brief  Get MMC clock enable or disable
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ipcfg_mmcen(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_MMCEN_MSK) >> RCU_IPCFG_MMCEN_POS);
}
/**
  * @brief  Set MMC pre clock
  * @param  rcu RCU Instance
  * @param  mmcdiv can be one of the following values:
  *         @arg @ref 0: no prescale MMC clock
  *         @arg @ref 1: 2 prescale MMC clock
  *         @arg @ref 2: 4 prescale MMC clock
  *         @arg @ref 3: 8 prescale MMC clock
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_mmcdiv(RCU_TypeDef *rcu, uint32_t mmcdiv)
{
    MODIFY_REG(rcu->IPCFG, RCU_IPCFG_MMCDIV_MSK, (mmcdiv << RCU_IPCFG_MMCDIV_POSS));
}
/**
  * @brief  Get MMC pre clock
  * @param  rcu RCU Instance
  * @param  mmcdiv can be one of the following values:
  *         @arg @ref 0: no prescale MMC clock
  *         @arg @ref 1: 2 prescale MMC clock
  *         @arg @ref 2: 4 prescale MMC clock
  *         @arg @ref 3: 8 prescale MMC clock
  */
__STATIC_INLINE uint32_t md_rcu_get_ipcfg_mmcdiv(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_MMCDIV_MSK) >> RCU_IPCFG_MMCDIV_POSS);
}
/**
  * @brief  Set HRC for VLCD clock
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_vlcdsw_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IPCFG, RCU_IPCFG_VLCDSW_MSK);
}
/**
  * @brief  Set HOSC for VLCD clock
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_vlcdsw_hrc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->IPCFG, RCU_IPCFG_VLCDSW_MSK);
}

/**
  * @brief  Get VLCD clock
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_ipcfg_vlcdsw(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_VLCDSW_MSK) >> RCU_IPCFG_VLCDSW_POS);
}
/**
  * @brief  Set VLCD clock enable
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ipcfg_vlcden(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IPCFG, RCU_IPCFG_VLCDEN_MSK);
}
/**
  * @brief  Set VLCD clock disable
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ipcfg_vlcden(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->IPCFG, RCU_IPCFG_VLCDEN_MSK);
}
/**
  * @brief  Get VLCD clock enable or disable
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ipcfg_vlcden(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_VLCDEN_MSK) >> RCU_IPCFG_VLCDEN_POS);
}
/**
  * @brief  Set VLCDCP pre clock
  * @param  rcu RCU Instance
  * @param  vlcdcpdiv can be one of the following values:
  *         @arg @ref 0 : Bypass
  *         @arg @ref n : Divider values is = n*2, Min.1 Max.63
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_vlcddiv(RCU_TypeDef *rcu, uint32_t vlcdcpdiv)
{
    MODIFY_REG(rcu->IPCFG, RCU_IPCFG_VLCDDIV_MSK, (vlcdcpdiv << RCU_IPCFG_VLCDDIV_POSS));
}
/**
  * @brief  Get VLCDCP pre clock
  * @param  rcu RCU Instance
  * @param  vlcdcpdiv can be one of the following values:
  *         @arg @ref 0 : Bypass
  *         @arg @ref n : Divider values is = n*2, Min.1 Max.63
  */
__STATIC_INLINE uint32_t md_rcu_get_ipcfg_vlcddiv(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_VLCDDIV_MSK) >> RCU_IPCFG_VLCDDIV_POSS);
}
/**
  * @brief  Set HRC for VPUMP clock
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_vpumpsw_hrc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->IPCFG, RCU_IPCFG_VPUMPSW_MSK);
}
/**
  * @brief  Set HOSC for VPUMP clock
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_vpumpsw_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IPCFG, RCU_IPCFG_VPUMPSW_MSK);
}

/**
  * @brief  Get VPUMP clock
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_ipcfg_vpumpsw(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_VPUMPSW_MSK) >> RCU_IPCFG_VPUMPSW_POS);
}

/**
  * @brief  Set VPUMP clock enable
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ipcfg_vpumpen(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IPCFG, RCU_IPCFG_VPUMPEN_MSK);
}
/**
  * @brief  Set VPUMP clock disable
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ipcfg_vpumpen(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->IPCFG, RCU_IPCFG_VPUMPEN_MSK);
}
/**
  * @brief  Get VPUMP clock enable or disable
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ipcfg_vpumpen(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_VPUMPEN_MSK) >> RCU_IPCFG_VPUMPEN_POS);
}
/**
  * @brief  Set VPUMP pre clock
  * @param  rcu RCU Instance
  * @param  vggcpdiv can be one of the following values:
  *         @arg @ref 0 : Bypass
  *         @arg @ref n : Divider values is = n*2, Min.1 Max.63
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ipcfg_vpumpdiv(RCU_TypeDef *rcu, uint32_t vpumpdiv)
{
    MODIFY_REG(rcu->IPCFG, RCU_IPCFG_VPUMPDIV_MSK, (vpumpdiv << RCU_IPCFG_VPUMPDIV_POSS));
}
/**
  * @brief  Get VPUMP pre clock
  * @param  rcu RCU Instance
  * @param  vggcpdiv can be one of the following values:
  *         @arg @ref 0 : Bypass
  *         @arg @ref n : Divider values is = n*2, Min.1 Max.63
  */
__STATIC_INLINE uint32_t md_rcu_get_ipcfg_vpumpdiv(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IPCFG, RCU_IPCFG_VPUMPDIV_MSK) >> RCU_IPCFG_VPUMPDIV_POSS);
}
/**
  * @brief  Clock security system interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_css_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_CSSHOSC_MSK);
}
/**
  * @brief  HOSC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_hosc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_HOSCRDY_MSK);
}
/**
  * @brief  HRC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_hrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_HRCRDY_MSK);
}
/**
  * @brief  LRC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_lrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_LRCRDY_MSK);
}
/**
  * @brief  Clock security system interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_css_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_CSSHOSC_MSK);
}
/**
  * @brief  HOSC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_hosc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_HOSCRDY_MSK);
}
/**
  * @brief  HRC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_hrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_HRCRDY_MSK);
}
/**
  * @brief  LRC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_lrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_LRCRDY_MSK);
}
/**
  * @brief  Clock security system interrupt active state
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_css_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_CSSHOSC_MSK) == (RCU_IVS_CSSHOSC_MSK));
}
/**
  * @brief  HOSC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_HOSCRDY_MSK) == (RCU_IVS_HOSCRDY_MSK));
}
/**
  * @brief  HRC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_HRCRDY_MSK) == (RCU_IVS_HRCRDY_MSK));
}
/**
  * @brief  LRC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_LRCRDY_MSK) == (RCU_IVS_LRCRDY_MSK));
}
/**
  * @brief  Clock security system unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_css_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_CSSHOSC_MSK) == (RCU_RIF_CSSHOSC_MSK));
}
/**
  * @brief  HOSC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_HOSCRDY_MSK) == (RCU_RIF_HOSCRDY_MSK));
}
/**
  * @brief  HRC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_HRCRDY_MSK) == (RCU_RIF_HRCRDY_MSK));
}
/**
  * @brief  LRC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_LRCRDY_MSK) == (RCU_RIF_LRCRDY_MSK));
}
/**
  * @brief  Clock security system interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_css_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_CSSHOSC_MSK) == (RCU_IFM_CSSHOSC_MSK));
}
/**
  * @brief  HOSC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_HOSCRDY_MSK) == (RCU_IFM_HOSCRDY_MSK));
}
/**
  * @brief  HRC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_HRCRDY_MSK) == (RCU_IFM_HRCRDY_MSK));
}
/**
  * @brief  LRC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_LRCRDY_MSK) == (RCU_IFM_LRCRDY_MSK));
}

/**
  * @brief  Clock security system interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_css_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_CSSHOSC_MSK);
}
/**
  * @brief  HOSC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_hosc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_HOSCRDY_MSK);
}
/**
  * @brief  HRC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_hrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_HRCRDY_MSK);
}
/**
  * @brief  LRC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_lrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_LRCRDY_MSK);
}

/**
  * @brief  BOR reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_bor(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTAT, RCU_RSTAT_BORRSTF_MSK) == (RCU_RSTAT_BORRSTF_MSK));
}
/**
  * @brief  LOW POWER reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_rstat_lprstf(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTAT, RCU_RSTAT_LPRSTF_MSK) == (RCU_RSTAT_LPRSTF_MSK));
}
/**
  * @brief  WWDT reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_wwdt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTAT, RCU_RSTAT_WWDTRSTF_MSK) == (RCU_RSTAT_WWDTRSTF_MSK));
}
/**
  * @brief  IWDT reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_iwdt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTAT, RCU_RSTAT_IWDTRSTF_MSK) == (RCU_RSTAT_IWDTRSTF_MSK));
}
/**
  * @brief  SW reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_software(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTAT, RCU_RSTAT_SWRSTF_MSK) == (RCU_RSTAT_SWRSTF_MSK));
}
/**
  * @brief  OBL reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_rstat_oblrstf(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTAT, RCU_RSTAT_OBLRSTF_MSK) == (RCU_RSTAT_OBLRSTF_MSK));
}
/**
  * @brief  NRST reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_nrst(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTAT, RCU_RSTAT_NRSTF_MSK) == (RCU_RSTAT_NRSTF_MSK));
}
/**
  * @brief  POR/PDR reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_por(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTAT, RCU_RSTAT_PORRSTF_MSK) == (RCU_RSTAT_PORRSTF_MSK));
}
/**
  * @brief  Clear flag reset
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_flag_reset_flag(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->RSTAT, RCU_RSTAT_CLRFLG_MSK);
}

/**
  * @brief  GPIOD reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiod_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_GPDEN_MSK);
}
/**
  * @brief  GPIOD reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiod_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_GPDEN_MSK);
}
/**
  * @brief  Get GPIOD reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiod_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_GPDEN_MSK) >> RCU_AHBRST_GPDEN_POS);
}
/**
  * @brief  GPIOC reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioc_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_GPCEN_MSK);
}
/**
  * @brief  GPIOC reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioc_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_GPCEN_MSK);
}
/**
  * @brief  Get GPIOC reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioc_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_GPCEN_MSK) >> RCU_AHBRST_GPCEN_POS);
}
/**
  * @brief  HDIV reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hdiven_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_HDIVEN_MSK);
}
/**
  * @brief  HDIV reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hdiven_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_HDIVEN_MSK);
}
/**
  * @brief  Get HDIV reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hdiven_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_HDIVEN_MSK) >> RCU_AHBRST_HDIVEN_POS);
}

/** @defgroup MD_RCU_AHBRST_GPDEN GPDEN
  * @brief  Set GPIOD reset request bit for md_rcu_set_ahbrst() function used
  * @param  gpden This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_ahbrst_gpden_fun(gpden)  (gpden<<RCU_AHBRST_GPDEN_POS)
/**
  * @} MD_RCU_AHBRST_GPDEN
  */
/** @defgroup MD_RCU_AHBRST_GPCEN GPCEN
  * @brief  Set GPIOC reset request bit for md_rcu_set_ahbrst() function used
  * @param  gpcen This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_ahbrst_gpcen_fun(gpcen)  (gpcen<<RCU_AHBRST_GPCEN_POS)
/**
  * @} MD_RCU_AHBRST_GPCEN
  */
/** @defgroup MD_RCU_AHBRST_HDIVEN HDIVEN
  * @brief  Set HDIV reset request bit for md_rcu_set_ahbrst() function used
  * @param  hdiven This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_ahbrst_hdiven_fun(hdiven)  (hdiven<<RCU_AHBRST_HDIVEN_POS)
/**
  * @} MD_RCU_AHBRST_HDIVEN
  */
/**
  * @brief  Set Peripheral Special Clock Control Register
  * @param  rcu RCU Instance
  * @param  gpden This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_ahbrst_gpden_fun(gpden)
  * @param  gpcen This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_ahbrst_gpcen_fun(gpcen)
  * @param  hdiven This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_ahbrst_hdiven_fun(hdiven)
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ahbrst(RCU_TypeDef *rcu, uint32_t gpden,  uint32_t gpcen,   uint32_t hdiven)
{
    WRITE_REG(rcu->AHBRST, (gpden | gpcen | hdiven));
}

/**
  * @brief  I2C1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_i2c1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_I2C1EN_MSK);
}
/**
  * @brief  I2C1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_i2c1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_I2C1EN_MSK);
}
/**
  * @brief  Get I2C1 reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_i2c1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_I2C1EN_MSK) >> RCU_APB1RST_I2C1EN_POS);
}
/**
  * @brief  SUART1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1rst_suart1en_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_SUART1EN_MSK);
}
/**
  * @brief  SUART1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1rst_suart1en_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_SUART1EN_MSK);
}
/**
  * @brief  Get SUART1 reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1rst_suart1en_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_SUART1EN_MSK) >> RCU_APB1RST_SUART1EN_POS);
}
/**
  * @brief  IWDT reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1rst_iwdten_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_IWDTEN_MSK);
}
/**
  * @brief  IWDT reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1rst_iwdten_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_IWDTEN_MSK);
}
/**
  * @brief  Get IWDT reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1rst_iwdten_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_IWDTEN_MSK) >> RCU_APB1RST_IWDTEN_POS);
}
/**
  * @brief  WWDT reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_wwdt_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_WWDTEN_MSK);
}
/**
  * @brief  WWDT reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_wwdt_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_WWDTEN_MSK);
}
/**
  * @brief  Get WWDT reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_wwdt_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_WWDTEN_MSK) >> RCU_APB1RST_WWDTEN_POS);
}
/**
  * @brief  BS16T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_bs16t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_BS16T1EN_MSK);
}
/**
  * @brief  BS16T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_bs16t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_BS16T1EN_MSK);
}
/**
  * @brief  Get BS16T1 reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_bs16t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_BS16T1EN_MSK) >> RCU_APB1RST_BS16T1EN_POS);
}
/**
  * @brief  GP32C4T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T1EN_MSK);
}
/**
  * @brief  GP32C4T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T1EN_MSK);
}
/**
  * @brief  Get GP32C4T1 reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T1EN_MSK) >> RCU_APB1RST_GP32C4T1EN_POS);
}

/** @defgroup MD_RCU_APB1RST_I2C1EN I2C1EN
  * @brief  Set I2C1 reset request bit for md_rcu_set_apb1rst() function used
  * @param  i2c1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1rst_i2c1en_fun(i2c1en)  (i2c1en<<RCU_APB1RST_I2C1EN_POS)
/**
  * @} MD_RCU_APB1RST_I2C1EN
  */
/** @defgroup MD_RCU_APB1RST_SUART1EN SUART1EN
  * @brief  Set SUART1 reset request bit for md_rcu_set_apb1rst() function used
  * @param  suart1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1rst_suart1en_fun(suart1en)  (suart1en<<RCU_APB1RST_SUART1EN_POS)
/**
  * @} MD_RCU_APB1RST_SUART1EN
  */
/** @defgroup MD_RCU_APB1RST_IWDTEN IWDTEN
  * @brief  Set IWDT reset request bit for md_rcu_set_apb1rst() function used
  * @param  iwdten This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1rst_iwdten_fun(iwdten)  (iwdten<<RCU_APB1RST_IWDTEN_POS)
/**
  * @} MD_RCU_APB1RST_IWDTEN
  */
/** @defgroup MD_RCU_APB1RST_WWDTEN WWDTEN
  * @brief  Set WWDT reset request bit for md_rcu_set_apb1rst() function used
  * @param  wwdten This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1rst_wwdten_fun(wwdten)  (wwdten<<RCU_APB1RST_WWDTEN_POS)
/**
  * @} MD_RCU_APB1RST_WWDTEN
  */
/** @defgroup MD_RCU_APB1RST_BS16T1EN BS16T1EN
  * @brief  Set BS16T1 reset request bit for md_rcu_set_apb1rst() function used
  * @param  bs16t1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1rst_bs16t1en_fun(bs16t1en)  (bs16t1en<<RCU_APB1RST_BS16T1EN_POS)
/**
  * @} MD_RCU_APB1RST_BS16T1EN
  */
/** @defgroup MD_RCU_APB1RST_GP32C4T1EN GP32C4T1EN
  * @brief  Set GP32C4T1 reset request bit for md_rcu_set_apb1rst() function used
  * @param  gp32c4t1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1rst_gp32c4t1en_fun(gp32c4t1en)  (gp32c4t1en<<RCU_APB1RST_GP32C4T1EN_POS)
/**
  * @} MD_RCU_APB1RST_GP32C4T1EN
  */
/**
  * @brief  Set APB1 Peripheral Reset Register
  * @param  rcu RCU Instance
  * @param  i2c1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1rst_i2c1en_fun(i2c1en)
  * @param  suart1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1rst_suart1en_fun(suart1en)
  * @param  iwdten This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1rst_iwdten_fun(iwdten)
  * @param  wwdten This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1rst_wwdten_fun(wwdten)
  * @param  bs16t1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1rst_bs16t1en_fun(bs16t1en)
  * @param  gp32c4t1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1rst_gp32c4t1en_fun(gp32c4t1en)
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_apb1rst(RCU_TypeDef *rcu, uint32_t i2c1en, uint32_t suart1en, uint32_t iwdten,
                                        uint32_t wwdten, uint32_t bs16t1en, uint32_t gp32c4t1en)
{
    WRITE_REG(rcu->APB1RST, (i2c1en | suart1en | iwdten | wwdten | bs16t1en | gp32c4t1en));
}

/**
  * @brief  GP16C2T2 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t2_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T2EN_MSK);
}
/**
  * @brief  GP16C2T2 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t2_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T2EN_MSK);
}
/**
  * @brief  Get GP16C2T2 reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t2_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T2EN_MSK) >> RCU_APB2RST_GP16C2T2EN_POS);
}
/**
  * @brief  GP16C2T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T1EN_MSK);
}
/**
  * @brief  GP16C2T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T1EN_MSK);
}
/**
  * @brief  Get GP16C2T1 reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T1EN_MSK) >> RCU_APB2RST_GP16C2T1EN_POS);
}
/**
  * @brief  UART1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_UART1EN_MSK);
}
/**
  * @brief  UART1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_UART1EN_MSK);
}
/**
  * @brief  Get UART1 reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_UART1EN_MSK) >> RCU_APB2RST_UART1EN_POS);
}
/**
  * @brief  SPI1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_spi1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_SPI1EN_MSK);
}
/**
  * @brief  SPI1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_spi1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_SPI1EN_MSK);
}
/**
  * @brief  Get SPI1 reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_spi1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_SPI1EN_MSK) >> RCU_APB2RST_SPI1EN_POS);
}
/**
  * @brief  MMC reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_mmcen_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_MMCEN_MSK);
}
/**
  * @brief  MMC reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_mmcen_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_MMCEN_MSK);
}
/**
  * @brief  Get MMC reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_mmcen_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_MMCEN_MSK) >> RCU_APB2RST_MMCEN_POS);
}
/**
  * @brief  LCD reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_lcden_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_LCDEN_MSK);
}
/**
  * @brief  LCD reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_lcden_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_LCDEN_MSK);
}
/**
  * @brief  Get LCD reset request bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_lcden_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_LCDEN_MSK) >> RCU_APB2RST_LCDEN_POS);
}

/** @defgroup MD_RCU_APB2RST_GP16C2T2EN GP16C2T2EN
  * @brief  Set GP16C2T2 reset request bit for md_rcu_set_apb2rst() function used
  * @param  gp16c2t2en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2rst_gp16c2t2en_fun(gp16c2t2en)  (gp16c2t2en<<RCU_APB2RST_GP16C2T2EN_POS)
/**
  * @} MD_RCU_APB2RST_GP16C2T2EN
  */
/** @defgroup MD_RCU_APB2RST_GP16C2T1EN GP16C2T1EN
  * @brief  Set GP16C2T1 reset request bit for md_rcu_set_apb2rst() function used
  * @param  gp16c2t1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2rst_gp16c2t1en_fun(gp16c2t1en)  (gp16c2t1en<<RCU_APB2RST_GP16C2T1EN_POS)
/**
  * @} MD_RCU_APB2RST_GP16C2T1EN
  */
/** @defgroup MD_RCU_APB2RST_UART1EN UART1EN
  * @brief  Set UART1 reset request bit for md_rcu_set_apb2rst() function used
  * @param  uart1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2rst_uart1en_fun(uart1en)  (uart1en<<RCU_APB2RST_UART1EN_POS)
/**
  * @} MD_RCU_APB2RST_UART1EN
  */
/** @defgroup MD_RCU_APB2RST_SPI1EN SPI1EN
  * @brief  Set SPI1 reset request bit for md_rcu_set_apb2rst() function used
  * @param  spi1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2rst_spi1en_fun(spi1en)  (spi1en<<RCU_APB2RST_SPI1EN_POS)
/**
  * @} MD_RCU_APB2RST_SPI1EN
  */
/** @defgroup MD_RCU_APB2RST_MMCEN MMCEN
  * @brief  Set MMC reset request bit for md_rcu_set_apb2rst() function used
  * @param  spi1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2rst_mmcen_fun(mmc)  (mmc<<RCU_APB2RST_MMCEN_POS)
/**
  * @} MD_RCU_APB2RST_MMCEN
  */
/** @defgroup MD_RCU_APB2RST_LCDEN LCDEN
  * @brief  Set LCD reset request bit for md_rcu_set_apb2rst() function used
  * @param  spi1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2rst_lcden_fun(lcd)  (lcd<<RCU_APB2RST_LCDEN_POS)
/**
  * @} MD_RCU_APB2RST_LCDEN
  */

/**
  * @brief  Set APB1 Peripheral Reset Register
  * @param  rcu RCU Instance
  * @param  gp16c2t2en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2rst_gp16c2t2en_fun(gp16c2t2en)
  * @param  gp16c2t1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2rst_gp16c2t1en_fun(gp16c2t1en)
  * @param  uart1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2rst_uart1en_fun(uart1en)
  * @param  spi1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2rst_spi1en_fun(spi1en)
  * @param  mmcen This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2rst_mmcen_fun(mmcen)
  * @param  lcden This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2rst_lcden_fun(lcden)
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_apb2rst(RCU_TypeDef *rcu, uint32_t gp16c2t2en, uint32_t gp16c2t1en, uint32_t uart1en, uint32_t spi1en,
                                        uint32_t mmcen, uint32_t lcden)
{
    WRITE_REG(rcu->APB2RST, (gp16c2t2en | gp16c2t1en | uart1en | spi1en | mmcen | lcden));
}

/**
  * @brief  GPIOD clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiod(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_GPDEN_MSK);
}
/**
  * @brief  GPIOD Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiod(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_GPDEN_MSK);
}
/**
  * @brief  Get GPIOD Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiod(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_GPDEN_MSK) >> RCU_AHBEN_GPDEN_POS);
}
/**
  * @brief  GPIOC clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_GPCEN_MSK);
}
/**
  * @brief  GPIOC Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_GPCEN_MSK);
}
/**
  * @brief  Get GPIOC Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_GPCEN_MSK) >> RCU_AHBEN_GPCEN_POS);
}
/**
  * @brief  HDIV clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hdiven(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_HDIVEN_MSK);
}
/**
  * @brief  HDIV Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hdiven(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_HDIVEN_MSK);
}
/**
  * @brief  Get HDIV Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hdiven(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_HDIVEN_MSK) >> RCU_AHBEN_HDIVEN_POS);
}

/** @defgroup MD_RCU_AHBEN_GPDEN GPDEN
  * @brief  Set GPIOD clock bit for md_rcu_set_ahben() function used
  * @param  gpden This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_ahben_gpden_fun(gpden)  (gpden<<RCU_AHBEN_GPDEN_POS)
/**
  * @} MD_RCU_AHBEN_GPDEN
  */
/** @defgroup MD_RCU_AHBEN_GPCEN GPCEN
  * @brief  Set GPIOC clock bit for md_rcu_set_ahben() function used
  * @param  gpcen This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_ahben_gpcen_fun(gpcen)  (gpcen<<RCU_AHBEN_GPCEN_POS)
/**
  * @} MD_RCU_AHBEN_GPCEN
  */
/** @defgroup MD_RCU_AHBEN_HDIVEN HDIVEN
  * @brief  Set HDIV clock bit for md_rcu_set_ahben() function used
  * @param  hdiven This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_ahben_hdiven_fun(hdiven)  (hdiven<<RCU_AHBEN_HDIVEN_POS)
/**
  * @} MD_RCU_AHBEN_HDIVEN
  */

/**
  * @brief  Set APB1 Peripheral Reset Register
  * @param  rcu RCU Instance
  * @param  gpden This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_ahben_gpden_fun(gpden)
  * @param  gpcen This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_ahben_gpcen_fun(gpcen)
  * @param  hdiven This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_ahben_hdiven_fun(hdiven)
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ahben(RCU_TypeDef *rcu, uint32_t gpden, uint32_t gpcen, uint32_t hdiven)
{
    WRITE_REG(rcu->AHBEN, (gpden | gpcen | hdiven));
}

/**
  * @brief  I2C1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_i2c1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_I2C1EN_MSK);
}
/**
  * @brief  I2C1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_i2c1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_I2C1EN_MSK);
}
/**
  * @brief  Get I2C1 Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_i2c1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_I2C1EN_MSK) >> RCU_APB1EN_I2C1EN_POS);
}
/**
  * @brief  SUART1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_suart1en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_SUART1EN_MSK);
}
/**
  * @brief  SUART1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_suart1en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_SUART1EN_MSK);
}
/**
  * @brief  Get SUART1 Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_suart1en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_SUART1EN_MSK) >> RCU_APB1EN_SUART1EN_POS);
}
/**
  * @brief  IWDT clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_iwdten(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_IWDTEN_MSK);
}
/**
  * @brief  IWDT Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_iwdten(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_IWDTEN_MSK);
}
/**
  * @brief  Get IWDT Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_iwdten(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_IWDTEN_MSK) >> RCU_APB1EN_IWDTEN_POS);
}
/**
  * @brief  WWDT clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_wwdt(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_WWDTEN_MSK);
}
/**
  * @brief  WWDT Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_wwdt(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_WWDTEN_MSK);
}
/**
  * @brief  Get WWDT Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_wwdt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_WWDTEN_MSK) >> RCU_APB1EN_WWDTEN_POS);
}
/**
  * @brief  BS16T1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_bs16t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_BS16T1EN_MSK);
}
/**
  * @brief  BS16T1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_bs16t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_BS16T1EN_MSK);
}
/**
  * @brief  Get BS16T1 Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_bs16t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_BS16T1EN_MSK) >> RCU_APB1EN_BS16T1EN_POS);
}
/**
  * @brief  GP32C4T1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T1EN_MSK);
}
/**
  * @brief  GP32C4T1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T1EN_MSK);
}
/**
  * @brief  Get GP32C4T1 Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T1EN_MSK) >> RCU_APB1EN_GP32C4T1EN_POS);
}

/** @defgroup MD_RCU_APB1EN_I2C1EN I2C1EN
  * @brief  Set I2C1 clock bit for md_rcu_set_apb1en() function used
  * @param  i2c1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1en_i2c1en_fun(i2c1en)  (i2c1en<<RCU_APB1EN_I2C1EN_POS)
/**
  * @} MD_RCU_APB1EN_I2C1EN
  */
/** @defgroup MD_RCU_APB1EN_SUART1EN SUART1EN
  * @brief  Set SUART1 clock bit for md_rcu_set_apb1en() function used
  * @param  suart1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1en_suart1en_fun(suart1en)  (suart1en<<RCU_APB1EN_SUART1EN_POS)
/**
  * @} MD_RCU_APB1EN_SUART1EN
  */
/** @defgroup MD_RCU_APB1EN_IWDTEN IWDTEN
  * @brief  Set IWDT clock bit for md_rcu_set_apb1en() function used
  * @param  iwdten This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1en_iwdten_fun(iwdten)  (iwdten<<RCU_APB1EN_IWDTEN_POS)
/**
  * @} MD_RCU_APB1EN_IWDTEN
  */
/** @defgroup MD_RCU_APB1EN_WWDTEN WWDTEN
  * @brief  Set WWDT clock bit for md_rcu_set_apb1en() function used
  * @param  wwdten This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1en_wwdten_fun(wwdten)  (wwdten<<RCU_APB1EN_WWDTEN_POS)
/**
  * @} MD_RCU_APB1EN_WWDTEN
  */
/** @defgroup MD_RCU_APB1EN_BS16T1EN BS16T1EN
  * @brief  Set BS16T1 clock bit for md_rcu_set_apb1en() function used
  * @param  bs16t1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1en_bs16t1en_fun(bs16t1en)  (bs16t1en<<RCU_APB1EN_BS16T1EN_POS)
/**
  * @} MD_RCU_APB1EN_BS16T1EN
  */
/** @defgroup MD_RCU_APB1EN_GP32C4T1EN GP32C4T1EN
  * @brief  Set GP32C4T1 clock bit for md_rcu_set_apb1en() function used
  * @param  gp32c4t1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb1en_gp32c4t1en_fun(gp32c4t1en)  (gp32c4t1en<<RCU_APB1EN_GP32C4T1EN_POS)
/**
  * @} MD_RCU_APB1EN_GP32C4T1EN
  */
/**
  * @brief  Set APB1 Peripheral Reset Register
  * @param  rcu RCU Instance
  * @param  i2c1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1en_i2c1en_fun(i2c1en)
  * @param  suart1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1en_suart1en_fun(suart1en)
  * @param  iwdten This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1en_iwdten_fun(iwdten)
  * @param  wwdten This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1en_wwdten_fun(wwdten)
  * @param  bs16t1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1en_bs16t1en_fun(bs16t1en)
  * @param  gp32c4t1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb1en_gp32c4t1en_fun(gp32c4t1en)
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_apb1en(RCU_TypeDef *rcu, uint32_t i2c1en, uint32_t suart1en, uint32_t iwdten,
                                       uint32_t wwdten, uint32_t bs16t1en, uint32_t gp32c4t1en)
{
    WRITE_REG(rcu->APB1EN, (i2c1en | suart1en | iwdten | wwdten | bs16t1en | gp32c4t1en));
}

/**
  * @brief  GP16C2T2EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t2(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T2EN_MSK);
}
/**
  * @brief  GP16C2T2EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t2(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T2EN_MSK);
}
/**
  * @brief  Get GP16C2T2EN Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t2(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T2EN_MSK) >> RCU_APB2EN_GP16C2T2EN_POS);
}
/**
  * @brief  GP16C2T1EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T1EN_MSK);
}
/**
  * @brief  GP16C2T1EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T1EN_MSK);
}
/**
  * @brief  Get GP16C2T1EN Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T1EN_MSK) >> RCU_APB2EN_GP16C2T1EN_POS);
}
/**
  * @brief  UART1EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_UART1EN_MSK);
}
/**
  * @brief  UART1EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_UART1EN_MSK);
}
/**
  * @brief  Get UART1EN Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_UART1EN_MSK) >> RCU_APB2EN_UART1EN_POS);
}
/**
  * @brief  SPI1EN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_spi1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_SPI1EN_MSK);
}
/**
  * @brief  SPI1EN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_spi1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_SPI1EN_MSK);
}
/**
  * @brief  Get SPI1EN Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_spi1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_SPI1EN_MSK) >> RCU_APB2EN_SPI1EN_POS);
}
/**
  * @brief  MMCEN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_mmcen(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_MMCEN_MSK);
}
/**
  * @brief  MMCEN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_mmcen(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_MMCEN_MSK);
}
/**
  * @brief  Get MMCEN Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_mmcen(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_MMCEN_MSK) >> RCU_APB2EN_MMCEN_POS);
}
/**
  * @brief  LCDEN clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_lcden(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_LCDEN_MSK);
}
/**
  * @brief  LCDEN Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_lcden(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_LCDEN_MSK);
}
/**
  * @brief  Get LCDEN Clock bit
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_lcden(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_LCDEN_MSK) >> RCU_APB2EN_LCDEN_POS);
}

/** @defgroup MD_RCU_APB2EN_GP16C2T2EN GP16C2T2EN
  * @brief  Set GP16C2T2EN clock bit for md_rcu_set_apb2en() function used
  * @param  gp16c2t2en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2en_gp16c2t2en_fun(gp16c2t2en)  (gp16c2t2en<<RCU_APB2EN_GP16C2T2EN_POS)
/**
  * @} MD_RCU_APB2EN_GP16C2T2EN
  */
/** @defgroup MD_RCU_APB2EN_GP16C2T1EN GP16C2T1EN
  * @brief  Set GP16C2T1EN clock bit for md_rcu_set_apb2en() function used
  * @param  gp16c2t1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2en_gp16c2t1en_fun(gp16c2t1en)  (gp16c2t1en<<RCU_APB2EN_GP16C2T1EN_POS)
/**
  * @} MD_RCU_APB2EN_GP16C2T1EN
  */
/** @defgroup MD_RCU_APB2EN_UART1EN UART1EN
  * @brief  Set UART1EN clock bit for md_rcu_set_apb2en() function used
  * @param  uart1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2en_uart1en_fun(uart1en)  (uart1en<<RCU_APB2EN_UART1EN_POS)
/**
  * @} MD_RCU_APB2EN_UART1EN
  */
/** @defgroup MD_RCU_APB2EN_SPI1EN SPI1EN
  * @brief  Set SPI1EN clock bit for md_rcu_set_apb2en() function used
  * @param  spi1en This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2en_spi1en_fun(spi1en)  (spi1en<<RCU_APB2EN_SPI1EN_POS)
/**
  * @} MD_RCU_APB2EN_SPI1EN
  */
/** @defgroup MD_RCU_APB2EN_MMCENN MMCEN
  * @brief  Set MMCEN clock bit for md_rcu_set_apb2en() function used
  * @param  mmcen This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2en_mmcen_fun(mmcen)  (mmcen<<RCU_APB2EN_MMCEN_POS)
/**
  * @} MD_RCU_APB2EN_MMCEN
  */
/** @defgroup MD_RCU_APB2EN_LCDEN LCDEN
  * @brief  Set LCDEN clock bit for md_rcu_set_apb2en() function used
  * @param  lcden This parameter can be one of the following values:
  *         @arg @ref DISABLE
  *         @arg @ref ENABLE
  * @{
  */
#define md_rcu_set_apb2en_lcden_fun(lcden)  (lcden<<RCU_APB2EN_LCDEN_POS)
/**
  * @} MD_RCU_APB2EN_LCDEN
  */
/**
  * @brief  Set APB2 Peripheral Reset Register
  * @param  rcu RCU Instance
  * @param  gp16c2t2en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2en_gp16c2t2en_fun(gp16c2t2en)
  * @param  gp16c2t1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2en_gp16c2t1en_fun(gp16c2t1en)
  * @param  uart1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2en_uart1en_fun(uart1en)
  * @param  spi1en This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2en_spi1en_fun(spi1en)
  * @param  mmcen This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2en_mmcen_fun(mmcen)
  * @param  lcden This parameter can be one of the following values:
  *         @arg @ref md_rcu_set_apb2en_lcden_fun(lcden)
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_apb2en(RCU_TypeDef *rcu, uint32_t gp16c2t2en, uint32_t gp16c2t1en, uint32_t uart1en,
                                       uint32_t spi1en, uint32_t mmcen, uint32_t lcden)
{
    WRITE_REG(rcu->APB2EN, (gp16c2t2en | gp16c2t1en | uart1en | spi1en | mmcen | lcden));
}

/**
  * @brief  GPDEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiod_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_GPDEN_MSK);
}
/**
  * @brief  GPDEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiod_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_GPDEN_MSK);
}
/**
  * @brief  Get GPDEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiod_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_GPDEN_MSK) >> RCU_AHBSL_GPDEN_POS);
}
/**
  * @brief  GPCEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioc_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_GPCEN_MSK);
}
/**
  * @brief  GPCEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioc_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_GPCEN_MSK);
}
/**
  * @brief  Get GPCEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioc_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_GPCEN_MSK) >> RCU_AHBSL_GPCEN_POS);
}
/**
  * @brief  HDIVEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ahbsl_hdiven(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_HDIVEN_MSK);
}
/**
  * @brief  HDIVEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ahbsl_hdiven(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_HDIVEN_MSK);
}
/**
  * @brief  Get HDIVEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ahbsl_hdiven(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_HDIVEN_MSK) >> RCU_AHBSL_HDIVEN_POS);
}

/**
  * @brief  I2C1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_i2c1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_I2C1EN_MSK);
}
/**
  * @brief  I2C1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_i2c1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_I2C1EN_MSK);
}
/**
  * @brief  Get I2C1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_i2c1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_I2C1EN_MSK) >> RCU_APB1SL_I2C1EN_POS);
}
/**
  * @brief  SUART1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1sl_suart1en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_SUART1EN_MSK);
}
/**
  * @brief  SUART1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1sl_suart1en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_SUART1EN_MSK);
}
/**
  * @brief  Get SUART1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1sl_suart1en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_SUART1EN_MSK) >> RCU_APB1SL_SUART1EN_POS);
}
/**
  * @brief  IWDTEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1sl_iwdten(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_IWDTEN_MSK);
}
/**
  * @brief IWDTEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1sl_iwdten(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_IWDTEN_MSK);
}
/**
  * @brief  Get IWDTEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1sl_iwdten(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_IWDTEN_MSK) >> RCU_APB1SL_IWDTEN_POS);
}
/**
  * @brief  WWDTEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_wwdt_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_WWDTEN_MSK);
}
/**
  * @brief WWDTEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_wwdt_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_WWDTEN_MSK);
}
/**
  * @brief  Get WWDTEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_wwdt_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_WWDTEN_MSK) >> RCU_APB1SL_WWDTEN_POS);
}
/**
  * @brief  BS16T1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_bs16t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_BS16T1EN_MSK);
}
/**
  * @brief  BS16T1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_bs16t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_BS16T1EN_MSK);
}
/**
  * @brief  Get BS16T1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_bs16t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_BS16T1EN_MSK) >> RCU_APB1SL_BS16T1EN_POS);
}
/**
  * @brief  GP32C4T1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T1EN_MSK);
}
/**
  * @brief  GP32C4T1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T1EN_MSK);
}
/**
  * @brief  Get GP32C4T1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T1EN_MSK) >> RCU_APB1SL_GP32C4T1EN_POS);
}

/**
  * @brief  GP16C2T2EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T2EN_MSK);
}
/**
  * @brief  GP16C2T2EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T2EN_MSK);
}
/**
  * @brief  Get GP16C2T2EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T2EN_MSK) >> RCU_APB2SL_GP16C2T2EN_POS);
}
/**
  * @brief  GP16C2T1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T1EN_MSK);
}
/**
  * @brief  GP16C2T1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T1EN_MSK);
}
/**
  * @brief  Get GP16C2T1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T1EN_MSK) >> RCU_APB2SL_GP16C2T1EN_POS);
}
/**
  * @brief  UART1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_UART1EN_MSK);
}
/**
  * @brief  UART1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_UART1EN_MSK);
}
/**
  * @brief  Get UART1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_UART1EN_MSK) >> RCU_APB2SL_UART1EN_POS);
}
/**
  * @brief  SPI1EN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_spi1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_SPI1EN_MSK);
}
/**
  * @brief  SPI1EN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_spi1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_SPI1EN_MSK);
}
/**
  * @brief  Get SPI1EN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_spi1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_SPI1EN_MSK) >> RCU_APB2SL_SPI1EN_POS);
}
/**
  * @brief  MMCEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb2sl_mmcen(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_MMCEN_MSK);
}
/**
  * @brief  MMCEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb2sl_mmcen(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_MMCEN_MSK);
}
/**
  * @brief  Get MMCEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb2sl_mmcen(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_MMCEN_MSK) >> RCU_APB2SL_MMCEN_POS);
}
/**
  * @brief  LCDEN clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb2sl_lcden(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_LCDEN_MSK);
}
/**
  * @brief  LCDEN clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb2sl_lcden(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_LCDEN_MSK);
}
/**
  * @brief  Get LCDEN clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb2sl_lcden(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_LCDEN_MSK) >> RCU_APB2SL_LCDEN_POS);
}

/**
  * @brief  GPIOD clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ahbdsl_gpden(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBDSL, RCU_AHBDSL_GPDEN_MSK);
}
/**
  * @brief  GPIOD clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ahbdsl_gpden(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBDSL, RCU_AHBDSL_GPDEN_MSK);
}
/**
  * @brief  Get GPIOD clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ahbdsl_gpden(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBDSL, RCU_AHBDSL_GPDEN_MSK) >> RCU_AHBDSL_GPDEN_POS);
}
/**
  * @brief  GPIOC clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ahbdsl_gpcen(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBDSL, RCU_AHBDSL_GPCEN_MSK);
}
/**
  * @brief  GPIOC clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ahbdsl_gpcen(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBDSL, RCU_AHBDSL_GPCEN_MSK);
}
/**
  * @brief  Get GPIOC clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ahbdsl_gpcen(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBDSL, RCU_AHBDSL_GPCEN_MSK) >> RCU_AHBDSL_GPCEN_POS);
}
/**
  * @brief  HDIV clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ahbdsl_hdiven(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBDSL, RCU_AHBDSL_HDIVEN_MSK);
}
/**
  * @brief  HDIV clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ahbdsl_hdiven(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBDSL, RCU_AHBDSL_HDIVEN_MSK);
}
/**
  * @brief  Get HDIV clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ahbdsl_hdiven(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBDSL, RCU_AHBDSL_HDIVEN_MSK) >> RCU_AHBDSL_HDIVEN_POS);
}

/**
  * @brief  I2C1 clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1dsl_i2c1en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1DSL, RCU_APB1DSL_I2C1EN_MSK);
}
/**
  * @brief  I2C1 clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1dsl_i2c1en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1DSL, RCU_APB1DSL_I2C1EN_MSK);
}
/**
  * @brief  Get I2C1 clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1dsl_i2c1en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1DSL, RCU_APB1DSL_I2C1EN_MSK) >> RCU_APB1DSL_I2C1EN_POS);
}
/**
  * @brief  SUART1 clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1dsl_suart1en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1DSL, RCU_APB1DSL_SUART1EN_MSK);
}
/**
  * @brief  SUART1 clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1dsl_suart1en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1DSL, RCU_APB1DSL_SUART1EN_MSK);
}
/**
  * @brief  Get SUART1 clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1dsl_suart1en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1DSL, RCU_APB1DSL_SUART1EN_MSK) >> RCU_APB1DSL_SUART1EN_POS);
}
/**
  * @brief  IWDT clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1dsl_iwdten(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1DSL, RCU_APB1DSL_IWDTEN_MSK);
}
/**
  * @brief  IWDT clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1dsl_iwdten(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1DSL, RCU_APB1DSL_IWDTEN_MSK);
}
/**
  * @brief  Get IWDT clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1dsl_iwdten(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1DSL, RCU_APB1DSL_IWDTEN_MSK) >> RCU_APB1DSL_IWDTEN_POS);
}
/**
  * @brief  WWDT clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1dsl_wwdten(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1DSL, RCU_APB1DSL_WWDTEN_MSK);
}
/**
  * @brief  WWDT clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1dsl_wwdten(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1DSL, RCU_APB1DSL_WWDTEN_MSK);
}
/**
  * @brief  Get WWDT clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1dsl_wwdten(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1DSL, RCU_APB1DSL_WWDTEN_MSK) >> RCU_APB1DSL_WWDTEN_POS);
}
/**
  * @brief  BS16T1 clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1dsl_bs16t1en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1DSL, RCU_APB1DSL_BS16T1EN_MSK);
}
/**
  * @brief  BS16T1 clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1dsl_bs16t1en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1DSL, RCU_APB1DSL_BS16T1EN_MSK);
}
/**
  * @brief  Get BS16T1 clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1dsl_bs16t1en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1DSL, RCU_APB1DSL_BS16T1EN_MSK) >> RCU_APB1DSL_BS16T1EN_POS);
}
/**
  * @brief  GP32C4T1 clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb1dsl_gp32c4t1en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1DSL, RCU_APB1DSL_GP32C4T1EN_MSK);
}
/**
  * @brief  GP32C4T1 clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb1dsl_gp32c4t1en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1DSL, RCU_APB1DSL_GP32C4T1EN_MSK);
}
/**
  * @brief  Get GP32C4T1 clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb1dsl_gp32c4t1en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1DSL, RCU_APB1DSL_GP32C4T1EN_MSK) >> RCU_APB1DSL_GP32C4T1EN_POS);
}

/**
  * @brief  GP16C2T2 clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb2dsl_gp16c2t2en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2DSL, RCU_APB2DSL_GP16C2T2EN_MSK);
}
/**
  * @brief  GP16C2T2 clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb2dsl_gp16c2t2en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2DSL, RCU_APB2DSL_GP16C2T2EN_MSK);
}
/**
  * @brief  Get GP16C2T2 clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb2dsl_gp16c2t2en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2DSL, RCU_APB2DSL_GP16C2T2EN_MSK) >> RCU_APB2DSL_GP16C2T2EN_POS);
}
/**
  * @brief  GP16C2T1 clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb2dsl_gp16c2t1en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2DSL, RCU_APB2DSL_GP16C2T1EN_MSK);
}
/**
  * @brief  GP16C2T1 clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb2dsl_gp16c2t1en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2DSL, RCU_APB2DSL_GP16C2T1EN_MSK);
}
/**
  * @brief  Get GP16C2T1 clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb2dsl_gp16c2t1en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2DSL, RCU_APB2DSL_GP16C2T1EN_MSK) >> RCU_APB2DSL_GP16C2T1EN_POS);
}
/**
  * @brief  UART1 clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb2dsl_uart1en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2DSL, RCU_APB2DSL_UART1EN_MSK);
}
/**
  * @brief  UART1 clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb2dsl_uart1en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2DSL, RCU_APB2DSL_UART1EN_MSK);
}
/**
  * @brief  Get UART1 clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb2dsl_uart1en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2DSL, RCU_APB2DSL_UART1EN_MSK) >> RCU_APB2DSL_UART1EN_POS);
}
/**
  * @brief  SPI1 clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb2dsl_spi1en(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2DSL, RCU_APB2DSL_SPI1EN_MSK);
}
/**
  * @brief  SPI1 clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb2dsl_spi1en(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2DSL, RCU_APB2DSL_SPI1EN_MSK);
}
/**
  * @brief  Get SPI1 clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb2dsl_spi1en(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2DSL, RCU_APB2DSL_SPI1EN_MSK) >> RCU_APB2DSL_SPI1EN_POS);
}
/**
  * @brief  MMC clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb2dsl_mmcen(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2DSL, RCU_APB2DSL_MMCEN_MSK);
}
/**
  * @brief  MMC clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb2dsl_mmcen(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2DSL, RCU_APB2DSL_MMCEN_MSK);
}
/**
  * @brief  Get MMC clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb2dsl_mmcen(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2DSL, RCU_APB2DSL_MMCEN_MSK) >> RCU_APB2DSL_MMCEN_POS);
}
/**
  * @brief  LCD clock enable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_apb2dsl_lcden(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2DSL, RCU_APB2DSL_LCDEN_MSK);
}
/**
  * @brief  LCD clock disable during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_apb2dsl_lcden(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2DSL, RCU_APB2DSL_LCDEN_MSK);
}
/**
  * @brief  Get LCD clock during Deep Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_apb2dsl_lcden(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2DSL, RCU_APB2DSL_LCDEN_MSK) >> RCU_APB2DSL_LCDEN_POS);
}

/**
  * @brief  Get HRC automatic calibration count value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_hrc_trim(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HTRIM, RCU_HTRIM_CNT_MSK) >> RCU_HTRIM_CNT_POSS);
}
/**
  * @brief  Get Calibration failed flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_htrim_fail(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HTRIM, RCU_HTRIM_FAIL_MSK) >> RCU_HTRIM_FAIL_POS);
}
/**
  * @brief  Get Calibration non-compliance sign
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_htrim_notequ(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HTRIM, RCU_HTRIM_NOTEQU_MSK) >> RCU_HTRIM_NOTEQU_POS);
}
/**
  * @brief  Get Automatic calibration of busy status flags
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_htrim_busy(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HTRIM, RCU_HTRIM_BUSY_MSK) >> RCU_HTRIM_BUSY_POS);
}
/**
  * @brief  Set HRC auto calibration value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_set_htrim_hrctrim(RCU_TypeDef *rcu, uint32_t hrctrim)
{
    MODIFY_REG(rcu->HTRIM, RCU_HTRIM_HRCTRIM_MSK, (hrctrim << RCU_HTRIM_HRCTRIM_POSS));
}
/**
  * @brief  Get HRC auto calibration value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_htrim_hrctrim(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HTRIM, RCU_HTRIM_HRCTRIM_MSK) >> RCU_HTRIM_HRCTRIM_POSS);
}
/**
  * @brief  HRC calibration value selection(HRCTRIM)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_htrim_insel_hrctrim(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->HTRIM, RCU_HTRIM_INSEL_MSK);
}
/**
  * @brief  HRC calibration value selection(option byte)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_htrim_insel_option_byte(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->HTRIM, RCU_HTRIM_INSEL_MSK);
}
/**
  * @brief  Get HRC calibration value selection
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_htrim_insel_selection(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HTRIM, RCU_HTRIM_INSEL_MSK) >> RCU_HTRIM_INSEL_POS);
}
/**
  * @brief  Set HRC calibration value interval
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_set_htrim_range(RCU_TypeDef *rcu, uint32_t range)
{
    MODIFY_REG(rcu->HTRIM, RCU_HTRIM_RANGE_MSK, (range << RCU_HTRIM_RANGE_POSS));
}
/**
  * @brief  Get HRC calibration value interval
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_htrim_range(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HTRIM, RCU_HTRIM_RANGE_MSK) >> RCU_HTRIM_RANGE_POSS);
}
/**
  * @brief  Calibration function selection(SW)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hrc_trim_software(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->HTRIM, RCU_HTRIM_SWEN_MSK);
}
/**
  * @brief  Calibration function selection(HW)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hrc_trim_hardware(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->HTRIM, RCU_HTRIM_SWEN_MSK);
}
/**
  * @brief  Get calibration value selection
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_hrc_trim_source(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HTRIM, RCU_HTRIM_SWEN_MSK) >> RCU_HTRIM_SWEN_POS);
}
/**
  * @brief  HRC auto calibration starts
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hrc_trim_start(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->HTRIM, RCU_HTRIM_TRIM_MSK);
}

/**
  * @brief  Get HRC auto calibration starts
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hrc_trim_start(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HTRIM, RCU_HTRIM_TRIM_MSK) >> RCU_HTRIM_TRIM_POS);
}

/**
  * @brief  Set HRC calibration target count value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_set_hcon_trim_tar(RCU_TypeDef *rcu, uint32_t trim_tar)
{
    MODIFY_REG(rcu->HCON, RCU_HCON_TRIM_TAR_MSK, (trim_tar << RCU_HCON_TRIM_TAR_POSS));
}
/**
  * @brief  Get HRC calibration target count value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_hcon_trim_tar(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HCON, RCU_HCON_TRIM_TAR_MSK) >> RCU_HCON_TRIM_TAR_POSS);
}
/**
  * @brief  Get HRC option byte Calibration Count
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_hcon_opt_hrctrim(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HCON, RCU_HCON_OPT_HRCTRIM_MSK) >> RCU_HCON_OPT_HRCTRIM_POSS);
}
/**
  * @brief  Set HRC stable time value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_set_hcon_settle(RCU_TypeDef *rcu, uint32_t settle)
{
    MODIFY_REG(rcu->HCON, RCU_HCON_SETTLE_MSK, (settle << RCU_HCON_SETTLE_POSS));
}
/**
  * @brief  Get HRC stable time value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_hcon_settle(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->HCON, RCU_HCON_SETTLE_MSK) >> RCU_HCON_SETTLE_POSS);
}

/**
  * @brief  Get LRC automatic calibration count value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_ltrim_cnt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LTRIM, RCU_LTRIM_CNT_MSK) >> RCU_LTRIM_CNT_POSS);
}
/**
  * @brief  Get Calibration failed flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_ltrim_fail(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LTRIM, RCU_LTRIM_FAIL_MSK) >> RCU_LTRIM_FAIL_POS);
}
/**
  * @brief  Get Calibration non-compliance sign
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_ltrim_notequ(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LTRIM, RCU_LTRIM_NOTEQU_MSK) >> RCU_LTRIM_NOTEQU_POS);
}
/**
  * @brief  Get Automatic calibration of busy status flags
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_ltrim_busy(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LTRIM, RCU_LTRIM_BUSY_MSK) >> RCU_LTRIM_BUSY_POS);
}
/**
  * @brief  Set LRC auto calibration value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_set_ltrim_hrctrim(RCU_TypeDef *rcu, uint32_t hrctrim)
{
    MODIFY_REG(rcu->LTRIM, RCU_LTRIM_LRCTRIM_MSK, (hrctrim << RCU_LTRIM_LRCTRIM_POSS));
}
/**
  * @brief  Get LRC auto calibration value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_ltrim_hrctrim(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LTRIM, RCU_LTRIM_LRCTRIM_MSK) >> RCU_LTRIM_LRCTRIM_POSS);
}
/**
  * @brief  LRC calibration value selection(HRCTRIM)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ltrim_insel_hrctrim(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->LTRIM, RCU_LTRIM_INSEL_MSK);
}
/**
  * @brief  LRC calibration value selection(option byte)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_ltrim_insel_option_byte(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->LTRIM, RCU_LTRIM_INSEL_MSK);
}
/**
  * @brief  Get LRC calibration value selection
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_ltrim_insel_selection(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LTRIM, RCU_LTRIM_INSEL_MSK) >> RCU_LTRIM_INSEL_POS);
}
/**
  * @brief  Set LRC calibration value interval
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_set_ltrim_range(RCU_TypeDef *rcu, uint32_t range)
{
    MODIFY_REG(rcu->LTRIM, RCU_LTRIM_RANGE_MSK, (range << RCU_LTRIM_RANGE_POSS));
}
/**
  * @brief  Get LRC calibration value interval
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_ltrim_range(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LTRIM, RCU_LTRIM_RANGE_MSK) >> RCU_LTRIM_RANGE_POSS);
}
/**
  * @brief  Calibration function selection(SW)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_lrc_trim_software(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->LTRIM, RCU_LTRIM_SWEN_MSK);
}
/**
  * @brief  Calibration function selection(HW)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_lrc_trim_hardware(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->LTRIM, RCU_LTRIM_SWEN_MSK);
}
/**
  * @brief  Get calibration value selection
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_lrc_trim_source(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LTRIM, RCU_LTRIM_SWEN_MSK) >> RCU_LTRIM_SWEN_POS);
}
/**
  * @brief  LRC auto calibration starts
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_lrc_trim_trim_start(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->LTRIM, RCU_LTRIM_TRIM_MSK);
}
/**
  * @brief  Get LRC auto calibration starts
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_lrc_trim_start(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LTRIM, RCU_LTRIM_TRIM_MSK) >> RCU_LTRIM_TRIM_POS);
}

/**
  * @brief  Set LRC calibration target count value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_set_lcon_trim_tar(RCU_TypeDef *rcu, uint32_t trim_tar)
{
    MODIFY_REG(rcu->LCON, RCU_LCON_TRIM_TAR_MSK, (trim_tar << RCU_LCON_TRIM_TAR_POSS));
}
/**
  * @brief  Get LRC calibration target count value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_lcon_trim_tar(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_TRIM_TAR_MSK) >> RCU_LCON_TRIM_TAR_POSS);
}
/**
  * @brief  Get LRC option byte Calibration Count
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_lcon_opt_lrctrim(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_OPT_LRCTRIM_MSK) >> RCU_LCON_OPT_LRCTRIM_POSS);
}
/**
  * @brief  Set LRC stable time value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_set_lcon_settle(RCU_TypeDef *rcu, uint32_t settle)
{
    MODIFY_REG(rcu->LCON, RCU_LCON_SETTLE_MSK, (settle << RCU_LCON_SETTLE_POSS));
}
/**
  * @brief  Get LRC stable time value
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_get_lcon_settle(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_SETTLE_MSK) >> RCU_LCON_SETTLE_POSS);
}

/**
  * @} MD_RCU_Public_Macros
  */

/* Public functions -----------------------------------------------------------*/
/** @defgroup MD_RCU_Public_Functions RCU Public Functions
  * @{
  */

/** @defgroup MD_RCU_PF_Basic_Configuration Basic Configuration
  * @{
  */
void md_rcu_init(void);
void md_rcu_init_set(RCU_TypeDef *, md_rcu_init_typedef *);
/**
  * @} MD_RCU_PF_Basic_Configuration
  */

/**
  * @} MD_RCU_Public_Functions
  */

#endif

/**
  * @} RCU
  */

/**
  * @} Micro_Driver
  */
#ifdef __cplusplus
}

#endif

#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

