/**********************************************************************************
 *
 * @file    md_mmc.h
 * @brief   header file of md_mmc.c
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_MMC_H__
#define __MD_MMC_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------*/
#include <stdbool.h>
#include "es32f0930.h"

/** @addtogroup Micro_Driver
  * @{
  */
#if defined (MMC)

/** @defgroup MMC MMC
  * @brief MMC module driver
  * @{
  */


/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private Macros ----------------------------------------------------------*/
/* Public Constants -----------------------------------------------------------*/

/** @defgroup MD_MMC_Public_Constants Public Constants
  * @brief MMC module driver
  * @{
  */

/* 0x00  MMC_CTRL   - MMC0 Control  */
#define EN_MMC(y)       ((y&0x01)<<0)   /*Enable MMC function*/
#define EN_VRGEN(y)     ((y&0x01)<<1)   /*Enable voltage reference generator*/
#define EN_VREFO(y)     ((y&0x01)<<2)   /*Enable bandgap voltage reference*/
#define EN_VDDA(y)      ((y&0x01)<<3)   /*Enable VDDA LDO*/
#define EN_VPUMP(y)     ((y&0x01)<<4)   /*Enable charge pump regulator*/
#define BN02_CTRL(y)    ((y&0x01)<<5)   /*BN<0> and BN<2> short control*/
/*0 : open*/
/*1 : short*/
#define VREF_SRC(y)     ((y&0x01)<<6)   /*Select reference voltage source*/
/*0 : Band-gap Voltage Reference*/
/*1 : BN4*/
#define VRG_K2(y)       ((y&0x01)<<7)   /*Voltoge reference genarator k2 paramater*/
/*0:Select K2B,(VREFO-VSS)>(2GREF-VSS),(GREF[P/N]<X>-GREF)=K2B(VREFO-GREF)+(K2B-1)(GREF-VSS)*/
/*1:Select K2, (VREFO-VSSA)<(2GREF-VSS),(GREF[P/N]<X>-GREF)=K2(VREFO -GREF)*/
#define SEL_GREF(y)     ((y&0x03)<<8)   /*Select GREF*/
/*00GFloating*/
/*01G0.1VDDA*/
/*10G0.3VDDA*/
/*11G0.5VDDA*/
#define SEL_VDDA(y)      ((y&0x03)<<10)  /*Select VDDA*/
/*00G3.6V*/
/*01G3.2V*/
#define VREF_SEL(y)     ((y&0x0F)<<12)  /*Select reference voltage combine with MODE[3] and CMPO [VREF_SEL, MODE<3>, CMPO] : VREF*/
/*[ 0000, 0, X ]: VDSC<17>*/
/*[ 0001, 0, X ]: VDSC<16>*/
/*[ 0010, 0, X ]: VDSC<15>*/
/*[ 0011, 0, X ]: GREFP<9>*/
/*[ 0100, 0, X ]: GREFP<8>*/
/*[ 0101, 0, X ]: GREFP<7>*/
/*[ 0110, 0, X ]: GREFP<6>*/
/*[ 0111, 0, X ]: GREFP<0>*/
/*[ 1000, 0, X ]: VDSC<1>*/
/*[ 1001, 0, X ]: VDSC<2>*/
/*[ 1010, 0, X ]: VDSC<3>*/
/*[ 1011, 0, X ]: GREFN<9>*/
/*[ 1100, 0, X ]: GREFN<8>*/
/*[ 1101, 0, X ]: GREFN<7>*/
/*[ 1110, 0, X ]: GREFN<6>*/
/*[ 1111, 0, X ]: BN5*/
/*[ X000, 1, 0 ]: VDSC<17>*/
/*[ X000, 1, 1 ]: VDSC<1>*/
/*[ X001, 1, 0 ]: VDSC<16>*/
/*[ X001, 1, 1 ]: VDSC<2>*/
/*[ X010, 1, 0 ]: VDSC<15>*/
/*[ X010, 1, 1 ]: VDSC<3>*/
/*[ X011, 1, 0 ]: GREFP<9>*/
/*[ X011, 1, 1 ]: GREFN<9>*/
/*[ X100, 1, 0 ]: GREFP<8>*/
/*[ X100, 1, 1 ]: GREFN<8>*/
/*[ X101, 1, 0 ]: GREFP<7>*/
/*[ X101, 1, 1 ]: GREFN<7>*/
/*[ X110, 1, 0 ]: GREFP<6>*/
/*[ X110, 1, 1 ]: GREFN<6>*/
/*[ X111, 1, 0 ]: GREFP<0>*/
/*[ X111, 1, 1 ]: BN5*/
#define MODE(y)         ((y&0xFF)<<16)  /*Measure mode*/
/*00000000G*/
/*00100000G*/
/*00010100G*/
/*00010101G*/
/*00000110G*/
/*00000111G*/
/*00010101G*/
/*10100000G*/
/*0001101xG*/
/*0001110xG*/
/*0000111xG*/
/*ReservedGReserved*/
#define EN_VDDA_FLOAT(y) ((y&0x01)<<24)  /*Enable VDDA LDO floating*/
/*0: Bypass*/
/*1: Floating*/
#define SEL_VPUMP(y)    ((y&0x03)<<25)  /*Charge pump regulator select*/
/*00 : 3.8V*/
/*01 : 4.0V*/
/*10 : 4.2V(default)*/
/*11 : 4.4V*/
#define VPUMP_BYP(y)    ((y&0x01)<<27)  /*Charge pump regulator bypass*/
/*0: VPUMP*/
/*1: Bypass*/
#define VREFO_CHOP(y)   ((y&0x01)<<28)  /*MMC VREFO chopper enable*/
#define VREFO_CKSEL(y)    ((y&0x01)<<29)  /*MMC VREFO chopper clock selec*/
/*0 : CLK/4*/
/*1 : CLK/2*/
#define EN_FUNC_OP(y)   ((y&0x01)<<30)  /*MMC Function OP enable*/

/*MMC VREFO CKSEL input marcros define*/
#define MD_MMC_VREFO_CKSEL_DIV4                    0                /*!< VREFO_CKSEL = CLK/4*/
#define MD_MMC_VREFO_CKSEL_DIV2                    1                /*!< VREFO_CKSEL = CLK/2*/

/*MMC VPUMP BYP input marcros define*/
#define MD_MMC_VPUMP_BYP_VPUMP                     0                /*!< VPUMP*/
#define MD_MMC_VPUMP_BYP_VDD                       1                /*!< Bypass*/

/*MMC VPUMP input marcros define*/
#define MD_MMC_SEL_VPUMP_3V8                       0                /*!< VPUMP = 3.8V*/
#define MD_MMC_SEL_VPUMP_4V0                       1                /*!< VPUMP = 4.0V*/
#define MD_MMC_SEL_VPUMP_4V2                       2                /*!< VPUMP = 4.2V*/
#define MD_MMC_SEL_VPUMP_4V4                       3                /*!< VPUMP = 4.4V*/

/*MMC VDDA input marcros define*/
#define MD_MMC_SEL_VDDA_3V6                         0                /*!< VDDA = 3.6V*/
#define MD_MMC_SEL_VDDA_3V2                         1                /*!< VDDA = 3.2V*/
#define MD_MMC_SEL_VDDA_2V8                         2                /*!< VDDA = 2.8V*/
#define MD_MMC_SEL_VDDA_2V4                         3                /*!< VDDA = 2.4V*/

/*MMC GREF input marcros define*/
#define MD_MMC_SEL_GREF_FLOATING                   0                /*!< GREF = FLOATING*/
#define MD_MMC_SEL_GREF_0V1_VDDA                   1                /*!< GREF = 0.1V*/
#define MD_MMC_SEL_GREF_0V3_VDDA                   2                /*!< GREF = 0.3V*/
#define MD_MMC_SEL_GREF_0V5_VDDA                   3                /*!< GREF = 0.5V*/

/*MMC VREFO CKSEL input marcros define*/
#define MD_MMC_VRG_K2_K2B                       0                /*!< Select K2B : (VREFO-VSS)>(2GREF-(GREF[P/N]<X> - GREF) = K2B(VREFO - GREF) + (K2B - 1)(GREF - VSS)*/
#define MD_MMC_VRG_K2_K2                        1                /*!< Select K2  : (VREFO-VSS)<(2GREF-(GREF[P/N]<X> - GREF) = K2(VREFO - GREF)*/

/* 0x04  PA_SWCTRL  - PAD Switch Control  */
#define SS0(y)  ((y&0x01)<<0)
#define FB0(y)  ((y&0x01)<<1)    /*FS0*/
#define OP0(y)  ((y&0x01)<<2)    /*DS0*/
#define PW0(y)  ((y&0x01)<<3)    /*PS0*/
#define SS1(y)  ((y&0x01)<<4)
#define FB1(y)  ((y&0x01)<<5)    /*FS1*/
#define OP1(y)  ((y&0x01)<<6)    /*DS1*/
#define PW1(y)  ((y&0x01)<<7)    /*PS1*/
#define SS2(y)  ((y&0x01)<<8)
#define FB2(y)  ((y&0x01)<<9)    /*FS2*/
#define OP2(y)  ((y&0x01)<<10)   /*DS2*/
#define PW2(y)  ((y&0x01)<<11)   /*PS2*/
#define SS3(y)  ((y&0x01)<<12)
#define FB3(y)  ((y&0x01)<<13)   /*FS3*/
#define OP3(y)  ((y&0x01)<<14)   /*DS3*/
#define PW3(y)  ((y&0x01)<<15)   /*PS3*/
#define SS4(y)  ((y&0x01)<<16)
#define FB4(y)  ((y&0x01)<<17)   /*FS4*/
#define OP4(y)  ((y&0x01)<<18)   /*DS4*/
#define PW4(y)  ((y&0x01)<<19)   /*PS4*/
#define SS5(y)  ((y&0x01)<<20)
#define FB5(y)  ((y&0x01)<<21)   /*FS5*/
#define OP5(y)  ((y&0x01)<<22)   /*DS5*/
#define PW5(y)  ((y&0x01)<<23)   /*PS5*/
#define SS6(y)  ((y&0x01)<<24)
#define FB6(y)  ((y&0x01)<<25)   /*FS6*/
#define OP6(y)  ((y&0x01)<<26)   /*DS6*/
#define PW6(y)  ((y&0x01)<<27)   /*PS6*/

/* 0x08  ADC0_CTRL  - MMC1 Control  */
/*AD NFILT input marcros define*/
#define MD_MMC_ADC_CTRL0_AD_NFILT_SENSE            0                /*!< AD_NFILT = SENSE*/
#define MD_MMC_ADC_CTRL0_AD_NFILT_AN8              1                /*!< AD_NFILT = AN8  */
#define MD_MMC_ADC_CTRL0_AD_NFILT_VSS              2                /*!< AD_NFILT = VSS  */
#define MD_MMC_ADC_CTRL0_AD_NFILT_GREF             3                /*!< AD_NFILT = GREF */
#define MD_MMC_ADC_CTRL0_AD_NFILT_BN2              4                /*!< AD_NFILT = BN2  */
#define MD_MMC_ADC_CTRL0_AD_NFILT_BN3              5                /*!< AD_NFILT = BN3  */
#define MD_MMC_ADC_CTRL0_AD_NFILT_BN4              6                /*!< AD_NFILT = BN4  */
#define MD_MMC_ADC_CTRL0_AD_NFILT_BN5              7                /*!< AD_NFILT = BN5  */

/*AD PFILT input marcros define*/
#define MD_MMC_ADC_CTRL0_AD_PFILT_SENSE            0                /*!< AD_PFILT = SENSE*/
#define MD_MMC_ADC_CTRL0_AD_PFILT_FB               1                /*!< AD_PFILT = FB   */
#define MD_MMC_ADC_CTRL0_AD_PFILT_AN8              2                /*!< AD_PFILT = AN8  */
#define MD_MMC_ADC_CTRL0_AD_PFILT_NON3             3                /*!< AD_PFILT = NONE */
#define MD_MMC_ADC_CTRL0_AD_PFILT_NON4             4                /*!< AD_PFILT = NONE */
#define MD_MMC_ADC_CTRL0_AD_PFILT_VDDA             5                /*!< AD_PFILT = VDDA */
#define MD_MMC_ADC_CTRL0_AD_PFILT_VREFO            6                /*!< AD_PFILT = VREFO*/
#define MD_MMC_ADC_CTRL0_AD_PFILT_VREF             7                /*!< AD_PFILT = VREF */
#define MD_MMC_ADC_CTRL0_AD_PFILT_BN0              8                /*!< AD_PFILT = BN0  */
#define MD_MMC_ADC_CTRL0_AD_PFILT_BN1              9                /*!< AD_PFILT = BN1  */
#define MD_MMC_ADC_CTRL0_AD_PFILT_BN2              10               /*!< AD_PFILT = BN2  */
#define MD_MMC_ADC_CTRL0_AD_PFILT_BN3              11               /*!< AD_PFILT = BN3  */
#define MD_MMC_ADC_CTRL0_AD_PFILT_BN4              12               /*!< AD_PFILT = BN4  */
#define MD_MMC_ADC_CTRL0_AD_PFILT_BN5              13               /*!< AD_PFILT = BN5  */
#define MD_MMC_ADC_CTRL0_AD_PFILT_NON14            14               /*!< AD_PFILT = NONE  */
#define MD_MMC_ADC_CTRL0_AD_PFILT_TS               15               /*!< AD_PFILT = TS   */

/*PFILT CH input marcros define*/
#define MD_MMC_ADC_CTRL0_PFILT_CH_100KOHM          0                /*!< PFILT_CH = 100KOHM*/
#define MD_MMC_ADC_CTRL0_PFILT_CH_10KOHM           1                /*!< PFILT_CH = 10KOHM*/
#define MD_MMC_ADC_CTRL0_PFILT_CH_0KOHM            2                /*!< PFILT_CH = 0KOHM*/
#define MD_MMC_ADC_CTRL0_PFILT_CH_NOFILTER         3                /*!< PFILT_CH = NOFILTER*/

/*AD CHOP input marcros define*/
#define MD_MMC_ADC_AD_CHOP_OFF                     0                /*!< AD_CHOP = OFF*/
#define MD_MMC_ADC_AD_CHOP_ON                      1                /*!< AD_CHOP = ON*/
#define MD_MMC_ADC_AD_CHOP_DIGITAL_CHOPPER_FILTER  2                /*!< AD_CHOP = DIGITAL_CHOPPER_FILTER*/
#define MD_MMC_ADC_AD_CHOP_INPUT_CHOPPER_ON        3                /*!< AD_CHOP = INPUT_CHOPPER_ON*/

/*SEL FRQ input marcros define*/
#define MD_MMC_ADC_SEL_FRQ_FREF_DIV_10             0                /*!< SEL_FRQ = Fref/10*/
#define MD_MMC_ADC_SEL_FRQ_FREF_DIV_20             1                /*!< SEL_FRQ = Fref/20*/
#define MD_MMC_ADC_SEL_FRQ_FREF_DIV_40             2                /*!< SEL_FRQ = Fref/40*/
#define MD_MMC_ADC_SEL_FRQ_NO                      3                /*!< SEL_FRQ = NO*/

/*SEL NADR input marcros define*/
#define MD_MMC_ADC_SEL_NADR_FB                     0                /*!< SEL_NADR = FB  */
#define MD_MMC_ADC_SEL_NADR_BN3                    1                /*!< SEL_NADR = BN3 */
#define MD_MMC_ADC_SEL_NADR_VREF                   2                /*!< SEL_NADR = VREF*/
#define MD_MMC_ADC_SEL_NADR_BN5                    3                /*!< SEL_NADR = BN5 */
#define MD_MMC_ADC_SEL_NADR_AN8                    4                /*!< SEL_NADR = AN8 */
#define MD_MMC_ADC_SEL_NADR_VSS                    5                /*!< SEL_NADR = VSS */
#define MD_MMC_ADC_SEL_NADR_GREF                   6                /*!< SEL_NADR = GREF*/
#define MD_MMC_ADC_SEL_NADR_NON7                   7                /*!< SEL_NADR = NONE*/

/*SEL PADR input marcros define*/
#define MD_MMC_ADC_SEL_PADR_FB                     0                /*!< SEL_PADR = FB   */
#define MD_MMC_ADC_SEL_PADR_VREFO                  1                /*!< SEL_PADR = VREFO*/
#define MD_MMC_ADC_SEL_PADR_VREF                   2                /*!< SEL_PADR = VREF */
#define MD_MMC_ADC_SEL_PADR_BN4                    3                /*!< SEL_PADR = BN4  */
#define MD_MMC_ADC_SEL_PADR_AN8                    4                /*!< SEL_PADR = AN8  */
#define MD_MMC_ADC_SEL_PADR_VDDA                   5                /*!< SEL_PADR = VDDA */
#define MD_MMC_ADC_SEL_PADR_GREF                   6                /*!< SEL_PADR = GREF */
#define MD_MMC_ADC_SEL_PADR_NON7                   7                /*!< SEL_PADR = NONE */

/*SEL ADI input marcros define*/
#define MD_MMC_ADC_SEL_ADI_ADSFP_AND_ADSFN         0                /*!< SEL_ADI = ADSFP_AND_ADSFN*/
#define MD_MMC_ADC_SEL_ADI_FB_AND_AN8              1                /*!< SEL_ADI = FB_AND_AN8*/
#define MD_MMC_ADC_SEL_ADI_TSP_AND_TSN             2                /*!< SEL_ADI = TSP_AND_TSN*/
#define MD_MMC_ADC_SEL_ADI_TSN_AND_TSP             3                /*!< SEL_ADI = TSN_AND_TSP*/

/*SEL ADI GAIN input marcros define*/
#define MD_MMC_ADC_ADR_GAIN_1X0                    0                /*!< ADR_GAIN = 1.0x*/
#define MD_MMC_ADC_ADR_GAIN_0X33                   1                /*!< ADR_GAIN = 0.33x*/

/*SEL ADI GAIN input marcros define*/
#define MD_MMC_ADC_ADI_GAIN_X1                    0                /*!< ADI_GAIN = x1*/
#define MD_MMC_ADC_ADI_GAIN_X2                    1                /*!< ADI_GAIN = x2*/
#define MD_MMC_ADC_ADI_GAIN_X3                    2                /*!< ADI_GAIN = x3*/
#define MD_MMC_ADC_ADI_GAIN_X4                    3                /*!< ADI_GAIN = x4*/

#define MD_MMC_ADC_RUN    0
#define MD_MMC_ADC_RESET  1

#define EN_AD(y)        ((y&0x01)<<0)
#define ADI_GAIN(y)     ((y&0x03)<<1)
#define ADR_GAIN(y)     ((y&0x01)<<3)
#define ADI_PBUF(y)     ((y&0x01)<<4)
#define ADI_NBUF(y)     ((y&0x01)<<5)
#define ADR_PBUF(y)     ((y&0x01)<<6)
#define ADR_NBUF(y)     ((y&0x01)<<7)
#define SEL_ADI(y)      ((y&0x03)<<8)
#define SEL_PADR(y)     ((y&0x07)<<10)
#define SEL_NADR(y)     ((y&0x07)<<13)
#define SEL_FRQ(y)      ((y&0x03)<<16)
#define SEL_VCM(y)      ((y&0x01)<<18)
#define EN_TS(y)        ((y&0x01)<<19)
#define AD_CHOP(y)      ((y&0x03)<<20)
#define PFILT_CH(y)     ((y&0x03)<<22)
#define AD_PFILT(y)     ((y&0x0F)<<24)
#define AD_NFILT(y)     ((y&0x07)<<28)
#define RST_AD(y)       ((y&0x01)<<31)

/* 0x0C  ADC1_CTRL  - MMC2 Control  */
/*CHOP FREQ input marcros define*/
#define MD_MMC_ADC_CHOP_FREQ_1XOSR                 0                /*!< CHOP_FREQ = 1XOSR*/
#define MD_MMC_ADC_CHOP_FREQ_2XOSR                 1                /*!< CHOP_FREQ = 2XOSR*/
#define MD_MMC_ADC_CHOP_FREQ_4XOSR                 2                /*!< CHOP_FREQ = 4XOSR*/
#define MD_MMC_ADC_CHOP_FREQ_8XOSR                 3                /*!< CHOP_FREQ = 8XOSR*/

/*AD PGA SEL input marcros define*/
#define MD_MMC_ADC_AD_PGA_SEL_X1                   0                /*!< AD_PGA_SEL = X1 */
#define MD_MMC_ADC_AD_PGA_SEL_X2                   1                /*!< AD_PGA_SEL = X2 */
#define MD_MMC_ADC_AD_PGA_SEL_X3                   2                /*!< AD_PGA_SEL = X3 */
#define MD_MMC_ADC_AD_PGA_SEL_X4                   3                /*!< AD_PGA_SEL = X4 */
#define MD_MMC_ADC_AD_PGA_SEL_X6                   4                /*!< AD_PGA_SEL = X6 */
#define MD_MMC_ADC_AD_PGA_SEL_X9                   5                /*!< AD_PGA_SEL = X9 */
#define MD_MMC_ADC_AD_PGA_SEL_X12                  6                /*!< AD_PGA_SEL = X12*/
#define MD_MMC_ADC_AD_PGA_SEL_X16                  7                /*!< AD_PGA_SEL = X16*/

/*RMS MODE input marcros define*/
#define MD_MMC_ADC_RMS_MODE_00                     0                /*!< RMS_MODE = 0 */
#define MD_MMC_ADC_RMS_MODE_01                     1                /*!< RMS_MODE = 1 */
#define MD_MMC_ADC_RMS_MODE_02                     2                /*!< RMS_MODE = 2 */
#define MD_MMC_ADC_RMS_MODE_03                     3                /*!< RMS_MODE = 3 */

/*SEL LPFOSR input marcros define*/
#define MD_MMC_ADC_SEL_LPFOSR_256                  0                /*!< SEL_LPFOSR = 256  */
#define MD_MMC_ADC_SEL_LPFOSR_512                  1                /*!< SEL_LPFOSR = 512  */
#define MD_MMC_ADC_SEL_LPFOSR_1024                 2                /*!< SEL_LPFOSR = 1024 */
#define MD_MMC_ADC_SEL_LPFOSR_2048                 3                /*!< SEL_LPFOSR = 2048 */

/*AD OSR input marcros define*/
#define MD_MMC_ADC_AD_OSR_32                       0                /*!< AD_OSR = 32    */
#define MD_MMC_ADC_AD_OSR_64                       1                /*!< AD_OSR = 64    */
#define MD_MMC_ADC_AD_OSR_128                      2                /*!< AD_OSR = 128   */
#define MD_MMC_ADC_AD_OSR_256                      3                /*!< AD_OSR = 256   */
#define MD_MMC_ADC_AD_OSR_2500                     4                /*!< AD_OSR = 2500  */
#define MD_MMC_ADC_AD_OSR_5000                     5                /*!< AD_OSR = 5000  */
#define MD_MMC_ADC_AD_OSR_10000                    6                /*!< AD_OSR = 10000 */
#define MD_MMC_ADC_AD_OSR_20000                    7                /*!< AD_OSR = 20000 */

#define EN_COMB(y)      ((y&0x01)<<0)
#define EN_LPF(y)       ((y&0x01)<<1)
#define EN_RMS(y)       ((y&0x01)<<2)
#define EN_PKHCMP(y)    ((y&0x01)<<3)
#define RST_COMB(y)     ((y&0x01)<<4)
#define RST_LPF(y)      ((y&0x01)<<5)
#define RST_RMS(y)      ((y&0x01)<<6)
#define RST_PKHCMP(y)   ((y&0x01)<<7)
#define AD_OSR(y)       ((y&0x07)<<8)
#define SEL_LPFOS(y)    ((y&0x03)<<11)
#define RMS_MODE(y)     ((y&0x03)<<13)
#define EN_DROOP(y)     ((y&0x01)<<15)
#define AD_PGA_SEL(y)   ((y&0x07)<<16)
#define AD_ACMBUF(y)    ((y&0x01)<<19)
#define AD_BUFCHOP(y)   ((y&0x01)<<20)
#define CHOP_FREQ(y)    ((y&0x03)<<21)
#define DECIM_MODE(y)   ((y&0x01)<<23)
#define SKIP_CNTINT(y)  ((y&0x0F)<<24)
#define AVFILT_MODE(y)  ((y&0x07)<<28)
#define ADRST_DIG(y)    ((y&0x01)<<31)


/* 0x10  CMP_CTRL   - LVD Control */
/*CMP HIGH input marcros define*/
#define MD_MMC_CMP_CMP_HIGH_VDSC16                 0                /*!< CMP_HIGH = VDSC16 */
#define MD_MMC_CMP_CMP_HIGH_VDSC13                 1                /*!< CMP_HIGH = VDSC13 */
#define MD_MMC_CMP_CMP_HIGH_VDSC11                 2                /*!< CMP_HIGH = VDSC11 */
#define MD_MMC_CMP_CMP_HIGH_VDSC10                 3                /*!< CMP_HIGH = VDSC10 */
#define MD_MMC_CMP_CMP_HIGH_VDSC9                  4                /*!< CMP_HIGH = VDSC9  */
#define MD_MMC_CMP_CMP_HIGH_VDSC8                  5                /*!< CMP_HIGH = VDSC8  */
#define MD_MMC_CMP_CMP_HIGH_VDSC7                  6                /*!< CMP_HIGH = VDSC7  */
#define MD_MMC_CMP_CMP_HIGH_BN5                    7                /*!< CMP_HIGH = BN5    */
#define MD_MMC_CMP_CMP_HIGH_GREFP6                 8                /*!< CMP_HIGH = GREFP6 */
#define MD_MMC_CMP_CMP_HIGH_GREFP5                 9                /*!< CMP_HIGH = GREFP5 */
#define MD_MMC_CMP_CMP_HIGH_GREFP4                 10               /*!< CMP_HIGH = GREFP4 */
#define MD_MMC_CMP_CMP_HIGH_GREFP3                 11               /*!< CMP_HIGH = GREFP3 */
#define MD_MMC_CMP_CMP_HIGH_GREFP2                 12               /*!< CMP_HIGH = GREFP2 */
#define MD_MMC_CMP_CMP_HIGH_GREFP1                 13               /*!< CMP_HIGH = GREFP1 */
#define MD_MMC_CMP_CMP_HIGH_GREFP0                 14               /*!< CMP_HIGH = GREFP0 */
#define MD_MMC_CMP_CMP_HIGH_GREFN1                 15               /*!< CMP_HIGH = GREFN1 */

/*CMP LOW input marcros define*/
#define MD_MMC_CMP_CMP_LOW_VDSC2                    0               /*!< CMP_LOW = VDSC2  */
#define MD_MMC_CMP_CMP_LOW_VDSC5                    1               /*!< CMP_LOW = VDSC5  */
#define MD_MMC_CMP_CMP_LOW_VDSC7                    2               /*!< CMP_LOW = VDSC7  */
#define MD_MMC_CMP_CMP_LOW_VDSC8                    3               /*!< CMP_LOW = VDSC8  */
#define MD_MMC_CMP_CMP_LOW_VDSC9                    4               /*!< CMP_LOW = VDSC9  */
#define MD_MMC_CMP_CMP_LOW_VDSC10                   5               /*!< CMP_LOW = VDSC10 */
#define MD_MMC_CMP_CMP_LOW_VDSC11                   6               /*!< CMP_LOW = VDSC11 */
#define MD_MMC_CMP_CMP_LOW_BN4                      7               /*!< CMP_LOW = BN4    */
#define MD_MMC_CMP_CMP_LOW_GREFN6                   8               /*!< CMP_LOW = GREFN6 */
#define MD_MMC_CMP_CMP_LOW_GREFN5                   9               /*!< CMP_LOW = GREFN5 */
#define MD_MMC_CMP_CMP_LOW_GREFN4                   10              /*!< CMP_LOW = GREFN4 */
#define MD_MMC_CMP_CMP_LOW_GREFN3                   11              /*!< CMP_LOW = GREFN3 */
#define MD_MMC_CMP_CMP_LOW_GREFN2                   12              /*!< CMP_LOW = GREFN2 */
#define MD_MMC_CMP_CMP_LOW_GREFN1                   13              /*!< CMP_LOW = GREFN1 */
#define MD_MMC_CMP_CMP_LOW_GREFP0                   14              /*!< CMP_LOW = GREFP0 */
#define MD_MMC_CMP_CMP_LOW_GREFP1                   15              /*!< CMP_LOW = GREFP1 */

/*CMP MID input marcros define*/
#define MD_MMC_CMP_CMP_MID_SENSE                    0               /*!< CMP_MID = SENSE */
#define MD_MMC_CMP_CMP_MID_FB                       1               /*!< CMP_MID = FB    */
#define MD_MMC_CMP_CMP_MID_BN2                      2               /*!< CMP_MID = BN2   */
#define MD_MMC_CMP_CMP_MID_BN0                      3               /*!< CMP_MID = BN0   */
#define MD_MMC_CMP_CMP_MID_BN1                      4               /*!< CMP_MID = BN1   */
#define MD_MMC_CMP_CMP_MID_AN7                      5               /*!< CMP_MID = AN7   */
#define MD_MMC_CMP_CMP_MID_BN3                      6               /*!< CMP_MID = BN3   */
#define MD_MMC_CMP_CMP_MID_BN4                      7               /*!< CMP_MID = BN4   */

#define EN_CMP(y)     ((y&0x01)<<0)
#define HYS_CMP(y)    ((y&0x01)<<1)
#define CMP_MID(y)    ((y&0x07)<<4)
#define CMP_LOW(y)    ((y&0x0F)<<8)
#define CMP_HIGH(y)   ((y&0x0F)<<12)
#define CMP_OT()      (1UL<<24)
#define CMPL_OT()     (1UL<<25)
#define CMPH_OT()     (1UL<<26)

/* 0x14  LVD_CTRL   - LVD Control */
/*LVD SEL NIN input marcros define*/
#define MD_MMC_LVD_SEL_NIN_16OF16_VDD               0               /*!< SEL_NIN = 16/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_15OF16_VDD               1               /*!< SEL_NIN = 15/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_14OF16_VDD               2               /*!< SEL_NIN = 14/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_13OF16_VDD               3               /*!< SEL_NIN = 13/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_12OF16_VDD               4               /*!< SEL_NIN = 12/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_11OF16_VDD               5               /*!< SEL_NIN = 11/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_10OF16_VDD               6               /*!< SEL_NIN = 10/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_09OF16_VDD               7               /*!< SEL_NIN =  9/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_08OF16_VDD               8               /*!< SEL_NIN =  8/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_07OF16_VDD               9               /*!< SEL_NIN =  7/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_06OF16_VDD               10              /*!< SEL_NIN =  6/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_05OF16_VDD               11              /*!< SEL_NIN =  5/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_04OF16_VDD               12              /*!< SEL_NIN =  4/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_03OF16_VDD               13              /*!< SEL_NIN =  3/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_02OF16_VDD               14              /*!< SEL_NIN =  2/16 X VDD */
#define MD_MMC_LVD_SEL_NIN_01OF16_VDD               15              /*!< SEL_NIN =  1/16 X VDD */

/*LVD SEL PIN input marcros define*/
#define MD_MMC_LVD_SEL_PIN_1V2                       0               /*!< SEL_PIN = 1.2V   */
#define MD_MMC_LVD_SEL_PIN_CMPIN0                    1               /*!< SEL_PIN = CMPIN0 */
#define MD_MMC_LVD_SEL_PIN_CMPIN1                    2               /*!< SEL_PIN = CMPIN1 */
#define MD_MMC_LVD_SEL_PIN_CMPIN2                    3               /*!< SEL_PIN = CMPIN2 */
#define MD_MMC_LVD_SEL_PIN_CMPIN3                    4               /*!< SEL_PIN = CMPIN3 */
#define MD_MMC_LVD_SEL_PIN_ACM                       5               /*!< SEL_PIN = ACM    */

/*LVD LVD MODE input marcros define*/
#define MD_MMC_LVD_LVD_MODE_BATTERY_OFF              0               /*!< LVD_MODE = Battery OFF     */
#define MD_MMC_LVD_LVD_MODE_12OF201_VDD              1               /*!< LVD_MODE = 1.2/2.01 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF211_VDD              2               /*!< LVD_MODE = 1.2/2.11 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF221_VDD              3               /*!< LVD_MODE = 1.2/2.21 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF231_VDD              4               /*!< LVD_MODE = 1.2/2.31 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF241_VDD              5               /*!< LVD_MODE = 1.2/2.41 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF251_VDD              6               /*!< LVD_MODE = 1.2/2.51 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF261_VDD              7               /*!< LVD_MODE = 1.2/2.61 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF271_VDD              8               /*!< LVD_MODE = 1.2/2.71 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF281_VDD              9               /*!< LVD_MODE = 1.2/2.81 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF291_VDD              10              /*!< LVD_MODE = 1.2/2.91 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF301_VDD              11              /*!< LVD_MODE = 1.2/3.01 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF311_VDD              12              /*!< LVD_MODE = 1.2/3.11 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF321_VDD              13              /*!< LVD_MODE = 1.2/3.21 X VDD */
#define MD_MMC_LVD_LVD_MODE_12OF331_VDD              14              /*!< LVD_MODE = 1.2/3.31 X VDD */
#define MD_MMC_LVD_LVD_MODE_LVDIN                    15              /*!< LVD_MODE = LVDIN           */

#define EN_LVD(y)     ((y&0x01)<<0)
#define HYS_LVD(y)    ((y&0x01)<<1)
#define LVD_MODE(y)   ((y&0x0F)<<4)
#define SEL_PIN(y)    ((y&0x07)<<8)
#define SEL_NIN(y)    ((y&0x0F)<<12)
#define ST_LDO()      (1UL<<24)

/* 0x20  CNT_CTRL   - Frequency Counter Control */
#define MD_MMC_CNT_CTRL_EN_FQCNTI_CMPO             0
#define MD_MMC_CNT_CTRL_EN_FQCNTI_PCNTI            1

#define MD_MMC_CNT_CTRL_FQCNT_DIV_DIV_1            0                /*!< FQCNT_DIV = 0           */
#define MD_MMC_CNT_CTRL_FQCNT_DIV_DIV_2            4                /*!< FQCNT_DIV = 2           */
#define MD_MMC_CNT_CTRL_FQCNT_DIV_DIV_4            5                /*!< FQCNT_DIV = 4           */
#define MD_MMC_CNT_CTRL_FQCNT_DIV_DIV_8            6                /*!< FQCNT_DIV = 8           */
#define MD_MMC_CNT_CTRL_FQCNT_DIV_DIV_16           7                /*!< FQCNT_DIV = 16           */

#define INIT_CNTA(y)  ((y&0xFFFF)<<8)
#define EN_FQCNT(y)   ((y&0x01)<<24)
#define EN_FQCNTI(y)  ((y&0x01)<<25)
#define FQCNT_DIV(y)  ((y&0x07)<<28)
#define RST_FQCNT(y)  ((y&0x01)<<31)

/* 0x40  RMS_VAL        - Root Mean Square   */
#define RMS_VAL()  (0x00FFFFFFUL<<8)
#define RdMmcRMS_VAL()     (MMC->RMS_VAL >> 8)

/* 0x44  LPF_VAL        - Low Pass Filter Value   */
#define LPF_VAL()  (0x00FFFFFFUL<<8)
#define RdMmcLPF_VAL()     (MMC->LPF_VAL >> 8)

/* 0x48  ADC_VAL        - ADC Value   */
#define ADC_VAL()  (0x00FFFFFFUL<<8)
#define RdMmcADC_VAL()     (MMC->ADC_VAL >> 8)

/* 0x50  IER        - Interrupt Enable Register   */
/* 0x54  IDR        - Interrupt Disable Register  */
/* 0x64  ICR        - Interrupt Clear Register    */
#define ADC_DONE(y)       ((y&0x01)<<0)
#define LPF_DONE(y)       ((y&0x01)<<1)
#define RMS_DONE(y)       ((y&0x01)<<2)
#define PKH_DONE(y)       ((y&0x01)<<3)
#define CMPH_CHG(y)       ((y&0x01)<<8)
#define CMPL_CHG(y)       ((y&0x01)<<9)
#define CMPO_CHG(y)       ((y&0x01)<<10)
#define LVDO_CHG(y)       ((y&0x01)<<11)
#define FRQCNT_END(y)     ((y&0x01)<<12)
#define FRQCNT_DONE(y)    ((y&0x01)<<13)
#define FRQCNT_ABN(y)     ((y&0x01)<<14)
#define CNTB_OV(y)        ((y&0x01)<<15)

#define MMC_CTRL_VREFO_CKSEL_DIV_2  (0)
#define MMC_CTRL_VREFO_CKSEL_DIV_4  (1)

#define MMC_CTRL_VPUMP_BYP_VPUMP    (0)
#define MMC_CTRL_VPUMP_BYP_VDD      (1)

#define MMC_CTRL_VRG_K2_SELECT_K2B  (0)
#define MMC_CTRL_VRG_K2_SELECT_K2   (1)

#define MMC_CTRL_VREFO_SRC_BAND_GAP (0)
#define MMC_CTRL_VREFO_SRC_BN4      (1)

#define MMC_CTRL_BN02_CTRL_OPEN     (0)
#define MMC_CTRL_BN02_CTRL_SHORT    (1)

#define MMC_ADC_CTRL0_SEL_VCM_VDDA_DIV2   (0)
#define MMC_ADC_CTRL0_SEL_VCM_1P2V        (1)

#define MMC_ADC_CTRL1_DECIM_MODE_2_CIC      (0)
#define MMC_ADC_CTRL1_DECIM_MODE_4_CIC      (1)

#define MMC_CNT_CTRL_EN_FQCNTI_CMPO         (0)
#define MMC_CNT_CTRL_EN_FQCNTI_PCNTI        (1)

/**
  * @} MD_MMC_Public_Constants
  */

/* Public Macro ------------------------------------------------------------*/
/** @defgroup MD_MMC_Public_Macro Public Macro
  * @brief MMC module driver
  * @{
  */
/**
  * @brief  MMC_CTRL Set.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_ctrl_reg(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->CTRL, DATA);
}


/**
  * @brief  MMC_CTRL Get.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_get_ctrl_reg(MMC_TypeDef *mmc)
{
    READ_REG(mmc->CTRL);
}

/**
  * @brief  MMC function op enable.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_func_op(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CTRL, MMC_CTRL_EN_FUNC_OP_MSK);
}

/**
  * @brief  MMC function op disable.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_func_op(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CTRL, MMC_CTRL_EN_FUNC_OP_MSK);
}

/**
  * @brief  Indicates whether the MMC function op is enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_func_op(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_EN_FUNC_OP_MSK) == (MMC_CTRL_EN_FUNC_OP_MSK));
}

/**
  * @brief  MMC function refo clock
  * @param  mmc MMC instance
    @param  MMC_CTRL_VREFO_CKSEL_DIV_2
    @param  MMC_CTRL_VREFO_CKSEL_DIV_4
  * @retval None
  */

__STATIC_INLINE void md_mmc_set_vrefo_clk(MMC_TypeDef *mmc, uint32_t div)
{
    if (div == MMC_CTRL_VREFO_CKSEL_DIV_4)
    {
        SET_BIT(mmc->CTRL, MMC_CTRL_VREFO_CKSEL_MSK);
    }
    else
    {
        CLEAR_BIT(mmc->CTRL, MMC_CTRL_VREFO_CKSEL_MSK);
    }
}

/**
  * @brief  Get MMC function refo clock
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_vrefo_clk(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_VREFO_CKSEL_MSK) >> MMC_CTRL_VREFO_CKSEL_POS);
}

/**
  * @brief  MMC function refo enable.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_vrefo_chop(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CTRL, MMC_CTRL_VREFO_CHOP_MSK);
}

/**
  * @brief  MMC function refo disable.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_vrefo_chop(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CTRL, MMC_CTRL_VREFO_CHOP_MSK);
}

/**
  * @brief  Indicates whether the MMC function refo is enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_vrefo_chop(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_VREFO_CHOP_MSK) == (MMC_CTRL_VREFO_CHOP_MSK));
}


/**
  * @brief  MMC function VPUMP bypass.
  * @param  mmc MMC instance
  * @param  MMC_CTRL_VPUMP_BYP_VPUMP
  * @param  MMC_CTRL_VPUMP_BYP_VDD
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_vpump_byp(MMC_TypeDef *mmc, uint32_t voltage)
{
    if (voltage == MMC_CTRL_VPUMP_BYP_VDD)
    {
        SET_BIT(mmc->CTRL, MMC_CTRL_VPUMP_BYP_MSK);
    }
    else
    {
        CLEAR_BIT(mmc->CTRL, MMC_CTRL_VPUMP_BYP_MSK);
    }
}

/**
  * @brief  Indicates whether the MMC function CP is bypassed
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_vpump_byp(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_VPUMP_BYP_MSK) == (MMC_CTRL_VPUMP_BYP_MSK));
}

/**
  * @brief  Set MMC function VPUMP select.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_vpump(MMC_TypeDef *mmc, uint32_t vpump)
{
    MODIFY_REG(mmc->CTRL, MMC_CTRL_SEL_VPUMP_MSK, vpump << MMC_CTRL_SEL_VPUMP_POSS);
}

/**
  * @brief  Get MMC function VPUMP select.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_vpump(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_SEL_VPUMP_MSK) >> MMC_CTRL_SEL_VPUMP_POSS);
}

/**
  * @brief  MMC function linear regulator floating switch enable.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_vdda_float(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CTRL, MMC_CTRL_EN_VDDA_FLOAT_MSK);
}

/**
  * @brief  MMC function linear regulator floating switch disable.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_vdda_float(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CTRL, MMC_CTRL_EN_VDDA_FLOAT_MSK);
}

/**
  * @brief  Indicates whether the MMC function linear regulator floating switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_vdda_float(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_EN_VDDA_FLOAT_MSK) == (MMC_CTRL_EN_VDDA_FLOAT_MSK));
}

/**
  * @brief  Set MMC function mode select.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_mode_sel(MMC_TypeDef *mmc, uint32_t mode)
{
    MODIFY_REG(mmc->CTRL, MMC_CTRL_MODE_MSK, mode << MMC_CTRL_MODE_POSS);
}

/**
  * @brief  Get MMC function mode select.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_mode_sel(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_MODE_MSK) >> MMC_CTRL_MODE_POSS);
}

/**
  * @brief  Set MMC function vref select.
  * @param  mmc MMC instance
  * @param  vref
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_vref_sel(MMC_TypeDef *mmc, uint32_t vref)
{
    MODIFY_REG(mmc->CTRL, MMC_CTRL_VREF_SEL_MSK, vref << MMC_CTRL_VREF_SEL_POSS);
}

/**
  * @brief  Get MMC function vref select.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_vref_sel(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_VREF_SEL_MSK) >> MMC_CTRL_VREF_SEL_POSS);
}

/**
  * @brief  Set MMC function ldo select.
  * @param  mmc MMC instance
  * @param  MD_MMC_SEL_VDDA_3V6
  * @param  MD_MMC_SEL_VDDA_3V2
  * @param  MD_MMC_SEL_VDDA_2V8
  * @param  MD_MMC_SEL_VDDA_2V4
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_vdda(MMC_TypeDef *mmc, uint32_t vref)
{
    MODIFY_REG(mmc->CTRL, MMC_CTRL_SEL_VDDA_MSK, vref << MMC_CTRL_SEL_VDDA_POSS);
}

/**
  * @brief  Get MMC function ldo select.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_vdda(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_SEL_VDDA_MSK) >> MMC_CTRL_SEL_VDDA_POSS);
}

/**
  * @brief  Set MMC function agnd select.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_gref(MMC_TypeDef *mmc, uint32_t gref)
{
    MODIFY_REG(mmc->CTRL, MMC_CTRL_SEL_GREF_MSK, gref << MMC_CTRL_SEL_GREF_POSS);
}

/**
  * @brief  Get MMC function agnd select.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_gref(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_SEL_GREF_MSK) >> MMC_CTRL_SEL_GREF_POSS);
}

/**
  * @brief  MMC function reference voltage select
  * @param  mmc MMC instance
  * @param  MMC_CTRL_VRG_K2_SELECT_K2B
  * @param  MMC_CTRL_VRG_K2_SELECT_K2
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_vrg_k2(MMC_TypeDef *mmc, uint32_t select)
{
    if (select == MMC_CTRL_VRG_K2_SELECT_K2)
    {
        SET_BIT(mmc->CTRL, MMC_CTRL_VRG_K2_MSK);
    }
    else
    {
        CLEAR_BIT(mmc->CTRL, MMC_CTRL_VRG_K2_MSK);
    }
}

/**
  * @brief  Indicates whether the MMC function reference voltage selection
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_vrg_k2(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_VRG_K2_MSK) == (MMC_CTRL_VRG_K2_MSK));
}

/**
  * @brief  MMC function refo source
  * @param  mmc MMC instance
  * @param  MMC_CTRL_VREFO_SRC_BN4
  * @param  MMC_CTRL_VREFO_SRC_BAND_GAP
  * @retval None
  */

__STATIC_INLINE void md_mmc_set_vref_src(MMC_TypeDef *mmc, uint32_t source)
{
    if (source == MMC_CTRL_VREFO_SRC_BN4)
    {
        SET_BIT(mmc->CTRL, MMC_CTRL_VREF_SRC_MSK);
    }
    else
    {
        CLEAR_BIT(mmc->CTRL, MMC_CTRL_VREF_SRC_MSK);
    }
}

/**
  * @brief  Indicates whether the MMC function refo source selection
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_vref_src(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_VREF_SRC_MSK) >> MMC_CTRL_VREF_SRC_POS);
}


/**
  * @brief  MMC function PB<0> and PB<2> open
  * @param  mmc MMC instance
  * @param  MMC_CTRL_BN02_CTRL_OPEN
  * @param  MMC_CTRL_BN02_CTRL_SHORT
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_bn02_ctrl(MMC_TypeDef *mmc, uint32_t select)
{
    if (select == MMC_CTRL_BN02_CTRL_OPEN)
    {
        CLEAR_BIT(mmc->CTRL, MMC_CTRL_BN02_CTRL_MSK);
    }
    else
    {
        SET_BIT(mmc->CTRL, MMC_CTRL_BN02_CTRL_MSK);
    }
}

/**
  * @brief  Indicates whether the MMC function PB<0> and PB<2> connect
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_bn02_ctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_BN02_CTRL_MSK) == (MMC_CTRL_BN02_CTRL_MSK));
}

/**
  * @brief  MMC function vpump enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_vpump(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CTRL, MMC_CTRL_EN_VPUMP_MSK);
}

/**
  * @brief  MMC function vpump disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_vpump(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CTRL, MMC_CTRL_EN_VPUMP_MSK);
}

/**
  * @brief  Indicates whether the MMC function vpump enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_vpump(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_EN_VPUMP_MSK) == (MMC_CTRL_EN_VPUMP_MSK));
}

/**
  * @brief  MMC function ldo enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_vdda(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CTRL, MMC_CTRL_EN_VDDA_MSK);
}

/**
  * @brief  MMC function ldo disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_vdda(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CTRL, MMC_CTRL_EN_VDDA_MSK);
}

/**
  * @brief  Indicates whether the MMC function ldo enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_vdda(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_EN_VDDA_MSK) == (MMC_CTRL_EN_VDDA_MSK));
}

/**
  * @brief  MMC function vrefo enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_vrefo(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CTRL, MMC_CTRL_EN_VREFO_MSK);
}

/**
  * @brief  MMC function vrefo disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_ctrl_en_vrefo(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CTRL, MMC_CTRL_EN_VREFO_MSK);
}

/**
  * @brief  Indicates whether the MMC function vrefo enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_vrefo(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_EN_VREFO_MSK) == (MMC_CTRL_EN_VREFO_MSK));
}

/**
  * @brief  MMC function vrgen enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_vrgen(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CTRL, MMC_CTRL_EN_VRGEN_MSK);
}

/**
  * @brief  MMC function vrgen disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_vrgen(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CTRL, MMC_CTRL_EN_VRGEN_MSK);
}

/**
  * @brief  Indicates whether the MMC function vrgen enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_vrgen(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_EN_VRGEN_MSK) == (MMC_CTRL_EN_VRGEN_MSK));
}

/**
  * @brief  MMC function mmc enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_mmc(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CTRL, MMC_CTRL_EN_MMC_MSK);
}

/**
  * @brief  MMC function mmc disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_mmc(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CTRL, MMC_CTRL_EN_MMC_MSK);
}

/**
  * @brief  Indicates whether the MMC function mmc enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_mmc(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CTRL, MMC_CTRL_EN_MMC_MSK) == (MMC_CTRL_EN_MMC_MSK));
}

/**
  * @brief  MMC_AN_SWCTRL Set.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_an_swctrl_reg(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->AN_SWCTRL, DATA);
}

/**
  * @brief  MMC_AN_SWCTRL Get.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_get_an_swctrl_reg(MMC_TypeDef *mmc)
{
    READ_REG(mmc->AN_SWCTRL);
}

/**
  * @brief  Power Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an6_pwctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_PWCTRL_MSK);
}

/**
  * @brief  Power Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an6_pwctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_PWCTRL_MSK);
}

/**
  * @brief  Indicates whether the Power Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an6_pwctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_PWCTRL_MSK) == (MMC_AN_SWCTRL_AN6_PWCTRL_MSK));
}

/**
  * @brief  OP Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an6_opctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_OPCTRL_MSK);
}

/**
  * @brief  OP Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an6_opctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_OPCTRL_MSK);
}

/**
  * @brief  Indicates whether the OP Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an6_opctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_OPCTRL_MSK) == (MMC_AN_SWCTRL_AN6_OPCTRL_MSK));
}

/**
  * @brief  Feedback Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an6_fbctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_FBCTRL_MSK);
}

/**
  * @brief  Feedback Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an6_fbctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_FBCTRL_MSK);
}

/**
  * @brief  Indicates whether the Feedback Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an6_fbctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_FBCTRL_MSK) == (MMC_AN_SWCTRL_AN6_FBCTRL_MSK));
}

/**
  * @brief  Sense Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an6_ssctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_SSCTRL_MSK);
}

/**
  * @brief  Sense Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an6_ssctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_SSCTRL_MSK);
}

/**
  * @brief  Indicates whether the Sense Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an6_ssctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN6_SSCTRL_MSK) == (MMC_AN_SWCTRL_AN6_SSCTRL_MSK));
}

/**
  * @brief  Power Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an5_pwctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_PWCTRL_MSK);
}

/**
  * @brief  Power Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an5_pwctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_PWCTRL_MSK);
}

/**
  * @brief  Indicates whether the Power Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an5_pwctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_PWCTRL_MSK) == (MMC_AN_SWCTRL_AN5_PWCTRL_MSK));
}

/**
  * @brief  OP Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an5_opctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_OPCTRL_MSK);
}

/**
  * @brief  OP Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an5_opctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_OPCTRL_MSK);
}

/**
  * @brief  Indicates whether the OP Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an5_opctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_OPCTRL_MSK) == (MMC_AN_SWCTRL_AN5_OPCTRL_MSK));
}

/**
  * @brief  Feedback Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an5_fbctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_FBCTRL_MSK);
}

/**
  * @brief  Feedback Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an5_fbctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_FBCTRL_MSK);
}

/**
  * @brief  Indicates whether the Feedback Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an5_fbctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_FBCTRL_MSK) == (MMC_AN_SWCTRL_AN5_FBCTRL_MSK));
}

/**
  * @brief  Sense Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an5_ssctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_SSCTRL_MSK);
}

/**
  * @brief  Sense Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an5_ssctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_SSCTRL_MSK);
}

/**
  * @brief  Indicates whether the Sense Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an5_ssctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN5_SSCTRL_MSK) == (MMC_AN_SWCTRL_AN5_SSCTRL_MSK));
}

/**
  * @brief  Power Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an4_pwctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_PWCTRL_MSK);
}

/**
  * @brief  Power Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an4_pwctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_PWCTRL_MSK);
}

/**
  * @brief  Indicates whether the Power Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an4_pwctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_PWCTRL_MSK) == (MMC_AN_SWCTRL_AN4_PWCTRL_MSK));
}

/**
  * @brief  OP Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an4_opctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_OPCTRL_MSK);
}

/**
  * @brief  OP Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an4_opctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_OPCTRL_MSK);
}

/**
  * @brief  Indicates whether the OP Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an4_opctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_OPCTRL_MSK) == (MMC_AN_SWCTRL_AN4_OPCTRL_MSK));
}

/**
  * @brief  Feedback Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an4_fbctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_FBCTRL_MSK);
}

/**
  * @brief  Feedback Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an4_fbctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_FBCTRL_MSK);
}

/**
  * @brief  Indicates whether the Feedback Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an4_fbctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_FBCTRL_MSK) == (MMC_AN_SWCTRL_AN4_FBCTRL_MSK));
}

/**
  * @brief  Sense Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an4_ssctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_SSCTRL_MSK);
}

/**
  * @brief  Sense Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an4_ssctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_SSCTRL_MSK);
}

/**
  * @brief  Indicates whether the Sense Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an4_ssctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN4_SSCTRL_MSK) == (MMC_AN_SWCTRL_AN4_SSCTRL_MSK));
}

/**
  * @brief  Power Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an3_pwctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_PWCTRL_MSK);
}

/**
  * @brief  Power Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an3_pwctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_PWCTRL_MSK);
}

/**
  * @brief  Indicates whether the Power Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an3_pwctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_PWCTRL_MSK) == (MMC_AN_SWCTRL_AN3_PWCTRL_MSK));
}

/**
  * @brief  OP Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an3_opctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_OPCTRL_MSK);
}

/**
  * @brief  OP Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an3_opctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_OPCTRL_MSK);
}

/**
  * @brief  Indicates whether the OP Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an3_opctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_OPCTRL_MSK) == (MMC_AN_SWCTRL_AN3_OPCTRL_MSK));
}

/**
  * @brief  Feedback Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an3_fbctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_FBCTRL_MSK);
}

/**
  * @brief  Feedback Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an3_fbctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_FBCTRL_MSK);
}

/**
  * @brief  Indicates whether the Feedback Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an3_fbctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_FBCTRL_MSK) == (MMC_AN_SWCTRL_AN3_FBCTRL_MSK));
}

/**
  * @brief  Sense Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an3_ssctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_SSCTRL_MSK);
}

/**
  * @brief  Sense Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an3_ssctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_SSCTRL_MSK);
}

/**
  * @brief  Indicates whether the Sense Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an3_ssctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN3_SSCTRL_MSK) == (MMC_AN_SWCTRL_AN3_SSCTRL_MSK));
}

/**
  * @brief  Power Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an2_pwctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_PWCTRL_MSK);
}

/**
  * @brief  Power Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an2_pwctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_PWCTRL_MSK);
}

/**
  * @brief  Indicates whether the Power Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an2_pwctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_PWCTRL_MSK) == (MMC_AN_SWCTRL_AN2_PWCTRL_MSK));
}

/**
  * @brief  OP Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an2_opctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_OPCTRL_MSK);
}

/**
  * @brief  OP Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an2_opctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_OPCTRL_MSK);
}

/**
  * @brief  Indicates whether the OP Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an2_opctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_OPCTRL_MSK) == (MMC_AN_SWCTRL_AN2_OPCTRL_MSK));
}

/**
  * @brief  Feedback Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an2_fbctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_FBCTRL_MSK);
}

/**
  * @brief  Feedback Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an2_fbctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_FBCTRL_MSK);
}

/**
  * @brief  Indicates whether the Feedback Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an2_fbctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_FBCTRL_MSK) == (MMC_AN_SWCTRL_AN2_FBCTRL_MSK));
}

/**
  * @brief  Sense Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an2_ssctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_SSCTRL_MSK);
}

/**
  * @brief  Sense Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an2_ssctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_SSCTRL_MSK);
}

/**
  * @brief  Indicates whether the Sense Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an2_ssctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN2_SSCTRL_MSK) == (MMC_AN_SWCTRL_AN2_SSCTRL_MSK));
}

/**
  * @brief  Power Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an1_pwctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_PWCTRL_MSK);
}

/**
  * @brief  Power Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an1_pwctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_PWCTRL_MSK);
}

/**
  * @brief  Indicates whether the Power Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an1_pwctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_PWCTRL_MSK) == (MMC_AN_SWCTRL_AN1_PWCTRL_MSK));
}

/**
  * @brief  OP Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an1_opctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_OPCTRL_MSK);
}

/**
  * @brief  OP Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an1_opctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_OPCTRL_MSK);
}

/**
  * @brief  Indicates whether the OP Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an1_opctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_OPCTRL_MSK) == (MMC_AN_SWCTRL_AN1_OPCTRL_MSK));
}

/**
  * @brief  Feedback Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an1_fbctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_FBCTRL_MSK);
}

/**
  * @brief  Feedback Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an1_fbctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_FBCTRL_MSK);
}

/**
  * @brief  Indicates whether the Feedback Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an1_fbctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_FBCTRL_MSK) == (MMC_AN_SWCTRL_AN1_FBCTRL_MSK));
}

/**
  * @brief  Sense Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an1_ssctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_SSCTRL_MSK);
}

/**
  * @brief  Sense Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an1_ssctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_SSCTRL_MSK);
}

/**
  * @brief  Indicates whether the Sense Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an1_ssctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN1_SSCTRL_MSK) == (MMC_AN_SWCTRL_AN1_SSCTRL_MSK));
}

/**
  * @brief  Power Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an0_pwctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_PWCTRL_MSK);
}

/**
  * @brief  Power Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an0_pwctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_PWCTRL_MSK);
}

/**
  * @brief  Indicates whether the Power Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an0_pwctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_PWCTRL_MSK) == (MMC_AN_SWCTRL_AN0_PWCTRL_MSK));
}

/**
  * @brief  OP Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an0_opctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_OPCTRL_MSK);
}

/**
  * @brief  OP Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an0_opctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_OPCTRL_MSK);
}

/**
  * @brief  Indicates whether the OP Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an0_opctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_OPCTRL_MSK) == (MMC_AN_SWCTRL_AN0_OPCTRL_MSK));
}

/**
  * @brief  Feedback Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an0_fbctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_FBCTRL_MSK);
}

/**
  * @brief  Feedback Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an0_fbctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_FBCTRL_MSK);
}

/**
  * @brief  Indicates whether the Feedback Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an0_fbctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_FBCTRL_MSK) == (MMC_AN_SWCTRL_AN0_FBCTRL_MSK));
}

/**
  * @brief  Sense Switch control enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_an0_ssctrl(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_SSCTRL_MSK);
}

/**
  * @brief  Sense Switch control disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_an0_ssctrl(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_SSCTRL_MSK);
}

/**
  * @brief  Indicates whether the Sense Switch control enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_an0_ssctrl(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->AN_SWCTRL, MMC_AN_SWCTRL_AN0_SSCTRL_MSK) == (MMC_AN_SWCTRL_AN0_SSCTRL_MSK));
}

/**
  * @brief  MMC_ADC_CTRL0 Set.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_adc_ctrl0_reg(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->ADC_CTRL0, DATA);
}

/**
  * @brief  MMC_ADC_CTRL0 Get.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_get_adc_ctrl0_reg(MMC_TypeDef *mmc)
{
    READ_REG(mmc->ADC_CTRL0);
}

/**
  * @brief  Set MMC MSK
  * @param  MMCx MMC instance
  * @param  a 32-bit value
            @arg MD_MMC_ADC_RUN
            @arg MD_MMC_ADC_RESET
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_rst_ad(MMC_TypeDef *MMCx, uint32_t value)
{
    MODIFY_REG(MMCx->ADC_CTRL0, MMC_ADC_CTRL0_RST_AD_MSK, value << MMC_ADC_CTRL0_RST_AD_POS);
}

/**
  * @brief  Get MMC MSK
  * @param  MMCx MMC instance
  * @retval a 32-bit value
            @arg MD_MMC_ADC_RUN
            @arg MD_MMC_ADC_RESET
  */
__STATIC_INLINE uint32_t md_mmc_get_rst_ad(MMC_TypeDef *MMCx)
{
    return (uint32_t)(READ_BIT(MMCx->ADC_CTRL0, MMC_ADC_CTRL0_RST_AD_MSK) >> MMC_ADC_CTRL0_RST_AD_POS);
}

/**
  * @brief  AMPINN and AMPINP filter negative input voltage selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_ad_nfilt(MMC_TypeDef *mmc, uint32_t ad_nfilt)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_AD_NFILT_MSK, ad_nfilt << MMC_ADC_CTRL0_AD_NFILT_POSS);
}

/**
  * @brief  Get AMPINN and AMPINP filter positive input voltage.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_ad_nfilt(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_AD_NFILT_MSK) >> MMC_ADC_CTRL0_AD_NFILT_POSS);
}

/**
  * @brief  AMPINN and AMPINP filter negative input voltage selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_ad_pfilt(MMC_TypeDef *mmc, uint32_t ad_pfilt)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_AD_PFILT_MSK, ad_pfilt << MMC_ADC_CTRL0_AD_PFILT_POSS);
}

/**
  * @brief  Get AMPINN and AMPINP filter positive input voltage.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_ad_pfilt(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_AD_PFILT_MSK) >> MMC_ADC_CTRL0_AD_PFILT_POSS);
}

/**
  * @brief  Set Pre-filter selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_ad_pfilt_ch(MMC_TypeDef *mmc, uint32_t pfilt_ch)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_PFILT_CH_MSK, pfilt_ch << MMC_ADC_CTRL0_PFILT_CH_POSS);
}

/**
  * @brief  Get Pre-filter selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_pfilt_ch(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_PFILT_CH_MSK) >> MMC_ADC_CTRL0_PFILT_CH_POSS);
}

/**
  * @brief  Analog-to-digital converter chopper mode selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_ad_chop(MMC_TypeDef *mmc, uint32_t ad_chop)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_AD_CHOP_MSK, ad_chop << MMC_ADC_CTRL0_AD_CHOP_POSS);
}

/**
  * @brief  Get Analog-to-digital converter chopper mode selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_ad_chop(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_AD_CHOP_MSK) >> MMC_ADC_CTRL0_AD_CHOP_POSS);
}

/**
  * @brief  Internal temperature sensing switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_ts(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_EN_TS_MSK);
}

/**
  * @brief  Internal temperature sensing switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_ts(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_EN_TS_MSK);
}

/**
  * @brief  Indicates whether the Internal temperature sensing switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_ts(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_EN_TS_MSK) == (MMC_ADC_CTRL0_EN_TS_MSK));
}

/**
  * @brief  ADC  voltage selection
  * @param  mmc MMC instance
  * @param  MMC_ADC_CTRL0_SEL_VCM_VDDA_DIV2
  * @param  MMC_ADC_CTRL0_SEL_VCM_1_2V
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_vcm(MMC_TypeDef *mmc, uint32_t select)
{
    if (select == MMC_ADC_CTRL0_SEL_VCM_VDDA_DIV2)
    {
        CLEAR_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_VCM_MSK);
    }
    else
    {
        SET_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_VCM_MSK);
    }
}

/**
  * @brief  Indicates ADC  voltage selection
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_vcm(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_VCM_MSK) == (MMC_ADC_CTRL0_EN_TS_MSK));
}

/**
  * @brief  Reference clock frequency selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_frq(MMC_TypeDef *mmc, uint32_t sel_frq)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_FRQ_MSK, sel_frq << MMC_ADC_CTRL0_SEL_FRQ_POSS);
}

/**
  * @brief  Get Reference clock frequency selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_frq(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_FRQ_MSK) >> MMC_ADC_CTRL0_SEL_FRQ_POSS);
}

/**
  * @brief  ADC Negative terminal reference voltage selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_nadr(MMC_TypeDef *mmc, uint32_t sel_nadr)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_NADR_MSK, sel_nadr << MMC_ADC_CTRL0_SEL_NADR_POSS);
}

/**
  * @brief  Get ADC Negative terminal reference voltage selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_nadr(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_NADR_MSK) >> MMC_ADC_CTRL0_SEL_NADR_POSS);
}

/**
  * @brief  ADC Positive terminal reference voltage selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_padr(MMC_TypeDef *mmc, uint32_t sel_padr)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_PADR_MSK, sel_padr << MMC_ADC_CTRL0_SEL_PADR_POSS);
}

/**
  * @brief  Get ADC Positive terminal reference voltage selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_padr(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_PADR_MSK) >> MMC_ADC_CTRL0_SEL_PADR_POSS);
}

/**
  * @brief  ADC input path selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_adi(MMC_TypeDef *mmc, uint32_t sel_adi)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_ADI_MSK, sel_adi << MMC_ADC_CTRL0_SEL_ADI_POSS);
}

/**
  * @brief  Get ADC input path selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_adi(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_SEL_ADI_MSK) >> MMC_ADC_CTRL0_SEL_ADI_POSS);
}

/**
  * @brief  ADC negative reference voltage input buffer switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_adr_nbuf(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADR_NBUF_MSK);
}

/**
  * @brief  ADC negative reference voltage input buffer switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_adr_nbuf(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADR_NBUF_MSK);
}

/**
  * @brief  Indicates whether the ADC negative reference voltage input buffer switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_adr_nbuf(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADR_NBUF_MSK) == (MMC_ADC_CTRL0_ADR_NBUF_MSK));
}

/**
  * @brief  ADC positive reference voltage input buffer switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_adr_pbuf(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADR_PBUF_MSK);
}

/**
  * @brief  ADC positive reference voltage input buffer switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_adr_pbuf(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADR_PBUF_MSK);
}

/**
  * @brief  Indicates whether the ADC positive reference voltage input buffer switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_adr_pbuf(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADR_PBUF_MSK) == (MMC_ADC_CTRL0_ADR_PBUF_MSK));
}

/**
  * @brief  ADC negative input voltage buffer switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_adi_nbuf(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADI_NBUF_MSK);
}

/**
  * @brief  ADC negative input voltage buffer switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_adi_nbuf(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADI_NBUF_MSK);
}

/**
  * @brief  Indicates whether the ADC negative input voltage buffer switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_adi_nbuf(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADI_NBUF_MSK) == (MMC_ADC_CTRL0_ADI_NBUF_MSK));
}

/**
  * @brief  ADC positive input voltage buffer switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_adi_pbuf(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADI_PBUF_MSK);
}

/**
  * @brief  ADC positive input voltage buffer switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_adi_pbuf(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADI_PBUF_MSK);
}

/**
  * @brief  Indicates whether the ADC positive input voltage buffer switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_adi_pbuf(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADI_PBUF_MSK) == (MMC_ADC_CTRL0_ADI_PBUF_MSK));
}

/**
  * @brief  ADC reference voltage gain selection(0.33x / 1x).
  * @param  mmc MMC instance
  * @param  MD_MMC_ADC_ADR_GAIN_1X0
            MD_MMC_ADC_ADR_GAIN_0X33
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_adr_gain(MMC_TypeDef *mmc, uint32_t adr_gain)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADR_GAIN_MSK, adr_gain << MMC_ADC_CTRL0_ADR_GAIN_POS);
}

/**
  * @brief  Get ADC reference voltage gain selection(0.33x / 1x).
  * @param  mmc MMC instance
  * @retval MD_MMC_ADC_ADR_GAIN_1X0
            MD_MMC_ADC_ADR_GAIN_0X33
  */
__STATIC_INLINE uint32_t md_mmc_get_adr_gain(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADR_GAIN_MSK) >> MMC_ADC_CTRL0_ADR_GAIN_POS);
}

/**
  * @brief  ADC input voltage gain selection(0.9x / 1.8x / 2.7x / 3.7x).
  * @param  mmc MMC instance
  * @param  MD_MMC_ADC_ADI_GAIN_X1
            MD_MMC_ADC_ADI_GAIN_X2
            MD_MMC_ADC_ADI_GAIN_X3
            MD_MMC_ADC_ADI_GAIN_X4
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_adi_gain(MMC_TypeDef *mmc, uint32_t adi_gain)
{
    MODIFY_REG(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADI_GAIN_MSK, adi_gain << MMC_ADC_CTRL0_ADI_GAIN_POSS);
}

/**
  * @brief  Get ADC input voltage gain selection(0.9x / 1.8x / 2.7x / 3.7x).
  * @param  mmc MMC instance
  * @retval MD_MMC_ADC_ADI_GAIN_X1
            MD_MMC_ADC_ADI_GAIN_X2
            MD_MMC_ADC_ADI_GAIN_X3
            MD_MMC_ADC_ADI_GAIN_X4
  */
__STATIC_INLINE uint32_t md_mmc_get_adi_gain(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_ADI_GAIN_MSK) >> MMC_ADC_CTRL0_ADI_GAIN_POSS);
}

/**
  * @brief  ADC enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_ad(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_EN_AD_MSK);
}

/**
  * @brief  ADC disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_ad(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_EN_AD_MSK);
}

/**
  * @brief  Indicates whether the ADC enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_ad(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL0, MMC_ADC_CTRL0_EN_AD_MSK) == (MMC_ADC_CTRL0_EN_AD_MSK));
}

/**
  * @brief  MMC_ADC_CTRL1 Set.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_adc_ctrl1_reg(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->ADC_CTRL1, DATA);
}

/**
  * @brief  MMC_ADC_CTRL1 Get.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_get_adc_ctrl1_reg(MMC_TypeDef *mmc)
{
    READ_REG(mmc->ADC_CTRL1);
}

/**
  * @brief  ADC digital reset
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_is_clear_flag_adrst_dig(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_ADRST_DIG_MSK);
}

/**
  * @brief  Average filter coefficient selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_avfilt_mode(MMC_TypeDef *mmc, uint32_t ad_nfilt)
{
    MODIFY_REG(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AVFILT_MODE_MSK, ad_nfilt << MMC_ADC_CTRL1_AVFILT_MODE_POSS);
}

/**
  * @brief  Get Average filter coefficient selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_avfilt_mode(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AVFILT_MODE_MSK) >> MMC_ADC_CTRL1_AVFILT_MODE_POSS);
}

/**
  * @brief  Skip sample when polarity is changed by system chopper..
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_skip_cntint(MMC_TypeDef *mmc, uint32_t ad_nfilt)
{
    MODIFY_REG(mmc->ADC_CTRL1, MMC_ADC_CTRL1_SKIP_CNTINT_MSK, ad_nfilt << MMC_ADC_CTRL1_SKIP_CNTINT_POSS);
}

/**
  * @brief  Get Skip sample when polarity is changed by system chopper..
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_skip_cntint(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_SKIP_CNTINT_MSK) >> MMC_ADC_CTRL1_SKIP_CNTINT_POSS);
}

/**
  * @brief  Down Frequency Filter Mode Selection(2 CIC)
  * @param  mmc MMC instance
  * @param  MMC_ADC_CTRL1_DECIM_MODE_2_CIC
  * @param  MMC_ADC_CTRL1_DECIM_MODE_4_CIC
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_decim_mode(MMC_TypeDef *mmc, uint32_t select)
{
    if (select == MMC_ADC_CTRL1_DECIM_MODE_2_CIC)
    {
        CLEAR_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_DECIM_MODE_MSK);
    }
    else
    {
        SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_DECIM_MODE_MSK);
    }
}

/**
  * @brief  Get Down Frequency Filter Mode Selection
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_decim_mode(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_DECIM_MODE_MSK) >> MMC_ADC_CTRL1_DECIM_MODE_POS);
}

/**
  * @brief  System chopper frequency selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_chop_freq(MMC_TypeDef *mmc, uint32_t chop_freq)
{
    MODIFY_REG(mmc->ADC_CTRL1, MMC_ADC_CTRL1_CHOP_FREQ_MSK, chop_freq << MMC_ADC_CTRL1_CHOP_FREQ_POSS);
}

/**
  * @brief  Get System chopper frequency selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_chop_freq(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_CHOP_FREQ_MSK) >> MMC_ADC_CTRL1_CHOP_FREQ_POSS);
}

/**
  * @brief  ADC chopper buffer switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_ad_bufchop(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_BUFCHOP_MSK);
}

/**
  * @brief  ADC chopper buffer switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_ad_bufchop(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_BUFCHOP_MSK);
}

/**
  * @brief  Indicates whether the ADC chopper buffer switch enables
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_ad_bufchop(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_BUFCHOP_MSK) == (MMC_ADC_CTRL1_AD_BUFCHOP_MSK));
}

/**
  * @brief  ADC acm buffer switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_ad_acmbuf(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_ACMBUF_MSK);
}

/**
  * @brief  ADC acm buffer switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_ad_acmbuf(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_ACMBUF_MSK);
}

/**
  * @brief  Indicates whether the ADC acm buffer switch enables
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_ad_acmbuf(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_ACMBUF_MSK) == (MMC_ADC_CTRL1_AD_ACMBUF_MSK));
}

/**
  * @brief  ADC programmable gain amplifier selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_ad_pga_sel(MMC_TypeDef *mmc, uint32_t ad_pga_sel)
{
    MODIFY_REG(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_PGA_SEL_MSK, ad_pga_sel << MMC_ADC_CTRL1_AD_PGA_SEL_POSS);
}

/**
  * @brief  Get ADC programmable gain amplifier selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_ad_pga_sel(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_PGA_SEL_MSK) >> MMC_ADC_CTRL1_AD_PGA_SEL_POSS);
}

/**
  * @brief  Filter voltage drop compensation switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_droop(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_DROOP_MSK);
}

/**
  * @brief  Filter voltage drop compensation switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_droop(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_DROOP_MSK);
}

/**
  * @brief  Indicates whether the Filter voltage drop compensation switch enables
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_droop(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_DROOP_MSK) == (MMC_ADC_CTRL1_EN_DROOP_MSK));
}

/**
  * @brief  Square Mode Selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_rms_mode(MMC_TypeDef *mmc, uint32_t rms_mode)
{
    MODIFY_REG(mmc->ADC_CTRL1, MMC_ADC_CTRL1_RMS_MODE_MSK, rms_mode << MMC_ADC_CTRL1_RMS_MODE_POSS);
}

/**
  * @brief  Get Square Mode Selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_rms_mode(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_RMS_MODE_MSK) >> MMC_ADC_CTRL1_RMS_MODE_POSS);
}

/**
  * @brief  Low-pass filter oversampling selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_lpfosr(MMC_TypeDef *mmc, uint32_t sel_lpfosr)
{
    MODIFY_REG(mmc->ADC_CTRL1, MMC_ADC_CTRL1_SEL_LPFOSR_MSK, sel_lpfosr << MMC_ADC_CTRL1_SEL_LPFOSR_POSS);
}

/**
  * @brief  Get Low-pass filter oversampling selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_lpfosr(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_SEL_LPFOSR_MSK) >> MMC_ADC_CTRL1_SEL_LPFOSR_POSS);
}

/**
  * @brief  ADC sampling rate selection bits.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_ad_osr(MMC_TypeDef *mmc, uint32_t ad_osr)
{
    MODIFY_REG(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_OSR_MSK, ad_osr << MMC_ADC_CTRL1_AD_OSR_POSS);
}

/**
  * @brief  Get ADC sampling rate selection bits.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_ad_osr(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_AD_OSR_MSK) >> MMC_ADC_CTRL1_AD_OSR_POSS);
}

/**
  * @brief  Reset peak hold compare module
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_is_clear_flag_rst_pkhcmp(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_RST_PKHCMP_MSK);
}

/**
  * @brief  Reset the root mean square module
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_is_clear_flag_rst_rms(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_RST_RMS_MSK);
}

/**
  * @brief  Reset low-pass filter
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_is_clear_flag_rst_lpf(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_RST_LPF_MSK);
}

/**
  * @brief  Reset comb filter
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_is_clear_flag_rst_comb(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_RST_COMB_MSK);
}

/**
  * @brief  Peak Hold Comparison Module Switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_pkhcmp(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_PKHCMP_MSK);
}

/**
  * @brief  Peak Hold Comparison Module Switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_pkhcmp(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_PKHCMP_MSK);
}

/**
  * @brief  Indicates whether the Peak Hold Comparison Module Switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_pkhcmp(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_PKHCMP_MSK) == (MMC_ADC_CTRL1_EN_PKHCMP_MSK));
}

/**
  * @brief  Root Mean Square Module Switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_rms(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_RMS_MSK);
}

/**
  * @brief  Root Mean Square Module Switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_rms(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_RMS_MSK);
}

/**
  * @brief  Indicates whether the Root Mean Square Module Switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_rms(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_RMS_MSK) == (MMC_ADC_CTRL1_EN_RMS_MSK));
}

/**
  * @brief  Low-pass filter switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_lpf(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_LPF_MSK);
}

/**
  * @brief  Low-pass filter switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_lpf(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_LPF_MSK);
}

/**
  * @brief  Indicates whether the Low-pass filter switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_lpf(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_LPF_MSK) == (MMC_ADC_CTRL1_EN_LPF_MSK));
}

/**
  * @brief  Comb filter switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_comb(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_COMB_MSK);
}

/**
  * @brief  Comb filter switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_comb(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_COMB_MSK);
}

/**
  * @brief  Indicates whether the Comb filter switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_comb(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_CTRL1, MMC_ADC_CTRL1_EN_COMB_MSK) == (MMC_ADC_CTRL1_EN_COMB_MSK));
}

/**
  * @brief  MMC_CMP_CTRL Set.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_cmp_ctrl_reg(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->CMP_CTRL, DATA);
}

/**
  * @brief  MMC_CMP_CTRL Get.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_get_cmp_ctrl_reg(MMC_TypeDef *mmc)
{
    READ_REG(mmc->CMP_CTRL);
}

/**
  * @brief  Get Upper comparator output status
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_active_flag_cmph_ot(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_CMPHO_MSK) >> MMC_CMP_CTRL_CMPHO_POS);
}

/**
  * @brief  Get Lower comparator output status
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_active_flag_cmpl_ot(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_CMPLO_MSK) >> MMC_CMP_CTRL_CMPLO_POS);
}

/**
  * @brief  Get Comparator output state after operation
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_active_flag_cmp_ot(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_CMPO_MSK) >> MMC_CMP_CTRL_CMPO_POS);
}

/**
  * @brief  Upper comparator input selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_cmp_high(MMC_TypeDef *mmc, uint32_t cmp_high)
{
    MODIFY_REG(mmc->CMP_CTRL, MMC_CMP_CTRL_CMP_HIGH_MSK, cmp_high << MMC_CMP_CTRL_CMP_HIGH_POSS);
}

/**
  * @brief  Get Upper comparator input selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_cmp_high(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_CMP_HIGH_MSK) >> MMC_CMP_CTRL_CMP_HIGH_POSS);
}

/**
  * @brief  Lower comparator input selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_cmp_low(MMC_TypeDef *mmc, uint32_t cmp_low)
{
    MODIFY_REG(mmc->CMP_CTRL, MMC_CMP_CTRL_CMP_LOW_MSK, cmp_low << MMC_CMP_CTRL_CMP_LOW_POSS);
}

/**
  * @brief  Get Lower comparator input selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_cmp_low(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_CMP_LOW_MSK) >> MMC_CMP_CTRL_CMP_LOW_POSS);
}

/**
  * @brief  Comparator input selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_cmp_mid(MMC_TypeDef *mmc, uint32_t cmp_mid)
{
    MODIFY_REG(mmc->CMP_CTRL, MMC_CMP_CTRL_CMP_MID_MSK, cmp_mid << MMC_CMP_CTRL_CMP_MID_POSS);
}

/**
  * @brief  Get Comparator input selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_cmp_mid(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_CMP_MID_MSK) >> MMC_CMP_CTRL_CMP_MID_POSS);
}

/**
  * @brief  Comparator hysteresis function switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_hys_cmp(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_HYS_CMP_MSK);
}

/**
  * @brief  Comparator hysteresis function switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_hys_cmp(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_HYS_CMP_MSK);
}

/**
  * @brief  Indicates whether the Comparator hysteresis function switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_hys_cmp(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_HYS_CMP_MSK) == (MMC_CMP_CTRL_HYS_CMP_MSK));
}

/**
  * @brief  Comparator function switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_cmp(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_EN_CMP_MSK);
}

/**
  * @brief  Comparator function switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_cmp(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_EN_CMP_MSK);
}

/**
  * @brief  Indicates whether the Comparator function switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_cmp(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CMP_CTRL, MMC_CMP_CTRL_EN_CMP_MSK) == (MMC_CMP_CTRL_EN_CMP_MSK));
}

/**
  * @brief  MMC_CTRL Set.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_lvd_ctrl_reg(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->LVD_CTRL, DATA);
}

/**
  * @brief  MMC_CTRL Get.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_get_lvd_ctrl_reg(MMC_TypeDef *mmc)
{
    READ_REG(mmc->LVD_CTRL);
}

/**
  * @brief  Get Low voltage detection output status
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_active_flag_st_lvd(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_ST_LVD_MSK) >> (MMC_LVD_CTRL_ST_LVD_POS));
}

/**
  * @brief  Low-voltage detection comparator negative input voltage selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_nin(MMC_TypeDef *mmc, uint32_t sel_nin)
{
    MODIFY_REG(mmc->LVD_CTRL, MMC_LVD_CTRL_SEL_NIN_MSK, sel_nin << MMC_LVD_CTRL_SEL_NIN_POSS);
}

/**
  * @brief  Get Low-voltage detection comparator negative input voltage selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_nin(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_SEL_NIN_MSK) >> MMC_LVD_CTRL_SEL_NIN_POSS);
}

/**
  * @brief  Low-voltage detection comparator positive input voltage selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_sel_pin(MMC_TypeDef *mmc, uint32_t sel_pin)
{
    MODIFY_REG(mmc->LVD_CTRL, MMC_LVD_CTRL_SEL_PIN_MSK, sel_pin << MMC_LVD_CTRL_SEL_PIN_POSS);
}

/**
  * @brief  Get Low-voltage detection comparator positive input voltage selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_sel_pin(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_SEL_PIN_MSK) >> MMC_LVD_CTRL_SEL_PIN_POSS);
}

/**
  * @brief  Low voltage detection mode selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_lvd_mode(MMC_TypeDef *mmc, uint32_t lvd_mode)
{
    MODIFY_REG(mmc->LVD_CTRL, MMC_LVD_CTRL_LVD_MODE_MSK, lvd_mode << MMC_LVD_CTRL_LVD_MODE_POSS);
}

/**
  * @brief  Get Low voltage detection mode selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_lvd_mode(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_LVD_MODE_MSK) >> MMC_LVD_CTRL_LVD_MODE_POSS);
}

/**
  * @brief  Low-voltage detection comparator hysteresis function switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_hys_lvd(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_HYS_LVD_MSK);
}

/**
  * @brief  Low-voltage detection comparator hysteresis function switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_hys_lvd(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_HYS_LVD_MSK);
}

/**
  * @brief  Indicates whether the Low-voltage detection comparator hysteresis function switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_hys_lvd(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_HYS_LVD_MSK) == (MMC_LVD_CTRL_HYS_LVD_MSK));
}

/**
  * @brief  Low-voltage detection comparator function switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_lvd(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_EN_LVD_MSK);
}

/**
  * @brief  Low-voltage detection comparator function switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_lvd(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_EN_LVD_MSK);
}

/**
  * @brief  Indicates whether the Low-voltage detection comparator function switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_lvd(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->LVD_CTRL, MMC_LVD_CTRL_EN_LVD_MSK) == (MMC_LVD_CTRL_EN_LVD_MSK));
}

/**
  * @brief  MMC_CNT_CTRL Set.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_cnt_ctrl_reg(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->CNT_CTRL, DATA);
}

/**
  * @brief  MMC_CNT_CTRL Get.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_cnt_ctrl_reg(MMC_TypeDef *mmc, uint32_t DATA)
{
    return READ_REG(mmc->CNT_CTRL);
}

/**
  * @brief  Reset frequency counter
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_is_clear_flag_rst_fqcnt(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_RST_FQCNT_MSK);
}

/**
  * @brief  Frequency counter clock selection.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_fqcnt_div(MMC_TypeDef *mmc, uint32_t fqcnt_div)
{
    MODIFY_REG(mmc->CNT_CTRL, MMC_CNT_CTRL_FQCNT_DIV_MSK, fqcnt_div << MMC_CNT_CTRL_FQCNT_DIV_POSS);
}

/**
  * @brief  Get Frequency counter clock selection.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_fqcnt_div(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_FQCNT_DIV_MSK) >> MMC_CNT_CTRL_FQCNT_DIV_POSS);
}

/**
  * @brief  Get Frequency counter signal status to be tested.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_clear_flag_fqcnt_in(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_FQCNTI_MSK) >> MMC_CNT_CTRL_FQCNTI_POS);
}

/**
  * @brief  Frequency counter signal input selection
  * @param  mmc MMC instance
  * @param  MMC_CNT_CTRL_EN_FQCNTI_CMPO
  * @param  MMC_CNT_CTRL_EN_FQCNTI_PCNTI
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_en_fqcnti(MMC_TypeDef *mmc, uint32_t select)
{
    if (select == MMC_CNT_CTRL_EN_FQCNTI_CMPO)
    {
        CLEAR_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_EN_FQCNTI_MSK);
    }
    else
    {
        SET_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_EN_FQCNTI_MSK);
    }
}

/**
  * @brief  Get Frequency counter signal input selection
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_en_fqcnti(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_EN_FQCNTI_MSK) >> MMC_CNT_CTRL_EN_FQCNTI_POS);
}

/**
  * @brief  Frequency counter switch enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_en_fqcnt(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_EN_FQCNT_MSK);
}

/**
  * @brief  Frequency counter switch disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_en_fqcnt(MMC_TypeDef *mmc)
{
    CLEAR_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_EN_FQCNT_MSK);
}

/**
  * @brief  Indicates whether the Frequency counter switch enabled
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_en_fqcnt(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_EN_FQCNT_MSK) == (MMC_CNT_CTRL_EN_FQCNT_MSK));
}

/**
  * @brief  Counter A initial value setting.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_init_cnta(MMC_TypeDef *mmc, uint32_t init_cnta)
{
    MODIFY_REG(mmc->CNT_CTRL, MMC_CNT_CTRL_INIT_CNTA_MSK, init_cnta << MMC_CNT_CTRL_INIT_CNTA_POSS);
}

/**
  * @brief  Get Counter A initial value setting.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_init_cnta(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->CNT_CTRL, MMC_CNT_CTRL_INIT_CNTA_MSK) >> MMC_CNT_CTRL_INIT_CNTA_POSS);
}

/**
  * @brief  Counter A status.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_fq_cnta(MMC_TypeDef *mmc, uint32_t fq_cnta)
{
    MODIFY_REG(mmc->FQ_CNTA, MMC_FQ_CNTA_FQ_CNTA_MSK, fq_cnta << MMC_FQ_CNTA_FQ_CNTA_POSS);
}

/**
  * @brief  Get Counter A status.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_fq_cnta(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->FQ_CNTA, MMC_FQ_CNTA_FQ_CNTA_MSK) >> MMC_FQ_CNTA_FQ_CNTA_POSS);
}

/**
  * @brief  Counter B status.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_fq_cntb(MMC_TypeDef *mmc, uint32_t fq_cntb)
{
    MODIFY_REG(mmc->FQ_CNTB, MMC_FQ_CNTB_FQ_CNTB_MSK, fq_cntb << MMC_FQ_CNTB_FQ_CNTB_POSS);
}

/**
  * @brief  Get Counter B status.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_fq_cntb(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->FQ_CNTB, MMC_FQ_CNTB_FQ_CNTB_MSK) >> MMC_FQ_CNTB_FQ_CNTB_POSS);
}

/**
  * @brief  Counter C status.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_fq_cntc(MMC_TypeDef *mmc, uint32_t fq_cntc)
{
    MODIFY_REG(mmc->FQ_CNTC, MMC_FQ_CNTC_FQ_CNTC_MSK, fq_cntc << MMC_FQ_CNTC_FQ_CNTC_POSS);
}

/**
  * @brief  Get Counter C status.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_fq_cntc(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->FQ_CNTC, MMC_FQ_CNTC_FQ_CNTC_MSK) >> MMC_FQ_CNTC_FQ_CNTC_POSS);
}

/**
  * @brief  Set Maximum peak value.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_pkh_max(MMC_TypeDef *mmc, uint32_t pkh_max)
{
    MODIFY_REG(mmc->PKH_MAX, MMC_PKH_MAX_PKH_MAX_MSK, pkh_max << MMC_PKH_MAX_PKH_MAX_POSS);
}

/**
  * @brief  Get Maximum peak value.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_pkh_max(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->PKH_MAX, MMC_PKH_MAX_PKH_MAX_MSK) >> MMC_PKH_MAX_PKH_MAX_POSS);
}

/**
  * @brief  Set Minimum peak value.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_pkh_min(MMC_TypeDef *mmc, uint32_t pkh_min)
{
    MODIFY_REG(mmc->PKH_MIN, MMC_PKH_MIN_PKH_MIN_MSK, pkh_min << MMC_PKH_MIN_PKH_MIN_POSS);
}

/**
  * @brief  Get Minimum peak value.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_pkh_min(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->PKH_MIN, MMC_PKH_MIN_PKH_MIN_MSK) >> MMC_PKH_MIN_PKH_MIN_POSS);
}

/**
  * @brief  Set Root mean square calculation value.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_dcest_val(MMC_TypeDef *mmc, uint32_t dcest_val)
{
    MODIFY_REG(mmc->DCEST_VAL, MMC_DCEST_VAL_DCEST_VAL_MSK, dcest_val << MMC_DCEST_VAL_DCEST_VAL_POSS);
}

/**
  * @brief  Get Root mean square calculation value.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_dcest_val(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->DCEST_VAL, MMC_DCEST_VAL_DCEST_VAL_MSK) >> MMC_DCEST_VAL_DCEST_VAL_POSS);
}


/**
  * @brief  Set Root mean square calculation value.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_rms_val(MMC_TypeDef *mmc, uint32_t rms_val)
{
    MODIFY_REG(mmc->RMS_VAL, MMC_RMS_VAL_RMS_VAL_MSK, rms_val << MMC_RMS_VAL_RMS_VAL_POSS);
}

/**
  * @brief  Get Root mean square calculation value.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_rms_val(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RMS_VAL, MMC_RMS_VAL_RMS_VAL_MSK) >> MMC_RMS_VAL_RMS_VAL_POSS);
}

/**
  * @brief  Set Low pass filter value.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_lpf_val(MMC_TypeDef *mmc, uint32_t lpf_val)
{
    MODIFY_REG(mmc->LPF_VAL, MMC_LPF_VAL_LPF_VAL_MSK, lpf_val << MMC_LPF_VAL_LPF_VAL_POSS);
}

/**
  * @brief  Get Low pass filter value.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_lpf_val(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->LPF_VAL, MMC_LPF_VAL_LPF_VAL_MSK) >> MMC_LPF_VAL_LPF_VAL_POSS);
}

/**
  * @brief  Set ADC value.
  * @param  mmc MMC instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_mmc_set_adc_val(MMC_TypeDef *mmc, uint32_t adc_val)
{
    MODIFY_REG(mmc->ADC_VAL, MMC_ADC_VAL_ADC_VAL_MSK, adc_val << MMC_ADC_VAL_ADC_VAL_POSS);
}

/**
  * @brief  Get ADC value.
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_mmc_get_adc_val(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->ADC_VAL, MMC_ADC_VAL_ADC_VAL_MSK) >> MMC_ADC_VAL_ADC_VAL_POSS);
}

/**
  * @brief  interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->IER, DATA);
}

/**
  * @brief  Counter B overflow interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_cntb_ov(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_CNTB_OV_MSK);
}

/**
  * @brief  Counter A abnormal interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_frqcnt_abn(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_FRQCNT_ABN_MSK);
}

/**
  * @brief  Counter measurement module end interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_frqcnt_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_FRQCNT_DONE_MSK);
}

/**
  * @brief  Low voltage detection output change interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_lvdo_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_LVDO_CHG_MSK);
}

/**
  * @brief  Upper and lower comparator output change interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_cmpo_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_CMPO_CHG_MSK);
}

/**
  * @brief  Lower comparator output change interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_cmpl_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_CMPL_CHG_MSK);
}

/**
  * @brief  Comparator output change interrupt enable after operation
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_cmph_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_CMPH_CHG_MSK);
}

/**
  * @brief  Peak hold operation end interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_pkh_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_PKH_DONE_MSK);
}

/**
  * @brief  End of rms operation, change interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_rms_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_RMS_DONE_MSK);
}

/**
  * @brief  Low-pass filter operation end interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_lpf_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_LPF_DONE_MSK);
}

/**
  * @brief  Analog converter operation end interrupt enable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_enable_it_adc_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IER, MMC_IER_ADC_DONE_MSK);
}

/**
  * @brief  interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->IDR, DATA);
}
/**
  * @brief  Counter B overflow interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_cntb_ov(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_CNTB_OV_MSK);
}

/**
  * @brief  Counter A abnormal interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_frqcnt_abn(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_FRQCNT_ABN_MSK);
}

/**
  * @brief  Counter measurement module end interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_frqcnt_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_FRQCNT_DONE_MSK);
}

/**
  * @brief  Low voltage detection output change interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_lvdo_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_LVDO_CHG_MSK);
}

/**
  * @brief  Upper and lower comparator output change interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_cmpo_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_CMPO_CHG_MSK);
}

/**
  * @brief  Lower comparator output change interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_cmpl_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_CMPL_CHG_MSK);
}

/**
  * @brief  Comparator output change interrupt disable after operation
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_cmph_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_CMPH_CHG_MSK);
}

/**
  * @brief  Peak hold operation end interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_pkh_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_PKH_DONE_MSK);
}

/**
  * @brief  End of rms operation, change interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_rms_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_RMS_DONE_MSK);
}

/**
  * @brief  Low-pass filter operation end interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_lpf_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_LPF_DONE_MSK);
}

/**
  * @brief  Analog converter operation end interrupt disable
  * @param  mmc MMC instance
  * @retval None
  */
__STATIC_INLINE void md_mmc_disable_it_adc_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->IDR, MMC_IDR_ADC_DONE_MSK);
}

/**
  * @brief  Check Counter B overflow valid bit status.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_cntb_ov(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_CNTB_OV_MSK) == (MMC_IVS_CNTB_OV_MSK));
}

/**
  * @brief  Check Counter A abnormal valid bit status.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_frqcnt_abn(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_FRQCNT_ABN_MSK) == (MMC_IVS_FRQCNT_ABN_MSK));
}

/**
  * @brief  Check Counter measurement module end valid bit status.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_frqcnt_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_FRQCNT_DONE_MSK) == (MMC_IVS_FRQCNT_DONE_MSK));
}

/**
  * @brief  Check Low voltage detection output change effective bit status.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_lvdo_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_LVDO_CHG_MSK) == (MMC_IVS_LVDO_CHG_MSK));
}

/**
  * @brief  Check Upper and lower comparator output change effective bit status.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_cmpo_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_CMPO_CHG_MSK) == (MMC_IVS_CMPO_CHG_MSK));
}

/**
  * @brief  Check Lower comparator output change effective bit status.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_cmpl_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_CMPL_CHG_MSK) == (MMC_IVS_CMPL_CHG_MSK));
}

/**
  * @brief  Check The comparator output changes the effective bit state after operation.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_cmph_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_CMPH_CHG_MSK) == (MMC_IVS_CMPH_CHG_MSK));
}

/**
  * @brief  Check Peak hold operation end valid bit status.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_pkh_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_PKH_DONE_MSK) == (MMC_IVS_PKH_DONE_MSK));
}

/**
  * @brief  Check End of RMS operation changes the effective bit state.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_rms_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_RMS_DONE_MSK) == (MMC_IVS_RMS_DONE_MSK));
}

/**
  * @brief  Check Low pass filter operation end valid bit status.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_lpf_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_LPF_DONE_MSK) == (MMC_IVS_LPF_DONE_MSK));
}

/**
  * @brief  Check The state of the valid bit after the operation of the analog converter.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_enabled_it_adc_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IVS, MMC_IVS_ADC_DONE_MSK) == (MMC_IVS_ADC_DONE_MSK));
}

/**
  * @brief  Get Counter B overflows unmasked interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_cntb_ov(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_CNTB_OV_MSK) == (MMC_RIF_CNTB_OV_MSK));
}

/**
  * @brief  Get Counter A abnormal unmasked interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_frqcnt_abn(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_FRQCNT_ABN_MSK) == (MMC_RIF_FRQCNT_ABN_MSK));
}

/**
  * @brief  Get Counter measurement module ends without masked interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_frqcnt_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_FRQCNT_DONE_MSK) == (MMC_RIF_FRQCNT_DONE_MSK));
}

/**
  * @brief  Get Low voltage detection output change without mask interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_lvdo_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_LVDO_CHG_MSK) == (MMC_RIF_LVDO_CHG_MSK));
}

/**
  * @brief  Get Unmasked interrupt flag for upper and lower comparator output changes.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_cmpo_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_CMPO_CHG_MSK) == (MMC_RIF_CMPO_CHG_MSK));
}

/**
  * @brief  Get Unmasked interrupt flag for lower comparator output change.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_cmpl_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_CMPL_CHG_MSK) == (MMC_RIF_CMPL_CHG_MSK));
}

/**
  * @brief  Get Comparator output changes after operation unmasked interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_cmph_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_CMPH_CHG_MSK) == (MMC_RIF_CMPH_CHG_MSK));
}

/**
  * @brief  Get Unmasked interrupt flag at the end of peak hold operation.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_pkh_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_PKH_DONE_MSK) == (MMC_RIF_PKH_DONE_MSK));
}

/**
  * @brief  Get End of root mean square operation change unmasked interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_rms_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_RMS_DONE_MSK) == (MMC_RIF_RMS_DONE_MSK));
}

/**
  * @brief  Get Unmasked interrupt flag at the end of low-pass filter operation.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_lpf_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_LPF_DONE_MSK) == (MMC_RIF_LPF_DONE_MSK));
}

/**
  * @brief  Get Unmasked interrupt flag at the end of analog converter operation.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_active_it_adc_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->RIF, MMC_RIF_ADC_DONE_MSK) == (MMC_RIF_ADC_DONE_MSK));
}

/**
  * @brief  Get Counter B overflows masked interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_cntb_ov(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_CNTB_OV_MSK) == (MMC_IFM_CNTB_OV_MSK));
}

/**
  * @brief  Get Counter A abnormal masked interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_frqcnt_abn(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_FRQCNT_ABN_MSK) == (MMC_IFM_FRQCNT_ABN_MSK));
}

/**
  * @brief  Get Counter measurement module end mask interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_frqcnt_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_FRQCNT_DONE_MSK) == (MMC_IFM_FRQCNT_DONE_MSK));
}

/**
  * @brief  Get Low voltage detection output change mask interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_lvdo_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_LVDO_CHG_MSK) == (MMC_IFM_LVDO_CHG_MSK));
}

/**
  * @brief  Get Upper and lower comparator output change mask interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_cmpo_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_CMPO_CHG_MSK) == (MMC_IFM_CMPO_CHG_MSK));
}

/**
  * @brief  Get Lower comparator output change mask interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_cmpl_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_CMPL_CHG_MSK) == (MMC_IFM_CMPL_CHG_MSK));
}

/**
  * @brief  Get Comparator outputs change mask interrupt flag after operation.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_cmph_chg(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_CMPH_CHG_MSK) == (MMC_IFM_CMPH_CHG_MSK));
}

/**
  * @brief  Get Peak hold operation end mask interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_pkh_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_PKH_DONE_MSK) == (MMC_IFM_PKH_DONE_MSK));
}

/**
  * @brief  Get End of root mean square operation change mask interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_rms_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_RMS_DONE_MSK) == (MMC_IFM_RMS_DONE_MSK));
}

/**
  * @brief  Get Low-pass filter operation end mask interrupt flag.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_lpf_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_LPF_DONE_MSK) == (MMC_IFM_LPF_DONE_MSK));
}

/**
  * @brief  Get Masking interrupt flag at the end of analog converter operation.
  * @param  mmc MMC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_mmc_is_masked_it_adc_done(MMC_TypeDef *mmc)
{
    return (READ_BIT(mmc->IFM, MMC_IFM_ADC_DONE_MSK) == (MMC_IFM_ADC_DONE_MSK));
}

/**
  * @brief interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr(MMC_TypeDef *mmc, uint32_t DATA)
{
    WRITE_REG(mmc->ICR, DATA);
}
/**
  * @brief  Counter B overflow interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_cntb_ov(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_CNTB_OV_MSK);
}

/**
  * @brief  Counter A abnormal interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_frqcnt_abn(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_FRQCNT_ABN_MSK);
}

/**
  * @brief  Counter measurement module end interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_frqcnt_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_FRQCNT_DONE_MSK);
}

/**
  * @brief  Low voltage detection output change interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_lvdo_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_LVDO_CHG_MSK);
}

/**
  * @brief  Upper and lower comparator output change interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_cmpo_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_CMPO_CHG_MSK);
}

/**
  * @brief  Lower comparator output change interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_cmpl_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_CMPL_CHG_MSK);
}

/**
  * @brief  Comparator output interrupt clear after operation.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_cmph_chg(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_CMPH_CHG_MSK);
}

/**
  * @brief  Peak hold operation end interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_pkh_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_PKH_DONE_MSK);
}

/**
  * @brief  End of rms operation, change interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_rms_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_RMS_DONE_MSK);
}

/**
  * @brief  Low-pass filter operation interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_lpf_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_LPF_DONE_MSK);
}

/**
  * @brief  Analog converter operation interrupt clear.
  * @param  mmc MMC instance.
  * @retval
  */
__STATIC_INLINE void md_mmc_clear_it_icr_adc_done(MMC_TypeDef *mmc)
{
    SET_BIT(mmc->ICR, MMC_ICR_ADC_DONE_MSK);
}

/**
  * @} MD_MMC_Public_Macro
  */

/* Public functions -----------------------------------------------------------*/

/**
  * @} MMC
  */


#endif

/**
  * @} Micro_Driver
  */


#ifdef __cplusplus
}
#endif

#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
