/**********************************************************************************
 *
 * @file    md_lcd.h
 * @brief   header file of md_lcd.c
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_LCD_H__
#define __MD_LCD_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------*/
#include <stdbool.h>
#include "es32f0930.h"
/** @addtogroup Micro_Driver
  * @{
  */
#if defined (LCD)

/** @defgroup LCD LCD
  * @brief LCD module driver
  * @{
  */


/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private Macros ----------------------------------------------------------*/
/* Public types ---------------------------------------------------------------*/
/** @defgroup MD_LCD_Public_Types LCD Public Init Type
  * @{
  */

typedef struct
{
    uint32_t VlcdCpSw;
    uint32_t VlcdCpDiv;
    uint32_t VlcdCpEn;
    uint32_t LcdSw;
    uint32_t LcdDiv;
} md_rcu_lcd_init_typedef;

typedef struct
{
    uint32_t CpEn;
    uint32_t BufEn;
    uint32_t Vsl;
    uint32_t Rsel;
    uint32_t Type;
    uint32_t Bias;
    uint32_t Duty;
} md_lcd_mode_typedef;

typedef struct
{
    uint32_t ComEn;
    uint32_t SegEn;
} md_lcd_comx_segx_typedef;

typedef struct
{
    uint32_t FrDiv;
    uint32_t UpdCtrl;
    uint32_t WsBlink;
    uint32_t FcValue;
    uint32_t FcCtrl;
    uint32_t SwUpdate;
    uint32_t LcdEn;
} md_lcd_ctrl_typedef;

/**
  * @} MD_LCD_Public_Types
  */

/* Public Constants -----------------------------------------------------------*/
/** @defgroup MD_LCD_Public_Constants Public Constants
  * @brief LCD module driver
  * @{
  */

/*LCD VSL input marcros define*/
#define MD_LCD_MODE_VLS_VLCD_3V3                       0                /*!< VLS = 3.3V*/
#define MD_LCD_MODE_VLS_VLCD_3V0                       1                /*!< VLS = 3.0V*/
#define MD_LCD_MODE_VLS_VLCD_2V8                       2                /*!< VLS = 2.8V*/
#define MD_LCD_MODE_VLS_VLCD_2V55                      3                /*!< VLS = 2.55V*/

/*LCD RSEL input marcros define*/
#define MD_LCD_MODE_RSEL_225K                          0                /*!< LCD BIAS RE = 225K*/
#define MD_LCD_MODE_RSEL_900K                          1                /*!< LCD BIAS RE = 900K*/

/*LCD RSEL input marcros define*/
#define MD_LCD_MODE_TYPE_A                             0                /*!< TYPE_A*/
#define MD_LCD_MODE_TYPE_B                             1                /*!< TYPE_B*/

/*LCD BIAS input marcros define*/
#define MD_LCD_MODE_BIAS_STATIC                        0                /*!< BIAS = STATIC*/
#define MD_LCD_MODE_BIAS_ONEHALF                       1                /*!< BIAS = 1/2*/
#define MD_LCD_MODE_BIAS_ONETHIRD                      2                /*!< BIAS = 1/3*/
#define MD_LCD_MODE_BIAS_ONEQUARTER                    3                /*!< BIAS = 1/4*/

/*LCD DUTY input marcros define*/
#define MD_LCD_MODE_DUTY_ONE                           0                /*!< DUTY = 1/1*/
#define MD_LCD_MODE_DUTY_ONEHALF                       1                /*!< DUTY = 1/2*/
#define MD_LCD_MODE_DUTY_ONETHIRD                      2                /*!< DUTY = 1/3*/
#define MD_LCD_MODE_DUTY_ONEQUARTER                    3                /*!< DUTY = 1/4*/

/*LCD WSBLINK input marcros define*/
#define MD_LCD_CTRL_WSBLINK_DISABLE                    0                /*!< WSBLINK = DISABLE*/
#define MD_LCD_CTRL_WSBLINK_MODE0                      1                /*!< WSBLINK = COMx BLINK REVERSE*/
#define MD_LCD_CTRL_WSBLINK_MODE1                      2                /*!< WSBLINK = COMx BLINK */
#define MD_LCD_CTRL_WSBLINK_MODE2                      3                /*!< WSBLINK = COMx 0~7 CONTROLLED BY BLINK REGISTER*/

/*LCD UPDCTRL input marcros define*/
#define MD_LCD_CTRL_UPDCTRL_EVERYFRAME                 0                /*!< UPDCTRL = EVERY FRAME*/
#define MD_LCD_CTRL_UPDCTRL_SWUPDATA                   1                /*!< UPDCTRL = SWUPDATA*/
#define MD_LCD_CTRL_UPDCTRL_COUNT_0                    2                /*!< UPDCTRL = COUNT TO 0 */
#define MD_LCD_CTRL_UPDCTRL_DISABLE                    3                /*!< UPDCTRL = DISABLE*/

/*LCD COM_EN input marcros define*/
#define MD_LCD_COMEN_COM_EN_COM0                       (1 << 0)         /*!< COM_EN = COM0*/
#define MD_LCD_COMEN_COM_EN_COM1                       (1 << 1)         /*!< COM_EN = COM1*/
#define MD_LCD_COMEN_COM_EN_COM2                       (1 << 2)         /*!< COM_EN = COM2*/
#define MD_LCD_COMEN_COM_EN_COM3                       (1 << 3)         /*!< COM_EN = COM3*/

/*LCD SEG_EN input marcros define*/
#define MD_LCD_SEGEN_SEG_EN_SEG0                       (1 << 0)         /*!< SEG_EN = SEG0 */
#define MD_LCD_SEGEN_SEG_EN_SEG1                       (1 << 1)         /*!< SEG_EN = SEG1 */
#define MD_LCD_SEGEN_SEG_EN_SEG2                       (1 << 2)         /*!< SEG_EN = SEG2 */
#define MD_LCD_SEGEN_SEG_EN_SEG3                       (1 << 3)         /*!< SEG_EN = SEG3 */
#define MD_LCD_SEGEN_SEG_EN_SEG4                       (1 << 4)         /*!< SEG_EN = SEG4 */
#define MD_LCD_SEGEN_SEG_EN_SEG5                       (1 << 5)         /*!< SEG_EN = SEG5 */
#define MD_LCD_SEGEN_SEG_EN_SEG6                       (1 << 6)         /*!< SEG_EN = SEG6 */
#define MD_LCD_SEGEN_SEG_EN_SEG7                       (1 << 7)         /*!< SEG_EN = SEG7 */
#define MD_LCD_SEGEN_SEG_EN_SEG8                       (1 << 8)         /*!< SEG_EN = SEG8 */
#define MD_LCD_SEGEN_SEG_EN_SEG9                       (1 << 9)         /*!< SEG_EN = SEG9 */
#define MD_LCD_SEGEN_SEG_EN_SEG10                      (1 << 10)        /*!< SEG_EN = SEG10*/
#define MD_LCD_SEGEN_SEG_EN_SEG11                      (1 << 11)        /*!< SEG_EN = SEG11*/
#define MD_LCD_SEGEN_SEG_EN_SEG12                      (1 << 12)        /*!< SEG_EN = SEG12*/
#define MD_LCD_SEGEN_SEG_EN_SEG13                      (1 << 13)        /*!< SEG_EN = SEG13*/
#define MD_LCD_SEGEN_SEG_EN_SEG14                      (1 << 14)        /*!< SEG_EN = SEG14*/
#define MD_LCD_SEGEN_SEG_EN_SEG15                      (1 << 15)        /*!< SEG_EN = SEG14*/

/*LCD SEGx input marcros define*/
#define MD_LCD_SEG0                                    (1 << 0)         /*!< SEG = SEG0 */
#define MD_LCD_SEG1                                    (1 << 1)         /*!< SEG = SEG1 */
#define MD_LCD_SEG2                                    (1 << 2)         /*!< SEG = SEG2 */
#define MD_LCD_SEG3                                    (1 << 3)         /*!< SEG = SEG3 */
#define MD_LCD_SEG4                                    (1 << 4)         /*!< SEG = SEG4 */
#define MD_LCD_SEG5                                    (1 << 5)         /*!< SEG = SEG5 */
#define MD_LCD_SEG6                                    (1 << 6)         /*!< SEG = SEG6 */
#define MD_LCD_SEG7                                    (1 << 7)         /*!< SEG = SEG7 */
#define MD_LCD_SEG8                                    (1 << 8)         /*!< SEG = SEG8 */
#define MD_LCD_SEG9                                    (1 << 9)         /*!< SEG = SEG9 */
#define MD_LCD_SEG10                                   (1 << 10)        /*!< SEG = SEG10*/
#define MD_LCD_SEG11                                   (1 << 11)        /*!< SEG = SEG11*/
#define MD_LCD_SEG12                                   (1 << 12)        /*!< SEG = SEG12*/
#define MD_LCD_SEG13                                   (1 << 13)        /*!< SEG = SEG13*/
#define MD_LCD_SEG14                                   (1 << 14)        /*!< SEG = SEG14*/


#define MD_LCD_COM0_REGISTER                           0                /*!< COM = COM0*/
#define MD_LCD_COM1_REGISTER                           1                /*!< COM = COM1*/
#define MD_LCD_COM2_REGISTER                           2                /*!< COM = COM2*/
#define MD_LCD_COM3_REGISTER                           3                /*!< COM = COM3*/

/**
  * @} MD_LCD__Public_Constants
  */

/* Public Macro ------------------------------------------------------------*/
/** @defgroup MD_LCD_Public_Macro Public Macro
  * @brief LCD module driver
  * @{
  */

/**
  * @brief  Set LVD voltage selection.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_vsl(LCD_TypeDef *lcd, uint32_t vsl)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_VSL_MSK, vsl << LCD_MODE_VSL_POSS);
}

/**
  * @brief  Get LVD voltage selection.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_vsl(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->MODE, LCD_MODE_VSL_MSK) >> LCD_MODE_VSL_POSS);
}

/**
  * @brief  Charge pump enable.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_cpen(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->MODE, LCD_MODE_CPEN_MSK);
}

/**
  * @brief  Charge pump disable.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_cpen(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->MODE, LCD_MODE_CPEN_MSK);
}

/**
  * @brief  Indicates whether the Charge pump is enabled
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_cpen(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->MODE, LCD_MODE_CPEN_MSK) == (LCD_MODE_CPEN_MSK));
}

/**
  * @brief  Output buffer enable.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_bufen(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->MODE, LCD_MODE_BUFEN_MSK);
}

/**
  * @brief  Output buffer disable.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_bufen(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->MODE, LCD_MODE_BUFEN_MSK);
}

/**
  * @brief  Indicates whether the Output buffer is enabled
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_bufen(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->MODE, LCD_MODE_BUFEN_MSK) == (LCD_MODE_BUFEN_MSK));
}

/**
  * @brief  LCD bias resistor selection. (900K re)
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_rsel_900k(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->MODE, LCD_MODE_RSEL_MSK);
}

/**
  * @brief  LCD bias resistor selection.(225K re)
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_rsel_225k(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->MODE, LCD_MODE_RSEL_MSK);
}

/**
  * @brief  Get the LCD bias resistor selection
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_rsel(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->MODE, LCD_MODE_RSEL_MSK) >> LCD_MODE_RSEL_POS);
}

/**
  * @brief  LCD output type selection (B TYPE)
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_typeB(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->MODE, LCD_MODE_TYPE_MSK);
}

/**
  * @brief  LCD output type selection (A TYPE)
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_typeA(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->MODE, LCD_MODE_TYPE_MSK);
}

/**
  * @brief  Get the LCD output type selection
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_type(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->MODE, LCD_MODE_TYPE_MSK) >> LCD_MODE_TYPE_POS);
}

/**
  * @brief  Set LCD bias selection.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_bias(LCD_TypeDef *lcd, uint32_t bias)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_BIAS_MSK, bias << LCD_MODE_BIAS_POSS);
}

/**
  * @brief  Get LCD bias selection.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_bias(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->MODE, LCD_MODE_BIAS_MSK) >> LCD_MODE_BIAS_POSS);
}

/**
  * @brief  Set LCD output selection.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_duty(LCD_TypeDef *lcd, uint32_t duty)
{
    MODIFY_REG(lcd->MODE, LCD_MODE_DUTY_MSK, duty << LCD_MODE_DUTY_POSS);
}

/**
  * @brief  Get LCD output selection.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_duty(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->MODE, LCD_MODE_DUTY_MSK) >> LCD_MODE_DUTY_POSS);
}

/**
  * @brief  Set Full frequency blink selection.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_setl_wsblink(LCD_TypeDef *lcd, uint32_t wsblink)
{
    MODIFY_REG(lcd->CTRL, LCD_CTRL_WSBLINK_MSK, wsblink << LCD_CTRL_WSBLINK_POSS);
}

/**
  * @brief  Get Full frequency blink selection.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_wsblink(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_WSBLINK_MSK) >> LCD_CTRL_WSBLINK_POSS);
}

/**
  * @brief  Set Frame counter value.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_fcvalue(LCD_TypeDef *lcd, uint32_t fcvalue)
{
    MODIFY_REG(lcd->CTRL, LCD_CTRL_FCVALUE_MSK, fcvalue << LCD_CTRL_FCVALUE_POSS);
}

/**
  * @brief  Get Frame counter value.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_fcvalue(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_FCVALUE_MSK) >> LCD_CTRL_FCVALUE_POSS);
}

/**
  * @brief  Frame counter control enable
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_fcctrl(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->CTRL, LCD_CTRL_FCCTRL_MSK);
}

/**
  * @brief  Frame counter control disable
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_fcctrl(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->CTRL, LCD_CTRL_FCCTRL_MSK);
}

/**
  * @brief  Indicates whether the Frame counter control is enabled
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_fcctrl(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_FCCTRL_MSK) == (LCD_CTRL_FCCTRL_MSK));
}

/**
  * @brief  Software update control
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_swupdate(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->CTRL, LCD_CTRL_SWUPDATE_MSK);
}

/**
  * @brief  Software update control
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_swupdate(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->CTRL, LCD_CTRL_SWUPDATE_MSK);
}

/**
  * @brief  Indicates whether the Software update control is enabled
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_swupdate(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_SWUPDATE_MSK) == (LCD_CTRL_SWUPDATE_MSK));
}

/**
  * @brief  Set Update control options.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_updctrl(LCD_TypeDef *lcd, uint32_t updctrl)
{
    MODIFY_REG(lcd->CTRL, LCD_CTRL_UPDCTRL_MSK, updctrl << LCD_CTRL_UPDCTRL_POSS);
}

/**
  * @brief  Get Update control options.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_ctrl_updctrl(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_UPDCTRL_MSK) >> LCD_CTRL_UPDCTRL_POSS);
}

/**
  * @brief  Set Frame divider.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_frdiv(LCD_TypeDef *lcd, uint32_t frdiv)
{
    MODIFY_REG(lcd->CTRL, LCD_CTRL_FRDIV_MSK, frdiv << LCD_CTRL_FRDIV_POSS);
}

/**
  * @brief  Get Frame divider.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_frdiv(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_FRDIV_MSK) >> LCD_CTRL_FRDIV_POSS);
}

/**
  * @brief  LCD control enable
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_lcden(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->CTRL, LCD_CTRL_LCDEN_MSK);
}

/**
  * @brief  LCD control disable
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_lcden(LCD_TypeDef *lcd)
{
    CLEAR_BIT(lcd->CTRL, LCD_CTRL_LCDEN_MSK);
}

/**
  * @brief  Indicates whether the LCD control enable is enabled
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_lcden(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->CTRL, LCD_CTRL_LCDEN_MSK) == (LCD_CTRL_LCDEN_MSK));
}

/**
  * @brief  LCD Blink update
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_blink_update(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->BLINK, LCD_BLINK_UPDATE_MSK);
}

/**
  * @brief  Update completion interrupt enable
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_ier_update(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->IER, LCD_IER_UPDATE_MSK);
}

/**
  * @brief  Update complete interrupt disable
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_idr_update(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->IDR, LCD_IDR_UPDATE_MSK);
}

/**
  * @brief  Update completion interrupt valid bit
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_ivs_update(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->IVS, LCD_IVS_UPDATE_MSK) == (LCD_IVS_UPDATE_MSK));
}

/**
  * @brief  Update complete original interrupt flag
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_is_active_flag_rif_update(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->RIF, LCD_RIF_UPDATE_MSK) == (LCD_RIF_UPDATE_MSK));
}

/**
  * @brief  Update complete original interrupt flag
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_is_active_flag_ifm_update(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->IFM, LCD_IFM_UPDATE_MSK) == (LCD_IFM_UPDATE_MSK));
}

/**
  * @brief  Update completes and the original interrupt flag is cleared
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE void md_lcd_is_clear_flag_icr_update(LCD_TypeDef *lcd)
{
    SET_BIT(lcd->ICR, LCD_ICR_UPDATE_MSK);
}

/**
  * @brief  Update run flag
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_is_active_flag_busy_update(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->BUSY, LCD_BUSY_UPDATE_MSK) == (LCD_BUSY_UPDATE_MSK));
}

/**
  * @brief  Set COM0 value.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_com0(LCD_TypeDef *lcd, uint32_t com0)
{
    MODIFY_REG(lcd->COM0, LCD_COM0_COM0_MSK, com0 << LCD_COM0_COM0_POSS);
}


/**
  * @brief  Get COM0 value.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_com0(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->COM0, LCD_COM0_COM0_MSK) >> LCD_COM0_COM0_POSS);
}

/**
  * @brief  Set COM1 value.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_com1(LCD_TypeDef *lcd, uint32_t com1)
{
    MODIFY_REG(lcd->COM1, LCD_COM1_COM1_MSK, com1 << LCD_COM1_COM1_POSS);
}
/**
  * @brief  Get COM1 value.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_com1(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->COM1, LCD_COM1_COM1_MSK) >> LCD_COM1_COM1_POSS);
}

/**
  * @brief  Set COM2 value.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_com2(LCD_TypeDef *lcd, uint32_t com2)
{
    MODIFY_REG(lcd->COM2, LCD_COM2_COM2_MSK, com2 << LCD_COM2_COM2_POSS);
}

/**
  * @brief  Get COM2 value.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_com2(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->COM2, LCD_COM2_COM2_MSK) >> LCD_COM2_COM2_POSS);
}

/**
  * @brief  Set COM3 value.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_com3(LCD_TypeDef *lcd, uint32_t com3)
{
    MODIFY_REG(lcd->COM3, LCD_COM3_COM3_MSK, com3 << LCD_COM3_COM3_POSS);
}

/**
  * @brief  Get COM3 value.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_com3(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->COM3, LCD_COM3_COM3_MSK) >> LCD_COM3_COM3_POSS);
}

/**
  * @brief  Set COM output enable.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_com_en(LCD_TypeDef *lcd, uint32_t com_en)
{
    MODIFY_REG(lcd->COMEN, LCD_COMEN_COM_EN_MSK, com_en << LCD_COMEN_COM_EN_POSS);
}

/**
  * @brief  Get COM output enable.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_com_en(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->COMEN, LCD_COMEN_COM_EN_MSK) >> LCD_COMEN_COM_EN_POSS);
}

/**
  * @brief  Set SEG output enable.
  * @param  lcd LCD instance
  * @param
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_seg_en(LCD_TypeDef *lcd, uint32_t seg_en)
{
    MODIFY_REG(lcd->SEGEN, LCD_SEGEN_SEG_EN_MSK, seg_en << LCD_SEGEN_SEG_EN_POSS);
}

/**
  * @brief  Get SEG output enable.
  * @param  lcd LCD instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_seg_en(LCD_TypeDef *lcd)
{
    return (READ_BIT(lcd->SEGEN, LCD_SEGEN_SEG_EN_MSK) >> LCD_SEGEN_SEG_EN_POSS);
}

/**
  * @} MD_LCD_Public_Macro
  */

/* Public functions -----------------------------------------------------------*/
void md_rcu_lcd_init(RCU_TypeDef *rcu, md_rcu_lcd_init_typedef *LCD_InitStruct);
void md_lcd_mode(LCD_TypeDef *lcd, md_lcd_mode_typedef *LCD_Mode_Struct);
void md_lcd_comx_segx_enable(LCD_TypeDef *lcd, md_lcd_comx_segx_typedef *LCD_Comx_Segx_Enable_Struct);
void md_lcd_ctrl(LCD_TypeDef *lcd, md_lcd_ctrl_typedef *LCD_InitStruct);
void md_lcd_display_all(void);
void md_lcd_clear_all(void);
void md_lcd_write(LCD_TypeDef *hlcd, uint8_t ComIndex, uint32_t RegisterMask, uint32_t Data);
void md_lcd_swupdate(void);
void md_lcd_display_float_number(float u32DispVal);



/**
  * @} LCD
  */


#endif

/**
  * @} Micro_Driver
  */


#ifdef __cplusplus
}
#endif

#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

