/**********************************************************************************
 *
 * @file    md_iwdt.h
 * @brief   header file of md_iwdt.c
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_IWDT_H
#define __MD_IWDT_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "es32f0930.h"

/** @addtogroup Micro_Driver
  * @{
  */

#if defined(IWDT)

/** @defgroup IWDT IWDT
  * @brief IWDT micro driver
  * @{
  */

/* Public constants -----------------------------------------------------------*/
/** @defgroup MD_IWDT_Public_Constants IWDT Public Constants
  * @{
  */
/** @defgroup MD_IWDT_Key IWDT Key
  * @{
  */
#define IWDT_KEY_RELOAD                 0x0000AAAAU               /*!< IWDT Reload Counter Enable   */
#define IWDT_KEY_ENABLE                 0x0000CCCCU               /*!< IWDT Peripheral Enable       */
#define IWDT_KEY_WR_ACCESS_ENABLE       0x00005555U               /*!< IWDT KR Write Access Enable  */
#define IWDT_KEY_WR_ACCESS_DISABLE      0x00000000U               /*!< IWDT KR Write Access Disable */

/**
  * @} MD_IWDT_Key
  */

/** @defgroup MD_IWDT_Prescaler IWDT Prescaler
  * @{
  */
#define IWDT_PRESCALER_4                0x00000000U                                    /*!< Divider by 4   */
#define IWDT_PRESCALER_8                1U<<IWDT_PR_PR_POSS                            /*!< Divider by 8   */
#define IWDT_PRESCALER_16               2U<<IWDT_PR_PR_POSS                            /*!< Divider by 16  */
#define IWDT_PRESCALER_32               3U<<IWDT_PR_PR_POSS                            /*!< Divider by 32  */
#define IWDT_PRESCALER_64               4U<<IWDT_PR_PR_POSS                            /*!< Divider by 64  */
#define IWDT_PRESCALER_128              5U<<IWDT_PR_PR_POSS                            /*!< Divider by 128 */
#define IWDT_PRESCALER_256              6U<<IWDT_PR_PR_POSS                            /*!< Divider by 256 */
/**
  * @} MD_IWDT_Prescaler
  */

/**
  * @} MD_IWDT_Public_Constants
  */

/* Exported macro ------------------------------------------------------------*/
/** @defgroup IWDT_Exported_Macros IWDT Exported Macros
  * @{
  */

/** @defgroup IWDT_EM_WRITE_READ Common Write and read registers Macros
  * @{
  */

/**
  * @brief  Write a value in IWDT register
  * @param  __INSTANCE__ IWDT Instance
  * @param  __REG__ Register to be written
  * @param  __VALUE__ Value to be written in the register
  * @retval None
  */
#define IWDT_WriteReg(__INSTANCE__, __REG__, __VALUE__) WRITE_REG(__INSTANCE__->__REG__, (__VALUE__))

/**
  * @brief  Read a value in IWDT register
  * @param  __INSTANCE__ IWDT Instance
  * @param  __REG__ Register to be read
  * @retval Register value
  */
#define IWDT_ReadReg(__INSTANCE__, __REG__) READ_REG(__INSTANCE__->__REG__)
/**
  * @} IWDT_EM_WRITE_READ
  */

/**
  * @} IWDT_Exported_Macros
  */


/* Public macro ---------------------------------------------------------------*/
/** @defgroup MD_IWDT_Public_Macros IWDT Public Macros
  * @{
  */

/**
  * @brief  Start the Independent Watchdog
  * @note   if the hardware watchdog option is selected
  * @param  IWDTx IWDT Instance
  * @retval None
  */

__STATIC_INLINE void md_iwdt_set_start(IWDT_TypeDef *IWDTx)
{
    WRITE_REG(IWDTx->KR, IWDT_KEY_ENABLE);
}

/**
  * @brief  Reloads IWDT counter with value defined in the reload register
  * @param  IWDTx IWDT Instance
  * @retval None
  */
__STATIC_INLINE void md_iwdt_set_reload_counter(IWDT_TypeDef *IWDTx)
{
    WRITE_REG(IWDTx->KR, IWDT_KEY_RELOAD);
}

/**
  * @brief  Enable write access to IWDT_PR, IWDT_RLR and IWDT_WINR registers
  * @param  IWDTx IWDT Instance
  * @retval None
  */
__STATIC_INLINE void md_iwdt_set_enable_write_access(IWDT_TypeDef *IWDTx)
{
    WRITE_REG(IWDTx->KR, IWDT_KEY_WR_ACCESS_ENABLE);
}

/**
  * @brief  Disable write access to IWDT_PR, IWDT_RLR and IWDT_WINR registers
  * @param  IWDTx IWDT Instance
  * @retval None
  */
__STATIC_INLINE void md_iwdt_set_disable_write_access(IWDT_TypeDef *IWDTx)
{
    WRITE_REG(IWDTx->KR, IWDT_KEY_WR_ACCESS_DISABLE);
}

/**
  * @brief  Select the prescaler of the IWDT
  * @param  IWDTx IWDT Instance
  * @param  Prescaler This parameter can be one of the following values:
  *         @arg @ref IWDT_PRESCALER_4
  *         @arg @ref IWDT_PRESCALER_8
  *         @arg @ref IWDT_PRESCALER_16
  *         @arg @ref IWDT_PRESCALER_32
  *         @arg @ref IWDT_PRESCALER_64
  *         @arg @ref IWDT_PRESCALER_128
  *         @arg @ref IWDT_PRESCALER_256
  * @retval None
  */
__STATIC_INLINE void md_iwdt_set_prescaler(IWDT_TypeDef *IWDTx, uint32_t Prescaler)
{
    WRITE_REG(IWDTx->PR, IWDT_PR_PR_MSK & Prescaler);
}

/**
  * @brief  Get the selected prescaler of the IWDT
  * @param  IWDTx IWDT Instance
  * @retval Returned value can be one of the following values:
  *         @arg @ref IWDT_PRESCALER_4
  *         @arg @ref IWDT_PRESCALER_8
  *         @arg @ref IWDT_PRESCALER_16
  *         @arg @ref IWDT_PRESCALER_32
  *         @arg @ref IWDT_PRESCALER_64
  *         @arg @ref IWDT_PRESCALER_128
  *         @arg @ref IWDT_PRESCALER_256
  */
__STATIC_INLINE uint32_t md_iwdt_get_prescaler(IWDT_TypeDef *IWDTx)
{
    return (uint32_t)(READ_REG(IWDTx->PR));
}

/**
  * @brief  Specify the IWDT down-counter reload value
  * @param  IWDTx IWDT Instance
  * @param  Counter Value between Min_Data=0 and Max_Data=0x0FFF
  * @retval None
  */
__STATIC_INLINE void md_iwdt_set_reload(IWDT_TypeDef *IWDTx, uint32_t Counter)
{
    WRITE_REG(IWDTx->RLR, IWDT_RLR_RL_MSK & Counter);
}

/**
  * @brief  Get the specified IWDT down-counter reload value
  * @param  IWDTx IWDT Instance
  * @retval Value between Min_Data=0 and Max_Data=0x0FFF
  */
__STATIC_INLINE uint32_t md_iwdt_get_reload(IWDT_TypeDef *IWDTx)
{
    return (uint32_t)(READ_REG(IWDTx->RLR));
}

/**
  * @brief  Specify high limit of the window value to be compared to the down-counter.
  * @param  IWDTx IWDT Instance
  * @param  Window Value between Min_Data=0 and Max_Data=0x0FFF
  * @retval None
  */
__STATIC_INLINE void md_iwdt_set_window(IWDT_TypeDef *IWDTx, uint32_t Window)
{
    WRITE_REG(IWDTx->WINR, IWDT_WINR_WIN_MSK & Window);
}

/**
  * @brief  Get the high limit of the window value specified.
  * @param  IWDTx IWDT Instance
  * @retval Value between Min_Data=0 and Max_Data=0x0FFF
  */
__STATIC_INLINE uint32_t md_iwdt_get_window(IWDT_TypeDef *IWDTx)
{
    return (uint32_t)(READ_REG(IWDTx->WINR));
}

/**
  * @} MD_IWDT_Public_Macros
  */

/** @defgroup IWDT_FLAG_Management FLAG_Management
  * @{
  */

/**
  * @brief  Check if flag BUSY is set or not
  * @param  IWDTx IWDT Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_iwdt_is_active_flag_busy(IWDT_TypeDef *IWDTx)
{
    return (uint32_t)(READ_BIT(IWDTx->FR, IWDT_FR_BUSY_MSK) == (IWDT_FR_BUSY_MSK));
}

/**
  * @brief  Get count value
  * @param  IWDTx IWDT Instance
  * @retval get count value.
  */
__STATIC_INLINE uint32_t md_iwdt_get_count_value(IWDT_TypeDef *IWDTx)
{
    return (uint32_t)(READ_BIT(IWDTx->SR, IWDT_SR_CNT_MSK) >> (IWDT_SR_CNT_POSS));
}


/**
  * @} IWDT_FLAG_Management
  */


/**
  * @} IWDT
  */

/**
  * @} Micro_Driver
  */

#endif /* IWDT) */

#ifdef __cplusplus
}
#endif

#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

