/**
  **************************************************************************************
  * @file    md_gp32c4t.h
  * @brief   GP32C4T Head File
  *
  * @version V1.00
  * @date    31/3/2020
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. AMD rights reserved.
  **************************************************************************************
  */
  
 /* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_GP32C4T_H__
#define __MD_GP32C4T_H__

#ifdef __cplusplus
extern "C"{
#endif

/* Includes -------------------------------------------------------*/
#include "es32f0930.h"

/** @addtogroup Micro_Driver
  * @{
  */
#if defined (GP32C4T1)

/** @defgroup GP32C4T GP32C4T
  * @brief GP32C4T module driver
  * @{
  */


/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private Macros ----------------------------------------------------------*/



/* Public Constants -----------------------------------------------------------*/

/** @defgroup MD_GP32C4T_Public_Constants Public Constants
  * @brief GP32C4T module driver
  * @{
  */
/* GP32C4T_CON1 macros define*/
#define MD_GP32C4T_DEBUGSELECT_INPUT        (0 << GP32C4T_CON1_DBGSEL_POS)      /*!<  Channel input*/
#define MD_GP32C4T_DEBUGSELECT_OUTPUT       (1 << GP32C4T_CON1_DBGSEL_POS)      /*!<  Channel output*/

#define MD_GP32C4T_CLOCKDIVSION_DIV1        (0 << GP32C4T_CON1_DFCKSEL_POSS)    /*!<  tDTS=tCK_INT*/
#define MD_GP32C4T_CLOCKDIVSION_DIV2        (1 << GP32C4T_CON1_DFCKSEL_POSS)    /*!<  tDTS=2*tCK_INT*/
#define MD_GP32C4T_CLOCKDIVSION_DIV4        (2 << GP32C4T_CON1_DFCKSEL_POSS)    /*!<  tDTS=4*tCK_INT*/

#define MD_GP32C4T_CENTERALIGNED_DISABLE    (0 << GP32C4T_CON1_CMSEL_POSS)      /*!<  Edge-aligned mode*/
#define MD_GP32C4T_CENTERALIGNED_DOWN       (1 << GP32C4T_CON1_CMSEL_POSS)      /*!<  Center-aligned mode 1*/
#define MD_GP32C4T_CENTERALIGNED_UP         (2 << GP32C4T_CON1_CMSEL_POSS)      /*!<  Center-aligned mode 2*/
#define MD_GP32C4T_CENTERALIGNED_BOTH       (3 << GP32C4T_CON1_CMSEL_POSS)      /*!<  Center-aligned mode 3*/

#define MD_GP32C4T_UPCOUNTER                (0 << GP32C4T_CON1_DIRSEL_POS)      /*!<  Counter used as upcounter*/
#define MD_GP32C4T_DOWNCOUNTER              (1 << GP32C4T_CON1_DIRSEL_POS)      /*!<  Counter used as downcounter*/

#define MD_GP32C4T_ONEPULSEMODE_DISABLE     (0 << GP32C4T_CON1_SPMEN_POS)       /*!<  Counter is not stopped at update event*/
#define MD_GP32C4T_ONEPULSEMODE_ENABLE      (1 << GP32C4T_CON1_SPMEN_POS)       /*!<  Counter stops counting at the next update event*/

#define MD_GP32C4T_UPDATESOURCE_NORMAL      (0 << GP32C4T_CON1_UERSEL_POS)      /*!<  Counter overflow/underflow, Setting the UG bit, Update generation through the slave mode controller generate an update interrupt*/
#define MD_GP32C4T_UPDATESOURCE_COUNTER     (1 << GP32C4T_CON1_UERSEL_POS)      /*!<  Only counter overflow/underflow generates an update interrupt or DMA request if Enabled*/

/* GP32C4T_CON2 macros define*/
#define MD_GP32C4T_IDLEOUTPUT_LOW           0                                   /*!<  OC1=0 (after a dead-time if OC1N is implemented) when MOE=0*/
#define MD_GP32C4T_IDLEOUTPUT_HIGH          1                                   /*!<  OC1=1 (after a dead-time if OC1N is implemented) when MOE=0*/

#define MD_GP32C4T_TI1INPUT_CH1             (0 << GP32C4T_CON2_I1SEL_POS)       /*!<  The TIMx_CH1 pin is connected to TI1 input*/
#define MD_GP32C4T_TI1INPUT_XOR             (1 << GP32C4T_CON2_I1SEL_POS)       /*!<  The TIMx_CH1, CH2 and CH3 pins are connected to the TI1 input (XOR combination)*/

#define MD_GP32C4T_MASTERMODE_RESET         (0 << GP32C4T_CON2_MMSEL_POSS)      /*!<  Reset mode*/
#define MD_GP32C4T_MASTERMODE_ENABLE        (1 << GP32C4T_CON2_MMSEL_POSS)      /*!<  Enable mode*/
#define MD_GP32C4T_MASTERMODE_UPDATE        (2 << GP32C4T_CON2_MMSEL_POSS)      /*!<  Update mode*/
#define MD_GP32C4T_MASTERMODE_COMPAREPULSE  (3 << GP32C4T_CON2_MMSEL_POSS)      /*!<  Compare Pulse mode*/
#define MD_GP32C4T_MASTERMODE_COMPARE1      (4 << GP32C4T_CON2_MMSEL_POSS)      /*!<  Compare 1 mode*/
#define MD_GP32C4T_MASTERMODE_COMPARE2      (5 << GP32C4T_CON2_MMSEL_POSS)      /*!<  Compare 2 mode*/
#define MD_GP32C4T_MASTERMODE_COMPARE3      (6 << GP32C4T_CON2_MMSEL_POSS)      /*!<  Compare 3 mode*/
#define MD_GP32C4T_MASTERMODE_COMPARE4      (7 << GP32C4T_CON2_MMSEL_POSS)      /*!<  Compare 4 mode*/

/* GP32C4T_SMCON macros define*/
#define MD_GP32C4T_ETRPOLARITY_NONINVERTED  (0 << GP32C4T_SMCON_ETPOL_POS)      /*!<  ETR is non-inverted*/
#define MD_GP32C4T_ETRPOLARITY_INVERTED     (1 << GP32C4T_SMCON_ETPOL_POS)      /*!<  ETR is inverted*/

#define MD_GP32C4T_ETRFILTER_FDIV1          (0 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  No filter*/
#define MD_GP32C4T_ETRFILTER_FDIV1N2        (1 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fCK_INT, N = 2*/
#define MD_GP32C4T_ETRFILTER_FDIV1N4        (2 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fCK_INT, N = 4*/
#define MD_GP32C4T_ETRFILTER_FDIV1N8        (3 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fCK_INT, N = 8*/
#define MD_GP32C4T_ETRFILTER_FDIV2N6        (4 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 2, N = 6*/
#define MD_GP32C4T_ETRFILTER_FDIV2N8        (5 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 2, N = 8*/
#define MD_GP32C4T_ETRFILTER_FDIV4N6        (6 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 4, N = 6*/
#define MD_GP32C4T_ETRFILTER_FDIV4N8        (7 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 4, N = 8*/
#define MD_GP32C4T_ETRFILTER_FDIV8N6        (8 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 8, N = 6*/
#define MD_GP32C4T_ETRFILTER_FDIV8N8        (9 << GP32C4T_SMCON_ETFLT_POSS)     /*!<  fSAMPLING = fDTS / 8, N = 8*/
#define MD_GP32C4T_ETRFILTER_FDIV16N5       (10 << GP32C4T_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 16, N = 5*/
#define MD_GP32C4T_ETRFILTER_FDIV16N6       (11 << GP32C4T_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 16, N = 6*/
#define MD_GP32C4T_ETRFILTER_FDIV16N8       (12 << GP32C4T_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 16, N = 8*/
#define MD_GP32C4T_ETRFILTER_FDIV32N5       (13 << GP32C4T_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 32, N = 5*/
#define MD_GP32C4T_ETRFILTER_FDIV32N6       (14 << GP32C4T_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 32, N = 6*/
#define MD_GP32C4T_ETRFILTER_FDIV32N8       (15 << GP32C4T_SMCON_ETFLT_POSS)    /*!<  fSAMPLING = fDTS / 32, N = 8*/

#define MD_GP32C4T_MASTERSLAVE_NODELAY      (0 << GP32C4T_SMCON_MSCFG_POS)      /*!<  No action*/
#define MD_GP32C4T_MASTERSLAVE_DELAY        (1 << GP32C4T_SMCON_MSCFG_POS)      /*!<  The effect of an event on the trigger input (TRGI) is delayed to allow a perfect*/

#define MD_GP32C4T_TRIGGERSELECT_ITR0       (0 << GP32C4T_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 0*/
#define MD_GP32C4T_TRIGGERSELECT_ITR1       (1 << GP32C4T_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 1*/
#define MD_GP32C4T_TRIGGERSELECT_ITR2       (2 << GP32C4T_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 2*/
#define MD_GP32C4T_TRIGGERSELECT_ITR3       (3 << GP32C4T_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 3*/
#define MD_GP32C4T_TRIGGERSELECT_TI1FED     (4 << GP32C4T_SMCON_TSSEL_POSS)     /*!<  TI1 Edge Detector*/
#define MD_GP32C4T_TRIGGERSELECT_TI1FP1     (5 << GP32C4T_SMCON_TSSEL_POSS)     /*!<  Filtered Timer Input 1*/
#define MD_GP32C4T_TRIGGERSELECT_TI2FP2     (6 << GP32C4T_SMCON_TSSEL_POSS)     /*!<  Filtered Timer Input 2*/
#define MD_GP32C4T_TRIGGERSELECT_ETRF       (7 << GP32C4T_SMCON_TSSEL_POSS)     /*!<  External Trigger input*/

#define MD_GP32C4T_OCREFCLEAR_CMP           (0 << GP32C4T_SMCON_CHCSEL_POS)     /*!<  OCREF_CLR is connected to the CMP_IN input*/
#define MD_GP32C4T_OCREFCLEAR_ETRF          (1 << GP32C4T_SMCON_CHCSEL_POS)     /*!<  OCREF_CLR is connected to ETRF*/

#define MD_GP32C4T_SLAVEMODE_DISABLE        (0 << GP32C4T_SMCON_SMODS_POSS)     /*!<  Slave mode disabled*/
#define MD_GP32C4T_SLAVEMODE_ENCODER1       (1 << GP32C4T_SMCON_SMODS_POSS)     /*!<  Encoder mode 1*/
#define MD_GP32C4T_SLAVEMODE_ENCODER2       (2 << GP32C4T_SMCON_SMODS_POSS)     /*!<  Encoder mode 2*/
#define MD_GP32C4T_SLAVEMODE_ENCODER3       (3 << GP32C4T_SMCON_SMODS_POSS)     /*!<  Encoder mode 3*/
#define MD_GP32C4T_SLAVEMODE_RESET          (4 << GP32C4T_SMCON_SMODS_POSS)     /*!<  Reset Mode*/
#define MD_GP32C4T_SLAVEMODE_GATED          (5 << GP32C4T_SMCON_SMODS_POSS)     /*!<  Gated Mode*/
#define MD_GP32C4T_SLAVEMODE_TRIGGER        (6 << GP32C4T_SMCON_SMODS_POSS)     /*!<  Trigger Mode*/
#define MD_GP32C4T_SLAVEMODE_EXTERNALCLOCK  (7 << GP32C4T_SMCON_SMODS_POSS)     /*!<  External Clock Mode 1*/

/* GP32C4T_IER IDR IVS RIF IFM ICR macros define*/
#define MD_GP32C4T_INTERRUPTFALG_CC4OI      (1 << GP32C4T_IER_CH4OVI_POS)       /*!<  Capture/Compare 4 overcapture interrupt*/
#define MD_GP32C4T_INTERRUPTFALG_CC3OI      (1 << GP32C4T_IER_CH3OVI_POS)       /*!<  Capture/Compare 3 overcapture interrupt*/
#define MD_GP32C4T_INTERRUPTFALG_CC2OI      (1 << GP32C4T_IER_CH2OVI_POS)       /*!<  Capture/Compare 2 overcapture interrupt*/
#define MD_GP32C4T_INTERRUPTFALG_CC1OI      (1 << GP32C4T_IER_CH1OVI_POS)       /*!<  Capture/Compare 1 overcapture interrupt*/
#define MD_GP32C4T_INTERRUPTFALG_TI         (1 << GP32C4T_IER_TRGI_POS)         /*!<  Trigger interrupt*/
#define MD_GP32C4T_INTERRUPTFALG_CC4I       (1 << GP32C4T_IER_CH4I_POS)         /*!<  Capture/Compare 4 interrupt*/
#define MD_GP32C4T_INTERRUPTFALG_CC3I       (1 << GP32C4T_IER_CH3I_POS)         /*!<  Capture/Compare 3 interrupt*/
#define MD_GP32C4T_INTERRUPTFALG_CC2I       (1 << GP32C4T_IER_CH2I_POS)         /*!<  Capture/Compare 2 interrupt*/
#define MD_GP32C4T_INTERRUPTFALG_CC1I       (1 << GP32C4T_IER_CH1I_POS)         /*!<  Capture/Compare 1 interrupt*/
#define MD_GP32C4T_INTERRUPTFALG_UI         (1 << GP32C4T_IER_UI_POS)           /*!<  Update interrupt*/

/* GP32C4T_SGE macros define*/
#define MD_GP32C4T_EVENTGENERATION_TG       (1 << GP32C4T_SGE_SGTRG_POS)        /*!<  Trigger generation*/  
#define MD_GP32C4T_EVENTGENERATION_CC4G     (1 << GP32C4T_SGE_SGCH4_POS)        /*!<  Capture/Compare 4 generation*/
#define MD_GP32C4T_EVENTGENERATION_CC3G     (1 << GP32C4T_SGE_SGCH3_POS)        /*!<  Capture/Compare 3 generation*/
#define MD_GP32C4T_EVENTGENERATION_CC2G     (1 << GP32C4T_SGE_SGCH2_POS)        /*!<  Capture/Compare 2 generation*/
#define MD_GP32C4T_EVENTGENERATION_CC1G     (1 << GP32C4T_SGE_SGCH1_POS)        /*!<  Capture/Compare 1 generation*/
#define MD_GP32C4T_EVENTGENERATION_UG       (1 << GP32C4T_SGE_SGU_POS)          /*!<  Update generation*/

/* GP32C4T_CHMR1 CHMR2 output macros define*/
#define MD_GP32C4T_OUTPUTMODE_DISABLE       0                                   /*!<  Frozen*/
#define MD_GP32C4T_OUTPUTMODE_HIGHONMSTCH   1                                   /*!<  Set channel 1 to active level on match*/
#define MD_GP32C4T_OUTPUTMODE_LOWONMSTCH    2                                   /*!<  Set channel 1 to inactive level on match*/
#define MD_GP32C4T_OUTPUTMODE_TOGGLE        3                                   /*!<  Toggle*/
#define MD_GP32C4T_OUTPUTMODE_FORCELOW      4                                   /*!<  Force inactive level*/
#define MD_GP32C4T_OUTPUTMODE_FORCEHIGH     5                                   /*!<  Force active level*/
#define MD_GP32C4T_OUTPUTMODE_PWMMODE1      6                                   /*!<  PWM mode 1*/
#define MD_GP32C4T_OUTPUTMODE_PWMMODE2      7                                   /*!<  PWM mode 2*/

#define MD_GP32C4T_CHMODE_OUTPUT            0                                   /*!<  CCx channel is configured as output*/
#define MD_GP32C4T_CHMODE_INPUT_DIRECT      1                                   /*!<  CCx channel is configured as input, ICx is mapped direct*/
#define MD_GP32C4T_CHMODE_INPUT_INDIRECT    2                                   /*!<  CCx channel is configured as input, ICx is mapped indirect*/
#define MD_GP32C4T_CHMODE_INPUT_TRC         3                                   /*!<  CCx channel is configured as input, ICx is mapped TRC*/

/* GP32C4T_CHMR1 CHMR2 input macros define*/
#define MD_GP32C4T_INPUTFILTER_FDIV1        0                                   /*!<  No filter*/
#define MD_GP32C4T_INPUTFILTER_FDIV1N2      1                                   /*!<  fSAMPLING = fCK_INT, N = 2*/
#define MD_GP32C4T_INPUTFILTER_FDIV1N4      2                                   /*!<  fSAMPLING = fCK_INT, N = 4*/
#define MD_GP32C4T_INPUTFILTER_FDIV1N8      3                                   /*!<  fSAMPLING = fCK_INT, N = 8*/
#define MD_GP32C4T_INPUTFILTER_FDIV2N6      4                                   /*!<  fSAMPLING = fDTS / 2, N = 6*/
#define MD_GP32C4T_INPUTFILTER_FDIV2N8      5                                   /*!<  fSAMPLING = fDTS / 2, N = 8*/
#define MD_GP32C4T_INPUTFILTER_FDIV4N6      6                                   /*!<  fSAMPLING = fDTS / 4, N = 6*/
#define MD_GP32C4T_INPUTFILTER_FDIV4N8      7                                   /*!<  fSAMPLING = fDTS / 4, N = 8*/
#define MD_GP32C4T_INPUTFILTER_FDIV8N6      8                                   /*!<  fSAMPLING = fDTS / 8, N = 6*/
#define MD_GP32C4T_INPUTFILTER_FDIV8N8      9                                   /*!<  fSAMPLING = fDTS / 8, N = 8*/
#define MD_GP32C4T_INPUTFILTER_FDIV16N5     10                                  /*!<  fSAMPLING = fDTS / 16, N = 5*/
#define MD_GP32C4T_INPUTFILTER_FDIV16N6     11                                  /*!<  fSAMPLING = fDTS / 16, N = 6*/
#define MD_GP32C4T_INPUTFILTER_FDIV16N8     12                                  /*!<  fSAMPLING = fDTS / 16, N = 8*/
#define MD_GP32C4T_INPUTFILTER_FDIV32N5     13                                  /*!<  fSAMPLING = fDTS / 32, N = 5*/
#define MD_GP32C4T_INPUTFILTER_FDIV32N6     14                                  /*!<  fSAMPLING = fDTS / 32, N = 6*/
#define MD_GP32C4T_INPUTFILTER_FDIV32N8     15                                  /*!<  fSAMPLING = fDTS / 32, N = 8*/

#define MD_GP32C4T_INPUTPRESCALE_DIV1       0                                   /*!<  no prescaler*/
#define MD_GP32C4T_INPUTPRESCALE_DIV2       1                                   /*!<  capture is done once every 2 events*/
#define MD_GP32C4T_INPUTPRESCALE_DIV4       2                                   /*!<  capture is done once every 4 events*/
#define MD_GP32C4T_INPUTPRESCALE_DIV8       3                                   /*!<  capture is done once every 8 events*/

/* GP32C4T_CCEP input macros define*/
#define MD_GP32C4T_OUTPUTPOLARITY_HIGH      0                                   /*!<  active high*/
#define MD_GP32C4T_OUTPUTPOLARITY_LOW       1                                   /*!<  active low*/

/**
  * @} MD_GP32C4T_Public_Constants
  */

/* Public Macro ------------------------------------------------------------*/
/** @defgroup MD_GP32C4T_Public_Macro Public Macro
  * @brief GP32C4T module driver
  * @{
  */
  
/**
  * @brief  Timer CON1 setup.
  * @param  timx GP32C4T instance
  * @param  value (DBGSEL | CMPSEL | CMPSELP | DFCKSEL | ARPEN | CMSEL | DIRSEL | SPMEN | UERSEL | DISUE | CNTEN)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_con1(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CON1, value);
}

/**
  * @brief  GET Timer CON1 register value.
  * @param  timx GP32C4T instance
  * @retval Timer CON1 register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_con1(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->CON1));
}

/**
  * @brief  Timer debug mode select.
  * @param  timx GP32C4T instance
  * @param  DebugSelect Debug mode output select
            @arg @ref MD_GP32C4T_DEBUGSELECT_INPUT
            @arg @ref MD_GP32C4T_DEBUGSELECT_OUTPUT
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_con1_dbgsel(GP32C4T_TypeDef *timx, uint32_t DebugSelect)
{
  MODIFY_REG(timx->CON1, GP32C4T_CON1_DBGSEL_MSK, DebugSelect);
}

/**
  * @brief  Get timer debug mode select.
  * @param  timx GP32C4T instance
  * @retval Timer debug mode select.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_con1_dbgsel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP32C4T_CON1_DBGSEL_MSK) >> GP32C4T_CON1_DBGSEL_POS);
}

/**
  * @brief  Timer clock division setup.
  * @param  timx GP32C4T instance
  * @param  ClockDivision Clock division
            @arg @ref MD_GP32C4T_CLOCKDIVSION_DIV1
            @arg @ref MD_GP32C4T_CLOCKDIVSION_DIV2
            @arg @ref MD_GP32C4T_CLOCKDIVSION_DIV4
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_con1_dfcksel(GP32C4T_TypeDef *timx, uint32_t ClockDivision)
{
  MODIFY_REG(timx->CON1, GP32C4T_CON1_DFCKSEL_MSK, ClockDivision);
}

/**
  * @brief  Get timer Clock division.
  * @param  timx GP32C4T instance
  * @retval Timer Clock division.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_con1_dfcksel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP32C4T_CON1_DFCKSEL_MSK) >> GP32C4T_CON1_DFCKSEL_POSS);
}

/**
  * @brief  Timer auto-reload preload enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_con1_arpen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CON1, GP32C4T_CON1_ARPEN_MSK);
}

/**
  * @brief  Timer auto-reload preload disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_con1_arpen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, GP32C4T_CON1_ARPEN_MSK);
}

/**
  * @brief  Indicates whether the timer auto-reload preload is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_con1_arpen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP32C4T_CON1_ARPEN_MSK) == (GP32C4T_CON1_ARPEN_MSK));
}

/**
  * @brief  Timer center-aligned mode selection setup.
  * @param  timx GP32C4T instance
  * @param  CenterAlignedMode center-aligned mode selection
            @arg @ref MD_GP32C4T_CENTERALIGNED_DISABLE
            @arg @ref MD_GP32C4T_CENTERALIGNED_DOWN
            @arg @ref MD_GP32C4T_CENTERALIGNED_UP
            @arg @ref MD_GP32C4T_CENTERALIGNED_BOTH
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_con1_cmsel(GP32C4T_TypeDef *timx, uint32_t CenterAlignedMode)
{
  MODIFY_REG(timx->CON1, GP32C4T_CON1_CMSEL_MSK, CenterAlignedMode);
}

/**
  * @brief  Get timer center-aligned mode selection.
  * @param  timx GP32C4T instance
  * @retval Timer center-aligned mode selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_con1_cmsel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP32C4T_CON1_CMSEL_MSK) >> GP32C4T_CON1_CMSEL_POSS);
}

/**
  * @brief  Timer counting direction setup.
  * @param  timx GP32C4T instance
  * @param  direction Counting direction
            @arg @ref MD_GP32C4T_UPCOUNTER
            @arg @ref MD_GP32C4T_DOWNCOUNTER
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_con1_dirsel(GP32C4T_TypeDef *timx, uint32_t direction)
{
  MODIFY_REG(timx->CON1, GP32C4T_CON1_DIRSEL_MSK, direction);
}

/**
  * @brief  Get timer counting direction.
  * @param  timx GP32C4T instance
  * @retval Timer counting direction.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_con1_dirsel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP32C4T_CON1_DIRSEL_MSK) >> GP32C4T_CON1_DIRSEL_POS);
}

/**
  * @brief  Timer one pulse mode enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_con1_spmen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CON1, GP32C4T_CON1_SPMEN_MSK);
}

/**
  * @brief  Timer one pulse mode disable.
  * @param  timx GP32C4T instance
  * @retval None.
  */

__STATIC_INLINE void md_gp32c4t_disable_con1_spmen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, GP32C4T_CON1_SPMEN_MSK);
}

/**
  * @brief  Indicates whether the timer one pulse mode is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_con1_spmen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP32C4T_CON1_SPMEN_MSK) == (GP32C4T_CON1_SPMEN_MSK));
}

/**
  * @brief  Timer update request source setup.
  * @param  timx GP32C4T instance
  * @param  UpdateSource Update request source select
            @arg @ref MD_GP32C4T_UPDATESOURCE_NORMAL
            @arg @ref MD_GP32C4T_UPDATESOURCE_COUNTER
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_con1_uersel(GP32C4T_TypeDef *timx, uint32_t UpdateSource)
{
  MODIFY_REG(timx->CON1, GP32C4T_CON1_UERSEL_MSK, UpdateSource);
}

/**
  * @brief  Get timer update request source.
  * @param  timx GP32C4T instance
  * @retval Timer update request source.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_con1_uersel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP32C4T_CON1_UERSEL_MSK) >> GP32C4T_CON1_UERSEL_POS);
}

/**
  * @brief  Timer update event enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_con1_disue(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, GP32C4T_CON1_DISUE_MSK);
}

/**
  * @brief  Timer update event disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disble_con1_disue(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CON1, GP32C4T_CON1_DISUE_MSK);
}

/**
  * @brief  Indicates whether the timer update event is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_con1_disue(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP32C4T_CON1_DISUE_MSK) == (GP32C4T_CON1_DISUE_MSK));
}

/**
  * @brief  Timer counter enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_con1_cnten(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CON1, GP32C4T_CON1_CNTEN_MSK);
}
  

/**
  * @brief  Timer counter disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_con1_cnten(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, GP32C4T_CON1_CNTEN_MSK);
}

/**
  * @brief  Indicates whether the timer counter is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_con1_cnten(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP32C4T_CON1_CNTEN_MSK) == (GP32C4T_CON1_CNTEN_MSK));
}

/**
  * @brief  Timer CON2 setup.
  * @param  timx GP32C4T instance
  * @param  value (I1SEL | MMSEL | CCDMASEL)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_con2(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CON2, value);
}

/**
  * @brief  GET Timer CON2 register value.
  * @param  timx GP32C4T instance
  * @retval Timer CON2 register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_con2(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->CON2));
}

/**
  * @brief  Timer TI1 selection setup.
  * @param  timx GP32C4T instance
  * @param  TI1Input TI1 input select
            @arg @ref MD_GP32C4T_TI1INPUT_CH1
            @arg @ref MD_GP32C4T_TI1INPUT_XOR
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_con2_i1sel(GP32C4T_TypeDef *timx, uint32_t TI1Input)
{
  MODIFY_REG(timx->CON2, GP32C4T_CON2_I1SEL_MSK, TI1Input);
}

/**
  * @brief  Get timer TI1 selection.
  * @param  timx GP32C4T instance
  * @retval Timer TI1 selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_con2_i1sel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON2, GP32C4T_CON2_I1SEL_MSK) >> GP32C4T_CON2_I1SEL_POS);
}

/**
  * @brief  Timer master mode selection setup.
  * @param  timx GP32C4T instance
  * @param  MasterMode Master mode selection
            @arg @ref MD_GP32C4T_MASTERMODE_RESET
            @arg @ref MD_GP32C4T_MASTERMODE_ENABLE
            @arg @ref MD_GP32C4T_MASTERMODE_UPDATE
            @arg @ref MD_GP32C4T_MASTERMODE_COMPAREPULSE
            @arg @ref MD_GP32C4T_MASTERMODE_COMPARE1
            @arg @ref MD_GP32C4T_MASTERMODE_COMPARE2
            @arg @ref MD_GP32C4T_MASTERMODE_COMPARE3
            @arg @ref MD_GP32C4T_MASTERMODE_COMPARE4
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_con2_mmsel(GP32C4T_TypeDef *timx, uint32_t MasterMode)
{
  MODIFY_REG(timx->CON2, GP32C4T_CON2_MMSEL_MSK, MasterMode);
}

/**
  * @brief  Get timer master mode selection.
  * @param  timx GP32C4T instance
  * @retval Timer master mode selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_con2_mmsel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CON2, GP32C4T_CON2_MMSEL_MSK) >> GP32C4T_CON2_MMSEL_POSS);
}

/**
  * @brief  Timer SMCON setup.
  * @param  timx GP32C4T instance
  * @param  value (ETPOL | ECM2EN | ETFLT | MSCFG | TSSEL | CHCSEL | SMODS)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_smcon(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->SMCON, value);
}

/**
  * @brief  GET Timer SMCON register value.
  * @param  timx GP32C4T instance
  * @retval Timer SMCON register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_smcon(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->SMCON));
}

/**
  * @brief  Timer external trigger polarity setup.
  * @param  timx GP32C4T instance
  * @param  ETRPolarity External trigger polarity
            @arg @ref MD_GP32C4T_ETRPOLARITY_NONINVERTED
            @arg @ref MD_GP32C4T_ETRPOLARITY_INVERTED
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_smcon_etpol(GP32C4T_TypeDef *timx, uint32_t ETRPolarity)
{
  MODIFY_REG(timx->SMCON, GP32C4T_SMCON_ETPOL_MSK, ETRPolarity);
}

/**
  * @brief  Get timer external trigger polarity.
  * @param  timx GP32C4T instance
  * @retval Timer external trigger polarity.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_smcon_etpol(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP32C4T_SMCON_ETPOL_MSK) >> GP32C4T_SMCON_ETPOL_POS);
}

/**
  * @brief  Timer external clock enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_smcon_ecm2en(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->SMCON, GP32C4T_SMCON_ECM2EN_MSK);
}
  

/**
  * @brief  Timer external clock disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_smcon_ecm2en(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->SMCON, GP32C4T_SMCON_ECM2EN_MSK);
}

/**
  * @brief  Indicates whether the timer external clock is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_smcon_ecm2en(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP32C4T_SMCON_ECM2EN_MSK) == (GP32C4T_SMCON_ECM2EN_MSK));
}

/**
  * @brief  Timer external trigger filter setup.
  * @param  timx GP32C4T instance 
  * @param  ETRFilter External trigger filter
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV1
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV1N2
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV1N4
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV1N8
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV2N6
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV2N8
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV4N6
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV4N8
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV8N6
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV8N8
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV16N5
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV16N6
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV16N8
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV32N5
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV32N6
            @arg @ref MD_GP32C4T_ETRFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_smcon_etflt(GP32C4T_TypeDef *timx, uint32_t ETRFilter)
{
  MODIFY_REG(timx->SMCON, GP32C4T_SMCON_ETFLT_MSK, ETRFilter);
}

/**
  * @brief  Get timer external trigger filter.
  * @param  timx GP32C4T instance
  * @retval Timer external trigger filter.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_smcon_etflt(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP32C4T_SMCON_ETFLT_MSK) >> GP32C4T_SMCON_ETFLT_POSS);
}

/**
  * @brief  Timer master/slave mode setup.
  * @param  timx GP32C4T instance 
  * @param  MSMode master/slave mode
            @arg @ref MD_GP32C4T_MASTERSLAVE_NODELAY
            @arg @ref MD_GP32C4T_MASTERSLAVE_DELAY
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_smcon_mscfg(GP32C4T_TypeDef *timx, uint32_t MSMode)
{
  MODIFY_REG(timx->SMCON, GP32C4T_SMCON_MSCFG_MSK, MSMode);
}

/**
  * @brief  Get timer master/slave mode.
  * @param  timx GP32C4T instance
  * @retval Timer master/slave mode.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_smcon_mscfg(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP32C4T_SMCON_MSCFG_MSK) >> GP32C4T_SMCON_MSCFG_POS);
}

/**
  * @brief  Timer trigger selection setup.
  * @param  timx GP32C4T instance 
  * @param  TriggerSelect Trigger selection
            @arg @ref MD_GP32C4T_TRIGGERSELECT_ITR0
            @arg @ref MD_GP32C4T_TRIGGERSELECT_ITR1
            @arg @ref MD_GP32C4T_TRIGGERSELECT_ITR2
            @arg @ref MD_GP32C4T_TRIGGERSELECT_ITR3
            @arg @ref MD_GP32C4T_TRIGGERSELECT_TI1FED
            @arg @ref MD_GP32C4T_TRIGGERSELECT_TI1FP1
            @arg @ref MD_GP32C4T_TRIGGERSELECT_TI2FP2
            @arg @ref MD_GP32C4T_TRIGGERSELECT_ETRF
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_smcon_tssel(GP32C4T_TypeDef *timx, uint32_t TriggerSelect)
{
  MODIFY_REG(timx->SMCON, GP32C4T_SMCON_TSSEL_MSK, TriggerSelect);
}

/**
  * @brief  Get timer trigger selection.
  * @param  timx GP32C4T instance
  * @retval Timer trigger selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_smcon_tssel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP32C4T_SMCON_TSSEL_MSK) >> GP32C4T_SMCON_TSSEL_POSS);
}

/**
  * @brief  Timer OCREF clear selection setup.
  * @param  timx GP32C4T instance 
  * @param  OCREFSelect OCREF clear selection
            @arg @ref MD_GP32C4T_OCREFCLEAR_CMP
            @arg @ref MD_GP32C4T_OCREFCLEAR_ETRF
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_smcon_chcsel(GP32C4T_TypeDef *timx, uint32_t OCREFSelect)
{
  MODIFY_REG(timx->SMCON, GP32C4T_SMCON_CHCSEL_MSK, OCREFSelect);
}

/**
  * @brief  Get timer OCREF clear selection.
  * @param  timx GP32C4T instance
  * @retval Timer OCREF clear selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_smcon_chcsel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP32C4T_SMCON_CHCSEL_MSK) >> GP32C4T_SMCON_CHCSEL_POS);
}

/**
  * @brief  Timer slave mode selection setup.
  * @param  timx GP32C4T instance 
  * @param  SlaveMode Slave mode selection
            @arg @ref MD_GP32C4T_SLAVEMODE_DISABLE
            @arg @ref MD_GP32C4T_SLAVEMODE_ENCODER1
            @arg @ref MD_GP32C4T_SLAVEMODE_ENCODER2
            @arg @ref MD_GP32C4T_SLAVEMODE_ENCODER3
            @arg @ref MD_GP32C4T_SLAVEMODE_RESET
            @arg @ref MD_GP32C4T_SLAVEMODE_GATED
            @arg @ref MD_GP32C4T_SLAVEMODE_TRIGGER
            @arg @ref MD_GP32C4T_SLAVEMODE_EXTERNALCLOCK
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_smcon_smods(GP32C4T_TypeDef *timx, uint32_t SlaveMode)
{
  MODIFY_REG(timx->SMCON, GP32C4T_SMCON_SMODS_MSK, SlaveMode);
}

/**
  * @brief  Get timer slave mode selection.
  * @param  timx GP32C4T instance
  * @retval Timer slave mode selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_smcon_smods(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP32C4T_SMCON_SMODS_MSK) >> GP32C4T_SMCON_SMODS_POSS);
}

/**
  * @brief  Timer IER setup.
  * @param  timx GP32C4T instance
  * @param  value (CH4OVI | CH3OVI | CH2OVI | CH1OVI | TRGI | CH4I | CH3I | CH2I | CH1I | UI)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ier(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->IER, value);
}

/**
  * @brief  Timer cpture/compare 4 overcapture interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_ch4ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_CH4OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 3 overcapture interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_ch3ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_CH3OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 2 overcapture interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_ch2ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_CH2OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 1 overcapture interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_ch1ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_CH1OVI_MSK);
}

/**
  * @brief  Timer trigger interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_trgi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_TRGI_MSK);
}

/**
  * @brief  Timer capture/compare 4 interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_ch4i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_CH4I_MSK);
}

/**
  * @brief  Timer capture/compare 3 interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_ch3i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_CH3I_MSK);
}

/**
  * @brief  Timer capture/compare 2 interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_ch2i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_CH2I_MSK);
}

/**
  * @brief  Timer capture/compare 1 interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_ch1i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_CH1I_MSK);
}

/**
  * @brief  Timer update interrupt enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ier_ui(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP32C4T_IER_UI_MSK);
}

/**
  * @brief  Timer IDR setup.
  * @param  timx GP32C4T instance
  * @param  value (CH4OVI | CH3OVI | CH2OVI | CH1OVI | TRGI | CH4I | CH3I | CH2I | CH1I | UI)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_idr(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->IDR, value);
}

/**
  * @brief  Timer cpture/compare 4 overcapture interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_ch4ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_CH4OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 3 overcapture interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_ch3ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_CH3OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 2 overcapture interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_ch2ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_CH2OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 1 overcapture interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_ch1ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_CH1OVI_MSK);
}

/**
  * @brief  Timer trigger interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_trgi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_TRGI_MSK);
}

/**
  * @brief  Timer capture/compare 4 interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_ch4i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_CH4I_MSK);
}

/**
  * @brief  Timer capture/compare 3 interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_ch3i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_CH3I_MSK);
}

/**
  * @brief  Timer capture/compare 2 interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_ch2i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_CH2I_MSK);
}

/**
  * @brief  Timer capture/compare 1 interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_ch1i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_CH1I_MSK);
}

/**
  * @brief  Timer update interrupt disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_idr_ui(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP32C4T_IDR_UI_MSK);
}

/**
  * @brief  Get timer IVS setup.
  * @param  timx GP32C4T instance
  * @retval Timer IVS setup.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ivs(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->IVS));
}

/**
  * @brief  Indicates whether the timer capture/compare 4 overcapture interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_ch4ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_CH4OVI_MSK) == (GP32C4T_IVS_CH4OVI_MSK));
}

/**
  * @brief  Indicates whether the timer capture/compare 3 overcapture interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_ch3ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_CH3OVI_MSK) == (GP32C4T_IVS_CH3OVI_MSK));
}

/**
  * @brief  Indicates whether the timer cpture/compare 2 overcapture interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_ch2ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_CH2OVI_MSK) == (GP32C4T_IVS_CH2OVI_MSK));
}

/**
  * @brief  Indicates whether the timer capture/compare 1 overcapture interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_ch1ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_CH1OVI_MSK) == (GP32C4T_IVS_CH1OVI_MSK));
}

/**
  * @brief  Indicates whether the timer trigger interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_trgi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_TRGI_MSK) == (GP32C4T_IVS_TRGI_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 4 interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_ch4i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_CH4I_MSK) == (GP32C4T_IVS_CH4I_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 3 interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_ch3i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_CH3I_MSK) == (GP32C4T_IVS_CH3I_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_ch2i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_CH2I_MSK) == (GP32C4T_IVS_CH2I_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_ch1i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_CH1I_MSK) == (GP32C4T_IVS_CH1I_MSK));
}

/**
  * @brief  Indicates whether the timer update interrupt is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ivs_ui(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP32C4T_IVS_UI_MSK) == (GP32C4T_IVS_UI_MSK));
}

/**
  * @brief  Get timer RIF flag.
  * @param  timx GP32C4T instance
  * @retval Timer RIF flag.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_rif(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->RIF));
}

/**
  * @brief  Get timer capture/compare 4 overcapture interrupt flag.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_ch4ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_CH4OVI_MSK) == (GP32C4T_RIF_CH4OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 3 overcapture interrupt flag.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_ch3ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_CH3OVI_MSK) == (GP32C4T_RIF_CH3OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 2 overcapture interrupt flag.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_ch2ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_CH2OVI_MSK) == (GP32C4T_RIF_CH2OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 1 overcapture interrupt flag.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_ch1ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_CH1OVI_MSK) == (GP32C4T_RIF_CH1OVI_MSK));
}

/**
  * @brief  Get timer trigger interrupt flag. 
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_trgi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_TRGI_MSK) == (GP32C4T_RIF_TRGI_MSK));
}

/**
  * @brief  Get timer capture/compare 4 interrupt flag.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_ch4i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_CH4I_MSK) == (GP32C4T_RIF_CH4I_MSK));
}

/**
  * @brief  Get timer capture/compare 3 interrupt flag.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_ch3i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_CH3I_MSK) == (GP32C4T_RIF_CH3I_MSK));
}

/**
  * @brief  Get timer capture/compare 2 interrupt flag.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_ch2i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_CH2I_MSK) == (GP32C4T_RIF_CH2I_MSK));
}

/**
  * @brief  Get timer capture/compare 1 interrupt flag.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_ch1i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_CH1I_MSK) == (GP32C4T_RIF_CH1I_MSK));
}

/**
  * @brief  Get timer update interrupt flag.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_rif_ui(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP32C4T_RIF_UI_MSK) == (GP32C4T_RIF_UI_MSK));
}

/**
  * @brief  Get timer IFM flag.
  * @param  timx GP32C4T instance
  * @retval Timer IFM flag.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ifm(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->IFM));
}

/**
  * @brief  Get timer capture/compare 4 overcapture interrupt flag masked.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_ch4ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_CH4OVI_MSK) == (GP32C4T_IFM_CH4OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 3 overcapture interrupt flag masked.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_ch3ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_CH3OVI_MSK) == (GP32C4T_IFM_CH3OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 2 overcapture interrupt flag masked.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_ch2ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_CH2OVI_MSK) == (GP32C4T_IFM_CH2OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 1 overcapture interrupt flag masked.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_ch1ovi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_CH1OVI_MSK) == (GP32C4T_IFM_CH1OVI_MSK));
}

/**
  * @brief  Get timer trigger interrupt flag masked. 
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_trgi(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_TRGI_MSK) == (GP32C4T_IFM_TRGI_MSK));
}

/**
  * @brief  Get timer capture/compare 4 interrupt flag masked.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_ch4i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_CH4I_MSK) == (GP32C4T_IFM_CH4I_MSK));
}

/**
  * @brief  Get timer capture/compare 3 interrupt flag masked.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_ch3i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_CH3I_MSK) == (GP32C4T_IFM_CH3I_MSK));
}

/**
  * @brief  Get timer capture/compare 2 interrupt flag masked.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_ch2i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_CH2I_MSK) == (GP32C4T_IFM_CH2I_MSK));
}

/**
  * @brief  Get timer capture/compare 1 interrupt flag masked.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_ch1i(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_CH1I_MSK) == (GP32C4T_IFM_CH1I_MSK));
}

/**
  * @brief  Get timer update interrupt flag masked.  
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp32c4t_is_active_flag_ifm_ui(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP32C4T_IFM_UI_MSK) == (GP32C4T_IFM_UI_MSK));
}

/**
  * @brief  Timer ICR setup.
  * @param  timx GP32C4T instance
  * @param  value (CC4OI | CC3OI | CC2OI | CC1OI | TI | CC4I | CC3I | CC2I | CC1I | UI)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_icr(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->ICR, value);
}

/**
  * @brief  Clear timer cpture/compare 4 overcapture interrupt flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_ch4ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_CH4OVI_MSK);
}

/**
  * @brief  Clear timer cpture/compare 3 overcapture interrupt flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_ch3ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_CH3OVI_MSK);
}

/**
  * @brief  Clear timer cpture/compare 2 overcapture interrupt flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_ch2ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_CH2OVI_MSK);
}

/**
  * @brief  Clear timer cpture/compare 1 overcapture interrupt flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_ch1ovi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_CH1OVI_MSK);
}

/**
  * @brief  Clear timer trigger interrupt flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_trgi(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_TRGI_MSK);
}

/**
  * @brief  Clear timer capture/compare 4 interrupt flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_ch4i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_CH4I_MSK);
}

/**
  * @brief  Clear timer capture/compare 3 interrupt flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_ch3i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_CH3I_MSK);
}

/**
  * @brief  Clear timer capture/compare 2 interrupt flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_ch2i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_CH2I_MSK);
}

/**
  * @brief  Clear timer capture/compare 1 interrupt flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_ch1i(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_CH1I_MSK);
}

/**
  * @brief  Clear timer upadte flag.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_clear_flag_icr_ui(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP32C4T_ICR_UI_MSK);
}

/**
  * @brief  Timer SGE setup.
  * @param  timx GP32C4T instance
  * @param  value (SGTRG | SGCH4 | SGCH3 | SGCH2 | SGCH1 | SGU)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_sge(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->SGE, value);
}

/**
  * @brief  Timer trigger generation.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_sge_sgtrg(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP32C4T_SGE_SGTRG_MSK);
}

/**
  * @brief  Timer Capture/Compare 4 generation.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_sge_sgch4(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP32C4T_SGE_SGCH4_MSK);
}

/**
  * @brief  Timer Capture/Compare 3 generation.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_sge_sgch3(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP32C4T_SGE_SGCH3_MSK);
}

/**
  * @brief  Timer Capture/Compare 2 generation.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_sge_sgch2(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP32C4T_SGE_SGCH2_MSK);
}

/**
  * @brief  Timer Capture/Compare 1 generation.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_sge_sgch1(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP32C4T_SGE_SGCH1_MSK);
}

/**
  * @brief  Timer update generation.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_sge_sgu(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP32C4T_SGE_SGU_MSK);
}

/**
  * @brief  Timer CHMR1 setup.
  * @param  timx GP32C4T instance
  * @param  value output (CH2OCLREN | CH2MOD | CH2PEN | CH2FEN | CC2SSEL | CH1OCLREN | CH1MOD | CH1PEN | CH1FEN | CC1SSEL)
  *               input  (I2FLT | I2PRES | CC2SSEL | I1FLT | I1PRES | CC1SSEL)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CHMR1, value);
}

/**
  * @brief  GET Timer CHMR1 register value.
  * @param  timx GP32C4T instance
  * @retval Timer CHMR1 register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->CHMR1));
}

/**
  * @brief  Timer output compare 2 clear enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr1_output_ch2oclren(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2OCLREN_MSK);
}

/**
  * @brief  Timer output compare 2 clear disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr1_output_ch2oclren(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2OCLREN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 2 clear is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr1_output_ch2oclren(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2OCLREN_MSK) == (GP32C4T_CHMR1_OUTPUT_CH2OCLREN_MSK));
}

/**
  * @brief  Timer output compare 2 mode setup.
  * @param  timx GP32C4T instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_GP32C4T_OUTPUTMODE_DISABLE
            @arg @ref MD_GP32C4T_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_GP32C4T_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_GP32C4T_OUTPUTMODE_TOGGLE
            @arg @ref MD_GP32C4T_OUTPUTMODE_FORCELOW
            @arg @ref MD_GP32C4T_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_GP32C4T_OUTPUTMODE_PWMMODE1
            @arg @ref MD_GP32C4T_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_output_ch2mod(GP32C4T_TypeDef *timx, uint32_t OutputMode)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2MOD_MSK, (OutputMode << GP32C4T_CHMR1_OUTPUT_CH2MOD_POSS));
}

/**
  * @brief  Get timer output compare 2 mode.
  * @param  timx GP32C4T instance
  * @retval Timer output compare 2 mode.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_output_ch2mod(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2MOD_MSK) >> GP32C4T_CHMR1_OUTPUT_CH2MOD_POSS);
}

/**
  * @brief  Timer output compare 2 preload enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr1_output_ch2pen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2PEN_MSK);
}

/**
  * @brief  Timer output compare 2 preload disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr1_output_ch2pen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 2 preload is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr1_output_ch2pen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2PEN_MSK) == (GP32C4T_CHMR1_OUTPUT_CH2PEN_MSK));
}

/**
  * @brief  Timer output compare 2 fast enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr1_output_ch2fen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2FEN_MSK);
}

/**
  * @brief  Timer output compare 2 fast disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr1_output_ch2fen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 2 fast is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr1_output_ch2fen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH2FEN_MSK) == (GP32C4T_CHMR1_OUTPUT_CH2FEN_MSK));
}

/** 
  * @brief  Timer cpture/compare 2 selection setup.
  * @param  timx GP32C4T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP32C4T_CHMODE_OUTPUT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_output_cc2ssel(GP32C4T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CC2SSEL_MSK, (ChannelMode << GP32C4T_CHMR1_OUTPUT_CC2SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 2 selection.
  * @param  timx GP32C4T instance
  * @retval Timer cpture/compare 2 selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_output_cc2ssel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CC2SSEL_MSK) >> GP32C4T_CHMR1_OUTPUT_CC2SSEL_POSS);
}

/**
  * @brief  Timer output compare 1 clear enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr1_output_ch1oclren(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1OCLREN_MSK);
}

/**
  * @brief  Timer output compare 1 clear disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr1_output_ch1oclren(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1OCLREN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 1 clear is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr1_output_ch1oclren(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1OCLREN_MSK) == (GP32C4T_CHMR1_OUTPUT_CH1OCLREN_MSK));
}

/**
  * @brief  Timer output compare 1 mode setup.
  * @param  timx GP32C4T instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_GP32C4T_OUTPUTMODE_DISABLE
            @arg @ref MD_GP32C4T_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_GP32C4T_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_GP32C4T_OUTPUTMODE_TOGGLE
            @arg @ref MD_GP32C4T_OUTPUTMODE_FORCELOW
            @arg @ref MD_GP32C4T_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_GP32C4T_OUTPUTMODE_PWMMODE1
            @arg @ref MD_GP32C4T_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_output_ch1mod(GP32C4T_TypeDef *timx, uint32_t OutputMode)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1MOD_MSK, (OutputMode << GP32C4T_CHMR1_OUTPUT_CH1MOD_POSS));
}

/**
  * @brief  Get timer output compare 1 mode.
  * @param  timx GP32C4T instance
  * @retval Timer output compare 1 mode.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_output_ch1mod(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1MOD_MSK) >> GP32C4T_CHMR1_OUTPUT_CH1MOD_POSS);
}

/**
  * @brief  Timer output compare 1 preload enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr1_output_ch1pen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1PEN_MSK);
}

/**
  * @brief  Timer output compare 1 preload disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr1_output_ch1pen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 1 preload is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr1_output_ch1pen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1PEN_MSK) == (GP32C4T_CHMR1_OUTPUT_CH1PEN_MSK));
}

/**
  * @brief  Timer output compare 1 fast enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr1_output_ch1fen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1FEN_MSK);
}

/**
  * @brief  Timer output compare 1 fast disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr1_output_ch1fen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 1 fast is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr1_output_ch1fen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CH1FEN_MSK) == (GP32C4T_CHMR1_OUTPUT_CH1FEN_MSK));
}

/** 
  * @brief  Timer cpture/compare 1 selection setup.
  * @param  timx GP32C4T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP32C4T_CHMODE_OUTPUT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_output_cc1ssel(GP32C4T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CC1SSEL_MSK, (ChannelMode << GP32C4T_CHMR1_OUTPUT_CC1SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 1 selection.
  * @param  timx GP32C4T instance
  * @retval Timer cpture/compare 1 selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_output_cc1ssel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_OUTPUT_CC1SSEL_MSK) >> GP32C4T_CHMR1_OUTPUT_CC1SSEL_POSS);
}

/** 
  * @brief  Timer input capture 2 filter setup.
  * @param  timx GP32C4T instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N2
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N4
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV2N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV2N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV4N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV4N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV8N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV8N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N5
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N5
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_input_i2flt(GP32C4T_TypeDef *timx, uint32_t InputFliter)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_INPUT_I2FLT_MSK, (InputFliter << GP32C4T_CHMR1_INPUT_I2FLT_POSS));
}

/**
  * @brief  Get timer input capture 2 filter.
  * @param  timx GP32C4T instance
  * @retval Timer input capture 2 filter.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_input_i2flt(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_INPUT_I2FLT_MSK) >> GP32C4T_CHMR1_INPUT_I2FLT_POSS);
}

/** 
  * @brief  Timer input capture 2 prescaler setup.
  * @param  timx GP32C4T instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV1
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV2
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV4
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_input_i2pres(GP32C4T_TypeDef *timx, uint32_t InputPrescale)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_INPUT_I2PRES_MSK, (InputPrescale << GP32C4T_CHMR1_INPUT_I2PRES_POSS));
}

/**
  * @brief  Get timer input capture 2 prescaler.
  * @param  timx GP32C4T instance
  * @retval Timer input capture 2 prescaler.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_input_i2pres(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_INPUT_I2PRES_MSK) >> GP32C4T_CHMR1_INPUT_I2PRES_POSS);
}

/** 
  * @brief  Timer cpture/compare 2 selection setup.
  * @param  timx GP32C4T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP32C4T_CHMODE_OUTPUT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_input_cc2ssel(GP32C4T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_INPUT_CC2SSEL_MSK, (ChannelMode << GP32C4T_CHMR1_INPUT_CC2SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 2 selection.
  * @param  timx GP32C4T instance
  * @retval Timer cpture/compare 2 selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_input_cc2ssel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_INPUT_CC2SSEL_MSK) >> GP32C4T_CHMR1_INPUT_CC2SSEL_POSS);
}

/** 
  * @brief  Timer input capture 1 filter setup.
  * @param  timx GP32C4T instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N2
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N4
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV2N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV2N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV4N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV4N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV8N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV8N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N5
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N5
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_input_i1flt(GP32C4T_TypeDef *timx, uint32_t InputFliter)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_INPUT_I1FLT_MSK, (InputFliter << GP32C4T_CHMR1_INPUT_I1FLT_POSS));
}

/**
  * @brief  Get timer input capture 1 filter.
  * @param  timx GP32C4T instance
  * @retval Timer input capture 1 filter.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_input_i1flt(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_INPUT_I1FLT_MSK) >> GP32C4T_CHMR1_INPUT_I1FLT_POSS);
}

/** 
  * @brief  Timer input capture 1 prescaler setup.
  * @param  timx GP32C4T instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV1
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV2
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV4
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_input_i1pres(GP32C4T_TypeDef *timx, uint32_t InputPrescale)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_INPUT_I1PRES_MSK, (InputPrescale << GP32C4T_CHMR1_INPUT_I1PRES_POSS));
}

/**
  * @brief  Get timer input capture 1 prescaler.
  * @param  timx GP32C4T instance
  * @retval Timer input capture 1 prescaler.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_input_i1pres(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_INPUT_I1PRES_MSK) >> GP32C4T_CHMR1_INPUT_I1PRES_POSS);
}

/** 
  * @brief  Timer cpture/compare 1 selection setup.
  * @param  timx GP32C4T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP32C4T_CHMODE_OUTPUT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr1_input_cc1ssel(GP32C4T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR1, GP32C4T_CHMR1_INPUT_CC1SSEL_MSK, (ChannelMode << GP32C4T_CHMR1_INPUT_CC1SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 1 selection.
  * @param  timx GP32C4T instance
  * @retval Timer cpture/compare 1 selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr1_input_cc1ssel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP32C4T_CHMR1_INPUT_CC1SSEL_MSK) >> GP32C4T_CHMR1_INPUT_CC1SSEL_POSS);
}

/**
  * @brief  Timer CHMR2 setup.

  * @param  timx GP32C4T instance
  * @param  value output (CH4OCLREN | CH4MOD | CH4PEN | CH4FEN | CC4SSEL | CH3OCLREN | CH3MOD | CH3PEN | CH3FEN | CC3SSEL)
  *               input  (I4FLT | I4PRES | CC4SSEL | I3FLT | I3PRES | CC3SSEL)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CHMR2, value);
}

/**
  * @brief  GET Timer CHMR2 register value.
  * @param  timx GP32C4T instance
  * @retval Timer CHMR2 register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->CHMR2));
}

/**
  * @brief  Timer output compare 4 clear enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr2_output_ch4oclren(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4OCLREN_MSK);
}

/**
  * @brief  Timer output compare 4 clear disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr2_output_ch4oclren(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4OCLREN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 4 clear is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr2_output_ch4oclren(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4OCLREN_MSK) == (GP32C4T_CHMR2_OUTPUT_CH4OCLREN_MSK));
}

/**
  * @brief  Timer output compare 4 mode setup.
  * @param  timx GP32C4T instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_GP32C4T_OUTPUTMODE_DISABLE
            @arg @ref MD_GP32C4T_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_GP32C4T_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_GP32C4T_OUTPUTMODE_TOGGLE
            @arg @ref MD_GP32C4T_OUTPUTMODE_FORCELOW
            @arg @ref MD_GP32C4T_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_GP32C4T_OUTPUTMODE_PWMMODE1
            @arg @ref MD_GP32C4T_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_output_ch4mod(GP32C4T_TypeDef *timx, uint32_t OutputMode)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4MOD_MSK, (OutputMode << GP32C4T_CHMR2_OUTPUT_CH4MOD_POSS));
}

/**
  * @brief  Get timer output compare 4 mode.
  * @param  timx GP32C4T instance
  * @retval Timer output compare 4 mode.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_output_ch4mod(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4MOD_MSK) >> GP32C4T_CHMR2_OUTPUT_CH4MOD_POSS);
}

/**
  * @brief  Timer output compare 4 preload enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr2_output_ch4pen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4PEN_MSK);
}

/**
  * @brief  Timer output compare 4 preload disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr2_output_ch4pen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 4 preload is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr2_output_ch4pen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4PEN_MSK) == (GP32C4T_CHMR2_OUTPUT_CH4PEN_MSK));
}

/**
  * @brief  Timer output compare 4 fast enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr2_output_ch4fen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4FEN_MSK);
}

/**
  * @brief  Timer output compare 4 fast disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr2_output_ch4fen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 4 fast is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr2_output_ch4fen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH4FEN_MSK) == (GP32C4T_CHMR2_OUTPUT_CH4FEN_MSK));
}

/** 
  * @brief  Timer cpture/compare 4 selection setup.
  * @param  timx GP32C4T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP32C4T_CHMODE_OUTPUT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_output_cc4ssel(GP32C4T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CC4SSEL_MSK, (ChannelMode << GP32C4T_CHMR2_OUTPUT_CC4SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 4 selection.
  * @param  timx GP32C4T instance
  * @retval Timer cpture/compare 4 selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_output_cc4ssel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CC4SSEL_MSK) >> GP32C4T_CHMR2_OUTPUT_CC4SSEL_POSS);
}

/**
  * @brief  Timer output compare 3 clear enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr2_output_ch3oclren(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3OCLREN_MSK);
}

/**
  * @brief  Timer output compare 3 clear disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr2_output_ch3oclren(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3OCLREN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 3 clear is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr2_output_ch3oclren(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3OCLREN_MSK) == (GP32C4T_CHMR2_OUTPUT_CH3OCLREN_MSK));
}

/**
  * @brief  Timer output compare 3 mode setup.
  * @param  timx GP32C4T instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_GP32C4T_OUTPUTMODE_DISABLE
            @arg @ref MD_GP32C4T_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_GP32C4T_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_GP32C4T_OUTPUTMODE_TOGGLE
            @arg @ref MD_GP32C4T_OUTPUTMODE_FORCELOW
            @arg @ref MD_GP32C4T_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_GP32C4T_OUTPUTMODE_PWMMODE1
            @arg @ref MD_GP32C4T_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_output_ch3mod(GP32C4T_TypeDef *timx, uint32_t OutputMode)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3MOD_MSK, (OutputMode << GP32C4T_CHMR2_OUTPUT_CH3MOD_POSS));
}

/**
  * @brief  Get timer output compare 3 mode.
  * @param  timx GP32C4T instance
  * @retval Timer output compare 3 mode.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_output_ch3mod(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3MOD_MSK) >> GP32C4T_CHMR2_OUTPUT_CH3MOD_POSS);
}

/**
  * @brief  Timer output compare 3 preload enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr2_output_ch3pen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3PEN_MSK);
}

/**
  * @brief  Timer output compare 3 preload disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr2_output_ch3pen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 3 preload is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr2_output_ch3pen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3PEN_MSK) == (GP32C4T_CHMR2_OUTPUT_CH3PEN_MSK));
}

/**
  * @brief  Timer output compare 3 fast enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_chmr2_output_ch3fen(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3FEN_MSK);
}

/**
  * @brief  Timer output compare 3 fast disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_chmr2_output_ch3fen(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 3 fast is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_chmr2_output_ch3fen(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CH3FEN_MSK) == (GP32C4T_CHMR2_OUTPUT_CH3FEN_MSK));
}

/** 
  * @brief  Timer cpture/compare 3 selection setup.
  * @param  timx GP32C4T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP32C4T_CHMODE_OUTPUT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_output_cc3ssel(GP32C4T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CC3SSEL_MSK, (ChannelMode << GP32C4T_CHMR2_OUTPUT_CC3SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 3 selection.
  * @param  timx GP32C4T instance
  * @retval Timer cpture/compare 3 selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_output_cc3ssel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_OUTPUT_CC3SSEL_MSK) >> GP32C4T_CHMR2_OUTPUT_CC3SSEL_POSS);
}

/** 
  * @brief  Timer input capture 4 filter setup.
  * @param  timx GP32C4T instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N2
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N4
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV2N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV2N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV4N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV4N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV8N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV8N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N5
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N5
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_input_i4flt(GP32C4T_TypeDef *timx, uint32_t InputFliter)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_INPUT_I4FLT_MSK, (InputFliter << GP32C4T_CHMR2_INPUT_I4FLT_POSS));
}

/**
  * @brief  Get timer input capture 4 filter.
  * @param  timx GP32C4T instance
  * @retval Timer input capture 4 filter.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_input_i4flt(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_INPUT_I4FLT_MSK) >> GP32C4T_CHMR2_INPUT_I4FLT_POSS);
}

/** 
  * @brief  Timer input capture 4 prescaler setup.
  * @param  timx GP32C4T instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV1
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV2
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV4
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_input_i4pres(GP32C4T_TypeDef *timx, uint32_t InputPrescale)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_INPUT_I4PRES_MSK, (InputPrescale << GP32C4T_CHMR2_INPUT_I4PRES_POSS));
}

/**
  * @brief  Get timer input capture 4 prescaler.
  * @param  timx GP32C4T instance
  * @retval Timer input capture 4 prescaler.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_input_i4pres(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_INPUT_I4PRES_MSK) >> GP32C4T_CHMR2_INPUT_I4PRES_POSS);
}

/** 
  * @brief  Timer cpture/compare 4 selection setup.
  * @param  timx GP32C4T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP32C4T_CHMODE_OUTPUT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_input_cc4ssel(GP32C4T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_INPUT_CC4SSEL_MSK, (ChannelMode << GP32C4T_CHMR2_INPUT_CC4SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 4 selection.
  * @param  timx GP32C4T instance
  * @retval Timer cpture/compare 4 selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_input_cc4ssel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_INPUT_CC4SSEL_MSK) >> GP32C4T_CHMR2_INPUT_CC4SSEL_POSS);
}

/** 
  * @brief  Timer input capture 3 filter setup.
  * @param  timx GP32C4T instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N2
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N4
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV1N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV2N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV2N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV4N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV4N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV8N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV8N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N5
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV16N8
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N5
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N6
            @arg @ref MD_GP32C4T_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_input_i3flt(GP32C4T_TypeDef *timx, uint32_t InputFliter)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_INPUT_I3FLT_MSK, (InputFliter << GP32C4T_CHMR2_INPUT_I3FLT_POSS));
}

/**
  * @brief  Get timer input capture 3 filter.
  * @param  timx GP32C4T instance
  * @retval Timer input capture 3 filter.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_input_i3flt(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_INPUT_I3FLT_MSK) >> GP32C4T_CHMR2_INPUT_I3FLT_POSS);
}

/** 
  * @brief  Timer input capture 3 prescaler setup.
  * @param  timx GP32C4T instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV1
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV2
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV4
            @arg @ref MD_GP32C4T_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_input_i3pres(GP32C4T_TypeDef *timx, uint32_t InputPrescale)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_INPUT_I3PRES_MSK, (InputPrescale << GP32C4T_CHMR2_INPUT_I3PRES_POSS));
}

/**
  * @brief  Get timer input capture 3 prescaler.
  * @param  timx GP32C4T instance
  * @retval Timer input capture 3 prescaler.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_input_i3pres(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_INPUT_I3PRES_MSK) >> GP32C4T_CHMR2_INPUT_I3PRES_POSS);
}

/** 
  * @brief  Timer cpture/compare 3 selection setup.
  * @param  timx GP32C4T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP32C4T_CHMODE_OUTPUT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP32C4T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_chmr2_input_cc3ssel(GP32C4T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR2, GP32C4T_CHMR2_INPUT_CC3SSEL_MSK, (ChannelMode << GP32C4T_CHMR2_INPUT_CC3SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 3 selection.
  * @param  timx GP32C4T instance
  * @retval Timer cpture/compare 3 selection.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_chmr2_input_cc3ssel(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR2, GP32C4T_CHMR2_INPUT_CC3SSEL_MSK) >> GP32C4T_CHMR2_INPUT_CC3SSEL_POSS);
}

/**
  * @brief  Timer CCEP setup.
  * @param  timx GP32C4T instance
  * @param  value (CC4NPOL | CC4POL | CC4EN | CC3NPOL | CC3POL | CC3EN | CC2NPOL |
  *                CC2POL | CC2EN | CC1NPOL | CC1POL | CC1EN )
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccep(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CCEP, value);
}

/**
  * @brief  GET Timer CCEP register value.
  * @param  timx GP32C4T instance
  * @retval Timer CCEP register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccep(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->CCEP));
}

/** 
  * @brief  Timer capture/compare 4 complementary output polarity setup.
  * @param  timx GP32C4T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccep_cc4npol(GP32C4T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP32C4T_CCEP_CC4NPOL_MSK, (OutputPolarity << GP32C4T_CCEP_CC4NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 4 complementary output polarity.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare 4 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccep_cc4npol(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC4NPOL_MSK) >> GP32C4T_CCEP_CC4NPOL_POS);
}

/** 
  * @brief  Timer capture/compare 4 output polarity setup.
  * @param  timx GP32C4T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccep_cc4pol(GP32C4T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP32C4T_CCEP_CC4POL_MSK, (OutputPolarity << GP32C4T_CCEP_CC4POL_POS));
}

/**
  * @brief  Get timer capture/compare 4 output polarity.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare 4 output polarity.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccep_cc4pol(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC4POL_MSK) >> GP32C4T_CCEP_CC4POL_POS);
}

/**
  * @brief  Timer Capture/Compare 4 output enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ccep_cc4en(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CCEP, GP32C4T_CCEP_CC4EN_MSK);
}

/**
  * @brief  Timer Capture/Compare 4 output disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_ccep_cc4en(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CCEP, GP32C4T_CCEP_CC4EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 4 output is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ccep_cc4en(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC4EN_MSK) == (GP32C4T_CCEP_CC4EN_MSK));
}

/** 
  * @brief  Timer capture/compare 3 complementary output polarity setup.
  * @param  timx GP32C4T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccep_cc3npol(GP32C4T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP32C4T_CCEP_CC3NPOL_MSK, (OutputPolarity << GP32C4T_CCEP_CC3NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 3 complementary output polarity.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare 3 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccep_cc3npol(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC3NPOL_MSK) >> GP32C4T_CCEP_CC3NPOL_POS);
}

/** 
  * @brief  Timer capture/compare 3 output polarity setup.
  * @param  timx GP32C4T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccep_cc3pol(GP32C4T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP32C4T_CCEP_CC3POL_MSK, (OutputPolarity << GP32C4T_CCEP_CC3POL_POS));
}

/**
  * @brief  Get timer capture/compare 3 output polarity.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare 3 output polarity.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccep_cc3pol(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC3POL_MSK) >> GP32C4T_CCEP_CC3POL_POS);
}

/**
  * @brief  Timer Capture/Compare 3 output enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ccep_cc3en(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CCEP, GP32C4T_CCEP_CC3EN_MSK);
}

/**
  * @brief  Timer Capture/Compare 3 output disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_ccep_cc3en(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CCEP, GP32C4T_CCEP_CC3EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 3 output is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ccep_cc3en(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC3EN_MSK) == (GP32C4T_CCEP_CC3EN_MSK));
}

/** 
  * @brief  Timer capture/compare 2 complementary output polarity setup.
  * @param  timx GP32C4T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccep_cc2npol(GP32C4T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP32C4T_CCEP_CC2NPOL_MSK, (OutputPolarity << GP32C4T_CCEP_CC2NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 2 complementary output polarity.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare 2 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccep_cc2npol(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC2NPOL_MSK) >> GP32C4T_CCEP_CC2NPOL_POS);
}

/** 
  * @brief  Timer capture/compare 2 output polarity setup.
  * @param  timx GP32C4T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccep_cc2pol(GP32C4T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP32C4T_CCEP_CC2POL_MSK, (OutputPolarity << GP32C4T_CCEP_CC2POL_POS));
}

/**
  * @brief  Get timer capture/compare 2 output polarity.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare 2 output polarity.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccep_cc2pol(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC2POL_MSK) >> GP32C4T_CCEP_CC2POL_POS);
}

/**
  * @brief  Timer Capture/Compare 2 output enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ccep_cc2en(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CCEP, GP32C4T_CCEP_CC2EN_MSK);
}

/**
  * @brief  Timer Capture/Compare 2 output disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_ccep_cc2en(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CCEP, GP32C4T_CCEP_CC2EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 output is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ccep_cc2en(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC2EN_MSK) == (GP32C4T_CCEP_CC2EN_MSK));
}

/** 
  * @brief  Timer capture/compare 1 complementary output polarity setup.
  * @param  timx GP32C4T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccep_cc1npol(GP32C4T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP32C4T_CCEP_CC1NPOL_MSK, (OutputPolarity << GP32C4T_CCEP_CC1NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 1 complementary output polarity.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare 1 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccep_cc1npol(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC1NPOL_MSK) >> GP32C4T_CCEP_CC1NPOL_POS);
}

/** 
  * @brief  Timer capture/compare 1 output polarity setup.
  * @param  timx GP32C4T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP32C4T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccep_cc1pol(GP32C4T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP32C4T_CCEP_CC1POL_MSK, (OutputPolarity << GP32C4T_CCEP_CC1POL_POS));
}

/**
  * @brief  Get timer capture/compare 1 output polarity.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare 1 output polarity.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccep_cc1pol(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC1POL_MSK) >> GP32C4T_CCEP_CC1POL_POS);
}

/**
  * @brief  Timer Capture/Compare 1 output enable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_enable_ccep_cc1en(GP32C4T_TypeDef *timx)
{
  SET_BIT(timx->CCEP, GP32C4T_CCEP_CC1EN_MSK);
}

/**
  * @brief  Timer Capture/Compare 1 output disable.
  * @param  timx GP32C4T instance
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_disable_ccep_cc1en(GP32C4T_TypeDef *timx)
{
  CLEAR_BIT(timx->CCEP, GP32C4T_CCEP_CC1EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 output is enabled.
  * @param  timx GP32C4T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp32c4t_is_enabled_ccep_cc1en(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP32C4T_CCEP_CC1EN_MSK) == (GP32C4T_CCEP_CC1EN_MSK));
}

/**
  * @brief  Timer COUNT setup.
  * @param  timx GP32C4T instance
  * @param  value COUNT
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_count(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->COUNT, value);
}

/**
  * @brief  GET Timer COUNT register value.
  * @param  timx GP32C4T instance
  * @retval Timer COUNT register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_count(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->COUNT));
}

/** 
  * @brief  Timer counter value setup.
  * @param  timx GP32C4T instance
  * @param  counter Counter value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_count_cntv(GP32C4T_TypeDef *timx, uint32_t counter)
{
  MODIFY_REG(timx->COUNT, GP32C4T_COUNT_CNTV_MSK, counter);
}

/**
  * @brief  Get timer counter value.
  * @param  timx GP32C4T instance
  * @retval Timer counter value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_cnt_cntv(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->COUNT, GP32C4T_COUNT_CNTV_MSK) >> GP32C4T_COUNT_CNTV_POSS);
}

/**
  * @brief  Timer PRES setup.
  * @param  timx GP32C4T instance
  * @param  value PRES
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_pres(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->PRES, value);
}

/**
  * @brief  GET Timer PRES register value.
  * @param  timx GP32C4T instance
  * @retval Timer PRES register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_pres(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->PRES));
}

/** 
  * @brief  Timer prescaler value setup.
  * @param  timx GP32C4T instance
  * @param  prescaler Prescaler value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_pres_pscv(GP32C4T_TypeDef *timx, uint32_t prescaler)
{
  MODIFY_REG(timx->PRES, GP32C4T_PRES_PSCV_MSK, prescaler);
}

/**
  * @brief  Get timer prescaler value.
  * @param  timx GP32C4T instance
  * @retval Timer prescaler value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_pres_pscv(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->PRES, GP32C4T_PRES_PSCV_MSK) >> GP32C4T_PRES_PSCV_POSS);
}

/**
  * @brief  Timer AR setup.
  * @param  timx GP32C4T instance
  * @param  value AR
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ar(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->AR, value);
}

/**
  * @brief  GET Timer AR register value.
  * @param  timx GP32C4T instance
  * @retval Timer AR register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ar(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->AR));
}

/** 
  * @brief  Timer auto-reload value setup.
  * @param  timx GP32C4T instance
  * @param  reload Auto-reload value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ar_arv(GP32C4T_TypeDef *timx, uint32_t reload)
{
  MODIFY_REG(timx->AR, GP32C4T_AR_ARV_MSK, reload);
}

/**
  * @brief  Get timer auto-reload value.
  * @param  timx GP32C4T instance
  * @retval Timer auto-reload value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ar_arv(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->AR, GP32C4T_AR_ARV_MSK) >> GP32C4T_AR_ARV_POSS);
}

/**
  * @brief  Timer CCVAL1 setup.
  * @param  timx GP32C4T instance
  * @param  value CCVAL1
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccval1(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CCVAL1, value);
}

/**
  * @brief  GET Timer CCVAL1 register value.
  * @param  timx GP32C4T instance
  * @retval Timer CCVAL1 register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccval1(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->CCVAL1));
}

/** 
  * @brief  Timer capture/compare value 1 setup.
  * @param  timx GP32C4T instance
  * @param  CapCompValue Capture/Compare value 1 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccval1_ccrv1(GP32C4T_TypeDef *timx, uint32_t CapCompValue)
{
  MODIFY_REG(timx->CCVAL1, GP32C4T_CCVAL1_CCRV1_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 1.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare value 1.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccval1_ccrv1(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCVAL1, GP32C4T_CCVAL1_CCRV1_MSK) >> GP32C4T_CCVAL1_CCRV1_POSS);
}

/**
  * @brief  Timer CCVAL2 setup.
  * @param  timx GP32C4T instance
  * @param  value CCVAL2
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccval2(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CCVAL2, value);
}

/**
  * @brief  GET Timer CCVAL2 register value.
  * @param  timx GP32C4T instance
  * @retval Timer CCVAL2 register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccval2(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->CCVAL2));
}

/** 
  * @brief  Timer capture/compare value 2 setup.
  * @param  timx GP32C4T instance
  * @param  CapCompValue Capture/Compare value 2 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccval2_ccrv2(GP32C4T_TypeDef *timx, uint32_t CapCompValue)
{
  MODIFY_REG(timx->CCVAL2, GP32C4T_CCVAL2_CCRV2_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 2.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare value 2.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccval2_ccrv2(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCVAL2, GP32C4T_CCVAL2_CCRV2_MSK) >> GP32C4T_CCVAL2_CCRV2_POSS);
}

/**
  * @brief  Timer CCVAL3 setup.
  * @param  timx GP32C4T instance
  * @param  value CCVAL3
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccval3(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CCVAL3, value);
}

/**
  * @brief  GET Timer CCVAL3 register value.
  * @param  timx GP32C4T instance
  * @retval Timer CCVAL3 register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccval3(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->CCVAL3));
}

/** 
  * @brief  Timer capture/compare value 3 setup.
  * @param  timx GP32C4T instance
  * @param  CapCompValue Capture/Compare value 3 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccval3_ccrv3(GP32C4T_TypeDef *timx, uint32_t CapCompValue)
{
  MODIFY_REG(timx->CCVAL3, GP32C4T_CCVAL3_CCRV3_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 3.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare value 3.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccval3_ccrv3(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCVAL3, GP32C4T_CCVAL3_CCRV3_MSK) >> GP32C4T_CCVAL3_CCRV3_POSS);
}

/**
  * @brief  Timer CCVAL4 setup.
  * @param  timx GP32C4T instance
  * @param  value CCVAL4
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccval4(GP32C4T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CCVAL4, value);
}

/**
  * @brief  GET Timer CCVAL4 register value.
  * @param  timx GP32C4T instance
  * @retval Timer CCVAL4 register value.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccval4(GP32C4T_TypeDef *timx)
{
  return (READ_REG(timx->CCVAL4));
}

/** 
  * @brief  Timer capture/compare value 4 setup.
  * @param  timx GP32C4T instance
  * @param  CapCompValue Capture/Compare value 4 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp32c4t_set_ccval4_ccrv4(GP32C4T_TypeDef *timx, uint32_t CapCompValue)
{
  MODIFY_REG(timx->CCVAL4, GP32C4T_CCVAL4_CCRV4_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 4.
  * @param  timx GP32C4T instance
  * @retval Timer capture/compare value 4.
  */

__STATIC_INLINE uint32_t md_gp32c4t_get_ccval4_ccrv4(GP32C4T_TypeDef *timx)
{
  return (READ_BIT(timx->CCVAL4, GP32C4T_CCVAL4_CCRV4_MSK) >> GP32C4T_CCVAL4_CCRV4_POSS);
}

/**
  * @} MD_GP32C4T_Public_Macro
  */

/* Public functions -----------------------------------------------------------*/



/**
  * @} GP32C4T
  */


#endif

/**
  * @} Micro_Driver
  */


#ifdef __cplusplus
}
#endif

#endif
