/**
  **************************************************************************************
  * @file    md_gp16c2t.h
  * @brief   GP16C2T Head File
  *
  * @version V1.00
  * @date    31/3/2020
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. AMD rights reserved.
  **************************************************************************************
  */
  
 /* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_GP16C2T_H__
#define __MD_GP16C2T_H__

#ifdef __cplusplus
extern "C"{
#endif

/* Includes -------------------------------------------------------*/
#include "es32f0930.h"

/** @addtogroup Micro_Driver
  * @{
  */
#if defined (GP16C2T1) | defined (GP16C2T2) 

/** @defgroup GP16C2T GP16C2T
  * @brief GP16C2T module driver
  * @{
  */


/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private Macros ----------------------------------------------------------*/



/* Public Constants -----------------------------------------------------------*/

/** @defgroup MD_GP16C2T_Public_Constants Public Constants
  * @brief GP16C2T module driver
  * @{
  */
/* GP16C2T_CON1 macros define*/
#define MD_GP16C2T_DEBUGSELECT_INPUT        (0 << GP16C2T_CON1_DBGSEL_POS)      /*!<  Channel input*/
#define MD_GP16C2T_DEBUGSELECT_OUTPUT       (1 << GP16C2T_CON1_DBGSEL_POS)      /*!<  Channel output*/

#define MD_GP16C2T_CLOCKDIVSION_DIV1        (0 << GP16C2T_CON1_DFCKSEL_POSS)    /*!<  tDTS=tCK_INT*/
#define MD_GP16C2T_CLOCKDIVSION_DIV2        (1 << GP16C2T_CON1_DFCKSEL_POSS)    /*!<  tDTS=2*tCK_INT*/
#define MD_GP16C2T_CLOCKDIVSION_DIV4        (2 << GP16C2T_CON1_DFCKSEL_POSS)    /*!<  tDTS=4*tCK_INT*/

#define MD_GP16C2T_ONEPULSEMODE_DISABLE     (0 << GP16C2T_CON1_SPMEN_POS)       /*!<  Counter is not stopped at update event*/
#define MD_GP16C2T_ONEPULSEMODE_ENABLE      (1 << GP16C2T_CON1_SPMEN_POS)       /*!<  Counter stops counting at the next update event*/

#define MD_GP16C2T_UPDATESOURCE_NORMAL      (0 << GP16C2T_CON1_UERSEL_POS)      /*!<  Counter overflow/underflow, Setting the UG bit, Update generation through the slave mode controller generate an update interrupt*/
#define MD_GP16C2T_UPDATESOURCE_COUNTER     (1 << GP16C2T_CON1_UERSEL_POS)      /*!<  Only counter overflow/underflow generates an update interrupt or DMA request if Enabled*/

/* GP16C2T_CON2 macros define*/
#define MD_GP16C2T_IDLEOUTPUT_LOW           0                                   /*!<  OC1=0 (after a dead-time if OC1N is implemented) when MOE=0*/
#define MD_GP16C2T_IDLEOUTPUT_HIGH          1                                   /*!<  OC1=1 (after a dead-time if OC1N is implemented) when MOE=0*/

#define MD_GP16C2T_MASTERMODE_RESET         (0 << GP16C2T_CON2_MMSEL_POSS)      /*!<  Reset mode*/
#define MD_GP16C2T_MASTERMODE_ENABLE        (1 << GP16C2T_CON2_MMSEL_POSS)      /*!<  Enable mode*/
#define MD_GP16C2T_MASTERMODE_UPDATE        (2 << GP16C2T_CON2_MMSEL_POSS)      /*!<  Update mode*/
#define MD_GP16C2T_MASTERMODE_COMPAREPULSE  (3 << GP16C2T_CON2_MMSEL_POSS)      /*!<  Compare Pulse mode*/
#define MD_GP16C2T_MASTERMODE_COMPARE1      (4 << GP16C2T_CON2_MMSEL_POSS)      /*!<  Compare 1 mode*/
#define MD_GP16C2T_MASTERMODE_COMPARE2      (5 << GP16C2T_CON2_MMSEL_POSS)      /*!<  Compare 2 mode*/

#define MD_GP16C2T_DMASELECTION_COMPARE     (0 << GP16C2T_CON2_CCDMASEL_POS)    /*!<  CCx DMA request sent when CCx event occurs*/
#define MD_GP16C2T_DMASELECTION_UPDATE      (1 << GP16C2T_CON2_CCDMASEL_POS)    /*!<  CCx DMA requests sent when update event occurs*/

#define MD_GP16C2T_UPDATESELECTION_COMG     (0 << GP16C2T_CON2_CCUSEL_POS)      /*!<  When capture/compare control bits are preloaded (CCPC=1), they are updated by setting the COMG bit only*/
#define MD_GP16C2T_UPDATESELECTION_BOTH     (1 << GP16C2T_CON2_CCUSEL_POS)      /*!<  When capture/compare control bits are preloaded (CCPC=1), they are updated by setting the COMG bit or when an rising edge occurs on TRGI*/

#define MD_GP16C2T_CCPRELOAD_DISABLE        (0 << GP16C2T_CON2_CCPCEN_POS)      /*!<  CCxE, CCxNE and OCxM bits are not preloaded*/
#define MD_GP16C2T_CCPRELOAD_ENABLE         (1 << GP16C2T_CON2_CCPCEN_POS)      /*!<  CCxE, CCxNE and OCxM bits are preloaded*/

/* GP16C2T_SMCON macros define*/
#define MD_GP16C2T_MASTERSLAVE_NODELAY      (0 << GP16C2T_SMCON_MSCFG_POS)      /*!<  No action*/
#define MD_GP16C2T_MASTERSLAVE_DELAY        (1 << GP16C2T_SMCON_MSCFG_POS)      /*!<  The effect of an event on the trigger input (TRGI) is delayed to allow a perfect*/

#define MD_GP16C2T_TRIGGERSELECT_ITR0       (0 << GP16C2T_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 0*/
#define MD_GP16C2T_TRIGGERSELECT_ITR1       (1 << GP16C2T_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 1*/
#define MD_GP16C2T_TRIGGERSELECT_ITR2       (2 << GP16C2T_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 2*/
#define MD_GP16C2T_TRIGGERSELECT_ITR3       (3 << GP16C2T_SMCON_TSSEL_POSS)     /*!<  Internal Trigger 3*/
#define MD_GP16C2T_TRIGGERSELECT_TI1FED     (4 << GP16C2T_SMCON_TSSEL_POSS)     /*!<  TI1 Edge Detector*/
#define MD_GP16C2T_TRIGGERSELECT_TI1FP1     (5 << GP16C2T_SMCON_TSSEL_POSS)     /*!<  Filtered Timer Input 1*/
#define MD_GP16C2T_TRIGGERSELECT_TI2FP2     (6 << GP16C2T_SMCON_TSSEL_POSS)     /*!<  Filtered Timer Input 2*/
#define MD_GP16C2T_TRIGGERSELECT_ETRF       (7 << GP16C2T_SMCON_TSSEL_POSS)     /*!<  External Trigger input*/

#define MD_GP16C2T_SLAVEMODE_DISABLE        (0 << GP16C2T_SMCON_SMODS_POSS)     /*!<  Slave mode disabled*/
#define MD_GP16C2T_SLAVEMODE_ENCODER1       (1 << GP16C2T_SMCON_SMODS_POSS)     /*!<  Encoder mode 1*/
#define MD_GP16C2T_SLAVEMODE_ENCODER2       (2 << GP16C2T_SMCON_SMODS_POSS)     /*!<  Encoder mode 2*/
#define MD_GP16C2T_SLAVEMODE_ENCODER3       (3 << GP16C2T_SMCON_SMODS_POSS)     /*!<  Encoder mode 3*/
#define MD_GP16C2T_SLAVEMODE_RESET          (4 << GP16C2T_SMCON_SMODS_POSS)     /*!<  Reset Mode*/
#define MD_GP16C2T_SLAVEMODE_GATED          (5 << GP16C2T_SMCON_SMODS_POSS)     /*!<  Gated Mode*/
#define MD_GP16C2T_SLAVEMODE_TRIGGER        (6 << GP16C2T_SMCON_SMODS_POSS)     /*!<  Trigger Mode*/
#define MD_GP16C2T_SLAVEMODE_EXTERNALCLOCK  (7 << GP16C2T_SMCON_SMODS_POSS)     /*!<  External Clock Mode 1*/

/* GP16C2T_IER IDR IVS RIF IFM ICR macros define*/
#define MD_GP16C2T_INTERRUPTFALG_CC2OI      (1 << GP16C2T_IER_CH2OVI_POS)       /*!<  Capture/Compare 2 overcapture interrupt*/
#define MD_GP16C2T_INTERRUPTFALG_CC1OI      (1 << GP16C2T_IER_CH1OVI_POS)       /*!<  Capture/Compare 1 overcapture interrupt*/
#define MD_GP16C2T_INTERRUPTFALG_BI         (1 << GP16C2T_IER_BRKI_POS)         /*!<  Break interrupt*/
#define MD_GP16C2T_INTERRUPTFALG_TI         (1 << GP16C2T_IER_TRGI_POS)         /*!<  Trigger interrupt*/
#define MD_GP16C2T_INTERRUPTFALG_COMI       (1 << GP16C2T_IER_COMI_POS)         /*!<  COM interrupt*/
#define MD_GP16C2T_INTERRUPTFALG_CC2I       (1 << GP16C2T_IER_CH2I_POS)         /*!<  Capture/Compare 2 interrupt*/
#define MD_GP16C2T_INTERRUPTFALG_CC1I       (1 << GP16C2T_IER_CH1I_POS)         /*!<  Capture/Compare 1 interrupt*/
#define MD_GP16C2T_INTERRUPTFALG_UI         (1 << GP16C2T_IER_UI_POS)           /*!<  Update interrupt*/

/* GP16C2T_SGE macros define*/
#define MD_GP16C2T_EVENTGENERATION_BG       (1 << GP16C2T_SGE_SGBRK_POS)        /*!<  Break generation*/  
#define MD_GP16C2T_EVENTGENERATION_TG       (1 << GP16C2T_SGE_SGTRG_POS)        /*!<  Trigger generation*/  
#define MD_GP16C2T_EVENTGENERATION_COMG     (1 << GP16C2T_SGE_SGCOM_POS)        /*!<  Capture/Compare control update generation*/
#define MD_GP16C2T_EVENTGENERATION_CC2G     (1 << GP16C2T_SGE_SGCH2_POS)        /*!<  Capture/Compare 2 generation*/
#define MD_GP16C2T_EVENTGENERATION_CC1G     (1 << GP16C2T_SGE_SGCH1_POS)        /*!<  Capture/Compare 1 generation*/
#define MD_GP16C2T_EVENTGENERATION_UG       (1 << GP16C2T_SGE_SGU_POS)          /*!<  Update generation*/

/* GP16C2T_CHMR1 output macros define*/
#define MD_GP16C2T_OUTPUTMODE_DISABLE       0                                   /*!<  Frozen*/
#define MD_GP16C2T_OUTPUTMODE_HIGHONMSTCH   1                                   /*!<  Set channel 1 to active level on match*/
#define MD_GP16C2T_OUTPUTMODE_LOWONMSTCH    2                                   /*!<  Set channel 1 to inactive level on match*/
#define MD_GP16C2T_OUTPUTMODE_TOGGLE        3                                   /*!<  Toggle*/
#define MD_GP16C2T_OUTPUTMODE_FORCELOW      4                                   /*!<  Force inactive level*/
#define MD_GP16C2T_OUTPUTMODE_FORCEHIGH     5                                   /*!<  Force active level*/
#define MD_GP16C2T_OUTPUTMODE_PWMMODE1      6                                   /*!<  PWM mode 1*/
#define MD_GP16C2T_OUTPUTMODE_PWMMODE2      7                                   /*!<  PWM mode 2*/

#define MD_GP16C2T_CHMODE_OUTPUT            0                                   /*!<  CCx channel is configured as output*/
#define MD_GP16C2T_CHMODE_INPUT_DIRECT      1                                   /*!<  CCx channel is configured as input, ICx is mapped direct*/
#define MD_GP16C2T_CHMODE_INPUT_INDIRECT    2                                   /*!<  CCx channel is configured as input, ICx is mapped indirect*/
#define MD_GP16C2T_CHMODE_INPUT_TRC         3                                   /*!<  CCx channel is configured as input, ICx is mapped TRC*/

/* GP16C2T_CHMR1 input macros define*/
#define MD_GP16C2T_INPUTFILTER_FDIV1        0                                   /*!<  No filter*/
#define MD_GP16C2T_INPUTFILTER_FDIV1N2      1                                   /*!<  fSAMPLING = fCK_INT, N = 2*/
#define MD_GP16C2T_INPUTFILTER_FDIV1N4      2                                   /*!<  fSAMPLING = fCK_INT, N = 4*/
#define MD_GP16C2T_INPUTFILTER_FDIV1N8      3                                   /*!<  fSAMPLING = fCK_INT, N = 8*/
#define MD_GP16C2T_INPUTFILTER_FDIV2N6      4                                   /*!<  fSAMPLING = fDTS / 2, N = 6*/
#define MD_GP16C2T_INPUTFILTER_FDIV2N8      5                                   /*!<  fSAMPLING = fDTS / 2, N = 8*/
#define MD_GP16C2T_INPUTFILTER_FDIV4N6      6                                   /*!<  fSAMPLING = fDTS / 4, N = 6*/
#define MD_GP16C2T_INPUTFILTER_FDIV4N8      7                                   /*!<  fSAMPLING = fDTS / 4, N = 8*/
#define MD_GP16C2T_INPUTFILTER_FDIV8N6      8                                   /*!<  fSAMPLING = fDTS / 8, N = 6*/
#define MD_GP16C2T_INPUTFILTER_FDIV8N8      9                                   /*!<  fSAMPLING = fDTS / 8, N = 8*/
#define MD_GP16C2T_INPUTFILTER_FDIV16N5     10                                  /*!<  fSAMPLING = fDTS / 16, N = 5*/
#define MD_GP16C2T_INPUTFILTER_FDIV16N6     11                                  /*!<  fSAMPLING = fDTS / 16, N = 6*/
#define MD_GP16C2T_INPUTFILTER_FDIV16N8     12                                  /*!<  fSAMPLING = fDTS / 16, N = 8*/
#define MD_GP16C2T_INPUTFILTER_FDIV32N5     13                                  /*!<  fSAMPLING = fDTS / 32, N = 5*/
#define MD_GP16C2T_INPUTFILTER_FDIV32N6     14                                  /*!<  fSAMPLING = fDTS / 32, N = 6*/
#define MD_GP16C2T_INPUTFILTER_FDIV32N8     15                                  /*!<  fSAMPLING = fDTS / 32, N = 8*/

#define MD_GP16C2T_INPUTPRESCALE_DIV1       0                                   /*!<  no prescaler*/
#define MD_GP16C2T_INPUTPRESCALE_DIV2       1                                   /*!<  capture is done once every 2 events*/
#define MD_GP16C2T_INPUTPRESCALE_DIV4       2                                   /*!<  capture is done once every 4 events*/
#define MD_GP16C2T_INPUTPRESCALE_DIV8       3                                   /*!<  capture is done once every 8 events*/

/* GP16C2T_CCEP input macros define*/
#define MD_GP16C2T_OUTPUTPOLARITY_HIGH      0                                   /*!<  active high*/
#define MD_GP16C2T_OUTPUTPOLARITY_LOW       1                                   /*!<  active low*/

/* GP16C2T_BDCFG input macros define*/
#define MD_GP16C2T_BREAKPOLARITY_LOW        (0 << GP16C2T_BDCFG_BRKP_POS)       /*!<  Break input BRK is active low*/
#define MD_GP16C2T_BREAKPOLARITY_HIGH       (1 << GP16C2T_BDCFG_BRKP_POS)       /*!<  Break input BRK is active high */

#define MD_GP16C2T_OFFSTATERUN_DISABLE      (0 << GP16C2T_BDCFG_OFFSSR_POS)     /*!<  This bit is used when MOE=1, when inactive, OC/OCN outputs are disabled*/
#define MD_GP16C2T_OFFSTATERUN_ENABLE       (1 << GP16C2T_BDCFG_OFFSSR_POS)     /*!<  This bit is used when MOE=1, when inactive, OC/OCN outputs are enabled*/

#define MD_GP16C2T_OFFSTATEIDLE_DISABLE     (0 << GP16C2T_BDCFG_OFFSSI_POS)     /*!<  This bit is used when MOE=0, when inactive, OC/OCN outputs are disabled*/
#define MD_GP16C2T_OFFSTATEIDLE_ENABLE      (1 << GP16C2T_BDCFG_OFFSSI_POS)     /*!<  This bit is used when MOE=0, when inactive, OC/OCN outputs are forced*/

#define MD_GP16C2T_LOCKLEVEL_0              (0 << GP16C2T_BDCFG_LOCKLVL_POSS)   /*!<  LOCK OFF*/
#define MD_GP16C2T_LOCKLEVEL_1              (1 << GP16C2T_BDCFG_LOCKLVL_POSS)   /*!<  LOCK Level 1*/
#define MD_GP16C2T_LOCKLEVEL_2              (2 << GP16C2T_BDCFG_LOCKLVL_POSS)   /*!<  LOCK Level 2*/
#define MD_GP16C2T_LOCKLEVEL_3              (3 << GP16C2T_BDCFG_LOCKLVL_POSS)   /*!<  LOCK Level 3*/

/**
  * @} MD_GP16C2T_Public_Constants
  */

/* Public Macro ------------------------------------------------------------*/
/** @defgroup MD_GP16C2T_Public_Macro Public Macro
  * @brief GP16C2T module driver
  * @{
  */
  
/**
  * @brief  Timer CON1 setup.
  * @param  timx GP16C2T instance
  * @param  value (DBGSEL | CMPSEL | CMPSELP | DFCKSEL | ARPEN | CMSEL | DIRSEL | SPMEN | UERSEL | DISUE | CNTEN)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con1(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CON1, value);
}

/**
  * @brief  GET Timer CON1 register value.
  * @param  timx GP16C2T instance
  * @retval Timer CON1 register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con1(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->CON1));
}

/**
  * @brief  Timer debug mode select.
  * @param  timx GP16C2T instance
  * @param  DebugSelect Debug mode output select
            @arg @ref MD_GP16C2T_DEBUGSELECT_INPUT
            @arg @ref MD_GP16C2T_DEBUGSELECT_OUTPUT
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con1_dbgsel(GP16C2T_TypeDef *timx, uint32_t DebugSelect)
{
  MODIFY_REG(timx->CON1, GP16C2T_CON1_DBGSEL_MSK, DebugSelect);
}

/**
  * @brief  Get timer debug mode select.
  * @param  timx GP16C2T instance
  * @retval Timer debug mode select.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con1_dbgsel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP16C2T_CON1_DBGSEL_MSK) >> GP16C2T_CON1_DBGSEL_POS);
}

/**
  * @brief  Timer clock division setup.
  * @param  timx GP16C2T instance
  * @param  ClockDivision Clock division
            @arg @ref MD_GP16C2T_CLOCKDIVSION_DIV1
            @arg @ref MD_GP16C2T_CLOCKDIVSION_DIV2
            @arg @ref MD_GP16C2T_CLOCKDIVSION_DIV4
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con1_dfcksel(GP16C2T_TypeDef *timx, uint32_t ClockDivision)
{
  MODIFY_REG(timx->CON1, GP16C2T_CON1_DFCKSEL_MSK, ClockDivision);
}

/**
  * @brief  Get timer Clock division.
  * @param  timx GP16C2T instance
  * @retval Timer Clock division.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con1_dfcksel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP16C2T_CON1_DFCKSEL_MSK) >> GP16C2T_CON1_DFCKSEL_POSS);
}

/**
  * @brief  Timer auto-reload preload enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_con1_arpen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CON1, GP16C2T_CON1_ARPEN_MSK);
}

/**
  * @brief  Timer auto-reload preload disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_con1_arpen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, GP16C2T_CON1_ARPEN_MSK);
}

/**
  * @brief  Indicates whether the timer auto-reload preload is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_con1_arpen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP16C2T_CON1_ARPEN_MSK) == (GP16C2T_CON1_ARPEN_MSK));
}

/**
  * @brief  Timer one pulse mode enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_con1_spmen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CON1, GP16C2T_CON1_SPMEN_MSK);
}

/**
  * @brief  Timer one pulse mode disable.
  * @param  timx GP16C2T instance
  * @retval None.
  */

__STATIC_INLINE void md_gp16c2t_disable_con1_spmen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, GP16C2T_CON1_SPMEN_MSK);
}

/**
  * @brief  Indicates whether the timer one pulse mode is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_con1_spmen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP16C2T_CON1_SPMEN_MSK) == (GP16C2T_CON1_SPMEN_MSK));
}

/**
  * @brief  Timer update request source setup.
  * @param  timx GP16C2T instance
  * @param  UpdateSource Update request source select
            @arg @ref MD_GP16C2T_UPDATESOURCE_NORMAL
            @arg @ref MD_GP16C2T_UPDATESOURCE_COUNTER
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con1_uersel(GP16C2T_TypeDef *timx, uint32_t UpdateSource)
{
  MODIFY_REG(timx->CON1, GP16C2T_CON1_UERSEL_MSK, UpdateSource);
}

/**
  * @brief  Get timer update request source.
  * @param  timx GP16C2T instance
  * @retval Timer update request source.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con1_uersel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP16C2T_CON1_UERSEL_MSK) >> GP16C2T_CON1_UERSEL_POS);
}

/**
  * @brief  Timer update event enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_con1_disue(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, GP16C2T_CON1_DISUE_MSK);
}

/**
  * @brief  Timer update event disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disble_con1_disue(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CON1, GP16C2T_CON1_DISUE_MSK);
}

/**
  * @brief  Indicates whether the timer update event is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_con1_disue(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP16C2T_CON1_DISUE_MSK) == (GP16C2T_CON1_DISUE_MSK));
}

/**
  * @brief  Timer counter enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_con1_cnten(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CON1, GP16C2T_CON1_CNTEN_MSK);
}
  

/**
  * @brief  Timer counter disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_con1_cnten(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, GP16C2T_CON1_CNTEN_MSK);
}

/**
  * @brief  Indicates whether the timer counter is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_con1_cnten(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, GP16C2T_CON1_CNTEN_MSK) == (GP16C2T_CON1_CNTEN_MSK));
}

/**
  * @brief  Timer CON2 setup.
  * @param  timx GP16C2T instance
  * @param  value (OISS2 | OISS1N | OISS1 | I1SEL | MMSEL | CCDMASEL | CCUSEL | CCPCEN)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con2(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CON2, value);
}

/**
  * @brief  GET Timer CON2 register value.
  * @param  timx GP16C2T instance
  * @retval Timer CON2 register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con2(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->CON2));
}

/**
  * @brief  Timer output idle state 2 setup.
  * @param  timx GP16C2T instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_GP16C2T_IDLEOUTPUT_LOW
            @arg @ref MD_GP16C2T_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con2_oiss2(GP16C2T_TypeDef *timx, uint32_t IdleOutput)
{
  MODIFY_REG(timx->CON2, GP16C2T_CON2_OISS2_MSK, (IdleOutput << GP16C2T_CON2_OISS2_POS));
}

/**
  * @brief  Get timer output idle state 2.
  * @param  timx GP16C2T instance
  * @retval Timer output idle state 2.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con2_oiss2(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON2, GP16C2T_CON2_OISS2_MSK) >> GP16C2T_CON2_OISS2_POS);
}

/**
  * @brief  Timer output idle state 1N setup.
  * @param  timx GP16C2T instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_GP16C2T_IDLEOUTPUT_LOW
            @arg @ref MD_GP16C2T_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con2_oiss1n(GP16C2T_TypeDef *timx, uint32_t IdleOutput)
{
  MODIFY_REG(timx->CON2, GP16C2T_CON2_OISS1N_MSK, (IdleOutput << GP16C2T_CON2_OISS1N_POS));
}

/**
  * @brief  Get timer output idle state 1N.
  * @param  timx GP16C2T instance
  * @retval Timer output idle state 1N.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con2_oiss1n(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON2, GP16C2T_CON2_OISS1N_MSK) >> GP16C2T_CON2_OISS1N_POS);
}

/**
  * @brief  Timer output idle state 1 setup.
  * @param  timx GP16C2T instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_GP16C2T_IDLEOUTPUT_LOW
            @arg @ref MD_GP16C2T_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con2_oiss1(GP16C2T_TypeDef *timx, uint32_t IdleOutput)
{
  MODIFY_REG(timx->CON2, GP16C2T_CON2_OISS1_MSK, (IdleOutput << GP16C2T_CON2_OISS1_POS));
}

/**
  * @brief  Get timer output idle state 1.
  * @param  timx GP16C2T instance
  * @retval Timer output idle state 1.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con2_oiss1(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON2, GP16C2T_CON2_OISS1_MSK) >> GP16C2T_CON2_OISS1_POS);
}

/**
  * @brief  Timer master mode selection setup.
  * @param  timx GP16C2T instance
  * @param  MasterMode Master mode selection
            @arg @ref MD_GP16C2T_MASTERMODE_RESET
            @arg @ref MD_GP16C2T_MASTERMODE_ENABLE
            @arg @ref MD_GP16C2T_MASTERMODE_UPDATE
            @arg @ref MD_GP16C2T_MASTERMODE_COMPAREPULSE
            @arg @ref MD_GP16C2T_MASTERMODE_COMPARE1
            @arg @ref MD_GP16C2T_MASTERMODE_COMPARE2
            @arg @ref MD_GP16C2T_MASTERMODE_COMPARE3
            @arg @ref MD_GP16C2T_MASTERMODE_COMPARE4
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con2_mmsel(GP16C2T_TypeDef *timx, uint32_t MasterMode)
{
  MODIFY_REG(timx->CON2, GP16C2T_CON2_MMSEL_MSK, MasterMode);
}

/**
  * @brief  Get timer master mode selection.
  * @param  timx GP16C2T instance
  * @retval Timer master mode selection.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con2_mmsel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON2, GP16C2T_CON2_MMSEL_MSK) >> GP16C2T_CON2_MMSEL_POSS);
}

/**
  * @brief  Timer capture/compare control update selection setup.
  * @param  timx GP16C2T instance
  * @param  UpdateSelection Capture/compare control update selection
            @arg @ref MD_GP16C2T_UPDATESELECTION_COMG
            @arg @ref MD_GP16C2T_UPDATESELECTION_BOTH
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_con2_ccusel(GP16C2T_TypeDef *timx, uint32_t UpdateSelection)
{
  MODIFY_REG(timx->CON2, GP16C2T_CON2_CCUSEL_MSK, UpdateSelection);
}

/**
  * @brief  Get timer capture/compare control update selection.
  * @param  timx GP16C2T instance
  * @retval Timer capture/compare control update selection.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_con2_ccusel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON2, GP16C2T_CON2_CCUSEL_MSK) >> GP16C2T_CON2_CCUSEL_POS);
}

/**
  * @brief  Timer capture/compare preloaded control enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_con2_ccpcen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CON2, GP16C2T_CON2_CCPCEN_MSK);
}

/**
  * @brief  Timer capture/compare preloaded control disable.
  * @param  timx GP16C2T instance
  * @retval None.
  */

__STATIC_INLINE void md_gp16c2t_disable_con2_ccpcen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON2, GP16C2T_CON2_CCPCEN_MSK);
}

/**
  * @brief  Indicates whether the timer capture/compare preloaded control is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_con2_ccpcen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CON2, GP16C2T_CON2_CCPCEN_MSK) == (GP16C2T_CON2_CCPCEN_MSK));
}

/**
  * @brief  Timer SMCON setup.
  * @param  timx GP16C2T instance
  * @param  value (MSCFG | TSSEL | SMODS)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_smcon(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->SMCON, value);
}

/**
  * @brief  GET Timer SMCON register value.
  * @param  timx GP16C2T instance
  * @retval Timer SMCON register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_smcon(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->SMCON));
}

/**
  * @brief  Timer master/slave mode setup.
  * @param  timx GP16C2T instance 
  * @param  MSMode master/slave mode
            @arg @ref MD_GP16C2T_MASTERSLAVE_NODELAY
            @arg @ref MD_GP16C2T_MASTERSLAVE_DELAY
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_smcon_mscfg(GP16C2T_TypeDef *timx, uint32_t MSMode)
{
  MODIFY_REG(timx->SMCON, GP16C2T_SMCON_MSCFG_MSK, MSMode);
}

/**
  * @brief  Get timer master/slave mode.
  * @param  timx GP16C2T instance
  * @retval Timer master/slave mode.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_smcon_mscfg(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP16C2T_SMCON_MSCFG_MSK) >> GP16C2T_SMCON_MSCFG_POS);
}

/**
  * @brief  Timer trigger selection setup.
  * @param  timx GP16C2T instance 
  * @param  TriggerSelect Trigger selection
            @arg @ref MD_GP16C2T_TRIGGERSELECT_ITR0
            @arg @ref MD_GP16C2T_TRIGGERSELECT_ITR1
            @arg @ref MD_GP16C2T_TRIGGERSELECT_ITR2
            @arg @ref MD_GP16C2T_TRIGGERSELECT_ITR3
            @arg @ref MD_GP16C2T_TRIGGERSELECT_TI1FED
            @arg @ref MD_GP16C2T_TRIGGERSELECT_TI1FP1
            @arg @ref MD_GP16C2T_TRIGGERSELECT_TI2FP2
            @arg @ref MD_GP16C2T_TRIGGERSELECT_ETRF
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_smcon_tssel(GP16C2T_TypeDef *timx, uint32_t TriggerSelect)
{
  MODIFY_REG(timx->SMCON, GP16C2T_SMCON_TSSEL_MSK, TriggerSelect);
}

/**
  * @brief  Get timer trigger selection.
  * @param  timx GP16C2T instance
  * @retval Timer trigger selection.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_smcon_tssel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP16C2T_SMCON_TSSEL_MSK) >> GP16C2T_SMCON_TSSEL_POSS);
}

/**
  * @brief  Timer slave mode selection setup.
  * @param  timx GP16C2T instance 
  * @param  SlaveMode Slave mode selection
            @arg @ref MD_GP16C2T_SLAVEMODE_DISABLE
            @arg @ref MD_GP16C2T_SLAVEMODE_ENCODER1
            @arg @ref MD_GP16C2T_SLAVEMODE_ENCODER2
            @arg @ref MD_GP16C2T_SLAVEMODE_ENCODER3
            @arg @ref MD_GP16C2T_SLAVEMODE_RESET
            @arg @ref MD_GP16C2T_SLAVEMODE_GATED
            @arg @ref MD_GP16C2T_SLAVEMODE_TRIGGER
            @arg @ref MD_GP16C2T_SLAVEMODE_EXTERNALCLOCK
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_smcon_smods(GP16C2T_TypeDef *timx, uint32_t SlaveMode)
{
  MODIFY_REG(timx->SMCON, GP16C2T_SMCON_SMODS_MSK, SlaveMode);
}

/**
  * @brief  Get timer slave mode selection.
  * @param  timx GP16C2T instance
  * @retval Timer slave mode selection.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_smcon_smods(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->SMCON, GP16C2T_SMCON_SMODS_MSK) >> GP16C2T_SMCON_SMODS_POSS);
}

/**
  * @brief  Timer IER setup.
  * @param  timx GP16C2T instance
  * @param  value (CH2OVI | CH1OVI | BRKI | TRGI | COMI | CH2I | CH1I | UI)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ier(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->IER, value);
}

/**
  * @brief  Timer cpture/compare 2 overcapture interrupt enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ier_ch2ovi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP16C2T_IER_CH2OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 1 overcapture interrupt enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ier_ch1ovi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP16C2T_IER_CH1OVI_MSK);
}

/**
  * @brief  Timer break interrupt enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ier_brki(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP16C2T_IER_BRKI_MSK);
}

/**
  * @brief  Timer trigger interrupt enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ier_trgi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP16C2T_IER_TRGI_MSK);
}

/**
  * @brief  Timer COM interrupt enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ier_comi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP16C2T_IER_COMI_MSK);
}

/**
  * @brief  Timer capture/compare 2 interrupt enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ier_ch2i(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP16C2T_IER_CH2I_MSK);
}

/**
  * @brief  Timer capture/compare 1 interrupt enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ier_ch1i(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP16C2T_IER_CH1I_MSK);
}

/**
  * @brief  Timer update interrupt enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ier_ui(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IER, GP16C2T_IER_UI_MSK);
}

/**
  * @brief  Timer IDR setup.
  * @param  timx GP16C2T instance
  * @param  value (CH2OVI | CH1OVI | BRKI | TRGI | COMI | CH2I | CH1I | UI)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_idr(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->IDR, value);
}

/**
  * @brief  Timer cpture/compare 2 overcapture interrupt disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_idr_ch2ovi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP16C2T_IDR_CH2OVI_MSK);
}

/**
  * @brief  Timer cpture/compare 1 overcapture interrupt disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_idr_ch1ovi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP16C2T_IDR_CH1OVI_MSK);
}

/**
  * @brief  Timer break interrupt disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_idr_brki(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP16C2T_IDR_BRKI_MSK);
}

/**
  * @brief  Timer trigger interrupt disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_idr_trgi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP16C2T_IDR_TRGI_MSK);
}

/**
  * @brief  Timer COM interrupt disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_idr_comi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP16C2T_IDR_COMI_MSK);
}

/**
  * @brief  Timer capture/compare 2 interrupt disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_idr_ch2i(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP16C2T_IDR_CH2I_MSK);
}

/**
  * @brief  Timer capture/compare 1 interrupt disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_idr_ch1i(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP16C2T_IDR_CH1I_MSK);
}

/**
  * @brief  Timer update interrupt disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_idr_ui(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->IDR, GP16C2T_IDR_UI_MSK);
}

/**
  * @brief  Get timer IVS setup.
  * @param  timx GP16C2T instance
  * @retval Timer IVS setup.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ivs(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->IVS));
}

/**
  * @brief  Indicates whether the timer cpture/compare 2 overcapture interrupt is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ivs_ch2ovi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP16C2T_IVS_CH2OVI_MSK) == (GP16C2T_IVS_CH2OVI_MSK));
}

/**
  * @brief  Indicates whether the timer capture/compare 1 overcapture interrupt is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ivs_ch1ovi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP16C2T_IVS_CH1OVI_MSK) == (GP16C2T_IVS_CH1OVI_MSK));
}

/**
  * @brief  Indicates whether the timer break interrupt is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ivs_brki(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP16C2T_IVS_BRKI_MSK) == (GP16C2T_IVS_BRKI_MSK));
}

/**
  * @brief  Indicates whether the timer trigger interrupt is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ivs_trgi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP16C2T_IVS_TRGI_MSK) == (GP16C2T_IVS_TRGI_MSK));
}

/**
  * @brief  Indicates whether the timer COM interrupt is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ivs_comi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP16C2T_IVS_COMI_MSK) == (GP16C2T_IVS_COMI_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 interrupt is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ivs_ch2i(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP16C2T_IVS_CH2I_MSK) == (GP16C2T_IVS_CH2I_MSK));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 interrupt is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ivs_ch1i(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP16C2T_IVS_CH1I_MSK) == (GP16C2T_IVS_CH1I_MSK));
}

/**
  * @brief  Indicates whether the timer update interrupt is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ivs_ui(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, GP16C2T_IVS_UI_MSK) == (GP16C2T_IVS_UI_MSK));
}

/**
  * @brief  Get timer RIF flag.
  * @param  timx GP16C2T instance
  * @retval Timer RIF flag.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_rif(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->RIF));
}

/**
  * @brief  Get timer capture/compare 2 overcapture interrupt flag.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_rif_ch2ovi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP16C2T_RIF_CH2OVI_MSK) == (GP16C2T_RIF_CH2OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 1 overcapture interrupt flag.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_rif_ch1ovi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP16C2T_RIF_CH1OVI_MSK) == (GP16C2T_RIF_CH1OVI_MSK));
}

/**
  * @brief  Get timer break interrupt flag. 
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_rif_brki(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP16C2T_RIF_BRKI_MSK) == (GP16C2T_RIF_BRKI_MSK));
}

/**
  * @brief  Get timer trigger interrupt flag. 
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_rif_trgi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP16C2T_RIF_TRGI_MSK) == (GP16C2T_RIF_TRGI_MSK));
}

/**
  * @brief  Get timer COM interrupt flag. 
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_rif_comi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP16C2T_RIF_COMI_MSK) == (GP16C2T_RIF_COMI_MSK));
}

/**
  * @brief  Get timer capture/compare 2 interrupt flag.  
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_rif_ch2i(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP16C2T_RIF_CH2I_MSK) == (GP16C2T_RIF_CH2I_MSK));
}

/**
  * @brief  Get timer capture/compare 1 interrupt flag.  
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_rif_ch1i(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP16C2T_RIF_CH1I_MSK) == (GP16C2T_RIF_CH1I_MSK));
}

/**
  * @brief  Get timer update interrupt flag.  
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_rif_ui(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, GP16C2T_RIF_UI_MSK) == (GP16C2T_RIF_UI_MSK));
}

/**
  * @brief  Get timer IFM flag.
  * @param  timx GP16C2T instance
  * @retval Timer IFM flag.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ifm(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->IFM));
}

/**
  * @brief  Get timer capture/compare 2 overcapture interrupt flag masked.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_ifm_ch2ovi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP16C2T_IFM_CH2OVI_MSK) == (GP16C2T_IFM_CH2OVI_MSK));
}

/**
  * @brief  Get timer capture/compare 1 overcapture interrupt flag masked.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_ifm_ch1ovi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP16C2T_IFM_CH1OVI_MSK) == (GP16C2T_IFM_CH1OVI_MSK));
}

/**
  * @brief  Get timer break interrupt flag masked. 
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_ifm_brki(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP16C2T_IFM_BRKI_MSK) == (GP16C2T_IFM_BRKI_MSK));
}

/**
  * @brief  Get timer trigger interrupt flag masked. 
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_ifm_trgi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP16C2T_IFM_TRGI_MSK) == (GP16C2T_IFM_TRGI_MSK));
}

/**
  * @brief  Get timer COM interrupt flag masked. 
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_ifm_comi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP16C2T_IFM_COMI_MSK) == (GP16C2T_IFM_COMI_MSK));
}

/**
  * @brief  Get timer capture/compare 2 interrupt flag masked.  
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_ifm_ch2i(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP16C2T_IFM_CH2I_MSK) == (GP16C2T_IFM_CH2I_MSK));
}

/**
  * @brief  Get timer capture/compare 1 interrupt flag masked.  
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_ifm_ch1i(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP16C2T_IFM_CH1I_MSK) == (GP16C2T_IFM_CH1I_MSK));
}

/**
  * @brief  Get timer update interrupt flag masked.  
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_gp16c2t_is_active_flag_ifm_ui(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, GP16C2T_IFM_UI_MSK) == (GP16C2T_IFM_UI_MSK));
}

/**
  * @brief  Timer ICR setup.
  * @param  timx GP16C2T instance
  * @param  value (CC2OI | CC1OI | BI | TI | COMI | CC2I | CC1I | UI)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_icr(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->ICR, value);
}

/**
  * @brief  Clear timer cpture/compare 2 overcapture interrupt flag.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_clear_flag_icr_ch2ovi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP16C2T_ICR_CH2OVI_MSK);
}

/**
  * @brief  Clear timer cpture/compare 1 overcapture interrupt flag.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_clear_flag_icr_ch1ovi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP16C2T_ICR_CH1OVI_MSK);
}

/**
  * @brief  Clear timer break interrupt flag.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_clear_flag_icr_brki(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP16C2T_ICR_BRKI_MSK);
}

/**
  * @brief  Clear timer trigger interrupt flag.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_clear_flag_icr_trgi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP16C2T_ICR_TRGI_MSK);
}

/**
  * @brief  Clear timer COM interrupt flag.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_clear_flag_icr_comi(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP16C2T_ICR_COMI_MSK);
}

/**
  * @brief  Clear timer capture/compare 2 interrupt flag.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_clear_flag_icr_ch2i(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP16C2T_ICR_CH2I_MSK);
}

/**
  * @brief  Clear timer capture/compare 1 interrupt flag.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_clear_flag_icr_ch1i(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP16C2T_ICR_CH1I_MSK);
}

/**
  * @brief  Clear timer upadte flag.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_clear_flag_icr_ui(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->ICR, GP16C2T_ICR_UI_MSK);
}

/**
  * @brief  Timer SGE setup.
  * @param  timx GP16C2T instance
  * @param  value (SGBRK | SGTRG | SGCOM | SGCH2 | SGCH1 | SGU)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_sge(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->SGE, value);
}

/**
  * @brief  Timer break generation.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_sge_sgbrk(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP16C2T_SGE_SGBRK_MSK);
}

/**
  * @brief  Timer trigger generation.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_sge_sgtrg(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP16C2T_SGE_SGTRG_MSK);
}

/**
  * @brief  Timer COM generation.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_sge_sgcom(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP16C2T_SGE_SGCOM_MSK);
}

/**
  * @brief  Timer Capture/Compare 2 generation.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_sge_sgch2(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP16C2T_SGE_SGCH2_MSK);
}

/**
  * @brief  Timer Capture/Compare 1 generation.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_sge_sgch1(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP16C2T_SGE_SGCH1_MSK);
}

/**
  * @brief  Timer update generation.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_sge_sgu(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->SGE, GP16C2T_SGE_SGU_MSK);
}

/**
  * @brief  Timer CHMR1 setup.
  * @param  timx GP16C2T instance
  * @param  value output (CH2MOD | CH2PEN | CH2FEN | CC2SSEL | CH1MOD | CH1PEN | CH1FEN | CC1SSEL)
  *               input  (I2FLT | I2PRES | CC2SSEL | I1FLT | I1PRES | CC1SSEL)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CHMR1, value);
}

/**
  * @brief  GET Timer CHMR1 register value.
  * @param  timx GP16C2T instance
  * @retval Timer CHMR1 register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->CHMR1));
}

/**
  * @brief  Timer output compare 2 mode setup.
  * @param  timx GP16C2T instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_GP16C2T_OUTPUTMODE_DISABLE
            @arg @ref MD_GP16C2T_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_GP16C2T_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_GP16C2T_OUTPUTMODE_TOGGLE
            @arg @ref MD_GP16C2T_OUTPUTMODE_FORCELOW
            @arg @ref MD_GP16C2T_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_GP16C2T_OUTPUTMODE_PWMMODE1
            @arg @ref MD_GP16C2T_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_output_ch2mod(GP16C2T_TypeDef *timx, uint32_t OutputMode)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH2MOD_MSK, (OutputMode << GP16C2T_CHMR1_OUTPUT_CH2MOD_POSS));
}

/**
  * @brief  Get timer output compare 2 mode.
  * @param  timx GP16C2T instance
  * @retval Timer output compare 2 mode.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_output_ch2mod(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH2MOD_MSK) >> GP16C2T_CHMR1_OUTPUT_CH2MOD_POSS);
}

/**
  * @brief  Timer output compare 2 preload enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_chmr1_output_ch2pen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH2PEN_MSK);
}

/**
  * @brief  Timer output compare 2 preload disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_chmr1_output_ch2pen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH2PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 2 preload is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_chmr1_output_ch2pen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH2PEN_MSK) == (GP16C2T_CHMR1_OUTPUT_CH2PEN_MSK));
}

/**
  * @brief  Timer output compare 2 fast enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_chmr1_output_ch2fen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH2FEN_MSK);
}

/**
  * @brief  Timer output compare 2 fast disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_chmr1_output_ch2fen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH2FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 2 fast is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_chmr1_output_ch2fen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH2FEN_MSK) == (GP16C2T_CHMR1_OUTPUT_CH2FEN_MSK));
}

/** 
  * @brief  Timer cpture/compare 2 selection setup.
  * @param  timx GP16C2T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP16C2T_CHMODE_OUTPUT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_output_cc2ssel(GP16C2T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CC2SSEL_MSK, (ChannelMode << GP16C2T_CHMR1_OUTPUT_CC2SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 2 selection.
  * @param  timx GP16C2T instance
  * @retval Timer cpture/compare 2 selection.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_output_cc2ssel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CC2SSEL_MSK) >> GP16C2T_CHMR1_OUTPUT_CC2SSEL_POSS);
}

/**
  * @brief  Timer output compare 1 mode setup.
  * @param  timx GP16C2T instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_GP16C2T_OUTPUTMODE_DISABLE
            @arg @ref MD_GP16C2T_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_GP16C2T_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_GP16C2T_OUTPUTMODE_TOGGLE
            @arg @ref MD_GP16C2T_OUTPUTMODE_FORCELOW
            @arg @ref MD_GP16C2T_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_GP16C2T_OUTPUTMODE_PWMMODE1
            @arg @ref MD_GP16C2T_OUTPUTMODE_PWMMODE2
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_output_ch1mod(GP16C2T_TypeDef *timx, uint32_t OutputMode)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH1MOD_MSK, (OutputMode << GP16C2T_CHMR1_OUTPUT_CH1MOD_POSS));
}

/**
  * @brief  Get timer output compare 1 mode.
  * @param  timx GP16C2T instance
  * @retval Timer output compare 1 mode.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_output_ch1mod(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH1MOD_MSK) >> GP16C2T_CHMR1_OUTPUT_CH1MOD_POSS);
}

/**
  * @brief  Timer output compare 1 preload enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_chmr1_output_ch1pen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH1PEN_MSK);
}

/**
  * @brief  Timer output compare 1 preload disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_chmr1_output_ch1pen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH1PEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 1 preload is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_chmr1_output_ch1pen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH1PEN_MSK) == (GP16C2T_CHMR1_OUTPUT_CH1PEN_MSK));
}

/**
  * @brief  Timer output compare 1 fast enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_chmr1_output_ch1fen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH1FEN_MSK);
}

/**
  * @brief  Timer output compare 1 fast disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_chmr1_output_ch1fen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH1FEN_MSK);
}

/**
  * @brief  Indicates whether the timer output compare 1 fast is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_chmr1_output_ch1fen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CH1FEN_MSK) == (GP16C2T_CHMR1_OUTPUT_CH1FEN_MSK));
}

/** 
  * @brief  Timer cpture/compare 1 selection setup.
  * @param  timx GP16C2T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP16C2T_CHMODE_OUTPUT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_output_cc1ssel(GP16C2T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CC1SSEL_MSK, (ChannelMode << GP16C2T_CHMR1_OUTPUT_CC1SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 1 selection.
  * @param  timx GP16C2T instance
  * @retval Timer cpture/compare 1 selection.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_output_cc1ssel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_OUTPUT_CC1SSEL_MSK) >> GP16C2T_CHMR1_OUTPUT_CC1SSEL_POSS);
}

/** 
  * @brief  Timer input capture 2 filter setup.
  * @param  timx GP16C2T instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV1
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV1N2
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV1N4
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV1N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV2N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV2N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV4N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV4N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV8N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV8N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV16N5
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV16N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV16N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV32N5
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV32N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_input_i2flt(GP16C2T_TypeDef *timx, uint32_t InputFliter)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_INPUT_I2FLT_MSK, (InputFliter << GP16C2T_CHMR1_INPUT_I2FLT_POSS));
}

/**
  * @brief  Get timer input capture 2 filter.
  * @param  timx GP16C2T instance
  * @retval Timer input capture 2 filter.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_input_i2flt(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_INPUT_I2FLT_MSK) >> GP16C2T_CHMR1_INPUT_I2FLT_POSS);
}

/** 
  * @brief  Timer input capture 2 prescaler setup.
  * @param  timx GP16C2T instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_GP16C2T_INPUTPRESCALE_DIV1
            @arg @ref MD_GP16C2T_INPUTPRESCALE_DIV2
            @arg @ref MD_GP16C2T_INPUTPRESCALE_DIV4
            @arg @ref MD_GP16C2T_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_input_i2pres(GP16C2T_TypeDef *timx, uint32_t InputPrescale)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_INPUT_I2PRES_MSK, (InputPrescale << GP16C2T_CHMR1_INPUT_I2PRES_POSS));
}

/**
  * @brief  Get timer input capture 2 prescaler.
  * @param  timx GP16C2T instance
  * @retval Timer input capture 2 prescaler.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_input_i2pres(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_INPUT_I2PRES_MSK) >> GP16C2T_CHMR1_INPUT_I2PRES_POSS);
}

/** 
  * @brief  Timer cpture/compare 2 selection setup.
  * @param  timx GP16C2T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP16C2T_CHMODE_OUTPUT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_input_cc2ssel(GP16C2T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_INPUT_CC2SSEL_MSK, (ChannelMode << GP16C2T_CHMR1_INPUT_CC2SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 2 selection.
  * @param  timx GP16C2T instance
  * @retval Timer cpture/compare 2 selection.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_input_cc2ssel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_INPUT_CC2SSEL_MSK) >> GP16C2T_CHMR1_INPUT_CC2SSEL_POSS);
}

/** 
  * @brief  Timer input capture 1 filter setup.
  * @param  timx GP16C2T instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV1
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV1N2
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV1N4
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV1N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV2N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV2N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV4N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV4N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV8N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV8N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV16N5
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV16N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV16N8
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV32N5
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV32N6
            @arg @ref MD_GP16C2T_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_input_i1flt(GP16C2T_TypeDef *timx, uint32_t InputFliter)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_INPUT_I1FLT_MSK, (InputFliter << GP16C2T_CHMR1_INPUT_I1FLT_POSS));
}

/**
  * @brief  Get timer input capture 1 filter.
  * @param  timx GP16C2T instance
  * @retval Timer input capture 1 filter.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_input_i1flt(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_INPUT_I1FLT_MSK) >> GP16C2T_CHMR1_INPUT_I1FLT_POSS);
}

/** 
  * @brief  Timer input capture 1 prescaler setup.
  * @param  timx GP16C2T instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_GP16C2T_INPUTPRESCALE_DIV1
            @arg @ref MD_GP16C2T_INPUTPRESCALE_DIV2
            @arg @ref MD_GP16C2T_INPUTPRESCALE_DIV4
            @arg @ref MD_GP16C2T_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_input_i1pres(GP16C2T_TypeDef *timx, uint32_t InputPrescale)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_INPUT_I1PRES_MSK, (InputPrescale << GP16C2T_CHMR1_INPUT_I1PRES_POSS));
}

/**
  * @brief  Get timer input capture 1 prescaler.
  * @param  timx GP16C2T instance
  * @retval Timer input capture 1 prescaler.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_input_i1pres(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_INPUT_I1PRES_MSK) >> GP16C2T_CHMR1_INPUT_I1PRES_POSS);
}

/** 
  * @brief  Timer cpture/compare 1 selection setup.
  * @param  timx GP16C2T instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_GP16C2T_CHMODE_OUTPUT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_DIRECT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_INDIRECT
            @arg @ref MD_GP16C2T_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_chmr1_input_cc1ssel(GP16C2T_TypeDef *timx, uint32_t ChannelMode)
{
  MODIFY_REG(timx->CHMR1, GP16C2T_CHMR1_INPUT_CC1SSEL_MSK, (ChannelMode << GP16C2T_CHMR1_INPUT_CC1SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 1 selection.
  * @param  timx GP16C2T instance
  * @retval Timer cpture/compare 1 selection.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_chmr1_input_cc1ssel(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CHMR1, GP16C2T_CHMR1_INPUT_CC1SSEL_MSK) >> GP16C2T_CHMR1_INPUT_CC1SSEL_POSS);
}

/**
  * @brief  Timer CCEP setup.
  * @param  timx GP16C2T instance
  * @param  value (CC2NPOL |
  *                CC2POL | CC2EN | CC1NPOL | CC1NEN | CC1POL | CC1EN )
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ccep(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CCEP, value);
}

/**
  * @brief  GET Timer CCEP register value.
  * @param  timx GP16C2T instance
  * @retval Timer CCEP register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ccep(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->CCEP));
}

/** 
  * @brief  Timer capture/compare 2 complementary output polarity setup.
  * @param  timx GP16C2T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP16C2T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP16C2T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ccep_cc2npol(GP16C2T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP16C2T_CCEP_CC2NPOL_MSK, (OutputPolarity << GP16C2T_CCEP_CC2NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 2 complementary output polarity.
  * @param  timx GP16C2T instance
  * @retval Timer capture/compare 2 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ccep_cc2npol(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP16C2T_CCEP_CC2NPOL_MSK) >> GP16C2T_CCEP_CC2NPOL_POS);
}

/** 
  * @brief  Timer capture/compare 2 output polarity setup.
  * @param  timx GP16C2T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP16C2T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP16C2T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ccep_cc2pol(GP16C2T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP16C2T_CCEP_CC2POL_MSK, (OutputPolarity << GP16C2T_CCEP_CC2POL_POS));
}

/**
  * @brief  Get timer capture/compare 2 output polarity.
  * @param  timx GP16C2T instance
  * @retval Timer capture/compare 2 output polarity.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ccep_cc2pol(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP16C2T_CCEP_CC2POL_MSK) >> GP16C2T_CCEP_CC2POL_POS);
}

/**
  * @brief  Timer Capture/Compare 2 output enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ccep_cc2en(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CCEP, GP16C2T_CCEP_CC2EN_MSK);
}

/**
  * @brief  Timer Capture/Compare 2 output disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_ccep_cc2en(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CCEP, GP16C2T_CCEP_CC2EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 output is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ccep_cc2en(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP16C2T_CCEP_CC2EN_MSK) == (GP16C2T_CCEP_CC2EN_MSK));
}

/** 
  * @brief  Timer capture/compare 1 complementary output polarity setup.
  * @param  timx GP16C2T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP16C2T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP16C2T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ccep_cc1npol(GP16C2T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP16C2T_CCEP_CC1NPOL_MSK, (OutputPolarity << GP16C2T_CCEP_CC1NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 1 complementary output polarity.
  * @param  timx GP16C2T instance
  * @retval Timer capture/compare 1 complementary output polarity.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ccep_cc1npol(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP16C2T_CCEP_CC1NPOL_MSK) >> GP16C2T_CCEP_CC1NPOL_POS);
}

/**
  * @brief  Timer Capture/Compare 1 complementary output enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ccep_cc1nen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CCEP, GP16C2T_CCEP_CC1NEN_MSK);
}

/**
  * @brief  Timer Capture/Compare 1 complementary output disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_ccep_cc1nen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CCEP, GP16C2T_CCEP_CC1NEN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 complementary output is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ccep_cc1nen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP16C2T_CCEP_CC1NEN_MSK) == (GP16C2T_CCEP_CC1NEN_MSK));
}

/** 
  * @brief  Timer capture/compare 1 output polarity setup.
  * @param  timx GP16C2T instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_GP16C2T_OUTPUTPOLARITY_HIGH
            @arg @ref MD_GP16C2T_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ccep_cc1pol(GP16C2T_TypeDef *timx, uint32_t OutputPolarity)
{
  MODIFY_REG(timx->CCEP, GP16C2T_CCEP_CC1POL_MSK, (OutputPolarity << GP16C2T_CCEP_CC1POL_POS));
}

/**
  * @brief  Get timer capture/compare 1 output polarity.
  * @param  timx GP16C2T instance
  * @retval Timer capture/compare 1 output polarity.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ccep_cc1pol(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP16C2T_CCEP_CC1POL_MSK) >> GP16C2T_CCEP_CC1POL_POS);
}

/**
  * @brief  Timer Capture/Compare 1 output enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_ccep_cc1en(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->CCEP, GP16C2T_CCEP_CC1EN_MSK);
}

/**
  * @brief  Timer Capture/Compare 1 output disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_ccep_cc1en(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->CCEP, GP16C2T_CCEP_CC1EN_MSK);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 output is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_ccep_cc1en(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CCEP, GP16C2T_CCEP_CC1EN_MSK) == (GP16C2T_CCEP_CC1EN_MSK));
}

/**
  * @brief  Timer COUNT setup.
  * @param  timx GP16C2T instance
  * @param  value COUNT
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_count(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->COUNT, value);
}

/**
  * @brief  GET Timer COUNT register value.
  * @param  timx GP16C2T instance
  * @retval Timer COUNT register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_count(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->COUNT));
}

/** 
  * @brief  Timer counter value setup.
  * @param  timx GP16C2T instance
  * @param  counter Counter value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_count_cntv(GP16C2T_TypeDef *timx, uint32_t counter)
{
  MODIFY_REG(timx->COUNT, GP16C2T_COUNT_CNTV_MSK, counter);
}

/**
  * @brief  Get timer counter value.
  * @param  timx GP16C2T instance
  * @retval Timer counter value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_cnt_cntv(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->COUNT, GP16C2T_COUNT_CNTV_MSK) >> GP16C2T_COUNT_CNTV_POSS);
}

/**
  * @brief  Timer PRES setup.
  * @param  timx GP16C2T instance
  * @param  value PRES
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_pres(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->PRES, value);
}

/**
  * @brief  GET Timer PRES register value.
  * @param  timx GP16C2T instance
  * @retval Timer PRES register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_pres(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->PRES));
}

/** 
  * @brief  Timer prescaler value setup.
  * @param  timx GP16C2T instance
  * @param  prescaler Prescaler value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_pres_pscv(GP16C2T_TypeDef *timx, uint32_t prescaler)
{
  MODIFY_REG(timx->PRES, GP16C2T_PRES_PSCV_MSK, prescaler);
}

/**
  * @brief  Get timer prescaler value.
  * @param  timx GP16C2T instance
  * @retval Timer prescaler value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_pres_pscv(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->PRES, GP16C2T_PRES_PSCV_MSK) >> GP16C2T_PRES_PSCV_POSS);
}

/**
  * @brief  Timer AR setup.
  * @param  timx GP16C2T instance
  * @param  value AR
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ar(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->AR, value);
}

/**
  * @brief  GET Timer AR register value.
  * @param  timx GP16C2T instance
  * @retval Timer AR register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ar(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->AR));
}

/** 
  * @brief  Timer auto-reload value setup.
  * @param  timx GP16C2T instance
  * @param  reload Auto-reload value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ar_arv(GP16C2T_TypeDef *timx, uint32_t reload)
{
  MODIFY_REG(timx->AR, GP16C2T_AR_ARV_MSK, reload);
}

/**
  * @brief  Get timer auto-reload value.
  * @param  timx GP16C2T instance
  * @retval Timer auto-reload value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ar_arv(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->AR, GP16C2T_AR_ARV_MSK) >> GP16C2T_AR_ARV_POSS);
}

/**
  * @brief  Timer REPAR setup.
  * @param  timx GP16C2T instance
  * @param  value REPAR
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_rcr(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->REPAR, value);
}

/**
  * @brief  GET Timer REPAR register value.
  * @param  timx GP16C2T instance
  * @retval Timer REPAR register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_repar(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->REPAR));
}

/** 
  * @brief  Timer repetition counter value setup.
  * @param  timx GP16C2T instance
  * @param  repetition Repetition counter value (between Min_Data=0 and Max_Data=0xFF)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_repar_repv(GP16C2T_TypeDef *timx, uint32_t repetition)
{
  MODIFY_REG(timx->REPAR, GP16C2T_REPAR_REPV_MSK, repetition);
}

/**
  * @brief  Get timer repetition counter value.
  * @param  timx GP16C2T instance
  * @retval Timer repetition counter value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_repar_repv(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->REPAR, GP16C2T_REPAR_REPV_MSK) >> GP16C2T_REPAR_REPV_POSS);
}

/**
  * @brief  Timer CCVAL1 setup.
  * @param  timx GP16C2T instance
  * @param  value CCVAL1
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ccval1(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CCVAL1, value);
}

/**
  * @brief  GET Timer CCVAL1 register value.
  * @param  timx GP16C2T instance
  * @retval Timer CCVAL1 register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ccval1(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->CCVAL1));
}

/** 
  * @brief  Timer capture/compare value 1 setup.
  * @param  timx GP16C2T instance
  * @param  CapCompValue Capture/Compare value 1 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ccval1_ccrv1(GP16C2T_TypeDef *timx, uint32_t CapCompValue)
{
  MODIFY_REG(timx->CCVAL1, GP16C2T_CCVAL1_CCRV1_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 1.
  * @param  timx GP16C2T instance
  * @retval Timer capture/compare value 1.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ccval1_ccrv1(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CCVAL1, GP16C2T_CCVAL1_CCRV1_MSK) >> GP16C2T_CCVAL1_CCRV1_POSS);
}

/**
  * @brief  Timer CCVAL2 setup.
  * @param  timx GP16C2T instance
  * @param  value CCVAL2
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ccval2(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CCVAL2, value);
}

/**
  * @brief  GET Timer CCVAL2 register value.
  * @param  timx GP16C2T instance
  * @retval Timer CCVAL2 register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ccval2(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->CCVAL2));
}

/** 
  * @brief  Timer capture/compare value 2 setup.
  * @param  timx GP16C2T instance
  * @param  CapCompValue Capture/Compare value 2 (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_ccval2_ccrv2(GP16C2T_TypeDef *timx, uint32_t CapCompValue)
{
  MODIFY_REG(timx->CCVAL2, GP16C2T_CCVAL2_CCRV2_MSK, CapCompValue);
}

/**
  * @brief  Get timer capture/compare value 2.
  * @param  timx GP16C2T instance
  * @retval Timer capture/compare value 2.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_ccval2_ccrv2(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->CCVAL2, GP16C2T_CCVAL2_CCRV2_MSK) >> GP16C2T_CCVAL2_CCRV2_POSS);
}

/**
  * @brief  Timer BDCFG setup.
  * @param  timx GP16C2T instance
  * @param  value (GOEN | AOEN | BRKP | BRKEN | OFFSSR | OFFSSI | LOCKLVL | DT)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_bdcfg(GP16C2T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->BDCFG, value);
}

/**
  * @brief  GET Timer BDCFG register value.
  * @param  timx GP16C2T instance
  * @retval Timer BDCFG register value.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_bdcfg(GP16C2T_TypeDef *timx)
{
  return (READ_REG(timx->BDCFG));
}

/**
  * @brief  Timer main output enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_bdcfg_goen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->BDCFG, GP16C2T_BDCFG_GOEN_MSK);
}

/**
  * @brief  Timer main output disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_bdcfg_goen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->BDCFG, GP16C2T_BDCFG_GOEN_MSK);
}

/**
  * @brief  Indicates whether the timer main output is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_bdcfg_goen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->BDCFG, GP16C2T_BDCFG_GOEN_MSK) == (GP16C2T_BDCFG_GOEN_MSK));
}

/**
  * @brief  Timer automatic output enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_bdcfg_aoen(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->BDCFG, GP16C2T_BDCFG_AOEN_MSK);
}

/**
  * @brief  Timer automatic output disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_bdcfg_aoen(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->BDCFG, GP16C2T_BDCFG_AOEN_MSK);
}

/**
  * @brief  Indicates whether the timer automatic output is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_bdcfg_aoen(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->BDCFG, GP16C2T_BDCFG_AOEN_MSK) == (GP16C2T_BDCFG_AOEN_MSK));
}

/** 
  * @brief  Timer break polarity setup.
  * @param  timx GP16C2T instance
  * @param  BreakPolarity Break polarity
  *         @arg @ref MD_GP16C2T_BREAKPOLARITY_LOW
  *         @arg @ref MD_GP16C2T_BREAKPOLARITY_HIGH
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_bdcfg_brkp(GP16C2T_TypeDef *timx, uint32_t BreakPolarity)
{
  MODIFY_REG(timx->BDCFG, GP16C2T_BDCFG_BRKP_MSK, BreakPolarity);
}

/**
  * @brief  Get timer break polarity.
  * @param  timx GP16C2T instance
  * @retval Timer break polarity.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_bdcfg_brkp(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->BDCFG, GP16C2T_BDCFG_BRKP_MSK) >> GP16C2T_BDCFG_BRKP_POS);
}

/**
  * @brief  Timer break enable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_enable_bdcfg_brken(GP16C2T_TypeDef *timx)
{
  SET_BIT(timx->BDCFG, GP16C2T_BDCFG_BRKEN_MSK);
}

/**
  * @brief  Timer break disable.
  * @param  timx GP16C2T instance
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_disable_bdcfg_brken(GP16C2T_TypeDef *timx)
{
  CLEAR_BIT(timx->BDCFG, GP16C2T_BDCFG_BRKEN_MSK);
}

/**
  * @brief  Indicates whether the timer break is enabled.
  * @param  timx GP16C2T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_gp16c2t_is_enabled_bdcfg_brken(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->BDCFG, GP16C2T_BDCFG_BRKEN_MSK) == (GP16C2T_BDCFG_BRKEN_MSK));
}

/** 
  * @brief  Timer off-state selection for run mode setup.
  * @param  timx GP16C2T instance
  * @param  OffStateRun Off-state selection for run mode
  *         @arg @ref MD_GP16C2T_OFFSTATERUN_DISABLE
  *         @arg @ref MD_GP16C2T_OFFSTATERUN_ENABLE
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_bdcfg_offssr(GP16C2T_TypeDef *timx, uint32_t OffStateRun)
{
  MODIFY_REG(timx->BDCFG, GP16C2T_BDCFG_OFFSSR_MSK, OffStateRun);
}

/**
  * @brief  Get timer off-state selection for run mode.
  * @param  timx GP16C2T instance
  * @retval Timer off-state selection for run mode.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_bdcfg_offssr(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->BDCFG, GP16C2T_BDCFG_OFFSSR_MSK) >> GP16C2T_BDCFG_OFFSSR_POS);
}

/** 
  * @brief  Timer off-state selection for idle mode setup.
  * @param  timx GP16C2T instance
  * @param  OffStateIdle Off-state selection for idle mode
  *         @arg @ref MD_GP16C2T_OFFSTATEIDLE_DISABLE
  *         @arg @ref MD_GP16C2T_OFFSTATEIDLE_ENABLE
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_bdcfg_offssi(GP16C2T_TypeDef *timx, uint32_t OffStateIdle)
{
  MODIFY_REG(timx->BDCFG, GP16C2T_BDCFG_OFFSSI_MSK, OffStateIdle);
}

/**
  * @brief  Get timer off-state selection for idle mode.
  * @param  timx GP16C2T instance
  * @retval Timer off-state selection for idle mode.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_bdcfg_offssi(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->BDCFG, GP16C2T_BDCFG_OFFSSI_MSK) >> GP16C2T_BDCFG_OFFSSI_POS);
}

/** 
  * @brief  Timer lock configuration setup.
  * @param  timx GP16C2T instance
  * @param  LockLevel Lock configuration
  *         @arg @ref MD_GP16C2T_LOCKLEVEL_0
  *         @arg @ref MD_GP16C2T_LOCKLEVEL_1
  *         @arg @ref MD_GP16C2T_LOCKLEVEL_2
  *         @arg @ref MD_GP16C2T_LOCKLEVEL_3
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_bdcfg_locklvl(GP16C2T_TypeDef *timx, uint32_t LockLevel)
{
  MODIFY_REG(timx->BDCFG, GP16C2T_BDCFG_LOCKLVL_MSK, LockLevel);
}

/**
  * @brief  Get timer lock configuration.
  * @param  timx GP16C2T instance
  * @retval Timer lock configuration.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_bdcfg_locklvl(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->BDCFG, GP16C2T_BDCFG_LOCKLVL_MSK) >> GP16C2T_BDCFG_LOCKLVL_POSS);
}

/** 
  * @brief  Timer dead-time generator setup.
  * @param  timx GP16C2T instance
  * @param  DeadTime Dead-time generator (between Min_Data=0 and Max_Data=0xFF)
  * @retval None
  */

__STATIC_INLINE void md_gp16c2t_set_bdcfg_dt(GP16C2T_TypeDef *timx, uint32_t DeadTime)
{
  MODIFY_REG(timx->BDCFG, GP16C2T_BDCFG_DT_MSK, DeadTime);
}

/**
  * @brief  Get timer dead-time generator.
  * @param  timx GP16C2T instance
  * @retval Timer dead-time generator.
  */

__STATIC_INLINE uint32_t md_gp16c2t_get_bdcfg_dt(GP16C2T_TypeDef *timx)
{
  return (READ_BIT(timx->BDCFG, GP16C2T_BDCFG_DT_MSK) >> GP16C2T_BDCFG_DT_POSS);
}

/**
  * @} MD_GP16C2T_Public_Macro
  */

/* Public functions -----------------------------------------------------------*/



/**
  * @} GP16C2T
  */


#endif

/**
  * @} Micro_Driver
  */


#ifdef __cplusplus
}
#endif

#endif
