/**********************************************************************************
 *
 * @file    md_fc.h
 * @brief   header file of md_fc.c
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_FC_H__
#define __MD_FC_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include "es32f0930.h"

/** @addtogroup Micro_Driver
  * @{
  */

#if defined (FC)

/** @defgroup  FC  FC
  * @brief FC micro driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/

/* Public types ---------------------------------------------------------------*/
/** @defgroup MD_FC_Public_Types FC Public Types
  * @{
  */

/**
  * @brief MD FC Parameter Structure definition
  */
typedef struct
{
    uint32_t u32SAddr;      /*!< Specifies the FC sector address to be erased.
                This parameter should be aligned with 0x400*/

    uint32_t u32SAddrC;    /*!< Specifies the complement of FC sector address to be erased.
                This parameter should be the complement of FC page address*/

    uint16_t u16BCnt;      /*!< Specifies the control byte count.
                For program, this parameter should be aligned with 8 and smaller than 0x400
                For read, this parameter should be aligned with 4*/

    uint32_t *pu32Buf;    /*!< Specifies the uint32_t data buffer to program.
                This parameter should be the pointer to uint32_t*/

} md_fc_ControlTypeDef;

/**
  * @brief MD FC Update Protect Level Structure definition
  */
typedef struct
{
    uint32_t UpdateL;    /*!< Specifies the protect page bit mapping low byte (page 31~0) (0=protect, 1=unprotect).
                This parameter can be any value between 0~0xffffffff*/

    uint32_t UpdateH;    /*!< Specifies the protect page bit mapping high byte (page 63~32) (0=protect, 1=unprotect).
                This parameter can be any value between 0~0xffffffff*/

} md_fc_UpdProtTypeDef;

/**
  * @} MD_FC_Public_Types
  */

/* Public constants -----------------------------------------------------------*/
/** @defgroup MD_FC_Public_Constants FC Public Constants
  * @{
  */

#define MD_FC_PC_EF_MPAGESZ                     (0x00000200UL)    /** @brief Select main page size */
#define MD_FC_PC_EF_IPAGESZ                     (0x00000200UL)    /** @brief Select information page size */
#define MD_FC_PC_EF_MERASEID                    (0xA5A5AA55UL)    /** @brief Select main erase ID */
#define MD_FC_PC_EF_UL_LOCK                     (0x00000000UL)    /** @brief EF Lock Key */
#define MD_FC_PC_EF_UL_KEY1                     (0x00112233UL)    /** @brief EF Unlock Key1 */
#define MD_FC_PC_EF_UL_KEY2                     (0x55667788UL)    /** @brief EF Unlock Key2 */

#define  MD_FC_PC_CMD_PROGRAM                           (0xF0)        /** @brief Program EFlash */
#define  MD_FC_PC_CMD_ERASE                            (0xF1)         /** @brief Page Erase       */
#define  MD_FC_PC_CMD_MERASE                            (0xF2)        /** @brief Master Erase       */
#define  MD_FC_PC_CMD_SETUCRP                           (0xF5)        /** @brief UCRP Protect       */
#define  MD_FC_PC_CMD_SETRP                             (0xF6)        /** @brief READ Protect       */
#define  MD_FC_PC_CMD_SETWP                             (0xF7)        /** @brief WRITE Protect  */

#define  MD_FC_PC_SREMAP_MAIN          (0x00)        /** @brief Main Flash memory mapped at 0x0000 0000  */
#define  MD_FC_PC_SREMAP_BOOTLOADER    (0x01)        /** @brief Bootloader mapped at 0x0000 0000  */
#define  MD_FC_PC_SREMAP_SRAM          (0x02)        /** @brief SRAM mapped at 0x0000 0000  */

/**
  * @} MD_FC_Public_Constants
  */

/* Public macro ---------------------------------------------------------------*/
/** @defgroup MD_FC_Public_Macros FC Public Macros
  * @{
  */

/** @defgroup MD_FC_PM_WRITE_READ Common write and read registers Macros
  * @{
  */

/**
  * @brief  Write a value in FC register
  * @param  __INSTANCE__ FC Instance
  * @param  __REG__ Register to be written
  * @param  __VALUE__ Value to be written in the register
  * @retval None
  */
#define MD_FC_WRITEREG(__INSTANCE__, __REG__, __VALUE__) WRITE_REG(__INSTANCE__->__REG__, (__VALUE__))

/**
  * @brief  Read a value in FC register
  * @param  __INSTANCE__ FC Instance
  * @param  __REG__ Register to be read
  * @retval Register value
  */
#define MD_FC_READREG(__INSTANCE__, __REG__) READ_REG(__INSTANCE__->__REG__)

/**
  * @brief  Modify value in FC register
  * @param  __INSTANCE__ FC Instance
  * @param  __REG__ Register to be written
  * @param  __MASK__ Mask value to be written in the register
  * @param  __VALUE__ Value to be written in the register
  * @retval None
  */
#define MD_FC_MODIFYREG(__INSTANCE__, __REG__, __MASK__, __VALUE__) MODIFY_REG(__INSTANCE__->__REG__, __MASK__, (__VALUE__))

/**
  * @brief  Set bit value in FC register
  * @param  __INSTANCE__ FC Instance
  * @param  __REG__ Register to be written
  * @param  __MASK__ Mask value to be set in the register
  * @retval None
  */
#define MD_FC_SET_BIT(__INSTANCE__, __REG__, __MASK__) SET_BIT(__INSTANCE__->__REG__, __MASK__)

/**
  * @brief  Clear bit value in FC register
  * @param  __INSTANCE__ FC Instance
  * @param  __REG__ Register to be written
  * @param  __MASK__ Mask value to be cleared in the register
  * @retval None
  */
#define MD_FC_CLEAR_BIT(__INSTANCE__, __REG__, __MASK__) CLEAR_BIT(__INSTANCE__->__REG__, __MASK__)

/**
  * @brief  Read bit value in FC register
  * @param  __INSTANCE__ FC Instance
  * @param  __REG__ Register to be read
  * @param  __MASK__ Mask value to be read in the register
  * @retval Register bit value
  */
#define MD_FC_READBIT(__INSTANCE__, __REG__, __MASK__) READ_BIT(__INSTANCE__->__REG__, __MASK__)

/**
  * @} MD_FC_PM_WRITE_READ
  */

/** @defgroup MD_FC_Macro_Drivers FC Public Macro Drivers
  * @{
  */

/**
  * @brief  Set FC CMD Flash Command
  * @note  After procedure is finished, Flash command will be cleaned by itself
  * @param  The retval can be one of the following values:
  *                 @arg @ref MD_FC_PC_CMD_PROGRAM(0xF0) : Program EFlash
  *         @arg @ref MD_FC_PC_CMD_SERASE(0xF1) : Sector Erase
  *         @arg @ref MD_FC_PC_CMD_MERASE(0xF2) : Master Erase
  *         @arg @ref MD_FC_PC_CMD_SETUCRP(0xF5) : Set UCRP
  *         @arg @ref MD_FC_PC_CMD_SETRP(0xF6) : Set RP
  *         @arg @ref MD_FC_PC_CMD_SETWP(0xF7) : Set WP
  *         @arg Else : Reserved
  * @retval None
  */
__STATIC_INLINE void md_fc_set_flash_command(uint32_t Cmd)
{
    MD_FC_MODIFYREG(FC, CMD, FC_CMD_FC_CMD_MSK, (Cmd << FC_CMD_FC_CMD_POSS));
}

/**
  * @brief  Get FC CMD Flash Command
  * @note  After procedure is finished, Flash command will be cleaned by itself
  * @param  None
  * @retval The retval can be one of the following values:
  *                 @arg @ref MD_FC_PC_CMD_PROGRAM(0xF0) : Program EFlash
  *         @arg @ref MD_FC_PC_CMD_SERASE(0xF1) : Sector Erase
  *         @arg @ref MD_FC_PC_CMD_MERASE(0xF2) : Master Erase
  *         @arg @ref MD_FC_PC_CMD_SETUCRP(0xF5) : Set UCRP
  *         @arg @ref MD_FC_PC_CMD_SETRP(0xF6) : Set RP
  *         @arg @ref MD_FC_PC_CMD_SETWP(0xF7) : Set WP
  *         @arg Else : Reserved
  */
__STATIC_INLINE uint32_t md_fc_get_flash_command(void)
{
    return (MD_FC_READBIT(FC, CMD, FC_CMD_FC_CMD_MSK) >> FC_CMD_FC_CMD_POSS);
}

/**
  * @brief  Set FC PA
  * @param  @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_pa(uint32_t u32Data)
{
    MD_FC_WRITEREG(FC, PA, u32Data);
}

/**
  * @brief  Get FC PA
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_pa(void)
{
    return (MD_FC_READREG(FC, PA));
}

/**
  * @brief  Set FC Program Counter
  * @note  Provide maximum 128 times continuous program
  * @note  Meanwhile user needs to fill out FC_PLD and FC_CMD
  * @param  @arg Max Value 0x7f
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_program_counter(uint32_t PCnt)
{
    MD_FC_MODIFYREG(FC, PA, FC_PA_PCNT_MSK, ((((PCnt + 3) >> 2) - 1) << FC_PA_PCNT_POSS));
}

/**
  * @brief  Get FC Program Counter
  * @note  Provide maximum 128 times continuous program
  * @note  Meanwhile user needs to fill out FC_PLD and FC_CMD
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0x7f
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint8_t md_fc_get_program_counter(void)
{
    return (MD_FC_READBIT(FC, PA, FC_PA_PCNT_MSK) >> FC_PA_PCNT_POSS);
}

/**
  * @brief  Enable FC Information Block
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_fc_enable_info_region(void)
{
    MD_FC_SET_BIT(FC, PA, FC_PA_IFREN_MSK);
}

/**
  * @brief  Disable FC Information Block
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_fc_disable_info_region(void)
{
    MD_FC_CLEAR_BIT(FC, PA, FC_PA_IFREN_MSK);
}

/**
  * @brief  Is FC Information Block Enabled
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Enable
  *         @arg 0x0 : Disable
  */
__STATIC_INLINE uint8_t md_fc_is_enabled_info_region(void)
{
    return (MD_FC_READBIT(FC, PA, FC_PA_IFREN_MSK) >> FC_PA_IFREN_POS);
}

/**
  * @brief  Set FC Program/Erase Address
  * @param  @arg Max Value 0xffc0
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_program_address(uint32_t PAddr)
{
    MD_FC_MODIFYREG(FC, PA, FC_PA_PROG_ADDR_MSK, (PAddr << FC_PA_PROG_ADDR_POSS));
}

/**
  * @brief  Get FC Program/Erase Address
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffc0
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_program_address(void)
{
    return (MD_FC_READBIT(FC, PA, FC_PA_PROG_ADDR_MSK) >> FC_PA_PROG_ADDR_POSS);
}

/**
  * @brief  Set FC Program Low Data
  * @note  Eflash supports 32 bits data program, this register provides Low 32 bits data
  * @param  @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_program_data(uint32_t u32Data)
{
    MD_FC_WRITEREG(FC, PLD, u32Data);
}

/**
  * @brief  Get FC Program Low Data
  * @note  Eflash supports 32 bits data program, this register provides Low 32 bits data
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_program_data(void)
{
    return (MD_FC_READREG(FC, PLD));
}

/**
  * @brief  Set FC CON
  * @param  @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_ctl(uint32_t u32Data)
{
    MD_FC_WRITEREG(FC, CON, u32Data);
}

/**
  * @brief  Get FC CON
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_ctl(void)
{
    return (MD_FC_READREG(FC, CON));
}

/**
  * @brief  Set FC CON OPRLD
  * @param  @arg Max Value 0xf
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_option_byte_reload(uint32_t oprld)
{
    MD_FC_MODIFYREG(FC, CON, FC_CON_OPRLD_MSK, (oprld << FC_CON_OPRLD_POSS));
}

/**
  * @brief  Get FC Control OPRLD
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xf
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint8_t md_fc_get_option_byte_reload_times(void)
{
    return (MD_FC_READBIT(FC, CON, FC_CON_OPRLD_MSK) >> FC_CON_OPRLD_POSS);
}

/**
  * @brief  Set FC CON Wait Cycle
  * @note  Eflash supports 32 bits data program, this register provides high 32 bits data
  * @param  @arg Max Value 0x3
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_read_latency(uint32_t Wait)
{
    MD_FC_MODIFYREG(FC, CON, FC_CON_WAIT_MSK, (Wait << FC_CON_WAIT_POSS));
}

/**
  * @brief  Get FC Control Wait Cycle
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0x3
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint8_t md_fc_get_read_latency(void)
{
    return (MD_FC_READBIT(FC, CON, FC_CON_WAIT_MSK) >> FC_CON_WAIT_POSS);
}

/**
  * @brief  Get FC STAT
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_sta(void)
{
    return (MD_FC_READREG(FC, STAT));
}

/**
  * @brief  Get FC Unlock Status
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0x1 : Option Byte Data is valid
  *         @arg Min Value 0x0 : Option Byte Data is invalid
  */
__STATIC_INLINE uint8_t md_fc_is_active_flag_flash_unlock(void)
{
    return (MD_FC_READBIT(FC, STAT, FC_STAT_FCUNLOCK_MSK) >> FC_STAT_FCUNLOCK_POS);
}

/**
  * @brief  Get FC Busy Status
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0x1 : Lv2 Protect Function is disabled
  *         @arg Min Value 0x0 : Lv2 Protect Function is enabled
  */
__STATIC_INLINE uint8_t md_fc_is_active_flag_flash_busy(void)
{
    return (MD_FC_READBIT(FC, STAT, FC_STAT_FCBUSY_MSK) >> FC_STAT_FCBUSY_POS);
}

/**
  * @brief  Get FC Write Protect Status
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0x1 : Lv1 Protect Function is disabled
  *         @arg Min Value 0x0 : Lv1 Protect Function is enabled
  */
__STATIC_INLINE uint8_t md_fc_is_active_flag_wp_disable(void)
{
    return (MD_FC_READBIT(FC, STAT, FC_STAT_WPDIS_MSK) >> FC_STAT_WPDIS_POS);
}

/**
  * @brief  Get FC Read Protect Status
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0x1 : Lv1 Protect Function is disabled
  *         @arg Min Value 0x0 : Lv1 Protect Function is enabled
  */
__STATIC_INLINE uint8_t md_fc_get_rp_level(void)
{
    return (MD_FC_READBIT(FC, STAT, FC_STAT_RPLV_MSK) >> FC_STAT_RPLV_POSS);
}

/**
  * @brief  Get FC UCR Protect Status
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0x1 : Lv1 Protect Function is disabled
  *         @arg Min Value 0x0 : Lv1 Protect Function is enabled
  */
__STATIC_INLINE uint8_t md_fc_is_active_flag_ucrp_disable(void)
{
    return (MD_FC_READBIT(FC, STAT, FC_STAT_UCRPDIS_MSK) >> FC_STAT_UCRPDIS_POS);
}

/**
  * @brief  Set FC Protect Update Low Data
  * @param  @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_protect0(uint32_t u32Data)
{
    MD_FC_WRITEREG(FC, UPL, u32Data);
}

/**
  * @brief  Get FC Protect Update Low Data
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_protect0(void)
{
    return (MD_FC_READREG(FC, UPL));
}

/**
  * @brief  Set FC Protect Update High Data
  * @param  @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_protect1(uint32_t u32Data)
{
    MD_FC_WRITEREG(FC, UPH, u32Data);
}

/**
  * @brief  Get FC Protect Update High Data
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_protect1(void)
{
    return (MD_FC_READREG(FC, UPH));
}

/**
  * @brief  Set FC Unlock Data
  * @param  @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  * @retval None
  */
__STATIC_INLINE void md_fc_set_flash_unlock(uint32_t u32Data)
{
    MD_FC_WRITEREG(FC, UL, u32Data);
}

/**
  * @brief  Get FC Unlock Data
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_flash_unlock(void)
{
    return (MD_FC_READREG(FC, UL));
}

/**
  * @brief  Get FC UCR Protect Low
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_ucrp0(void)
{
    return (MD_FC_READREG(FC, UCRPL));
}

/**
  * @brief  Get FC UCR Protect High
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_ucrp1(void)
{
    return (MD_FC_READREG(FC, UCRPH));
}

/**
  * @brief  Get FC Read Protect
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_rp(void)
{
    return (MD_FC_READREG(FC, RP));
}

/**
  * @brief  Get FC Write Protect Low
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_wp0(void)
{
    return (MD_FC_READREG(FC, WPL));
}

/**
  * @brief  Get FC Write Protect High
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_wp1(void)
{
    return (MD_FC_READREG(FC, WPH));
}

/**
  * @brief  Get FC Remap
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xffffffff
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_remap(void)
{
    return (MD_FC_READREG(FC, REMAP));
}

/**
  * @brief  Get FC Remap HRemap
  * @param  None
  * @retval The retval can be one of the following values:
    *         @arg 0xA5 :   Remap to Main Flash memory and Skip First Bootloader
  *         @arg Else : Remap to First Bootloader
  */
__STATIC_INLINE uint32_t md_fc_get_option_bootrom_bypass(void)
{
    return (MD_FC_READBIT(FC, WPH, FC_REMAP_HREMAP_MSK) >> FC_REMAP_HREMAP_POSS);
}

/**
  * @brief  Get FC Remap SRemap
  * @param  None
  * @retval The retval can be one of the following values:
    *         @arg MD_FC_PC_SREMAP_MAIN(0x00) :             Main Flash memory mapped at 0x00000000
    *         @arg MD_FC_PC_SREMAP_BOOTLOADER(0x01) :   Bootloader mapped at 0x00000000
    *         @arg MD_FC_PC_SREMAP_SRAM(0x02) :             SRAM mapped at 0x00000000
  */
__STATIC_INLINE uint32_t md_fc_get_option_memory_mapping(void)
{
    return (MD_FC_READBIT(FC, WPH, FC_REMAP_SREMAP_MSK) >> FC_REMAP_SREMAP_POSS);
}

/**
  * @brief  Get FC Remap SEFbase
  * @param  None
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0x3f
  *         @arg Min Value 0x0
  */
__STATIC_INLINE uint32_t md_fc_get_option_flash_remap_base(void)
{
    return (MD_FC_READBIT(FC, WPH, FC_REMAP_SEFBASE_MSK) >> FC_REMAP_SEFBASE_POSS);
}


/**
  * @} MD_FC_Macro_Drivers
  */

/**
  * @} MD_FC_Public_Macros
  */

/* Public functions -----------------------------------------------------------*/
/** @defgroup MD_FC_Public_Functions FC Public Functions
  * @{
  */
ErrorStatus  md_fc_unlock(void);
ErrorStatus  md_fc_lock(void);
ErrorStatus  md_fc_merase(uint32_t MEraseID);
ErrorStatus  md_fc_page_erase(md_fc_ControlTypeDef *pSErasePara);
ErrorStatus  md_fc_program(md_fc_ControlTypeDef *pProgramPara);
ErrorStatus  md_fc_read(md_fc_ControlTypeDef *pReadPara);
ErrorStatus  md_fc_updprotl1(md_fc_UpdProtTypeDef *pUpdPL1Para);
ErrorStatus  md_fc_updprotl2(md_fc_UpdProtTypeDef *pUpdPL2Para);
ErrorStatus  md_fc_update_ucrp(md_fc_UpdProtTypeDef *pUpdPL1Para);
ErrorStatus  md_fc_update_rp(md_fc_UpdProtTypeDef *pUpdPL1Para);
ErrorStatus  md_fc_update_wp(md_fc_UpdProtTypeDef *pUpdPL1Para);
ErrorStatus  md_fc_update_remap(uint32_t UpdRemap);

ErrorStatus  md_fc_read_adc_temperature(uint32_t *ADCTEMP);
ErrorStatus  md_fc_read_info(uint32_t info_addr, uint32_t *info);
double  md_fc_read_adc_temperature_voltage(void);
/**
  * @}  MD_FC_Public_Functions
  */

#endif

/**
  * @}  FC
  */

/**
  * @}  Micro_Driver
  */

#ifdef __cplusplus
}
#endif

#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


