/**
  **************************************************************************************
  * @file    md_bs16t.h
  * @brief   BS16T Head File
  *
  * @version V1.00
  * @date    31/3/2020
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. AMD rights reserved.
  **************************************************************************************
  */
  
 /* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_BS16T_H__
#define __MD_BS16T_H__

#ifdef __cplusplus
extern "C"{
#endif

/* Includes -------------------------------------------------------*/
#include "es32f0930.h"

/** @addtogroup Micro_Driver
  * @{
  */
#if defined (BS16T1)

/** @defgroup BS16T BS16T
  * @brief BS16T module driver
  * @{
  */


/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private constants ----------------------------------------------------------*/
/* Private Macros ----------------------------------------------------------*/



/* Public Constants -----------------------------------------------------------*/

/** @defgroup MD_BS16T_Public_Constants Public Constants
  * @brief BS16T module driver
  * @{
  */
/* BS16T_CON1 macros define*/
#define MD_BS16T_ONEPULSEMODE_DISABLE     (0 << BS16T_CON1_SPMEN_POS)       /*!<  Counter is not stopped at update event*/
#define MD_BS16T_ONEPULSEMODE_ENABLE      (1 << BS16T_CON1_SPMEN_POS)       /*!<  Counter stops counting at the next update event*/

#define MD_BS16T_UPDATESOURCE_NORMAL      (0 << BS16T_CON1_UERSEL_POS)      /*!<  Counter overflow/underflow, Setting the UG bit, Update generation through the slave mode controller generate an update interrupt*/
#define MD_BS16T_UPDATESOURCE_COUNTER     (1 << BS16T_CON1_UERSEL_POS)      /*!<  Only counter overflow/underflow generates an update interrupt or DMA request if Enabled*/

/* BS16T_CON2 macros define*/
#define MD_BS16T_IDLEOUTPUT_LOW           0                                   /*!<  OC1=0 (after a dead-time if OC1N is implemented) when MOE=0*/
#define MD_BS16T_IDLEOUTPUT_HIGH          1                                   /*!<  OC1=1 (after a dead-time if OC1N is implemented) when MOE=0*/

#define MD_BS16T_MASTERMODE_RESET         (0 << BS16T_CON2_MMSEL_POSS)      /*!<  Reset mode*/
#define MD_BS16T_MASTERMODE_ENABLE        (1 << BS16T_CON2_MMSEL_POSS)      /*!<  Enable mode*/
#define MD_BS16T_MASTERMODE_UPDATE        (2 << BS16T_CON2_MMSEL_POSS)      /*!<  Update mode*/
#define MD_BS16T_MASTERMODE_COMPAREPULSE  (3 << BS16T_CON2_MMSEL_POSS)      /*!<  Compare Pulse mode*/
#define MD_BS16T_MASTERMODE_COMPARE1      (4 << BS16T_CON2_MMSEL_POSS)      /*!<  Compare 1 mode*/
#define MD_BS16T_MASTERMODE_COMPARE2      (5 << BS16T_CON2_MMSEL_POSS)      /*!<  Compare 2 mode*/
#define MD_BS16T_MASTERMODE_COMPARE3      (6 << BS16T_CON2_MMSEL_POSS)      /*!<  Compare 3 mode*/
#define MD_BS16T_MASTERMODE_COMPARE4      (7 << BS16T_CON2_MMSEL_POSS)      /*!<  Compare 4 mode*/

/* BS16T_IER IDR IVS RIF IFM ICR macros define*/
#define MD_BS16T_INTERRUPTFALG_UI         (1 << BS16T_IER_UI_POS)           /*!<  Update interrupt*/

/* BS16T_SGE macros define*/
#define MD_BS16T_EVENTGENERATION_UG       (1 << BS16T_SGE_SGU_POS)          /*!<  Update generation*/

/**
  * @} MD_BS16T_Public_Constants
  */

/* Public Macro ------------------------------------------------------------*/
/** @defgroup MD_BS16T_Public_Macro Public Macro
  * @brief BS16T module driver
  * @{
  */
  
/**
  * @brief  Timer CON1 setup.
  * @param  timx BS16T instance
  * @param  value (ARPEN | SPMEN | UERSEL | DISUE | CNTEN)
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_con1(BS16T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CON1, value);
}

/**
  * @brief  GET Timer CON1 register value.
  * @param  timx BS16T instance
  * @retval Timer CON1 register value.
  */

__STATIC_INLINE uint32_t md_bs16t_get_con1(BS16T_TypeDef *timx)
{
  return (READ_REG(timx->CON1));
}

/**
  * @brief  Timer auto-reload preload enable.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_enable_con1_arpen(BS16T_TypeDef *timx)
{
  SET_BIT(timx->CON1, BS16T_CON1_ARPEN_MSK);
}

/**
  * @brief  Timer auto-reload preload disable.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_disable_con1_arpen(BS16T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, BS16T_CON1_ARPEN_MSK);
}

/**
  * @brief  Indicates whether the timer auto-reload preload is enabled.
  * @param  timx BS16T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_bs16t_is_enabled_con1_arpen(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, BS16T_CON1_ARPEN_MSK) == (BS16T_CON1_ARPEN_MSK));
}

/**
  * @brief  Timer one pulse mode enable.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_enable_con1_spmen(BS16T_TypeDef *timx)
{
  SET_BIT(timx->CON1, BS16T_CON1_SPMEN_MSK);
}

/**
  * @brief  Timer one pulse mode disable.
  * @param  timx BS16T instance
  * @retval None.
  */

__STATIC_INLINE void md_bs16t_disable_con1_spmen(BS16T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, BS16T_CON1_SPMEN_MSK);
}

/**
  * @brief  Indicates whether the timer one pulse mode is enabled.
  * @param  timx BS16T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_bs16t_is_enabled_con1_spmen(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, BS16T_CON1_SPMEN_MSK) == (BS16T_CON1_SPMEN_MSK));
}

/**
  * @brief  Timer update request source setup.
  * @param  timx BS16T instance
  * @param  UpdateSource Update request source select
            @arg @ref MD_BS16T_UPDATESOURCE_NORMAL
            @arg @ref MD_BS16T_UPDATESOURCE_COUNTER
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_con1_uersel(BS16T_TypeDef *timx, uint32_t UpdateSource)
{
  MODIFY_REG(timx->CON1, BS16T_CON1_UERSEL_MSK, UpdateSource);
}

/**
  * @brief  Get timer update request source.
  * @param  timx BS16T instance
  * @retval Timer update request source.
  */

__STATIC_INLINE uint32_t md_bs16t_get_con1_uersel(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, BS16T_CON1_UERSEL_MSK) >> BS16T_CON1_UERSEL_POS);
}

/**
  * @brief  Timer update event enable.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_enable_con1_disue(BS16T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, BS16T_CON1_DISUE_MSK);
}

/**
  * @brief  Timer update event disable.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_disble_con1_disue(BS16T_TypeDef *timx)
{
  SET_BIT(timx->CON1, BS16T_CON1_DISUE_MSK);
}

/**
  * @brief  Indicates whether the timer update event is enabled.
  * @param  timx BS16T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_bs16t_is_enabled_con1_disue(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, BS16T_CON1_DISUE_MSK) == (BS16T_CON1_DISUE_MSK));
}

/**
  * @brief  Timer counter enable.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_enable_con1_cnten(BS16T_TypeDef *timx)
{
  SET_BIT(timx->CON1, BS16T_CON1_CNTEN_MSK);
}
  

/**
  * @brief  Timer counter disable.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_disable_con1_cnten(BS16T_TypeDef *timx)
{
  CLEAR_BIT(timx->CON1, BS16T_CON1_CNTEN_MSK);
}

/**
  * @brief  Indicates whether the timer counter is enabled.
  * @param  timx BS16T instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_bs16t_is_enabled_con1_cnten(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->CON1, BS16T_CON1_CNTEN_MSK) == (BS16T_CON1_CNTEN_MSK));
}

/**
  * @brief  Timer CON2 setup.
  * @param  timx BS16T instance
  * @param  value (MMSEL)
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_con2(BS16T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->CON2, value);
}

/**
  * @brief  GET Timer CON2 register value.
  * @param  timx BS16T instance
  * @retval Timer CON2 register value.
  */

__STATIC_INLINE uint32_t md_bs16t_get_con2(BS16T_TypeDef *timx)
{
  return (READ_REG(timx->CON2));
}

/**
  * @brief  Timer master mode selection setup.
  * @param  timx BS16T instance
  * @param  MasterMode Master mode selection
            @arg @ref MD_BS16T_MASTERMODE_RESET
            @arg @ref MD_BS16T_MASTERMODE_ENABLE
            @arg @ref MD_BS16T_MASTERMODE_UPDATE
            @arg @ref MD_BS16T_MASTERMODE_COMPAREPULSE
            @arg @ref MD_BS16T_MASTERMODE_COMPARE1
            @arg @ref MD_BS16T_MASTERMODE_COMPARE2
            @arg @ref MD_BS16T_MASTERMODE_COMPARE3
            @arg @ref MD_BS16T_MASTERMODE_COMPARE4
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_con2_mmsel(BS16T_TypeDef *timx, uint32_t MasterMode)
{
  MODIFY_REG(timx->CON2, BS16T_CON2_MMSEL_MSK, MasterMode);
}

/**
  * @brief  Get timer master mode selection.
  * @param  timx BS16T instance
  * @retval Timer master mode selection.
  */

__STATIC_INLINE uint32_t md_bs16t_get_con2_mmsel(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->CON2, BS16T_CON2_MMSEL_MSK) >> BS16T_CON2_MMSEL_POSS);
}

/**
  * @brief  Timer IER setup.
  * @param  timx BS16T instance
  * @param  value (UI)
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_ier(BS16T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->IER, value);
}

/**
  * @brief  Timer update interrupt enable.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_enable_ier_ui(BS16T_TypeDef *timx)
{
  SET_BIT(timx->IER, BS16T_IER_UI_MSK);
}

/**
  * @brief  Timer IDR setup.
  * @param  timx BS16T instance
  * @param  value (UI)
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_idr(BS16T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->IDR, value);
}

/**
  * @brief  Timer update interrupt disable.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_disable_idr_ui(BS16T_TypeDef *timx)
{
  SET_BIT(timx->IDR, BS16T_IDR_UI_MSK);
}

/**
  * @brief  Get timer IVS setup.
  * @param  timx BS16T instance
  * @retval Timer IVS setup.
  */

__STATIC_INLINE uint32_t md_bs16t_get_ivs(BS16T_TypeDef *timx)
{
  return (READ_REG(timx->IVS));
}

__STATIC_INLINE uint32_t md_bs16t_is_enabled_ivs_ui(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->IVS, BS16T_IVS_UI_MSK) == (BS16T_IVS_UI_MSK));
}

/**
  * @brief  Get timer RIF flag.
  * @param  timx BS16T instance
  * @retval Timer RIF flag.
  */

__STATIC_INLINE uint32_t md_bs16t_get_rif(BS16T_TypeDef *timx)
{
  return (READ_REG(timx->RIF));
}

/**
  * @brief  Get timer update interrupt flag.  
  * @param  timx BS16T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_bs16t_is_active_flag_rif_ui(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->RIF, BS16T_RIF_UI_MSK) == (BS16T_RIF_UI_MSK));
}

/**
  * @brief  Get timer IFM flag.
  * @param  timx BS16T instance
  * @retval Timer IFM flag.
  */

__STATIC_INLINE uint32_t md_bs16t_get_ifm(BS16T_TypeDef *timx)
{
  return (READ_REG(timx->IFM));
}

/**
  * @brief  Get timer update interrupt flag masked.  
  * @param  timx BS16T instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_bs16t_is_active_flag_ifm_ui(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->IFM, BS16T_IFM_UI_MSK) == (BS16T_IFM_UI_MSK));
}

/**
  * @brief  Timer ICR setup.
  * @param  timx BS16T instance
  * @param  value (UI)
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_icr(BS16T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->ICR, value);
}

/**
  * @brief  Clear timer upadte flag.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_clear_flag_icr_ui(BS16T_TypeDef *timx)
{
  SET_BIT(timx->ICR, BS16T_ICR_UI_MSK);
}

/**
  * @brief  Timer SGE setup.
  * @param  timx BS16T instance
  * @param  value (SGU)
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_sge(BS16T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->SGE, value);
}

/**
  * @brief  Timer update generation.
  * @param  timx BS16T instance
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_sge_sgu(BS16T_TypeDef *timx)
{
  SET_BIT(timx->SGE, BS16T_SGE_SGU_MSK);
}

/**
  * @brief  Timer COUNT setup.
  * @param  timx BS16T instance
  * @param  value COUNT
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_count(BS16T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->COUNT, value);
}

/**
  * @brief  GET Timer COUNT register value.
  * @param  timx BS16T instance
  * @retval Timer COUNT register value.
  */

__STATIC_INLINE uint32_t md_bs16t_get_count(BS16T_TypeDef *timx)
{
  return (READ_REG(timx->COUNT));
}

/** 
  * @brief  Timer counter value setup.
  * @param  timx BS16T instance
  * @param  counter Counter value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_count_cntv(BS16T_TypeDef *timx, uint32_t counter)
{
  MODIFY_REG(timx->COUNT, BS16T_COUNT_CNTV_MSK, counter);
}

/**
  * @brief  Get timer counter value.
  * @param  timx BS16T instance
  * @retval Timer counter value.
  */

__STATIC_INLINE uint32_t md_bs16t_get_cnt_cntv(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->COUNT, BS16T_COUNT_CNTV_MSK) >> BS16T_COUNT_CNTV_POSS);
}

/**
  * @brief  Timer PRES setup.
  * @param  timx BS16T instance
  * @param  value PRES
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_pres(BS16T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->PRES, value);
}

/**
  * @brief  GET Timer PRES register value.
  * @param  timx BS16T instance
  * @retval Timer PRES register value.
  */

__STATIC_INLINE uint32_t md_bs16t_get_pres(BS16T_TypeDef *timx)
{
  return (READ_REG(timx->PRES));
}

/** 
  * @brief  Timer prescaler value setup.
  * @param  timx BS16T instance
  * @param  prescaler Prescaler value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_pres_pscv(BS16T_TypeDef *timx, uint32_t prescaler)
{
  MODIFY_REG(timx->PRES, BS16T_PRES_PSCV_MSK, prescaler);
}

/**
  * @brief  Get timer prescaler value.
  * @param  timx BS16T instance
  * @retval Timer prescaler value.
  */

__STATIC_INLINE uint32_t md_bs16t_get_pres_pscv(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->PRES, BS16T_PRES_PSCV_MSK) >> BS16T_PRES_PSCV_POSS);
}

/**
  * @brief  Timer AR setup.
  * @param  timx BS16T instance
  * @param  value AR
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_ar(BS16T_TypeDef *timx, uint32_t value)
{
  WRITE_REG(timx->AR, value);
}

/**
  * @brief  GET Timer AR register value.
  * @param  timx BS16T instance
  * @retval Timer AR register value.
  */

__STATIC_INLINE uint32_t md_bs16t_get_ar(BS16T_TypeDef *timx)
{
  return (READ_REG(timx->AR));
}

/** 
  * @brief  Timer auto-reload value setup.
  * @param  timx BS16T instance
  * @param  reload Auto-reload value (between Min_Data=0 and Max_Data=0xFFFF)
  * @retval None
  */

__STATIC_INLINE void md_bs16t_set_ar_arv(BS16T_TypeDef *timx, uint32_t reload)
{
  MODIFY_REG(timx->AR, BS16T_AR_ARV_MSK, reload);
}

/**
  * @brief  Get timer auto-reload value.
  * @param  timx BS16T instance
  * @retval Timer auto-reload value.
  */

__STATIC_INLINE uint32_t md_bs16t_get_ar_arv(BS16T_TypeDef *timx)
{
  return (READ_BIT(timx->AR, BS16T_AR_ARV_MSK) >> BS16T_AR_ARV_POSS);
}

/**
  * @} MD_BS16T_Public_Macro
  */

/* Public functions -----------------------------------------------------------*/



/**
  * @} BS16T
  */


#endif

/**
  * @} Micro_Driver
  */


#ifdef __cplusplus
}
#endif

#endif
