/**
  *********************************************************************************
  *
  * @file    md_uart.c
  * @brief   UART module driver.
  *
  * @version V1.0
  * @date    20 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          20 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "md_conf.h"

/** @addtogroup Micro_Driver
  * @{
  */
#ifdef MD_UART

/** @addtogroup MD_UART
  * @{
  */

/** @addtogroup MD_UART_Public_Functions
  * @{
  */
/** @addtogroup MD_UART_Public_Functions_Group1
  * @{
  */
/**
  * @brief  Reset UART peripheral
  * @param  UARTx: UART peripheral
  * @retval None
  */
void md_uart_reset(UART_TypeDef *UARTx)
{
	WRITE_REG(UARTx->BRR, 0x0);
	WRITE_REG(UARTx->LCR, 0x0);
	WRITE_REG(UARTx->MCR, 0x0);
	WRITE_REG(UARTx->CR, 0x0);
	WRITE_REG(UARTx->RTOR, 0x0);
	WRITE_REG(UARTx->FCR, 0x0);
	WRITE_REG(UARTx->IDR, 0xFFF);
}

/**
  * @brief  Initializes the UARTx according to the specified
  *         parameters in the uart_init_t.
  * @param  UARTx: UART peripheral
  * @param  init: Pointer to a md_uart_init_t structure that contains
  *         the configuration information for the specified UART module.
  * @retval None
  */
void md_uart_init(UART_TypeDef *UARTx, md_uart_init_t *init)
{
	md_uart_reset(UARTx);

	MODIFY_REG(UARTx->LCR, UART_LCR_DLS_MSK, init->word_length);
	MODIFY_REG(UARTx->LCR, UART_LCR_STOP_MSK, init->stop_bits);
	MODIFY_REG(UARTx->LCR, UART_LCR_PEN_MSK | UART_LCR_PS_MSK, init->parity);
	MODIFY_REG(UARTx->MCR, UART_MCR_AFCEN_MSK, init->fctl);
	SET_BIT(UARTx->LCR, UART_LCR_BRWEN_MSK);
	WRITE_REG(UARTx->BRR, md_cmu_get_pclk1_clock() / init->baud);
	CLEAR_BIT(UARTx->LCR, UART_LCR_BRWEN_MSK);
	MODIFY_REG(UARTx->FCR, UART_FCR_TXTL_MSK, 1 << UART_FCR_TXTL_POSS);
	SET_BIT(UARTx->FCR, UART_FCR_FIFOEN_MSK);
	SET_BIT(UARTx->LCR, UART_LCR_RXEN_MSK);
}

/**
  * @brief  Initialize the UARTx peripheral using the default parameters.
  * @param  init: Pointer to a md_uart_init_t structure
  * @retval None
  */
void md_uart_init_struct(md_uart_init_t *init)
{
	init->baud        = 115200;
	init->word_length = MD_UART_WORD_LENGTH_8B;
	init->stop_bits   = MD_UART_STOP_BITS_1;
	init->parity      = MD_UART_PARITY_NONE;
	init->fctl        = MD_UART_FLOW_CTL_DISABLE;

	return;
}
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
#endif
/**
 * @}
 */
