/**
  *********************************************************************************
  *
  * @file    md_lpuart.c
  * @brief   LPUART module driver.
  *
  * @version V1.0
  * @date    20 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          20 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "md_conf.h"

/** @addtogroup Micro_Driver
  * @{
  */
#ifdef MD_LPUART

/** @addtogroup MD_LPUART
  * @{
  */

/** @addtogroup MD_LPUART_Public_Functions
  * @{
  */
/** @addtogroup MD_LPUART_Public_Functions_Group1
  * @{
  */
/**
  * @brief  Reset LPUART peripheral
  * @param  LPUARTx: LPUART peripheral
  * @retval None
  */
void md_lpuart_reset(LPUART_TypeDef *LPUARTx)
{
	WRITE_REG(LPUARTx->CON0, 0x3000);
	WRITE_REG(LPUARTx->CON1, 0x4);
	WRITE_REG(LPUARTx->CLKDIV, 0x0);
	WRITE_REG(LPUARTx->FIFOCON, 0x0);
	WRITE_REG(LPUARTx->IER, 0x0);
}

/**
  * @brief  Initializes the LPUART according to the specified
  *         parameters in the md_lpuart_init_t.
  * @param  LPUARTx: LPUART peripheral
  * @param  init: Pointer to a md_lpuart_init_t structure that contains
  *         the configuration information for the specified LPUART module.
  * @retval None
  */
void md_lpuart_init(LPUART_TypeDef *LPUARTx, md_lpuart_init_t *init)
{
	SYSCFG_UNLOCK();
	md_cmu_set_lpuart0_clock_source(0x5); /* Select LRC */
	SYSCFG_LOCK();

	md_lpuart_reset(LPUARTx);
	SET_BIT(LPUARTx->UPDATE, LPUART_UPDATE_UDIS_MSK);
	MODIFY_REG(LPUARTx->CON0, LPUART_CON0_DATLENTH_MSK, init->word_length);
	MODIFY_REG(LPUARTx->CON0, LPUART_CON0_STPLENTH_MSK, init->stop_bits);
	MODIFY_REG(LPUARTx->CON0, (LPUART_CON0_PARCHKE_MSK | LPUART_CON0_EVENPARSEL_MSK), init->parity);
	MODIFY_REG(LPUARTx->CON0, (LPUART_CON0_ATRTSE_MSK | LPUART_CON0_ATCTSE_MSK), init->fctl);
	MODIFY_REG(LPUARTx->CON0, LPUART_CON0_MODESEL_MSK, (init->mode << LPUART_CON0_MODESEL_POSS));
	WRITE_REG(LPUARTx->CLKDIV, (32768 << 8) / init->baud);

	if (init->mode == MD_LPUART_MODE_IrDA)
		CLEAR_BIT(LPUARTx->CON1, LPUART_CON1_IRRXINV_MSK);

	CLEAR_BIT(LPUARTx->UPDATE, LPUART_UPDATE_UDIS_MSK);
	while (LPUARTx->SYNCSTAT & 0xF)
		;

	return;
}
/**
  * @brief  Initialize the LPUARTx peripheral using the default parameters.
  * @param  init: Pointer to a md_lpuart_init_t structure
  * @retval None
  */
void md_lpuart_init_struct(md_lpuart_init_t *init)
{
	init->baud        = 2400;
	init->word_length = MD_LPUART_WORD_LENGTH_8B;
	init->stop_bits   = MD_LPUART_STOP_BITS_1;
	init->parity      = MD_LPUART_PARITY_NONE;
	init->fctl        = MD_LPUART_FLOW_CTL_DISABLE;
	init->mode        = MD_LPUART_MODE_UART;

	return;
}
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
#endif
/**
 * @}
 */
