/**
  *********************************************************************************
  *
  * @file    md_lpuart.h
  * @brief   Header file of LPUART module driver.
  *
  * @version V1.0
  * @date    20 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          20 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */


#ifndef __MD_LPUART_H__
#define __MD_LPUART_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "md_utils.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_LPUART LPUART
  * @brief LPUART micro driver
  * @{
  */

/**
  * @defgroup MD_LPUART_Public_Macros LPUART Public Macros
  * @{
  */

/**
  * @brief LPUART word length
  */
typedef enum {
	MD_LPUART_WORD_LENGTH_5B = 0x0U,
	MD_LPUART_WORD_LENGTH_6B = 0x1U,
	MD_LPUART_WORD_LENGTH_7B = 0x2U,
	MD_LPUART_WORD_LENGTH_8B = 0x3U,
	MD_LPUART_WORD_LENGTH_9B = 0x4U,
} md_lpuart_word_length_t;

/**
  * @brief  LPUART stop bits
  */
typedef enum {
	MD_LPUART_STOP_BITS_1 = 0x0U,
	MD_LPUART_STOP_BITS_2 = 0x8U,
} md_lpuart_stop_bits_t;

/**
  * @brief LPUART parity
  */
typedef enum {
	MD_LPUART_PARITY_NONE = 0x0U,
	MD_LPUART_PARITY_ODD  = 0x10U,
	MD_LPUART_PARITY_EVEN = 0x30U,
} md_lpuart_parity_t;

/**
  * @brief LPUART hardware flow control
  */
typedef enum {
	MD_LPUART_FLOW_CTL_DISABLE = 0x0U,
	MD_LPUART_FLOW_CTL_RTS     = 0x400U,
	MD_LPUART_FLOW_CTL_CTS     = 0x800U,
	MD_LPUART_FLOW_CTL_RTS_CTS = 0xC00U,
} md_lpuart_flow_ctl_t;

/**
  * @brief  LPUART Mode
  */
typedef enum {
	MD_LPUART_MODE_UART  = 0x0U,
	MD_LPUART_MODE_IrDA  = 0x2U,
	MD_LPUART_MODE_RS485 = 0x3U,
} md_lpuart_mode_t;
/**
  * @}
  */
/**
  * @defgroup MD_LPUART_Public_Types LPUART Public Types
  * @{
  */
/**
  * @brief LPUART Init structure definition
  */
typedef struct {
	uint32_t baud;				/**< Specifies the lpuart communication baud rate */
	md_lpuart_word_length_t word_length;	/**< Specifies the number of data bits transmitted or received in a frame */
	md_lpuart_stop_bits_t stop_bits;	/**< Specifies the number of stop bits transmitted */
	md_lpuart_parity_t parity;		/**< Specifies the parity mode */
	md_lpuart_mode_t mode;			/**< Specifies uart mode */
	md_lpuart_flow_ctl_t fctl;		/**< Specifies wether the hardware flow control mode is enabled or disabled */
} md_lpuart_init_t;
/**
  * @}
  */
/** @defgroup MD_LPUART_Public_Functions LPUART Public Functions
  * @{
  */
/** @defgroup MD_LPUART_Public_Functions_Group2 CON0
  * @{
  */
/**
  * @brief  Set LPUART data width
  * @param  LPUARTx LPUART Instance
  * @param  width This parameter can be one of the following values:
  *         @arg @ref MD_LPUART_WORD_LENGTH_5B
  *         @arg @ref MD_LPUART_WORD_LENGTH_6B
  *         @arg @ref MD_LPUART_WORD_LENGTH_7B
  *         @arg @ref MD_LPUART_WORD_LENGTH_8B
  *         @arg @ref MD_LPUART_WORD_LENGTH_9B
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_datawidth(LPUART_TypeDef *LPUARTx, uint32_t width)
{
	MODIFY_REG(LPUARTx->CON0, LPUART_CON0_DATLENTH_MSK, width);
}

/**
  * @brief  Get LPUART data width
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LPUART_WORD_LENGTH_5B
  *         @arg @ref MD_LPUART_WORD_LENGTH_6B
  *         @arg @ref MD_LPUART_WORD_LENGTH_7B
  *         @arg @ref MD_LPUART_WORD_LENGTH_8B
  *         @arg @ref MD_LPUART_WORD_LENGTH_9B
  */
__STATIC_INLINE uint32_t md_pluart_get_datawidth(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->CON0, LPUART_CON0_DATLENTH_MSK));
}

/**
  * @brief  Set LPUART data stop bit
  * @param  LPUARTx: LPUART Instance
  * @param  stop This parameter can be one of the following values:
  *         @arg @ref MD_LPUART_STOP_BITS_1
  *         @arg @ref MD_LPUART_STOP_BITS_2
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_stop(LPUART_TypeDef *LPUARTx, uint32_t stop)
{
	MODIFY_REG(LPUARTx->CON0, LPUART_CON0_STPLENTH_MSK, stop);
}

/**
  * @brief  Get LPUART data stop bit
  * @note   This bit can only be written when UART is disabled(TXEN and RXEN=0)
  * @param  LPUARTx UART Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LPUART_STOP_BITS_1
  *         @arg @ref MD_LPUART_STOP_BITS_2
  */
__STATIC_INLINE uint32_t md_lpuart_get_stop(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->CON0, LPUART_CON0_STPLENTH_MSK));
}

/**
  * @brief  Set LPUART data parity
  * @param  LPUARTx: LPUART Instance
  * @param  parity This parameter can be one of the following values:
  *         @arg @ref MD_LPUART_PARITY_NONE
  *         @arg @ref MD_LPUART_PARITY_ODD
  *         @arg @ref MD_LPUART_PARITY_EVEN
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_parity(LPUART_TypeDef *LPUARTx, uint32_t parity)
{
	MODIFY_REG(LPUARTx->CON0, LPUART_CON0_PARCHKE_MSK | LPUART_CON0_EVENPARSEL_MSK, parity);
}

/**
  * @brief  Get LPUART data parity
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LPUART_PARITY_NONE
  *         @arg @ref MD_LPUART_PARITY_ODD
  *         @arg @ref MD_LPUART_PARITY_EVEN
  */
__STATIC_INLINE uint32_t md_lpuart_get_parity(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->CON0, LPUART_CON0_PARCHKE_MSK | LPUART_CON0_EVENPARSEL_MSK));
}
/**
  * @brief  Enable LPUART data loop back mode
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_loop_back(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON0, LPUART_CON0_LPBMOD_MSK);
}

/**
  * @brief  Disable LPUART data loop back mode
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_loop_back(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON0, LPUART_CON0_LPBMOD_MSK);
}

/**
  * @brief  Check if is enable LPUART data loop back mode
  * @param  LPUARTx: LPUART Instance
  * @retval state of 0 or 1
  */
__STATIC_INLINE uint32_t md_lpuart_is_enable_loop_back(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->CON0, LPUART_CON0_LPBMOD_MSK) != LPUART_CON0_LPBMOD_MSK);
}

/**
  * @brief  Set LPUART flow control
  * @param  LPUARTx: LPUART Instance
  * @param  ctl This parameter can be one of the following values:
  *         @arg @ref MD_LPUART_FLOW_CTL_DISABLE
  *         @arg @ref MD_LPUART_FLOW_CTL_RTS	
  *         @arg @ref MD_LPUART_FLOW_CTL_CTS
  *         @arg @ref MD_LPUART_FLOW_CTL_RTS_CTS
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_flow_ctl(LPUART_TypeDef *LPUARTx, uint32_t ctl)
{
	MODIFY_REG(LPUARTx->CON0, LPUART_CON0_ATRTSE_MSK | LPUART_CON0_ATCTSE_MSK, ctl);
}

/**
  * @brief  Get LPUART flow control
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LPUART_FLOW_CTL_DISABLE
  *         @arg @ref MD_LPUART_FLOW_CTL_RTS	
  *         @arg @ref MD_LPUART_FLOW_CTL_CTS
  *         @arg @ref MD_LPUART_FLOW_CTL_RTS_CTS
  */
__STATIC_INLINE uint32_t md_lpuart_get_flow_ctl(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->CON0, LPUART_CON0_ATRTSE_MSK | LPUART_CON0_ATCTSE_MSK));
}

/**
  * @brief  Set LPUART mode
  * @param  LPUARTx: LPUART Instance
  * @param  mode This parameter can be one of the following values:
  *         @arg @ref MD_LPUART_MODE_UART
  *         @arg @ref MD_LPUART_MODE_IrDA	
  *         @arg @ref MD_LPUART_MODE_RS485
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_mode(LPUART_TypeDef *LPUARTx, uint32_t mode)
{
	MODIFY_REG(LPUARTx->CON0, LPUART_CON0_MODESEL_MSK, mode);
}

/**
  * @brief  Get LPUART mode
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_LPUART_MODE_UART
  *         @arg @ref MD_LPUART_MODE_IrDA	
  *         @arg @ref MD_LPUART_MODE_RS485
  */
__STATIC_INLINE uint32_t md_lpuart_get_mode(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->CON0, LPUART_CON0_MODESEL_MSK));
}

/**
  * @brief  Set LPUART TX interval
  * @param  LPUARTx: LPUART Instance
  * @param  val This parameter can be 0x0~0xFF
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_tx_interval(LPUART_TypeDef *LPUARTx, uint32_t val)
{
	MODIFY_REG(LPUARTx->CON0, LPUART_CON0_INTERVAL_MSK, val);
}

/**
  * @brief  Get LPUART TX interval
  * @param  LPUARTx: LPUART Instance
  * @retval The value
  */
__STATIC_INLINE uint32_t md_lpuart_get_tx_interval(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->CON0, LPUART_CON0_INTERVAL_MSK));
}

/**
  * @brief  LPUART TXDMA Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_txdma(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON0, LPUART_CON0_TXDMAE_MSK);
}

/**
  * @brief  LPUART TXDMA Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_txdma(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON0, LPUART_CON0_TXDMAE_MSK);
}

/**
  * @brief  Indicate LPUART TXDMA is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_txdma(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON0, LPUART_CON0_TXDMAE_MSK) == (LPUART_CON0_TXDMAE_MSK));
}

/**
  * @brief  LPUART RXDMA Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_rxdma(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON0, LPUART_CON0_RXDMAE_MSK);
}

/**
  * @brief  LPUART RXDMA Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_rxdma(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON0, LPUART_CON0_RXDMAE_MSK);
}

/**
  * @brief  Indicate LPUART RXDMA is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_rxdma(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON0, LPUART_CON0_RXDMAE_MSK) == (LPUART_CON0_RXDMAE_MSK));
}

/**
  * @}
  */
/** @defgroup MD_LPUART_Public_Functions_Group3 CON1
  * @{
  */

/**
  * @brief  LPUART RTS signal Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_rts_signal(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON1, LPUART_CON1_RTS_MSK);
}

/**
  * @brief  LPUART RTS signal Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_rts_signal(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON1, LPUART_CON1_RTS_MSK);
}

/**
  * @brief  Indicate LPUART RTS signal is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_rts_signal(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON1, LPUART_CON1_RTS_MSK) == (LPUART_CON1_RTS_MSK));
}

/**
  * @brief  LPUART IrDA TX Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_irda_tx(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON1, LPUART_CON1_IRTXE_MSK);
}

/**
  * @brief  LPUART IrDA TX Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_irda_tx(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON1, LPUART_CON1_IRTXE_MSK);
}

/**
  * @brief  Indicate LPUART IrDA TX is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_irda_tx(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON1, LPUART_CON1_IRTXE_MSK) == (LPUART_CON1_IRTXE_MSK));
}

/**
  * @brief  LPUART IrDA TX inversion Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_irda_tx_inv(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON1, LPUART_CON1_IRTXINV_MSK);
}

/**
  * @brief  LPUART IrDA TX inversion Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_irda_tx_inv(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON1, LPUART_CON1_IRTXINV_MSK);
}

/**
  * @brief  Indicate LPUART IrDA TX inversion is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_irda_tx_inv(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON1, LPUART_CON1_IRTXINV_MSK) == (LPUART_CON1_IRTXINV_MSK));
}

/**
  * @brief  LPUART IrDA RX inversion Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_irda_rx_inv(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON1, LPUART_CON1_IRRXINV_MSK);
}

/**
  * @brief  LPUART IrDA RX inversion Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_irda_rx_inv(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON1, LPUART_CON1_IRRXINV_MSK);
}

/**
  * @brief  Indicate LPUART IrDA RX inversion is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_irda_rx_inv(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON1, LPUART_CON1_IRRXINV_MSK) == (LPUART_CON1_IRRXINV_MSK));
}

/**
  * @brief  Set LPUART IrDA pulse width
  * @param  LPUARTx: LPUART Instance
  * @param  width: This parameter can be one of the following values:
  *           - 0: 3/16 bit
  *           - 1: 5/16 bit
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_irda_pulse_width(LPUART_TypeDef *LPUARTx, uint32_t width)
{
	MODIFY_REG(LPUARTx->CON1, LPUART_CON1_IRWIDTH_MSK, width << LPUART_CON1_IRWIDTH_POS);
}

/**
  * @brief  Get LPUART IrDA pulse width
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be one of the following values:
  *           - 0: 3/16 bit
  *           - 1: 5/16 bit
  */
__STATIC_INLINE uint32_t md_lpuart_get_irda_pulse_width(LPUART_TypeDef *LPUARTx)
{
	return (READ_BITS(LPUARTx->CON1, LPUART_CON1_IRWIDTH_MSK, LPUART_CON1_IRWIDTH_POS));
}

/**
  * @brief  LPUART RS485 NMM Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_rs485_nmm(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON1, LPUART_CON1_NMPMOD_MSK);
}

/**
  * @brief  LPUART RS485 NMM Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_rs485_nmm(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON1, LPUART_CON1_NMPMOD_MSK);
}

/**
  * @brief  Indicate LPUART RS485 NMM is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_rs485_nmm(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON1, LPUART_CON1_NMPMOD_MSK) == (LPUART_CON1_NMPMOD_MSK));
}

/**
  * @brief  LPUART RS485 AAD Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_rs485_aad(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON1, LPUART_CON1_ATADETE_MSK);
}

/**
  * @brief  LPUART RS485 AAD Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_rs485_aad(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON1, LPUART_CON1_ATADETE_MSK);
}

/**
  * @brief  Indicate LPUART RS485 AAD is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_rs485_aad(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON1, LPUART_CON1_ATADETE_MSK) == (LPUART_CON1_ATADETE_MSK));
}

/**
  * @brief  LPUART RS485 AUD Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_rs485_aud(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON1, LPUART_CON1_ATDIRM_MSK);
}

/**
  * @brief  LPUART RS485 AUD Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_rs485_aud(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON1, LPUART_CON1_ATDIRM_MSK);
}

/**
  * @brief  Indicate LPUART RS485 AUD is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_rs485_aud(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON1, LPUART_CON1_ATDIRM_MSK) == (LPUART_CON1_ATDIRM_MSK));
}

/**
  * @brief  LPUART RS485 ADE Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_rs485_ade(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->CON1, LPUART_CON1_ADETE_MSK);
}

/**
  * @brief  LPUART RS485 ADE Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_rs485_ade(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->CON1, LPUART_CON1_ADETE_MSK);
}

/**
  * @brief  Indicate LPUART RS485 ADE is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_rs485_ade(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->CON1, LPUART_CON1_ADETE_MSK) == (LPUART_CON1_ADETE_MSK));
}

/**
  * @brief  Set LPUART RS485 address
  * @param  LPUARTx: LPUART Instance
  * @param  addr: This parameter can be 0x0~0xFF
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_rs485_addr(LPUART_TypeDef *LPUARTx, uint32_t addr)
{
	MODIFY_REG(LPUARTx->CON1, LPUART_CON1_ADDCMP_MSK, addr << LPUART_CON1_ADDCMP_POSS);
}

/**
  * @brief  Get LPUART RS485 address
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be 0x0~0xFF
  */
__STATIC_INLINE uint32_t md_lpuart_get_rs485_addr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BITS(LPUARTx->CON1, LPUART_CON1_ADDCMP_MSK, LPUART_CON1_ADDCMP_POSS));
}

/**
  * @}
  */

/** @defgroup MD_LPUART_Public_Functions_Group4 CLKDIV
  * @{
  */

/**
  * @brief  Set LPUART baudrate
  * @param  LPUARTx: LPUART Instance
  * @param  baud: This parameter can be 0x0~0xFF
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_baudrate(LPUART_TypeDef *LPUARTx, uint32_t baud)
{
	MODIFY_REG(LPUARTx->CLKDIV, LPUART_CLKDIV_CLKDIV_MSK, baud << LPUART_CLKDIV_CLKDIV_POSS);
}

/**
  * @brief  Get LPUART baudrate
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be 0x0~0xFF
  */
__STATIC_INLINE uint32_t md_lpuart_get_baudrate(LPUART_TypeDef *LPUARTx)
{
	return (READ_BITS(LPUARTx->CLKDIV, LPUART_CLKDIV_CLKDIV_MSK, LPUART_CLKDIV_CLKDIV_POSS));
}
/**
  * @}
  */

/** @defgroup MD_LPUART_Public_Functions_Group5 FIFOCON
  * @{
  */
/**
  * @brief  Reset LPUART RX FIFO
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_reset_rxfifo(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->FIFOCON, LPUART_FIFOCON_RXRESET_MSK);
}

/**
  * @brief  Reset LPUART TX FIFO
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_reset_txfifo(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->FIFOCON, LPUART_FIFOCON_TXRESET_MSK);
}

/**
  * @brief  LPUART RS485 RX Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_rs485_rx(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->FIFOCON, LPUART_FIFOCON_NMPMRXDIS_MSK);
}

/**
  * @brief  LPUART RS485 RX Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_rs485_rx(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->FIFOCON, LPUART_FIFOCON_NMPMRXDIS_MSK);
}

/**
  * @brief  Indicate LPUART RS485 RX is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_rs485_rx(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->FIFOCON, LPUART_FIFOCON_NMPMRXDIS_MSK) == (LPUART_FIFOCON_NMPMRXDIS_MSK));
}

/**
  * @brief  Set LPUART RX FIFO level
  * @param  LPUARTx: LPUART Instance
  * @param  level: This parameter can be one of the following values:
  *           - 0: 1-Byte
  *           - 1: 4-Bytes
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_rxfifo_level(LPUART_TypeDef *LPUARTx, uint32_t level)
{
	MODIFY_REG(LPUARTx->FIFOCON, LPUART_FIFOCON_RXTRGLVL_MSK, level << LPUART_FIFOCON_RXTRGLVL_POSS);
}

/**
  * @brief  Get LPUART RX FIFO level
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be one of the following values:
  *           - 0: 1-Byte
  *           - 1: 4-Bytes
  */
__STATIC_INLINE uint32_t md_lpuart_get_rxfifo_level(LPUART_TypeDef *LPUARTx)
{
	return (READ_BITS(LPUARTx->FIFOCON, LPUART_FIFOCON_RXTRGLVL_MSK, LPUART_FIFOCON_RXTRGLVL_POSS));
}

/**
  * @brief  Set LPUART TX FIFO level
  * @param  LPUARTx: LPUART Instance
  * @param  level: This parameter can be one of the following values:
  *           - 0: 1-Byte
  *           - 1: 4-Bytes
  * @retval None
  */
__STATIC_INLINE void md_lpuart_set_txfifo_level(LPUART_TypeDef *LPUARTx, uint32_t level)
{
	MODIFY_REG(LPUARTx->FIFOCON, LPUART_FIFOCON_RTSTRGLVL_MSK, level << LPUART_FIFOCON_RTSTRGLVL_POSS);
}

/**
  * @brief  Get LPUART TX FIFO level
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be one of the following values:
  *           - 0: 1-Byte
  *           - 1: 4-Bytes
  */
__STATIC_INLINE uint32_t md_lpuart_get_txfifo_level(LPUART_TypeDef *LPUARTx)
{
	return (READ_BITS(LPUARTx->FIFOCON, LPUART_FIFOCON_RTSTRGLVL_MSK, LPUART_FIFOCON_RTSTRGLVL_POSS));
}
/**
  * @}
  */

/** @defgroup MD_LPUART_Public_Functions_Group6 RXDR
  * @{
  */

/**
  * @brief  Read data from the register
  * @param  LPUARTx LPUART Instance
  * @retval Data Value between Min_Data=0x00 and Max_Data=0x1FF
  */
__STATIC_INLINE uint16_t md_lpuart_recv_data(LPUART_TypeDef *LPUARTx)
{
	return (uint16_t)(READ_REG(LPUARTx->RXDR));
}
/**
  * @}
  */

/** @defgroup MD_LPUART_Public_Functions_Group7 TXDR
  * @{
  */
/**
  * @brief  Write data to the data register
  * @param  LPUARTx LPUART Instance
  * @param  data Value between Min_Data=0x00 and Max_Data=0x1FF
  * @retval None
  */
__STATIC_INLINE void md_lpuart_send_data(LPUART_TypeDef *LPUARTx, uint16_t data)
{
	LPUARTx->TXDR = data;
}

/**
  * @}
  */

/** @defgroup MD_LPUART_Public_Functions_Group8 STAT
  * @{
  */

/**
  * @brief  Get LPUART RXPTR state
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be 0x0~0xF
  */
__STATIC_INLINE uint32_t md_lpuart_get_state_rxptr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BITS(LPUARTx->STAT, LPUART_STAT_RXPTR_MSK, LPUART_STAT_RXPTR_POSS));
}

/**
  * @brief  Get LPUART RXEMP state
  * @param  LPUARTx: LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_state_rxemp(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->STAT, LPUART_STAT_RXEMP_MSK) == (LPUART_STAT_RXEMP_MSK));
}

/**
  * @brief  Get LPUART RXFULL state
  * @param  LPUARTx: LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_state_rxfull(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->STAT, LPUART_STAT_RXFULL_MSK) == (LPUART_STAT_RXFULL_MSK));
}

/**
  * @brief  Get LPUART TXPTR state
  * @param  LPUARTx: LPUART Instance
  * @retval The retval can be 0x0~0xF
  */
__STATIC_INLINE uint32_t md_lpuart_get_state_txptr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BITS(LPUARTx->STAT, LPUART_STAT_TXPTR_MSK, LPUART_STAT_TXPTR_POSS));
}

/**
  * @brief  Get LPUART TXEMP state
  * @param  LPUARTx: LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_state_txemp(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->STAT, LPUART_STAT_TXEMP_MSK) == (LPUART_STAT_TXEMP_MSK));
}

/**
  * @brief  Get LPUART TXFULL state
  * @param  LPUARTx: LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_state_txfull(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->STAT, LPUART_STAT_TXFULL_MSK) == (LPUART_STAT_TXFULL_MSK));
}

/**
  * @brief  Get LPUART TXIDLE state
  * @param  LPUARTx: LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_state_txidle(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->STAT, LPUART_STAT_TXIDLE_MSK) == (LPUART_STAT_TXIDLE_MSK));
}

/**
  * @brief  Get LPUART CTS state
  * @param  LPUARTx: LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_state_cts(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->STAT, LPUART_STAT_CTSSTAT_MSK) == (LPUART_STAT_CTSSTAT_MSK));
}

/**
  * @brief  Get LPUART RTS state
  * @param  LPUARTx: LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_state_rts(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->STAT, LPUART_STAT_RTSSTAT_MSK) == (LPUART_STAT_RTSSTAT_MSK));
}

/**
  * @}
  */
/** @defgroup MD_LPUART_Public_Functions_Group9 Interrupt
  * @{
  */
/**
  * @brief  Enable RBR interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_rbr(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_RBRIE_MSK);
}

/**
  * @brief  Disable RBR interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_rbr(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_RBRIE_MSK);
}

/**
  * @brief  Check if RBR interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_rbr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_RBRIE_MSK) == (LPUART_IER_RBRIE_MSK));
}

/**
  * @brief  Enable TBEMP interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_tbemp(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_TBEMPIE_MSK);
}

/**
  * @brief  Disable TBEMP interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_tbemp(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_TBEMPIE_MSK);
}

/**
  * @brief  Check if TBEMP interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_tbemp(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_TBEMPIE_MSK) == (LPUART_IER_TBEMPIE_MSK));
}

/**
  * @brief  Enable CTSDET interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_ctsdet(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_CTSDETIE_MSK);
}

/**
  * @brief  Disable CTSDET interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_ctsdet(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_CTSDETIE_MSK);
}

/**
  * @brief  Check if LPUART CTSDET interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_ctsdet(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_CTSDETIE_MSK) == (LPUART_IER_CTSDETIE_MSK));
}

/**
  * @brief  Enable RXTO interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_rxto(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_RXTOIE_MSK);
}

/**
  * @brief  Disable RXTO interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_rxto(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_RXTOIE_MSK);
}

/**
  * @brief  Check if RXTO interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_rxto(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_RXTOIE_MSK) == (LPUART_IER_RXTOIE_MSK));
}

/**
  * @brief  Enable RXOV interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_rxov(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_RXOVIE_MSK);
}

/**
  * @brief  Disable RXOV interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_rxov(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_RXOVIE_MSK);
}

/**
  * @brief  Check if RXOV interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_rxov(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_RXOVIE_MSK) == (LPUART_IER_RXOVIE_MSK));
}

/**
  * @brief  Enable TXOV interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_txov(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_TXOVIE_MSK);
}

/**
  * @brief  Disable TXOV interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_txov(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_TXOVIE_MSK);
}

/**
  * @brief  Check if TXOV interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_txov(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_TXOVIE_MSK) == (LPUART_IER_TXOVIE_MSK));
}

/**
  * @brief  Enable CTSWK interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_ctswk(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_CTSWKIE_MSK);
}

/**
  * @brief  Disable CTSWK interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_ctswk(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_CTSWKIE_MSK);
}

/**
  * @brief  Check if LPUART CTSWK interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_ctswk(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_CTSWKIE_MSK) == (LPUART_IER_CTSWKIE_MSK));
}

/**
  * @brief  Enable DATWK interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_datwk(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_DATWKIE_MSK);
}

/**
  * @brief  Disable DATWK interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_datwk(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_DATWKIE_MSK);
}

/**
  * @brief  Check if DATWK interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_datwk(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_DATWKIE_MSK) == (LPUART_IER_DATWKIE_MSK));
}

/**
  * @brief  Enable PERR interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_perr(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_PERRIE_MSK);
}

/**
  * @brief  Disable PERR interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_perr(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_PERRIE_MSK);
}

/**
  * @brief  Check if PERR interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_perr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_PERRIE_MSK) == (LPUART_IER_PERRIE_MSK));
}

/**
  * @brief  Enable FERR interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_ferr(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_FERRIE_MSK);
}

/**
  * @brief  Disable FERR interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_ferr(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_FERRIE_MSK);
}

/**
  * @brief  Check if FERR interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_ferr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_FERRIE_MSK) == (LPUART_IER_FERRIE_MSK));
}

/**
  * @brief  Enable BRKERR interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_brkerr(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_BRKERRIE_MSK);
}

/**
  * @brief  Disable BRKERR interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_brkerr(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_BRKERRIE_MSK);
}

/**
  * @brief  Check if LPUART BRKERR interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_brkerr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_BRKERRIE_MSK) == (LPUART_IER_BRKERRIE_MSK));
}

/**
  * @brief  Enable ADET interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_adet(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_ADETIE_MSK);
}

/**
  * @brief  Disable ADET interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_adet(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_ADETIE_MSK);
}

/**
  * @brief  Check if LPUART ADET interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_adet(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_ADETIE_MSK) == (LPUART_IER_ADETIE_MSK));
}

/**
  * @brief  Enable TC interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_it_tc(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IER, LPUART_IER_TCIE_MSK);
}

/**
  * @brief  Disable TC interrupt
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_it_tc(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->IER, LPUART_IER_TCIE_MSK);
}

/**
  * @brief  Check if TC interrupt is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_it_tc(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IER, LPUART_IER_TCIE_MSK) == (LPUART_IER_TCIE_MSK));
}
/**
  * @brief  Get RBR flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_rbr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_RBRIF_MSK) == (LPUART_IFLAG_RBRIF_MSK));
}

/**
  * @brief  Clear RBR flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_rbr(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_RBRIFC_MSK);
}

/**
  * @brief  Get TBEMP flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_tbemp(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_TBEMPIF_MSK) == (LPUART_IFLAG_TBEMPIF_MSK));
}

/**
  * @brief  Clear TBEMP flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_tbemp(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_TBEMPIFC_MSK);
}

/**
  * @brief  Get CTSDET flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_ctsdet(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_CTSDETIF_MSK) == (LPUART_IFLAG_CTSDETIF_MSK));
}

/**
  * @brief  Clear CTSDET flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_ctsdet(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_CTSDETIFC_MSK);
}

/**
  * @brief  Get RXTO flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_rxto(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_RXTOIF_MSK) == (LPUART_IFLAG_RXTOIF_MSK));
}

/**
  * @brief  Get RXOV flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_rxov(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_RXOVIF_MSK) == (LPUART_IFLAG_RXOVIF_MSK));
}

/**
  * @brief  Clear RXOV flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_rxov(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_RXOVIFC_MSK);
}

/**
  * @brief  Get TXOV flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_txov(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_TXOVIF_MSK) == (LPUART_IFLAG_TXOVIF_MSK));
}

/**
  * @brief  Clear TXOV flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_txov(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_TXOVIFC_MSK);
}

/**
  * @brief  Get CTSWK flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_ctswk(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_CTSWKIF_MSK) == (LPUART_IFLAG_CTSWKIF_MSK));
}

/**
  * @brief  Clear CTSWK flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_ctswk(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_CTSWKIFC_MSK);
}

/**
  * @brief  Get DATWK flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_datwk(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_DATWKIF_MSK) == (LPUART_IFLAG_DATWKIF_MSK));
}

/**
  * @brief  Clear DATWK flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_datwk(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_DATWKIFC_MSK);
}

/**
  * @brief  Get PERR flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_perr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_PERRIF_MSK) == (LPUART_IFLAG_PERRIF_MSK));
}

/**
  * @brief  Clear PERR flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_perr(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_PERRIFC_MSK);
}

/**
  * @brief  Get FERR flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_ferr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_FERRIF_MSK) == (LPUART_IFLAG_FERRIF_MSK));
}

/**
  * @brief  Clear FERR flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_ferr(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_FERRIFC_MSK);
}

/**
  * @brief  Get BRKERR flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_brkerr(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_BRKERRIF_MSK) == (LPUART_IFLAG_BRKERRIF_MSK));
}

/**
  * @brief  Clear BRKERR flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_brkerr(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_BRKERRIFC_MSK);
}

/**
  * @brief  Get ADET flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_adet(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_ADETIF_MSK) == (LPUART_IFLAG_ADETIF_MSK));
}

/**
  * @brief  Clear ADET flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_adet(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_ADETIFC_MSK);
}

/**
  * @brief  Get TC flag
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_active_flag_tc(LPUART_TypeDef *LPUARTx)
{
	return (READ_BIT(LPUARTx->IFLAG, LPUART_IFLAG_TCIF_MSK) == (LPUART_IFLAG_TCIF_MSK));
}

/**
  * @brief  Clear TC flag
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_clear_flag_tc(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->IFC, LPUART_IFC_TCIFC_MSK);
}
/**
  * @}
  */
/** @defgroup MD_LPUART_Public_Functions_Group10 UPDATE
  * @{
  */
/**
  * @brief  LPUART update Enable
  * @param  LPUARTx: LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_enable_update(LPUART_TypeDef *LPUARTx)
{
	CLEAR_BIT(LPUARTx->UPDATE, LPUART_UPDATE_UDIS_MSK);
}

/**
  * @brief  LPUART update Disable
  * @param  LPUARTx LPUART Instance
  * @retval None
  */
__STATIC_INLINE void md_lpuart_disable_update(LPUART_TypeDef *LPUARTx)
{
	SET_BIT(LPUARTx->UPDATE, LPUART_UPDATE_UDIS_MSK);
}

/**
  * @brief  Indicate LPUART update is enabled
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_is_enabled_update(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->UPDATE, LPUART_UPDATE_UDIS_MSK) == (LPUART_UPDATE_UDIS_MSK));
}
/**
  * @}
  */
/** @defgroup MD_LPUART_Public_Functions_Group11 SYNCSTAT
  * @{
  */
/**
  * @brief  Get LPUART update CON0 state
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_update_con0(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->SYNCSTAT, LPUART_SYNCSTAT_CON0WBSY_MSK) == (LPUART_SYNCSTAT_CON0WBSY_MSK));
}

/**
  * @brief  Get LPUART update CON1 state
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_update_con1(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->SYNCSTAT, LPUART_SYNCSTAT_CON1WBSY_MSK) == (LPUART_SYNCSTAT_CON1WBSY_MSK));
}

/**
  * @brief  Get LPUART update CLKDIV state
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_update_clkdiv(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->SYNCSTAT, LPUART_SYNCSTAT_CLKDIVWBSY_MSK) == (LPUART_SYNCSTAT_CLKDIVWBSY_MSK));
}

/**
  * @brief  Get LPUART update FIFOCON state
  * @param  LPUARTx LPUART Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_lpuart_get_update_fifocon(LPUART_TypeDef *LPUARTx)
{
	return (uint32_t)(READ_BIT(LPUARTx->SYNCSTAT, LPUART_SYNCSTAT_FIFOCONWBSY_MSK) == (LPUART_SYNCSTAT_FIFOCONWBSY_MSK));
}
/**
  * @}
  */
/** @defgroup MD_LPUART_Public_Functions_Group1 Initialization
  * @{
  */
extern void md_lpuart_reset(LPUART_TypeDef *LPUARTx);
extern void md_lpuart_init(LPUART_TypeDef *LPUARTx, md_lpuart_init_t *init);
extern void md_lpuart_init_struct(md_lpuart_init_t *init);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif
#endif
