/**
  *********************************************************************************
  *
  * @file    md_lcd.h
  * @brief   Header file of LCD module driver.
  *
  * @version V1.0
  * @date    20 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          20 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */


#ifndef __MD_LCD_H__
#define __MD_LCD_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "md_utils.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_LCD LCD
  * @brief LCD micro driver
  * @{
  */
/**
  * @defgroup LCD_Public_Types LCD Public Types
  * @{
  */
/**
  * @brief Lcd vlcd voltage type
  */
typedef enum {
	MD_LCD_VCHPS_3V2 = 0U,	/**< 3.2V */
	MD_LCD_VCHPS_3V8 = 1U,	/**< 3.8V */
	MD_LCD_VCHPS_4V8 = 2U,	/**< 4.8V */
	MD_LCD_VCHPS_5V4 = 3U,	/**< 5.4V */
} md_lcd_vchps_t;

/**
  * @brief Lcd voltage type
  */
typedef enum {
	MD_LCD_VSEL_VDD  = 0U,	/**< VDD */
	MD_LCD_VSEL_CP   = 1U,	/**< Charge pump output */
	MD_LCD_VSEL_VLCD = 2U,	/**< VLCD input */
} md_lcd_vsel_t;

/**
  * @brief Lcd resistance select bit
  */
typedef enum {
	MD_LCD_RES_1MOHM = 0U,	/**< 1M ohm */
	MD_LCD_RES_2MOHM = 1U,	/**< 2M ohm */
	MD_LCD_RES_3MOHM = 2U,	/**< 3M ohm */
} md_lcd_res_t;

/**
  * @brief Lcd bias selector
  */
typedef enum {
	MD_LCD_BIAS_1_4 = 0U,	/**< 1/4 bias */
	MD_LCD_BIAS_1_2 = 2U,	/**< 1/2 bias */
	MD_LCD_BIAS_1_3 = 3U,	/**< 1/3 bias */
} md_lcd_bias_t;

/**
  * @brief Lcd duty
  */
typedef enum {
	MD_LCD_DUTY_STATIC = 0U,	/**< Static duty (COM0) */
	MD_LCD_DUTY_1_2    = 1U,	/**< 1/2 duty (COM0~COM1) */
	MD_LCD_DUTY_1_3    = 2U,	/**< 1/3 duty (COM0~COM2) */
	MD_LCD_DUTY_1_4    = 3U,	/**< 1/4 duty (COM0~COM3) */
	MD_LCD_DUTY_1_6    = 4U,	/**< 1/6 duty (COM0~COM5) */
	MD_LCD_DUTY_1_8    = 5U,	/**< 1/8 duty (COM0~COM7) */
} md_lcd_duty_t;

/**
  * @brief Lcd prescaler
  */
typedef enum {
	MD_LCD_PRS_1     = 0U, 	/**< CLKPRS = LCDCLK / 1 */
	MD_LCD_PRS_2     = 1U, 	/**< CLKPRS = LCDCLK / 2 */
	MD_LCD_PRS_4     = 2U, 	/**< CLKPRS = LCDCLK / 4 */
	MD_LCD_PRS_8     = 3U, 	/**< CLKPRS = LCDCLK / 8 */
	MD_LCD_PRS_16    = 4U, 	/**< CLKPRS = LCDCLK / 16 */
	MD_LCD_PRS_32    = 5U, 	/**< CLKPRS = LCDCLK / 32 */
	MD_LCD_PRS_64    = 6U, 	/**< CLKPRS = LCDCLK / 64 */
	MD_LCD_PRS_128   = 7U, 	/**< CLKPRS = LCDCLK / 128 */
	MD_LCD_PRS_256   = 8U, 	/**< CLKPRS = LCDCLK / 256 */
	MD_LCD_PRS_512   = 9U, 	/**< CLKPRS = LCDCLK / 512 */
	MD_LCD_PRS_1024  = 10U,	/**< CLKPRS = LCDCLK / 1024 */
	MD_LCD_PRS_2048  = 11U,	/**< CLKPRS = LCDCLK / 2048 */
	MD_LCD_PRS_4096  = 12U,	/**< CLKPRS = LCDCLK / 4096 */
	MD_LCD_PRS_8192  = 13U,	/**< CLKPRS = LCDCLK / 8192 */
	MD_LCD_PRS_16384 = 14U,	/**< CLKPRS = LCDCLK / 16384 */
	MD_LCD_PRS_32768 = 15U,	/**< CLKPRS = LCDCLK / 32768 */
} md_lcd_prs_t;

/**
  * @brief Lcd divider
  */
typedef enum {
	MD_LCD_DIV_16 = 0U, 	/**< DIVCLK = CLKPRS / 16 */
	MD_LCD_DIV_17 = 1U, 	/**< DIVCLK = CLKPRS / 17 */
	MD_LCD_DIV_18 = 2U, 	/**< DIVCLK = CLKPRS / 18 */
	MD_LCD_DIV_19 = 3U, 	/**< DIVCLK = CLKPRS / 19 */
	MD_LCD_DIV_20 = 4U, 	/**< DIVCLK = CLKPRS / 20 */
	MD_LCD_DIV_21 = 5U, 	/**< DIVCLK = CLKPRS / 21 */
	MD_LCD_DIV_22 = 6U, 	/**< DIVCLK = CLKPRS / 22 */
	MD_LCD_DIV_23 = 7U, 	/**< DIVCLK = CLKPRS / 23 */
	MD_LCD_DIV_24 = 8U, 	/**< DIVCLK = CLKPRS / 24 */
	MD_LCD_DIV_25 = 9U, 	/**< DIVCLK = CLKPRS / 25 */
	MD_LCD_DIV_26 = 10U,	/**< DIVCLK = CLKPRS / 26 */
	MD_LCD_DIV_27 = 11U,	/**< DIVCLK = CLKPRS / 27 */
	MD_LCD_DIV_28 = 12U,	/**< DIVCLK = CLKPRS / 28 */
	MD_LCD_DIV_29 = 13U,	/**< DIVCLK = CLKPRS / 29 */
	MD_LCD_DIV_30 = 14U,	/**< DIVCLK = CLKPRS / 30 */
	MD_LCD_DIV_31 = 15U,	/**< DIVCLK = CLKPRS / 31 */
} md_lcd_div_t;

/**
  * @brief Lcd dead time
  */
typedef enum {
	MD_LCD_DEAD_TIME_NONE     = 0U,	/**< No dead time */
	MD_LCD_DEAD_TIME_1_DIVCLK = 1U,	/**< Dead time is 1 divclk */
	MD_LCD_DEAD_TIME_2_DIVCLK = 2U,	/**< Dead time is 2 divclk */
	MD_LCD_DEAD_TIME_3_DIVCLK = 3U,	/**< Dead time is 3 divclk */
	MD_LCD_DEAD_TIME_4_DIVCLK = 4U,	/**< Dead time is 4 divclk */
	MD_LCD_DEAD_TIME_5_DIVCLK = 5U,	/**< Dead time is 5 divclk */
	MD_LCD_DEAD_TIME_6_DIVCLK = 6U,	/**< Dead time is 6 divclk */
	MD_LCD_DEAD_TIME_7_DIVCLK = 7U,	/**< Dead time is 7 divclk */
} md_lcd_dead_t;

/**
  * @brief Lcd pulse keep time
  */
typedef enum {
	MD_LCD_PON_NONE     = 0U,	/**< No pulse keep time */
	MD_LCD_PON_1_PRSCLK = 1U,	/**< Pulse keep 1 prsclk */
	MD_LCD_PON_2_PRSCLK = 2U,	/**< Pulse keep 2 prsclk */
	MD_LCD_PON_3_PRSCLK = 3U,	/**< Pulse keep 3 prsclk */
	MD_LCD_PON_4_PRSCLK = 4U,	/**< Pulse keep 4 prsclk */
	MD_LCD_PON_5_PRSCLK = 5U,	/**< Pulse keep 5 prsclk */
	MD_LCD_PON_6_PRSCLK = 6U,	/**< Pulse keep 6 prsclk */
	MD_LCD_PON_7_PRSCLK = 7U,	/**< Pulse keep 7 prsclk */
} md_lcd_pluse_on_t;

/**
  * @brief Lcd vgs select
  */
typedef enum {
	MD_LCD_VGS_0  = 0U, 	/**< Grey level display voltage is 30/45 vlcd */
	MD_LCD_VGS_1  = 1U, 	/**< Grey level display voltage is 31/45 vlcd */
	MD_LCD_VGS_2  = 2U, 	/**< Grey level display voltage is 32/45 vlcd */
	MD_LCD_VGS_3  = 3U, 	/**< Grey level display voltage is 33/45 vlcd */
	MD_LCD_VGS_4  = 4U, 	/**< Grey level display voltage is 34/45 vlcd */
	MD_LCD_VGS_5  = 5U, 	/**< Grey level display voltage is 35/45 vlcd */
	MD_LCD_VGS_6  = 6U, 	/**< Grey level display voltage is 36/45 vlcd */
	MD_LCD_VGS_7  = 7U, 	/**< Grey level display voltage is 37/45 vlcd */
	MD_LCD_VGS_8  = 8U, 	/**< Grey level display voltage is 38/45 vlcd */
	MD_LCD_VGS_9  = 9U, 	/**< Grey level display voltage is 39/45 vlcd */
	MD_LCD_VGS_10 = 10U,	/**< Grey level display voltage is 40/45 vlcd */
	MD_LCD_VGS_11 = 11U,	/**< Grey level display voltage is 41/45 vlcd */
	MD_LCD_VGS_12 = 12U,	/**< Grey level display voltage is 42/45 vlcd */
	MD_LCD_VGS_13 = 13U,	/**< Grey level display voltage is 43/45 vlcd */
	MD_LCD_VGS_14 = 14U,	/**< Grey level display voltage is 44/45 vlcd */
	MD_LCD_VGS_15 = 15U,	/**< Grey level display voltage is equal to vlcd */
} md_lcd_vgs_t;

/**
  * @brief Lcd wave choose
  */
typedef enum {
	MD_LCD_WAVE_A = 0U,	/**< Wave type is A */
	MD_LCD_WAVE_B = 1U,	/**< Wave type is B */
} md_lcd_wfs_t;


/**
  * @brief Lcd configure
  */
typedef struct
{
	md_lcd_vsel_t lcd_vsel;      	/**< Lcd power choose */
	md_lcd_vchps_t lcd_vchps;    	/**< Charge pump voltage choose */
	type_func_t lcd_vbufld;    	/**< Low drive mode function */
	type_func_t lcd_vbufhd;    	/**< High drive mode function */
	uint32_t lcd_dsld;        	/**< Low drive mode level */
	uint32_t lcd_dshd;        	/**< High drive mode level */
	md_lcd_res_t lcd_resld;      	/**< Low dirve mode resistance choose */
	md_lcd_res_t lcd_reshd;      	/**< High dirve mode resistance choose */
	md_lcd_bias_t lcd_bias;      	/**< LCD bias */
	md_lcd_duty_t lcd_duty;      	/**< LCD duty */
	md_lcd_wfs_t lcd_wfs;        	/**< Wave choose */
	md_lcd_prs_t lcd_prs;        	/**< Lcd clock prs */
	md_lcd_div_t lcd_div;        	/**< Lcd div */
	md_lcd_dead_t lcd_dead;      	/**< Lcd dead time */
	md_lcd_pluse_on_t lcd_pon;   	/**< Lcd pluse on time */
	md_lcd_vgs_t lcd_vgs;        	/**< Lcd gray level display voltage */
} md_lcd_init_t;
/**
  * @}
  */

/** @defgroup MD_LCD_Public_Functions LCD Public Functions
  * @{
  */
/** @defgroup MD_LCD_Public_Functions_Group2 CR
  * @{
  */
/**
  * @brief  Sets charge pump
  * @param  sel Value of following:      
            - 00：3.2V 
            - 01：3.8V 
            - 10：4.8V 
            - 11：5.4V
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_vchps(uint32_t sel)
{
	MODIFY_REG(LCD->CR, LCD_CR_VCHPS_MSK, sel << LCD_CR_VCHPS_POSS);
}

/**
  * @brief  Gets charge pump
  * @retval Value of following:
            - 0x0：3.2V 
            - 0x1：3.8V 
            - 0x2：4.8V 
            - 0x3：5.4V
  */
__STATIC_INLINE uint32_t md_lcd_get_vchps(void)
{
	return READ_BITS(LCD->CR, LCD_CR_VCHPS_MSK, LCD_CR_VCHPS_POSS);
}

/**
  * @brief  Sets lcd dsld
  * @param  sel Value of following:      
            - 0x0：min 
            - ...... 
            - 0xf：max
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_dsld(uint32_t sel)
{
	MODIFY_REG(LCD->CR, LCD_CR_DSLD_MSK, sel << LCD_CR_DSLD_POSS);
}

/**
  * @brief  Gets lcd dsld
  * @retval Value of following:
            - 0x0：min 
            - ...... 
            - 0x1: max
  */
__STATIC_INLINE uint32_t md_lcd_get_dsld(void)
{
	return READ_BITS(LCD->CR, LCD_CR_DSLD_MSK, LCD_CR_DSLD_POSS);
}


/**
  * @brief  Sets lcd dshd
  * @param  sel Value of following:      
            - 0x0：min 
            - ...... 
            - 0xf：max
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_dshd(uint32_t sel)
{
	MODIFY_REG(LCD->CR, LCD_CR_DSHD_MSK, sel << LCD_CR_DSHD_POSS);
}

/**
  * @brief  Gets lcd dshd
  * @retval Value of following:
            - 0x0：min 
            - ...... 
            - 0x1: max
  */
__STATIC_INLINE uint32_t md_lcd_get_dshd(void)
{
	return READ_BITS(LCD->CR, LCD_CR_DSHD_MSK, LCD_CR_DSHD_POSS);
}


/**
  * @brief  Enable VBUFLD. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_vbufld(void)
{
	SET_BIT(LCD->CR, LCD_CR_VBUFLD_MSK);
}

/**
  * @brief  Disable VBUFLD. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_vbufld(void)
{
	CLEAR_BIT(LCD->CR, LCD_CR_VBUFLD_MSK);
}

/**
  * @brief  Check VBUFLD. 
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_vbufld(void)
{
	return READ_BITS(LCD->CR, LCD_CR_VBUFLD_MSK, LCD_CR_VBUFLD_POS);
}

/**
  * @brief  Enable VBUFHD. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_vbufhd(void)
{
	SET_BIT(LCD->CR, LCD_CR_VBUFHD_MSK);
}

/**
  * @brief  Disable VBUFHD. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_vbufhd(void)
{
	CLEAR_BIT(LCD->CR, LCD_CR_VBUFHD_MSK);
}

/**
  * @brief  Check VBUFHD. 
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_vbufhd(void)
{
	return READ_BITS(LCD->CR, LCD_CR_VBUFHD_MSK, LCD_CR_VBUFHD_POS);
}


/**
  * @brief  Sets RESLD
  * @param  sel Value of following:      
            - 0x0：1MΩ 
            - 0x1：2MΩ 
            - 0x2：3MΩ
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_resld(uint32_t sel)
{
	MODIFY_REG(LCD->CR, LCD_CR_RESLD_MSK, sel << LCD_CR_RESLD_POSS);
}

/**
  * @brief  Gets RESLD
  * @retval Value of following:
            - 0x0：1MΩ 
            - 0x1：2MΩ 
            - 0x2：3MΩ
  */
__STATIC_INLINE uint32_t md_lcd_get_resld(void)
{
	return READ_BITS(LCD->CR, LCD_CR_RESLD_MSK, LCD_CR_RESLD_POSS);
}

/**
  * @brief  Sets RESHD
  * @param  sel Value of following:      
            - 0x0：1MΩ 
            - 0x1：2MΩ 
            - 0x2：3MΩ
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_reshd(uint32_t sel)
{
	MODIFY_REG(LCD->CR, LCD_CR_RESHD_MSK, sel << LCD_CR_RESHD_POSS);
}

/**
  * @brief  Gets RESHD
  * @retval Value of following:
            - 0x0：1MΩ 
            - 0x1：2MΩ 
            - 0x2：3MΩ
  */
__STATIC_INLINE uint32_t md_lcd_get_reshd(void)
{
	return READ_BITS(LCD->CR, LCD_CR_RESHD_MSK, LCD_CR_RESHD_POSS);
}

/**
  * @brief  Sets BIAS
  * @param  sel Value of following:      
            - 0x1：1/4 
            - 0x2：1/2 
            - 0x3：1/3
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_bias(uint32_t sel)
{
	MODIFY_REG(LCD->CR, LCD_CR_BIAS_MSK, sel << LCD_CR_BIAS_POSS);
}

/**
  * @brief  Gets BIAS
  * @retval Value of following:
            - 0x1：1/4 
            - 0x2：1/2 
            - 0x3：1/3
  */
__STATIC_INLINE uint32_t md_lcd_get_bias(void)
{
	return READ_BITS(LCD->CR, LCD_CR_BIAS_MSK, LCD_CR_BIAS_POSS);
}

/**
  * @brief  Sets DUTY
  * @param  sel Value of following:      
            - 0x0:（COM0） static
            - 0x1：1/2（COM0~COM1） 
            - 0x2：1/3（COM0~COM2） 
            - 0x3：1/4（COM0~COM3） 
            - 0x4：1/6（COM0~COM5） 
            - 0x5：1/8（COM0~COM7） 
            - others：reserved
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_duty(uint32_t sel)
{
	MODIFY_REG(LCD->CR, LCD_CR_DUTY_MSK, sel << LCD_CR_DUTY_POSS);
}

/**
  * @brief  Gets DUTY
  * @retval Value of following:
            - 0x0:（COM0） static
            - 0x1：1/2（COM0~COM1） 
            - 0x2：1/3（COM0~COM2） 
            - 0x3：1/4（COM0~COM3） 
            - 0x4：1/6（COM0~COM5） 
            - 0x5：1/8（COM0~COM7） 
            - others：reserved
  */
__STATIC_INLINE uint32_t md_lcd_get_duty(void)
{
	return READ_BITS(LCD->CR, LCD_CR_DUTY_MSK, LCD_CR_DUTY_POSS);
}



/**
  * @brief  Enable OE. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_oe(void)
{
	SET_BIT(LCD->CR, LCD_CR_OE_MSK);
}

/**
  * @brief  Disable OE. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_oe(void)
{
	CLEAR_BIT(LCD->CR, LCD_CR_OE_MSK);
}

/**
  * @brief  Check OE. 
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_oe(void)
{
	return READ_BITS(LCD->CR, LCD_CR_OE_MSK, LCD_CR_OE_POS);
}


/**
  * @brief  Sets VSEL
  * @param  sel Value of following:      
            - 0x0：VDD 
            - 0x1: internal charge pump output 
            - 0x2：VLCD input
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_vsel(uint32_t sel)
{
	MODIFY_REG(LCD->CR, LCD_CR_VSEL_MSK, sel << LCD_CR_VSEL_POSS);
}

/**
  * @brief  Gets VSEL
  * @retval Value of following:
            - 0x0：VDD 
            - 0x1: internal charge pump output 
            - 0x2：VLCD input
  */
__STATIC_INLINE uint32_t md_lcd_get_vsel(void)
{
	return READ_BITS(LCD->CR, LCD_CR_VSEL_MSK, LCD_CR_VSEL_POSS);
}


/**
  * @brief  Enable LCD. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_en(void)
{
	SET_BIT(LCD->CR, LCD_CR_EN_MSK);
}

/**
  * @brief  Disable LCD. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_en(void)
{
	CLEAR_BIT(LCD->CR, LCD_CR_EN_MSK);
}

/**
  * @brief  Check EN. 
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_en(void)
{
	return READ_BITS(LCD->CR, LCD_CR_EN_MSK, LCD_CR_EN_POS);
}
/**
  * @}
  */
/** @defgroup MD_LCD_Public_Functions_Group3 FCR
  * @{
  */
/**
  * @brief Set WFS 
  * @param sel Input select:
  *         - 0：A wave 
  *         - 1：B wave
  * @retval None  
  */
__STATIC_INLINE uint32_t md_lcd_set_active_wfs(uint32_t sel)
{
	return READ_BITS(LCD->FCR, LCD_FCR_WFS_MSK, sel << LCD_FCR_WFS_POS);
}

/**
  * @brief  Get WFS 
  * @retval input select:
  *         - 0：A wave 
  *         - 1：B wave 
  */
__STATIC_INLINE uint32_t md_lcd_get_active_wfs(void)
{
	return READ_BITS(LCD->FCR, LCD_FCR_WFS_MSK, LCD_FCR_WFS_POS);
}

/**
  * @brief  Sets PRS
  * @param  sel Value of following:      
  *         - 0x0：FLCDCLK 
  *         - 0x1：FLCDCLK/2 
  *         - 0x2：FLCDCLK/4 
  *         - ......
  *         - 0xf：FLCDCLK/32768
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_PRS(uint32_t sel)
{
	MODIFY_REG(LCD->FCR, LCD_FCR_PRS_MSK, sel << LCD_FCR_PRS_POSS);
}

/**
  * @brief  Gets PRS
  * @retval Value of following:
  *         - 0x0：FLCDCLK 
  *         - 0x1：FLCDCLK/2 
  *         - 0x2：FLCDCLK/4 
  *         - ......
  *         - 0xf：FLCDCLK/32768
  */
__STATIC_INLINE uint32_t md_pis_get_PRS(void)
{
	return READ_BITS(LCD->FCR, LCD_FCR_PRS_MSK, LCD_FCR_PRS_POSS);
}

/**
  * @brief  Sets DIV
  * @param  sel Value of following:      
  *         - 0x0：16 
  *         - 0x1：17 
  *         - 0x2：18 
  *         - ......
  *         - 0xf：31
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_div(uint32_t sel)
{
	MODIFY_REG(LCD->FCR, LCD_FCR_DIV_MSK, sel << LCD_FCR_DIV_POSS);
}

/**
  * @brief  Gets DIV
  * @retval Value of following:
  *         - 0x0：16 
  *         - 0x1：17 
  *         - 0x2：18 
  *         - ......
  *         - 0xf：31
  */
__STATIC_INLINE uint32_t md_lcd_get_div(void)
{
	return READ_BITS(LCD->FCR, LCD_FCR_DIV_MSK, LCD_FCR_DIV_POSS);
}

/**
  * @brief  Sets BLMOD
  * @param  sel Value of following:      
  *         - 0x0：disable
  *         - 0x1：SEG0，COM0 blink（single pixel） 
  *         - 0x2：SEG0，COMx2 blink（8 pixel max） 
  *         - 0x3：SEGx，COMx blink（all pixel）
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_blmod(uint32_t sel)
{
	MODIFY_REG(LCD->FCR, LCD_FCR_BLMOD_MSK, sel << LCD_FCR_BLMOD_POSS);
}

/**
  * @brief  Gets BLMOD
  * @retval Value of following:
  *         - 0x0：disable
  *         - 0x1：SEG0，COM0 blink（single pixel） 
  *         - 0x2：SEG0，COMx2 blink（8 pixel max） 
  *         - 0x3：SEGx，COMx blink（all pixel）
  */
__STATIC_INLINE uint32_t md_pis_get_blmod(void)
{
	return READ_BITS(LCD->FCR, LCD_FCR_BLMOD_MSK, LCD_FCR_BLMOD_POSS);
}

/**
  * @brief  Sets BLFRQ
  * @param  sel Value of following:      
  *         - 0x0：FDIVCLK/8 
  *         - 0x1：FDIVCLK /16 
  *         - 0x2：FDIVCLK /32 
  *         - 0x3：FDIVCLK /64 
  *         - 0x4：FDIVCLK /128 
  *         - 0x5：FDIVCLK /256 
  *         - 0x6：FDIVCLK /512 
  *         - 0x7：FDIVCLK /1024
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_blfrq(uint32_t sel)
{
	MODIFY_REG(LCD->FCR, LCD_FCR_BLFRQ_MSK, sel << LCD_FCR_BLFRQ_POSS);
}

/**
  * @brief  Gets BLFRQ
  * @retval Value of following:
  *         - 0x0：FDIVCLK/8 
  *         - 0x1：FDIVCLK /16 
  *         - 0x2：FDIVCLK /32 
  *         - 0x3：FDIVCLK /64 
  *         - 0x4：FDIVCLK /128 
  *         - 0x5：FDIVCLK /256 
  *         - 0x6：FDIVCLK /512 
  *         - 0x7：FDIVCLK /1024
  */
__STATIC_INLINE uint32_t md_lcd_get_blfrq(void)
{
	return READ_BITS(LCD->FCR, LCD_FCR_BLFRQ_MSK, LCD_FCR_BLFRQ_POSS);
}


/**
  * @brief  Sets DEAD
  * @param  sel Value of following:      
  *         - 0x0：no dead time 
  *         - 0x1：1 DIVCLK clock duration 
  *         - 0x2：2 DIVCLK clock duration
  *         - ......
  *         - 0x7：7 DIVCLK clock duration
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_dead(uint32_t sel)
{
	MODIFY_REG(LCD->FCR, LCD_FCR_DEAD_MSK, sel << LCD_FCR_DEAD_POSS);
}

/**
  * @brief  Gets DEAD
  * @retval Value of following:
  *         - 0x0：no dead time 
  *         - 0x1：1 DIVCLK clock duration 
  *         - 0x2：2 DIVCLK clock duration
  *         - ......
  *         - 0x7：7 DIVCLK clock duration
  */
__STATIC_INLINE uint32_t md_pis_get_dead(void)
{
	return READ_BITS(LCD->FCR, LCD_FCR_DEAD_MSK, LCD_FCR_DEAD_POSS);
}


/**
  * @brief  Enable HD. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_hd(void)
{
	SET_BIT(LCD->FCR, LCD_FCR_HD_MSK);
}

/**
  * @brief  Disable HD. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_hd(void)
{
	CLEAR_BIT(LCD->FCR, LCD_FCR_HD_MSK);
}

/**
  * @brief  Check HD. 
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_hd(void)
{
	return READ_BITS(LCD->FCR, LCD_FCR_HD_MSK, LCD_FCR_HD_POS);
}


/**
  * @brief  Sets PON
  * @param  sel Value of following:      
  *         - 0x: no last time 
  *         - 0x1：1 LCD prescaler clock 
  *         - 0x2：2 LCD prescaler clock
  *         - ......
  *         - 111：7 LCD prescaler clock
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_pon(uint32_t sel)
{
	MODIFY_REG(LCD->FCR, LCD_FCR_PON_MSK, sel << LCD_FCR_PON_POSS);
}

/**
  * @brief  Gets PON
  * @retval Value of following:
  *         - 0x: no last time 
  *         - 0x1：1 LCD prescaler clock 
  *         - 0x2：2 LCD prescaler clock
  *         - ......
  *         - 111：7 LCD prescaler clock
  */
__STATIC_INLINE uint32_t md_pis_get_pon(void)
{
	return READ_BITS(LCD->FCR, LCD_FCR_PON_MSK, LCD_FCR_PON_POSS);
}

/**
  * @brief  Sets VGS
  * @param  sel Value of following:      
  *         - 0x0：VLCD*(45/75) 
  *         - 0x1：VLCD*(47/75) 
  *         - 0x2：VLCD*(49/75) 
  *         - ......
  *         - 0xd：VLCD*(71/75) 
  *         - 0xe：VLCD*(73/75)
  *         - 0xf: VLCD
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_vgs(uint32_t sel)
{
	MODIFY_REG(LCD->FCR, LCD_FCR_VGS_MSK, sel << LCD_FCR_VGS_POSS);
}

/**
  * @brief  Gets VGS
  * @retval Value of following:
  *         - 0x0：VLCD*(45/75) 
  *         - 0x1：VLCD*(47/75) 
  *         - 0x2：VLCD*(49/75) 
  *         - ......
  *         - 0xd：VLCD*(71/75) 
  *         - 0xe：VLCD*(73/75)
  *         - 0xf: VLCD
  */
__STATIC_INLINE uint32_t md_pis_get_vgs(void)
{
	return READ_BITS(LCD->FCR, LCD_FCR_VGS_MSK, LCD_FCR_VGS_POSS);
}
/**
  * @}
  */
/** @defgroup MD_LCD_Public_Functions_Group4 SEGCR0
  * @{
  */
/**
  * @brief  Set SEGCR0 value
  * @param  sel Value of following:      
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_segcr0_value(uint32_t sel)
{
	MODIFY_REG(LCD->SEGCR0, LCD_SEGCR0_SEG_OE_MSK,  sel << LCD_SEGCR0_SEG_OE_POSS);
}

/**
  * @brief  Get SEGCR0 value. 
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_segcr0_value(void)
{
	return READ_BITS(LCD->SEGCR0, LCD_SEGCR0_SEG_OE_MSK, LCD_SEGCR0_SEG_OE_POSS);
}
/**
  * @}
  */
/** @defgroup MD_LCD_Public_Functions_Group5 SEGCR1
  * @{
  */
/**
  * @brief  Set SEGCR1 value
  * @param  sel Value of following:      
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_segcr1_value(uint32_t sel)
{
	MODIFY_REG(LCD->SEGCR1, LCD_SEGCR1_SEG_OE_MSK,  sel << LCD_SEGCR1_SEG_OE_POSS);
}

/**
  * @brief  Get SEGCR1 value. 
  * @retval None
  */
__STATIC_INLINE uint32_t md_lcd_get_segcr1_value(void)
{
	return READ_BITS(LCD->SEGCR1, LCD_SEGCR1_SEG_OE_MSK, LCD_SEGCR1_SEG_OE_POSS);
}
/**
  * @}
  */
/** @defgroup MD_LCD_Public_Functions_Group6 IE
  * @{
  */
/**
  * @brief  Enable UDDIE. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_uddie(void)
{
	SET_BIT(LCD->IE, LCD_IE_UDDIE_MSK);
}

/**
  * @brief  Disable UDDIE. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_uddie(void)
{
	CLEAR_BIT(LCD->IE, LCD_IE_UDDIE_MSK);
}

/**
  * @brief  Check UDDIE. 
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_uddie(void)
{
	return READ_BITS(LCD->IE, LCD_IE_UDDIE_MSK, LCD_IE_UDDIE_POS);
}

/**
  * @brief  Enable SOFIE. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_enable_sofie(void)
{
	SET_BIT(LCD->IE, LCD_IE_SOFIE_MSK);
}

/**
  * @brief  Disable SOFIE. 
  * @retval None
  */
__STATIC_INLINE void md_lcd_disable_sofie(void)
{
	CLEAR_BIT(LCD->IE, LCD_IE_SOFIE_MSK);
}

/**
  * @brief  Check SOFIE. 
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_sofie(void)
{
	return READ_BITS(LCD->IE, LCD_IE_SOFIE_MSK, LCD_IE_SOFIE_POS);
}
/**
  * @}
  */
/** @defgroup MD_LCD_Public_Functions_Group7 IF
  * @{
  */
/**
  * @brief  Get UDDIF
  * @retval input select:
  *         - 0：on-going or cleared
  *         - 1：completely
  */
__STATIC_INLINE uint32_t md_lcd_get_active_uddif(void)
{
	return READ_BITS(LCD->IF, LCD_IF_UDDIF_MSK, LCD_IF_UDDIF_POS);
}

/**
  * @brief  Get SOFIF
  * @retval input select:
  *         - 0：on-going or cleared
  *         - 1：completely
  */
__STATIC_INLINE uint32_t md_lcd_get_active_sofif(void)
{
	return READ_BITS(LCD->IF, LCD_IF_SOFIF_MSK, LCD_IF_SOFIF_POS);
}
/**
  * @}
  */
/** @defgroup MD_LCD_Public_Functions_Group8 IFCR
  * @{
  */
/**
  * @brief  Sets UDDIFC
  * @param  sel Value of following:      
  *         - 0: no action
  *         - 1：display refreshed interrupt clear
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_uddifc(uint32_t sel)
{
	MODIFY_REG(LCD->IFCR, LCD_IFCR_UDDIFC_MSK, sel << LCD_IFCR_UDDIFC_POS);
}

/**
  * @brief  Sets SOFIFC
  * @param  sel Value of following:      
  *         - 0：no action
  *         - 1：frame start interrupt clear
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_sofifc(uint32_t sel)
{
	MODIFY_REG(LCD->IFCR, LCD_IFCR_SOFIFC_MSK, sel << LCD_IFCR_SOFIFC_POS);
}
/**
  * @}
  */
/** @defgroup MD_LCD_Public_Functions_Group9 SR
  * @{
  */
/**
  * @brief  Get FCRSF
  * @retval input select:
  *         - 0: syncronization finished 
  *         - 1: syncronization on going
  */
__STATIC_INLINE uint32_t md_lcd_get_active_fcrsf(void)
{
	return READ_BITS(LCD->SR, LCD_SR_FCRSF_MSK, LCD_SR_FCRSF_POS);
}

/**
  * @brief  Get UDR
  * @retval input select:
  *         - 0：no need to display refresh 
  *         - 1：need to display refresh
  */
__STATIC_INLINE uint32_t md_lcd_get_active_udr(void)
{
	return READ_BITS(LCD->SR, LCD_SR_UDR_MSK, LCD_SR_UDR_POS);
}

/**
  * @brief  Get ENS. 
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_lcd_enable_ens(void)
{
	return READ_BITS(LCD->SR, LCD_SR_ENS_MSK, LCD_SR_ENS_POS);
}

/**
  * @brief  Check SOFIE. 
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_lcd_is_enabled_ens(void)
{
	return READ_BITS(LCD->SR, LCD_SR_ENS_MSK, LCD_SR_ENS_POS);
}

/**
  * @brief  Get RDY
  * @retval input select:
  *         - 0：not enabled or not steady
  *         - 1：enabled and steady
  */
__STATIC_INLINE uint32_t md_lcd_get_active_rdy(void)
{
	return READ_BITS(LCD->SR, LCD_SR_RDY_MSK, LCD_SR_RDY_POS);
}
/**
  * @}
  */
/** @defgroup MD_LCD_Public_Functions_Group10 SEG_DATA
  * @{
  */
/**
  * @brief  Sets SEG_DATA
  * @param  bufn Seg buffer number, 0x0 ~ 0xf
  * @param  val  Value of seg buf      
  * @retval None
  */
__STATIC_INLINE void md_lcd_set_seg_buf_data(uint8_t bufn, uint32_t val)
{
	MODIFY_REG(LCD->BUF[bufn], LCD_BUF_SEG_DATA_MSK, val << LCD_BUF_SEG_DATA_POSS);
}

/**
  * @brief  Gets SEG_DATA
  * @param  bufn Seg buffer number, 0x0 ~ 0xf
  * @retval val  Value of seg buf
  */
__STATIC_INLINE uint32_t md_lcd_get_seg_buf_data(uint8_t bufn)
{
	return READ_BITS(LCD->BUF[bufn], LCD_BUF_SEG_DATA_MSK, LCD_BUF_SEG_DATA_POSS);
}
/**
  * @}
  */
/** @defgroup MD_LCD_Public_Functions_Group1 Initialization
  * @{
  */
extern md_status_t md_lcd_init(LCD_TypeDef *lcd, md_lcd_init_t *init);
extern void md_lcd_struct_init(md_lcd_init_t *init);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif
#endif
