/**********************************************************************************
 *
 * @file    md_pis.c
 * @brief   PIS module driver.
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "md_pis.h"

/** @addtogroup Micro_Driver
  * @{
  */
//#ifdef MD_PIS

/** @addtogroup MD_PIS
  * @{
  */

/** @addtogroup MD_PIS_Public_Functions
  * @{
  */
/** @addtogroup MD_PIS_Public_Functions_Group1
  * @{
  */
/**
  * @brief  Reset the PIS register
  * @retval None
  */
void md_pis_reset(void)
{
    uint32_t i;

    for (i = 0; i < 6; ++i)
        WRITE_REG(PIS->CH_CON[i], 0x0);

    WRITE_REG(PIS->CH_OER, 0x0);
    WRITE_REG(PIS->TAR_CON0, 0x0);
    WRITE_REG(PIS->TAR_CON1, 0x0);
    WRITE_REG(PIS->TXMCR, 0x0);

    return;
}

/**
  * @brief  Initializes the PIS mode according to the specified parameters in
  *         the md_pis_init_t.
  * @param  init: Pointer to a md_pis_init_t structure that contains
  *         the configuration information for the specified PIS module.
  * @retval Status, see @ref md_status_t.
  */
md_status_t md_pis_init(md_pis_init_t *init)
{
    uint8_t ch, c_idx, shift;

    if (init == 0)
        return MD_ERROR;

    ch    = init->c_trig & 0xF;
    c_idx = (init->c_trig >> 4) & 0xF;
    shift = (init->c_trig >> 8) & 0xFF;

    MODIFY_REG(PIS->CH_CON[ch], PIS_CH0_CON_SRCS_MSK, ((init->p_src) >> 4) << PIS_CH0_CON_SRCS_POSS);
    MODIFY_REG(PIS->CH_CON[ch], PIS_CH0_CON_MSIGS_MSK, ((init->p_src) & 0xF) << PIS_CH0_CON_MSIGS_POSS);

    switch (c_idx)
    {
        case 0:
            PIS->TAR_CON0 |= (1 << shift);
            break;

        case 1:
            PIS->TAR_CON1 |= (1 << shift);
            break;

        default:
            break;
    }

    return MD_OK;
}
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
//#endif
/**
 * @}
 */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
