/**********************************************************************************
 *
 * @file    es32f0943.h
 * @brief   REG Head File
 *
 * @date    20 Jul 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Jul 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/** @addtogroup CMSIS
  * @{
  */

/** @addtogroup
  * @{
  */

#ifndef __ES32F0943_H__
#define __ES32F0943_H__

#ifdef __cplusplus
extern "C" {
#endif

/**
  * @brief defines 'read only' 、'write only'、 'read / write' permissions.
  */
#define __I   volatile const  /* defines 'read only' permissions */
#define __O   volatile        /* defines 'write only' permissions */
#define __IO  volatile        /* defines 'read / write' permissions */

/** @addtogroup Configuration_section_for_CMSIS
  * @{
  */
/**
  * @brief configuration of the cortex-M3 processor and core peripherals.
  */

#define __MPU_PRESENT           0       /*!< cm0ikmcu does not provide a MPU present or not       */
#define __NVIC_PRIO_BITS        2       /*!< cm0ikmcu Supports 2 Bits for the Priority Levels     */
#define __Vendor_SysTickConfig  0       /*!< Set to 1 if different SysTick Config is used         */
/**
  * @}
  */

/** @addtogroup peripherals_interrupt_number_definition
  * @{
  */

typedef enum IRQn
{
    /******  Cortex-M0 Processor Exceptions Numbers *************************************************/
    NonMaskableInt_IRQn   = -14,      /*!< 2 Non Maskable Interrupt                               */
    HardFault_IRQn        = -13,      /*!< 3 Cortex-M0 Hard Fault Interrupt                       */
    SVCall_IRQn           = -5,       /*!< 11 Cortex-M0 SV Call Interrupt                         */
    PendSV_IRQn           = -2,       /*!< 14 Cortex-M0 Pend SV Interrupt                         */
    SysTick_IRQn          = -1,       /*!< 15 Cortex-M0 System Tick Interrupt                     */

    /******  ES32F0943 Cortex-M0 specific Interrupt Numbers *****************************************/
    WWDT_IRQn             = 0,        /*  0       | WWDT          Interrupt                       */
    LVD_IRQn              = 1,        /*  1       | LVD           Interrupt   EXTI[20]            */
    RTC_IRQn              = 2,        /*  2       | RTC           Interrupt                       */
    WAKEUP_IRQn           = 3,        /*  3       | WAKEUP        Interrupt   EXTI[21]            */
    RCU_IRQn              = 4,        /*  4       | RCU           Interrupt                       */
    EXTI_0to1_IRQn        = 5,        /*  5       | EXTI[1:0]     Interrupt   GPIOx[1:0]          */
    EXTI_2to3_IRQn        = 6,        /*  6       | EXTI[3:2]     Interrupt   GPIOx[3:2]          */
    EXTI_4to15_IRQn       = 7,        /*  7       | EXTI[15:4]    Interrupt   GPIOx[15:4]         */
    CM0IKMCU_IRQ8_IRQn    = 8,        /*  8       | Reserved      Interrupt                       */
    CM0IKMCU_IRQ9_IRQn    = 9,        /*  9       | Reserved      Interrupt                       */
    LCD_IRQn              = 10,       /*  10      | LCD           Interrupt                       */
    ADC_IRQn              = 11,       /*  11      | ADC           Interrupt                       */
    COMP_IRQn             = 12,       /*  12      | COMP          Interrupt                       */
    OPAMP_IRQn            = 13,       /*  13      | OPAMP         Interrupt                       */
    BS16T1_IRQn           = 14,       /*  14      | BS16T1        Interrupt                       */
    GP32C4T1_IRQn         = 15,       /*  15      | GP32C4T1      Interrupt                       */
    CM0IKMCU_IRQ16_IRQn   = 16,       /*  16      | Reserved      Interrupt                       */
    CM0IKMCU_IRQ17_IRQn   = 17,       /*  17      | Reserved      Interrupt                       */
    CM0IKMCU_IRQ18_IRQn   = 18,       /*  18      | Reserved      Interrupt                       */
    GP16C2T1_IRQn         = 19,       /*  19      | GP16C2T1      Interrupt                       */
    GP16C2T2_IRQn         = 20,       /*  20      | GP16C2T2      Interrupt                       */
    GP16C2T3_IRQn         = 21,       /*  21      | GP16C2T3      Interrupt                       */
    GP16C2T4_IRQn         = 22,       /*  22      | GP16C2T4      Interrupt                       */
    I2C1_IRQn             = 23,       /*  23      | I2C1          Interrupt                       */
    CM0IKMCU_IRQ24_IRQn   = 24,       /*  24      | Reserved      Interrupt                       */
    SPI1_IRQn             = 25,       /*  25      | SPI1          Interrupt                       */
    CM0IKMCU_IRQ26_IRQn   = 26,       /*  26      | Reserved      Interrupt                       */
    UART1_IRQn            = 27,       /*  27      | UART1         Interrupt                       */
    UART2_IRQn            = 28,       /*  28      | UART2         Interrupt                       */
    UART3_IRQn            = 29,       /*  29      | UART3         Interrupt                       */
    UART4_IRQn            = 30,       /*  30      | UART4         Interrupt                       */
    CM0IKMCU_IRQ31_IRQn   = 31        /*  31      | Reserved      Interrupt                       */
} IRQn_Type;

/**
  * @}
  */

/* Includes-------------------------------------------------------------------------------*/

#include "core_cm0.h"
#include <stdint.h>
#include <stdbool.h>
#include "system_es32f0943.h"
/**
  * @brief these files declare Peripheral register bit_definition.
  */
#include "reg_gpio.h"
#include "reg_calc.h"
#include "reg_rtc.h"

#include "reg_exti.h"
#include "reg_uart.h"
#include "reg_spi.h"
#include "reg_i2c.h"
#include "reg_timer.h"
#include "reg_wwdt.h"
#include "reg_iwdt.h"
#include "reg_lcd.h"
#include "reg_anpwr.h"
#include "reg_adc.h"
#include "reg_opamp.h"
#include "reg_cmp.h"

#include "reg_rcu.h"
#include "reg_syscfg.h"
#include "reg_fc.h"
#include "reg_tick.h"

/******************************************************************************/
/*                              Peripheral register                           */
/******************************************************************************/

/*   allow anonymous structures and anonymous enumeration    */
/* #pragma anon_unions */

/******************************************************************************/
/*                         Peripheral memory map                              */
/******************************************************************************/
#define CM0IN_BASE          ((     uint32_t)0xE0000000) // Cortex®-M0 internal Peripherals
#define AHB3_BASE           ((     uint32_t)0x90000000) // AHB3 Peripheral
#define AHB2_BASE           ((     uint32_t)0x48000000) // AHB2 Peripheral
#define AHB1_BASE           ((     uint32_t)0x40020000) // AHB1 Peripheral
#define APB2_BASE           ((     uint32_t)0x40010000) // APB2 Peripheral
#define APB1_BASE           ((     uint32_t)0x40000000) // APB1 Peripheral
#define SRAM_BASE           ((     uint32_t)0x20000000) // SRAM Memory  (128K Bytes)
#define FLASH_INFO_BASE     ((     uint32_t)0x1FFFE000) // EFLASH Info  (  4K Bytes) - Boot Loader Memory
#define FLASH_BASE          ((     uint32_t)0x08000000) // EFLASH Main  (512K Bytes)

/* Cortex®-M0 internal Peripherals - CM0IN_BASE: 0xE0000000*/
#define TICK_BASE           (CM0IN_BASE     + 0xE010)

#define SRAM1_BASE          (SRAM_BASE      + 0x00000)   // SRAM1 Memory (4K Bytes)
#define SRAM2_BASE          (SRAM_BASE      + 0x01000)   // SRAM2 Memory (4K Bytes)
#define STACK_SIZE           0x00000400>>2              // Stack size (in Words)

/* AHB3 Peripheral memory map - AHB3_BASE: 0x90000000*/
#define EXSTORE_BASE        (AHB3_BASE      + 0x0000)
/* AHB2 Peripheral memory map - AHB2_BASE: 0x48000000*/
#define GPIOA_BASE          (AHB2_BASE      + 0x0000)
#define GPIOB_BASE          (AHB2_BASE      + 0x0400)
#define GPIOC_BASE          (AHB2_BASE      + 0x0800)
#define GPIOD_BASE          (AHB2_BASE      + 0x0C00)
#define GPIOE_BASE          (AHB2_BASE      + 0x1000)
#define GPIOF_BASE          (AHB2_BASE      + 0x1400)
#define MBIST_BASE          (AHB2_BASE      + 0x3C00)
/* AHB1 Peripheral memory map - AHB1_BASE: 0x40020000*/
#define RCU_BASE            (AHB1_BASE      + 0x1000)
#define SYSCFG_BASE         (AHB1_BASE      + 0x1400)
#define OPT_BASE            (AHB1_BASE      + 0x14F0)
#define RTC_BASE            (AHB1_BASE      + 0x1800)  // 
#define EFLASH_BASE         (AHB1_BASE      + 0x2000)
#define CALC_BASE           (AHB1_BASE      + 0x3C00)
/* APB2 Peripheral memory map - APB2_BASE: 0x40010000*/
#define EXTI_BASE           (APB2_BASE      + 0x0400)
#define LCD_BASE            (APB2_BASE      + 0x0800)
#define ADC_BASE            (APB2_BASE      + 0x0C00)
#define ANPWR_BASE          (APB2_BASE      + 0x1000)
#define OPAMP_BASE          (APB2_BASE      + 0x1400)
#define SPI1_BASE           (APB2_BASE      + 0x3000)
#define UART1_BASE          (APB2_BASE      + 0x3800)
#define GP16C2T1_BASE       (APB2_BASE      + 0x4000)
#define GP16C2T2_BASE       (APB2_BASE      + 0x4400)
#define GP16C2T3_BASE       (APB2_BASE      + 0x4800)
#define GP16C2T4_BASE       (APB2_BASE      + 0x4C00)
#define DBGMCU_BASE         (APB2_BASE      + 0x5800)
#define CMP_BASE            (APB2_BASE      + 0x5C00)
/* APB1 Peripheral memory map - APB1_BASE: 0x40000000*/
#define GP32C4T1_BASE       (APB1_BASE      + 0x0000)
#define BS16T1_BASE         (APB1_BASE      + 0x1000)
#define WWDT_BASE           (APB1_BASE      + 0x2C00)
#define IWDT_BASE           (APB1_BASE      + 0x3000)
#define UART2_BASE          (APB1_BASE      + 0x4400)
#define UART3_BASE          (APB1_BASE      + 0x4800)
#define UART4_BASE          (APB1_BASE      + 0x4C00)
#define I2C1_BASE           (APB1_BASE      + 0x5400)
/******************************************************************************/
/*                         Peripheral declaration                             */
/******************************************************************************/
#define TICK                ((  SYST_TypeDef *)     TICK_BASE)
#define GPIOA               ((  GPIO_TypeDef *)    GPIOA_BASE)
#define GPIOB               ((  GPIO_TypeDef *)    GPIOB_BASE)
#define GPIOC               ((  GPIO_TypeDef *)    GPIOC_BASE)
#define GPIOD               ((  GPIO_TypeDef *)    GPIOD_BASE)
#define MBIST               (( MBIST_TypeDef *)    MBIST_BASE)
#define RCU                 ((   RCU_TypeDef *)      RCU_BASE)
#define SYSCFG              ((SYSCFG_TypeDef *)   SYSCFG_BASE)
#define OPT                 ((   OPT_TypeDef *)      OPT_BASE)
#define RTC                 ((   RTC_TypeDef *)      RTC_BASE)
#define FC                  ((    FC_TypeDef *)   EFLASH_BASE)
#define CALC                ((  CALC_TypeDef *)     CALC_BASE)
#define IWDT                ((  IWDT_TypeDef *)     IWDT_BASE)
#define WWDT                ((  WWDT_TypeDef *)     WWDT_BASE)
#define GP32C4T1            (( TIMER_TypeDef *) GP32C4T1_BASE)
#define GP16C2T1            (( TIMER_TypeDef *) GP16C2T1_BASE)
#define GP16C2T2            (( TIMER_TypeDef *) GP16C2T2_BASE)
#define GP16C2T3            (( TIMER_TypeDef *) GP16C2T3_BASE)
#define GP16C2T4            (( TIMER_TypeDef *) GP16C2T4_BASE)
#define BS16T1              (( TIMER_TypeDef *)   BS16T1_BASE)
#define SPI1                ((  SPI_TypeDef *)      SPI1_BASE)
#define I2C1                ((   I2C_TypeDef *)     I2C1_BASE)
#define UART1               ((  UART_TypeDef *)    UART1_BASE)
#define UART2               ((  UART_TypeDef *)    UART2_BASE)
#define UART3               ((  UART_TypeDef *)    UART3_BASE)
#define UART4               ((  UART_TypeDef *)    UART4_BASE)
#define EXTI                ((  EXTI_TypeDef *)     EXTI_BASE)
#define CMP                 ((   CMP_TypeDef *)      CMP_BASE)
#define LCD                 ((   LCD_TypeDef *)      LCD_BASE)
#define ADC                 ((   ADC_TypeDef *)      ADC_BASE)
#define ANPWR               (( ANPWR_TypeDef *)    ANPWR_BASE)
#define OPAMP               (( OPAMP_TypeDef *)    OPAMP_BASE)
/*   Special stuff */
#define MSG_INCR            ( __IO uint8_t  *) (DBGMCU_BASE + 0x50)
#define MSG_WR              ( __IO uint32_t *) (DBGMCU_BASE + 0x54)
#define MSG_END             ( __IO uint32_t *) (DBGMCU_BASE + 0x58)
#define MSG_INCR_S          ( __IO uint32_t *) (DBGMCU_BASE + 0x5C)
#define MSG_INCR_X          ( __IO uint32_t *) (DBGMCU_BASE + 0x60)
#define MSG_INCR_B          ( __IO uint32_t *) (DBGMCU_BASE + 0x64)
#define MSG_PTR             ( __IO uint8_t  *) (DBGMCU_BASE + 0x68)
#define MSG_INCR_D          ( __IO uint32_t *) (DBGMCU_BASE + 0x78)
#define INCR_ERR_CNT        ( __IO uint32_t *) (DBGMCU_BASE + 0x6C)

#define END_SIM     *MSG_END
#define SIM_FAIL    0xFFFFAAAA
#define SIM_PASS    0xFFFF5555
/******************************************************************************/
/*                                          macros                            */
/******************************************************************************/

/** @addtogroup Public_macros
  * @{
  */
#if defined(__CC_ARM) || (defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050))
#define __INLINE__  __inline
#else
#define __INLINE__  inline
#endif

#define __isr__

#define UNUSED(x) ((void)(x))

#ifdef USE_ASSERT
#define assert_param(x)   \
    do                        \
    {                         \
        if (!(x))               \
        {                       \
            __disable_irq();      \
            while (1)             \
                ;                     \
        }                       \
    } while (0)
#else
#define assert_param(x)
#endif

#define BIT(x)                   ((uint32_t)((uint32_t)0x01U<<(x)))

#define BITS(start, end)         ((0xFFFFFFFFUL << (start)) &\
                                  (0xFFFFFFFFUL >> (31U - (uint32_t)(end))))

#define SET_BIT(REG, SETMASK)     ((REG) |= (SETMASK))

#define CLEAR_BIT(REG, SETMASK)   ((REG) &= ~(SETMASK))

#define READ_BIT(REG, SETMASK)    ((REG) & (SETMASK))

#define CLEAR_REG(REG)            ((REG) = (0x0))

#define WRITE_REG(REG, VAL)       ((REG) = (VAL))

#define READ_REG(REG)             ((REG))

#define MODIFY_REG(REG, CLEARMASK, SETMASK)  WRITE_REG((REG), (((READ_REG(REG)) & (~(CLEARMASK))) | (SETMASK)))

#define POSITION_VAL(VAL)         (__CLZ(__RBIT(VAL)))

#define BYTE0(x)                  (uint8_t)(((x)>>0)&0xff)

#define BYTE1(x)                  (uint8_t)(((x)>>8)&0xff)

#define BYTE2(x)                  (uint8_t)(((x)>>16)&0xff)

#define BYTE3(x)                  (uint8_t)(((x)>>24)&0xff)

#define DATA16(d1,d0)             (uint16_t)(((uint16_t)(d1)<<8)|(d0))

#define DATA32(d3,d2,d1,d0)       (uint32_t)(((uint32_t)(d3)<<24)|((uint32_t)(d2)<<16)|((uint32_t)(d1)<<8)|(d0))

/**
  * @}
  */

/******************************************************************************/
/*                                types                                       */
/******************************************************************************/
/** @addtogroup Public_types
  * @{
  */
typedef enum
{
    DISABLE = 0,
    ENABLE = !DISABLE
} TypeFunc, FuncState;
#define IS_FUNC_STATE(x) (((x) == DISABLE) || ((x) == ENABLE))

typedef enum
{
    RESET = 0,
    SET = !RESET
} FlagStatus, ITStatus, PinStatus;

typedef enum
{
    ERROR = 0,
    SUCCESS = !ERROR
} ErrorStatus;

typedef enum
{
    BIT_RESET = 0x0,
    BIT_SET   = 0x1,
} BitStatus;

typedef enum
{
    FALSE = 0x0,
    TRUE  = 0x1,
} TypeBool;

typedef enum
{
    UNLOCK = 0x0,
    LOCK   = 0x1,
} LockState;
#define IS_LOCK_STATE(x) (((x) == UNLOCK) || ((x) == LOCK))

/**
  * @} Public_types
  */

#ifdef __cplusplus
}
#endif

#endif

/**
  * @}
  */

/**
  * @} CMSIS
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
