/**********************************************************************************
 *
 * @file    es32f0930.h
 * @brief   REG Head File
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/** @addtogroup CMSIS
  * @{
  */

/** @addtogroup es32f0930
  * @{
  */

#ifndef __ES32F0930_H__
#define __ES32F0930_H__

#ifdef __cplusplus
extern "C" {
#endif

/**
  * @brief defines 'read only' 、'write only'、 'read / write' permissions.
  */
#define __I   volatile const  /* defines 'read only' permissions */
#define __O   volatile        /* defines 'write only' permissions */
#define __IO  volatile        /* defines 'read / write' permissions */

/** @addtogroup Configuration_section_for_CMSIS
  * @{
  */
/**
  * @brief configuration of the cortex-M3 processor and core peripherals.
  */

#define __MPU_PRESENT           0       /*!< cm0ikmcu does not provide a MPU present or not       */
#define __NVIC_PRIO_BITS        2       /*!< cm0ikmcu Supports 2 Bits for the Priority Levels     */
#define __Vendor_SysTickConfig  0       /*!< Set to 1 if different SysTick Config is used         */
/**
  * @}
  */

/** @addtogroup peripherals_interrupt_number_definition
  * @{
  */

typedef enum IRQn
{
    /****** Cortex-M0 Processor Exceptions Numbers ******************************************************/
    RST_IRQn              = -15,
    NonMaskableInt_IRQn   = -14,      /*!< 2 Non Maskable Interrupt                               */
    HardFault_IRQn        = -13,      /*!< 3 Cortex-M0 Hard Fault Interrupt                       */
    SVCall_IRQn           = -5,       /*!< 11 Cortex-M0 SV Call Interrupt                         */
    PendSV_IRQn           = -2,       /*!< 14 Cortex-M0 Pend SV Interrupt                         */
    SysTick_IRQn          = -1,       /*!< 15 Cortex-M0 System Tick Interrupt                     */

    /****************        ES32F0930 specific Interrupt Numbers        **************************/
    WWDT_IRQn             = 0,        /*  0       | WWDT          Interrupt                       */
    PVD_IRQn              = 1,        /*  1       | PVD           Interrupt   EXTI[20]            */
    CM0IKMCU_IRQ02_IRQn   = 2,        /*  2       |               Interrupt                       */
    WAKEUP_IRQn           = 3,        /*  3       | WAKEUP        Interrupt   EXTI[21]            */
    RCU_IRQn              = 4,        /*  4       | RCU           Interrupt                       */
    EXTI_0to1_IRQn        = 5,        /*  5       | EXTI[1:0]     Interrupt   GPIOx[1:0]          */
    EXTI_2to3_IRQn        = 6,        /*  6       | EXTI[3:2]     Interrupt   GPIOx[3:2]          */
    EXTI_4to15_IRQn       = 7,        /*  7       | EXTI[15:4]    Interrupt   GPIOx[15:4]         */
    CM0IKMCU_IRQ08_IRQn   = 8,        /*  8       |               Interrupt                       */
    CM0IKMCU_IRQ09_IRQn   = 9,        /*  9       |               Interrupt                       */
    LCD_IRQn              = 10,       /*  10      | LCD           Interrupt                       */
    MMC_IRQn              = 11,       /*  11      | MMC           Interrupt                       */
    EXTI_16to19_IRQn      = 12,       /*  12      | EXTI[19:16]   Interrupt                      */
    CM0IKMCU_IRQ13_IRQn   = 13,       /*  13      |               Interrupt                       */
    BS16T1_IRQn           = 14,       /*  14      | BS16T1        Interrupt                       */
    GP32C4T1_IRQn         = 15,       /*  15      | GP32C4T1      Interrupt                       */
    CM0IKMCU_IRQ16_IRQn   = 16,       /*  16      |               Interrupt                       */
    CM0IKMCU_IRQ17_IRQn   = 17,       /*  17      |               Interrupt                       */
    CM0IKMCU_IRQ18_IRQn   = 18,       /*  18      |               Interrupt                       */
    GP16C2T1_IRQn         = 19,       /*  19      | GP16C2T1      Interrupt                       */
    GP16C2T2_IRQn         = 20,       /*  20      | GP16C2T2      Interrupt                       */
    CM0IKMCU_IRQ21_IRQn   = 21,       /*  21      |               Interrupt                       */
    CM0IKMCU_IRQ22_IRQn   = 22,       /*  22      |               Interrupt                       */
    I2C1_IRQn             = 23,       /*  23      | I2C1          Interrupt                       */
    CM0IKMCU_IRQ24_IRQn   = 24,       /*  24      |               Interrupt                       */
    SPI1_IRQn             = 25,       /*  25      | SPI1          Interrupt                       */
    CM0IKMCU_IRQ26_IRQn   = 26,       /*  26      |               Interrupt                       */
    UART1_IRQn            = 27,       /*  27      | UART1         Interrupt                       */
    CM0IKMCU_IRQ28_IRQn   = 28,       /*  28      |               Interrupt                       */
    CM0IKMCU_IRQ29_IRQn   = 29,       /*  29      |               Interrupt                       */
    SUART1_IRQn           = 30,       /*  30      | SUART1        Interrupt                       */
    CM0IKMCU_IRQ31_IRQn   = 31        /*  31      |               Interrupt                       */
} IRQn_Type;

/**
  * @}
  */

/* Includes-------------------------------------------------------------------------------*/

#include "core_cm0.h"
#include <stdint.h>
#include <stdbool.h>
#include "system_es32f0930.h"
/**
  * @brief these files declare Peripheral register bit_definition.
  */
#include "reg_rcu.h"
#include "reg_syscfg.h"
#include "reg_fc.h"
#include "reg_gpio.h"
#include "reg_i2c.h"
#include "reg_uart.h"
#include "reg_spi.h"
#include "reg_hdiv.h"
#include "reg_wwdt.h"
#include "reg_iwdt.h"
#include "reg_exti.h"
#include "reg_tick.h"
#include "reg_mmc.h"
#include "reg_lcd.h"
#include "reg_timer.h"
/******************************************************************************/
/*                              Peripheral register                           */
/******************************************************************************/

/*   allow anonymous structures and anonymous enumeration    */
/* #pragma anon_unions */

/******************************************************************************/
/*                            Peripheral memory map                           */
/******************************************************************************/

/*********************      Base addresses        *************************/
#define CM0IN_BASE          ((uint32_t)0xE0000000) // Cortex®-M0 internal Peripherals
#define AHB2_BASE           ((uint32_t)0x48000000) // AHB2 Peripheral
#define AHB1_BASE           ((uint32_t)0x40020000) // AHB1 Peripheral
#define APB2_BASE           ((uint32_t)0x40010000) // APB2 Peripheral
#define APB1_BASE           ((uint32_t)0x40000000) // APB1 Peripheral
#define SRAM_BASE           ((uint32_t)0x20000000) // SRAM Memory  ( 8K Bytes)
#define HWLIB_BASE          ((uint32_t)0x1FFFFC00) // HW Library   ( 1K Bytes)
#define EBDOP_BASE          ((uint32_t)0x1FFFF800) // EFLASH Option( 1K Bytes)
#define EBDINFO_BASE        ((uint32_t)0x1FFFE000) // EFLASH Info  ( 6K Bytes) - Boot Loader Memory
#define EBDDATA_BASE        ((uint32_t)0x08000000) // EFLASH Main  (64K Bytes)

/* Cortex®-M0 internal Peripherals - CM0IN_BASE: 0xE0000000*/
#define TICK_BASE         (CM0IN_BASE     + 0xE010)

#define SRAM1_BASE        (SRAM_BASE      + 0x0000)   // SRAM1 Memory ( 4K Bytes)
#define SRAM2_BASE        (SRAM_BASE      + 0x1000)   // SRAM2 Memory ( 4K Bytes)
#define STACK_SIZE        0x00000400>>2               // Stack size (in Words)

/* AHB2 Peripheral memory map - AHB2_BASE: 0x48000000*/
#define GPIOA_BASE        (AHB2_BASE      + 0x0000)
#define GPIOB_BASE        (AHB2_BASE      + 0x0400)
#define GPIOC_BASE        (AHB2_BASE      + 0x0800)
#define GPIOD_BASE        (AHB2_BASE      + 0x0C00)
#define GPIOE_BASE        (AHB2_BASE      + 0x1000)
#define GPIOF_BASE        (AHB2_BASE      + 0x1400)

/* AHB1 Peripheral memory map - AHB1_BASE: 0x40020000*/
#define RCU_BASE          (AHB1_BASE      + 0x1000)
#define SYSCFG_BASE       (AHB1_BASE      + 0x1400)
#define OPT_BASE          (AHB1_BASE      + 0x1480)
#define EFLASH_BASE       (AHB1_BASE      + 0x2000)
#define MSWD_BASE         (AHB1_BASE      + 0x2C00)
#define HDIV_BASE         (AHB1_BASE      + 0x3C00)

/* APB2 Peripheral memory map - APB2_BASE: 0x40010000*/
#define EXTI_BASE           (APB2_BASE      + 0x0400)
#define LCD_BASE            (APB2_BASE      + 0x0800)
#define MMC_BASE            (APB2_BASE      + 0x0C00)
#define SPI1_BASE           (APB2_BASE      + 0x3000)
#define UART1_BASE          (APB2_BASE      + 0x3800)
#define GP16C2T1_BASE       (APB2_BASE      + 0x4000)
#define GP16C2T2_BASE       (APB2_BASE      + 0x4400)
#define DBGMCU_BASE         (APB2_BASE      + 0x5800)

/* APB1 Peripheral memory map - APB1_BASE: 0x40000000*/
#define GP32C4T1_BASE       (APB1_BASE      + 0x0000)
#define BS16T1_BASE         (APB1_BASE      + 0x1000)
#define WWDT_BASE           (APB1_BASE      + 0x2C00)
#define IWDT_BASE           (APB1_BASE      + 0x3000)
#define SUART1_BASE         (APB1_BASE      + 0x4C00)
#define I2C1_BASE           (APB1_BASE      + 0x5400)
#define PWR_BASE            (APB1_BASE      + 0x7000)

/*********************      Peripheral declaration        *************************/
#define TICK              (( SYST_TypeDef     *)     TICK_BASE)
#define GPIOA             (( GPIO_TypeDef     *)    GPIOA_BASE)
#define GPIOB             (( GPIO_TypeDef     *)    GPIOB_BASE)
#define GPIOC             (( GPIO_TypeDef     *)    GPIOC_BASE)
#define GPIOD             (( GPIO_TypeDef     *)    GPIOD_BASE)
#define RCU               (( RCU_TypeDef      *)      RCU_BASE)
#define SYSCFG            (( SYSCFG_TypeDef   *)   SYSCFG_BASE)
#define OPT               (( OPT_TypeDef      *)      OPT_BASE)
#define FC                (( FC_TypeDef       *)   EFLASH_BASE)
#define HDIV              (( HDIV_TypeDef     *)     HDIV_BASE)
#define MSWD              (( MSWD_TypeDef     *)     MSWD_BASE)
#define IWDT              (( IWDT_TypeDef     *)     IWDT_BASE)
#define WWDT              (( WWDT_TypeDef     *)     WWDT_BASE)
#define GP32C4T1          (( TIMER_TypeDef    *) GP32C4T1_BASE)
#define GP16C2T1          (( TIMER_TypeDef    *) GP16C2T1_BASE)
#define GP16C2T2          (( TIMER_TypeDef    *) GP16C2T2_BASE)
#define BS16T1            (( TIMER_TypeDef    *)   BS16T1_BASE)
#define SPI1              (( SPI_TypeDef      *)     SPI1_BASE)
#define I2C1              (( I2C_TypeDef      *)     I2C1_BASE)
#define UART1             (( UART_TypeDef     *)    UART1_BASE)
#define SUART1            (( UART_TypeDef     *)   SUART1_BASE)
#define EXTI              (( EXTI_TypeDef     *)     EXTI_BASE)
#define LCD               (( LCD_TypeDef      *)      LCD_BASE)
#define MMC               (( MMC_TypeDef      *)      MMC_BASE)

/******************************************************************************/
/*                                          macros                            */
/******************************************************************************/

/** @addtogroup Public_macros
  * @{
  */
#if defined (__CC_ARM)
#define __INLINE__  __inline
#else
#define __INLINE__  inline
#endif

#define __isr__

#define UNUSED(x) ((void)(x))

#ifdef USE_ASSERT
#define assert_param(x)   \
    do                        \
    {                         \
        if (!(x))               \
        {                       \
            __disable_irq();      \
            while (1)             \
                ;                     \
        }                       \
    } while (0)
#else
#define assert_param(x)
#endif

#define BIT(x)                   ((uint32_t)((uint32_t)0x01U<<(x)))

#define BITS(start, end)         ((0xFFFFFFFFUL << (start)) &\
                                  (0xFFFFFFFFUL >> (31U - (uint32_t)(end))))

#define SET_BIT(REG, SETMASK)     ((REG) |= (SETMASK))

#define CLEAR_BIT(REG, SETMASK)   ((REG) &= ~(SETMASK))

#define READ_BIT(REG, SETMASK)    ((REG) & (SETMASK))

#define CLEAR_REG(REG)            ((REG) = (0x0))

#define WRITE_REG(REG, VAL)       ((REG) = (VAL))

#define READ_REG(REG)             ((REG))

#define MODIFY_REG(REG, CLEARMASK, SETMASK)  WRITE_REG((REG), (((READ_REG(REG)) & (~(CLEARMASK))) | (SETMASK)))

#define POSITION_VAL(VAL)         (__CLZ(__RBIT(VAL)))

/**
  * @}
  */

/******************************************************************************/
/*                                types                                       */
/******************************************************************************/
/** @addtogroup Public_types
  * @{
  */
typedef enum
{
    DISABLE = 0,
    ENABLE = !DISABLE
} TypeFunc, FuncState;
#define IS_FUNC_STATE(x) (((x) == DISABLE) || ((x) == ENABLE))

typedef enum
{
    RESET = 0,
    SET = !RESET
} FlagStatus, ITStatus, PinStatus;

typedef enum
{
    ERROR = 0,
    SUCCESS = !ERROR
} ErrorStatus;

typedef enum
{
    BIT_RESET = 0x0,
    BIT_SET   = 0x1,
} BitStatus;

typedef enum
{
    FALSE = 0x0,
    TRUE  = 0x1,
} TypeBool;

typedef enum
{
    UNLOCK = 0x0,
    LOCK   = 0x1,
} LockState;
#define IS_LOCK_STATE(x) (((x) == UNLOCK) || ((x) == LOCK))

/**
  * @} Public_types
  */

#ifdef __cplusplus
}
#endif

#endif

/**
  * @} es32f0930
  */

/**
  * @} CMSIS
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
