/**
  *********************************************************************************
  *
  * @file    bsp_sram.c
  * @brief   SRAM driver
  *
  * @version V1.0
  * @date    24 Apr 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <stdio.h>
#include "bsp_sram.h"

/** @addtogroup ES32F3xxx_BSP
  * @{
  */

/** @defgroup SRAM sram 
  * @{
  */

/** @defgroup SRAM_Private_Variables SRAM Private Variables
  * @{
  */

sram_handle_t hsram;
ald_ebi_nor_sram_timing_t sram_timing;

GPIO_TypeDef * EBI_SRAM_ADD_PORT[EBI_SRAM_ADDRESS_PORTn] = { 
				   EBI_A0_PORT, EBI_A1_PORT,
				   EBI_A2_PORT, EBI_A3_PORT,
				   EBI_A4_PORT, EBI_A5_PORT,
				   EBI_A6_PORT, EBI_A7_PORT,
				   EBI_A8_PORT, EBI_A9_PORT,
				   EBI_A10_PORT,EBI_A11_PORT,
				   EBI_A12_PORT,EBI_A13_PORT,
				   EBI_A14_PORT,EBI_A15_PORT,
				   EBI_A16_PORT,EBI_A17_PORT,
				   EBI_A18_PORT };
				   				   
const uint16_t EBI_SRAM_ADD_PIN[EBI_SRAM_ADDRESS_PINn] = {  
				   EBI_A0_PIN, EBI_A1_PIN,
				   EBI_A2_PIN, EBI_A3_PIN,			   
				   EBI_A4_PIN, EBI_A5_PIN,			   
				   EBI_A6_PIN, EBI_A7_PIN,			   
				   EBI_A8_PIN, EBI_A9_PIN,			   
				   EBI_A10_PIN,EBI_A11_PIN,			   
				   EBI_A12_PIN,EBI_A13_PIN,			   
				   EBI_A14_PIN,EBI_A15_PIN,
				   EBI_A16_PIN,EBI_A17_PIN,
				   EBI_A18_PIN };

GPIO_TypeDef * EBI_SRAM_DATA_PORT[EBI_SRAM_DATA_PORTn] = { 
				   EBI_D0_PORT, EBI_D1_PORT,
				   EBI_D2_PORT, EBI_D3_PORT,
				   EBI_D4_PORT, EBI_D5_PORT,
				   EBI_D6_PORT, EBI_D7_PORT,
				   EBI_D8_PORT, EBI_D9_PORT,
				   EBI_D10_PORT,EBI_D11_PORT,
				   EBI_D12_PORT,EBI_D13_PORT,
				   EBI_D14_PORT,EBI_D15_PORT };

const uint16_t EBI_SRAM_DATA_PIN[EBI_SRAM_DATA_PINn] = {  
				   EBI_D0_PIN, EBI_D1_PIN,
				   EBI_D2_PIN, EBI_D3_PIN,			   
				   EBI_D4_PIN, EBI_D5_PIN,			   
				   EBI_D6_PIN, EBI_D7_PIN,			   
				   EBI_D8_PIN, EBI_D9_PIN,			   
				   EBI_D10_PIN,EBI_D11_PIN,			   
				   EBI_D12_PIN,EBI_D13_PIN,			   
				   EBI_D14_PIN,EBI_D15_PIN };
/**
  * @}
  */
				   
/** @defgroup SRAM_Public_Functions SRAM Public Functions
  * @{
  */
  
/**
  * @brief  Sram pin initiation
  * @retval None
  */
void bsp_sram_pin_init(void)
{
	uint8_t i = 0;
	gpio_init_t x;
	
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_7;
	
	/* Address pin init */
	for(i = 0;i < EBI_SRAM_ADDRESS_PINn; ++i) {
		ald_gpio_init(EBI_SRAM_ADD_PORT[i], EBI_SRAM_ADD_PIN[i], &x);
	}
	
	/* Data pin init */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.func = GPIO_FUNC_7;
	for(i = 0;i < EBI_SRAM_DATA_PINn; ++i) {
		ald_gpio_init(EBI_SRAM_DATA_PORT[i], EBI_SRAM_DATA_PIN[i], &x);
	}
	
	/* Control pin init */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_7;
	/* Output enable pin init EBI_NOE */
	ald_gpio_init(EBI_NOE_PORT, EBI_NOE_PIN, &x);
	/* Write enable pin init EBI_NWE*/
	ald_gpio_init(EBI_NWE_PORT, EBI_NWE_PIN, &x);
	/* Chip select pin init EBI_NE1 */
	ald_gpio_init(EBI_NE3_PORT, EBI_NE3_PIN, &x);
	/* Low byte select EBI_NBL0 */
	ald_gpio_init(EBI_NBL0_PORT, EBI_NBL0_PIN, &x);
	/* High byte selec EBI_NBL1 */
	ald_gpio_init(EBI_NBL1_PORT, EBI_NBL1_PIN, &x);
}

/**
  * @brief  Sram initiation
  * @retval None
  */
void bsp_sram_init(void)
{
	hsram.instance	= EBI_NOR_SRAM_DEVICE;
	hsram.ext	= EBI_NOR_SRAM_EXTENDED_DEVICE;
	
	sram_timing.addr_setup		= 8;
	sram_timing.addr_hold		= 4;
	sram_timing.data_setup		= 7;
	sram_timing.bus_dur		= 1;
	sram_timing.div			= 0;
	sram_timing.latency		= 4;
	sram_timing.mode		= EBI_ACCESS_MODE_A;
	hsram.init.bank			= EBI_NORSRAM_BANK3;
	hsram.init.mux			= EBI_DATA_ADDRESS_MUX_DISABLE;
	hsram.init.type			= EBI_MEMORY_TYPE_SRAM;
	hsram.init.width		= SRAM_MEMORY_WIDTH;
	hsram.init.acc_mode		= EBI_BURST_ACCESS_MODE_DISABLE;
	hsram.init.polarity		= EBI_WAIT_SIGNAL_POLARITY_LOW;
	hsram.init.wrap_mode		= EBI_WRAP_MODE_DISABLE;
	hsram.init.active		= EBI_WAIT_TIMING_BEFORE_WS;
	hsram.init.write		= EBI_WRITE_OPERATION_ENABLE;
	hsram.init.signal		= EBI_WAIT_SIGNAL_DISABLE;
	hsram.init.ext_mode		= EBI_EXTENDED_MODE_DISABLE;
	hsram.init.wait			= EBI_ASYNCHRONOUS_WAIT_DISABLE;
	hsram.init.burst		= EBI_WRITE_BURST_DISABLE;
	ald_sram_init(&hsram, &sram_timing, &sram_timing);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
