/**
  *********************************************************************************
  *
  * @file    bsp_record.c
  * @brief   Record driver
  *
  * @version V1.0
  * @date    10 May 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          10 May 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
  
#include "bsp_audio.h"
#include "bsp_record.h"
#include "bsp_display.h"
#include "ff.h"
#include <string.h>
#include <stdio.h>

/** @addtogroup ES32F3xxx_BSP
  * @{
  */

/** @defgroup Record Record
  * @{
  */

/** @defgroup Record_Public_Variables Record Public Variables
  * @{
  */
#define PATCH_LEN 40
const uint16_t wav_patch[PATCH_LEN]=
{ 
	0x0007, 0x0001, 0x8010, 0x0006, 0x001c, 0x3e12, 0xb817, 0x3e14, 
	0xf812, 0x3e01, 0xb811, 0x0007, 0x9717, 0x0020, 0xffd2, 0x0030, 
	0x11d1, 0x3111, 0x8024, 0x3704, 0xc024, 0x3b81, 0x8024, 0x3101, 
	0x8024, 0x3b81, 0x8024, 0x3f04, 0xc024, 0x2808, 0x4800, 0x36f1, 
	0x9811, 0x0007, 0x0001, 0x8028, 0x0006, 0x0002, 0x2a00, 0x040e,  
}; 

uint8_t recbuf[512] = {0};
record_flag_t recf = {0};
FIL ftmp;
FIL frec;
wave_header_t wvh;

/**
  * @}
  */

/** @defgroup Record_Public_Functions Record Public Functions
  * @{
  */
/**
  * @brief  load vs1053 patch
  * @param  spatch: patch data start address
  * @param  len: patch data length
  * @retval None
  */
void load_wav_patch(uint16_t *spatch, uint16_t len)
{
	int i = 0;
	uint16_t addr, n, val;

	while (i < PATCH_LEN) {
		addr = spatch[i++];
		n = spatch[i++];
		if (n & 0x8000U) {
			/* RLE run, replicate n samples */
			n &= 0x7FFF;
			val = spatch[i++];
			while (n--) {
				vs1053_write_cmd(addr, val);
			}
		}
		else {
			/* Copy run, copy n samples */
			while (n--) {
				val = spatch[i++];
				vs1053_write_cmd(addr, val);
			}
		}
	}
	if (vs1053_wait_timeout()) {
		return;
	}
}
  
/**
  * @brief  Record init
  * @retval None
  */ 
void bps_record_init(void)
{
	uint16_t i = 0;
	
	vs1053_write_cmd(SCI_BASS   , 0x0000);
	vs1053_write_cmd(SCI_AICTRL0, 8000);
	vs1053_write_cmd(SCI_AICTRL1, 4096);
	vs1053_write_cmd(SCI_AICTRL2, 0);
	vs1053_write_cmd(SCI_AICTRL3, 6);
	vs1053_write_cmd(SCI_CLOCKF, 0x2000);
	vs1053_write_cmd(SCI_MODE  , 0x1804);
	for (i = 0; i < 0x8fff; ++i);
	load_wav_patch((uint16_t*)wav_patch, 40);
	for (i = 0; i < 0x2fff; ++i);
}

/**
  * @brief  Record file header default data init
  * @param  whead: pointer to header struct
  * @retval None
  */
void record_header_default_init(wave_header_t *whead)
{
	whead->riff.chunk_id       = 0X46464952;	
	whead->riff.chunk_size     = 0;		
	whead->riff.format         = 0X45564157; 	
	whead->fmt.chunk_id        = 0X20746D66; 	
	whead->fmt.chunk_size      = 16; 		
	whead->fmt.audio_format    = 0X01; 		
 	whead->fmt.num_of_channels = 1;		
 	whead->fmt.sample_rate     = 8000;		
 	whead->fmt.byte_rate       = whead->fmt.sample_rate * 2;
 	whead->fmt.block_align     = 2;			
 	whead->fmt.bits_persample  = 16;		
   	whead->data.chunk_id       = 0X61746164;	
 	whead->data.chunk_size     = 0;			
}

/**
  * @brief  Record time stamp display 
  * @param  stamp: write sdcard times
  * @retval None
  */
void record_time_stamp_dislay(uint32_t stamp)
{		 
	lcd_show_string(30, 260, 200, 16, 16, "->TIME:");	  	  
	lcd_show_xnumber(30 + 60, 260, stamp/60, 2, 16, 0X80);	
	lcd_show_char(30 + 80, 260, ':', 16, 0);
	lcd_show_xnumber(30 + 90, 260, stamp % 60, 2, 16, 0X80);	
}

/**
  * @brief  Check Record for the new record file name
  * @param  name: new file name start address
  * @retval None
  */
void record_check_new_name(char *name)
{	 
	uint8_t result = 0;					 
	uint16_t idx = 0;
	
	for (idx = 0; idx < 0xffff; ++idx) {
		sprintf((char*)name, "0:Record/REC%05d.wav", idx); 
		result = f_open(&ftmp, (const TCHAR*)name, FA_READ);
		if(result == FR_NO_FILE)
			break;		
	}
}
  
/**
  * @}
  */

/**
  * @}
  */
  
/**
  * @}
  */

