/**
  *********************************************************************************
  *
  * @file    bsp_pwm.c
  * @brief   Pwm driver
  *
  * @version V1.0
  * @date    16 Apr 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          16 Apr 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include "bsp_pwm.h"
#include "ald_gpio.h"
#include "ald_cmu.h"

 /** @addtogroup ES32F3xxx_BSP
   * @{
   */

/** @defgroup PWM pwm
  * @{
  */

/** @defgroup PWM_Private_Variables pwm Private Variables
  * @{
  */
timer_handle_t  h_timer;
struct pwm_config pwm_cfg = {0};
/**
  * @}
  */

/** @defgroup PWM_Private_Functions pwm Private Functions
  * @{
  */

/**
  * @brief  PWM duty set function.
  * @param  timer_initstruct: pointer to timer handler
  * @param  ch: timer channel
  * @param  ns: timer duty time in unit of ns
  * @retval None.
  */
static void bsp_pwm_set_duty(timer_handle_t *timer_initstruct, timer_channel_t ch, uint32_t ns)
{
	uint64_t tmp = (uint64_t)ald_cmu_get_pclk1_clock() * ns / 1000000000 /
			(timer_initstruct->init.prescaler + 1);

	if (ch == TIMER_CHANNEL_1)
		WRITE_REG(timer_initstruct->perh->CCVAL1, (uint32_t)tmp);
	else if (ch == TIMER_CHANNEL_2)
		WRITE_REG(timer_initstruct->perh->CCVAL2, (uint32_t)tmp);
	else if (ch == TIMER_CHANNEL_3)
		WRITE_REG(timer_initstruct->perh->CCVAL3, (uint32_t)tmp);
	else if (ch == TIMER_CHANNEL_4)
		WRITE_REG(timer_initstruct->perh->CCVAL4, (uint32_t)tmp);
}

/**
  * @brief  PWM frequency set function.
  * @param  timer_initstruct: pointer to timer handler
  * @param  ns: timer duty time in unit of ns
  * @retval None.
  */
static void bsp_pwm_set_freq(timer_handle_t *timer_initstruct, uint32_t ns)
{
	uint64_t _arr = (uint64_t)ald_cmu_get_pclk1_clock() * ns / 1000000000 /
			(timer_initstruct->init.prescaler + 1);

	WRITE_REG(timer_initstruct->perh->AR, (uint32_t)_arr);
	timer_initstruct->init.period   = (uint32_t)_arr;
}
/**
  * @}
  */

/** @defgroup PWM_Public_Functions pwm Public Functions
  * @brief    BSP EEPROM Functions
  */

/**
  * @brief  PWM pin and timer initiation
  * @param  hperh: pointer to timer handler
  * @retval None.
  */
void bsp_pwm_init(timer_handle_t *hperh)
{
	gpio_init_t x;
	
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv= GPIO_OUT_DRIVE_1;
	x.nodrv= GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	hperh->perh = GP16C4T1;
	ald_timer_pwm_init(hperh);
	/* gpio initialization */
	x.func = GPIO_FUNC_5;
	ald_gpio_init(GPIOA, GPIO_PIN_3, &x);
}

/**
  * @brief  PWM pin and timer initiation
  * @param  hperh: pointer to timer handler
  * @param  cmd: control command
  * @param  arg: pwm control parameter
  * @retval None.
  */
void bsp_pwm_control(timer_handle_t *hperh, int cmd, void *arg)
{
	uint32_t ccep;
	
	timer_channel_t pwm_channel;
	timer_oc_init_t tim_ocinit;
	timer_handle_t *timer_initstruct = hperh;
	struct pwm_config *cfg = (struct pwm_config *)arg;


	tim_ocinit.oc_mode      = TIMER_OC_MODE_PWM1;
	tim_ocinit.oc_polarity  = TIMER_OC_POLARITY_HIGH;
	tim_ocinit.oc_fast_en   = DISABLE;
	tim_ocinit.ocn_polarity = TIMER_OCN_POLARITY_HIGH;
	tim_ocinit.ocn_idle     = TIMER_OCN_IDLE_RESET;
	tim_ocinit.oc_idle      = TIMER_OC_IDLE_RESET;
	
	/* select pwm output channel */
	if (0 == cfg->channel) {
		pwm_channel = TIMER_CHANNEL_1;
	}
	else if (1 == cfg->channel) {
		pwm_channel = TIMER_CHANNEL_2;
	}
	else if (2 == cfg->channel) {
		pwm_channel = TIMER_CHANNEL_3;
	}
	else if (3 == cfg->channel) {
		pwm_channel = TIMER_CHANNEL_4;
	}
	switch (cmd)
	{
	case PWM_CMD_ENABLE:
		ald_timer_pwm_start(timer_initstruct, pwm_channel);
		break;
	case PWM_CMD_DISABLE:
		ald_timer_pwm_stop(timer_initstruct, pwm_channel);
		break;
	case PWM_CMD_SET:
		ccep = timer_initstruct->perh->CCEP;
		/* count registers max 0xFFFF, auto adjust prescaler */
		do {
			bsp_pwm_set_freq(timer_initstruct, cfg->period);
			timer_initstruct->init.prescaler ++;
		}
		while (timer_initstruct->init.period > 0xFFFF);
		/* update prescaler */
		WRITE_REG(timer_initstruct->perh->PRES, --timer_initstruct->init.prescaler);
		ald_timer_oc_config_channel(timer_initstruct, &tim_ocinit, pwm_channel);
		bsp_pwm_set_duty(timer_initstruct, pwm_channel, cfg->pulse);
		timer_initstruct->perh->CCEP = ccep;
		break;
	case PWM_CMD_GET:
		cfg->pulse = ald_timer_read_capture_value(timer_initstruct, pwm_channel) * 100 /
			READ_REG(timer_initstruct->perh->AR);
		break;
	default:
		break;
    }
}

/**
  * @brief  PWM parameter set.
  * @param  channel: timer channel
  * @param  period: pwm period
  * @param  pulse: pwm pulse
  * @retval None.
  */
void bsp_pwm_set(timer_handle_t *hperh, int channel, uint32_t period, uint32_t pulse)
{
	pwm_cfg.channel = channel;
	pwm_cfg.period  = period;
	pwm_cfg.pulse   = pulse;
	bsp_pwm_control(hperh, PWM_CMD_SET, &pwm_cfg);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

