/**
  *********************************************************************************
  *
  * @file    bsp_micro_sd.c
  * @brief   MICRO-SD driver
  *
  * @version V1.0
  * @date    10 May 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          10 May 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "ald_gpio.h"
#include "ald_spi.h"
#include "bsp_micro_sd.h"
#include "bsp_audio.h"

/** @addtogroup ES32F3xxx_BSP
  * @{
  */

/** @defgroup SD micro-sd
  * @{
  */

/** @defgroup SD_Public_Variables Public Variables
  * @{
  */
bsp_sd_env_t bsp_sd_env;
uint8_t __buf[20];
uint8_t sd_type = 2;
/**
  * @}
  */

/** @defgroup SD_Private_Functions Private Functions
  * @{
  */

/**
  * @brief  micro-sd module pin initiation
  * @retval None.
  */
static void sd_pin_config(void)
{
	gpio_init_t x;

	/* Initialize sck pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = SD_SCK_FUNC;
	ald_gpio_init(SD_SCK_PORT, SD_SCK_PIN, &x);

	/* Initialize miso pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = SD_MISO_FUNC;
	ald_gpio_init(SD_MISO_PORT, SD_MISO_PIN, &x);
	
	

	/* Initialize mosi pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = SD_MOSI_FUNC;
	ald_gpio_init(SD_MOSI_PORT, SD_MOSI_PIN, &x);

	/* Initialize cs pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = SD_SCS_FUNC;
	ald_gpio_init(SD_SCS_PORT, SD_SCS_PIN, &x);

	ald_gpio_write_pin(SD_SCS_PORT, SD_SCS_PIN, 1);

	return;
}

/**
  * @brief  Micro-sd module spi confiugure.
  * @retval None.
  */
static void sd_spi_config(void)
{
	/* Initialize SPI */
	bsp_sd_env.h_spi.perh           = SPI0;
	bsp_sd_env.h_spi.init.mode      = SPI_MODE_MASTER;
	bsp_sd_env.h_spi.init.dir       = SPI_DIRECTION_2LINES;
	bsp_sd_env.h_spi.init.data_size = SPI_DATA_SIZE_8;
	bsp_sd_env.h_spi.init.baud      = SPI_BAUD_256;
	bsp_sd_env.h_spi.init.phase     = SPI_CPHA_SECOND;
	bsp_sd_env.h_spi.init.polarity  = SPI_CPOL_HIGH;
	bsp_sd_env.h_spi.init.first_bit = SPI_FIRSTBIT_MSB;
	bsp_sd_env.h_spi.init.ss_en     = DISABLE;
	bsp_sd_env.h_spi.init.crc_calc  = DISABLE;
	bsp_sd_env.h_spi.init.crc_poly  = 7;
	ald_spi_init(&bsp_sd_env.h_spi);

	return;
}

/**
  * @brief  Delay.
  * @param  delay: times of delay.
  * @retval None.
  */
static void sd_delay(uint32_t delay)
{
	while (delay--);
}

/**
  * @brief  Send a command to card.
  * @param  cmd: Type of command.
  * @param  arg: Parameter of command.
  * @param  crc: CRC of command.
  * @retval Card's status.
  */
static uint8_t sd_send_cmd(sd_card_cmd_t cmd, uint32_t arg, uint8_t crc)
{
	uint8_t buf[6];
	uint8_t ret, i;
	uint16_t retry = 10;
	int r_flag = 0;

	ald_gpio_write_pin(SD_SCS_PORT, SD_SCS_PIN, 1);
	ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);
	ald_gpio_write_pin(SD_SCS_PORT, SD_SCS_PIN, 0);
	sd_delay(DELAY_CNT);

	buf[0] = cmd | 0x40;
	buf[1] = (arg >> 24) & 0xFF;
	buf[2] = (arg >> 16) & 0xFF;
	buf[3] = (arg >> 8) & 0xFF;
	buf[4] = arg & 0xFF;
	buf[5] = crc;

	for (i = 0; i < 6; ++i)
		ald_spi_send_byte_fast(&bsp_sd_env.h_spi, buf[i]);

	do {
		ret = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
	} while ((ret == 0xFF) && (retry--));

	return ret;
}

/**
  * @brief  Write data to card.
  * @param  buf: Pointer to data.
  * @param  cmd: Type of command
  * @retval Card's status.
  */
static int sd_write_block(uint8_t *buf, uint8_t cmd)
{
	uint8_t ret = 0;
	uint16_t retry = 10;
	int r_flag = 0;

	do {
		ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);
	} while (--retry);

	ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFE);

	for (retry = 0; retry < 512; ++retry) {
		ald_spi_send_byte_fast(&bsp_sd_env.h_spi, buf[retry]);
	}

	ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);
	ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);
	sd_delay(DELAY_CNT * 4);

	ret = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
	if ((ret & 0x1f) != 0x5) {
		ret = 1;
		goto end;
	}

	retry = 0xFFFF;
	do {
		ret = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
	} while ((ret != 0xFF) && (--retry));

	ret = retry == 0 ? 2 : 0;

end:
	ald_gpio_write_pin(SD_SCS_PORT, SD_SCS_PIN, 1);
	sd_delay(DELAY_CNT);
	ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);
	sd_delay(DELAY_CNT);

	return ret;
}

/**
  * @brief  Read data from card.
  * @param  buf: Pointer to data.
  * @param  len: Length of data.
  * @retval Card's status.
  */
static int sc_read_data(uint8_t *buf, uint16_t len)
{
	uint8_t ret;
	uint16_t retry = 0xFFFF;
	int r_flag = 0;

	do {
		ret = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
	} while ((ret != 0xFE) && (--retry));

	if (!retry)
		return -1;

	while (len--) {
		*buf++ = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
	}

	/* Read CRC */
	__buf[18] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
	__buf[19] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);

	ald_gpio_write_pin(SD_SCS_PORT, SD_SCS_PIN, 1);
	sd_delay(DELAY_CNT);
	ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);
	sd_delay(DELAY_CNT);

	return 0;
}
/**
  * @}
  */

/** @defgroup SD_Public_Functions SD Public Functions
  * @brief    BSP SD-CARD Functions
  * @verbatim
==================================================================================
               ##### BSP SD-CARD Functions #####
==================================================================================
 [..]   This section provides functions allowing to:
    (+) Initialize sd-card module.
    (+) Get the size of the card
    (+) Get CID of the card
    (+) Erase the sector of card.
    (+) Write the sector of card.
    (+) Read the sector of card.

    @endverbatim
  * @{
  */


uint8_t CIDBUF[16] = {0};

/**
  * @brief  SD initialize.
  * @retval None.
  */
int bsp_sd_init(void)
{
	uint32_t retry = 2, i;
	uint8_t ret;
	int r_flag = 0;

	memset(__buf, 0xFF, 20);
	memset(&bsp_sd_env, 0x0, sizeof(bsp_sd_env_t));
	sd_pin_config();
	sd_spi_config();

	ald_gpio_write_pin(SD_SCS_PORT, SD_SCS_PIN, 1);
	
	for (i = 0; i < 20; ++i)
		ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);

	sd_delay(DELAY_CNT);

	/* Send CMD0 */
	do {
		ret = sd_send_cmd(SD_CMD0, 0, 0x95);
	} while ((ret != 1) && (--retry));

	/* Send CMD8 */
	ret = sd_send_cmd(SD_CMD8, 0x1AA, 0x87);
	if (ret == 0x1) {
		__buf[0] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
		__buf[1] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
		__buf[2] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
		__buf[3] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
	}

	if ((__buf[2] != 0x1) || (__buf[3] != 0xAA))
		return -2;

	/* Send CMD55+ACMD41 */
	retry = 500;
	do {
		ret = sd_send_cmd(SD_CMD55, 0, 0x1);
		if (ret == 1)
			ret = sd_send_cmd(SD_ACMD41, 0x40000000, 0x1);
	} while ((ret != 0) && (--retry));

	if (ret != 0)
		return -3;

	/* Send CMD58 */
	ret = sd_send_cmd(SD_CMD58, 0, 0x1);
	if (ret == 0x0) {
		__buf[10] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
		__buf[11] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
		__buf[12] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
		__buf[13] = ald_spi_recv_byte_fast(&bsp_sd_env.h_spi, &r_flag);
	}
	if ((__buf[10] & 0x40) == 0x40)
		sd_type = 4; /* V2-HC */

	ald_gpio_write_pin(SD_SCS_PORT, SD_SCS_PIN, 1);
	sd_delay(DELAY_CNT);
	ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);
	sd_delay(DELAY_CNT);

	ald_spi_speed_config(&bsp_sd_env.h_spi, SPI_BAUD_8);
	return 0;
}

/**
  * @brief  SDCard detect pin init
  * @retval None
  */
void bsp_sd_detect_pin_init(void)
{
	gpio_init_t x;
	
	/* Initialize det pin */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odos = GPIO_PUSH_PULL;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(SD_DET_PORT, SD_DET_PIN, &x);
}

/**
  * @brief  Get size of the card.
  * @retval Number of sectors.
  */
uint32_t bsp_sd_get_size(void)
{
	uint8_t ret, n;
	uint32_t size, capacity;

	ret = sd_send_cmd(SD_CMD9, 0, 0x1);
	if (ret == 0x0)
		ret = sc_read_data(__buf, 16);

	/* SDHC */
	if ((__buf[0] & 0xC0) == 0x40) {
		size = __buf[9] + ((uint16_t)__buf[8] << 8) + 1;
		capacity = (uint32_t)size << 10;
	} else { /* V1 */
		n = (__buf[5] & 15) + ((__buf[10] & 128) >> 7) + ((__buf[9] & 3) << 1) + 2;
		size = (__buf[8] >> 6) + ((uint16_t)__buf[7] << 2) + ((uint16_t)(__buf[6] & 3) << 10) + 1;
		capacity = (uint32_t)size << (n - 9);
	}

	return capacity;
}

/**
  * @brief  Get CID of the card.
  * @param  buf: Pointer to buffer.
  * @retval Status.
  */
int bsp_sd_get_cid(uint8_t *buf)
{
	uint8_t ret;

	ret = sd_send_cmd(SD_CMD10, 0, 0x1);
	if (ret == 0x0)
		ret = sc_read_data(buf, 16);

	return 0;
}

/**
  * @brief  Erase the sectors in the card
  * @param  s_sector: starting of sectors.
  * @param  e_sector: ending of sectors.
  * @retval Status:
  *          -  0 Success
  *          - -1 Failed
  */
int bsp_sd_erase(uint32_t s_sector, uint32_t e_sector)
{
	uint32_t retry = 10;
	uint8_t ret;

	do {
		ret = sd_send_cmd(SD_CMD32, s_sector, 0x1);
	} while ((ret != 0) && (--retry));

	if (ret != 0)
		goto end;

	retry = 10;
	do {
		ret = sd_send_cmd(SD_CMD33, e_sector, 0x1);
	} while ((ret != 0) && (--retry));

	if (ret != 0)
		goto end;

	retry = 10;
	do {
		ret = sd_send_cmd(SD_CMD38, 0, 0x1);
	} while ((ret != 0) && (--retry));

	if (ret != 0)
		goto end;

	ald_gpio_write_pin(SD_SCS_PORT, SD_SCS_PIN, 1);
	sd_delay(DELAY_CNT);
	ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);
	sd_delay(DELAY_CNT);
	return 0;

end:
	ald_gpio_write_pin(SD_SCS_PORT, SD_SCS_PIN, 1);
	sd_delay(DELAY_CNT);
	ald_spi_send_byte_fast(&bsp_sd_env.h_spi, 0xFF);
	sd_delay(DELAY_CNT);
	return -1;
}

/**
  * @brief  Write data to the sector in the card
  * @param  buf: Pointer to the data.
  * @param  idx_sector: Index of sector in the card.
  * @retval Status:
  *          -  0 Success
  *          - -1 Failed
  */
int bsp_sd_write_sector(uint8_t *buf, uint32_t idx_sector)
{
	uint8_t ret, retry = 10;
	int result = 1;

	do {
		ret = sd_send_cmd(SD_CMD24, idx_sector, 0x1);
	} while ((ret != 0) && (--retry));

	if (ret == 0x0)
		result = sd_write_block(buf, 0xFE);

	return result;
}

/**
  * @brief  Read data from the sector in the card
  * @param  buf: Pointer to the data.
  * @param  idx_sector: Index of sector in the card.
  * @retval Status:
  *          -  0 Success
  *          - -1 Failed
  */
int bsp_sd_read_sector(uint8_t *buf, uint32_t idx_sector)
{
	uint8_t ret, retry = 10;

	do {
		ret = sd_send_cmd(SD_CMD17, idx_sector, 0x1);
	} while ((ret != 0) && (--retry));

	if (ret == 0x0)
		ret = sc_read_data(buf, 512);

	return ret;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
