/**
  *********************************************************************************
  *
  * @file    bsp_lcd.c
  * @brief   Lcd driver
  *
  * @version V1.0
  * @date    16 Apr 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          16 Apr 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
  
#include "bsp_display.h"
#include "ald_gpio.h"
#include "ald_sram.h"
#include "font.h"
#include "lv_color.h"

 /**
   * @addtogroup ES32F3xxx_BSP
   * @{
   */

/**
  * @defgroup LCD lcd
  * @{
  */

/**
  * @defgroup LCD_Private_Params lcd Private Variables
  * @{
  */
sram_handle_t hperh;
lcd_dev_t lcd_device;

GPIO_TypeDef * EBI_DATA_PORT[EBI_DATA_PORTn] = { 
				   EBI_D0_PORT, EBI_D1_PORT,
				   EBI_D2_PORT, EBI_D3_PORT,
				   EBI_D4_PORT, EBI_D5_PORT,
				   EBI_D6_PORT, EBI_D7_PORT,
				   EBI_D8_PORT, EBI_D9_PORT,
				   EBI_D10_PORT,EBI_D11_PORT,
				   EBI_D12_PORT,EBI_D13_PORT,
				   EBI_D14_PORT,EBI_D15_PORT };

const uint16_t EBI_DATA_PIN[EBI_DATA_PINn] = {  
				   EBI_D0_PIN, EBI_D1_PIN,
				   EBI_D2_PIN, EBI_D3_PIN,			   
				   EBI_D4_PIN, EBI_D5_PIN,			   
				   EBI_D6_PIN, EBI_D7_PIN,			   
				   EBI_D8_PIN, EBI_D9_PIN,			   
				   EBI_D10_PIN,EBI_D11_PIN,			   
				   EBI_D12_PIN,EBI_D13_PIN,			   
				   EBI_D14_PIN,EBI_D15_PIN };

/**
  * @}
  */

				   
/** @defgroup LCD_Private_Functions LCD Private Functions
  * @{
  */				   
	   
/**
  * @brief  LCD write register
  * @param  regval: write value
  * @retval None
  */
static void lcd_wr_reg(volatile uint16_t regval)
{   
	regval = regval;		
	LCD_ADDR->lcd_reg = regval;	 
}

/**
  * @brief  LCD write data
  * @param  data: write data
  * @retval None
  */
static void lcd_wr_data(volatile uint16_t data)
{	  
	data = data;
	LCD_ADDR->lcd_ram = data;		 
}

/**
  * @brief  LCD read data
  * @retval return ram data
  */
static uint16_t lcd_rd_data(void)
{
	volatile uint16_t ram;
	
	ram = LCD_ADDR->lcd_ram;
	
	return ram;	 
}

/**
  * @brief  LCD write value to register
  * @param  reg: register address
  * @param  regvalue: register value
  * @retval None
  */
static void lcd_write_reg(uint16_t reg, uint16_t regvalue)
{	
	LCD_ADDR->lcd_reg = reg;		 
	LCD_ADDR->lcd_ram = regvalue;	    		 
}	   

/**
  * @brief  LCD read value to register
  * @param  reg: register address
  * @retval return register value
  */
uint16_t lcd_read_reg(uint16_t reg)
{										   
	uint16_t i = 0;
	
	lcd_wr_reg(reg);		
	for (i = 0; i < 0xfff; ++i) ;
	return lcd_rd_data();		
}

/**
  * @brief  LCD operation delay 
  * @param  i: delay value
  * @retval None
  */
void opt_delay(uint8_t i)
{
	while(i--);
}

/**
  * @brief  n power of m
  * @param  m: Value 
  * @param  n: Power of first parameter
  * @retval None
  */
uint32_t lcd_pow(uint8_t m, uint8_t n)
{
	uint32_t result = 1;
	
	while(n--) result*=m;    
	
	return result;
}

/**
  * @}
  */

/** @defgroup LCD_Public_Functions LCD Public Functions
  * @{
  */

/**
  * @brief  Start to write lcd command
  * @retval None
  */
void lcd_write_ram_prepare(void)
{
 	LCD_ADDR->lcd_reg = lcd_device.wramcmd;	  
}

/**
  * @brief  LCD pin initiation
  * @retval None
  */			   
void bsp_lcd_pin_init(void)
{
	uint8_t i = 0;
	gpio_init_t x;
		
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.flt  = GPIO_FILTER_DISABLE;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.type  = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_7;
	
	for(i = 0;i < EBI_DATA_PINn; ++i) {
		ald_gpio_init(EBI_DATA_PORT[i], EBI_DATA_PIN[i], &x);
	}
	
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_7;
	
	ald_gpio_init(EBI_NOE_PORT, EBI_NOE_PIN, &x); /* NT35510 output enable pin EBI_NOE */
	ald_gpio_init(EBI_NWE_PORT, EBI_NWE_PIN, &x); /* NT35510 write enbale pin EBI_NWE*/
	ald_gpio_init(EBI_NE4_PORT, EBI_NE4_PIN, &x); /* NT35510 chip selected pin EBI_NE4 */
	ald_gpio_init(EBI_RS_PORT, EBI_RS_PIN, &x);   /* NT35510 CMD/DATA control pin EBI_A18 */
	
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(EBI_TFT2_RST_PORT, EBI_TFT2_RST_PIN, &x);
}

/**
  * @brief  Reset lcd function
  * @retval None
  */	
void bsp_reset_lcd(void)
{
	uint32_t i = 0;
	
	ald_gpio_write_pin(EBI_TFT2_RST_PORT, EBI_TFT2_RST_PIN, 0);
	for ( i = 0xfffff; i > 0; --i);
	ald_gpio_write_pin(EBI_TFT2_RST_PORT, EBI_TFT2_RST_PIN, 1);
	for ( i = 0xfffff; i > 0; --i);
}

/**
  * @brief  LCD initiation
  * @retval None
  */
void bsp_lcd_init(void)
{
	ald_ebi_nor_sram_timing_t ebi_rwt = {1};
	ald_ebi_nor_sram_timing_t ebi_wt= {1};
	uint32_t i = 0;
	
	hperh.instance	= EBI_NOR_SRAM_DEVICE;
	hperh.ext	= EBI_NOR_SRAM_EXTENDED_DEVICE;
	
	hperh.init.bank		= EBI_NORSRAM_BANK4;
	hperh.init.mux		= EBI_DATA_ADDRESS_MUX_DISABLE;
	hperh.init.type		= EBI_MEMORY_TYPE_SRAM;
	hperh.init.width	= EBI_NORSRAM_MEM_BUS_WIDTH_16;
	hperh.init.acc_mode	= EBI_BURST_ACCESS_MODE_DISABLE;
	hperh.init.polarity	= EBI_WAIT_SIGNAL_POLARITY_LOW;
	hperh.init.wrap_mode	= EBI_WRAP_MODE_DISABLE;
	hperh.init.active	= EBI_WAIT_TIMING_BEFORE_WS;
	hperh.init.write	= EBI_WRITE_OPERATION_ENABLE;
	hperh.init.signal	= EBI_WAIT_SIGNAL_DISABLE;
	hperh.init.ext_mode	= EBI_EXTENDED_MODE_ENABLE;
	hperh.init.wait		= EBI_ASYNCHRONOUS_WAIT_DISABLE;
	hperh.init.burst	= EBI_WRITE_BURST_DISABLE;
	
	#if 0
	/* EBI read time */
	ebi_rwt.addr_setup	= 0x0A;	/* Address setuptime(ADDSET) 1/96M * 10 = 10.4ns * 10 = 104ns */
	ebi_rwt.addr_hold	= 0x01;
	ebi_rwt.data_setup	= 0x28;	/* Data save time(DATAST) 40 HCLK = 10.4ns * 40 = 416ns */
	ebi_rwt.mode		= EBI_ACCESS_MODE_A;
	ebi_rwt.latency		= 0x01;
	
	/* EBI write time */
	ebi_wt.addr_setup	= 0x0A;	/* Address setuptime (ADDSET) 1/96M * 10 = 10.4ns * 10 = 104ns */
	ebi_wt.addr_hold	= 0x01;
	ebi_wt.data_setup	= 0x28;	/* Data save time(DATAST) 40 HCLK = 10.4ns * 40 = 416ns */
	ebi_wt.mode		= EBI_ACCESS_MODE_A;
	ebi_wt.latency		= 0x01;
	#endif
	
	/* EBI read time */
	ebi_rwt.addr_setup	= 0x0A;	/* Address setuptime(ADDSET) 1/96M * 10 = 10.4ns * 10 = 104ns */
	ebi_rwt.addr_hold	= 0x01;
	ebi_rwt.data_setup	= 0x0f;	/* Data save time(DATAST) 40 HCLK = 10.4ns * 40 = 416ns */
	ebi_rwt.mode		= EBI_ACCESS_MODE_A;
	ebi_rwt.latency		= 0x01;
	
	/* EBI write time */
	ebi_wt.addr_setup	= 0x0A;	/* Address setuptime (ADDSET) 1/96M * 10 = 10.4ns * 10 = 104ns */
	ebi_wt.addr_hold	= 0x01;
	ebi_wt.data_setup	= 0x06;	/* Data save time(DATAST) 40 HCLK = 10.4ns * 40 = 416ns */
	ebi_wt.mode		= EBI_ACCESS_MODE_A;
	ebi_wt.latency		= 0x01;
	
	ald_sram_init(&hperh, &ebi_rwt, &ebi_wt);
	
	/* Read ILI9488 ID */
	lcd_wr_reg(0XD3);				   
        lcd_device.id = lcd_rd_data();	
        lcd_device.id = lcd_rd_data();	
  	lcd_device.id = lcd_rd_data();   								   
        lcd_device.id <<= 8;
	lcd_device.id |= lcd_rd_data();  	
	
	/* Read NT35510 ID */
	if (lcd_device.id != 0X9488) { 
		lcd_wr_reg(0XDA00);
		lcd_device.id = lcd_rd_data();
		lcd_wr_reg(0XDB00);	
		lcd_device.id = lcd_rd_data();		
		lcd_device.id <<= 8;	
		lcd_wr_reg(0XDC00);	
		lcd_device.id |= lcd_rd_data();
		if(lcd_device.id == 0x8000)
			lcd_device.id = 0x5510;  /* NT35510's ID is 8000H */
	}
	
	if(lcd_device.id == 0x5510) { 
		lcd_write_reg(0xF000, 0x55);
		lcd_write_reg(0xF001, 0xAA);
		lcd_write_reg(0xF002, 0x52);
		lcd_write_reg(0xF003, 0x08);
		lcd_write_reg(0xF004, 0x01);
		/* AVDD Set AVDD 5.2V */
		lcd_write_reg(0xB000, 0x0D);
		lcd_write_reg(0xB001, 0x0D);
		lcd_write_reg(0xB002, 0x0D);
		/* AVDD ratio */
		lcd_write_reg(0xB600, 0x34);
		lcd_write_reg(0xB601, 0x34);
		lcd_write_reg(0xB602, 0x34);
		/* AVEE -5.2V */
		lcd_write_reg(0xB100, 0x0D);
		lcd_write_reg(0xB101, 0x0D);
		lcd_write_reg(0xB102, 0x0D);
		/* AVEE ratio */
		lcd_write_reg(0xB700, 0x34);
		lcd_write_reg(0xB701, 0x34);
		lcd_write_reg(0xB702, 0x34);
		/* VCL -2.5V */
		lcd_write_reg(0xB200, 0x00);
		lcd_write_reg(0xB201, 0x00);
		lcd_write_reg(0xB202, 0x00);
		/* VCL ratio */
		lcd_write_reg(0xB800, 0x24);
		lcd_write_reg(0xB801, 0x24);
		lcd_write_reg(0xB802, 0x24);
		/* VGH 15V (Free pump) */
		lcd_write_reg(0xBF00, 0x01);
		lcd_write_reg(0xB300, 0x0F);
		lcd_write_reg(0xB301, 0x0F);
		lcd_write_reg(0xB302, 0x0F);
		/* VGH ratio */
		lcd_write_reg(0xB900, 0x34);
		lcd_write_reg(0xB901, 0x34);
		lcd_write_reg(0xB902, 0x34);
		/* VGL_REG -10V */
		lcd_write_reg(0xB500, 0x08);
		lcd_write_reg(0xB501, 0x08);
		lcd_write_reg(0xB502, 0x08);
		lcd_write_reg(0xC200, 0x03);
		/* VGLX ratio */
		lcd_write_reg(0xBA00, 0x24);
		lcd_write_reg(0xBA01, 0x24);
		lcd_write_reg(0xBA02, 0x24);
		/* VGMP/VGSP 4.5V/0V */
		lcd_write_reg(0xBC00, 0x00);
		lcd_write_reg(0xBC01, 0x78);
		lcd_write_reg(0xBC02, 0x00);
		/* VGMN/VGSN -4.5V/0V */
		lcd_write_reg(0xBD00, 0x00);
		lcd_write_reg(0xBD01, 0x78);
		lcd_write_reg(0xBD02, 0x00);
		/* VCOM */
		lcd_write_reg(0xBE00, 0x00);
		lcd_write_reg(0xBE01, 0x64);
		/* Gamma Setting */
		lcd_write_reg(0xD100, 0x00);
		lcd_write_reg(0xD101, 0x33);
		lcd_write_reg(0xD102, 0x00);
		lcd_write_reg(0xD103, 0x34);
		lcd_write_reg(0xD104, 0x00);
		lcd_write_reg(0xD105, 0x3A);
		lcd_write_reg(0xD106, 0x00);
		lcd_write_reg(0xD107, 0x4A);
		lcd_write_reg(0xD108, 0x00);
		lcd_write_reg(0xD109, 0x5C);
		lcd_write_reg(0xD10A, 0x00);
		lcd_write_reg(0xD10B, 0x81);
		lcd_write_reg(0xD10C, 0x00);
		lcd_write_reg(0xD10D, 0xA6);
		lcd_write_reg(0xD10E, 0x00);
		lcd_write_reg(0xD10F, 0xE5);
		lcd_write_reg(0xD110, 0x01);
		lcd_write_reg(0xD111, 0x13);
		lcd_write_reg(0xD112, 0x01);
		lcd_write_reg(0xD113, 0x54);
		lcd_write_reg(0xD114, 0x01);
		lcd_write_reg(0xD115, 0x82);
		lcd_write_reg(0xD116, 0x01);
		lcd_write_reg(0xD117, 0xCA);
		lcd_write_reg(0xD118, 0x02);
		lcd_write_reg(0xD119, 0x00);
		lcd_write_reg(0xD11A, 0x02);
		lcd_write_reg(0xD11B, 0x01);
		lcd_write_reg(0xD11C, 0x02);
		lcd_write_reg(0xD11D, 0x34);
		lcd_write_reg(0xD11E, 0x02);
		lcd_write_reg(0xD11F, 0x67);
		lcd_write_reg(0xD120, 0x02);
		lcd_write_reg(0xD121, 0x84);
		lcd_write_reg(0xD122, 0x02);
		lcd_write_reg(0xD123, 0xA4);
		lcd_write_reg(0xD124, 0x02);
		lcd_write_reg(0xD125, 0xB7);
		lcd_write_reg(0xD126, 0x02);
		lcd_write_reg(0xD127, 0xCF);
		lcd_write_reg(0xD128, 0x02);
		lcd_write_reg(0xD129, 0xDE);
		lcd_write_reg(0xD12A, 0x02);
		lcd_write_reg(0xD12B, 0xF2);
		lcd_write_reg(0xD12C, 0x02);
		lcd_write_reg(0xD12D, 0xFE);
		lcd_write_reg(0xD12E, 0x03);
		lcd_write_reg(0xD12F, 0x10);
		lcd_write_reg(0xD130, 0x03);
		lcd_write_reg(0xD131, 0x33);
		lcd_write_reg(0xD132, 0x03);
		lcd_write_reg(0xD133, 0x6D);
		lcd_write_reg(0xD200, 0x00);
		lcd_write_reg(0xD201, 0x33);
		lcd_write_reg(0xD202, 0x00);
		lcd_write_reg(0xD203, 0x34);
		lcd_write_reg(0xD204, 0x00);
		lcd_write_reg(0xD205, 0x3A);
		lcd_write_reg(0xD206, 0x00);
		lcd_write_reg(0xD207, 0x4A);
		lcd_write_reg(0xD208, 0x00);
		lcd_write_reg(0xD209, 0x5C);
		lcd_write_reg(0xD20A, 0x00);

		lcd_write_reg(0xD20B, 0x81);
		lcd_write_reg(0xD20C, 0x00);
		lcd_write_reg(0xD20D, 0xA6);
		lcd_write_reg(0xD20E, 0x00);
		lcd_write_reg(0xD20F, 0xE5);
		lcd_write_reg(0xD210, 0x01);
		lcd_write_reg(0xD211, 0x13);
		lcd_write_reg(0xD212, 0x01);
		lcd_write_reg(0xD213, 0x54);
		lcd_write_reg(0xD214, 0x01);
		lcd_write_reg(0xD215, 0x82);
		lcd_write_reg(0xD216, 0x01);
		lcd_write_reg(0xD217, 0xCA);
		lcd_write_reg(0xD218, 0x02);
		lcd_write_reg(0xD219, 0x00);
		lcd_write_reg(0xD21A, 0x02);
		lcd_write_reg(0xD21B, 0x01);
		lcd_write_reg(0xD21C, 0x02);
		lcd_write_reg(0xD21D, 0x34);
		lcd_write_reg(0xD21E, 0x02);
		lcd_write_reg(0xD21F, 0x67);
		lcd_write_reg(0xD220, 0x02);
		lcd_write_reg(0xD221, 0x84);
		lcd_write_reg(0xD222, 0x02);
		lcd_write_reg(0xD223, 0xA4);
		lcd_write_reg(0xD224, 0x02);
		lcd_write_reg(0xD225, 0xB7);
		lcd_write_reg(0xD226, 0x02);
		lcd_write_reg(0xD227, 0xCF);
		lcd_write_reg(0xD228, 0x02);
		lcd_write_reg(0xD229, 0xDE);
		lcd_write_reg(0xD22A, 0x02);
		lcd_write_reg(0xD22B, 0xF2);
		lcd_write_reg(0xD22C, 0x02);
		lcd_write_reg(0xD22D, 0xFE);
		lcd_write_reg(0xD22E, 0x03);
		lcd_write_reg(0xD22F, 0x10);
		lcd_write_reg(0xD230, 0x03);
		lcd_write_reg(0xD231, 0x33);
		lcd_write_reg(0xD232, 0x03);
		lcd_write_reg(0xD233, 0x6D);
		lcd_write_reg(0xD300, 0x00);
		lcd_write_reg(0xD301, 0x33);
		lcd_write_reg(0xD302, 0x00);
		lcd_write_reg(0xD303, 0x34);
		lcd_write_reg(0xD304, 0x00);
		lcd_write_reg(0xD305, 0x3A);
		lcd_write_reg(0xD306, 0x00);
		lcd_write_reg(0xD307, 0x4A);
		lcd_write_reg(0xD308, 0x00);
		lcd_write_reg(0xD309, 0x5C);
		lcd_write_reg(0xD30A, 0x00);
				      
		lcd_write_reg(0xD30B, 0x81);
		lcd_write_reg(0xD30C, 0x00);
		lcd_write_reg(0xD30D, 0xA6);
		lcd_write_reg(0xD30E, 0x00);
		lcd_write_reg(0xD30F, 0xE5);
		lcd_write_reg(0xD310, 0x01);
		lcd_write_reg(0xD311, 0x13);
		lcd_write_reg(0xD312, 0x01);
		lcd_write_reg(0xD313, 0x54);
		lcd_write_reg(0xD314, 0x01);
		lcd_write_reg(0xD315, 0x82);
		lcd_write_reg(0xD316, 0x01);
		lcd_write_reg(0xD317, 0xCA);
		lcd_write_reg(0xD318, 0x02);
		lcd_write_reg(0xD319, 0x00);
		lcd_write_reg(0xD31A, 0x02);
		lcd_write_reg(0xD31B, 0x01);
		lcd_write_reg(0xD31C, 0x02);
		lcd_write_reg(0xD31D, 0x34);
		lcd_write_reg(0xD31E, 0x02);
		lcd_write_reg(0xD31F, 0x67);
		lcd_write_reg(0xD320, 0x02);
		lcd_write_reg(0xD321, 0x84);
		lcd_write_reg(0xD322, 0x02);
		lcd_write_reg(0xD323, 0xA4);
		lcd_write_reg(0xD324, 0x02);
		lcd_write_reg(0xD325, 0xB7);
		lcd_write_reg(0xD326, 0x02);
		lcd_write_reg(0xD327, 0xCF);
		lcd_write_reg(0xD328, 0x02);
		lcd_write_reg(0xD329, 0xDE);
		lcd_write_reg(0xD32A, 0x02);
		lcd_write_reg(0xD32B, 0xF2);
		lcd_write_reg(0xD32C, 0x02);
		lcd_write_reg(0xD32D, 0xFE);
		lcd_write_reg(0xD32E, 0x03);
		lcd_write_reg(0xD32F, 0x10);
		lcd_write_reg(0xD330, 0x03);
		lcd_write_reg(0xD331, 0x33);
		lcd_write_reg(0xD332, 0x03);
		lcd_write_reg(0xD333, 0x6D);
		lcd_write_reg(0xD400, 0x00);
		lcd_write_reg(0xD401, 0x33);
		lcd_write_reg(0xD402, 0x00);
		lcd_write_reg(0xD403, 0x34);
		lcd_write_reg(0xD404, 0x00);
		lcd_write_reg(0xD405, 0x3A);
		lcd_write_reg(0xD406, 0x00);
		lcd_write_reg(0xD407, 0x4A);
		lcd_write_reg(0xD408, 0x00);
		lcd_write_reg(0xD409, 0x5C);
		lcd_write_reg(0xD40A, 0x00);
		lcd_write_reg(0xD40B, 0x81);
				      
		lcd_write_reg(0xD40C, 0x00);
		lcd_write_reg(0xD40D, 0xA6);
		lcd_write_reg(0xD40E, 0x00);
		lcd_write_reg(0xD40F, 0xE5);
		lcd_write_reg(0xD410, 0x01);
		lcd_write_reg(0xD411, 0x13);
		lcd_write_reg(0xD412, 0x01);
		lcd_write_reg(0xD413, 0x54);
		lcd_write_reg(0xD414, 0x01);
		lcd_write_reg(0xD415, 0x82);
		lcd_write_reg(0xD416, 0x01);
		lcd_write_reg(0xD417, 0xCA);
		lcd_write_reg(0xD418, 0x02);
		lcd_write_reg(0xD419, 0x00);
		lcd_write_reg(0xD41A, 0x02);
		lcd_write_reg(0xD41B, 0x01);
		lcd_write_reg(0xD41C, 0x02);
		lcd_write_reg(0xD41D, 0x34);
		lcd_write_reg(0xD41E, 0x02);
		lcd_write_reg(0xD41F, 0x67);
		lcd_write_reg(0xD420, 0x02);
		lcd_write_reg(0xD421, 0x84);
		lcd_write_reg(0xD422, 0x02);
		lcd_write_reg(0xD423, 0xA4);
		lcd_write_reg(0xD424, 0x02);
		lcd_write_reg(0xD425, 0xB7);
		lcd_write_reg(0xD426, 0x02);
		lcd_write_reg(0xD427, 0xCF);
		lcd_write_reg(0xD428, 0x02);
		lcd_write_reg(0xD429, 0xDE);
		lcd_write_reg(0xD42A, 0x02);
		lcd_write_reg(0xD42B, 0xF2);
		lcd_write_reg(0xD42C, 0x02);
		lcd_write_reg(0xD42D, 0xFE);
		lcd_write_reg(0xD42E, 0x03);
		lcd_write_reg(0xD42F, 0x10);
		lcd_write_reg(0xD430, 0x03);
		lcd_write_reg(0xD431, 0x33);
		lcd_write_reg(0xD432, 0x03);
		lcd_write_reg(0xD433, 0x6D);
		lcd_write_reg(0xD500, 0x00);
		lcd_write_reg(0xD501, 0x33);
		lcd_write_reg(0xD502, 0x00);
		lcd_write_reg(0xD503, 0x34);
		lcd_write_reg(0xD504, 0x00);
		lcd_write_reg(0xD505, 0x3A);
		lcd_write_reg(0xD506, 0x00);
		lcd_write_reg(0xD507, 0x4A);
		lcd_write_reg(0xD508, 0x00);
		lcd_write_reg(0xD509, 0x5C);
		lcd_write_reg(0xD50A, 0x00);
		lcd_write_reg(0xD50B, 0x81);
				      
		lcd_write_reg(0xD50C, 0x00);
		lcd_write_reg(0xD50D, 0xA6);
		lcd_write_reg(0xD50E, 0x00);
		lcd_write_reg(0xD50F, 0xE5);
		lcd_write_reg(0xD510, 0x01);
		lcd_write_reg(0xD511, 0x13);
		lcd_write_reg(0xD512, 0x01);
		lcd_write_reg(0xD513, 0x54);
		lcd_write_reg(0xD514, 0x01);
		lcd_write_reg(0xD515, 0x82);
		lcd_write_reg(0xD516, 0x01);
		lcd_write_reg(0xD517, 0xCA);
		lcd_write_reg(0xD518, 0x02);
		lcd_write_reg(0xD519, 0x00);
		lcd_write_reg(0xD51A, 0x02);
		lcd_write_reg(0xD51B, 0x01);
		lcd_write_reg(0xD51C, 0x02);
		lcd_write_reg(0xD51D, 0x34);
		lcd_write_reg(0xD51E, 0x02);
		lcd_write_reg(0xD51F, 0x67);
		lcd_write_reg(0xD520, 0x02);
		lcd_write_reg(0xD521, 0x84);
		lcd_write_reg(0xD522, 0x02);
		lcd_write_reg(0xD523, 0xA4);
		lcd_write_reg(0xD524, 0x02);
		lcd_write_reg(0xD525, 0xB7);
		lcd_write_reg(0xD526, 0x02);
		lcd_write_reg(0xD527, 0xCF);
		lcd_write_reg(0xD528, 0x02);
		lcd_write_reg(0xD529, 0xDE);
		lcd_write_reg(0xD52A, 0x02);
		lcd_write_reg(0xD52B, 0xF2);
		lcd_write_reg(0xD52C, 0x02);
		lcd_write_reg(0xD52D, 0xFE);
		lcd_write_reg(0xD52E, 0x03);
		lcd_write_reg(0xD52F, 0x10);
		lcd_write_reg(0xD530, 0x03);
		lcd_write_reg(0xD531, 0x33);
		lcd_write_reg(0xD532, 0x03);
		lcd_write_reg(0xD533, 0x6D);
		lcd_write_reg(0xD600, 0x00);
		lcd_write_reg(0xD601, 0x33);
		lcd_write_reg(0xD602, 0x00);
		lcd_write_reg(0xD603, 0x34);
		lcd_write_reg(0xD604, 0x00);
		lcd_write_reg(0xD605, 0x3A);
		lcd_write_reg(0xD606, 0x00);
		lcd_write_reg(0xD607, 0x4A);
		lcd_write_reg(0xD608, 0x00);
		lcd_write_reg(0xD609, 0x5C);
		lcd_write_reg(0xD60A, 0x00);
		lcd_write_reg(0xD60B, 0x81);
				      
		lcd_write_reg(0xD60C, 0x00);
		lcd_write_reg(0xD60D, 0xA6);
		lcd_write_reg(0xD60E, 0x00);
		lcd_write_reg(0xD60F, 0xE5);
		lcd_write_reg(0xD610, 0x01);
		lcd_write_reg(0xD611, 0x13);
		lcd_write_reg(0xD612, 0x01);
		lcd_write_reg(0xD613, 0x54);
		lcd_write_reg(0xD614, 0x01);
		lcd_write_reg(0xD615, 0x82);
		lcd_write_reg(0xD616, 0x01);
		lcd_write_reg(0xD617, 0xCA);
		lcd_write_reg(0xD618, 0x02);
		lcd_write_reg(0xD619, 0x00);
		lcd_write_reg(0xD61A, 0x02);
		lcd_write_reg(0xD61B, 0x01);
		lcd_write_reg(0xD61C, 0x02);
		lcd_write_reg(0xD61D, 0x34);
		lcd_write_reg(0xD61E, 0x02);
		lcd_write_reg(0xD61F, 0x67);
		lcd_write_reg(0xD620, 0x02);
		lcd_write_reg(0xD621, 0x84);
		lcd_write_reg(0xD622, 0x02);
		lcd_write_reg(0xD623, 0xA4);
		lcd_write_reg(0xD624, 0x02);
		lcd_write_reg(0xD625, 0xB7);
		lcd_write_reg(0xD626, 0x02);
		lcd_write_reg(0xD627, 0xCF);
		lcd_write_reg(0xD628, 0x02);
		lcd_write_reg(0xD629, 0xDE);
		lcd_write_reg(0xD62A, 0x02);
		lcd_write_reg(0xD62B, 0xF2);
		lcd_write_reg(0xD62C, 0x02);
		lcd_write_reg(0xD62D, 0xFE);
		lcd_write_reg(0xD62E, 0x03);
		lcd_write_reg(0xD62F, 0x10);
		lcd_write_reg(0xD630, 0x03);
		lcd_write_reg(0xD631, 0x33);
		lcd_write_reg(0xD632, 0x03);
		lcd_write_reg(0xD633, 0x6D);
		/* LV2 Page 0 enable */
		lcd_write_reg(0xF000, 0x55);
		lcd_write_reg(0xF001, 0xAA);
		lcd_write_reg(0xF002, 0x52);
		lcd_write_reg(0xF003, 0x08);
		lcd_write_reg(0xF004, 0x00);
		/* Display control */
		lcd_write_reg(0xB100, 0xCC);
		lcd_write_reg(0xB101, 0x00);
		/* Source hold time */
		lcd_write_reg(0xB600, 0x05);
		/* Gate EQ control */
		lcd_write_reg(0xB700, 0x70);
		lcd_write_reg(0xB701, 0x70);
		/* Source EQ control (Mode 2) */
		lcd_write_reg(0xB800, 0x01);
		lcd_write_reg(0xB801, 0x03);
		lcd_write_reg(0xB802, 0x03);
		lcd_write_reg(0xB803, 0x03);
		/* Inversion mode (2-dot) */
		lcd_write_reg(0xBC00, 0x02);
		lcd_write_reg(0xBC01, 0x00);
		lcd_write_reg(0xBC02, 0x00);
		/* Timing control 4H w/ 4-delay */
		lcd_write_reg(0xC900, 0xD0);
		lcd_write_reg(0xC901, 0x02);
		lcd_write_reg(0xC902, 0x50);
		lcd_write_reg(0xC903, 0x50);
		lcd_write_reg(0xC904, 0x50);
		lcd_write_reg(0x3500, 0x00);
		/* 16-bit/pixel */
		lcd_write_reg(0x3A00, 0x55);  
		lcd_wr_reg(0x1100);
		/* delay 120 */
		for ( i = 0xffff; i > 0; --i);
		lcd_wr_reg(0x2900);  
		
		/* Config write time */  	 							    
		EBI_Bank1E->BWTR[0] &= ~(0XF << 0); /* Clear address setuptime */	 
		EBI_Bank1E->BWTR[0] &= ~(0XF << 8); /* Clear data save time */
		EBI_Bank1E->BWTR[0] |= 4 << 0;	    /* Address setuptime (ADDSET) 4*HCLK = 21ns */  	 
		EBI_Bank1E->BWTR[0]|= 4 << 8; 	    /* Data save time(DATAST) 5.2ns*4HCLK = 21ns */
		
		lcd_display_dir(0);		    /* Default Vertical Screen */
		lcd_clear(WHITE);
	} else if (lcd_device.id == 0x9488) {
		lcd_wr_reg(0XF7);
		lcd_wr_data(0xA9);
		lcd_wr_data(0x51);
		lcd_wr_data(0x2C);
		lcd_wr_data(0x82);

		lcd_wr_reg(0xC0);
		lcd_wr_data(0x11);
		lcd_wr_data(0x09);
	
		lcd_wr_reg(0xC1);
		lcd_wr_data(0x41);
	
		lcd_wr_reg(0XC5);
		lcd_wr_data(0x00);
		lcd_wr_data(0x0A);
		lcd_wr_data(0x80);
	
		lcd_wr_reg(0xB1);
		lcd_wr_data(0xB0);
		lcd_wr_data(0x11);
	
		lcd_wr_reg(0xB4);
		lcd_wr_data(0x02);
	
		lcd_wr_reg(0xB6);
		lcd_wr_data(0x02);
		lcd_wr_data(0x22);
	
		lcd_wr_reg(0xB7);
		lcd_wr_data(0xc6);
	
		lcd_wr_reg(0xBE);
		lcd_wr_data(0x00);
		lcd_wr_data(0x04);
	
		lcd_wr_reg(0xE9);
		lcd_wr_data(0x00);
	
		lcd_wr_reg(0x36);
		lcd_wr_data(0x08);
	
		lcd_wr_reg(0x3A);
		lcd_wr_data(0x55);
	
		lcd_wr_reg(0xE0);
		lcd_wr_data(0x00);
		lcd_wr_data(0x07);
		lcd_wr_data(0x10);
		lcd_wr_data(0x09);
		lcd_wr_data(0x17);
		lcd_wr_data(0x0B);
		lcd_wr_data(0x41);
		lcd_wr_data(0x89);
		lcd_wr_data(0x4B);
		lcd_wr_data(0x0A);
		lcd_wr_data(0x0C);
		lcd_wr_data(0x0E);
		lcd_wr_data(0x18);
		lcd_wr_data(0x1B);
		lcd_wr_data(0x0F);

		lcd_wr_reg(0XE1);
		lcd_wr_data(0x00);
		lcd_wr_data(0x17);
		lcd_wr_data(0x1A);
		lcd_wr_data(0x04);
		lcd_wr_data(0x0E);
		lcd_wr_data(0x06);
		lcd_wr_data(0x2F);
		lcd_wr_data(0x45);
		lcd_wr_data(0x43);
		lcd_wr_data(0x02);
		lcd_wr_data(0x0A);
		lcd_wr_data(0x09);
		lcd_wr_data(0x32);
		lcd_wr_data(0x36);
		lcd_wr_data(0x0F);

		lcd_wr_reg(0x11);
		for ( i = 0xffff; i > 0; --i);
		lcd_wr_reg(0x29);
		
		/* Config write time */  	 							    
		EBI_Bank1E->BWTR[0] &= ~(0XF << 0); /* Clear address setuptime */	 
		EBI_Bank1E->BWTR[0] &= ~(0XF << 8); /* Clear data save time */
		EBI_Bank1E->BWTR[0] |= 4 << 0;	    /* Address setuptime (ADDSET) 4*HCLK = 21ns */  	 
		EBI_Bank1E->BWTR[0]|= 4 << 8; 	    /* Data save time(DATAST) 5.2ns*4HCLK = 21ns */
		
		lcd_display_dir(0);
		lcd_clear(WHITE);
	}
}


/**
  * @brief  Read (x, y) point color value
  * @param  x: X coordinate
  * @param  y: Y coordinate 
  * @retval Color value
  */
uint32_t lcd_read_point(uint16_t x, uint16_t y)
{
 	uint16_t r = 0, g = 0, b = 0;
	
	if(x >= lcd_device.width || y >= lcd_device.height) 
		return 0;	
	lcd_set_cursor(x, y);	    
	if(lcd_device.id == 0x9488)
		lcd_wr_reg(0x2E);
	else if(lcd_device.id == 0X5510)
		lcd_wr_reg(0X2E00);	
 	r = lcd_rd_data();  		  						
	opt_delay(2);	                                    
	b = lcd_rd_data(); 
	g = r & 0XFF;		
	g <<= 8; 
	return (((r >> 11) << 11)|((g >> 10) <<5 )|(b >> 11));
}	

/**
  * @brief  LCD display on
  * @retval None
  */
void lcd_display_on(void)
{					   
	if(lcd_device.id == 0x9488)
		lcd_wr_reg(0x29);
	else if(lcd_device.id==0X5510)
		lcd_wr_reg(0X2900); 
}	 

/**
  * @brief  LCD display off
  * @retval None
  */
void lcd_display_off(void)
{	
	if(lcd_device.id == 0x9488)
		lcd_wr_reg(0x28);
	else if(lcd_device.id==0X5510)
		lcd_wr_reg(0X2800);	
}   

/**
  * @brief  LCD set cursor
  * @param  xpos: X coordinate
  * @param  ypos: Y coordinate 
  * @retval None
  */
void lcd_set_cursor(uint16_t xpos, uint16_t ypos)
{	
	if (lcd_device.id == 0X9488) {
		lcd_wr_reg(lcd_device.setxcmd); 
		lcd_wr_data(xpos >> 8);
		lcd_wr_data(xpos & 0XFF); 
		lcd_wr_data((lcd_device.width-1) >> 8);
		lcd_wr_data((lcd_device.width-1) & 0XFF);
		lcd_wr_reg(lcd_device.setycmd); 
		lcd_wr_data(ypos >> 8);
		lcd_wr_data(ypos & 0XFF);
		lcd_wr_data((lcd_device.height - 1) >> 8);
		lcd_wr_data((lcd_device.height - 1) & 0XFF);
	}
	else if(lcd_device.id==0X5510) {
		lcd_wr_reg(lcd_device.setxcmd); 
		lcd_wr_data(xpos >> 8); 		
		lcd_wr_reg(lcd_device.setxcmd + 1); 
		lcd_wr_data(xpos & 0XFF);			 
		lcd_wr_reg(lcd_device.setycmd); 
		lcd_wr_data(ypos >> 8);  		
		lcd_wr_reg(lcd_device.setycmd + 1); 
		lcd_wr_data(ypos & 0XFF);			
	} 
} 		 

/**
  * @brief  Set LCD scan direction
  * @param  direct: Scan mode(0 ~ 7) 
  * @retval None
  */
void lcd_scan_dir(uint8_t direct)
{
	uint16_t regval = 0;
	uint16_t dirreg = 0;
        
	if(lcd_device.id == 0X5510 || lcd_device.id == 0X9488) {
		switch(direct) {
			case L2R_U2D:
				regval |= (0 << 7) | (0 << 6) | (0 << 5); 
				break;                                
			case L2R_D2U:                                 
				regval |= (1 << 7) | (0 << 6) | (0 << 5); 
				break;                                
			case R2L_U2D:                                 
				regval |= (0 << 7) | (1 << 6) | (0 << 5); 
				break;
			case R2L_D2U:
				regval |= (1 << 7) | (1 << 6) | (0 << 5); 
				break; 	                              
			case U2D_L2R:                                 
				regval |= (0 << 7) | (0 << 6) | (1 << 5); 
				break;                                
			case U2D_R2L:                                 
				regval |= (0 << 7) | (1 << 6) | (1 << 5); 
				break;                                
			case D2U_L2R:                                 
				regval |= (1 << 7) | (0 << 6) | (1 << 5); 
				break;                                
			case D2U_R2L:                                 
				regval |= (1 << 7) | (1 << 6) | (1 << 5); 
				break;	 
		}
		if(lcd_device.id == 0X5510) 
			dirreg = 0X3600;
		else 
			dirreg = 0X36;
 		if(lcd_device.id == 0X9488) 
			regval |= 0X08;
 		lcd_write_reg(dirreg, regval);
		
		if(lcd_device.id == 0X5510) {
			lcd_wr_reg(lcd_device.setxcmd); 
			lcd_wr_data(0); 
			lcd_wr_reg(lcd_device.setxcmd + 1); 
			lcd_wr_data(0); 
			lcd_wr_reg(lcd_device.setxcmd + 2); 
			lcd_wr_data((lcd_device.width - 1) >> 8); 
			lcd_wr_reg(lcd_device.setxcmd + 3); 
			lcd_wr_data((lcd_device.width-1) & 0XFF); 
			lcd_wr_reg(lcd_device.setycmd); 
			lcd_wr_data(0); 
			lcd_wr_reg(lcd_device.setycmd + 1); 
			lcd_wr_data(0); 
			lcd_wr_reg(lcd_device.setycmd + 2); 
			lcd_wr_data((lcd_device.height - 1) >> 8); 
			lcd_wr_reg(lcd_device.setycmd + 3); 
			lcd_wr_data((lcd_device.height - 1) & 0XFF);
		} else { /* lcd_device.id == 0x9488 */
			lcd_wr_reg(lcd_device.setxcmd); 
			lcd_wr_data(0);
			lcd_wr_data(0);
			lcd_wr_data((lcd_device.width-1)>>8);
			lcd_wr_data((lcd_device.width-1)&0XFF);
			lcd_wr_reg(lcd_device.setycmd); 
			lcd_wr_data(0);
			lcd_wr_data(0);
			lcd_wr_data((lcd_device.height-1)>>8);
			lcd_wr_data((lcd_device.height-1)&0XFF);  
		
		}
  	} 
} 

/**
  * @brief  Draw a point with special color
  * @param  x: X coordinate
  * @param  y: Y coordinate 
  * @param  color: color value 
  * @retval None
  */
void lcd_draw_point_color(uint16_t x, uint16_t y, uint16_t color)
{
	lcd_set_cursor(x, y);
	lcd_write_ram_prepare();
	LCD_ADDR->lcd_ram = color;
}


/**
  * @brief  Draw a point with default color
  * @param  x: X coordinate
  * @param  y: Y coordinate 
  * @param  color: color value 
  * @retval None
  */
void lcd_draw_point(uint16_t x, uint16_t y)
{ 
	lcd_set_cursor(x,y);		
	lcd_write_ram_prepare();	
	LCD_ADDR->lcd_ram = lcd_device.front_color;  
}

/**
  * @brief  Draw a point with special color fastly
  * @param  x: X coordinate
  * @param  y: Y coordinate 
  * @param  color: color value 
  * @retval None
  */
void lcd_fast_draw_point(uint16_t x,uint16_t y,uint32_t color)
{	   
	if (lcd_device.id == 0x5510) {
		lcd_wr_reg(lcd_device.setxcmd); 
		lcd_wr_data(x >> 8);  
		lcd_wr_reg(lcd_device.setxcmd + 1); 
		lcd_wr_data(x & 0XFF);	  
		lcd_wr_reg(lcd_device.setycmd); 
		lcd_wr_data(y >> 8);  
		lcd_wr_reg(lcd_device.setycmd + 1); 
		lcd_wr_data(y & 0XFF); 
	} else if (lcd_device.id == 0x9488) {
		lcd_wr_reg(lcd_device.setxcmd); 
		lcd_wr_data(x >> 8);
		lcd_wr_data(x & 0XFF); 		
		lcd_wr_data(x >> 8);
		lcd_wr_data(x & 0XFF); 		
		lcd_wr_reg(lcd_device.setycmd); 
		lcd_wr_data(y >> 8);
		lcd_wr_data(y & 0XFF); 		
		lcd_wr_data(y >> 8);
		lcd_wr_data(y & 0XFF); 
	}
		 
	LCD_ADDR->lcd_reg = lcd_device.wramcmd; 
	LCD_ADDR->lcd_ram = color; 
}

/**
  * @brief Set LCD display direction 
  * @param  dir: 0,vertical 1,horizontal 
  * @retval None
  */
void lcd_display_dir(uint8_t direct)
{
	lcd_device.dir = direct;
	
	if(direct == 0) {
		if (lcd_device.id == 0X5510) {
			lcd_device.wramcmd = 0X2C00;
			lcd_device.setxcmd = 0X2A00;
			lcd_device.setycmd = 0X2B00; 
			lcd_device.width  = 480;
			lcd_device.height = 800;
		} else { /* 0x9488 */
			lcd_device.wramcmd = 0X2C;
			lcd_device.setxcmd = 0X2A;
			lcd_device.setycmd = 0X2B;
			lcd_device.width  = 320;
			lcd_device.height = 480;
		}
	} else { 
		if (lcd_device.id == 0x5510) {
			lcd_device.wramcmd = 0X2C00;
			lcd_device.setxcmd = 0X2A00;
			lcd_device.setycmd = 0X2B00; 
			lcd_device.width  = 800;
			lcd_device.height = 480;
		} else { /* 0X9488 */
			lcd_device.wramcmd = 0X2C;
			lcd_device.setxcmd = 0X2A;
			lcd_device.setycmd = 0X2B;
			lcd_device.width  = 480;
			lcd_device.height = 320; 
		}
	}
	lcd_device.front_color = 0XFF000000;
	lcd_device.back_color  = 0XFFFFFFFF;
	lcd_scan_dir(LCD_DFT_SCAN_DIR);	
}	 

/**
  * @brief  Display a window on the LCD
  * @param  sx: Start X-coordinate
  * @param  sy: Start Y-coordinate 
  * @param  width: Window width 
  * @param  height: Window height 
  * @retval None
  */
void lcd_set_window(uint16_t sx, uint16_t sy, uint16_t width, uint16_t height)
{     
	uint16_t twidth, theight;
	
	twidth = sx + width - 1;
	theight = sy + height - 1;
	
	if(lcd_device.id == 0X9488) {
		lcd_wr_reg(lcd_device.setxcmd); 
		lcd_wr_data(sx >> 8); 
		lcd_wr_data(sx & 0XFF);	 
		lcd_wr_data(twidth >> 8); 
		lcd_wr_data(twidth & 0XFF);  
		lcd_wr_reg(lcd_device.setycmd); 
		lcd_wr_data(sy >> 8); 
		lcd_wr_data(sy & 0XFF); 
		lcd_wr_data(theight >> 8); 
		lcd_wr_data(theight & 0XFF);
		lcd_wr_reg(lcd_device.wramcmd);	/* Need to check */
	}
	else if (lcd_device.id == 0X5510) {
		/* Column Address Set (2A00h~2A03h) */
		lcd_wr_reg(lcd_device.setxcmd); 
		lcd_wr_data(sx >> 8);  
		lcd_wr_reg(lcd_device.setxcmd + 1); 
		lcd_wr_data(sx & 0XFF);	  
		lcd_wr_reg(lcd_device.setxcmd + 2); 
		lcd_wr_data(twidth >> 8);   
		lcd_wr_reg(lcd_device.setxcmd + 3); 
		lcd_wr_data(twidth & 0XFF);
		/*  Row Address Set (2B00h~2B03h) */
		lcd_wr_reg(lcd_device.setycmd); 
		lcd_wr_data(sy >> 8);   
		lcd_wr_reg(lcd_device.setycmd + 1); 
		lcd_wr_data(sy & 0XFF);  
		lcd_wr_reg(lcd_device.setycmd + 2); 
		lcd_wr_data(theight >> 8);   
		lcd_wr_reg(lcd_device.setycmd + 3); 
		lcd_wr_data(theight & 0XFF);
		/* Memory Write (2C00h) */
		lcd_wr_reg(lcd_device.wramcmd);
	}
}

/**
  * @brief  Display a rectangle with dma
  * @param  sx: Start X-coordinate
  * @param  sy: Start Y-coordinate 
  * @param  width: Rectangle width 
  * @param  height: Rectangle height
  * @param  psbuf: Color buffer address 
  * @retval None
  */
void lcd_rectangle_flush(uint16_t sx, uint16_t sy, uint16_t width, uint16_t height, lv_color_t *psbuf)
{
	uint32_t i = 0, j = 0;

	lcd_set_window( sx, sy, width, height);

	if (psbuf == NULL) 
		return ;
	
	for (i = 0; i < width; ++i) {
		for (j = 0; j < height ; ++j) {
			LCD_ADDR->lcd_ram = psbuf->full;
			psbuf++;
		}	
	}	
}	

/**
  * @brief  Clear LCD with special color
  * @param  color: Color value
  * @retval None
  */
void lcd_clear(uint32_t color)
{
	uint32_t index = 0; 	
	uint32_t totalpoint = lcd_device.width; 
	
	totalpoint *= lcd_device.height; 			
	lcd_set_cursor(0x00, 0x0000);			
	lcd_write_ram_prepare();     		 	  
	for(index=0; index < totalpoint; index++) {
		LCD_ADDR->lcd_ram = color;	
	} 
}  

/**
  * @brief  Fill a rectangle with special color
  * @param  xs: Start X-coordinate
  * @param  ys: Start Y-coordinate
  * @param  xe: End X-coordinate
  * @param  ye: End Y-coordinate 
  * @param  color: Color value
  * @retval None
  */
void lcd_fill(uint16_t xs, uint16_t ys, uint16_t xe, uint16_t ye, uint32_t color)
{          
	uint16_t i,j;
	uint16_t xlen = 0; 
	
	xlen = xe - xs + 1;	 
	for(i = ys; i <= ye; i++) {
		lcd_set_cursor(xs, i);      				
		lcd_write_ram_prepare();     			 
		for(j=0; j < xlen; j++)
			LCD_ADDR->lcd_ram = color;	    
	}  
} 

/**
  * @brief  Fill a rectangle with different color
  * @param  xs: Start X-coordinate
  * @param  ys: Start Y-coordinate
  * @param  xe: End X-coordinate
  * @param  ye: End Y-coordinate 
  * @param  color: point to color buffer address
  * @retval None
  */
void lcd_fill_colors(uint16_t xs, uint16_t ys, uint16_t xe, uint16_t ye, uint16_t *color)
{  
	uint16_t height, width;
	uint16_t i, j;
	
	width = xe - xs + 1;
	height = ye - ys + 1;			
	for(i=0; i < height; i++) {
		lcd_set_cursor(xs, ys + i);   	
		lcd_write_ram_prepare();     
		for(j = 0;j < width; j++) 
			LCD_ADDR->lcd_ram = color[i * width + j];
	} 
}  

/**
  * @brief  LCD draw a line
  * @param  xs: Start X-coordinate
  * @param  ys: Start Y-coordinate
  * @param  xe: End X-coordinate
  * @param  ye: End Y-coordinate 
  * @retval None
  */
void lcd_draw_line(uint16_t xs, uint16_t ys, uint16_t xe, uint16_t ye)
{
	uint16_t t; 
	int xerr = 0, yerr = 0, delta_x, delta_y, distance; 
	int incx, incy, uRow, uCol; 
	
	delta_x = xe - xs; 
	delta_y = ye - ys; 
	uRow = xs; 
	uCol = ys; 
	if(delta_x > 0) 
		incx = 1; 
	else if(delta_x == 0) 
		incx = 0;
	else {
		incx = -1; 
		delta_x = -delta_x;
	} 
	if(delta_y > 0) 
		incy = 1; 
	else if(delta_y == 0) 
		incy = 0; 
	else {
		incy = -1; 
		delta_y = -delta_y;
	} 
	if( delta_x > delta_y) 
		distance = delta_x;  
	else distance = delta_y; 
	for(t = 0;t <= distance + 1; t++ ) {
		lcd_draw_point(uRow, uCol);
		xerr += delta_x ; 
		yerr += delta_y ; 
		if(xerr > distance) {
			xerr -= distance; 
			uRow += incx; 
		} 
		if(yerr > distance) {
			yerr -= distance; 
			uCol += incy; 
		} 
	}  
}    

/**
  * @brief  LCD draw a rectangle 
  * @param  xs: Start X-coordinate
  * @param  ys: Start Y-coordinate
  * @param  xe: End X-coordinate
  * @param  ye: End Y-coordinate 
  * @retval None
  */
void lcd_draw_rectangle(uint16_t xs, uint16_t ys, uint16_t xe, uint16_t ye)
{
	lcd_draw_line(xs, ys, xe, ys);
	lcd_draw_line(xs, ys, xs, ye);
	lcd_draw_line(xs, ye, xe, ye);
	lcd_draw_line(xe, ys, xe, ye);
}

/**
  * @brief  LCD draw a circle 
  * @param  xc: Center of circle X-coordinate
  * @param  yc: Center of circle Y-coordinate
  * @param  r: Radius
  * @retval None
  */
void lcd_draw_circle(uint16_t xc, uint16_t yc, uint8_t r)
{
	int a, b;
	int di;
	a=0; b=r;
	
	di = 3 - (r << 1);             
	while(a <= b) {
		lcd_draw_point(xc + a, yc - b);             
 		lcd_draw_point(xc + b, yc - a);                       
		lcd_draw_point(xc + b, yc + a);                           
		lcd_draw_point(xc + a, yc + b);             
		lcd_draw_point(xc - a, yc + b);                   
 		lcd_draw_point(xc - b, yc + a);             
		lcd_draw_point(xc - a, yc - b);                         
  		lcd_draw_point(xc - b, yc - a);                 	         
		a++;
		if(di < 0) di += 4*a + 6;	  
		else {
			di += 10 + 4*(a - b);   
			b--;
		} 						    
	}
} 									  

/**
  * @brief  Show a character with special size and mode 
  * @param  x: Start of X-coordinate 
  * @param  y: Start of Y-coordinate
  * @param  num: Show char
  * @param  size: Size of the char
  * @param  mode: 0 or 1
  * @retval None
  */
void lcd_show_char(uint16_t x, uint16_t y, uint8_t num, uint8_t size, uint8_t mode)
{  							  
	uint8_t temp, t1, t;
	uint16_t y0 = y;
	
	uint8_t csize = (size / 8 + ((size % 8) ? 1 : 0)) * (size / 2);		
 	num = num - ' ';
	for(t = 0; t < csize; t++) {
		if(size == 12) 
			temp = asc2_1206[num][t]; 	 	
		else if(size == 16) 
			temp = asc2_1608[num][t];	
		else if(size == 24) 
			temp = asc2_2412[num][t];	
		else if(size == 32) 
			temp = asc2_3216[num][t];	
		else 
			return;				
		for(t1=0; t1 < 8; t1++) {    
			if(temp & 0x80) 
				lcd_fast_draw_point(x, y, lcd_device.front_color);
			else if(mode == 0) 
				lcd_fast_draw_point(x, y, lcd_device.back_color);
			temp <<= 1;
			y++;
			if(y >= lcd_device.height) 
				return;		
			if((y - y0) == size) {
				y = y0;
				x++;
				if(x >= lcd_device.width) 
					return;	
				break;
			}
		}  	 
	}  	    	   	 	  
}   
			 
/**
  * @brief  Show a train of number
  * @param  x: Start of X-coordinate 
  * @param  y: Start of Y-coordinate
  * @param  num: Show numbers
  * @param  len: Length of the numbers
  * @param  size: Size of the number
  * @retval None
  */	 
void lcd_show_num(uint16_t x, uint16_t y, uint32_t num, uint8_t len, uint8_t size)
{         	
	uint8_t t,temp;
	uint8_t enshow = 0;
	
	for(t = 0;t < len; t++) {
		temp = (num / lcd_pow(10, len - t - 1)) % 10;
		if(enshow == 0 && t < (len - 1)) {
			if(temp == 0) {
				lcd_show_char(x + (size / 2) * t, y,' ', size, 0);
				continue;
			} else enshow = 1; 
		 	 
		}
	 	lcd_show_char(x + (size / 2) * t, y, temp +'0', size, 0); 
	}
} 

/**
  * @brief  Show a train of number
  * @param  x: Start of X-coordinate 
  * @param  y: Start of Y-coordinate
  * @param  num: Show numbers
  * @param  len: Length of the numbers
  * @param  size: Size of the number
  * @param  mode: 0 or 1
  * @retval None
  */
void lcd_show_xnum(uint16_t x, uint16_t y, uint32_t num, uint8_t len, uint8_t size, uint8_t mode)
{  
	uint8_t t, temp;
	uint8_t enshow = 0;
	
	for(t = 0; t < len; t++) {
		temp=(num / lcd_pow(10, len - t - 1)) % 10;
		if(enshow == 0 && t < (len - 1)) {
			if(temp == 0) {
				if(mode & 0X80) lcd_show_char(x + (size / 2) * t, y, '0', size, mode & 0X01);  
				else lcd_show_char(x + (size / 2) * t, y, ' ', size, mode & 0X01);  
 				continue;
			}else enshow = 1; 
		 	 
		}
	 	lcd_show_char(x + (size / 2) * t, y, temp + '0', size, mode & 0X01); 
	}
} 

/**
  * @brief  Show string
  * @param  x: Start of X-coordinate 
  * @param  y: Start of Y-coordinate
  * @param  width: Show numbers
  * @param  height: String height
  * @param  size: Size of the number
  * @param  p: Start address of the string
  * @retval None
  */	  
void lcd_show_string(uint16_t x, uint16_t y, uint16_t width, uint16_t height, uint8_t size, uint8_t *p)
{         
	uint8_t x0 = x;
	
	width += x;
	height += y;
	while((*p <= '~') && (*p >= ' ')) {
		if(x >= width){
			x = x0; 
			y += size;
		}
		if(y >= height) 
			break;
		lcd_show_char(x, y, *p, size, 0);
		x += size / 2;
		p++;
	}  
}

/**
  * @brief  Lcd set draw point color
  * @param  colr: draw point color
  * @retval None
  */	
void lcd_set_front_color(uint32_t colr)
{
	lcd_device.front_color = colr;
}

/**
  * @brief  Lcd set draw back color
  * @param  colr: back color
  * @retval None
  */	
void lcd_set_back_color(uint32_t colr)
{
	lcd_device.back_color = colr;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */


















