/**
  *********************************************************************************
  *
  * @file    bsp_key.c
  * @brief   KEY driver
  *
  * @version V1.0
  * @date    24 Apr 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include "ald_gpio.h"
#include "bsp_key.h"

 /** @addtogroup ES32F3xxx_BSP
   * @{
   */

/** @defgroup KEY key
  * @{
  */
 
/** @defgroup KEY_Private_Params Key Private Variables
  * @{
  */

#define GPIO_PORT_K3    GPIOA
#define GPIO_PIN_K3     GPIO_PIN_0
#define GPIO_PORT_K2    GPIOA
#define GPIO_PIN_K2     GPIO_PIN_2

static key_property_t keys_property[KEY_COUNT];
/* Key fifo variable, struct */
static key_fifo_t keys_fifo;

/**
  * @}
  */

/** @defgroup KEY_Private_Function_Prototypes Key Private Function Prototypes
  * @{
  */
static void bsp_init_key_var(void);
static void bsp_init_key_hard(void);
static void bsp_detect_key(uint8_t i);
/**
  * @}
  */

/** @defgroup KEY_Private_Function Key Private Function 
  * @{
  */
/**
  * @brief Read key2 I/O state
  * @retval None
  */
static uint8_t is_key_down3(void) 
{
	if(ald_gpio_read_pin(GPIO_PORT_K3, GPIO_PIN_K3) == RESET) 
		return 1; 
	else 
		return 0;
}

/**
  * @brief Read key2 I/O state.
  * @retval None
  */
static uint8_t is_key_down2(void) 
{
	if(ald_gpio_read_pin(GPIO_PORT_K2, GPIO_PIN_K2) == RESET) 
		return 1; 
	else return 0;
}

/**
  * @brief Init key hardware.
  * @retval None
  */
static void bsp_init_key_hard(void)
{
	gpio_init_t x;

	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_DOWN;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.odos = GPIO_PUSH_PULL;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(GPIO_PORT_K3, GPIO_PIN_K3, &x);
	ald_gpio_init(GPIO_PORT_K2, GPIO_PIN_K2, &x);
}

/**
  * @brief Init key variable.
  * @retval None
  */
static void bsp_init_key_var(void)
{
	uint8_t i;

	/* Clear key write/read pointer fifo */
	keys_fifo.read = 0;
	keys_fifo.write = 0;
	keys_fifo.read2 = 0;
	/* Set key struct default value */
	for (i = 0; i < KEY_COUNT; i++) {
		keys_property[i].long_time = KEY_LONG_TIME;			
		keys_property[i].count = KEY_FILTER_TIME / 2;		
		keys_property[i].state = 0;
#if 0		
		keys_property[i].KeyCodeDown = 3 * i + 1;		
		keys_property[i].KeyCodeUp   = 3 * i + 2;		
		keys_property[i].KeyCodeLong = 3 * i + 3;		
#endif
		keys_property[i].repeat_speed = 0;					
		keys_property[i].repeat_count = 0;						
	}

	keys_property[0].key_pressed_detect = is_key_down3;
	keys_property[1].key_pressed_detect = is_key_down2;	
}

/**
  * @brief Detect key state.
  * @param i: key pointer
  * @retval None
  */
static void bsp_detect_key(uint8_t i)
{
	key_property_t *pbtn;

	pbtn = &keys_property[i];
	if (pbtn->key_pressed_detect()) {     
		if (pbtn->count < KEY_FILTER_TIME) {
			pbtn->count = KEY_FILTER_TIME;
		}
		else if(pbtn->count < 2 * KEY_FILTER_TIME) {
			pbtn->count++;
		}
		else {
			if (pbtn->state == 0) {
				pbtn->state = 1;
				bsp_put_key((uint8_t)(3 * i + 1));
			}
			if (pbtn->long_time > 0) {
				if (pbtn->long_count < pbtn->long_time) {
					if (++pbtn->long_count == pbtn->long_time) {
						bsp_put_key((uint8_t)(3 * i + 3));
					}
				}
				else {
					if (pbtn->repeat_speed > 0) {
						if (++pbtn->repeat_count >= pbtn->repeat_speed) {
							pbtn->repeat_count = 0;
							bsp_put_key((uint8_t)(3 * i + 1));
						}
					}
				}
			}
		}
	}
	else {
		if(pbtn->count > KEY_FILTER_TIME) {
			pbtn->count = KEY_FILTER_TIME;
		}
		else if(pbtn->count != 0) {
			pbtn->count--;
		}
		else {
			if (pbtn->state == 1) {
				pbtn->state = 0;
				bsp_put_key((uint8_t)(3 * i + 2));
			}
		}
		pbtn->long_count = 0;
		pbtn->repeat_count = 0;
	}
}

/**
  * @}
  */

/** @defgroup Key_Public_Functions Key Public Functions
  * @{
  */

/**
  * @brief  Init key variable and key hardware
  * @retval None
  */	
void bsp_key_init(void)
{
	bsp_init_key_var();		
	bsp_init_key_hard();		
}

/**
  * @brief  Send a key value in fifo.
  * @param  keycode: key value
  * @retval None
  */
void bsp_put_key(uint8_t keycode)
{
	keys_fifo.buffer[keys_fifo.write] = keycode;
	if (++keys_fifo.write  >= KEY_FIFO_SIZE) {
		keys_fifo.write = 0;
	}
}

/**
  * @brief  Get a key value in fifo.
  * @retval Return the first key value
  */
uint8_t bsp_get_key(void)
{
	uint8_t ret;

	if (keys_fifo.read == keys_fifo.write) {
		return KEY_NONE;
	}
	else {
		ret = keys_fifo.buffer[keys_fifo.read];
		if (++keys_fifo.read >= KEY_FIFO_SIZE) {
			keys_fifo.read = 0; 
		}
		return ret;
	}
}

/**
  * @brief  Get a key value in fifo.
  * @retval Return the first key value
  */
uint8_t bsp_get_key2(void)
{
	uint8_t ret;

	if (keys_fifo.read2 == keys_fifo.write) {
		return KEY_NONE;
	}
	else {
		ret = keys_fifo.buffer[keys_fifo.read2];

		if (++keys_fifo.read2 >= KEY_FIFO_SIZE) {
			keys_fifo.read2 = 0;
		}
		return ret;
	}
}

/**
  * @brief  Get a key state.
  * @param  keyid: Key id
  * @retval Return key state, 1, pressed; 0, not pressed
  */
uint8_t bsp_get_key_state(key_id_e keyid)
{
	return keys_property[keyid].state;
}

/**
  * @brief  Set key parameter.
  * @param  keyid: Key id
  * @param  longtime: how long time index long-time pressed
  * @param  repeatspeed: repeat speed
  * @retval None
  */
void bsp_set_key_param(uint8_t keyid, uint16_t longtime, uint8_t  repeatspeed)
{
	keys_property[keyid].long_time = longtime;	 /* 0, not detect long-time pressed */
	keys_property[keyid].repeat_speed = repeatspeed; /* 0, not support repeat-pressed */
	keys_property[keyid].repeat_count = 0;					
}

/**
  * @brief  Clear key fifo.
  * @retval None
  */
void bsp_clear_key(void)
{
	keys_fifo.read = keys_fifo.write;
}

/**
  * @brief Scan key state periodically.
  * @retval None
  */
void bsp_key_scan(void)
{
	uint8_t i;

	for (i = 0; i < KEY_COUNT; i++) {
		bsp_detect_key(i);
	}
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

