/**
  *********************************************************************************
  *
  * @file    bsp_infrared.c
  * @brief   Infrared driver
  *
  * @version V1.0
  * @date    24 Apr 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
  
#include "bsp_infrared.h"
#include "ald_gpio.h"
#include "ald_timer.h"
  
/** @addtogroup ES32F3xxx_BSP
* @{
*/

/** @defgroup Infrared infrared 
  * @{
  */

/** @defgroup Infrared_Private_Variable Infrared Private Variable
  * @{
  */
  
typedef struct capture {
	uint8_t  capsta;
	uint32_t recval;
	uint8_t  keycnt;
} capture_sta_t;

capture_sta_t capture_sta = {0};
timer_handle_t h_tim;
timer_clock_config_t clock;
timer_ic_init_t init;

/**
  * @}
  */ 

/** @defgroup Infrared_Private_Functions Infrared Private Functions 
  * @{
  */

/**
  * @brief  Swtich infrared pin from func5 to func1, read the i/o state.
  * @retval  1 or 0
  */
static uint8_t infrared_pin_read(void)
{
	uint32_t tmp = 0;
	
	tmp = INFRARED_PORT->FUNC0;
	INFRARED_PORT->FUNC0 &= ~0XF000000;
	INFRARED_PORT->FUNC0 = 0x1000000;
	if (READ_BIT(INFRARED_PORT->DIN, INFRARED_PIN)) {
		INFRARED_PORT->FUNC0 = tmp;
		return 1;
	} else {
		INFRARED_PORT->FUNC0 = tmp;
		return 0;
	}
}

/**
  * @brief  Capture event callback function
  * @param  arg: Pointer to a timer_handle_t structure  
  * @retval None
  */
static void timer_ic_cbk(struct timer_handle_s *arg)
{
	uint16_t cval = 0;
	
	if (infrared_pin_read()) {               /* Rising edge detected */
		arg->perh->CCEP |= (1 << 1);     /* Channel1 capture falling edge */
		arg->perh->COUNT = 0;            /* Clear timer counter */
		capture_sta.capsta |= ( 1 << 4); /* Label rising edge detected */
	} else { /* Falling edge detected */
		cval = ald_timer_read_capture_value(arg, TIMER_CHANNEL_1); /* Get capture value */
		arg->perh->CCEP &= ~(1 << 1);    /* Channel1 capture rising edge */		
		if (capture_sta.capsta & 0x10) { /* if detected rising edge */
			if (capture_sta.capsta & 0x80) {                    /* if detected leadcode */
				if ((cval > 300) && (cval < 800)) {         /* 0 received */
					capture_sta.recval <<= 1;
					capture_sta.recval |= 0;
				} else if ((cval > 1400) && (cval < 1800)) {/* 1 received */
					capture_sta.recval <<= 1;
					capture_sta.recval |= 1;
				} else if ((cval > 2200) && (cval < 2600)) { /* Repeat code */
					capture_sta.keycnt += 1;
					capture_sta.capsta &= 0xF0;
				}
			} else if ((cval > 4200) && (cval < 4700)){ 
				capture_sta.capsta |= 0x80;
				capture_sta.keycnt = 0;
			}
		}
	}
}

/**
  * @brief  Capture timer 10ms update event callback function
  * @param  arg: Pointer to a timer_handle_t structure  
  * @retval None
  */
static void timer_update_event_cbk (struct timer_handle_s *arg)
{
	if (capture_sta.capsta & 0x80) {
		capture_sta.capsta &= ~ 0x10;
		if ((capture_sta.capsta & 0x0f) == 0) 
			capture_sta.capsta |= 1 << 6;
		if ((capture_sta.capsta & 0x0f) < 14)
			capture_sta.capsta++;
		else {
			capture_sta.capsta &= ~ (1 << 7);
			capture_sta.capsta &= 0xf0;
		}
	}
}

/**
  * @}
  */ 

/** @defgroup Infrared_Public_Functions Infrared Public Functions
  * @{
  */

/**
  * @brief  Infrared init
  * @param  None  
  * @retval None
  */
void bsp_infrared_init(void)
{
	gpio_init_t x;

	/* Initialize pin */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(INFRARED_PORT, INFRARED_PIN, &x);
	/* Initialize GP32C4T1 */
	h_tim.perh           = GP32C4T1;
	h_tim.init.prescaler = (72 - 1);
	h_tim.init.mode      = TIMER_CNT_MODE_UP;
	h_tim.init.period    = 9999;
	h_tim.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim.init.re_cnt    = 0;
        h_tim.capture_cbk    = NULL;    
	h_tim.period_elapse_cbk = NULL;
	ald_timer_ic_init(&h_tim);
	/* Initialize clock source */
	clock.source     = TIMER_SRC_INTER;
	ald_timer_config_clock_source(&h_tim, &clock);
	/* Initialize input capture channel */
	init.polarity = TIMER_IC_POLARITY_RISE;
	init.sel      = TIMER_IC_SEL_DIRECT;
	init.psc      = TIMER_IC_PSC_DIV1;
	init.filter   = 0x03;
	ald_timer_ic_config_channel(&h_tim,&init,TIMER_CHANNEL_1);
	ald_mcu_irq_config(GP32C4T1_IRQn, 0, 1, ENABLE);
	/* Enable CC interrupt */
	ald_timer_interrupt_config(&h_tim, TIMER_IT_CC1, ENABLE);
	/* Enable UPDATE interrupt */
	ald_timer_interrupt_config(&h_tim, TIMER_IT_UPDATE, ENABLE);
        /* Start input capture */
	ald_timer_ic_start(&h_tim, TIMER_CHANNEL_1);
        ald_timer_ic_start_by_it(&h_tim, TIMER_CHANNEL_1);
}

/**
  * @brief  GP32C4T1 Handler
  * @param  None  
  * @retval None
  */
void GP32C4T1_Handler(void)
{
	uint32_t reg = GP32C4T1->IFM;
	
	/* Capture or compare 3 event */
	if (READ_BIT(reg, TIMER_FLAG_CC1)) {
		timer_ic_cbk(&h_tim);
		ald_timer_clear_flag_status(&h_tim, TIMER_FLAG_CC1);
	}
	/* TIMER Update event */
	if (READ_BIT(reg, TIMER_FLAG_UPDATE)) {
		timer_update_event_cbk(&h_tim);
		ald_timer_clear_flag_status(&h_tim, TIMER_FLAG_UPDATE);
	}
	
}

/**
  * @brief  Infrared key scan
  * @param  None
  * @retval key value
  */
uint8_t bsp_infrared_scan(void)
{
	uint8_t sta = 0;
	uint8_t add = 0;
	uint8_t iadd = 0;
	
	if (capture_sta.capsta & (0x40)) {
		add = capture_sta.recval >> 24;
		iadd = (capture_sta.recval >> 16) & 0xff;
		if (add == 0 ) {
			if (iadd == 0xff) {
				add = capture_sta.recval >> 8;
				iadd = capture_sta.recval;
				if (add == (0xff - iadd)) 
					sta = add;
			}
		}
		if ((sta == 0) || ((capture_sta.capsta & 0x80) == 0)) {
			capture_sta.capsta &= ~(1 << 6);
			capture_sta.keycnt = 0;
		}
		capture_sta.capsta = 0;
		capture_sta.keycnt = 0;
		capture_sta.recval = 0;
	}
	return sta;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */ 
