/**
  *********************************************************************************
  *
  * @file    bsp_ethernet.c
  * @brief   ethernet driver
  *
  * @version V1.0
  * @date    16 Apr 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          16 Apr 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include "bsp_ethernet.h"

 /** @addtogroup ES32F3xxx_BSP
   * @{
   */

/** @defgroup ETHERNET ethernet 
  * @{
  */
  
/** @defgroup Ethernet_SPI_LOW_LEVEL_Private_Function_Prototypes Ethernet SPI LOW LEVEL Private Function_Prototypes
  * @{
  */
static void spi_cs_select(void);
static void spi_cs_deselect(void);
static uint8_t spi_rb(void);
static void spi_wb(uint8_t wb);
static void spi_cris_enter(void);
static void spi_cris_exit(void);
static void w5500_pin_config(void);
/**
  * @}
  */ 

/** @defgroup Ethernet_Private_Variavle Ethernet Private Variable
  * @{
  */
/* Default Network Configuration */
w5500_net_info_t net_info = { .mac  = {0x00, 0x08, 0xdc,0x00, 0xab, 0xcd},
			      .ip   = {192, 168, 1, 199},
			      .sn   = {255,255,255,0},
			      .gw   = {192, 168, 1, 1},
			      .dns  = {0,0,0,0},
			      .dhcp = NETINFO_STATIC };
/* DNS server ip address */
static uint8_t    _DNS_[4];      
static dhcp_mode  _DHCP_;        		      
static uint16_t sock_any_port = SOCK_ANY_PORT_NUM;
static uint16_t sock_io_mode = 0;
static uint16_t sock_remained_size[8] = {0,0,};
uint8_t  sock_pack_info[8] = {0,};
static uint16_t sock_is_sending = 0;

/**
  * @}
  */ 

/** @defgroup Ethernet_Private_Macro Ethernet Private Macro
  * @{
  */
#define CHECK_SOCKNUM()   \
   do {                    \
      if(sn > _WIZCHIP_SOCK_NUM_) return SOCKERR_SOCKNUM;   \
   } while(0);             \

#define CHECK_SOCKMODE(mode)  \
   do {                     \
      if((get_sn_mr(sn) & 0x0F) != mode) return SOCKERR_SOCKMODE;  \
   } while(0);              \

#define CHECK_SOCKINIT()   \
   do {                     \
      if((get_sn_sr(sn) != SOCK_INIT)) return SOCKERR_SOCKINIT; \
   } while(0);              \

#define CHECK_SOCKDATA()   \
   do {                     \
      if(len == 0) return SOCKERR_DATALEN;   \
   } while(0);      

#define CHECK_SOCKMODE(mode)  \
   do {                     \
      if((get_sn_mr(sn) & 0x0F) != mode) return SOCKERR_SOCKMODE;  \
   } while(0);  

#define get_sn_rx_rd(sn) \
		(((uint16_t)w5500_read(Sn_RX_RD(sn)) << 8) + w5500_read(W5500_OFFSET_INC(Sn_RX_RD(sn),1)))		

#define set_sn_rx_rd(sn, rxrd) { \
		w5500_write(Sn_RX_RD(sn),   (uint8_t)(rxrd>>8)); \
		w5500_write(W5500_OFFSET_INC(Sn_RX_RD(sn),1), (uint8_t) rxrd); \
	}

#define set_sn_cr(sn, cr) \
		w5500_write(Sn_CR(sn), cr)

#define get_sn_cr(sn) w5500_read(Sn_CR(sn))
	
#define set_sn_ir(sn, ir) w5500_write(Sn_IR(sn), (ir & 0x1F))
	
#define get_sn_ir(sn) (w5500_read(Sn_IR(sn)) & 0x1F)
	
#define get_sn_rxbuf_size(sn) w5500_read(Sn_RXBUF_SIZE(sn))
	
#define get_sn_rx_max(sn) (((uint16_t)get_sn_rxbuf_size(sn)) << 10)
	
#define get_sn_sr(sn) w5500_read(Sn_SR(sn))
		
#define get_sn_tx_wr(sn) (((uint16_t)w5500_read(Sn_TX_WR(sn)) << 8) + w5500_read(W5500_OFFSET_INC(Sn_TX_WR(sn),1)))		
		
#define set_sn_tx_wr(sn, txwr) { \
		w5500_write(Sn_TX_WR(sn),   (uint8_t)(txwr>>8)); \
		w5500_write(W5500_OFFSET_INC(Sn_TX_WR(sn),1), (uint8_t) txwr); \
		}

#define get_sn_txbuf_size(sn) w5500_read(Sn_TXBUF_SIZE(sn))
		
#define get_sn_tx_max(sn) (((uint16_t)get_sn_txbuf_size(sn)) << 10)
		
#define get_sn_mr(sn) w5500_read(Sn_MR(sn))
		
#define get_sipr(sipr) w5500_read_buf(SIPR, sipr, 4)
		
#define set_sn_port(sn, port)  { \
		w5500_write(Sn_PORT(sn),   (uint8_t)(port >> 8)); \
		w5500_write(W5500_OFFSET_INC(Sn_PORT(sn),1), (uint8_t) port); \
	}		

#define set_sn_mr(sn, mr) w5500_write(Sn_MR(sn),mr)
	
#define set_sn_dipr(sn, dipr) w5500_write_buf(Sn_DIPR(sn), dipr, 4)
	
#define set_sn_dport(sn, dport) { \
		w5500_write(Sn_DPORT(sn),   (uint8_t) (dport>>8)); \
		w5500_write(W5500_OFFSET_INC(Sn_DPORT(sn),1), (uint8_t)  dport); \
	}
/**
  * @}
  */ 

/** @defgroup W5500_SPI_LOW_LEVEL_Private_Functions W5500 SPI LOW LEVEL Private Functions
  * @{
  */
  
/**
  * @brief  SPI delay
  * @retval None
  */
void bsp_spi_w5500_delay(void)
{
	uint32_t i;

	for (i = 0; i < 1; i++);
}  

/**
  * @brief  SPI chip write byte
  * @retval None
  */
void bsp_spi_w5500_write1(uint8_t dat)
{
	uint8_t i;

	for(i = 0; i < 8; i++) {
		if (dat & 0x80) {
			MOSI_1();
		}
		else {
			MOSI_0();
		}
		SCK_0();
		dat <<= 1;
		bsp_spi_w5500_delay();
		SCK_1();				
		bsp_spi_w5500_delay();
	}
}

/**
  * @brief  SPI chip read byte
  * @retval None
  */
uint8_t bsp_spi_w5500_read1(void)
{
	uint8_t i;
	uint8_t read = 0;

	for (i = 0; i < 8; i++) {
		SCK_0();
		bsp_spi_w5500_delay();
		read = read << 1;
		if (MISO_IS_HIGH()) {
			read++;
		}
		SCK_1();
		bsp_spi_w5500_delay();
	}
	return read;
}

/**
  * @brief  SPI chip seleced
  * @retval None
  */
static void spi_cs_select(void)
{
	CLEAR_BIT(W5500_CS_PORT->DOUT, W5500_CS_PIN);
}

/**
  * @brief  SPI chip deseleced
  * @retval None
  */
static void spi_cs_deselect(void)
{
	SET_BIT(W5500_CS_PORT->DOUT, W5500_CS_PIN);
}

/**
  * @brief  SPI read buffer
  * @retval None
  */
static uint8_t spi_rb(void)
{
	return bsp_spi_w5500_read1();
}

/**
  * @brief  SPI write buffer
  * @retval None
  */
static void spi_wb(uint8_t wb)
{
	bsp_spi_w5500_write1(wb);
}

/**
  * @brief  SPI  enter critical
  * @retval None
  */
static void spi_cris_enter(void)
{
	__set_PRIMASK(1);
}

/**
  * @brief  SPI exit from critical
  * @retval None
  */
static void spi_cris_exit(void)
{
	__set_PRIMASK(0);
}

/**
  * @brief  w5500 module pin initialize.
  * @retval None.
  */
static void w5500_pin_config(void)
{
	gpio_init_t x;

	/* Initialize sck pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(W5500_SCLK_PORT, W5500_SCLK_PIN, &x);

	/* Initialize miso pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(W5500_MISO_PORT, W5500_MISO_PIN, &x);

	/* Initialize mosi pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(W5500_MOSI_PORT, W5500_MOSI_PIN, &x);

	/* Initialize cs pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(W5500_CS_PORT, W5500_CS_PIN, &x);

	/* Initialize reset pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(W5500_RST_PORT, W5500_RST_PIN, &x);

	ald_gpio_write_pin(W5500_CS_PORT, W5500_CS_PIN, 1);   /* CS SET */
	ald_gpio_write_pin(W5500_RST_PORT, W5500_RST_PIN, 1); /* RST RESET */	
}
/**
  * @}
  */

/** @defgroup W5500_SPI_Public_Functions W5500 SPI Public Functions
  * @{
  */

/**
  * @brief  w5500 module reset.
  * @retval reset value.
  */
void bsp_w5500_spi_init(void)
{
	w5500_pin_config();
	/* Hardware reset */
	bsp_w5500_hw_reset();
}

/**
  * @brief  W5500 module reset
  * @retval reset value
  */
void bsp_w5500_hw_reset(void)
{
	uint32_t i = 0;
	
	ald_gpio_write_pin(W5500_RST_PORT, W5500_RST_PIN, 0);
	for (i = 0; i < 0xffff; ++i);
	ald_gpio_write_pin(W5500_RST_PORT, W5500_RST_PIN, 1);
	for (i = 0; i < 0xffff; ++i);
}

/**
  * @brief  w5500 module write buffer
  * @param  addr: w5500 address
  * @param  pbuf: source data start address
  * @param  len: data length
  * @retval reset value
  */
void w5500_write_buf(uint32_t addr, uint8_t *pbuf, uint16_t len)
{
	uint16_t i;
	
	spi_cris_enter();
	spi_cs_select();
	
	addr |= (_W5500_SPI_WRITE_ | _W5500_SPI_VDM_OP_);
	spi_wb((addr & 0x00ff0000) >> 16);
	spi_wb((addr & 0x0000ff00) >> 8);
	spi_wb((addr & 0x000000ff) >> 0);
	for (i = 0; i < len; ++i) 
		spi_wb(pbuf[i]);
	
	spi_cs_deselect();
	spi_cris_exit();
}

/**
  * @brief  w5500 module read buffer
  * @param  addr: w5500 address
  * @param  pbuf: source data start address
  * @param  len: data length
  * @retval reset value
  */
void w5500_read_buf(uint32_t addr, uint8_t *pbuf, uint16_t len)
{
	uint16_t i;
	
	spi_cris_enter();
	spi_cs_select();
	
	addr |= (_W5500_SPI_READ_ | _W5500_SPI_VDM_OP_);
	spi_wb((addr & 0x00ff0000) >> 16);
	spi_wb((addr & 0x0000ff00) >> 8);
	spi_wb((addr & 0x000000ff) >> 0);
	for (i = 0; i < len; ++i) 
		pbuf[i] = spi_rb();
	
	spi_cs_deselect();
	spi_cris_exit();
}

/**
  * @brief  w5500 module write byte
  * @param  addr: w5500 address
  * @param  wb: source data 
  * @retval reset value
  */
void w5500_write(uint32_t addr, uint8_t wb)
{
	spi_cris_enter();
	spi_cs_select();
	
	addr |= (_W5500_SPI_WRITE_ | _W5500_SPI_VDM_OP_);
	spi_wb((addr & 0x00FF0000) >> 16);
	spi_wb((addr & 0x0000FF00) >>  8);
	spi_wb((addr & 0x000000FF) >>  0);
	spi_wb(wb);

	spi_cs_deselect();
	spi_cris_exit();
}

/**
  * @brief  w5500 module read byte
  * @param  addr: w5500 address
  * @retval ret: return value
  */
uint8_t w5500_read(uint32_t addr)
{
	uint8_t ret;
	
	spi_cris_enter();
	spi_cs_select();
	
	addr |= (_W5500_SPI_READ_ | _W5500_SPI_VDM_OP_);
	spi_wb((addr & 0x00FF0000) >> 16);
	spi_wb((addr & 0x0000FF00) >>  8);
	spi_wb((addr & 0x000000FF) >>  0);
	ret = spi_rb();
	spi_cs_deselect();
	spi_cris_exit();
	
	return ret;
}

/**
  * @brief  w5500 module set net information
  * @param  pnetinfo: pointer to net info struct
  * @retval None
  */
static void w5500_set_net_info(w5500_net_info_t* pnetinfo)
{
	w5500_write_buf(SHAR, pnetinfo->mac, 6);
	w5500_write_buf(GAR, pnetinfo->gw, 4);
	w5500_write_buf(SUBR, pnetinfo->sn, 4);
	w5500_write_buf(SIPR, pnetinfo->ip, 4);
	_DNS_[0] = pnetinfo->dns[0];
	_DNS_[1] = pnetinfo->dns[1];
	_DNS_[2] = pnetinfo->dns[2];
	_DNS_[3] = pnetinfo->dns[3];
	_DHCP_   = pnetinfo->dhcp;
}

/**
  * @brief  w5500 module get net information
  * @param  pnetinfo: pointer to net info struct
  * @retval None
  */
static void w5500_get_net_info(w5500_net_info_t* pnetinfo)
{
	w5500_read_buf(SHAR, pnetinfo->mac, 6);
	w5500_read_buf(GAR, pnetinfo->gw, 4);
	w5500_read_buf(SUBR, pnetinfo->sn, 4);
	w5500_read_buf(SIPR, pnetinfo->ip, 4);
	pnetinfo->dns[0]= _DNS_[0];
	pnetinfo->dns[1]= _DNS_[1];
	pnetinfo->dns[2]= _DNS_[2];
	pnetinfo->dns[3]= _DNS_[3];
	pnetinfo->dhcp  = _DHCP_;
}

/**
  * @brief  w5500 module soft reset 
  * @retval None
  */
static void w5500_sw_reset(void)
{
	uint8_t gw[4], sn[4], sip[4];
	uint8_t mac[6];

	w5500_read_buf(SHAR, mac, 6);
	w5500_read_buf(GAR, gw, 4);  
	w5500_read_buf(SUBR, sn, 4); 
	w5500_read_buf(SIPR, sip, 4);
	w5500_write(MR, MR_RST);
	w5500_read(MR); // for delay
	w5500_write_buf(SHAR, mac, 6);  
	w5500_write_buf(GAR, gw,4);
	w5500_write_buf(SUBR, sn,4);
	w5500_write_buf(SIPR, sip, 4);
}

/**
  * @brief  Intialize the network information 
  * @retval None
  */
void bsp_network_init(void)
{
	w5500_set_net_info((void*)&net_info);
	w5500_get_net_info((void*)&net_info);
	w5500_sw_reset();
}

/**
 * @brief Get Sn_RX_RSR register
 * @param sn: Socket number, it should be 0 ~ 7
 * @return Value of Sn_RX_RSR
 */
uint16_t get_sn_rx_rsr(uint8_t sn)
{
	uint16_t val=0,val1=0;

	do {
		val1 = w5500_read(Sn_RX_RSR(sn));
		val1 = (val1 << 8) + w5500_read(W5500_OFFSET_INC(Sn_RX_RSR(sn),1));
		if (val1 != 0) {
			val = w5500_read(Sn_RX_RSR(sn));
			val = (val << 8) + w5500_read(W5500_OFFSET_INC(Sn_RX_RSR(sn),1));
		}
	} while (val != val1);

	return val;
}

/**
 * @brief Discard the received data in RX memory
 * @param sn: Socket number 0 ~ 7
 * @param len: Data length
 * @return None
 */
void w5500_recv_ignore(uint8_t sn, uint16_t len)
{
	uint16_t ptr = 0;

	ptr = get_sn_rx_rd(sn);
	ptr += len;
	set_sn_rx_rd(sn, ptr);
}

/**
 * @brief Copy data to your buffer from internal RX memory
 * @param sn: Socket number 0 ~ 7
 * @param data: Pointer buffer to read data
 * @param len: Read data length
 * @return None
 */
void w5500_recv_data(uint8_t sn, uint8_t *data, uint16_t len)
{
	uint16_t ptr = 0;
	uint32_t addrsel = 0;

	if(len == 0) 
		return;
	ptr = get_sn_rx_rd(sn);
	addrsel = ((uint32_t)ptr << 8) + (W5500_RXBUF_BLOCK(sn) << 3);
	w5500_read_buf(addrsel, data, len);
	ptr += len;
	set_sn_rx_rd(sn,ptr);
}

/**
 * @brief Copy data to internal TX memory
 * @param sn: Socket number 0 ~ 7
 * @param data: Pointer buffer to write data
 * @param len: Write data length
 * @return None
 */
void w5500_send_data(uint8_t sn, uint8_t *data, uint16_t len)
{
	uint16_t ptr = 0;
	uint32_t addrsel = 0;

	if(len == 0)  
		return;
	ptr = get_sn_tx_wr(sn);
	addrsel = ((uint32_t)ptr << 8) + (W5500_TXBUF_BLOCK(sn) << 3);
	w5500_write_buf(addrsel,data, len);
	ptr += len;
	set_sn_tx_wr(sn,ptr);
}

/**
 * @brief Get Sn_TX_FSR register
 * @param sn: Socket number 0 ~ 7
 * @return Value of Sn_TX_FSR
 */
uint16_t get_sn_tx_fsr(uint8_t sn)
{
	uint16_t val=0, val1=0;

	do {
		val1 = w5500_read(Sn_TX_FSR(sn));
		val1 = (val1 << 8) + w5500_read(W5500_OFFSET_INC(Sn_TX_FSR(sn),1));
		if (val1 != 0) {
			val = w5500_read(Sn_TX_FSR(sn));
			val = (val << 8) + w5500_read(W5500_OFFSET_INC(Sn_TX_FSR(sn),1));
		}
	} while (val != val1);
	
	return val;
}

/**
 * @brief Close a socket.
 * @param sn: Socket number 0 ~ 7.
 * @return return SOCK_OK or SOCKERR_SOCKNUM
 */
int8_t socket_close(uint8_t sn)
{
	CHECK_SOCKNUM();

	w5500_write(Sn_CR(sn), Sn_CR_CLOSE);
	while( w5500_read(Sn_CR(sn)));
	set_sn_ir(sn, 0xFF);
	sock_io_mode &= ~(1<<sn);
	sock_is_sending &= ~(1<<sn);
	sock_remained_size[sn] = 0;
	sock_pack_info[sn] = 0;
	while(w5500_read(Sn_SR(sn)) != SOCK_CLOSED);
	
	return SOCK_OK;
}

/**
 * @brief Try to disconnect a connection socket, valid only in TCP server or client mode
 * @param sn: Socket number 0 ~ 7
 * @return @ref SOCK_OK 
 *         @ref SOCKERR_SOCKNUM  
 *         @ref SOCKERR_SOCKMODE 
 *         @ref SOCKERR_TIMEOUT  
 *         @ref SOCK_BUSY        
 */
int8_t socket_disconnect(uint8_t sn)
{
	CHECK_SOCKNUM();
	CHECK_SOCKMODE(Sn_MR_TCP);
	set_sn_cr(sn, Sn_CR_DISCON);
	while(get_sn_cr(sn));
	sock_is_sending &= ~(1 << sn);
	if(sock_io_mode & (1 << sn)) 
		return SOCK_BUSY;
	while(get_sn_sr(sn) != SOCK_CLOSED) {
	   if(get_sn_ir(sn) & Sn_IR_TIMEOUT) {
	      socket_close(sn);
	      return SOCKERR_TIMEOUT;
	   }
	}
	
	return SOCK_OK;
}

/**
 * @brief Receive datagram of UDP or MACRAW
 * @param sn  : Socket number 0 ~ 7
 * @param buf : Pointer buffer to read incoming data
 * @param len : Length of data in buf. 
 * @param addr: Pointer variable of destination IP address
 * @param port: Pointer variable of destination port number.
 * @return @ref SOCKERR_DATALEN    
 *         @ref SOCKERR_SOCKMODE   
 *         @ref SOCKERR_SOCKNUM    
 *         @ref SOCKBUSY           
 */
int32_t socket_recvfrom(uint8_t sn, uint8_t * buf, uint16_t len, uint8_t * addr, uint16_t *port)
{
  
	uint8_t  mr;
	uint8_t  head[8];
	uint16_t pack_len=0;

	CHECK_SOCKNUM();
	switch((mr=get_sn_mr(sn)) & 0x0F) {
	  case Sn_MR_UDP:
	  case Sn_MR_IPRAW:
	  case Sn_MR_MACRAW:
		break;
	  default:
		return SOCKERR_SOCKMODE;
	}
	CHECK_SOCKDATA();
	if(sock_remained_size[sn] == 0) {
		while(1) {
			pack_len = get_sn_rx_rsr(sn);
			if(w5500_read(Sn_SR(sn)) == SOCK_CLOSED) 
				return SOCKERR_SOCKCLOSED;
			if( (sock_io_mode & (1<<sn)) && (pack_len == 0) ) 
				return SOCK_BUSY;
			if(pack_len != 0) break;
		};
	}
	switch (mr & 0x07) {
	   case Sn_MR_UDP :
	      if(sock_remained_size[sn] == 0) {
   			w5500_recv_data(sn, head, 8);
   			set_sn_cr(sn,Sn_CR_RECV);
   			while(get_sn_cr(sn));
			addr[0] = head[0];
      			addr[1] = head[1];
      			addr[2] = head[2];
      			addr[3] = head[3];
      			*port = head[4];
      			*port = (*port << 8) + head[5];
      			sock_remained_size[sn] = head[6];
      			sock_remained_size[sn] = (sock_remained_size[sn] << 8) + head[7];
   			sock_pack_info[sn] = PACK_FIRST;
		}
		if(len < sock_remained_size[sn]) 
			pack_len = len;
		else 
			pack_len = sock_remained_size[sn];
		len = pack_len;
		w5500_recv_data(sn, buf, pack_len); 
		break;
	   case Sn_MR_MACRAW :
	      if(sock_remained_size[sn] == 0) {
   			w5500_recv_data(sn, head, 2);
   			set_sn_cr(sn,Sn_CR_RECV);
   			while(get_sn_cr(sn));
    			sock_remained_size[sn] = head[0];
   			sock_remained_size[sn] = (sock_remained_size[sn] <<8) + head[1] -2;
   			if(sock_remained_size[sn] > 1514) {
				socket_close(sn);
				return SOCKFATAL_PACKLEN;
   			}
   			sock_pack_info[sn] = PACK_FIRST;
		}
		if(len < sock_remained_size[sn]) 
			pack_len = len;
		else 
			pack_len = sock_remained_size[sn];
		w5500_recv_data(sn,buf,pack_len);
		break;

	    case Sn_MR_IPRAW:
	      if(sock_remained_size[sn] == 0) {
			w5500_recv_data(sn, head, 6);
			set_sn_cr(sn,Sn_CR_RECV);
			while(get_sn_cr(sn));
			addr[0] = head[0];
			addr[1] = head[1];
			addr[2] = head[2];
			addr[3] = head[3];
			sock_remained_size[sn] = head[4];
			sock_remained_size[sn] = (sock_remained_size[sn] << 8) + head[5];
			sock_pack_info[sn] = PACK_FIRST;
		}
		if(len < sock_remained_size[sn]) 
			pack_len = len;
		else 
			pack_len = sock_remained_size[sn];
   		w5500_recv_data(sn, buf, pack_len); 
		break;
	    default:
		w5500_recv_ignore(sn, pack_len); 
		sock_remained_size[sn] = pack_len;
		break;
	}
	set_sn_cr(sn, Sn_CR_RECV);
	while(get_sn_cr(sn)) ;
	sock_remained_size[sn] -= pack_len;
	if(sock_remained_size[sn] != 0) {
		sock_pack_info[sn] |= PACK_REMAINED;
	}
	else 
		sock_pack_info[sn] = PACK_COMPLETED;

	return (int32_t)pack_len;
}

/**
 * @brief Send data to the connected peer in TCP socket.
 * @param sn: Socket number 0 ~ 7
 * @param buf: Pointer buffer containing data to be sent.
 * @param len: Length of data in buf
 * @return Success : The sent data size 
 *         Fail    : @ref SOCKERR_SOCKSTATUS 
 *                   @ref SOCKERR_TIMEOUT    
 *                   @ref SOCKERR_SOCKMODE 
 *                   @ref SOCKERR_SOCKNUM    
 *                   @ref SOCKERR_DATALEN    
 *                   @ref SOCK_BUSY          
 */
int32_t socket_send(uint8_t sn, uint8_t * buf, uint16_t len)
{
	uint8_t tmp=0;
	uint16_t freesize=0;
   
	CHECK_SOCKNUM();
	CHECK_SOCKMODE(Sn_MR_TCP);
	CHECK_SOCKDATA();
	tmp = get_sn_sr(sn);
	if(tmp != SOCK_ESTABLISHED && tmp != SOCK_CLOSE_WAIT) 
		return SOCKERR_SOCKSTATUS;
	if( sock_is_sending & (1<<sn) ) {
		tmp = get_sn_ir(sn);
		if(tmp & Sn_IR_SENDOK) {
			set_sn_ir(sn, Sn_IR_SENDOK);
			sock_is_sending &= ~(1<<sn);         
		} else if(tmp & Sn_IR_TIMEOUT) {
			socket_close(sn);
			return SOCKERR_TIMEOUT;
		} else return SOCK_BUSY;
	}
	freesize = get_sn_tx_max(sn);
	if (len > freesize) 
		len = freesize; 
	while(1) {
		freesize = get_sn_tx_fsr(sn);
		tmp = get_sn_sr(sn);
		if ((tmp != SOCK_ESTABLISHED) && (tmp != SOCK_CLOSE_WAIT)) {
			socket_close(sn);
			return SOCKERR_SOCKSTATUS;
		}
		if( (sock_io_mode & (1<<sn)) && (len > freesize) ) 
			return SOCK_BUSY;
		if(len <= freesize) 
			break;
	}
	w5500_send_data(sn, buf, len);
   
	set_sn_cr(sn,Sn_CR_SEND);
	while(get_sn_cr(sn));
	sock_is_sending |= (1 << sn);

	return (int32_t)len;
}

/**
 * @brief Listen to a connection request from a client
 * @param sn: Socket number 0 ~ 7
 * @return Success : @ref SOCK_OK 
 *         Fail    : @ref SOCKERR_SOCKINIT   
 *                   @ref SOCKERR_SOCKCLOSED 
 */
int8_t socket_listen(uint8_t sn)
{
	CHECK_SOCKNUM();
	CHECK_SOCKMODE(Sn_MR_TCP);
	CHECK_SOCKINIT();
	set_sn_cr(sn, Sn_CR_LISTEN);
	while(get_sn_cr(sn));
	while(get_sn_sr(sn) != SOCK_LISTEN) {
		socket_close(sn);
		return SOCKERR_SOCKCLOSED;
	}
	
	return SOCK_OK;
}

/**
 * @brief Receive data from the connected peer
 * @param sn:  Socket number 0 ~ 7
 * @param buf: Pointer buffer to read incoming data
 * @param len: Length of data in buf
 * @return  Success : The real received data size 
 *          Fail    :  @ref SOCKERR_SOCKSTATUS 
 *                     @ref SOCKERR_SOCKMODE   
 *                     @ref SOCKERR_SOCKNUM    
 *                     @ref SOCKERR_DATALEN    
 *                     @ref SOCK_BUSY          
 */
int32_t recv(uint8_t sn, uint8_t * buf, uint16_t len)
{
	uint8_t  tmp = 0;
	uint16_t recvsize = 0;

	CHECK_SOCKNUM();
	CHECK_SOCKMODE(Sn_MR_TCP);
	CHECK_SOCKDATA();
   
	recvsize = get_sn_rx_max(sn);
	if(recvsize < len) 
		len = recvsize;
	while(1) {
		recvsize = get_sn_rx_rsr(sn);
		tmp = get_sn_sr(sn);
		if (tmp != SOCK_ESTABLISHED) {
			if(tmp == SOCK_CLOSE_WAIT) {
				if(recvsize != 0) 
					break;
				else if(get_sn_tx_fsr(sn) == get_sn_tx_max(sn)) {
					socket_close(sn);
					return SOCKERR_SOCKSTATUS;
				}
			} else {
				socket_close(sn);
				return SOCKERR_SOCKSTATUS;
			}
		}
		if((sock_io_mode & (1<<sn)) && (recvsize == 0)) 
			return SOCK_BUSY;
		if(recvsize != 0) 
			break;
	};

	if(recvsize < len) 
		len = recvsize;   
	w5500_recv_data(sn, buf, len);
	set_sn_cr(sn, Sn_CR_RECV);
	while(get_sn_cr(sn));

	return (int32_t)len;
}

/**
 * @brief Open a socket
 * @param sn: Socket number 0 ~ 7
 * @param protocol: TCP, UDP and MACRAW
 * @param port: Port number to be bined
 * @param flag: Socket flags 
 * @return  Success : The socket number 
 *          Fail    : @ref SOCKERR_SOCKNUM    
 *                    @ref SOCKERR_SOCKMODE   
 *                    @ref SOCKERR_SOCKFLAG   
 */
int8_t socket_open(uint8_t sn, uint8_t protocol, uint16_t port, uint8_t flag)
{
	CHECK_SOCKNUM();
	switch(protocol) {
	  case Sn_MR_TCP :
	  {
		uint32_t taddr;
	 
		get_sipr((uint8_t*)&taddr);
		if(taddr == 0) 
			return SOCKERR_SOCKINIT;
	  }
          case Sn_MR_UDP :
          case Sn_MR_MACRAW :
	  case Sn_MR_IPRAW :
		break;
	  default :
		return SOCKERR_SOCKMODE;
	}
	if((flag & 0x04) != 0) 
		return SOCKERR_SOCKFLAG;
	   
	if(flag != 0) {
		switch(protocol) {
		  case Sn_MR_TCP:
			if((flag & (SF_TCP_NODELAY|SF_IO_NONBLOCK))==0) 
				return SOCKERR_SOCKFLAG;
		  break;
		  case Sn_MR_UDP:
			if(flag & SF_IGMP_VER2) {
				if((flag & SF_MULTI_ENABLE)==0) 
					return SOCKERR_SOCKFLAG;
			}
			if(flag & SF_UNI_BLOCK) {
				if((flag & SF_MULTI_ENABLE) == 0) 
					return SOCKERR_SOCKFLAG;
			}
		  break;
		  default:
			break;
		}
	}
	socket_close(sn);
	set_sn_mr(sn, (protocol | (flag & 0xF0)));
	if(!port) {
		port = sock_any_port++;
		if(sock_any_port == 0xFFF0) sock_any_port = SOCK_ANY_PORT_NUM;
	}
	set_sn_port(sn,port);	
	set_sn_cr(sn,Sn_CR_OPEN);
	while(get_sn_cr(sn));
	sock_io_mode &= ~(1 <<sn);
	sock_io_mode |= ((flag & SF_IO_NONBLOCK) << sn);   
	sock_is_sending &= ~(1<<sn);
	sock_remained_size[sn] = 0;
	sock_pack_info[sn] = PACK_COMPLETED;

	while(get_sn_sr(sn) == SOCK_CLOSED);
	return (int8_t)sn;
}

/**
 * @brief Sends datagram to the peer with destination IP address and port number passed as parameter
 * @param sn:    Socket number 0 ~ 7
 * @param buf:   Pointer buffer to send outgoing data
 * @param len:   Length of data in buf
 * @param addr:  Pointer variable of destination IP address
 * @param port:  Destination port number
 * @return Success : The sent data size 
 *         Fail    : @ref SOCKERR_SOCKNUM     
 *                   @ref SOCKERR_SOCKMODE    
 *                   @ref SOCKERR_SOCKSTATUS  
 *                   @ref SOCKERR_DATALEN     
 *                   @ref SOCKERR_IPINVALID   
 *                   @ref SOCKERR_PORTZERO    
 *                   @ref SOCKERR_SOCKCLOSED  
 *                   @ref SOCKERR_TIMEOUT     
 *                   @ref SOCK_BUSY           
 */
int32_t socket_sendto(uint8_t sn, uint8_t * buf, uint16_t len, uint8_t * addr, uint16_t port)
{
	uint8_t tmp = 0;
	uint16_t freesize = 0;
	uint32_t taddr;

	CHECK_SOCKNUM();
	switch(get_sn_mr(sn) & 0x0F) {
	  case Sn_MR_UDP:
	  case Sn_MR_MACRAW:
	  case Sn_MR_IPRAW:
	  break;
	  default:
		return SOCKERR_SOCKMODE;
	}
	CHECK_SOCKDATA();
	taddr = ((uint32_t)addr[0]) & 0x000000FF;
	taddr = (taddr << 8) + ((uint32_t)addr[1] & 0x000000FF);
	taddr = (taddr << 8) + ((uint32_t)addr[2] & 0x000000FF);
	taddr = (taddr << 8) + ((uint32_t)addr[3] & 0x000000FF);
   
	if((taddr == 0) && ((get_sn_mr(sn)&Sn_MR_MACRAW) != Sn_MR_MACRAW)) 
		return SOCKERR_IPINVALID;
	if((port  == 0) && ((get_sn_mr(sn)&Sn_MR_MACRAW) != Sn_MR_MACRAW)) 
		return SOCKERR_PORTZERO;
	tmp = get_sn_sr(sn);
	if((tmp != SOCK_MACRAW) && (tmp != SOCK_UDP) && (tmp != SOCK_IPRAW)) 
		return SOCKERR_SOCKSTATUS;
	set_sn_dipr(sn,addr);
	set_sn_dport(sn,port);      
	freesize = get_sn_tx_max(sn);
	if (len > freesize) 
		len = freesize; 
	while(1) {
		freesize = get_sn_tx_fsr(sn);
		if(get_sn_sr(sn) == SOCK_CLOSED) 
			return SOCKERR_SOCKCLOSED;
		if( (sock_io_mode & (1<<sn)) && (len > freesize) ) 
			return SOCK_BUSY;
		if(len <= freesize) 
			break;
	};
	w5500_send_data(sn, buf, len);
	set_sn_cr(sn,Sn_CR_SEND);
	while(get_sn_cr(sn));
	while(1) {
		tmp = get_sn_ir(sn);
		if(tmp & Sn_IR_SENDOK) {
			set_sn_ir(sn, Sn_IR_SENDOK);
			break;
		} else if(tmp & Sn_IR_TIMEOUT) {
			set_sn_ir(sn, Sn_IR_TIMEOUT);
			return SOCKERR_TIMEOUT;
		}
	}

	return (int32_t)len;
}

/**
  * @brief  TCP server Loopback test example
  * @param  sn: socket number
  * @param  buf: data buffer
  * @param  port: port number
  * @retval state
  */
int32_t loopback_tcps(uint8_t sn, uint8_t* buf, uint16_t port)
{
	int32_t ret;
	uint16_t size = 0, sentsize=0;

	switch(get_sn_sr(sn)) {
	  case SOCK_ESTABLISHED :
		if(get_sn_ir(sn) & Sn_IR_CON) {
			set_sn_ir(sn, Sn_IR_CON);
		}
		if((size = get_sn_rx_rsr(sn)) > 0) {
			if(size > DATA_BUF_SIZE) size = DATA_BUF_SIZE;
				ret = recv(sn, buf, size);
			if(ret <= 0) 
				return ret;      
			size = (uint16_t) ret;
			sentsize = 0;
			while(size != sentsize) {
				ret = socket_send(sn, buf+sentsize, size-sentsize);
				if(ret < 0) {
					socket_close(sn);
					return ret;
				}
				sentsize += ret; 
			}
		}
         break;
	case SOCK_CLOSE_WAIT :
		if((ret = socket_disconnect(sn)) != SOCK_OK) 
			return ret;
         break;
	case SOCK_INIT :
		if( (ret = socket_listen(sn)) != SOCK_OK) 
			return ret;
         break;
	case SOCK_CLOSED:
		if((ret = socket_open(sn, Sn_MR_TCP, port, 0x00)) != sn) return ret;
         break;
	default:
		break;
	}
	
	return 1;
}

/**
  * @brief  UDP Loopback test example
  * @param  sn: socket number
  * @param  buf: data buffer
  * @param  port: port number
  * @retval state
  */
int32_t loopback_udps(uint8_t sn, uint8_t* buf, uint16_t port)
{
	int32_t  ret;
	uint16_t size, sentsize;
	uint8_t  destip[4];
	uint16_t destport;

	switch(get_sn_sr(sn)) {
	case SOCK_UDP :
		if((size = get_sn_rx_rsr(sn)) > 0) {
			if(size > DATA_BUF_SIZE) size = DATA_BUF_SIZE;
				ret = socket_recvfrom(sn, buf, size, destip, (uint16_t*)&destport);
			if(ret <= 0) {
				return ret;
			}
			size = (uint16_t) ret;
			sentsize = 0;
			while(sentsize != size) {
				ret = socket_sendto(sn, buf + sentsize, size - sentsize, destip, destport);
				if(ret < 0) {
					return ret;
				}
				sentsize += ret; 
			}
		}
		break;
	case SOCK_CLOSED:
		if((ret = socket_open(sn, Sn_MR_UDP, port, 0x00)) != sn)
			return ret;
		break;
	default :
		break;
	}
	
	return 1;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */



