/**
  *********************************************************************************
  *
  * @file    bsp_eeprom.c
  * @brief   Eeprom driver
  *
  * @version V1.0
  * @date    16 Apr 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          16 Apr 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

 #include <stdio.h>
 #include "bsp_eeprom.h"
 #include "ald_gpio.h"

 /** @addtogroup ES32F3xxx_BSP
   * @{
   */

/** @defgroup EEPROM eeprom
  * @{
  */

/** @defgroup EEPROM_Private_Params eeprom Private Variables
  * @{
  */
i2c_handle_t h_i2c;
/**
  * @}
  */

/** @defgroup EEPROM_Private_Functions eeprom Private Functions
  * @{
  */

/**
  * @brief  Eeprom pin initiation
  * @retval None.
  */
static void eeprom_pin_init(void)
{
	gpio_init_t a;

	/* Initialize scl pin */
	a.mode = GPIO_MODE_OUTPUT;
	a.odos = GPIO_OPEN_DRAIN;
	a.pupd = GPIO_PUSH_UP;
	a.nodrv = GPIO_OUT_DRIVE_6;
	a.podrv = GPIO_OUT_DRIVE_6;
	a.flt  = GPIO_FILTER_DISABLE;
	a.type = GPIO_TYPE_TTL;
	a.func = EEPROM_SCL_FUN;
	ald_gpio_init(EEPROM_SCL_PORT, EEPROM_SCL_PIN, &a);

	/* Initialize sda pin */
	a.mode = GPIO_MODE_OUTPUT;
	a.odos = GPIO_OPEN_DRAIN;
	a.pupd = GPIO_PUSH_UP;
	a.nodrv = GPIO_OUT_DRIVE_6;
	a.podrv = GPIO_OUT_DRIVE_6;
	a.flt  = GPIO_FILTER_DISABLE;
	a.type = GPIO_TYPE_TTL;
	a.func = EEPROM_SDA_FUN;
	ald_gpio_init(EEPROM_SDA_PORT, EEPROM_SDA_PIN, &a);
}

/**
  * @}
  */

/** @defgroup EEPROM_Public_Functions eeprom Public Functions
  * @brief    BSP EEPROM Functions
  */

/**
  * @brief  Initiate eeprom moudle
  * @retval None.
  */
void bsp_eeprom_init(void)
{
	eeprom_pin_init();

	/* Initialize I2C struct */
	memset(&h_i2c, 0, sizeof(h_i2c));
	
	/* Enable i2c interrupt */
	ald_mcu_irq_config(I2C1_EV_IRQn, 3, 3, ENABLE);
	ald_mcu_irq_config(I2C1_ERR_IRQn, 3, 3, ENABLE);

	/* Initialize I2C */
	h_i2c.perh = I2C0;
	h_i2c.init.clk_speed = 100000;
	h_i2c.init.module    = I2C_MODULE_MASTER;
	h_i2c.init.dual_addr = I2C_DUALADDR_ENABLE;
	h_i2c.init.own_addr1 = 0xA0;
	h_i2c.init.addr_mode = I2C_ADDR_7BIT;
	h_i2c.init.general_call = I2C_GENERALCALL_ENABLE;
	h_i2c.init.no_stretch = I2C_NOSTRETCH_ENABLE;

	ald_i2c_reset(&h_i2c);
	ald_i2c_init(&h_i2c);
}

/**
  * @brief  Write an amount of data in non-blocking mode with Interrupt to a specific memory address
  * @param  dev_addr: Target device address
  * @param  mem_addr: Internal memory address
  * @param  add_size: size of internal memory address
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref eep_operat_t.
  */
eep_operat_t bsp_eeprom_write(uint16_t dev_addr, uint16_t mem_addr,
                              i2c_addr_size_t add_size, uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return EEP_ERROR;

	if (ald_i2c_mem_write(&h_i2c, dev_addr, mem_addr, add_size, buf, size, 500) != OK)
		return EEP_ERROR;

	return EEP_OK;
}

/**
  * @brief  Read an amount of data in non-blocking mode with Interrupt to a specific memory address
  * @param  dev_addr: Target device address
  * @param  mem_addr: Internal memory address
  * @param  add_size: size of internal memory address
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be read
  * @retval Status, see @ref eep_operat_t.
  */
eep_operat_t bsp_eeprom_read(uint16_t dev_addr, uint16_t mem_addr,
                              i2c_addr_size_t add_size, uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return EEP_ERROR;

	if (ald_i2c_mem_read(&h_i2c, dev_addr, mem_addr, add_size, buf, size, 500) != OK)
		return EEP_ERROR;

	return EEP_OK;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
