/**
  *********************************************************************************
  *
  * @file    bsp_can.c
  * @brief   can driver
  *
  * @version V1.0
  * @date    24 Apr 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <stdio.h>
#include "bsp_can.h"
#include "stdio.h"

/** @addtogroup ES32F0xx_BSP
  * @{
  */

/** @defgroup CAN can
  * @{
  */

/** @defgroup Can_Public_Variables CAN Public Variables
  * @{
  */
uint8_t can_flag = 0;
can_handle_t h_can;
can_rx_msg_t rx_msg;
can_tx_msg_t tx_msg;
can_filter_t filter;

/**
  * @}
  */

/** @defgroup Can_Private_Functions Can Private Functions
  * @{
  */

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to can_handle_t structure.
  * @param  num: Index of the RxFIFO.
  * @retval None.
  */
void can_recv_complete(can_handle_t *arg, can_rx_fifo_t num)
{
	can_flag = 1;
	ald_can_recv_by_it(&h_can, CAN_RX_FIFO0, &rx_msg);
}

 /**
  * @brief  Initialize pin of can module  
  * @retval None
  */ 
static void can_pin_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_9, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_8, &x);
}
/**
  * @}
  */

/** @defgroup Can_Public_Functions CAN Public Functions
  * @brief    BSP can Functions
  */

/**
  * @brief  can module initialize function
  * @retval None
  */
void bsp_can_init(void)
{
	/* Clear can_handle_t structure */
	memset(&h_can, 0x0, sizeof(h_can));
	memset(&filter, 0x0, sizeof(filter));
	/* Initialize pin */
	can_pin_init();
	/* Initialize can handler */
	h_can.perh        = CAN0;
	h_can.rx0_msg     = &rx_msg;
	h_can.init.psc    = 12;
	h_can.init.mode   = CAN_MODE_NORMAL;
	h_can.init.sjw    = CAN_SJW_1;
	h_can.init.seg1   = CAN_SEG1_7;
	h_can.init.seg2   = CAN_SEG2_4;
	h_can.init.ttcm   = DISABLE;
	h_can.init.abom   = DISABLE;
	h_can.init.awk    = DISABLE;
	h_can.init.artx   = ENABLE;
	h_can.init.rfom   = DISABLE;
	h_can.init.txmp   = DISABLE;
	h_can.tx_cplt_cbk = NULL;
	h_can.rx_cplt_cbk = can_recv_complete;
	h_can.error_cbk   = NULL;
	ald_can_init(&h_can);
	
	filter.id_high      = 0;
	filter.id_low       = 0;
	filter.mask_id_high = 0;
	filter.mask_id_low  = 0;
	filter.fifo         = CAN_FILTER_FIFO0;
	filter.number       = 0;
	filter.mode         = CAN_FILTER_MODE_MASK;
	filter.scale        = CAN_FILTER_SCALE_32;
	filter.active	    = ENABLE;
	ald_can_filter_config(&h_can, &filter);
}

/**
  * @brief  Sends an amount of data in blocking mode.
  * @param  tx_msg: struct of data to be sent.
  * @retval Status, see @ref can_status_t.
  */
can_status_t can_send_msg(void)
{
	static uint8_t cnt = 1;
#ifdef CAN_ID_EXTEND
	tx_msg.type = CAN_ID_EXT;
	tx_msg.ext  = 0x486;
#else
	tx_msg.type = CAN_ID_STD;
	tx_msg.ext  = 0x486;
#endif
	tx_msg.rtr  = CAN_RTR_DATA;
	tx_msg.len  = cnt % 9;
        cnt = cnt + 1;
	memset(tx_msg.data, cnt, cnt%9);
	return (can_status_t)ald_can_send(&h_can, &tx_msg, 100);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
