/**
  *********************************************************************************
  *
  * @file    bsp_audio.c
  * @brief   Audio driver
  *
  * @version V1.0
  * @date    10 May 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          10 May 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
  
#include "bsp_audio.h"
#include "ald_gpio.h"
#include "ald_sram.h"
#include <string.h>
  
/** @addtogroup ES32F3xxx_BSP
  * @{
  */
   
/** @defgroup Audio audio
  * @{ 
  */
  
/** @defgroup VS1053_SPI__Bus_Private_Variables Audio Private Variables
  * @{
  */
uint8_t vs1053_spi_busy_flag = 0;
#define SCK_0()		VS1053_SCK_PORT->BSRR = (VS1053_SCK_PIN << 16)
#define SCK_1()		VS1053_SCK_PORT->BSRR = VS1053_SCK_PIN
#define MOSI_0()	VS1053_MOSI_PORT->BSRR = (VS1053_MOSI_PIN << 16)
#define MOSI_1()	VS1053_MOSI_PORT->BSRR = VS1053_MOSI_PIN
#define MISO_IS_HIGH()	((VS1053_MISO_PORT->DIN & VS1053_MISO_PIN) != 0) 
/**
  * @}
  */
  
/** @defgroup VS1053_Private_Params vs1053 Private Variables
  * @{
  */
uint8_t vs1053ram[5] = {0,0,0,0,250};

const uint16_t plugin[605] = { /* Compressed plugin */
  0x0007, 0x0001, 0x8300, 0x0006, 0x01f2, 0xb080, 0x0024, 0x0007, /*    0 */
  0x9257, 0x3f00, 0x0024, 0x0030, 0x0297, 0x3f00, 0x0024, 0x0006, /*    8 */
  0x0017, 0x3f10, 0x0024, 0x3f00, 0x0024, 0x0000, 0xf6d7, 0xf400, /*   10 */
  0x55c0, 0x0000, 0x0817, 0xf400, 0x57c0, 0x0000, 0x004d, 0x000a, /*   18 */
  0x708f, 0x0000, 0xc44e, 0x280f, 0xe100, 0x0006, 0x2016, 0x0000, /*   20 */
  0x028d, 0x0014, 0x1b01, 0x2800, 0xc795, 0x0015, 0x59c0, 0x0000, /*   28 */
  0xfa0d, 0x0039, 0x324f, 0x0000, 0xd20e, 0x2920, 0x41c0, 0x0000, /*   30 */
  0x0024, 0x000a, 0x708f, 0x0000, 0xc44e, 0x280a, 0xcac0, 0x0000, /*   38 */
  0x028d, 0x6fc2, 0x0024, 0x000c, 0x0981, 0x2800, 0xcad5, 0x0000, /*   40 */
  0x18c2, 0x290c, 0x4840, 0x3613, 0x0024, 0x290c, 0x4840, 0x4086, /*   48 */
  0x184c, 0x6234, 0x0024, 0x0000, 0x0024, 0x2800, 0xcad5, 0x0030, /*   50 */
  0x0317, 0x3f00, 0x0024, 0x280a, 0x71c0, 0x002c, 0x9d40, 0x3613, /*   58 */
  0x0024, 0x3e10, 0xb803, 0x3e14, 0x3811, 0x3e11, 0x3805, 0x3e00, /*   60 */
  0x3801, 0x0007, 0xc390, 0x0006, 0xa011, 0x3010, 0x0444, 0x3050, /*   68 */
  0x4405, 0x6458, 0x0302, 0xff94, 0x4081, 0x0003, 0xffc5, 0x48b6, /*   70 */
  0x0024, 0xff82, 0x0024, 0x42b2, 0x0042, 0xb458, 0x0003, 0x4cd6, /*   78 */
  0x9801, 0xf248, 0x1bc0, 0xb58a, 0x0024, 0x6de6, 0x1804, 0x0006, /*   80 */
  0x0010, 0x3810, 0x9bc5, 0x3800, 0xc024, 0x36f4, 0x1811, 0x36f0, /*   88 */
  0x9803, 0x283e, 0x2d80, 0x0fff, 0xffc3, 0x003e, 0x2d4f, 0x2800, /*   90 */
  0xe380, 0x0000, 0xcb4e, 0x3413, 0x0024, 0x2800, 0xd405, 0xf400, /*   98 */
  0x4510, 0x2800, 0xd7c0, 0x6894, 0x13cc, 0x3000, 0x184c, 0x6090, /*   a0 */
  0x93cc, 0x38b0, 0x3812, 0x3004, 0x4024, 0x0000, 0x0910, 0x3183, /*   a8 */
  0x0024, 0x3100, 0x4024, 0x6016, 0x0024, 0x000c, 0x8012, 0x2800, /*   b0 */
  0xd711, 0xb884, 0x104c, 0x6894, 0x3002, 0x2939, 0xb0c0, 0x3e10, /*   b8 */
  0x93cc, 0x4084, 0x9bd2, 0x4282, 0x0024, 0x0000, 0x0041, 0x2800, /*   c0 */
  0xd9c5, 0x6212, 0x0024, 0x0000, 0x0040, 0x2800, 0xdec5, 0x000c, /*   c8 */
  0x8390, 0x2a00, 0xe240, 0x34c3, 0x0024, 0x3444, 0x0024, 0x3073, /*   d0 */
  0x0024, 0x3053, 0x0024, 0x3000, 0x0024, 0x6092, 0x098c, 0x0000, /*   d8 */
  0x0241, 0x2800, 0xe245, 0x32a0, 0x0024, 0x6012, 0x0024, 0x0000, /*   e0 */
  0x0024, 0x2800, 0xe255, 0x0000, 0x0024, 0x3613, 0x0024, 0x3001, /*   e8 */
  0x3844, 0x2920, 0x0580, 0x3009, 0x3852, 0xc090, 0x9bd2, 0x2800, /*   f0 */
  0xe240, 0x3800, 0x1bc4, 0x000c, 0x4113, 0xb880, 0x2380, 0x3304, /*   f8 */
  0x4024, 0x3800, 0x05cc, 0xcc92, 0x05cc, 0x3910, 0x0024, 0x3910, /*  100 */
  0x4024, 0x000c, 0x8110, 0x3910, 0x0024, 0x39f0, 0x4024, 0x3810, /*  108 */
  0x0024, 0x38d0, 0x4024, 0x3810, 0x0024, 0x38f0, 0x4024, 0x34c3, /*  110 */
  0x0024, 0x3444, 0x0024, 0x3073, 0x0024, 0x3063, 0x0024, 0x3000, /*  118 */
  0x0024, 0x4080, 0x0024, 0x0000, 0x0024, 0x2839, 0x53d5, 0x4284, /*  120 */
  0x0024, 0x3613, 0x0024, 0x2800, 0xe585, 0x6898, 0xb804, 0x0000, /*  128 */
  0x0084, 0x293b, 0x1cc0, 0x3613, 0x0024, 0x000c, 0x8117, 0x3711, /*  130 */
  0x0024, 0x37d1, 0x4024, 0x4e8a, 0x0024, 0x0000, 0x0015, 0x2800, /*  138 */
  0xe845, 0xce9a, 0x0024, 0x3f11, 0x0024, 0x3f01, 0x4024, 0x000c, /*  140 */
  0x8197, 0x408a, 0x9bc4, 0x3f15, 0x4024, 0x2800, 0xea85, 0x4284, /*  148 */
  0x3c15, 0x6590, 0x0024, 0x0000, 0x0024, 0x2839, 0x53d5, 0x4284, /*  150 */
  0x0024, 0x0000, 0x0024, 0x2800, 0xd2d8, 0x458a, 0x0024, 0x2a39, /*  158 */
  0x53c0, 0x3009, 0x3851, 0x3e14, 0xf812, 0x3e12, 0xb817, 0x0006, /*  160 */
  0xa057, 0x3e11, 0x9fd3, 0x0023, 0xffd2, 0x3e01, 0x0024, 0x0006, /*  168 */
  0x0011, 0x3111, 0x0024, 0x6498, 0x07c6, 0x868c, 0x2444, 0x3901, /*  170 */
  0x8e06, 0x0030, 0x0551, 0x3911, 0x8e06, 0x3961, 0x9c44, 0xf400, /*  178 */
  0x44c6, 0xd46c, 0x1bc4, 0x36f1, 0xbc13, 0x2800, 0xf615, 0x36f2, /*  180 */
  0x9817, 0x002b, 0xffd2, 0x3383, 0x188c, 0x3e01, 0x8c06, 0x0006, /*  188 */
  0xa097, 0x468c, 0xbc17, 0xf400, 0x4197, 0x2800, 0xf304, 0x3713, /*  190 */
  0x0024, 0x2800, 0xf345, 0x37e3, 0x0024, 0x3009, 0x2c17, 0x3383, /*  198 */
  0x0024, 0x3009, 0x0c06, 0x468c, 0x4197, 0x0006, 0xa052, 0x2800, /*  1a0 */
  0xf544, 0x3713, 0x2813, 0x2800, 0xf585, 0x37e3, 0x0024, 0x3009, /*  1a8 */
  0x2c17, 0x36f1, 0x8024, 0x36f2, 0x9817, 0x36f4, 0xd812, 0x2100, /*  1b0 */
  0x0000, 0x3904, 0x5bd1, 0x2a00, 0xeb8e, 0x3e11, 0x7804, 0x0030, /*  1b8 */
  0x0257, 0x3701, 0x0024, 0x0013, 0x4d05, 0xd45b, 0xe0e1, 0x0007, /*  1c0 */
  0xc795, 0x2800, 0xfd95, 0x0fff, 0xff45, 0x3511, 0x184c, 0x4488, /*  1c8 */
  0xb808, 0x0006, 0x8a97, 0x2800, 0xfd45, 0x3009, 0x1c40, 0x3511, /*  1d0 */
  0x1fc1, 0x0000, 0x0020, 0xac52, 0x1405, 0x6ce2, 0x0024, 0x0000, /*  1d8 */
  0x0024, 0x2800, 0xfd41, 0x68c2, 0x0024, 0x291a, 0x8a40, 0x3e10, /*  1e0 */
  0x0024, 0x2921, 0xca80, 0x3e00, 0x4024, 0x36f3, 0x0024, 0x3009, /*  1e8 */
  0x1bc8, 0x36f0, 0x1801, 0x2808, 0x9300, 0x3601, 0x5804, 0x0007, /*  1f0 */
  0x0001, 0x802e, 0x0006, 0x0002, 0x2800, 0xf700, 0x0007, 0x0001, /*  1f8 */
  0x8050, 0x0006, 0x0028, 0x3e12, 0x3800, 0x2911, 0xf140, 0x3e10, /*  200 */
  0x8024, 0xf400, 0x4595, 0x3593, 0x0024, 0x35f3, 0x0024, 0x3500, /*  208 */
  0x0024, 0x0021, 0x6d82, 0xd024, 0x44c0, 0x0006, 0xa402, 0x2800, /*  210 */
  0x1815, 0xd024, 0x0024, 0x0000, 0x0000, 0x2800, 0x1815, 0x000b, /*  218 */
  0x6d57, 0x3009, 0x3c00, 0x36f0, 0x8024, 0x36f2, 0x1800, 0x2000, /*  220 */
  0x0000, 0x0000, 0x0024, 0x0007, 0x0001, 0x8030, 0x0006, 0x0002, /*  228 */
  0x2800, 0x1400, 0x0007, 0x0001, 0x8064, 0x0006, 0x001c, 0x3e12, /*  230 */
  0xb817, 0x3e14, 0xf812, 0x3e01, 0xb811, 0x0007, 0x9717, 0x0020, /*  238 */
  0xffd2, 0x0030, 0x11d1, 0x3111, 0x8024, 0x3704, 0xc024, 0x3b81, /*  240 */
  0x8024, 0x3101, 0x8024, 0x3b81, 0x8024, 0x3f04, 0xc024, 0x2808, /*  248 */
  0x4800, 0x36f1, 0x9811, 0x0007, 0x0001, 0x8028, 0x0006, 0x0002, /*  250 */
  0x2a00, 0x190e, 0x000a, 0x0001, 0x0300,
};

const uint16_t bitrate[2][16]=
{
	{0,8,16,24,32,40,48,56,64,80,96,112,128,144,160,0},
	{0,32,40,48,56,64,80,96,112,128,160,192,224,256,320,0}
};

/**
  * @}
  */
  
/** @defgroup VS1053_SPI_Bus_Functions VS1053 SPI Bus Public Functions
  * @{
  */

/**
  * @brief  Inilize spi pins
  * @retval None
  */
void bsp_init_spi(void)
{	
	/* Initialize led pin */
	gpio_init_t x;
	
	x.mode = GPIO_MODE_OUTPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odos = GPIO_PUSH_PULL;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(VS1053_MOSI_PORT, VS1053_MOSI_PIN, &x);	/* VS-MOSI */
	ald_gpio_init(VS1053_SCK_PORT, VS1053_SCK_PIN, &x);	/* VS-SCK */
	
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odos = GPIO_OPEN_SOURCE;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	ald_gpio_init(VS1053_MISO_PORT, VS1053_MISO_PIN, &x);	/* VS-MISO */
}

/**
  * @brief Spi delay
  * @retval None
  */
void bsp_spi_delay(void)
{
	uint32_t i;

	for (i = 0; i < 2; i++);
}

/**
  * @brief  Send a byte and SCLK gather data at rising edge, CPOL = 0
  * @param  dat: Send data
  * @retval None
  */
void bsp_spi_write0(uint8_t dat)
{
	uint8_t i;

	for(i = 0; i < 8; i++) {
		if (dat & 0x80) {
			MOSI_1();
		}
		else {
			MOSI_0();
		}
		bsp_spi_delay();
		SCK_1();
		dat <<= 1;
		bsp_spi_delay();
		SCK_0();
	}
	bsp_spi_delay();
}

/**
  * @brief  Read a byte and SCLK gather data at rising edge, CPOL = 0
  * @retval Read a byte
  */
uint8_t bsp_spi_read0(void)
{
	uint8_t i;
	uint8_t read = 0;

	for (i = 0; i < 8; i++) {
		read = read << 1;
		if (MISO_IS_HIGH()) {
			read++;
		}
		SCK_1();
		bsp_spi_delay();
		SCK_0();
		bsp_spi_delay();
	}
	
	return read;
}

/**
  * @brief  Send a byte. SCLK gather data at rising edge, CPOL = 1
  * @param  dat: Send data
  * @retval None
  */
void bsp_spi_write1(uint8_t dat)
{
	uint8_t i;

	for(i = 0; i < 8; i++) {
		if (dat & 0x80) {
			MOSI_1();
		}
		else {
			MOSI_0();
		}
		SCK_0();
		dat <<= 1;
		bsp_spi_delay();
		SCK_1();				
		bsp_spi_delay();
	}
}

/**
  * @brief  Read a byte and SCLK gather data at rising edge, CPOL = 1
  * @retval Read a byte
  */
uint8_t bsp_spi_read1(void)
{
	uint8_t i;
	uint8_t read = 0;

	for (i = 0; i < 8; i++) {
		SCK_0();
		bsp_spi_delay();
		read = read << 1;
		if (MISO_IS_HIGH()) {
			read++;
		}
		SCK_1();
		bsp_spi_delay();
	}
	
	return read;
}

/**
  * @brief SPI BUS enter busy state
  * @retval None
  */
void bsp_spi_bus_enter(void)
{
	vs1053_spi_busy_flag = 1;
}

/**
  * @brief  SPI BUS exit busy state
  * @retval None
  */
void bsp_spi_bus_exit(void)
{
	vs1053_spi_busy_flag = 0;
}

/**
  * @brief  Get spi bus busy sate
  * @retval 0, idle 1, busy
  */
uint8_t bsp_spi_bus_busy(void)
{
	return vs1053_spi_busy_flag;
}

/**
  * @brief  Set spi sck state
  * @param  dat: 0 or 1
  * @retval None
  */
void bsp_set_spi_sck(uint8_t dat)
{
	if (dat == 0) {
		SCK_0();
	}
	else {
		SCK_1();
	}
}
/**
  * @}
  */

/** @defgroup VS1053_Public_Functions VS1053 Public Functions
  * @{
  */
/**
  * @brief  VS1053 initiation
  * @retval None
  */
void vs1053_init(void)
{
	gpio_init_t x;
	
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odos = GPIO_PUSH_PULL;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_1;
	/* VS_DREQ */
	ald_gpio_init(GPIOE, GPIO_PIN_6, &x);
	/* VS_XCS */
	x.mode = GPIO_MODE_OUTPUT;
	ald_gpio_init(GPIOF, GPIO_PIN_6, &x);
	/* VS_XDCS */
	ald_gpio_init(GPIOD, GPIO_PIN_6, &x);
	/* VS_RST */
	ald_gpio_init(GPIOF, GPIO_PIN_0, &x);
	ald_gpio_write_pin(GPIOF, GPIO_PIN_0, 1);
	VS1053_CS_1();
	VS1053_DS_1();
}

/**
  * @brief  Set vs1053 cs pin
  * @param  level: 0 or 1
  * @retval None.
  */
static void vs1053_set_cs(uint8_t level)
{
	if (level == 0) {
		bsp_spi_bus_enter();	
		bsp_set_spi_sck(0);
		VS1053_CS_0();
	}
	else {
		VS1053_CS_1();
		bsp_spi_bus_exit();	
	}
}

/**
  * @brief  Set vs1053 ds pin
  * @param  level: 0 or 1
  * @retval None.
  */
static void vs1053_set_ds(uint8_t level)
{
	if (level == 0) {
		bsp_spi_bus_enter();	
		bsp_set_spi_sck(0);
		VS1053_DS_0();
	}
	else {
		VS1053_DS_1();
		bsp_spi_bus_exit();	
	}
}

/**
  * @brief  if vs1053 is busy
  * @retval 0, idle 1,busy
  */
uint8_t vs1053_wait_timeout(void)
{
	uint32_t i;

	for (i = 0; i < 500000; i++) {
		if (!VS1053_IS_BUSY()) {
			break;
		}
	}
	if (i >= 500000) {
		return 1;	
	}

	return 0;	
}

/**
  * @brief  Write data to vs1053
  * @param  addr: address
  * @param  data: data
  * @retval None.
  */
void vs1053_write_cmd(uint8_t addr, uint16_t data)
{
	if (vs1053_wait_timeout()) 
		return;
	vs1053_set_cs(0);
	bsp_spi_write0(VS_WRITE_COMMAND);		
	bsp_spi_write0(addr); 			
	bsp_spi_write0(data >> 8); 			
	bsp_spi_write0(data);	 			
	vs1053_set_cs(1);
}

/**
  * @brief  if vs1053 request new data
  * @retval 0, idle 1, busy
  */
uint8_t vs1053_req_new_data(void)
{
	if (VS1053_IS_BUSY()) {
		return 1;
	}
	else {
		return 0;
	}
}

/**
  * @brief  Prepare fot writting
  * @retval None
  */
void vs1053_prewrite_data(void)
{
	VS1053_CS_1();
	VS1053_DS_0();
}

/**
  * @brief  Write data
  * @param  data: write data value
  * @retval None
  */
void vs1053_write_data(uint8_t data)
{
	vs1053_set_ds(0);
	bsp_spi_write0(data);
	vs1053_set_ds(1);
}

/**
  * @brief  Read vs1053 register
  * @param  addr: write address
  * @retval Register value
  */
uint16_t vs1053_read_reg(uint8_t addr)
{
	uint16_t temp;

	if (vs1053_wait_timeout()) {
		return 0;
	}
	vs1053_set_cs(0);
	bsp_spi_write0(VS_READ_COMMAND);		
	bsp_spi_write0(addr);			
	temp = bsp_spi_read0() << 8;		
	temp += bsp_spi_read0();		
	vs1053_set_cs(1);
	
	return temp;
}

/**
  * @brief  Read vs1053 ID
  * @retval Chip id
  */
uint8_t vs1053_read_chip_id(void)
{
	uint16_t usStatus;
	/*      pdf page 40
		SCI STATUS status Bit7:4 chip version
		4 for VS1053,
	*/
	usStatus = vs1053_read_reg(SCI_STATUS);

	return ((usStatus >> 4) & 0x000F);
}

/**
  * @brief  Load user patch
  * @retval None
  */
void load_user_patch(void)
{
	int i = 0;

	while (i < sizeof(plugin) / sizeof(plugin[0])) {
		unsigned short addr, n, val;
		addr = plugin[i++];
		n = plugin[i++];
		if (n & 0x8000U) {
			/* RLE run, replicate n samples */
			n &= 0x7FFF;
			val = plugin[i++];
			while (n--) {
				vs1053_write_cmd(addr, val);
			}
		}
		else {
			/* Copy run, copy n samples */
			while (n--) {
				val = plugin[i++];
				vs1053_write_cmd(addr, val);
			}
		}
	}
	if (vs1053_wait_timeout()) {
		return;
	}
}

/**
  * @brief  Test vs1053 inner ram
  * @retval 0, ok 1,fail
  */
uint8_t vs1053_test_ram(void)
{
	uint32_t i = 0;
	uint16_t regval;

 	vs1053_write_cmd(SCI_MODE, 0x0820);	

	if (vs1053_wait_timeout()) {
		return 0;
	}
	
	for (i = 0; i < 0xffff; ++i);

	vs1053_set_ds(0);
	for (i = 0; i < 0xff; ++i);
	bsp_spi_write0(0x4d);
	for (i = 0; i < 0xff; ++i);
	bsp_spi_write0(0xea);
	for (i = 0; i < 0xff; ++i);
	bsp_spi_write0(0x6d);
	for (i = 0; i < 0xff; ++i);
	bsp_spi_write0(0x54);
	for (i = 0; i < 0xff; ++i);
	bsp_spi_write0(0x00);
	bsp_spi_write0(0x00);
	bsp_spi_write0(0x00);
	bsp_spi_write0(0x00);
	vs1053_set_ds(1);
	for (i = 0; i < 0xffffff; ++i);
	if (vs1053_wait_timeout()) {
		return 0;
	}
	regval = vs1053_read_reg(SCI_HDAT0); 
	if (regval == 0x83FF) {
		return 1;
	}
	else {
		return 0;
	}
}

/**
  * @brief  Test sine
  * @retval None
  */
void vs1053_test_sine(void)
{
	vs1053_write_cmd(0x0b,0x2020);	  	
 	vs1053_write_cmd(SCI_MODE, 0x0820);	
	if (vs1053_wait_timeout()) {
		return;
	}
	vs1053_set_ds(0);
	bsp_spi_write0(0x53);
	bsp_spi_write0(0xef);
	bsp_spi_write0(0x6e);
	bsp_spi_write0(0x24);	/* 0x24 or 0x44 */
	bsp_spi_write0(0x00);
	bsp_spi_write0(0x00);
	bsp_spi_write0(0x00);
	bsp_spi_write0(0x00);
	vs1053_set_ds(1);
	vs1053_set_ds(0);
	bsp_spi_write0(0x45);
	bsp_spi_write0(0x78);
	bsp_spi_write0(0x69);
	bsp_spi_write0(0x74);
	bsp_spi_write0(0x00);
	bsp_spi_write0(0x00);
	bsp_spi_write0(0x00);
	bsp_spi_write0(0x00);
	vs1053_set_ds(1);
}

/**
  * @brief  Reset vs1053 in hard
  * @retval None
  */
void vs1053_hd_reset(void)
{
	uint32_t i = 0;
	
	if (vs1053_wait_timeout()) {
		return;
	}
	VS1053_RST_0();
	for(i = 0; i < 0xfff; ++i);
	VS1053_DS_1();
	VS1053_CS_1();
	VS1053_RST_1();
	while(VS1053_IS_BUSY());
	for(i = 0; i < 0xfff; ++i);
}

/**
  * @brief  Reset vs1053 in soft
  * @retval None
  */
void vs1053_soft_reset(void)
{
	uint8_t retry = 0;
	uint16_t i = 0;
	
	if (vs1053_wait_timeout()) {
		return;
	}
	bsp_spi_write0(0X00);
	while(vs1053_read_reg(SCI_MODE) != 0x0800) {
		vs1053_write_cmd(SCI_MODE, 0x0804);
		for (i = 0; i < 0xfff; ++i);
		if (vs1053_wait_timeout()) {
			return;
		}
		
		if (retry++ > 5) {
			break;
		}
	}
	/* Set clock register, doubler etc. */
	vs1053_write_cmd(SCI_CLOCKF, 0x9800);

	#if 0
	vs1053_write_cmd(SCI_BASS, 0x0000);	
	vs1053_write_cmd(SCI_VOL, 0x2020); 	
	#endif
	if (vs1053_wait_timeout()) {
		return;
	}
	load_user_patch();
}

/**
  * @brief  Set vs1053 volume
  * @param  vol: 0 ~ 254
  * @retval None
  */
void vs1053_set_volume(uint8_t vol)
{
	if (vol == 0) {
		vol = 254;
	}
	else if (vol == 255) {
		vol = 0;
	}
	else {
		vol = 254 - vol;
	}
	vs1053_write_cmd(SCI_VOL, (vol << 8) | vol);
}

/**
  * @brief  Set vs1053 bass
  * @param  highamp: [-8, 7]
  * @param  highfreqcut: [1000, 15000]HZ
  * @param  lowamp: [0, 15]
  * @param  lowfreqcut: [20, 150]HZ
  * @retval None
  */
void vs1053_set_bass(int8_t highamp, uint16_t highfreqcut, uint8_t lowamp, uint16_t lowfreqcut)
{
	uint16_t value;

	/*
		SCI_BASS  register define

		Bit15:12  high sound -8 ... 7  (0 close)
		Bit11:8   bottom unit 1KHz,  1...15

		Bit7:4    low sound 0...15 (0 close)
		Bit3:0    top unit 10Hz, 2...15
	*/
	if (highamp < -8) {
		highamp = -8;
	}
	else if (highamp > 7) {
		highamp = 7;
	}
	value = highamp << 12;

	if (highfreqcut < 1000) {
		highfreqcut = 1000;
	}
	else if (highfreqcut > 15000) {
		highfreqcut = 15000;
	}
	value  += ((highfreqcut / 1000) << 8);

	if (lowamp > 15) {
		lowamp = 15;
	}
	value  += (lowamp << 4);

	if (lowfreqcut < 20) {
		lowfreqcut = 20;
	}
	else if (lowfreqcut > 150) {
		lowfreqcut = 150;
	}
	value  += (lowfreqcut / 10);

	vs1053_write_cmd(SCI_BASS, value);
}

/**
  * @brief  Write inner RAM
  * @param  addr: address
  * @param  val: value
  * @retval None
  */
static void vs_wram_write(uint16_t addr, uint16_t val)
{
	vs1053_write_cmd(SCI_WRAMADDR, addr);
	if (vs1053_wait_timeout()) {
		return;
	}
	vs1053_write_cmd(SCI_WRAM, val);	
}

/**
  * @brief  Set speaker state 
  * @param  sw: 0,close 1,open
  * @retval None
  */
void vs1053_spk_set(uint8_t sw)
{
	vs_wram_write(GPIO_DDR, 1 << 4);
	vs_wram_write(GPIO_ODATA, sw << 4);
} 

/**
  * @brief  Reset decode time 
  * @retval None
  */
void reset_decode_time(void)
{
	vs1053_write_cmd(SCI_DECODE_TIME, 0x0000);
}

/**
  * @brief  Audio init
  * @retval None
  */
void bsp_audio_init(void)
{
	bsp_init_spi();
	vs1053_init();
	vs1053_hd_reset();
	vs1053_soft_reset();
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
