/**
  *********************************************************************************
  *
  * @file    bsp_ethernet.c
  * @brief   ethernet driver
  *
  * @version V1.0
  * @date    24 Apr 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __BSP_ETHERNET_H__
#define __BSP_ETHERNET_H__

#include "ald_spi.h"
#include "ald_gpio.h"

#ifdef _cplusplus
extern "C" {
#endif
/** @addtogroup ES32F3xxx_BSP
  * @{
  */

/** @addtogroup ETHERNET ethernet
  * @{
  */
  
/** @defgroup Ethernet_Public_Variable Ethernet Public Variable
  * @{
  */
extern spi_handle_t h_spi;
/**
  * @}
  */

/** @defgroup W5500_Public_Macros W5500 Public Macros
  * @{
  */
#define W5500_RST_PORT	 GPIOB
#define W5500_RST_PIN	 GPIO_PIN_2
#define W5500_INT_PORT   GPIOF
#define W5500_INT_PIN    GPIO_PIN_1
#define W5500_MISO_PORT  GPIOB
#define W5500_MISO_PIN   GPIO_PIN_0
#define W5500_MOSI_PORT  GPIOB
#define W5500_MOSI_PIN   GPIO_PIN_1
#define W5500_CS_PORT    GPIOE
#define W5500_CS_PIN     GPIO_PIN_4
#define W5500_SCLK_PORT  GPIOE
#define W5500_SCLK_PIN   GPIO_PIN_5

#define SCK_0()		W5500_SCLK_PORT->BSRR = (W5500_SCLK_PIN << 16)
#define SCK_1()		W5500_SCLK_PORT->BSRR = W5500_SCLK_PIN
#define MOSI_0()	W5500_MOSI_PORT->BSRR = (W5500_MOSI_PIN << 16)
#define MOSI_1()	W5500_MOSI_PORT->BSRR = W5500_MOSI_PIN
#define MISO_IS_HIGH()	((W5500_MISO_PORT->DIN & W5500_MISO_PIN) != 0)

#define DATA_BUF_SIZE	          2048
#define _WIZCHIP_SOCK_NUM_        8  
#define _W5500_SPI_VDM_OP_        0x00
#define _W5500_IO_BASE_           0x00000000
#define _W5500_SPI_READ_	  (0x00 << 2) 
#define _W5500_SPI_WRITE_	  (0x01 << 2) 

#define W5500_CREG_BLOCK          0x00 	
#define W5500_SREG_BLOCK(N)       (1 + 4*N) 
#define W5500_TXBUF_BLOCK(N)      (2 + 4*N) 
#define W5500_RXBUF_BLOCK(N)      (3 + 4*N) 
#define W5500_OFFSET_INC(ADDR, N) (ADDR + (N << 8)) 

#define MR_RST             0x80
#define SHAR               (_W5500_IO_BASE_ + (0x0009 << 8) + (W5500_CREG_BLOCK << 3))
#define MR                 (_W5500_IO_BASE_ + (0x0000 << 8) + (W5500_CREG_BLOCK << 3))
#define GAR                (_W5500_IO_BASE_ + (0x0001 << 8) + (W5500_CREG_BLOCK << 3))
#define SUBR               (_W5500_IO_BASE_ + (0x0005 << 8) + (W5500_CREG_BLOCK << 3))
#define SIPR               (_W5500_IO_BASE_ + (0x000F << 8) + (W5500_CREG_BLOCK << 3))

#define Sn_RX_RD(N)        (_W5500_IO_BASE_ + (0x0028 << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_CR(N)           (_W5500_IO_BASE_ + (0x0001 << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_IR(N)           (_W5500_IO_BASE_ + (0x0002 << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_RXBUF_SIZE(N)   (_W5500_IO_BASE_ + (0x001E << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_TX_FSR(N)       (_W5500_IO_BASE_ + (0x0020 << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_TX_WR(N)        (_W5500_IO_BASE_ + (0x0024 << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_TXBUF_SIZE(N)   (_W5500_IO_BASE_ + (0x001F << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_SR(N)           (_W5500_IO_BASE_ + (0x0003 << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_RX_RSR(N)       (_W5500_IO_BASE_ + (0x0026 << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_PORT(N)         (_W5500_IO_BASE_ + (0x0004 << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_MR(N)           (_W5500_IO_BASE_ + (0x0000 << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_DIPR(N)         (_W5500_IO_BASE_ + (0x000C << 8) + (W5500_SREG_BLOCK(N) << 3))
#define Sn_DPORT(N)        (_W5500_IO_BASE_ + (0x0010 << 8) + (W5500_SREG_BLOCK(N) << 3))

#define SOCK_OK               1        
#define SOCK_BUSY             0        
#define SOCK_FATAL            -1000    
#define SOCK_ERROR            0        
#define SOCKERR_SOCKNUM       (SOCK_ERROR - 1)     
#define SOCKERR_SOCKOPT       (SOCK_ERROR - 2)     
#define SOCKERR_SOCKINIT      (SOCK_ERROR - 3)     
#define SOCKERR_SOCKCLOSED    (SOCK_ERROR - 4)     
#define SOCKERR_SOCKMODE      (SOCK_ERROR - 5)     
#define SOCKERR_SOCKFLAG      (SOCK_ERROR - 6)     
#define SOCKERR_SOCKSTATUS    (SOCK_ERROR - 7)     
#define SOCKERR_ARG           (SOCK_ERROR - 10)    
#define SOCKERR_PORTZERO      (SOCK_ERROR - 11)    
#define SOCKERR_IPINVALID     (SOCK_ERROR - 12)    
#define SOCKERR_TIMEOUT       (SOCK_ERROR - 13)    
#define SOCKERR_DATALEN       (SOCK_ERROR - 14)    
#define SOCKERR_BUFFER        (SOCK_ERROR - 15)    
#define SOCKFATAL_PACKLEN     (SOCK_FATAL - 1)     

#define Sn_CR_SEND             0x20
#define Sn_CR_CLOSE            0x10
#define Sn_CR_DISCON           0x08
#define Sn_CR_SEND_MAC         0x21
#define Sn_CR_RECV             0x40
#define Sn_IR_SENDOK           0x10
#define Sn_IR_TIMEOUT          0x08
#define SOCK_CLOSED            0x00
#define Sn_MR_TCP              0x01
#define Sn_MR_UDP              0x02
#define Sn_MR_IPRAW            0x03    
#define Sn_MR_MACRAW           0x04
#define Sn_IR_CON              0x01

#define Sn_MR_ND               0x20
#define Sn_MR_MMB              Sn_MR_ND
#define Sn_MR_MC               Sn_MR_ND
#define Sn_MR_MULTI            0x80
#define Sn_MR_UCASTB           0x10

#define PACK_FIRST               0x80              
#define PACK_REMAINED            0x01              
#define PACK_COMPLETED           0x00

#define SOCK_MACRAW              0x42
#define SOCK_ESTABLISHED         0x17
#define SOCK_FIN_WAIT            0x18
#define SOCK_SYNRECV             0x16
#define SOCK_CLOSE_WAIT          0x1C
#define SOCK_LISTEN              0x14
#define SOCK_INIT                0x13
#define SOCK_UDP                 0x22
#define SOCK_IPRAW               0x32     
#define SOCK_ANY_PORT_NUM        0xC000

#define Sn_CR_LISTEN             0x02
#define Sn_CR_OPEN               0x01
#define SF_IO_NONBLOCK           0x01              
#define SF_TCP_NODELAY         (Sn_MR_ND)          
#define SF_IGMP_VER2           (Sn_MR_MC)            
#define SF_MULTI_ENABLE        (Sn_MR_MULTI)       
#define SF_UNI_BLOCK           (Sn_MR_UCASTB)

typedef enum {
	NETINFO_STATIC = 1,    
	NETINFO_DHCP           
} dhcp_mode;

typedef struct w5500_net_info {
	uint8_t mac[6];  
	uint8_t ip[4];   
	uint8_t sn[4];   
	uint8_t gw[4];   
	uint8_t dns[4];  
	dhcp_mode dhcp;  
} w5500_net_info_t;

/**
  * @}
  */

/** @addtogroup W5500_Public_Functions
  * @{
  */
void bsp_w5500_spi_init(void);
void bsp_w5500_hw_reset(void);
void bsp_network_init(void);
int32_t loopback_tcps(uint8_t sn, uint8_t* buf, uint16_t port);
int32_t loopback_udps(uint8_t sn, uint8_t* buf, uint16_t port);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef _cplusplus
}
#endif

#endif


