/**
  *********************************************************************************
  *
  * @file    bsp_display.h
  * @brief   Display driver
  *
  * @version V1.0
  * @date    10 May 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          10 May 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
  
#ifndef __BSP_DISPLAY_H__
#define __BSP_DISPLAY_H__

#include <stdint.h>
#if defined LVGL_GUI
#include "lv_color.h"
#endif

#ifdef _cplusplus
extern "C" {
#endif

/** @addtogroup ES32F3xxx_BSP
  * @{
  */
  
/** @addtogroup DISPLAY
  * @{
  */

/** @defgroup Lcd_Private_Macros Lcd Private Macros
  * @{
  */
#define EBI_LCD_BASE            ((uint32_t)(0x6C000000 | 0x0007FFFE))
#define LCD_ADDR                ((lcd_addr_t *) EBI_LCD_BASE)
	 
#define EBI_D0_PORT		GPIOD
#define EBI_D0_PIN		GPIO_PIN_14
#define EBI_D1_PORT		GPIOD
#define EBI_D1_PIN		GPIO_PIN_15
#define EBI_D2_PORT		GPIOD
#define EBI_D2_PIN		GPIO_PIN_0
#define EBI_D3_PORT		GPIOD
#define EBI_D3_PIN		GPIO_PIN_1
#define EBI_D4_PORT		GPIOE
#define EBI_D4_PIN		GPIO_PIN_7
#define EBI_D5_PORT		GPIOE
#define EBI_D5_PIN		GPIO_PIN_8
#define EBI_D6_PORT		GPIOE
#define EBI_D6_PIN		GPIO_PIN_9
#define EBI_D7_PORT		GPIOE
#define EBI_D7_PIN		GPIO_PIN_10
#define EBI_D8_PORT		GPIOE
#define EBI_D8_PIN		GPIO_PIN_11
#define EBI_D9_PORT		GPIOE
#define EBI_D9_PIN		GPIO_PIN_12
#define EBI_D10_PORT		GPIOE
#define EBI_D10_PIN		GPIO_PIN_13
#define EBI_D11_PORT		GPIOE
#define EBI_D11_PIN		GPIO_PIN_14
#define EBI_D12_PORT		GPIOE
#define EBI_D12_PIN		GPIO_PIN_15
#define EBI_D13_PORT		GPIOD
#define EBI_D13_PIN		GPIO_PIN_8
#define EBI_D14_PORT		GPIOD
#define EBI_D14_PIN		GPIO_PIN_9
#define EBI_D15_PORT		GPIOD
#define EBI_D15_PIN		GPIO_PIN_10

#define	EBI_TFT2_RST_PORT	GPIOD
#define	EBI_TFT2_RST_PIN	GPIO_PIN_7

#define EBI_NE4_PORT		GPIOC
#define EBI_NE4_PIN		GPIO_PIN_12
        
#define EBI_NWE_PORT		GPIOD
#define EBI_NWE_PIN		GPIO_PIN_5
        
#define EBI_NOE_PORT		GPIOD
#define EBI_NOE_PIN		GPIO_PIN_4
        
#define EBI_RS_PORT		GPIOD
#define EBI_RS_PIN		GPIO_PIN_13

typedef enum {
	L2R_U2D = 0,	/*< From left to right and up to down */
        L2R_D2U = 1,    /*< From left to right and down to up */
        R2L_U2D = 2,    /*< From right to left and up to down */
        R2L_D2U = 3,    /*< From right to left and down to up */
	U2D_L2R = 4,    /*< From up to down and left to right */
	U2D_R2L = 5,    /*< From up to down and right to left */
	D2U_L2R = 6,    /*< From down to up and left to right */
	D2U_R2L = 7,    /*< From down to up and right to left */
} lcd_scan_dir_e;

#define LCD_DFT_SCAN_DIR  L2R_U2D   /*< Default direction */

#define WHITE       0xFFFF	
#define BLACK       0x0000		  
#define BLUE        0x001F	
#define BRED        0XF81F	
#define GRED 	    0XFFE0	
#define GBLUE	    0X07FF	
#define RED         0xF800	
#define MAGENTA     0xF81F	
#define GREEN       0x07E0	
#define CYAN        0x7FFF	
#define YELLOW      0xFFE0	
#define BROWN 	    0XBC40 
#define BRRED 	    0XFC07 
#define GRAY  	    0X8430 
#define DARKBLUE    0X01CF	
#define LIGHTBLUE   0X7D7C	  
#define GRAYBLUE    0X5458 
#define LIGHTGREEN  0X841F 
#define LGRAY 	    0XC618 
#define LGRAYBLUE   0XA651 
#define LBBLUE      0X2B12 

/**
  * @}
  */
  
/** @defgroup Touch_IIC_Public_Macros Touch IIC Public Macros
  * @{
  */

#define	TOUCH_IIC_SDA_PORT  GPIOD
#define	TOUCH_IIC_SDA_PIN   GPIO_PIN_2
			 
#define	TOUCH_IIC_SCL_PORT  GPIOD
#define	TOUCH_IIC_SCL_PIN   GPIO_PIN_3

#define	TOUCH_INT_PORT	    GPIOC
#define	TOUCH_INT_PIN	    GPIO_PIN_7
			 
#define	TOUCH_RST_PORT	    GPIOC
#define	TOUCH_RST_PIN	    GPIO_PIN_8
			 
#define TOUCH_IIC_SCL_1	ald_gpio_write_pin(TOUCH_IIC_SCL_PORT, TOUCH_IIC_SCL_PIN, 1)
#define TOUCH_IIC_SCL_0	ald_gpio_write_pin(TOUCH_IIC_SCL_PORT, TOUCH_IIC_SCL_PIN, 0)
#define TOUCH_IIC_SDA_1	ald_gpio_write_pin(TOUCH_IIC_SDA_PORT, TOUCH_IIC_SDA_PIN, 1)
#define TOUCH_IIC_SDA_0	ald_gpio_write_pin(TOUCH_IIC_SDA_PORT, TOUCH_IIC_SDA_PIN, 0)
#define TOUCH_READ_SDA	ald_gpio_read_pin(TOUCH_IIC_SDA_PORT, TOUCH_IIC_SDA_PIN) 

/**
  * @}
  */

/** @defgroup GT9147_Private_Macros GT9147 Private Macros
  * @{
  */
 
#define GT9147_CMD_WR 	0X28     	
#define GT9147_CMD_RD 	0X29		
  
#define GT9147_CTRL_ADDR   0X8040   	
#define GT9147_CFGS_ADDR   0X8047   	
#define GT9147_CHECK_ADDR  0X80FF   	
#define GT9147_ID_ADDR     0X8140   	
#define GT9147_GSTID_ADDR  0X814E   	

/**
  * @}
  */
  
/** @defgroup GT911_Private_Macros GT911 Private Macros
  * @{
  */
 
#define GT911_IIC_RADDR 0xBB	
#define GT911_IIC_WADDR 0xBA	

#define GT911_READ_ADDR 0x814E	
#define GT911_ID_ADDR   0x8140	

/**
  * @}
  */
  
/** @defgroup TOUCH_Public_Macros Touch Public Macros
  * @{
  */  
#define TOUCH_PRES_DOWN 0x80  /* Touched code */
#define TOUCH_CATH_PRES 0x40  /* Pressed */
#define TOUCH_MAX_POINT 5     /* Support touch pad number */

/**
  * @brief Touch screen control handler
  */
typedef struct {
	uint8_t (*init)(void);		/* Inilization function */
	uint8_t (*scan)(void);		/* Scan touch status */
	uint16_t x[TOUCH_MAX_POINT]; 	/* Current coordinate*/
	uint16_t y[TOUCH_MAX_POINT];	
	uint8_t  sta;			/* Current state */	
					/* bit7: 1,pressed 0,released */
					/* bit6: 0,no touch pressed 1, touch pressed */
					/* bit5: reserved */
					/* bit4 ~ bit0: pressed numbers */
	uint8_t touchtype;
} touch_device_t;

extern touch_device_t touch_device;	/* Defined in touch.c file */

/**
  * @}
  */

/** @defgroup Lcd_Variable_Type Lcd Variable Types
  * @{
  */

/**
  * @brief Parameters set of device
  */
typedef struct { 	 
	uint16_t width;			/*< LCD width  */
	uint16_t height;		/*< LCD height */
	uint16_t id;			/*< LCD ID */
	uint8_t  dir;			/*< 0,Vertiacl 1,Horizontal */
	uint16_t wramcmd;		/*< GRAM cmd */
	uint16_t setxcmd;		/*< X coordinate cmd */
	uint16_t setycmd;		/*< Y coordinate cmd */
	uint32_t front_color;
	uint32_t back_color;
} lcd_dev_t; 

/**
  * @brief LCD address struct
  */
typedef struct {
	volatile uint16_t lcd_reg;
	volatile uint16_t lcd_ram;
} lcd_addr_t;
/**
  * @}
  */

/** @defgroup LCD_Public_Variable lcd Public Varible
  * @{
  */
extern lcd_dev_t lcd_device;	

/**
  * @}
  */

/** @addtogroup LCD_Public_Functions
  * @{
  */
void lcd_set_front_color(uint32_t color);
void lcd_set_back_color(uint32_t color);
void lcd_display_on(void);
void lcd_display_off(void);
void lcd_screen_clear(uint32_t color);													
void lcd_set_cursor_pos(uint16_t xpos, uint16_t ypos);
void lcd_draw_point(uint16_t x, uint16_t y);
void lcd_draw_point_color(uint16_t x, uint16_t y, uint32_t color);
void lcd_draw_circle(uint16_t xc, uint16_t yc, uint8_t r);
void lcd_draw_line(uint16_t xs, uint16_t ys, uint16_t xe, uint16_t ye);
void lcd_draw_rectangle(uint16_t xs, uint16_t ys, uint16_t xe, uint16_t ye);
void lcd_fill_rectangle_color(uint16_t xs, uint16_t ys, uint16_t xe, uint16_t ye, uint16_t color);
void lcd_fill_rectangle_colors(uint16_t xs, uint16_t ys, uint16_t xe, uint16_t ye, uint16_t *color);
void lcd_show_char(uint16_t x, uint16_t y, char num, uint8_t size, uint8_t mode);
void lcd_show_number(uint16_t x, uint16_t y, uint32_t num, uint8_t len, uint8_t size); 
void lcd_show_xnumber(uint16_t x, uint16_t y, uint32_t num, uint8_t len, uint8_t size, uint8_t mode);
void lcd_show_string(uint16_t x, uint16_t y, uint16_t width, uint16_t height, uint8_t size, char *p);
void lcd_write_reg(uint16_t reg, uint16_t regval);
void lcd_write_ram_prepare(void);
void lcd_scan_dir(lcd_scan_dir_e direct);
void lcd_display_dir_param_set(uint8_t direct);
void lcd_set_window(uint16_t sx, uint16_t sy, uint16_t width, uint16_t height);
void bsp_lcd_pin_init(void);
void bsp_lcd_init(void);
#if defined LVGL_GUI
void lcd_rectangle_flush(uint16_t sx, uint16_t sy, uint16_t width, uint16_t height, lv_color_t *psbuf);
#endif
void bsp_reset_lcd(void);
uint32_t lcd_read_point(uint16_t x, uint16_t y);
/**
  * @}
  */
  
/** @addtogroup Touch_IIC_Public_Functions
  * @{
  */
void touch_iic_init(void);                						 
void touch_iic_start(void);			
void touch_iic_stop(void);	  			
void touch_iic_send_byte(uint8_t txd);				
void touch_iic_ack(void);				
void touch_iic_nack(void);
uint8_t touch_iic_read_byte(unsigned char ack);	
uint8_t touch_iic_wait_ack(void); 
/**
  * @}
  */

/** @addtogroup GT9147_Public_Functions
  * @{
  */
void gt9147_send_cfg(uint8_t mode);
void gt9147_rd_reg(uint16_t reg, uint8_t *buf, uint8_t len);
uint8_t gt9147_wr_reg(uint16_t reg, uint8_t *buf, uint8_t len);
uint8_t gt9147_init(void);
uint8_t gt9147_touch_scan(void); 
/**
  * @}
  */
  
/** @addtogroup GT911_Public_Functions
  * @{
  */
void gt911_init(void);
void gt911_write_prepare(uint16_t addr);
void gt911_write_data (uint16_t addr, uint8_t value);
void gt911_slave_address_config(void);
void gt911_int_sync(uint16_t ms);
uint8_t gt911_touch_scan(void);
uint8_t gt911_read_data (uint16_t addr, uint8_t cnt, uint8_t *value);
/**
  * @}
  */
  
/** @addtogroup Touch_Public_Functions
  * @{
  */
uint8_t bsp_touch_init(void);
/**
  * @}
  */


/**
  * @}
  */

/**
  * @}
  */
#ifdef _cplusplus
}
#endif	

#endif

