/**********************************************************************************
 *
 * @file    bsp_led.c
 * @brief   bsp_led C file
 *
 * @date    12 Jul 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          12 Jul 2022     Kevin          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "bsp_led.h"
/** @addtogroup Micro_Driver
  * @{
  */

#if defined __BS_LED_H__
//define word

/** @defgroup MD Board function ES32F0930 GPIO
  * @brief MD Board function ES32F0930 micro driver
  * @{
  */

/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/

/** @defgroup BSP_GPIO_Public_Macros GPIO Public Macros
  * @{
  */

/**
  * @brief BSP_GPIO_PIN Pin
  */
#define BSP_GPIO_PIN_0                      (0x0001U)  /*!< Select pin 0 */
#define BSP_GPIO_PIN_1                      (0x0002U)  /*!< Select pin 1 */
#define BSP_GPIO_PIN_2                      (0x0004U)  /*!< Select pin 2 */
#define BSP_GPIO_PIN_3                      (0x0008U)  /*!< Select pin 3 */
#define BSP_GPIO_PIN_4                      (0x0010U)  /*!< Select pin 4 */
#define BSP_GPIO_PIN_5                      (0x0020U)  /*!< Select pin 5 */
#define BSP_GPIO_PIN_6                      (0x0040U)  /*!< Select pin 6 */
#define BSP_GPIO_PIN_7                      (0x0080U)  /*!< Select pin 7 */
#define BSP_GPIO_PIN_8                      (0x0100U)  /*!< Select pin 8 */
#define BSP_GPIO_PIN_9                      (0x0200U)  /*!< Select pin 9 */
#define BSP_GPIO_PIN_10                     (0x0400U) /*!< Select pin 10 */
#define BSP_GPIO_PIN_11                     (0x0800U) /*!< Select pin 11 */
#define BSP_GPIO_PIN_12                     (0x1000U) /*!< Select pin 12 */
#define BSP_GPIO_PIN_13                     (0x2000U) /*!< Select pin 13 */
#define BSP_GPIO_PIN_14                     (0x4000U) /*!< Select pin 14 */
#define BSP_GPIO_PIN_15                     (0x8000U) /*!< Select pin 15 */
#define BSP_GPIO_PIN_ALL                    (0xFFFFU) /*!< Select all pins */

/**
  * @brief BSP_GPIO_LCK Lock
  */
#define BSP_GPIO_UNLOCK                  (0x00000000UL)       /** @brief Select input mode */
#define BSP_GPIO_LOCK                    (0X00000001UL)       /** @brief Select output mode */


/**
  * @brief BSP_GPIO_MODE Mode
  */
#define BSP_GPIO_MODE_INPUT                 (0x00000000UL)       /** @brief Select input mode */
#define BSP_GPIO_MODE_OUTPUT                (0X00000001UL)       /** @brief Select output mode */
#define BSP_GPIO_MODE_FUNCTION              (0X00000002UL)       /** @brief Select function mode */
#define BSP_GPIO_MODE_ANALOG                (0x00000003UL)       /** @brief Select analog mode */

/**
  * @brief BSP_GPIO_OT Output Type
  */
#define BSP_GPIO_OUTPUT_PUSHPULL            (0x00000000U) /** @brief Select push-pull as output type */
#define BSP_GPIO_OUTPUT_OPENDRAIN           (0x00000001U) /** @brief Select open-drain as output type */

/**
  * @brief BSP_GPIO_PULL FLOATING/ Pull Up / Pull Down
  */
#define BSP_GPIO_PULL_FLOATING              (0x00000000UL) /** @brief Select I/O no pull */
#define BSP_GPIO_PULL_UP                    (0x00000001UL) /** @brief Select I/O pull up */
#define BSP_GPIO_PULL_DOWN                  (0x00000002UL) /** @brief Select I/O pull down */

/**
  * @brief BSP_GPIO_DS Output driving/sinking current
  */
#define BSP_GPIO_DRIVING_16MA                    (0x00000001U) /** @brief Select I/O 16mA driving/sinking current */
#define BSP_GPIO_DRIVING_8MA                     (0x00000000U) /** @brief Select I/O 8mA driving/sinking current */

/**
  * @brief BSP_GPIO_FUNCTION Alternate Function
  */
#define BSP_GPIO_AF0                       (0x00000000UL) /** @brief Select alternate function 0 */
#define BSP_GPIO_AF1                       (0x00000001UL) /** @brief Select alternate function 1 */
#define BSP_GPIO_AF2                       (0x00000002UL) /** @brief Select alternate function 2 */
#define BSP_GPIO_AF3                       (0x00000003UL) /** @brief Select alternate function 3 */
#define BSP_GPIO_AF4                       (0x00000004UL) /** @brief Select alternate function 4 */
#define BSP_GPIO_AF5                       (0x00000005UL) /** @brief Select alternate function 5 */
#define BSP_GPIO_AF6                       (0x00000006UL) /** @brief Select alternate function 6 */
#define BSP_GPIO_AF7                       (0x00000007UL) /** @brief Select alternate function 7 */
#define BSP_GPIO_AF8                       (0x00000008UL) /** @brief Select alternate function 8 */
#define BSP_GPIO_AF9                       (0x00000009UL) /** @brief Select alternate function 9 */

/**
  * @brief BSP_GPIO_FILTER Input Filter
  */
#define BSP_GPIO_FILTER_BYPASS             (0x00000000UL) /** @brief Select bypass */
#define BSP_GPIO_FILTER_GLITCH             (0x00000001UL) /** @brief Select glitch free for 20ns plus */

/**
  * @brief BSP_GPIO_IST Input Type
  */
#define BSP_GPIO_IST_TTL                   (0x00000000UL) /** @brief Select TTL IO Level */
#define BSP_GPIO_IST_CMOS                  (0x00000001UL) /** @brief Select CMOS IO Level */

/**
  * @}
  */
/** @defgroup MD_GPIO_Pubulic_Types GPIO Pubulic Types
  * @{
  */

/**
  * @brief MD GPIO Init Structure definition
  */
typedef struct
{
    uint32_t Pin;          /*!< Specifies the GPIO pins to be configured.
                              This parameter can be any value of @ref MD_GPIO_PC_PIN */

    uint32_t Mode;         /*!< Specifies the operating mode for the selected pins.
                              This parameter can be a value of @ref MD_GPIO_PC_MODE.

                              GPIO HW configuration can be modified afterwards using unitary function @ref md_gpio_set_mode().*/

    uint32_t OutputType;   /*!< Specifies the operating output type for the selected pins.
                              This parameter can be a value of @ref MD_GPIO_PC_OT.

                              GPIO HW configuration can be modified afterwards using unitary function @ref md_gpio_set_output_type().*/


    uint32_t Pull;         /*!< Specifies the operating Pull-up/Pull down for the selected pins.
                              This parameter can be a value of @ref MD_GPIO_PC_PULL.

                              GPIO HW configuration can be modified afterwards using unitary function @ref md_gpio_set_pull().*/

    uint32_t OutDrive;     /*!< Specifies the output driving current for the selected pins.
                              This parameter can be a value of @ref MD_GPIO_PC_DS.

                              GPIO HW configuration can be modified afterwards using unitary function @ref md_gpio_get_ds().*/

    uint32_t Function;     /*!< Specifies the Peripheral to be connected to the selected pins.
                              This parameter can be a value of @ref MD_GPIO_PC_FUNCTION.

                              GPIO HW configuration can be modified afterwards using unitary function @ref md_gpio_set_function0_7() and md_gpio_set_function8_15().*/
} bsp_gpio_inittypedef;

/**
  * @brief BSP Board function ES32F0930 GPIO Init Structure definition
  */


static bsp_gpio_inittypedef GPIOC_PIN5_Init =    /**< GPIOC Pin5(LED1) init structure */
{
    BSP_GPIO_PIN_5,
    BSP_GPIO_MODE_OUTPUT,
    BSP_GPIO_OUTPUT_PUSHPULL,
    BSP_GPIO_PULL_FLOATING,
    BSP_GPIO_DRIVING_8MA,
    BSP_GPIO_AF0
};

static bsp_gpio_inittypedef GPIOD_PIN0_Init =    /**< GPIOD Pin0(LED2) init structure */
{
    BSP_GPIO_PIN_0,
    BSP_GPIO_MODE_OUTPUT,
    BSP_GPIO_OUTPUT_PUSHPULL,
    BSP_GPIO_PULL_FLOATING,
    BSP_GPIO_DRIVING_8MA,
    BSP_GPIO_AF0
};

static bsp_gpio_inittypedef GPIOD_PIN6_Init =    /**< GPIOD Pin6(LED3) init structure */
{
    BSP_GPIO_PIN_6,
    BSP_GPIO_MODE_OUTPUT,
    BSP_GPIO_OUTPUT_PUSHPULL,
    BSP_GPIO_PULL_FLOATING,
    BSP_GPIO_DRIVING_8MA,
    BSP_GPIO_AF0
};

static bsp_gpio_inittypedef GPIOC_PIN2_Init =    /**< GPIOC Pin5(LED4) init structure */
{
    BSP_GPIO_PIN_2,
    BSP_GPIO_MODE_OUTPUT,
    BSP_GPIO_OUTPUT_PUSHPULL,
    BSP_GPIO_PULL_FLOATING,
    BSP_GPIO_DRIVING_8MA,
    BSP_GPIO_AF0
};

/**
  * @}
  */

/* Private constants ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/**
  * @} MD Board function ES32F0930_Private_Macros
  */

/* Private function prototypes ------------------------------------------------*/
static ErrorStatus bsp_gpio_init(GPIO_TypeDef *GPIOx, bsp_gpio_inittypedef *GPIO_InitStruct)
{
    ErrorStatus status = ERROR;

    /* Check the GPIO Instance GPIOx*/
    assert_param(IS_MD_GPIO_ALL_INSTANCE(GPIOx));

    /* Check the GPIO parameters from GPIO_InitStruct*/
    assert_param(IS_MD_GPIO_PIN(GPIO_InitStruct->Pin));
    assert_param(IS_MD_GPIO_MODE(GPIO_InitStruct->Mode));
    assert_param(IS_MD_GPIO_OUTPUT_TYPE(GPIO_InitStruct->OutputType));
    assert_param(IS_MD_GPIO_PULL(GPIO_InitStruct->Pull));
    assert_param(IS_MD_GPIO_DS(GPIO_InitStruct->OutDrive));
    assert_param(IS_MD_GPIO_FUNCTION(GPIO_InitStruct->Function));

    if (POSITION_VAL(GPIO_InitStruct->Pin) <= 7)
    {
        MODIFY_REG(GPIOx->AFL, (GPIO_AFL_AFSEL0_MSK  << (POSITION_VAL(GPIO_InitStruct->Pin) * 4U)), (GPIO_InitStruct->Function << (POSITION_VAL(GPIO_InitStruct->Pin) * 4U)));
    }
    else
    {
        MODIFY_REG(GPIOx->AFH, (GPIO_AFH_AFSEL8_MSK  << (POSITION_VAL(GPIO_InitStruct->Pin >> 8U) * 4U)), (GPIO_InitStruct->Function << (POSITION_VAL(GPIO_InitStruct->Pin >> 8U) * 4U)));
    }

    MODIFY_REG(GPIOx->OT, (GPIO_OT_OT_MSK  << (POSITION_VAL(GPIO_InitStruct->Pin))), (GPIO_InitStruct->OutputType << (POSITION_VAL(GPIO_InitStruct->Pin))));
    MODIFY_REG(GPIOx->PUD, (GPIO_PUD_PUD0_MSK  << (POSITION_VAL(GPIO_InitStruct->Pin) * 2U)), (GPIO_InitStruct->Pull << (POSITION_VAL(GPIO_InitStruct->Pin) * 2U)));
    MODIFY_REG(GPIOx->DS, (GPIO_DS_DS_MSK  << (POSITION_VAL(GPIO_InitStruct->Pin))), (GPIO_InitStruct->OutDrive << (POSITION_VAL(GPIO_InitStruct->Pin))));
    MODIFY_REG(GPIOx->MOD, (GPIO_MOD_MOD0_MSK  << (POSITION_VAL(GPIO_InitStruct->Pin) * 2U)), (GPIO_InitStruct->Mode << (POSITION_VAL(GPIO_InitStruct->Pin) * 2U)));

    status = SUCCESS;

    return status;
}
/* Public functions -----------------------------------------------------------*/
/** @addtogroup MD Board function ES32F0930_Public_Functions GPIO Public Functions
  * @{
  */

/**
* @brief  Initialize the GPIO settings of the peripheral circuit and turn on the GPIO clock used in the RCU.
* @note   PC5: GPIO LED1.
*         PD0: GPIO LED2.
*         PD6: GPIO LED3.
*         PC2: GPIO LED4.
* @param  None
* @retval None
*/
void bsp_led_init()
{
    bsp_led1_gpio_init();
    bsp_led2_gpio_init();
    bsp_led3_gpio_init();
    bsp_led4_gpio_init();
}

void bsp_led1_gpio_init(void)
{
    SET_BIT(RCU->AHBEN, RCU_AHBEN_GPCEN_MSK);
    bsp_gpio_init(GPIOC, &GPIOC_PIN5_Init);
}
void bsp_led2_gpio_init(void)
{
    SET_BIT(RCU->AHBEN, RCU_AHBEN_GPDEN_MSK);
    bsp_gpio_init(GPIOD, &GPIOD_PIN0_Init);
}
void bsp_led3_gpio_init(void)
{
    SET_BIT(RCU->AHBEN, RCU_AHBEN_GPDEN_MSK);
    bsp_gpio_init(GPIOD, &GPIOD_PIN6_Init);
}
void bsp_led4_gpio_init(void)
{
    SET_BIT(RCU->AHBEN, RCU_AHBEN_GPCEN_MSK);
    bsp_gpio_init(GPIOC, &GPIOC_PIN2_Init);
}




/**
  * @brief  Turn-on LED1.
  * @param  None
  * @retval None
  */
void bsp_led1_on()
{
    WRITE_REG(GPIOC->BSBR, (BSP_GPIO_PIN_5 << 16));
}

/**
  * @brief  Turn-off LED1.
  * @param  None
  * @retval None
  */
void bsp_led1_off()
{
    WRITE_REG(GPIOC->BSBR, BSP_GPIO_PIN_5);
}

/**
  * @brief  Turn-on LED2.
  * @param  None
  * @retval None
  */
void bsp_led2_on()
{
    WRITE_REG(GPIOD->BSBR, (BSP_GPIO_PIN_0 << 16));
}

/**
  * @brief  Turn-off LED2.
  * @param  None
  * @retval None
  */
void bsp_led2_off()
{
    WRITE_REG(GPIOD->BSBR, BSP_GPIO_PIN_0);
}

/**
  * @brief  Turn-on LED3.
  * @param  None
  * @retval None
  */
void bsp_led3_on()
{
    WRITE_REG(GPIOD->BSBR, (BSP_GPIO_PIN_6 << 16));
}

/**
  * @brief  Turn-off LED3.
  * @param  None
  * @retval None
  */
void bsp_led3_off()
{
    WRITE_REG(GPIOD->BSBR, BSP_GPIO_PIN_6);
}

/**
  * @brief  Turn-on LED4.
  * @param  None
  * @retval None
  */
void bsp_led4_on()
{
    WRITE_REG(GPIOC->BSBR, (BSP_GPIO_PIN_2 << 16));
}

/**
  * @brief  Turn-off LED4.
  * @param  None
  * @retval None
  */
void bsp_led4_off()
{
    WRITE_REG(GPIOC->BSBR, BSP_GPIO_PIN_2);
}

/**
  * @} MD Board function ES32F0930_Public_Functions
  */
#endif
/**
  * @} Micro_Driver
  */
