/**
  *********************************************************************************
  *
  * @file    bsp_w5500.c
  * @brief   Network driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "ald_gpio.h"
#include "bsp_w5500.h"

/**
  * @addtogroup ES32F03xx_BSP
  * @{
  */

/**
  * @defgroup ETH ethernet
  * @{
  */

/**
  * @defgroup ETH_Public_Variable eth Public Variables
  * @{
  */
bsp_w5500_env_t bsp_w5500_env;
/**
  * @}
  */

/**
  * @defgroup ETH_Private_Function eth Private Functions
  * @{
  */

/*
 * SCK  - PB3
 * MISO - PB4
 * MOSI - PB5
 * CS   - PF0
 * RST  - PE1
 */

/**
  * @brief  w5500 module pin initialize.
  * @retval None.
  */
static void w5500_pin_config(void)
{
	gpio_init_t x;

	/* Initialize sck pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_4;
	ald_gpio_init(GPIOB, GPIO_PIN_3, &x);

	/* Initialize miso pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_4;
	ald_gpio_init(GPIOB, GPIO_PIN_4, &x);

	/* Initialize mosi pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_4;
	ald_gpio_init(GPIOB, GPIO_PIN_5, &x);

	/* Initialize cs pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(W5500_SCS_PORT, W5500_SCS, &x);

	/* Initialize reset pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(W5500_RST_PORT, W5500_RST, &x);

	ald_gpio_write_pin(GPIOB, GPIO_PIN_3, 1);
	ald_gpio_write_pin(GPIOB, GPIO_PIN_4, 1);
	ald_gpio_write_pin(GPIOB, GPIO_PIN_5, 1);
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);
	ald_gpio_write_pin(W5500_RST_PORT, W5500_RST, 0);

	return;
}

/**
  * @brief  w5500 module spi confiugure.
  * @retval None.
  */
static void w5500_spi_config(void)
{
	/* Initialize SPI */
	bsp_w5500_env.h_spi.perh           = SPI0;
	bsp_w5500_env.h_spi.init.mode      = SPI_MODE_MASTER;
	bsp_w5500_env.h_spi.init.dir       = SPI_DIRECTION_2LINES;
	bsp_w5500_env.h_spi.init.data_size = SPI_DATA_SIZE_8;
	bsp_w5500_env.h_spi.init.baud      = SPI_BAUD_4;
	bsp_w5500_env.h_spi.init.phase     = SPI_CPHA_FIRST;
	bsp_w5500_env.h_spi.init.polarity  = SPI_CPOL_LOW;
	bsp_w5500_env.h_spi.init.first_bit = SPI_FIRSTBIT_MSB;
	bsp_w5500_env.h_spi.init.ss_en     = DISABLE;
	bsp_w5500_env.h_spi.init.crc_calc  = DISABLE;
	bsp_w5500_env.h_spi.init.crc_poly  = 7;
	ald_spi_init(&bsp_w5500_env.h_spi);

	return;
}

/**
  * @brief  write 1byte data to w5500 register by spi.
  * @param  reg: specific register.
  * @param  dat: write data.
  * @retval None.
  */
static void w5500_write_1byte(uint16_t reg, uint8_t dat)
{
	uint8_t buf[4];

	buf[0] = (reg >> 8) & 0xFF;
	buf[1] = reg & 0xFF;
	buf[2] = FDM1 | RWB_WRITE | COMMON_R;
	buf[3] = dat;

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 4, 1000);
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);
}

/**
  * @brief  read data from w5500 register by spi.
  * @param  reg: specific register.
  * @param  dat: read data.
  * @retval None.
  */
static void w5500_write_2byte(uint16_t reg, uint16_t dat)
{
	uint8_t buf[5];

	buf[0] = (reg >> 8) & 0xFF;
	buf[1] = reg & 0xFF;
	buf[2] = FDM2 | RWB_WRITE | COMMON_R;
	buf[3] = (dat >> 8) & 0xFF;
	buf[4] = dat & 0xFF;

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 5, 1000);
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);
}

/**
  * @brief  write specific length data to w5500 register by spi.
  * @param  reg: specific register.
  * @param  dat_ptr: write data point.
  * @param  size: specific length.
  * @retval None.
  */

static void w5500_write_nbyte(uint16_t reg, uint8_t *dat_ptr, uint16_t size)
{
	uint8_t buf[3];

	buf[0] = (reg >> 8) & 0xFF;
	buf[1] = reg & 0xFF;
	buf[2] = VDM | RWB_WRITE | COMMON_R;

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 3, 1000);
	ald_spi_send(&bsp_w5500_env.h_spi, dat_ptr, size, 1000);
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);
}

/**
  * @brief  write 1byte data to specific socket.
  * @param  s_idx: specific socket index.
  * @param  reg: specific register.
  * @param  dat: specific data.
  * @retval None.
  */

static void w5500_write_socket_1byte(uint8_t s_idx, uint16_t reg, uint8_t dat)
{
	uint8_t buf[4];

	buf[0] = (reg >> 8) & 0xFF;
	buf[1] = reg & 0xFF;
	buf[2] = FDM1 | RWB_WRITE | (s_idx * 0x20 + 0x08);
	buf[3] = dat;

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 4, 1000);
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);
}

/**
  * @brief  write 2byte data to specific socket.
  * @param  s_idx: specific socket index.
  * @param  reg: specific register.
  * @param  dat: write data.
  * @retval None.
  */
static void w5500_write_socket_2byte(uint8_t s_idx, uint16_t reg, uint16_t dat)
{
	uint8_t buf[5];

	buf[0] = (reg >> 8) & 0xFF;
	buf[1] = reg & 0xFF;
	buf[2] = FDM2 | RWB_WRITE | (s_idx * 0x20 + 0x08);
	buf[3] = (dat >> 8) & 0xFF;
	buf[4] = dat & 0xFF;

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 5, 1000);
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);
}

/**
  * @brief  write 4byte data to specific socket.
  * @param  s_idx: specific socket index.
  * @param  reg: specific register.
  * @param  dat_ptr: write data point.
  * @retval None.
  */
static void w5500_write_socket_4byte(uint8_t s_idx, uint16_t reg, uint8_t *dat_ptr)
{
	uint8_t buf[7];

	buf[0] = (reg >> 8) & 0xFF;
	buf[1] = reg & 0xFF;
	buf[2] = FDM4 | RWB_WRITE | (s_idx * 0x20 + 0x08);
	buf[3] = dat_ptr[0];
	buf[4] = dat_ptr[1];
	buf[5] = dat_ptr[2];
	buf[6] = dat_ptr[3];

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 7, 1000);
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);
}

/**
  * @brief  read 1byte data from w5500 register by spi.
  * @param  reg: specific register.
  * @retval data: read value.
  */
static uint8_t w5500_read_1byte(uint16_t reg)
{
	uint8_t data;
	uint8_t buf[3];

	buf[0] = (reg >> 8) & 0xff;
	buf[1] = reg & 0xff;
	buf[2] = FDM1 | RWB_READ | COMMON_R;

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 3, 1000);
	ald_spi_recv(&bsp_w5500_env.h_spi, &data, 1, 10000);
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);

	return data;
}

/**
  * @brief  read 1byte data from w5500 socket.
  * @param  s_idx: specific socket index.
  * @param  reg: specific register.
  * @retval data: read value.
  */
static uint8_t w5500_read_socket_1byte(uint8_t s_idx, uint16_t reg)
{
	uint8_t buf[3];

	buf[0] = (reg >> 8) & 0xff;
	buf[1] = reg & 0xff;
	buf[2] = FDM1 | RWB_READ | (s_idx * 0x20 + 0x08);

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 3, 1000);
	ald_spi_recv(&bsp_w5500_env.h_spi, buf, 1, 10000);
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);

	return buf[0];
}

/**
  * @brief  read 2byte data from w5500 socket.
  * @param  s_idx: specific socket index.
  * @param  reg: specific register.
  * @retval data: read value.
  */
static uint16_t w5500_read_socket_2byte(uint8_t s_idx, uint16_t reg)
{
	uint8_t buf[3];
	uint16_t data;

	buf[0] = (reg >> 8) & 0xff;
	buf[1] = reg & 0xff;
	buf[2] = FDM2 | RWB_READ | (s_idx * 0x20 + 0x08);

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 3, 1000);
	ald_spi_recv(&bsp_w5500_env.h_spi, buf, 2, 10000);
	data = (buf[0] << 8) | buf[1];
	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);

	return data;
}

/**
  * @}
  */

/**
  * @defgroup ETH_Public_Function eth Public Functions
  * @brief    BSP ETH Functions
  * @verbatim
==================================================================================
               ##### BSP ETH Functions #####
==================================================================================
 [..]   This section provides functions allowing to:
    (+) reset eth module.
    (+) test connect by eth.
    (+) Initialize eth module.
    (+) Set eth dectect gateway.
    (+) Initialize eth socket.
    (+) Connect by socket.
    (+) Socket listen
    (+) Set Socket by UDP Module.
    (+) Read Socket data.
    (+) Write Socket data.
    (+) Process eth interrupt.
    
    @endverbatim
  * @{
  */

/**
  * @brief  w5500 module reset.
  * @retval reset value.
  */
uint8_t bsp_w5500_hw_reset(void)
{
	ald_gpio_write_pin(W5500_RST_PORT, W5500_RST, 0);
	ald_delay_ms(50);
	ald_gpio_write_pin(W5500_RST_PORT, W5500_RST, 1);
	ald_delay_ms(3000);
	if ((w5500_read_1byte(PHYCFGR) & LINK) == 0)
		return 1;

	return 0;
}

/**
  * @brief  test connect by eth.
  * @retval test value.
  */
uint8_t test_connect(void)
{
	if ((w5500_read_1byte(PHYCFGR) & LINK) == 0) {
		ald_delay_ms(200);
		if ((w5500_read_1byte(PHYCFGR) & LINK) == 0)
			return 1;
	}

	return 0;
}

/**
  * @brief  w5500 moudle initialize.
  * @param  con_err: w5500 initialize error data. 
  * @retval None.
  */
void bsp_w5500_init(uint8_t *con_err)
{
	uint8_t i;

	w5500_pin_config();
	w5500_spi_config();
	*con_err = bsp_w5500_hw_reset();
	w5500_write_1byte(MR, RST);
	ald_delay_ms(10);

	w5500_write_nbyte(GAR, bsp_w5500_env.geteway, 4);
	w5500_write_nbyte(SUBR, bsp_w5500_env.sub_mask, 4);
	w5500_write_nbyte(SHAR, bsp_w5500_env.phy_addr, 6);
	w5500_write_nbyte(SIPR, bsp_w5500_env.local_ip, 4);

	for (i = 0; i < 8; ++i) {
		w5500_write_socket_1byte(i, Sn_RXBUF_SIZE, 0x02);
		w5500_write_socket_1byte(i, Sn_TXBUF_SIZE, 0x02);
	}

	w5500_write_2byte(RTR, 0x07d0);
	w5500_write_1byte(RCR, 8);

	bsp_w5500_detect_gateway();
	bsp_w5500_socket_init(0);

	return;
}

/**
  * @brief  set detect gateway.
  * @retval set value
  *         TURE: set ok; FALSE: set error.
  */
uint8_t bsp_w5500_detect_gateway(void)
{
	uint8_t tmp, ip_adde[4];

	ip_adde[0] = bsp_w5500_env.local_ip[0] + 1;
	ip_adde[1] = bsp_w5500_env.local_ip[1] + 1;
	ip_adde[2] = bsp_w5500_env.local_ip[2] + 1;
	ip_adde[3] = bsp_w5500_env.local_ip[3] + 1;

	w5500_write_socket_4byte(0, Sn_DIPR, ip_adde);
	w5500_write_socket_1byte(0, Sn_MR, MR_TCP);
	w5500_write_socket_1byte(0, Sn_CR, OPEN);
	ald_delay_ms(5);

	if (w5500_read_socket_1byte(0, Sn_SR) != SOCK_INIT) {
		w5500_write_socket_1byte(0, Sn_CR, CLOSE);
		return FALSE;
	}

	w5500_write_socket_1byte(0, Sn_CR, CONNECT);

	do {
		tmp = w5500_read_socket_1byte(0, Sn_IR);
		if (tmp != 0)
			w5500_write_socket_1byte(0, Sn_IR, tmp);

		ald_delay_ms(5);

		if ((tmp & IR_TIMEOUT) == IR_TIMEOUT) {
			return FALSE;
		}
		else if (w5500_read_socket_1byte(0, Sn_DHAR) != 0xff) {
			w5500_write_socket_1byte(0, Sn_CR, CLOSE);
			return TRUE;
		}
	} while (1);
}

/**
  * @brief  eth index initialize.
  * @param  s_idx: specific socket.
  * @retval None.
  */
void bsp_w5500_socket_init(uint8_t s_idx)
{
	w5500_write_socket_2byte(0, Sn_MSSR, 1460);
	switch (s_idx) {
	case 0:
		w5500_write_socket_2byte(0, Sn_PORT, (bsp_w5500_env.local_port[0] << 8) | bsp_w5500_env.local_port[1]);
		w5500_write_socket_2byte(0, Sn_DPORTR, (bsp_w5500_env.dest_port[0] << 8) | bsp_w5500_env.dest_port[1]);
		w5500_write_socket_4byte(0, Sn_DIPR, bsp_w5500_env.dest_ip);
		break;

	default:
		break;
	}
}

/**
  * @brief  w5500 connect by eth socket.
  * @param  s_idx: specific socket.
  * @retval connect value
  *         TURE: connect ok; FALSE: connect error.
  */
uint8_t bsp_w5500_socket_connect(uint8_t s_idx)
{
	w5500_write_socket_1byte(s_idx, Sn_MR, MR_TCP);
	w5500_write_socket_1byte(s_idx, Sn_CR, OPEN);
	ald_delay_ms(5);

	if (w5500_read_socket_1byte(s_idx, Sn_SR) != SOCK_INIT) {
		w5500_write_socket_1byte(s_idx, Sn_CR, CLOSE);
		return FALSE;
	}

	w5500_write_socket_1byte(s_idx, Sn_CR, CONNECT);
	return TRUE;
}
/**
  * @brief  w5500 listen by specific socket.
  * @param  s_idx: specific socket.
  * @retval listen value
  *         TURE: listen ok; FALSE: listen error.
  */
uint8_t bsp_w5500_socket_listen(uint8_t s_idx)
{
	w5500_write_socket_1byte(s_idx, Sn_MR, MR_TCP);
	w5500_write_socket_1byte(s_idx, Sn_CR, OPEN);
	ald_delay_ms(5);

	if (w5500_read_socket_1byte(s_idx, Sn_SR) != SOCK_INIT) {
		w5500_write_socket_1byte(s_idx, Sn_CR, CLOSE);
		return FALSE;
	}

	w5500_write_socket_1byte(s_idx, Sn_CR, LISTEN);
	ald_delay_ms(5);

	if (w5500_read_socket_1byte(s_idx, Sn_SR) != SOCK_LISTEN) {
		w5500_write_socket_1byte(s_idx, Sn_CR, CLOSE);
		return FALSE;
	}

	return TRUE;
}
/**
  * @brief  w5500 read by specific socket in udp module.
  * @param  s_idx: specific socket.
  * @retval read value
  *         TURE: read ok; FALSE: read error.
  */
uint8_t bsp_w5500_socket_udp(uint8_t s_idx)
{
	w5500_write_socket_1byte(s_idx, Sn_MR, MR_UDP);
	w5500_write_socket_1byte(s_idx, Sn_CR, OPEN);
	ald_delay_ms(5);

	if (w5500_read_socket_1byte(s_idx, Sn_SR) != SOCK_UDP) {
		w5500_write_socket_1byte(s_idx, Sn_CR, CLOSE);
		return FALSE;
	}
	else
		return TRUE;
}

/**
  * @brief  w5500 read by specific socket.
  * @param  s_idx: specific socket.
  * @param  dat_ptr: read data point.
  * @retval read data length.
  */
uint16_t bsp_w5500_read_socket_data(uint8_t s_idx, uint8_t *dat_ptr)
{
	uint8_t buf[3];
	uint16_t rx_size;
	uint16_t offset, offset1;

	rx_size = w5500_read_socket_2byte(s_idx , Sn_RX_RSR);

	if (rx_size == 0)
		return 0;
	if (rx_size > 1460)
		rx_size = 1460;

	offset  = w5500_read_socket_2byte(s_idx, Sn_RX_RD);
	offset1 = offset;
	offset &= (S_RX_SIZE - 1);

	buf[0] = (offset >> 8) & 0xFF;
	buf[1] = offset & 0xFF;
	buf[2] = VDM | RWB_READ | (s_idx * 0x20 + 0x18);

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 3, 1000);

	if ((offset + rx_size) < S_RX_SIZE) {
		ald_spi_recv(&bsp_w5500_env.h_spi, dat_ptr, rx_size, 10000);
	}
	else {
		offset = S_RX_SIZE - offset;
		ald_spi_recv(&bsp_w5500_env.h_spi, dat_ptr, offset, 10000);

		ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);
		ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);

		buf[0] = 0x0;
		buf[1] = 0x0;
		buf[2] = VDM | RWB_READ | (s_idx * 0x20 + 0x18);
		ald_spi_send(&bsp_w5500_env.h_spi, buf, 3, 1000);
		ald_spi_recv(&bsp_w5500_env.h_spi, dat_ptr + offset, rx_size - offset, 10000);
	}

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);

	offset1 += rx_size;
	w5500_write_socket_2byte(s_idx, Sn_RX_RD, offset1);
	w5500_write_socket_1byte(s_idx, Sn_CR, RECV);

	return rx_size;
}

/**
  * @brief  w5500 write to specific socket.
  * @param  s_idx: specific socket.
  * @param  dat_ptr: write data point.
  * @param  size: write data length.
  * @retval read data length.
  */
void bsp_w5500_write_socket_data(uint8_t s_idx, uint8_t *dat_ptr, uint16_t size)
{
	uint8_t buf[3];
	uint16_t offset, offset1;

	if ((w5500_read_socket_1byte(s_idx, Sn_MR) & 0x0f) != SOCK_UDP) {
		w5500_write_socket_4byte(s_idx, Sn_DIPR, bsp_w5500_env.udp_dip);
		w5500_write_socket_2byte(s_idx, Sn_DPORTR,
				(bsp_w5500_env.udp_dport[0] << 8) | bsp_w5500_env.udp_dport[1]);
	}

	offset  = w5500_read_socket_2byte(s_idx, Sn_TX_WR);
	offset1 = offset;
	offset &= (S_TX_SIZE - 1);

	buf[0] = (offset >> 8) & 0xFF;
	buf[1] = offset & 0xFF;
	buf[2] = VDM | RWB_WRITE | (s_idx * 0x20 + 0x10);

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);
	ald_spi_send(&bsp_w5500_env.h_spi, buf, 3, 1000);

	if ((offset + size) < S_TX_SIZE) {
		ald_spi_send(&bsp_w5500_env.h_spi, dat_ptr, size, 1000);
	}
	else {
		offset = S_TX_SIZE - offset;
		ald_spi_send(&bsp_w5500_env.h_spi, dat_ptr, offset, 1000);

		ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);
		ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 0);


		buf[0] = 0x0;
		buf[1] = 0x0;
		buf[2] = VDM | RWB_WRITE | (s_idx * 0x20 + 0x10);
		ald_spi_send(&bsp_w5500_env.h_spi, buf, 3, 1000);
		ald_spi_recv(&bsp_w5500_env.h_spi, dat_ptr + offset, size - offset, 10000);
	}

	ald_gpio_write_pin(W5500_SCS_PORT, W5500_SCS, 1);

	offset1 += size;
	w5500_write_socket_2byte(s_idx, Sn_TX_WR, offset1);
	w5500_write_socket_1byte(s_idx, Sn_CR, SEND);
}

/**
  * @brief  w5500 interrupt process.
  * @retval None.
  */
void bsp_w5500_interrupt_process(void)
{
	uint8_t tmp, sr;

	if ((tmp = w5500_read_1byte(SIR)) == 0)
		return;
	if ((tmp & S0_INT) != S0_INT)
		return;

	while (tmp) {
		sr = w5500_read_socket_1byte(0, Sn_IR);
		w5500_write_socket_1byte(0, Sn_IR, sr);

		if (sr & IR_CON)
			bsp_w5500_env.state |= S_CONN;

		if (sr & IR_DISCON) {
			w5500_write_socket_1byte(0, Sn_CR, CLOSE);
			bsp_w5500_socket_init(0);
			bsp_w5500_env.state = 0;
		}

		if (sr & IR_SEND_OK)
			bsp_w5500_env.flag |= S_TRANSMITOK;

		if (sr & IR_RECV)
			bsp_w5500_env.flag |= S_RECEIVE;

		if (sr & IR_TIMEOUT) {
			w5500_write_socket_1byte(0, Sn_CR, CLOSE);
			bsp_w5500_env.state = 0;
		}

		tmp = w5500_read_1byte(SIR);
	}

	return;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
