/**
  *********************************************************************************
  *
  * @file    bsp_w25q32.c
  * @brief   W25Q32 driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include "bsp_w25q32.h"
#include "ald_gpio.h"

 /**
   * @addtogroup ES32F03xx_BSP
   * @{
   */

/**
  * @defgroup W25Q32 spi flash
  * @{
  */

/**
  * @defgroup W25Q32_Private_Params spi flash Private Variables
  * @{
  */

static spi_handle_t h_spi;

/**
  * @}
  */

/**
  * @defgroup W25Q32_Private_Functions spi flash Private Functions
  * @{
  */

/**
  * @brief  Initializate spi_flash pin
  * @retval None.
  */
static void w25q32_pin_init(void)
{
	gpio_init_t x;

	/* Initialize cs pin PE0 */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
        x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;

	ald_gpio_init(GPIOE, GPIO_PIN_0, &x);
	ald_gpio_write_pin(GPIOE, GPIO_PIN_0, 1);
	/* Enable spi lcd cs set */
	ald_gpio_init(GPIOE, GPIO_PIN_14, &x);
	ald_gpio_write_pin(GPIOE, GPIO_PIN_14, 1);
	ald_gpio_init(GPIOE, GPIO_PIN_15, &x);
	ald_gpio_write_pin(GPIOE, GPIO_PIN_15, 1);

	/* Initialize MOSI pin PB1 */
	x.func = GPIO_FUNC_4;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);

	/* Initialize CLK pin PB1 */
	x.func = GPIO_FUNC_4;
	ald_gpio_init(GPIOB, GPIO_PIN_13, &x);

	/* Initialize MISO pin PB0 */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
        x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_4;
        ald_gpio_init(GPIOB, GPIO_PIN_14, &x);
}

/**
  * @defgroup W25Q32_Public_Functions spi flash Public Functions
  * @brief    BSP SPI_Flash Functions
  * @verbatim
 ===============================================================================
              ##### BSP W25Q32 Functions #####
 ===============================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp w25q32 module
      (+) Write an amount of data to a specific spi_flash address
      (+) Read an amount of data to a specific spi_flash address
      (+) Erase a specific spi_flash sector address
    @endverbatim
  * @{
  */

/**
  * @brief  Initializate w25q32 moudle
  * @retval None.
  */
void w25q32_module_init(void)
{
	memset(&h_spi, 0, sizeof(h_spi));

	w25q32_pin_init();

	h_spi.perh           = SPI1;
	h_spi.init.mode      = SPI_MODE_MASTER;
	h_spi.init.dir       = SPI_DIRECTION_2LINES;
	h_spi.init.data_size = SPI_DATA_SIZE_8;
	h_spi.init.baud      = SPI_BAUD_8;
	h_spi.init.phase     = SPI_CPHA_FIRST;
	h_spi.init.polarity  = SPI_CPOL_LOW;
	h_spi.init.first_bit = SPI_FIRSTBIT_MSB;
	h_spi.init.ss_en     = DISABLE;
	h_spi.init.crc_calc  = DISABLE;

	ald_spi_init(&h_spi);
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_write(uint32_t addr, uint8_t *buf, uint16_t size)
{
	uint8_t write_cmd  = 0x02;
	uint8_t write_enable_cmd = 0x06;
	uint8_t tx_buf[261] = {0};
	uint8_t tmp = 0;

	if (buf == NULL)
		return W25_ERROR;

	tx_buf[0] = write_cmd;
	tx_buf[1] = (addr >> 16) & 0xff;
	tx_buf[2] = (addr >> 8) & 0xff;
	tx_buf[3] = addr & 0xff;

	for (tmp = 0; tmp < size; tmp++) {
		tx_buf[4 + tmp] = *buf++;
	}

	W25Q32_CS_CLR();

	if (ald_spi_send(&h_spi, &write_enable_cmd, 1, 500) != OK) {
		W25Q32_CS_SET();
		return W25_ERROR;
	}

	W25Q32_CS_SET();
	W25Q32_CS_CLR();

	if (ald_spi_send(&h_spi, tx_buf, size + 4, 500) != OK) {
		W25Q32_CS_SET();
		return W25_ERROR;
	}

	W25Q32_CS_SET();
	return W25_OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_read(uint32_t addr, uint8_t *buf, uint16_t size)
{
	uint8_t read_cmd = 0x03;
	uint8_t add_buf[5] = {0};

	W25Q32_CS_CLR();

	add_buf[0] = read_cmd;
	add_buf[1] = (addr >> 16) & 0xff;
	add_buf[2] = (addr >> 8) & 0xff;
	add_buf[3] = addr & 0xff;

	if (buf == NULL)
		return W25_ERROR;

	if (ald_spi_send(&h_spi, add_buf, 4, 500) != OK) {
		W25Q32_CS_SET();
		return W25_ERROR;
	}

	if (ald_spi_recv(&h_spi, buf, size, 500) != OK) {
		W25Q32_CS_SET();
		return W25_ERROR;
	}

	W25Q32_CS_SET();

	return W25_OK;
}
/**
  * @brief  Erase an sector of data in blocking mode.
  * @param  addr: specific sector address want erase.
  * @retval Status, see @ref w25_status_t.
  */
w25_status_t w25q32_sector_erase(uint8_t addr)
{
	uint8_t sector_erase_cmd = 0x20;
	uint8_t write_enable_cmd = 0x06;
	uint8_t add_buf[4] = {0};

	add_buf[0] = sector_erase_cmd;
	add_buf[1] = (addr >> 16) & 0xff;
	add_buf[2] = (addr >> 8) & 0xff;
	add_buf[3] = addr & 0xff;

	W25Q32_CS_CLR();

	if (ald_spi_send(&h_spi, &write_enable_cmd, 1, 500) != OK) {
		W25Q32_CS_SET();
		return W25_ERROR;
	}

	W25Q32_CS_SET();

	W25Q32_CS_CLR();

	if (ald_spi_send(&h_spi, add_buf, 4, 500) != OK) {
		W25Q32_CS_SET();
		return W25_ERROR;
	}

	W25Q32_CS_SET();

	return W25_OK;
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
