/**
  *********************************************************************************
  *
  * @file    bsp_uart_usb.c
  * @brief   uart_usb driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <stdio.h>
#include "bsp_uart_usb.h"

/** @addtogroup ES32F03xx_BSP
  * @{
  */

/**
  * @defgroup UART_USB uart usb
  * @{
  */

/**
  * @defgroup UART_usb_Private_Params uart usb Private Variables
  * @{
  */

static uart_handle_t h_uart;

/**
  * @}
  */
/**
  * @defgroup UART_usb_Private_Functions uart usb Private Functions
  * @{
  */
 /**
  * @brief  initialize pin of uart_usb module  
  * @retval None
  */ 
static void uart_usb_pin_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_5;
	ald_gpio_init(GPIOC, GPIO_PIN_12, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_5;
	ald_gpio_init(GPIOD, GPIO_PIN_2, &x);
}
/**
  * @}
  */

/**
  * @defgroup UART_USB_Public_Functions uart usb Public Functions
  * @brief    BSP UART_usb Functions
  * @verbatim
=========================================================================================
              ##### BSP UART_usb Functions #####
=========================================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp uart_usb module.
      (+) Send data by uart_usb module.
      (+) Recv data by uart_usb module.
    @endverbatim
  * @{
  */

/**
  * @brief  uart_usb module initialize function
  * @param  baud: communication baud rate.
  * @retval None
  */
void uart_usb_module_init(uint32_t baud)
{
	uart_usb_pin_init();
	memset(&h_uart, 0, sizeof(h_uart));
	/* Initialize uart */
	h_uart.perh             = UART2;
	h_uart.init.baud        = baud;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;

	ald_uart_init(&h_uart);
}

/**
  * @brief  Sends an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref uart_usb_status_t.
  */
uart_usb_status_t bsp_uart_usb_send(uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return UART_USB_ERROR;

	if (ald_uart_send(&h_uart, buf, size, 500) != 0)
		return UART_USB_ERROR;

	return UART_USB_OK;
}

/**
  * @brief  Recvs an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref uart_usb_status_t.
  */
uart_usb_status_t bsp_uart_usb_recv(uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return UART_USB_ERROR;

	if (ald_uart_recv(&h_uart, buf, size, 500) != 0)
		return UART_USB_ERROR;

	return UART_USB_OK;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
